#include <stdio.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <sys/fcntl.h>
#include <sys/types.h>
#include <string.h>
#include <stdarg.h>
#ifdef HAVE_GETOPT_H
#include <getopt.h>
#endif
#include "atacmds.h"
#include "ataprint.h"
#include "extern.h"
#include "knowndrives.h"
#include "scsicmds.h"
#include "scsiprint.h"
#include "smartctl.h"
#include "utility.h"

extern const char *atacmds_c_cvsid, *ataprint_c_cvsid, *knowndrives_c_cvsid, *scsicmds_c_cvsid, *scsiprint_c_cvsid, *utility_c_cvsid;

// This is a block containing all the "control variables".  We declare
// this globally in this file, and externally in other files.
smartmonctrl *con=NULL;


/*  void prints help information for command syntax */
void Usage (void){
  printf("Usage: sendidle [options] device\n\n");
  printf("============================================ SHOW INFORMATION OPTIONS =====\n\n");
#ifdef HAVE_GETOPT_LONG
  printf(
"  -h, -?, --help, --usage\n"
"         Display this help and exit\n\n"
"  -V, --version\n"
"         Print version information and exit\n\n"
"  -i, --immediate                                                       \n"
"         Send IDLE_IMMEDITATE command for device\n\n"
"  -t VALUE, --timer VALUE                                                        \n"
"         The standby timer value\n\n"
  );
#else
  printf(
"  -h, -?    Display this help and exit\n"
"  -V        Print version information\n"
"  -i        Send IDLE_IMMEDITATE command for device\n"
"  -t VALUE  The standby timer value\n\n"
  );
#endif
}

unsigned char tryata=0,tryscsi=1;
unsigned char immediate = 0, power_mode=0;
unsigned int timer = 0;
int smartcommandhandler(int device, smart_command_set command, int select, char *data);

/*      Takes command options and sets features to be run */    
void ParseOpts (int argc, char** argv){
  int optchar;
  int badarg;
  int captive;
  extern char *optarg;
  extern int optopt, optind, opterr;
  // Please update getvalidarglist() if you edit shortopts
  const char *shortopts = "h?V:t:i:p";
#ifdef HAVE_GETOPT_LONG
//  char *arg;
  // Please update getvalidarglist() if you edit longopts
  struct option longopts[] = {
    { "help",            no_argument,       0, 'h' },
    { "usage",           no_argument,       0, 'h' },
    { "version",         no_argument,       0, 'V' },
    { "device",          required_argument, 0, 'd' },
    { "timer",           required_argument, 0, 't' },
    { "immediate",       no_argument,       0, 'i' },
    { "power",       no_argument,       0, 'p' },
    { 0,                 0,                 0, 0   }
  };
#endif
  
  memset(con,0,sizeof(*con));
  con->testcase=-1;
  opterr=optopt=0;
  badarg = captive = FALSE;
  
  // This miserable construction is needed to get emacs to do proper indenting. Sorry!
  while (-1 != (optchar = 
#ifdef HAVE_GETOPT_LONG
        getopt_long(argc, argv, shortopts, longopts, NULL)
#else
        getopt(argc, argv, shortopts)
#endif
        )){
    switch (optchar){
    case 'i':
        immediate = 1;
      break;
    case 'p':
        power_mode = 1;
      break;      
    case 't':
        timer = strtol(optarg, NULL, 0); 
        printf(" timer %d\n", timer);
      break;
    case 'h':
    case '?':
    default:
      Usage();
      exit(0);  
    } // closes switch statement to process command-line options
    }  
}
// Printing function (controlled by global con->veryquietmode) 
// [From GLIBC Manual: Since the prototype doesn't specify types for
// optional arguments, in a call to a variadic function the default
// argument promotions are performed on the optional argument
// values. This means the objects of type char or short int (whether
// signed or not) are promoted to either int or unsigned int, as
// appropriate.]
void pout(char *fmt, ...){
  va_list ap;
  
  // initialize variable argument list 
  va_start(ap,fmt);
  if (con->veryquietmode){
    va_end(ap);
    return;
  }

  // print out
  vprintf(fmt,ap);
  va_end(ap);
  fflush(stdout);
  return;
}

int ataSendIdle (int device ){ 
  
  if (smartcommandhandler(device, SEND_IDLE, timer, NULL)){
    syserror("Error Send IDLE command failed");
    return -1;
  }
  return 0;
}
int ataSendIdleImmediate (int device ){ 
  
  if (smartcommandhandler(device, SEND_IDLE_IMMEDIATE, 0, NULL)){
    syserror("Error Send IDLE IMMEDIATE command failed");
    return -1;
  }
  return 0;
}


/* Main Program */
int main (int argc, char **argv){
  int fd,retval=0;
  char *device;
  smartmonctrl control;
  int dev_type, flags;

  // define control block for external functions
  con=&control;

  // Part input arguments
  ParseOpts(argc,argv);

  device = argv[argc-1];
  dev_type = guess_linux_device_type(device);
  if (GUESS_DEVTYPE_SCSI == dev_type) {
    flags = O_RDWR | O_NONBLOCK;
    tryscsi = 1;
  } else { /* treat "don't know" case as an ATA device as well */
    flags = O_RDONLY | O_NONBLOCK;
    tryata = 1;
  }

//    printf(" immediate = %d, timer %d, tryscsi %d\n", immediate, timer, tryscsi); 
 // open device - SCSI devices are opened (O_RDWR | O_NONBLOCK) so the
  // scsi generci device can be used (needs write permission for MODE
  // SELECT command) plus O_NONBLOCK to stop open hanging if media not
  // present (e.g. with st).
	fd = open(device, flags);
	if (fd<0) {
	    char errmsg[256];
	    snprintf(errmsg,256,"Sendidle open device: %s failed",argv[argc-1]);
	    errmsg[255]='\0';
	    syserror(errmsg);
	    return FAILDEV;
	}
    
    con->ismvsata = 0;
    retval = scsiPrintMain(device, fd);
    if (con->ismvsata)
    {   
    	if(power_mode){
		    int powermode=0;
		   
		    FILE *ff=NULL;
#define	SATA_POWER_MODE	"/tmp/sata_power"		    
		    
		    powermode  = ataCheckPowerMode(fd);
		    ff=fopen(SATA_POWER_MODE, "wt");
		    if(!ff){
		    	close(fd);
		    	return -1;
		    }
		    switch (powermode) {
		      case -1:
		        if (errno == ENOSYS) {
		          pout("CHECK POWER STATUS not implemented, ignoring -n Option\n"); break;
		        }
		        printf("Power Mode: SLEEP\n");
		        fprintf(ff,"Power Mode: SLEEP\n");
		        break;
		      case 0:
		        printf("Power Mode: STANDBY\n");
		        fprintf(ff,"Power Mode: STANDBY\n");
		        break;
		      case 0x80:
		        printf("Power Mode: IDLE\n");
		        fprintf(ff,"Power Mode: IDLE\n");
		        break;
		      case 0xff:
		     	printf("Power Mode: ACTIVE/IDLE\n");
				fprintf(ff,"Power Mode: ACTIVE/IDLE\n");
		        break;
		      default:
		        printf("Power Mode: \n");
		        fprintf(ff,"Power Mode: \n");
		        break;
		    }    	
		    close(fd);
		    fclose(ff);
		    return 0;
	    }
        if (immediate)
        {
            if (ataSendIdleImmediate(fd)) {
                pout("Sendidle: IDLE IMMEDIATE Failed.\n\n");
            }
            else
                pout("IDLE IMMEDIATE command was sent.\n");

        }
        else
        {
            if (ataSendIdle(fd)) {
                pout("Sendidle: IDLE Failed.\n\n");
            }
            else
                pout("IDLE command was sent.\n");
        }
    }
    else {
        Usage();
		close(fd);
        return FAILCMD;
    }
	close(fd);
	return retval;
}

