/*
 *  x1205.c - An i2c driver for the Xicor X1205 RTC
 *  Copyright 2004 Karen Spearel
 *  Copyright 2005 Alessandro Zummo
 *
 *  please send all reports to:
 *   kas11 at tampabay dot rr dot com
 *      a dot zummo at towertech dot it
 *
 *  based on the other drivers in this same directory.
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/slab.h>
#include <linux/i2c.h>
#include <linux/string.h>
#include <linux/bcd.h>
#include <linux/rtc.h>
#include <linux/list.h>
#include <linux/miscdevice.h>
#include <linux/delay.h>

#include <linux/proc_fs.h>
#include <linux/spinlock.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#define DRV_VERSION "0.9.9"

/* Addresses to scan: none. This chip is located at
 * 0x6f and uses a two bytes register addressing.
 * Two bytes need to be written to read a single register,
 * while most other chips just require one and take the second
 * one as the data to be written. To prevent corrupting
 * unknown chips, the user must explicitely set the probe parameter.
 */

/* offsets into CCR area */

#define CCR_SEC         0
#define CCR_MIN         1
#define CCR_HOUR      2
#define CCR_MDAY      3
#define CCR_MONTH      4
#define CCR_YEAR      5
#define CCR_WDAY      6
#define CCR_Y2K         7

#define X1205_REG_SR      0x3F   /* status register */
#define X1205_REG_Y2K      0x37
#define X1205_REG_DW      0x36
#define X1205_REG_YR      0x35
#define X1205_REG_MO      0x34
#define X1205_REG_DT      0x33
#define X1205_REG_HR      0x32
#define X1205_REG_MN      0x31
#define X1205_REG_SC      0x30
#define X1205_REG_DTR      0x13
#define X1205_REG_ATR      0x12
#define X1205_REG_INT      0x11
#define X1205_REG_0      0x10
#define X1205_REG_Y2K1      0x0F
#define X1205_REG_DWA1      0x0E
#define X1205_REG_YRA1      0x0D
#define X1205_REG_MOA1      0x0C
#define X1205_REG_DTA1      0x0B
#define X1205_REG_HRA1      0x0A
#define X1205_REG_MNA1      0x09
#define X1205_REG_SCA1      0x08
#define X1205_REG_Y2K0      0x07
#define X1205_REG_DWA0      0x06
#define X1205_REG_YRA0      0x05
#define X1205_REG_MOA0      0x04
#define X1205_REG_DTA0      0x03
#define X1205_REG_HRA0      0x02
#define X1205_REG_MNA0      0x01
#define X1205_REG_SCA0      0x00

//Base Address
#define X1205_CCR_BASE      0x30   /* Base address of CCR */
#define X1205_ALM0_BASE     0x00   /* Base address of ALARM0 */
#define X1205_ALM1_BASE	    0x08	
//#define	X1205_INT_BASE		0x11   /* Base address of Interrupt Control*/

#define X1205_ALM0_INT_BIT	0x05	

#define X1205_SR_RTCF      0x01   /* Clock failure */
#define X1205_SR_WEL       0x02   /* Write Enable Latch */
#define X1205_SR_RWEL      0x04   /* Register Write Enable */
#define X1205_ALARM_EN	   0x80
	
#define X1205_DTR_DTR0      0x01
#define X1205_DTR_DTR1      0x02
#define X1205_DTR_DTR2      0x04

#define X1205_HR_MIL      0x80   /* Set in ccr.hour for 24 hr mode */
#define EPOCH		2000
#define SYS_EPOCH	1900

//#define	__RTC_DEBUG__
static int x1205_use_count = 0;
extern void mvRtcCharRead(unsigned int offset, unsigned char *data);
extern void mvRtcCharWrite(unsigned int  offset, unsigned char data);
extern void i2c_init(int speed,int slaveaddr);

/*
 * In the routines that deal directly with the x1205 hardware, we use
 * rtc_time -- month 0-11, hour 0-23, yr = calendar year-epoch
 * Epoch is initialized as 2000. Time is set to UTC.
 */
 
static int x1205_set_datetime(struct rtc_time *tm,int datetoo, u8 reg_base);
int check_if_RTC_on(struct rtc_time * tm){
	/* Just judge if the rtc starts simply. If no, start it*/
	if((tm->tm_sec==0)&&(tm->tm_min ==0)){
		tm->tm_sec=1;
		x1205_set_datetime( tm,1,X1205_CCR_BASE);
	}	
	return 0;
	
} 	

static int x1205_get_datetime( struct rtc_time *tm, u8 reg_base)
{
   int i=0;
   unsigned char buf[8];
   
   for (i = 0; i <  8 ; i++) {
    	mvRtcCharRead(reg_base + i, &buf[i]);
    }
    	   	
   tm->tm_sec = BCD2BIN(buf[CCR_SEC]);
   tm->tm_min = BCD2BIN(buf[CCR_MIN]);
   tm->tm_hour = BCD2BIN(buf[CCR_HOUR] & 0x3F); /* hr is 0-23 */
   tm->tm_mday = BCD2BIN(buf[CCR_MDAY]);
   tm->tm_mon = BCD2BIN(buf[CCR_MONTH])-1;
   tm->tm_year = BCD2BIN(buf[CCR_YEAR]) + EPOCH - SYS_EPOCH;
   tm->tm_wday = buf[CCR_WDAY];
   check_if_RTC_on(tm);
#ifdef __RTC_DEBUG__   
   printk("%s: tm is secs=%d, mins=%d, hours=%d, "
      "mday=%d, mon=%d, year=%d, wday=%d\n",
      __FUNCTION__,
      tm->tm_sec, tm->tm_min, tm->tm_hour,
      tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_wday);
#endif	
   return 0;
}


static int x1205_set_datetime(struct rtc_time *tm,int datetoo, u8 reg_base)
{
   int i;
   unsigned char buf[8];

#ifdef __RTC_DEBUG__
	unsigned char t;
	
   printk("%s: secs=%d, mins=%d, hours=%d, "
      "mday=%d, mon=%d, year=%d, wday=%d\n",
      __FUNCTION__,
      tm->tm_sec, tm->tm_min, tm->tm_hour,
      tm->tm_mday, tm->tm_mon, tm->tm_year, tm->tm_wday);
#endif
   buf[CCR_SEC] = BIN2BCD(tm->tm_sec);
   buf[CCR_MIN] = BIN2BCD(tm->tm_min);

   /* set hour and 24hr bit */
   buf[CCR_HOUR] = BIN2BCD(tm->tm_hour) | X1205_HR_MIL;

   /* should we also set the date? */
   if (datetoo) {
      buf[CCR_MDAY] = BIN2BCD(tm->tm_mday);

      /* month, 0 - 11 */
      buf[CCR_MONTH] = BIN2BCD(tm->tm_mon+1);

      /* year, since 1900 */
      buf[CCR_YEAR] = BIN2BCD(tm->tm_year + SYS_EPOCH - EPOCH);
      buf[CCR_WDAY] = tm->tm_wday & 0x07;
      buf[CCR_Y2K] = BIN2BCD(EPOCH / 100);
   }
   
   /*unlock the chip*/
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
    
    for (i = 0; i < (datetoo ? 8 : 3); i++) {
    	mvRtcCharWrite(reg_base + i, buf[i]);
    	mvRtcCharWrite(X1205_REG_SR,0);	
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
    }	
#ifdef __RTC_DEBUG__    
    mvRtcCharRead(X1205_REG_SR,&t);
	printk("Status Register: %x\n",t);
#endif	    
    /*lock the chip*/
    mdelay(1000);
    mvRtcCharWrite(X1205_REG_SR,0);	
    
    	
   return 0;
}

static int x1205_diable_alarm(u8 reg_base)
{
	int i;
	unsigned char buf[8]={0},t;
	
   buf[CCR_MIN] = 0;
   /* set hour and 24hr bit */
   buf[CCR_HOUR] = 0;
   buf[CCR_WDAY] = 0;
   
    for (i = 0; i < 7; i++) {
    	mvRtcCharRead(X1205_REG_SR,&t);/* it seems that we must read the status register first for writing alarm register*/
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
    	mvRtcCharWrite(reg_base + i, buf[i]);
    }	
    mdelay(1000);
    mvRtcCharRead(X1205_REG_SR,&t);
#ifdef __RTC_DEBUG__    
	printk("read status :%x\n",t);
#endif	
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);		
    mvRtcCharWrite(X1205_REG_INT,0x00);
    mdelay(1000);

   return 0;
}


static int x1205_set_alarm2(struct rtc_time *tm, u8 reg_base)
{
	int i;
	unsigned char buf[8]={0},t;
	
#ifdef __RTC_DEBUG__
   printk("%s: wday=%d,  mins=%d, hours=%d "  , __FUNCTION__, tm->tm_wday,
       tm->tm_min, tm->tm_hour
      );
#endif

   buf[CCR_MIN] = BIN2BCD(tm->tm_min )| X1205_ALARM_EN;

   /* set hour and 24hr bit */
   buf[CCR_HOUR] = BIN2BCD(tm->tm_hour) | X1205_ALARM_EN;
   buf[CCR_WDAY] = (tm->tm_wday & 0x07)|X1205_ALARM_EN;
   
    for (i = 0; i < 6; i++) {
    	mvRtcCharRead(X1205_REG_SR,&t);/* it seems that we must read the status register first for writing alarm register*/
#ifdef __RTC_DEBUG__    	
    	printk("read status :%x\n",t);
#endif    	
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
    	mvRtcCharWrite(reg_base + i, buf[i]);
    }	
    mdelay(1000);
    //set day of week
    mvRtcCharRead(X1205_REG_SR,&t);
#ifdef __RTC_DEBUG__    
    printk("read status :%x\n",t);
#endif  
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
    mvRtcCharWrite(reg_base + CCR_WDAY, buf[CCR_WDAY]);
    mdelay(1000);
    mvRtcCharRead(X1205_REG_SR,&t);
#ifdef __RTC_DEBUG__    
	printk("read status :%x\n",t);
#endif	
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);		
    mvRtcCharWrite(X1205_REG_INT,0x20);
   // mvRtcCharWrite(X1205_REG_SR,0);	
    

   return 0;
}


int rtc_open(struct inode *minode, struct file *mfile)
{
	if(x1205_use_count > 0) {
		return -EBUSY;
	}
	
	++x1205_use_count;
	return 0;
}

int rtc_release(struct inode *minode, struct file *mfile)
{
	--x1205_use_count;
	return 0;
}

static loff_t rtc_llseek(struct file *mfile, loff_t offset, int origint)
{
	return -ESPIPE;
}
int clr_status(void){
	
//	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
//	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);		
//	mvRtcCharWrite(X1205_REG_INT,0x0);
	return 0;
}	

static int rtc_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		     unsigned long arg)
{
	struct rtc_time rtc_tm;
	struct rtc_time alarm_tm;

	switch (cmd) {
	case RTC_RD_TIME:	/* Read the time/date from RTC  */
		x1205_get_datetime( &rtc_tm, X1205_CCR_BASE);
		return copy_to_user((void *)arg, &rtc_tm, sizeof(rtc_tm)) ? 
			-EFAULT : 0;
	case RTC_SET_TIME:	/* Set the RTC */
		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		if (copy_from_user(&rtc_tm, 
				   (struct rtc_time *) arg,
		                   sizeof(struct rtc_time))) 
			return -EFAULT;
			
		return 	x1205_set_datetime( &rtc_tm,1,X1205_CCR_BASE);
		
	case RTC_ALM_SET: /*set the alarm time*/
		if (!capable(CAP_SYS_TIME)){
#ifdef __RTC_DEBUG__			
			printk("capable() fail\n");
#endif			
			return -EACCES;
		}

		if (copy_from_user(&alarm_tm, 
				   (struct rtc_time *) arg,
		                   sizeof(struct rtc_time))){
#ifdef __RTC_DEBUG__		                   	
			printk("copy_from_user() fail\n");
#endif			
			return -EFAULT;
		}
		return x1205_set_alarm2( &alarm_tm,X1205_ALM0_BASE);
	case RTC_AIE_OFF:
		if (!capable(CAP_SYS_TIME))
			return -EACCES;
		return x1205_diable_alarm(X1205_ALM0_BASE);
				
	default:
		return -EINVAL;
	}
}


static struct file_operations rtc_fops = {
	owner:		THIS_MODULE,
	llseek:		rtc_llseek,
	ioctl:		rtc_ioctl,
	open:		rtc_open,
	release:	rtc_release,
};

static struct miscdevice x1205rtc_miscdev = {
	RTC_MINOR,
	"rtc",
	&rtc_fops
};
static int rtc_proc_output(char *buf)
{
	char *p;
	struct rtc_time tm;
	int ret;
	
	if ((ret = x1205_get_datetime( &tm, X1205_CCR_BASE)) < 0)
		return ret;

	p = buf;

	/*
	 * There is no way to tell if the luser has the RTC set for local
	 * time or for Universal Standard Time (GMT). Probably local though.
	 */
	p += sprintf(p,
		     "rtc_time\t: %02d:%02d:%02d\n"
		     "rtc_date\t: %04d-%02d-%02d\n"
		     "rtc_epoch\t: %04d\n",
		     tm.tm_hour, tm.tm_min, tm.tm_sec,
		     tm.tm_year + SYS_EPOCH, tm.tm_mon + 1,
		     tm.tm_mday, EPOCH);

	return p - buf;
}

static int rtc_read_proc(char *page, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int len = rtc_proc_output(page);
	if (len <= off + count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

static int __init  x1205_init(void)
{
	int ret;
	unsigned char  outvalue;
	
	printk("X1205 RTC driver.\n");
	mvRtcCharRead(X1205_REG_SR,&outvalue);
#if 0 //Enable Alarm Interrupt?	
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
    mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);	
	mvRtcCharWrite(X1205_REG_INT,0x20);
    mvRtcCharWrite(X1205_REG_SR,0);
#endif	

	/* adjust rtc,slow down about 6 seconds one day*/
	mvRtcCharRead(X1205_REG_SR,&outvalue);	   
	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
	mvRtcCharWrite(X1205_REG_DTR,0x07);/*-30ppm*/
	mdelay(1000);
	mvRtcCharRead(X1205_REG_SR,&outvalue);
	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL);
	mvRtcCharWrite(X1205_REG_SR,X1205_SR_WEL | X1205_SR_RWEL);
	mvRtcCharWrite(X1205_REG_ATR,0x3f);/*-37ppm*/

	ret = misc_register(&x1205rtc_miscdev);
	if (ret) {
		printk("Register misc driver failed, errno is %d\n", ret);
		return ret;
	}
	printk(":--- Register RTC driver OK.\n");
	create_proc_read_entry("driver/rtc", 0, 0, rtc_read_proc, NULL);

	return 0;
}

static void __exit x1205_exit(void)
{
	remove_proc_entry("driver/rtc", NULL);
	misc_deregister(&x1205rtc_miscdev);
}


module_init(x1205_init);
module_exit(x1205_exit);

/*
 *	Info exported via "/proc/driver/rtc".
 */


MODULE_LICENSE("GPL");
