/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell 
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.


********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File in accordance with the terms and conditions of the General 
Public License Version 2, June 1991 (the "GPL License"), a copy of which is 
available along with the File in the license.txt file or by writing to the Free 
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or 
on the worldwide web at http://www.gnu.org/licenses/gpl.txt. 

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED 
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY 
DISCLAIMED.  The GPL License provides additional details about this warranty 
disclaimer.
*******************************************************************************/
/* 
* 
* DESCRIPTION:
*		Ethernet driver for UniMAC
*
* DEPENDENCIES:   
*
* FILE REVISION NUMBER:
*
*******************************************************************************/
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/config.h>
#include <linux/init.h>
#include <linux/sched.h>
#include <linux/ptrace.h>
#include <linux/fcntl.h>
#include <linux/ioport.h>
#include <linux/interrupt.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/errno.h>
#include <linux/ip.h>
#include <linux/mii.h>

#include <asm/bitops.h>
#include <asm/io.h>
#include <asm/types.h>
#include <asm/pgtable.h>
#include <asm/system.h>

#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/ethtool.h>
#include <linux/skbuff.h>
#include <linux/ctype.h>

#include "mvCtrlEnvLib.h"
#include "mvOs.h"
#include "mvSysHwConfig.h"
#include "mv_eth_regs.h"

#include "mv_unimac.h"
#include "mv_eth.h"
#include "mv_eth_addrtbl.h"
#include "mv_unm_netconf.h"

#include "msApiDefs.h"
#include "mv_unimac.h"
#include "qdModule.h"
#include "mv_unm_netconf.h"
#define MIN_ETH_PACKET_LEN 60
#define HEADER_SIZE        2
#define TRAILER_SIZE       4

#define MV_INT_UNIMAC	21
#define ASSERT	mvOsAssert
extern GT_QD_DEV* qd_dev;
int if_stopped = 0;

#undef MC_BRIDGE_EXAMPLE /* Use MC bridge example */


/***************************************************/
/* Definitions                                     */
/***************************************************/

#define BIT(n) (1<<(n))

#if !defined(HEADERS) && !defined(TRAILERS)
	#error "You must define headers or trailers!!!"
#endif

#if defined(HEADERS) && defined(TRAILERS)
	#error "You must define headers or trailers not both!!!"
#endif


/***************************************************/
/* Gloabl vars                                     */
/***************************************************/
extern u32 overEthAddr;
extern u8	mvMacAddr[6];

/* The _binding structs represents per vlan binding 
 * the binding is between the ethernet device (struct net_device)
 * and its fields: HW access, link state, dev state and trailer/header
 */ 
typedef struct _binding
{
	struct net_device *dev;		  /* the device associated with this binding entry */
	mv_eth_hw         *pmvEthHw;  /* the global HW HAL */
	bool               boLinkUp;  /* link state */
	bool               boOpened;  /* device has allready been opened by the stack (hasn't been closed yet) */
#ifdef HEADERS
	char               header[HEADER_SIZE];
#elif defined (TRAILERS)
	char               trailer[TRAILER_SIZE];
#endif
} BINDING, *PBINDING;

/* the irq number assigned to our MAC HW by the kernel */
static u32  irq;	   /* irq of our shared MAC HW */

/*
 * The static global array of bindings.
 * each vlan has a binding that represents it
 * the index in the array is the binding for it
 * we are allocating the number of needed bindings in 
 * mv_eth_start
 */ 
static BINDING * mvBindings[MV_UNM_VID_ISOLATED]= {0};

/* a global spin lock to sync transmits */
spinlock_t lock;

/* the sole HW MAC presentation is SW */
static mv_eth_hw   mvEthHw;

/* In some implemantation we use the broadcast MAC to update HW tables */
static unsigned char brdcast_mac[GT_ETHERNET_HEADER_SIZE] = {0xff, 0xff, 0xff, 0xff, 0xff, 0xff}; 

#ifdef MC_BRIDGE_EXAMPLE
/* a few mcast addresses for example */
static unsigned char multicast_mac1[GT_ETHERNET_HEADER_SIZE] = {0x01, 0x00, 0x5E, 0x00, 0x00, 0x02}; 
static unsigned char multicast_mac2[GT_ETHERNET_HEADER_SIZE] = {0x01, 0x00, 0x5E, 0x00, 0x00, 0x03}; 
static unsigned char multicast_mac3[GT_ETHERNET_HEADER_SIZE] = {0x01, 0x00, 0x5E, 0x00, 0x00, 0x04}; 
#endif

#ifdef USE_NAPI
static int mv_eth_rx_poll(struct net_device *dev, int *budget);
#endif

static int mv_eth_receive_queue(unsigned long data);
static int mv_eth_free_tx_queue(unsigned long queue);

#ifndef USE_NAPI
DECLARE_WORK(rxComplete, mv_eth_receive_queue, Q_INDEX_LIMIT);
DECLARE_WORK(txComplete, mv_eth_free_tx_queue, MV_ETH_TX_QUEUE);
#endif

extern void qdStatus(void);



/*********************************************************** 
 * string helpers for mac address setting                  *
 ***********************************************************/
static unsigned int mv_eth_str_to_hex( char ch ) 
{
	if ( (ch >= '0') && (ch <= '9') )
		return( ch - '0' );

	if ( (ch >= 'a') && (ch <= 'f') )
		return( ch - 'a' + 10 );

	if ( (ch >= 'A') && (ch <= 'F') )
		return( ch - 'A' + 10 );

	return 0;
}

static void mv_eth_convert_str_to_mac( char *source , char *dest ) 
{
    dest[0] = (mv_eth_str_to_hex( source[0] ) << 4) + mv_eth_str_to_hex( source[1] );
    dest[1] = (mv_eth_str_to_hex( source[2] ) << 4) + mv_eth_str_to_hex( source[3] );
    dest[2] = (mv_eth_str_to_hex( source[4] ) << 4) + mv_eth_str_to_hex( source[5] );
    dest[3] = (mv_eth_str_to_hex( source[6] ) << 4) + mv_eth_str_to_hex( source[7] );
    dest[4] = (mv_eth_str_to_hex( source[8] ) << 4) + mv_eth_str_to_hex( source[9] );
    dest[5] = (mv_eth_str_to_hex( source[10] ) << 4) + mv_eth_str_to_hex( source[11] );
}



/*
 * ----------------------------------------------------------------------------
 * Set the MAC address for the specified interface
 * to the specified value, forsaking all others.
 *
 * Input : pointer to ethernet interface network device structure and
 *         a pointer to the designated entry to be added to the cache.
 * Output : zero upon success, negative upon failure
 */
s32
mv_eth_set_mac_address(struct net_device *dev, void *addr)
{
	u32 i;
	struct sockaddr *sock;
	u8 port = ((gt_eth_priv *) (dev->priv))->port;

#ifdef ETH_DBG_TRACE
	printk("mv_eth_set_mac_address\n");
#endif

	sock = (struct sockaddr *) addr;
	for (i = 0; i < 6; i++)
	{
		dev->dev_addr[i] = sock->sa_data[i];
	}
#ifdef ETH_DBG_INFO
	printk("Setting MAC address for dev [%d]\n",((gt_eth_priv *) (dev->priv))->vid);
#endif
	/* addressTableClear(port); */	  /* Does flush */
	mv_eth_update_mac_address(dev->dev_addr, port );
	return(0);
}

/*****************************************************************************
*
* int etherReadMIIReg (unsigned int portNumber , unsigned int MIIReg,
* unsigned int* value)
*
* Description
* This function will access the MII registers and will read the value of
* the MII register , and will retrieve the value in the pointer.
* Inputs
* portNumber - one of the 2 possiable Ethernet ports (0-1).
* MIIReg - the MII register offset.
* Outputs
* value - pointer to unsigned int which will receive the value.
* Returns Value
* 1 if success.
* 0 if fail to make the assignment.
* Error types (and exceptions if exist)
*/
static int etherReadMIIReg(unsigned int portNum, unsigned int miiReg,
						   unsigned int *value);

/*****************************************************************************
* 
* int etherWriteMIIReg (unsigned int portNumber , unsigned int MIIReg,
* unsigned int value)
* 
* Description
* This function will access the MII registers and will write the value
* to the MII register.
* Inputs
* portNumber - one of the 2 possiable Ethernet ports (0-1).
* MIIReg - the MII register offset.
* value -the value that will be written.
* Outputs
* Returns Value
* 1 if success.
* 0 if fail to make the assignment.
* Error types (and exceptions if exist)
*/
static int etherWriteMIIReg(unsigned int portNum, unsigned int miiReg,
							unsigned int value);



/* set port config value */
void set_port_config(u32 value, unsigned int port) {
	MV_REG_WRITE(MV_ETH_PCR + ETHERNET_PORTS_DIFFERENCE_OFFSETS * port, value);
}

/* get port config value */
u32 get_port_config(unsigned int port) {
	return MV_REG_READ(MV_ETH_PCR + ETHERNET_PORTS_DIFFERENCE_OFFSETS * port);
}


void enableFiltering(u32 port)
{
	ETHERNET_PCR pcr;

	pcr = get_port_config(port);
	pcr &= ~0x1; /* disable promiscuous unicast mode */
	set_port_config(pcr, port);
}

void disableFiltering(u32 port)
{
	ETHERNET_PCR pcr;

	pcr = get_port_config(port);
	pcr |= 0x1;	/* enable promiscuous unicast mode */
#ifdef ENABLE_MC
	pcr |= 0x8;	/* enable promiscuous multicast mode */
#endif
	set_port_config(pcr, port);
}


static inline void disableMulticastFiltering(u32 port)
{
	ETHERNET_PCR pcr;

	pcr = get_port_config(port);
	pcr |= 0x8;	/* enable promiscuous multicast mode */
	set_port_config(pcr, port);
}


/* Sometimes it seems the phy and the GT don't always agree. 
 * Make sure they do, or restart autoneg. */
static void
check_phy_state(struct net_device *dev)
{
	gt_eth_priv *private = (gt_eth_priv *) dev->priv;
	struct mii_if_info *mii = &private->mii_if;
	int bmsr = mii->mdio_read(mii->dev, mii->phy_id, MII_BMSR);

	if (mii_link_ok(&private->mii_if) && (bmsr & BMSR_ANEGCOMPLETE))
	{
		int advert = mii->mdio_read(dev, mii->phy_id, MII_ADVERTISE);
		int lpa = mii->mdio_read(dev, mii->phy_id, MII_LPA);
		int nego = mii_nway_result(advert & lpa);
		int psr, wanted = -1;

		switch (nego)
		{
		case LPA_100FULL:
			wanted = 0x3;
			break;
		case LPA_100HALF:
			wanted = 0x1;
			break;
		case LPA_10FULL:
			wanted = 0x2;
			break;
		case LPA_10HALF:
			wanted = 0x0;
			break;
		default:
#ifdef ETH_DBG_WARN
			printk("%s: MII negotiated strange settings %d\n",dev->name, nego);
#endif
			break;
		}
		psr = MV_REG_READ(MV_ETH_PSR + (ETH_ADDR_GAP * private->port));

		if ((psr & 0x3) != wanted)
		{
#ifdef ETH_DBG_ERROR
			printk("%s: MII said %x, GT said %x, restarting autoneg\n", dev->name, wanted, psr);
#endif
			mii_nway_restart(mii);
		}
	}
}

/*
 * dump_link_state - prints the link state
 * it informs: dev->name, speed, link, duplex, flow-control
 */
#ifdef ETH_DBG_INFO
static void dump_link_state(struct net_device *dev)
{
	gt_eth_priv *private = (gt_eth_priv *) dev->priv;
	unsigned int psr;

	psr = MV_REG_READ(MV_ETH_PSR + (ETH_ADDR_GAP * private->port));
	printk("%s: link state:\n"
		   "  GT:%s:%s:%s:%s\n",
		   dev->name,
		   psr & 1 ? "100" : " 10",
		   psr & 8 ? " Link" : "nLink",
		   psr & 2 ? "FD" : "HD", psr & 4 ? " FC" : "nFC");

}
#endif /* ETH_DBG_INFO */

/*
 * ----------------------------------------------------------------------------
 *  Update the statistics structure in the private data structure
 *
 *  Input : pointer to ethernet interface network device structure
 *  Output : N/A
 */
void
mv_eth_update_stat(struct net_device *dev)
{
	gt_eth_priv             *private_data;
	struct net_device_stats *stat;
	static int              first_time = 1;
#if 0
	u32 base    =   MV_ENET_0_MIB_CTRS;
#endif

#ifdef ETH_DBG_TRACE
	printk("mv_eth_update_stat\n");
#endif

	private_data =   dev->priv;
	stat    = &(private_data->stat);

	if (first_time)
	{
		stat->rx_bytes = 0;
		stat->tx_bytes = 0;
		stat->rx_packets = 0;
		stat->tx_packets = 0;
		stat->rx_errors = 0;
		stat->rx_dropped = 0;
		stat->multicast = 0;
		stat->collisions = 0;
		stat->rx_length_errors = 0;
		stat->rx_length_errors = 0;
		stat->rx_crc_errors = 0;
#if 0

		/* Clear chips MIB counters */
		MV_REG_READ( base + 0x00 );
		MV_REG_READ( base + 0x04 );
		MV_REG_READ( base + 0x08 );
		MV_REG_READ( base + 0x0c );
		MV_REG_READ( base + 0x50 );
		MV_REG_READ( base + 0x20 );
		MV_REG_READ( base + 0x1c );
		MV_REG_READ( base + 0x30 );
		MV_REG_READ( base + 0x60 );
		MV_REG_READ( base + 0x24 );
		MV_REG_READ( base + 0x20 );
#endif
		first_time = 0;
	}
#if 0
	stat->rx_bytes         += MV_REG_READ( base + 0x00 );
	stat->tx_bytes         += MV_REG_READ( base + 0x04 );
	stat->rx_packets       += MV_REG_READ( base + 0x08 );
	stat->tx_packets       += MV_REG_READ( base + 0x0c );
	stat->rx_errors        += MV_REG_READ( base + 0x50 );

	/*
	 * Rx dropped is for received packet with CRC error
	 */
	stat->rx_dropped       += MV_REG_READ( base + 0x20 );
	stat->multicast        += MV_REG_READ( base + 0x1c );
	stat->collisions       += MV_REG_READ( base + 0x30 );

	/*
	 * detailed rx errors
	 */
	stat->rx_length_errors += MV_REG_READ( base + 0x60 );
	stat->rx_length_errors += MV_REG_READ( base + 0x24 );
	stat->rx_crc_errors    += MV_REG_READ( base + 0x20 );
#endif
}

/*
 * ----------------------------------------------------------------------------
 * Returns a pointer to the interface statistics.
 *
 * Input : dev - a pointer to the required interface
 *
 * Output : a pointer to the interface's statistics
 */
struct net_device_stats *
mv_eth_get_stats(struct net_device *dev)
{
	gt_eth_priv* priv = (gt_eth_priv*)(dev->priv); 

#ifdef ETH_DBG_TRACE
	printk("mv_eth_get_stats\n");
#endif

	mv_eth_update_stat(dev);
	return &priv->stat;
}

/*
 * ----------------------------------------------------------------------------
 *  change rx mode
 *
 *  Input : pointer to ethernet interface network device structure
 *  Output : N/A
 */
void
mv_eth_set_rx_mode(struct net_device *dev)
{
	gt_eth_priv *private;
	int i;
	struct dev_mc_list *mcList = dev->mc_list;
	char* addr;

#ifdef ETH_DBG_TRACE
	printk("mv_eth_set_rx_mode\n");
#endif

	private = dev->priv;

	if (dev->flags & IFF_PROMISC)
	{
		disableFiltering(private->port);
#ifdef ETH_DBG_TRACE
		printk ("Promiscuous mode is on.\n");
#endif
	} else if (dev->flags & IFF_ALLMULTI)
	{
		disableMulticastFiltering(private->port);
#ifdef ETH_DBG_MC
		printk ("All Multicast mode is on.\n");
#endif
	} else
	{
#ifdef ENABLE_MC
		disableFiltering(private->port);
#ifdef ETH_DBG_MC
		printk ("Multicast Addresses (%i)\n", dev->mc_count);
#endif
		for (i = 0; i < dev->mc_count; i++)
		{
			addr = mcList->dmi_addr;
#ifdef ETH_DBG_MC
			printk("%02x:%02x:%02x:%02x:%02x:%02x\n",
				   addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
#endif
			mcList = mcList->next;
		}
#else /* !ENABLE_MC */
		enableFiltering(private->port);
#ifdef ETH_DBG_MC
		printk ("Multicast Addresses (%i)\n", dev->mc_count);
#endif
		for (i = 0; i < dev->mc_count; i++)
		{
			addr = mcList->dmi_addr;
			mv_eth_update_mac_address(addr, private->port);
#ifdef ETH_DBG_MC
			printk("%02x:%02x:%02x:%02x:%02x:%02x\n",
				   addr[0], addr[1], addr[2], addr[3], addr[4], addr[5]);
#endif
			mcList = mcList->next;
		}
#ifdef MC_BRIDGE_EXAMPLE
		mv_eth_update_mac_address(multicast_mac1, private->port);
		mv_eth_update_mac_address(multicast_mac2, private->port);
		mv_eth_update_mac_address(multicast_mac3, private->port);
		printk("Multicast Addresses (01-00-5e-00-00-02,-03,-04) are added\n");    
#endif
#endif /* ENABLE_MC */
	}
}

static int
mdio_read(struct net_device *dev, int phy_id, int location)
{
	unsigned int val;

	etherReadMIIReg(phy_id & 0x1f, location & 0x1f, &val);

	return(val & 0xffff);
}

static void
mdio_write(struct net_device *dev, int phy_id, int location, int value)
{
	etherWriteMIIReg(phy_id & 0x1f, location & 0x1f, value);
}

#ifdef ETHERTOOL_FIXED
static int
netdev_ethtool_ioctl(struct net_device *dev, void *useraddr)
{
	gt_eth_priv *private = (gt_eth_priv *) dev->priv;
	u32 ethcmd;

	if (copy_from_user(&ethcmd, useraddr, sizeof (ethcmd)))
		return -EFAULT;

	switch (ethcmd)
	{

	/* Get driver info */
	case ETHTOOL_GDRVINFO:{
			struct ethtool_drvinfo info = { ETHTOOL_GDRVINFO};
			strncpy(info.driver, "unimac",
					sizeof (info.driver) - 1);
			strncpy(info.version, version,
					sizeof (info.version) - 1);
			if (copy_to_user(useraddr, &info, sizeof (info)))
				return -EFAULT;
			return 0;
		}
		/* get settings */
	case ETHTOOL_GSET:{
			struct ethtool_cmd ecmd = { ETHTOOL_GSET};
			spin_lock_irq(&private->lock);
			mii_ethtool_gset(&private->mii_if, &ecmd);
			spin_unlock_irq(&private->lock);
			if (copy_to_user(useraddr, &ecmd, sizeof (ecmd)))
				return -EFAULT;
			return 0;
		}
		/* set settings */
	case ETHTOOL_SSET:{
			int r;
			struct ethtool_cmd ecmd;
			if (copy_from_user(&ecmd, useraddr, sizeof (ecmd)))
				return -EFAULT;
			spin_lock_irq(&private->lock);
			r = mii_ethtool_sset(&private->mii_if, &ecmd);
			spin_unlock_irq(&private->lock);
			return r;
		}
		/* restart autonegotiation */
	case ETHTOOL_NWAY_RST:{
			return mii_nway_restart(&private->mii_if);
		}
		/* get link status */
	case ETHTOOL_GLINK:{
			struct ethtool_value edata = { ETHTOOL_GLINK};
			edata.data = mii_link_ok(&private->mii_if);
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}
		/* get message-level */
	case ETHTOOL_GMSGLVL:{
			struct ethtool_value edata = { ETHTOOL_GMSGLVL};
			edata.data = 0;	/* XXX */
			if (copy_to_user(useraddr, &edata, sizeof (edata)))
				return -EFAULT;
			return 0;
		}
		/* set message-level */
	case ETHTOOL_SMSGLVL:{
			struct ethtool_value edata;
			if (copy_from_user(&edata, useraddr, sizeof (edata)))
				return -EFAULT;
			/* debug = edata.data; */
			return 0;
		}
	}
	return -EOPNOTSUPP;
}
#endif

static int
mv_eth_ioctl(struct net_device *dev, struct ifreq *rq, int cmd)
{
	struct mii_ioctl_data *data = (struct mii_ioctl_data *) &rq->ifr_data;
	int phy = dev->base_addr & 0x1f;
	int retval = -EOPNOTSUPP;

	switch (cmd)
	{
	case SIOCETHTOOL:
#ifdef ETHERTOOL_FIXED
		retval = netdev_ethtool_ioctl(dev, (void *) rq->ifr_data);
#endif
		break;

	case SIOCGMIIPHY: /* Get address of MII PHY in use. */
	case SIOCDEVPRIVATE:  /* for binary compat, remove in 2.5 */
		data->phy_id = phy;
		/* Fall through */

	case SIOCGMIIREG: /* Read MII PHY register. */
	case SIOCDEVPRIVATE + 1:  /* for binary compat, remove in 2.5 */
		data->val_out =
		mdio_read(dev, data->phy_id & 0x1f, data->reg_num & 0x1f);
		retval = 0;
		break;

	case SIOCSMIIREG: /* Write MII PHY register. */
	case SIOCDEVPRIVATE + 2:  /* for binary compat, remove in 2.5 */
		if (!capable(CAP_NET_ADMIN))
		{
			retval = -EPERM;
		} else
		{
			mdio_write(dev, data->phy_id & 0x1f,
					   data->reg_num & 0x1f, data->val_in);
			retval = 0;
		}
		break;

	default:
		retval = -EOPNOTSUPP;
		break;
	}
	return retval;
}

/*
 * ----------------------------------------------------------------------------
 * Initializes the ethernet interface's private structure.
 * Statistics, descriptors, etc...
 *
 * Input : pointer to network device structure to be filled
 * Output : N/A
 */
void
mv_eth_init_priv(struct net_device *dev)
{
	gt_eth_priv *private;

#ifdef ETH_DBG_TRACE
	printk("mv_eth_init_priv\n");
#endif


	private = (gt_eth_priv *) kmalloc(sizeof (*private), GFP_KERNEL);
	if (!private)
	{
		panic("mv_eth_init_priv : kmalloc failed\n");
	}
	dev->priv = (void *) private;
	memset(private, 0, sizeof (*(private)));


	private->mii_if.dev = dev;
	private->mii_if.phy_id = dev->base_addr;
	private->mii_if.mdio_read = mdio_read;
	private->mii_if.mdio_write = mdio_write;
	private->mii_if.advertising = mdio_read(dev, dev->base_addr, MII_ADVERTISE);

	spin_lock_init(&private->lock);
	private->inited = false;

#ifdef ETH_DBG_INFO
	printk("Successful init of priv\n");
#endif
}

/*
 * ----------------------------------------------------------------------------
 * Currently a no-op.  According to previous commentary, the hardware
 * "cannot [!] be stuck because it is a built-in hardware - if we reach
 * here the ethernet port might be [physically] disconnected..."
 */
void
mv_eth_tx_timeout(struct net_device *dev)
{
#ifdef ETH_DBG_PKT
	printk("mv_eth_tx_timeout\n");
#endif

#ifdef ETH_DBG_WARN
	u32 cause = MV_REG_READ(MV_ETH_ICR );
	u32 mask = MV_REG_READ(MV_ETH_IMR );
	printk("mv_eth_tx_timeout: cause =%x, mask=%x\n", cause, mask);
#endif /* ETH_DBG_WARN */
}

/*
 * ----------------------------------------------------------------------------
 *  First function called after registering the network device.
 *  It's purpose is to initialize the device as an ethernet device,
 *  fill the structure that was given in registration with pointers
 *  to functions, and setting the MAC address of the interface
 *
 *  Input : pointer to network device structure to be filled
 *  Output : -ENONMEM if failed, 0 if success
 */


s32
mv_eth_init(struct net_device * dev)
{
	int iNumOfVlans, i;
	u32 queue;
	gt_eth_priv *privateData;
	bool boFound = false;
	BINDING *pBinding = NULL;
	int vid = 0;
	int size;
	char *namebuf;
	dma_addr_t phyAddr;

#ifdef ETH_DBG_TRACE
	printk("mv_eth_init\n");
#endif

	/* find location in device array */
	iNumOfVlans = mvUnmGetNumOfVlans();
	for (i = 1; i <= iNumOfVlans; i++)
	{
		if (mvBindings[i] && (mvBindings[i]->dev == dev))
		{
			/* found - initialize it */
			boFound = true;
			pBinding = mvBindings[i];
			vid = i;
			break;
		}
	}

	if (!boFound || !pBinding)
	{
		panic("mv_eth_init: can't find my own device\n");
	}
	if ((vid == 0) || (vid > mvUnmGetNumOfVlans()))
	{
		panic("mv_eth_init: init_etherdev dev->vid error\n");
	}

	mv_eth_init_priv(dev);
	privateData = (gt_eth_priv*)(dev->priv);
	privateData->vid = vid; 

	/* get the interface names from netconf */
	mv_nc_GetVIDName(vid, &size, &namebuf);

	printk("dev[%d]->name = %s\n", vid, dev->name);
	dev->open = mv_eth_open;
	dev->stop = mv_eth_stop;
	dev->set_config = NULL;	  /* no runtime config support for now */
	dev->hard_start_xmit = mv_eth_start_xmit;
	dev->do_ioctl = mv_eth_ioctl;
	dev->get_stats = mv_eth_get_stats;
	dev->set_mac_address = mv_eth_set_mac_address;
	dev->set_multicast_list = mv_eth_set_rx_mode;
	dev->tx_timeout = mv_eth_tx_timeout;  /* Currently no-op */
	dev->watchdog_timeo = 5 * HZ;	
	dev->flags &= ~IFF_RUNNING;
#ifdef USE_NAPI
	dev->poll = &mv_eth_rx_poll;
	dev->weight = Q_INDEX_LIMIT;
#endif	
	if (!privateData->inited)
	{
		//  memcpy(dev->dev_addr, mv_nc_GetMacOfVlan(vid), GT_ETHERNET_HEADER_SIZE);
		privateData->port = 0;
		privateData->inited = true;
	}

	if (mvEthHw.hwState == HW_UNKNOWN)
	{
		MV_REG_WRITE(MV_ETH_SDCR,
					 (0x3 << 12) | (1 << 9)	/*| (0xf << 2)*/
					 | (1 << 6) | (1 << 7)); /*  0000.203c */

		/* enable */
		MV_REG_WRITE(MV_ETH_PCR, MV_REG_READ(MV_ETH_PCR) | (1 << 7));

		/* no promisc */
		MV_REG_WRITE(MV_ETH_PCR, MV_REG_READ(MV_ETH_PCR) & ~(1 << 0));

		/* set full duplex  */
		MV_REG_WRITE(MV_ETH_PCR, MV_REG_READ(MV_ETH_PCR) | (1 << 15));

#ifdef HEADERS
		MV_REG_WRITE(MV_ETH_PCXR,
					 ((1<<22) | (1<<19) | (1<<18) /*| (1<<16)*/
					  | (1<<14) | (1<<10) | (1<<9) /* |(1<<8)*/));
#else
		MV_REG_WRITE(MV_ETH_PCXR, (/*(1<<16) | */(1<<14) | (1<<8)));
#endif

#ifdef ETH_DBG_TRACE
		printk(" PCR = %x PCXR %x SDMA Conf %x\n", MV_REG_READ(MV_ETH_PCR) , 
			   MV_REG_READ(MV_ETH_PCXR),MV_REG_READ(MV_ETH_SDCR) );
#endif			   
		/*
		 * The UniMAC Enet engine accesses its queues of DMA
		 * descriptors in uncached mode, so we dedicate a page
		 * for each queue and mark each such page uncached.
		 */
		for (queue = 0; queue < NUM_TX_QUEUES; queue++)
		{
			ASSERT(mvEthHw.TXqueue[queue] ==  NULL);
			if (queue == MV_ETH_TX_QUEUE)
			{				
#ifdef DESCRIPTORS_IN_CACHED_MEM
				mvEthHw.TXqueue[queue] = (gt_dma_desc*)mvOsIoCachedMalloc(NULL, PAGE_SIZE, &phyAddr);
#else				
				mvEthHw.TXqueue[queue] = (gt_dma_desc*)mvOsIoUncachedMalloc(NULL, PAGE_SIZE, &phyAddr);
				mvEthHw.physTXqueue[queue] = phyAddr;
#ifdef ETH_DBG_TRACE
				printk("mvEthHw.TXqueue[queue] %x Txqueue %p ,mvOsIoVirtToPhy %x Txqueue phyAddr %x\n", 
						mvEthHw.TXqueue[queue],
					   	mvEthHw.TXqueue[0],  
						mvOsIoVirtToPhy(NULL, (void *)mvEthHw.TXqueue[0]),phyAddr);
#endif

#endif
				if (mvEthHw.TXqueue[queue] == NULL)
				{
					panic("Can't allocate memory for descriptors\n");
				}
			} 
			else
			{
				mvEthHw.TXqueue[queue] = (gt_dma_desc*)0;				
			}
		}

		for (queue = 0; queue < NUM_RX_QUEUES; queue++)
		{
			ASSERT(mvEthHw.RXqueue[queue] == NULL);
			if (queue == MV_ETH_RX_QUEUE)
			{
#ifdef DESCRIPTORS_IN_CACHED_MEM
				mvEthHw.RXqueue[queue] = (gt_dma_desc*)mvOsIoCachedMalloc(NULL, PAGE_SIZE, &phyAddr);
#else
				mvEthHw.RXqueue[queue] = (gt_dma_desc*)mvOsIoUncachedMalloc(NULL, PAGE_SIZE, &phyAddr);
				mvEthHw.physRXqueue[queue] = phyAddr;
#endif
				if (mvEthHw.RXqueue[queue] == NULL)
				{
					panic("Can't allocate memory for descriptors\n");
				}
			} else
			{
				mvEthHw.RXqueue[queue] = (gt_dma_desc*)0;
			}
		}

		/* initialize address table for hash mode 0 with 1/2K size
		 * the address will be added to address table at eth_open
		 */
		/* todo - move address management to ATU tables */
		initAddressTable(privateData->port, 0, 1, 0);
		mvEthHw.hwState = HW_INIT;
	}

	/* read MIB counters in order to reset them, then zero all the
	 * stats fields in memory
	 */
	mv_eth_update_stat(dev);
	return 0;
}


/*
 * ----------------------------------------------------------------------------
 *  This function is called when opening the network device. The function
 *  should initialize all the hardware, initialize cyclic Rx/Tx
 *  descriptors chain and buffers and allocate an IRQ to the network
 *  device.
 *
 *  Input : a pointer to the network device structure
 *
 *  Output : zero if success, nonzero if fails.
 */
s32
mv_eth_open(struct net_device * dev)
{
	gt_dma_desc *desc;
	gt_eth_priv *priv;
	s32 retval;
	struct sk_buff *sk;
	u32 count;
	u32 gap = 0;
	u32 port;
	u32 port_status;
	u32 queue;
	unsigned int portsBitMask;
	GT_STATUS status;
	priv = dev->priv;
	
#ifdef ETH_DBG_TRACE
	printk("mv_eth_open: dev = %p\n", dev);
#endif

	ASSERT(mvEthHw.hwState != HW_UNKNOWN); /* already initialized by mv_eth_init */

	/* check if we need to move into HW_READY state */
	if (mvEthHw.hwState == HW_INIT)
	{
		/*
		 * Initialize the lists of Tx/Rx descriptors (as circular chains,
		 * each in its own uncached page) using the physical addresses in
		 * the "next" pointers that the Enet DMA engine expects.  The Rx
		 * descriptors also get an sk_buff pre-allocated for them and their
		 * "data" pointers set to point to the corresponding sk_buff buffer.
		 */
		for (queue = 0; queue < NUM_TX_QUEUES; queue++)
		{
			mvEthHw.TXskbIndex[queue] =  mvEthHw.TXindex[queue] = 0;
			desc = mvEthHw.TXqueue[queue];
			if (desc)
			{
#ifndef DESCRIPTORS_IN_CACHED_MEM
				u32 phys = mvEthHw.physTXqueue[queue];
#endif
			
				memset((void*)desc, 0, PAGE_SIZE); /* clean all descriptors */
				for (count = 0; count < Q_INDEX_LIMIT; count++, desc++)
				{
#ifdef DESCRIPTORS_IN_CACHED_MEM
					desc->next = mvOsIoVirtToPhy(NULL, (void*)(desc + 1));
#else
					phys += sizeof(gt_dma_desc);
					desc->next = phys;					
#endif					
					mvEthHw.TXskbuff[queue][count] = 0;
#ifdef DESCRIPTORS_IN_CACHED_MEM
					mvOsCacheLineFlushInv((MV_ULONG)(desc));
#endif
				}
				--desc;	/* link last desc to first desc, to make a ring */
#ifdef DESCRIPTORS_IN_CACHED_MEM
				desc->next = mvOsIoVirtToPhy(NULL, (void*)(mvEthHw.TXqueue[queue]));

				mvOsCacheLineFlushInv((MV_ULONG)(desc));
#else				
				desc->next = mvEthHw.physTXqueue[queue];
#endif				
			}
		}

		for (queue = 0; queue < NUM_RX_QUEUES; queue++)
		{
			mvEthHw.RXindex[queue] = 0;
			desc = mvEthHw.RXqueue[queue];
			if (desc)
			{
#ifndef DESCRIPTORS_IN_CACHED_MEM
				u32 phys = mvEthHw.physRXqueue[queue];			
#endif				
				memset((void*)desc, 0, PAGE_SIZE); /* clean all descriptors */
				for (count = 0; count < Q_INDEX_LIMIT; count++, desc++)
				{
#ifdef DESCRIPTORS_IN_CACHED_MEM
					desc->next = mvOsIoVirtToPhy(NULL, (void*)(desc + 1));
#else
					phys += sizeof(gt_dma_desc);
					desc->next = phys;
#endif					
					desc->count.rx.bufferBytes = MAX_BUFF_SIZE;
					desc->command_status = GT_ENET_DESC_OWNERSHIP | GT_ENET_DESC_INT_ENABLE;

					sk = dev_alloc_skb(MAX_BUFF_SIZE);
					desc->data = (void*)mvOsIoVirtToPhy(NULL, (void*)(sk->data));
					mvEthHw.RXskbuff[queue][count] = sk;
					mvOsCacheInvalidate(NULL, sk->data, MAX_BUFF_SIZE);
#ifdef DESCRIPTORS_IN_CACHED_MEM
					mvOsCacheLineFlushInv((MV_ULONG)(desc));
#endif

				}
				--desc;	/* link last desc to first desc, to make a ring */
#ifdef DESCRIPTORS_IN_CACHED_MEM
				desc->next = mvOsIoVirtToPhy(NULL, (void *) (mvEthHw.RXqueue[queue]));

				mvOsCacheLineFlushInv((MV_ULONG)(desc));
#else
				desc->next = (void*)mvEthHw.physRXqueue[queue];
#endif
			}
		}

		/* initialize DMA descriptor-pointer registers */
		port = priv->port;
		gap = ETH_ADDR_GAP * port;
#ifdef DESCRIPTORS_IN_CACHED_MEM
		/* Set Tx queues 0..1 */
		MV_REG_WRITE(MV_ETH_CTDP0 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.TXqueue[0]));
		MV_REG_WRITE(MV_ETH_CTDP1 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.TXqueue[1]));
#ifdef ETH_DBG_TRACE
		printk("MV_ETH_CTDP0 %x, Txqueue %p , Txqueue phyAddr %x\n", MV_REG_READ(MV_ETH_CTDP0), 
			   mvEthHw.TXqueue[0],     mvOsIoVirtToPhy(NULL, (void *)mvEthHw.TXqueue[0]));
#endif			   
		/* Set Rx queues 0..3 */
		MV_REG_WRITE(MV_ETH_FRDP0 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[0]));
		MV_REG_WRITE(MV_ETH_CRDP0 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[0]));
					 
		MV_REG_WRITE(MV_ETH_FRDP1 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[1]));
		MV_REG_WRITE(MV_ETH_CRDP1 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[1]));
					 
		MV_REG_WRITE(MV_ETH_FRDP2 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[2]));
		MV_REG_WRITE(MV_ETH_CRDP2 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[2]));
					 
		MV_REG_WRITE(MV_ETH_FRDP3 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[3]));
		MV_REG_WRITE(MV_ETH_CRDP3 + gap,
					 mvOsIoVirtToPhy(NULL, (void *)mvEthHw.RXqueue[3]));
#else
		/* Set Tx queues 0..1 */
		MV_REG_WRITE(MV_ETH_CTDP0 + gap, mvEthHw.physTXqueue[0]);
		MV_REG_WRITE(MV_ETH_CTDP1 + gap, mvEthHw.physTXqueue[1]);
#ifdef ETH_DBG_TRACE
		printk("MV_ETH_CTDP0 %x, Txqueue %p , Txqueue phyAddr %x\n", MV_REG_READ(MV_ETH_CTDP0), 
			   mvEthHw.TXqueue[0], mvEthHw.physTXqueue[0]);
#endif			   
		/* Set Rx queues 0..3 */
		MV_REG_WRITE(MV_ETH_FRDP0 + gap, mvEthHw.physRXqueue[0]);
		MV_REG_WRITE(MV_ETH_CRDP0 + gap, mvEthHw.physRXqueue[0]);
							 
		MV_REG_WRITE(MV_ETH_FRDP1 + gap, mvEthHw.physRXqueue[1]); 
		MV_REG_WRITE(MV_ETH_CRDP1 + gap, mvEthHw.physRXqueue[1]);
		
		MV_REG_WRITE(MV_ETH_FRDP2 + gap, mvEthHw.physRXqueue[2]);
		MV_REG_WRITE(MV_ETH_CRDP2 + gap, mvEthHw.physRXqueue[2]);
							 
		MV_REG_WRITE(MV_ETH_FRDP3 + gap, mvEthHw.physRXqueue[3]);
		MV_REG_WRITE(MV_ETH_CRDP3 + gap, mvEthHw.physRXqueue[3]);
#endif
		/* allocate IRQ */
		retval = request_irq(MV_INT_UNIMAC, mv_eth_int_handler,
							 SA_INTERRUPT /* | SA_SAMPLE_RANDOM */, "mv_eth", dev);
		if (!retval)
		{
			irq = MV_INT_UNIMAC;
#ifdef ETH_DBG_TRACE
			printk("mv_eth_open: assigned IRQ %d to ISR\n", irq);
#endif
		} else
		{
			panic("mv_eth_open: cannot assign IRQ number to ISR\n");
		}

		/* clear all interrupts */
		MV_REG_WRITE(MV_ETH_ICR + gap, 0x000000000);

		/* enable interrupts */
#ifdef DEBUG
		MV_REG_WRITE(MV_ETH_IMR + gap, 0x90f10109);
#else

		/* enable the following interrupts:
		 * bit 0 = Rx Buffer
		 * bit 3 = Tx Done
		 * bit 8 = Rx Error (to catch Rx ring overrun)
		 */
		MV_REG_WRITE(MV_ETH_IMR + gap, 0x90000109);
#endif

		mvEthHw.hwState = HW_READY;
	}

	/* check link status on UniMAC.
	 * no link on UniMAC indicates a HW unrecoverable error
	 */
	port_status = MV_REG_READ(MV_ETH_PSR + gap);
	if ((port_status & 0x8) == 0)
	{
		panic("mv_eth_open: no link on UniMAC\n");
	}

	/* check phy state */
	check_phy_state(dev);
#ifdef ETH_DBG_INFO
	dump_link_state(dev);
#endif

	//TODO remove this
#ifdef ETH_DBG_TRACE
	printk("mv_eth_open: eth%d MAC address to %02x:%02x:%02x:%02x:%02x:%02x\n",
		priv->port,
		dev->dev_addr[0],
		dev->dev_addr[1],
		dev->dev_addr[2],
		dev->dev_addr[3],
		dev->dev_addr[4],
		dev->dev_addr[5]);
#endif	
	mv_eth_update_mac_address(dev->dev_addr, priv->port);  
	if (mvEthHw.hwState == HW_READY)
	{
		/* start RX (BIT(7) == EnableRXdma) */
		MV_REG_WRITE(MV_ETH_SDCMR + gap, BIT(7));
		mvEthHw.hwState = HW_ACTIVE;
		mv_eth_update_mac_address(brdcast_mac, priv->port);
	}

	spin_lock_init(&dev->xmit_lock);
	spin_lock_init(&lock);

	portsBitMask = mvUnmGetPortMaskOfVid(priv->vid);

#ifdef HEADERS  
	mvBindings[priv->vid]->header[0] = 0; /* DBNUM = 0 */
	mvBindings[priv->vid]->header[1] = portsBitMask;
#elif defined (TRAILERS)
	mvBindings[priv->vid]->trailer[0] = (1<<7);	/* DBNUM = 0 */
	mvBindings[priv->vid]->trailer[1] = portsBitMask;
	mvBindings[priv->vid]->trailer[2] = 0; /* DBNUM = 0 */
	mvBindings[priv->vid]->trailer[3] = 0;
#endif

	mvBindings[priv->vid]->boOpened = true;
#ifdef ETH_DBG_TRACE
	printk("mv_eth_open: VID %d is %s\n",
		   priv->vid, (portsBitMask? "UP": "DOWN"));
#endif

	/* update logical link state, RUNNING flag and queue state */
	if (portsBitMask == 0)
	{
#ifdef USE_NAPI
		netif_poll_disable(dev);
#endif
		netif_stop_queue(dev);
		dev->flags &= ~IFF_RUNNING;
		mvBindings[priv->vid]->boLinkUp = false;
		netif_carrier_off(dev);
	} else
	{
		netif_start_queue(dev);
		dev->flags |= IFF_RUNNING;
		mvBindings[priv->vid]->boLinkUp = true;
		netif_carrier_on(dev);
#ifdef USE_NAPI
		netif_poll_enable(dev);
#endif
	}

	status = mvUnmCreateVlan(priv->vid, portsBitMask);
	if ( status != GT_OK)
	{
		panic("mv_eth_open: cannot create VID %d with bitmask %x\n",
			  priv->vid, portsBitMask);
		return -1;
	}

	return(0);
}

/*
 * ----------------------------------------------------------------------------
 * This function is used when closing the network device.  It should update
 * the hardware, release all memory that holds buffers and descriptors and
 * release the IRQ.
 * Input : a pointer to the device structure
 * Output : zero if success, nonzero if fails
 */
int
mv_eth_stop(struct net_device *dev)
{
	gt_eth_priv *priv;
	u32 queue;
	u32 count;
	bool boLastDev = false;
	int iNumOfVlans, i;
	ulong flags;
    local_irq_save(flags);

#ifdef ETH_DBG_TRACE
	printk("mv_eth_stop\n");
#endif

	iNumOfVlans = mvUnmGetNumOfVlans();
	priv = dev->priv;

	ASSERT(priv->vid);
#ifdef ETH_DBG_INFO
	printk("eth_stop for [%d]\n", priv->vid);
#endif

	if (mvBindings[priv->vid]->boOpened == false)
	{
#ifdef ETH_DBG_ERROR
		printk("Kernel is trying to close a closed interface!\n");
#endif
		local_irq_restore(flags);
		return 0;
	}

	/* mark this interface as closed */
	mvBindings[priv->vid]->boOpened = false; 
#ifdef USE_NAPI
	netif_poll_disable(dev);
#endif
	netif_stop_queue(dev);
	netif_carrier_off(dev);
	
	for (i = 1; i <= iNumOfVlans; i++)
	{
		if (mvBindings[i]->boOpened)
		{
			break; /* there are still open devices */
		}
	}
	if (i == iNumOfVlans)
	{
		boLastDev = true;
	}
#ifdef ETH_DBG_TRACE
	printk("mv_eth_stop: closing (%s) interface (VID = %d)\n",
		   boLastDev? "last": "not last", priv->vid );
#endif

	if (boLastDev)
	{
		/* stop RX and mask interrupts */
		MV_REG_WRITE(MV_ETH_SDCMR + (ETH_ADDR_GAP * priv->port), 0x00008000);
		free_irq(irq, NULL); /* free_irq(dev->irq, dev); */
		/* going down to ready */
		mvEthHw.hwState = HW_READY;
#ifdef ETH_DBG_TRACE
		printk("mv_eth_stop: release RX buffers\n");
#endif
		for (queue = 0; queue < NUM_RX_QUEUES; queue++)
		{
			for (count = 0; count < Q_INDEX_LIMIT; count++)
			{
				if (mvEthHw.RXskbuff[queue][count])
				{
					dev_kfree_skb(mvEthHw.RXskbuff[queue][count]);
					mvEthHw.RXskbuff[queue][count] = 0;
				}
			}
		}
#ifdef ETH_DBG_TRACE
		printk("mv_eth_stop: release data structures\n");
#endif
		for (i = 1; i < iNumOfVlans; i++)
		{
			if (mvBindings[i])
			{
				if (mvBindings[i]->dev)
				{
					unregister_netdev(mvBindings[i]->dev);
					if (mvBindings[i]->dev->priv)
					{
						kfree(mvBindings[i]->dev->priv);
					}
					kfree(mvBindings[i]->dev);
				}
				kfree(mvBindings[i]);
			}
		}
		memset(mvBindings, 0, sizeof(mvBindings));
	}
	local_irq_restore(flags);
	return(0);
}


static int start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	gt_eth_priv *priv = dev->priv;
	u32 queue = MV_ETH_TX_QUEUE;
	gt_dma_desc *tx;
	u32 TXindex;
	int i;
	
#if defined (HEADERS) || defined (TRAILERS) 
	bool boCopy = false;
	struct sk_buff *skb_copy;
#endif

    if( netif_queue_stopped( dev ) ) {
#ifdef ETH_DBG_ERROR
        printk( KERN_ERR "%s: transmitting while stopped\n", dev->name );
#endif
        return 1;
    }

#ifdef ETH_DBG_PKT
	printk("start_xmit: skb = %p, dev = %p\n", skb, dev);
#endif


#ifdef HEADERS
	/*
	 * Here we need to check header insertion and padding,
	 * if we have no place at the beginning of buffer for header, or we have no sufficient padding space
	 * we copy the packet to a fresh SKB 
	 */
#ifdef ETH_DBG_PKT
	printk("start_xmit: skb->data - skb->head == %d\n", (skb->data - skb->head));
	{
		int i;
		for ( i = 0; i < skb->len; i++)
		{
			if ((i & 0xf) == 0)
			{
				printk("\ndata[%d]:", i);
			}
			printk(" 0x%02x", skb->data[i]);
		}

	}
#endif /*  ETH_DBG_ECHO */

	
	/* header check */
	if ( (skb->data - skb->head) < 2 )
	{
		boCopy = true;
#ifdef ETH_DBG_ERROR
		printk("start_xmit: dev %d: need to copy beacuse of header\n", priv->vid);
#endif /* ETH_DBG_ERROR */
	}

	if (boCopy)
	{
#ifdef ETH_DBG_PKT
		printk("mv_eth_start_xmit: buffer copy for header insertion\n");
#endif
		skb_copy = dev_alloc_skb(MAX_BUFF_SIZE);
		priv->privstat.tx_needed_copy++;
		if ( skb_copy == NULL )
		{ /* silently drop */
#ifdef ETH_DBG_ERROR
			printk(KERN_CRIT "silently drop\n");
#endif /* ETH_DBG_ERROR */
			priv->stat.tx_dropped++;
			return -EBUSY;
		}
		skb_copy->dev = dev;
		/* let's copy (we are placing the data in the real data, idea is to reflect a 'normal' packet */
		skb_copy->data = skb_copy->head + HEADER_SIZE;
		memcpy( skb_copy->data, skb->data, skb->len );
		skb_copy->len = skb->len;
		/* free the original skb */ 
		dev_kfree_skb(skb);

		/* in any case skb is now pointing to an skb with a place to
		 * put the header in front and the length is not updated but
		 * we do have a sufficient valid buffer at packet's end
		 */
		skb = skb_copy;
	}
	/* we need to check the length before adding the header's two bytes, after switch will extract
	 * header the packet (with 4 bytes crc) should be 64 or longer
	 */
	if (skb->len < MIN_ETH_PACKET_LEN)
	{
		skb->len = MIN_ETH_PACKET_LEN; /* this is safe because we allocated this skb */
	}

	/* prepare the header place and insert the header */
	skb->data -= 2;
	skb->len += 2;
	ASSERT(((skb->data - skb->head) >= 0));
	((unsigned short*)skb->data)[0] =
	((unsigned short*)(mvBindings[priv->vid]->header))[0];

#elif defined (TRAILERS)
	/*
	 * Here we need to check trailer insertion and padding,
	 * if we have no place at the end of buffer for trailer, or we have no sufficient padding space
	 * we copy the packet to a fresh SKB 
	 */
#ifdef ETH_DBG_ECHO
	printk(" skb->end - skb->tail == %d\n",(skb->end - skb->tail));
#endif 

	/* padding check */
	if ( skb->len < MIN_ETH_PACKET_LEN )
	{ /*  MINIMAL_ETH_PACKET_LEN == 60 */
		if ( (skb->end - skb->tail) < (MIN_ETH_PACKET_LEN - skb->len +4 )  )
		{
			boCopy = true;
#ifdef ETH_DBG_ERROR
			printk("Dev [%d] - need to copy beacuse of padding\n", priv->vid);
#endif
		}
	}

	/* trailer check */
	else if ( (skb->end - skb->tail) < 4 )
	{
		boCopy = true;
#ifdef ETH_DBG_ERROR
		printk("Dev [%d] - need to copy beacuse of trailer\n", priv->vid);
#endif
	}

	if ( boCopy )
	{
		skb_copy = dev_alloc_skb(MAX_BUFF_SIZE);
		priv->privstat.tx_needed_copy++;
		if ( skb_copy == NULL )
		{ /* silently drop */
			priv->stat.tx_dropped++;
			return -EBUSY;
		}
		skb_copy->dev = dev;
		/* let's copy (we are placing the data in the real data, idea is to reflect a 'normal' packet */
		memcpy( skb_copy->data, skb->data, skb->len );
		skb_copy->len = skb->len;
		/* free the original skb */ 
		dev_kfree_skb(skb);

		/* in any casy skb is now pointing to an skb with a place to put the header in front and
		 * the length is not updated but we do have a sufficient valid buffer at packet's end
		 */
		skb = skb_copy;
	}
	/* we need to check the length before adding the header's two bytes, after switch will extract
	 * header the packet (with 4 bytes crc) should be 64 or longer
	 */
	if (skb->len < MIN_ETH_PACKET_LEN )
	{
		skb->len = MIN_ETH_PACKET_LEN; /* this is safe because we allocated this skb */
	}

	/* prepare the trailer place and insert the trailer */

	ASSERT( ((skb->end - skb->tail) >= 0) );
	/* TODO - use a faster memcpy */
	skb->data[skb->len]   = mvBindings[priv->vid]->trailer[0];
	skb->data[skb->len+1] = mvBindings[priv->vid]->trailer[1];
	skb->data[skb->len+2] = mvBindings[priv->vid]->trailer[2];
	skb->data[skb->len+3] = mvBindings[priv->vid]->trailer[3];
	skb->len += 4;
#endif /*  HEADERS elif TRAILERS */

	dev->trans_start = jiffies;	  /* timestamp */
	TXindex = mvEthHw.TXindex[queue];
	if (mvEthHw.TXskbuff[queue][TXindex])
	{
		panic("Error on mv_eth device driver");
	}

	mvEthHw.TXskbuff[queue][TXindex] = skb;
	tx = &mvEthHw.TXqueue[queue][TXindex];

#ifdef DESCRIPTORS_IN_CACHED_MEM
	mvOsCacheLineFlushInv((MV_ULONG)(tx));
#endif
	tx->data = (void*)mvOsIoVirtToPhy(NULL, (void*)skb->data);
	tx->count.tx.bytes = skb->len;
	tx->count.tx.reserved = (u16)priv->vid;

	/* flush/sync before transferring ownership */
	mvOsCacheFlush(NULL, skb->data, skb->len);
	wmb();

	/* Officially transfer ownership of descriptor
	   31: Owner = 1 HW
	   23: EI    = 1 Interrupt Enabled
	   22: GC    = 1 Generate CRC
	   18: P     = 1 Pad zeros
	   17: F     = 1 First in packet
	   16: L     = 1 Last in packet
	*/
	tx->command_status = (1 << 31) | (1 << 23) | (1 << 22) | (7 << 16);
//  printk("dump tx desc:skb->len 0x%x, reserved 0x%x data %x\n", skb->len, priv->vid, tx->data  );
//  printk("MV_ETH_CTDP0 %x\n", MV_REG_READ(MV_ETH_CTDP0));
#ifdef DESCRIPTORS_IN_CACHED_MEM
	mvOsCacheLineFlushInv((MV_ULONG)(tx));
#endif
	TXindex++;
	if (TXindex == Q_INDEX_LIMIT)
	{
		TXindex = 0;
	}

	/* if next descriptor is GT owned then the tx queue is full */
	if (mvEthHw.TXskbuff[queue][TXindex])
	{
		/* we will stop all queues, as HW is a shared resource */
		for (i = 1; i <= mvUnmGetNumOfVlans(); i++)
		{
#ifdef ETH_DBG_TRACE
			printk ("start_xmit: stopping queue vid %d\n", i);			
#endif
			if_stopped = 1;
			netif_stop_queue(mvBindings[i]->dev);
		}
	}
	mvEthHw.TXindex[queue] = TXindex;

	/* start Tx LOW dma */
	MV_REG_WRITE(MV_ETH_SDCMR + (ETH_ADDR_GAP * priv->port),
				 queue? (1 << 23): (1 << 24));
	return 0; /* success */
}


/*
 * ----------------------------------------------------------------------------
 * This function queues a packet in the Tx descriptor for required
 * port.  It checks the IPTOS_PREC_FLASHOVERRIDE bit of the ToS
 * field in the IP header and decides where to queue the packet,
 * in the high priority queue or in the low priority queue.
 *
 * Input : skb - a pointer to socket buffer
 *         dev - a pointer to the required port
 *
 * Output : zero upon success, negative number upon failure
 *         (-EBUSY when the interface is busy)
 */
int mv_eth_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
	unsigned long flags;
	int stat;

#ifdef ETH_DBG_PKT
	printk("mv_eth_start_xmit: skb = %p, dev = %p\n", skb, dev);
#endif

	ASSERT(dev && dev->priv);

	if (!skb)
	{
		((gt_eth_priv*)(dev->priv))->stat.tx_dropped++;
#ifdef ETH_DBG_ERROR
		printk(KERN_CRIT "Fatal Error SKB == NULL in start tx\n");
#endif /* ETH_DBG_ERROR */
		return -EBUSY;
	}
	ASSERT(netif_carrier_ok(mvBindings[((gt_eth_priv*)(dev->priv))->vid]->dev));
	ASSERT(mvBindings[((gt_eth_priv*)(dev->priv))->vid]->boLinkUp == true);

	spin_lock_irqsave(&lock, flags);
	stat = start_xmit(skb, dev);
	spin_unlock_irqrestore(&lock, flags);

	return stat;
}



/*
 * ----------------------------------------------------------------------------
 * This function is forward packets that are received from the port's
 * queues toward kernel core or FastRoute them to another interface.
 *
 * Input : dev - a pointer to the required interface
 *
 * Output : number of served packets
 */
int mv_eth_receive_queue(unsigned long data)
{
	gt_dma_desc *rx;
	struct sk_buff *skb;
	struct sk_buff *new_skb;
	u32 RXindex;
	u32 served;
	MV_UNM_VID vid;
#ifndef USE_NAPI	
	u32 eth_int_mask;
#endif
	int queue = MV_ETH_RX_QUEUE;
	unsigned int max = (unsigned int)data;
	gt_eth_priv* priv;

#if defined (HEADERS) || defined (TRAILERS)
	unsigned char ucSrcPort;
#endif

#ifdef MC_BRIDGE_EXAMPLE
	int isMulticast;
#endif

#ifdef ETH_DBG_PKT
	printk("mv_eth_receive_queue: max %d\n", max);
#endif

	/* Taken from original ISR */
	/* We are clearing the interrupt rx bit here, in order to avoid spending unneccesary
	 * time in acknowledging sparious interrupts
	 */
	MV_REG_WRITE(MV_ETH_ICR, (u32)~0x1);

	RXindex =  mvEthHw.RXindex[queue];	  /* Where we left off... */
	served = 0;
	while (max)
	{
		rx = &(mvEthHw.RXqueue[queue][RXindex]);
#ifdef DESCRIPTORS_IN_CACHED_MEM
		mvOsCacheLineInv((MV_ULONG)(rx));
// printk(" 0x%08x,  0x%08x,  0x%08x,  0x%08x\n", 
		//           *(u32 *)rx, *(((u32 *)rx) +1 ), *(((u32 *)rx) +2 ) , *(((u32 *)rx) +3 )  );

#endif

		if (rx->command_status & GT_ENET_DESC_OWNERSHIP)
		{
			break;
		}
		max--;

#ifdef ETH_DBG_PKT
		printk("mv_eth_receive_queue %d: rx == %p\n", max, rx);
#endif

		/*
		 * If received packet has errors, keep the socket buffer and change
		 * descriptor to GT ownership and continue analyzing next descriptor.
		 */
		if (rx->command_status & GT_ENET_DESC_ERROR_SUMMARY)
		{
			rx->command_status = GT_ENET_DESC_OWNERSHIP | GT_ENET_DESC_INT_ENABLE;
#ifdef DESCRIPTORS_IN_CACHED_MEM
			flush_dcache(rx);
#endif
			RXindex++;
			if (RXindex == Q_INDEX_LIMIT)
			{
				RXindex = 0;
			}
			continue;
		}

#ifdef ETH_DBG_PKT
		printk("mv_eth_receive_queue %d: good rx\n", max);
#endif

		/*
		 * We never leave descriptors without an skb
		 * we will allocate new skb, if we cannot allocate we will return the 'just-closed'
		 * descriptor to HW ownership and will not! indicate current packet up the stack
		 */
		new_skb = dev_alloc_skb(MAX_BUFF_SIZE);
		if ( new_skb == NULL )
		{
#ifdef ETH_DBG_ERROR
			printk(KERN_CRIT "Could not alloc new skb\n");
#endif /* ETH_DBG_ERROR */
			rx->command_status = GT_ENET_DESC_OWNERSHIP | GT_ENET_DESC_INT_ENABLE;

#ifdef DESCRIPTORS_IN_CACHED_MEM
			flush_dcache(rx);
#endif

			RXindex++;
			if (RXindex == Q_INDEX_LIMIT)
			{
				RXindex = 0;
			}
			continue;
		}

#ifdef ETH_DBG_PKT
		printk("mv_eth_receive_queue %d: new_skb == %p\n", max, new_skb);
#endif

		/* okay, let's deal with the packet */
		served++;
		skb = mvEthHw.RXskbuff[queue][RXindex];

#ifdef DESCRIPTORS_IN_CACHED_MEM
		invalidate_dcache(rx);
#endif


#ifdef ETH_DBG_ERROR
		if ( skb == NULL || skb->len != 0 )
			printk("mv_eth_receive_queue: SKB error\n");
#endif /* ETH_DBG_ERROR */

#ifdef ETH_DBG_PKT
		printk("mv_eth_receive_queue %d: skb == %p\n", max, skb);
#endif

		if (skb)
		{
			if (skb->len)
			{
#ifdef ETH_DBG_ERROR
				printk("mv_eth_receive_queue: nonzero existing SKB\n");
#endif /* ETH_DBG_ERROR */
				dev_kfree_skb(skb);
				skb = dev_alloc_skb(MAX_BUFF_SIZE);
			}
			/* tail - point to data end, len and data fileds will be updated manualy to extract header */

#ifdef ETH_DBG_PKT
			printk("mv_eth_receive_queue %d: skb->len == %d rx->count.rx.bytesReceived %d\n",
				   max, skb->len, rx->count.rx.bytesReceived );
			printk("mv_eth_receive_queue: \n");
			{
				int i;
				for ( i = 0; i <  rx->count.rx.bytesReceived; i++)
				{
					if ((i & 0xf) == 0)
					{
						printk("\ndata[%d]:", i);
					}
					printk(" 0x%02x", skb->data[i]);
				}
			}
#endif

#ifdef HEADERS
			ucSrcPort = ((skb->data[1]) & 0xf );

#ifdef MC_BRIDGE_EXAMPLE
			if (((skb->data[2] & 0x1) != 0) &&
				(*(unsigned int*)&skb->data[4] != 0xFFFFFFFF))
			{
				isMulticast = 1;
#ifdef ETH_DBG_MC
				printk("Received a multi packet (PCR : %0#x)\n",
					   get_port_config(0));
				printk("%0#x-%0#x-%0#x-%0#x-%0#x-%0#x\n",
					   skb->data[2], skb->data[3], skb->data[4],
					   skb->data[5],skb->data[6],skb->data[7]);
#endif
			} else
			{
				isMulticast = 0;
			}
#endif /* MC_BRIDGE_EXAMPLE */

			skb_put(skb, (rx->count.rx.bytesReceived - 4 ));
			skb->data += 2;
			skb->len = skb->len - 2; /* skb_put above updated the init value */

#elif defined (TRAILERS)
			ucSrcPort = ((skb->data[rx->count.rx.bytesReceived - 8 + 1])    & 0xf );
			memmove( skb->data + 2, skb->data, (rx->count.rx.bytesReceived - 8));
			skb->data += 2;
			skb_put(skb, (rx->count.rx.bytesReceived - 8 ));
#else
			memmove( skb->data + 2, skb->data, (rx->count.rx.bytesReceived - 4));
			skb->data += 2;
			skb_put(skb, (rx->count.rx.bytesReceived - 4 ));
#endif /* HEADERS */


#ifdef DESCRIPTORS_IN_CACHED_MEM
			invalidate_dcache(rx);
#endif

#ifdef ETH_DBG_PKT
			printk("mv_eth_receive_queue %d: skb_put done\n", max);
#endif

#if defined (HEADERS) || defined (TRAILERS) 

			/* we need to find our device from the source port */
			//  printk("%s %d\n", __FUNCTION__, __LINE__);

			ASSERT( ucSrcPort <= qd_dev->numOfPorts);
			ASSERT( ucSrcPort != qd_dev->cpuPortNum );
			vid = mvUnmGetVidOfPort(ucSrcPort);
			ASSERT( vid != 0);
			ASSERT( mvBindings[vid] != NULL );
			ASSERT( mvBindings[vid]->dev != NULL);

			skb->dev = mvBindings[vid]->dev;
			skb->protocol = eth_type_trans(skb,  mvBindings[vid]->dev);

#else /*  ! (defined (HEADERS) || defined (TRAILERS)) */
			skb->dev = mvBindings[1]->dev;
			skb->protocol = eth_type_trans(skb,  mvBindings[1]->dev);
#endif /*  defined (HEADERS) || defined (TRAILERS) */

#ifdef ETH_DBG_PKT
			printk("mv_eth_receive_queue %d: skb details filled\n", max);
#endif

			priv = (gt_eth_priv*)(skb->dev->priv);
			priv->stat.rx_packets++;

#ifdef MC_BRIDGE_EXAMPLE
			if (isMulticast)
			{
				skb->data -= 14;
				skb->len += 14;
				start_xmit(skb, mvBindings[vid^3]->dev); /* vid 1 -> vid 2, vid 2 -> vid 1 */
			} else
#endif
#ifndef USE_NAPI
            netif_rx(skb);
#else
#ifdef ETH_DBG_INFO
			if (netif_receive_skb(skb))
			{
				printk("mv_eth_receive_queue: skb dropped.\n");
			}
#else
			netif_receive_skb(skb);
#endif
#endif

#ifdef ETH_DBG_PKT
			printk("mv_eth_receive_queue %d: netif_rx done\n", max);
#endif
		}

		skb = new_skb; /* dev_alloc_skb(MAX_BUFF_SIZE);*/
		mvEthHw.RXskbuff[queue][RXindex] = skb;
		ASSERT(skb);
		mvOsCacheInvalidate(NULL, skb->data, MAX_BUFF_SIZE);		
		rx->data = (void *) mvOsIoVirtToPhy(NULL, (void *) skb->data);
		/*
		 * Officially transfer ownership of descriptor to GT
		 */
		rx->command_status = 0x80800000;	/* GT owner bit */
#ifdef DESCRIPTORS_IN_CACHED_MEM
		flush_dcache(rx);
#endif
		RXindex++;
		if (RXindex == Q_INDEX_LIMIT)
		{
			RXindex = 0;
		}
#ifdef ETH_DBG_PKT
		printk("mv_eth_receive_queue %d: rx == %p done\n", max, rx);
#endif
	}
	mvEthHw.RXindex[queue] = RXindex;
#ifdef ETH_DBG_ECHO
	if (served)
	{
		printk("Rx served %d\n",served);
	}
#endif /* ETH_DBG_ECHO */

#ifndef USE_NAPI
	/* here we open the rx interrupt back again */
#ifdef DEBUG
	MV_REG_WRITE(MV_ETH_IMR + (ETH_ADDR_GAP * port), 0x90f10109);
#else
	eth_int_mask = MV_REG_READ(MV_ETH_IMR);	/* Enhance to shadow this register */
	MV_REG_WRITE(MV_ETH_IMR, (eth_int_mask | 0x00000101) );
#endif
#endif	

#ifdef ETH_DBG_PKT
	printk("mv_eth_receive_queue: %d packets served\n", served);
#endif
	return	served;
}


/*
 * ----------------------------------------------------------------------------
 * Input : dev - a pointer to the required interface
 *
 * Output : N/A
 */
int mv_eth_free_tx_queue(unsigned long queue)
{
	gt_eth_priv *priv;
	gt_dma_desc *tx;
	struct sk_buff *sk;
	u32 freed_skbs;
	u32 TXskbIndex;
	struct net_device * dev;
	MV_UNM_VID vid;
	int i;
#ifndef USE_NAPI	
	u32 eth_int_mask;
#endif

#ifdef ETH_DBG_PKT 
	printk("mv_eth_free_tx_queue: queue = %d\n",(int)queue);
#endif

	spin_lock(&lock);
	freed_skbs = 0;
	TXskbIndex = mvEthHw.TXskbIndex[queue];

#ifdef ETH_DBG_TRACE 	
	if (if_stopped)
	{
		printk("mv_eth_free_tx_queue: queue = %d\n",(int)queue);
		printk(" TXskbIndex %d mvEthHw.TXindex %d\n",  mvEthHw.TXskbIndex[queue],
			   mvEthHw.TXindex[queue]);
	}
#endif	
	while (1)
	{
		sk = mvEthHw.TXskbuff[queue][TXskbIndex];
		if (!sk)
		{
#ifdef ETH_DBG_TRACE
			if (if_stopped)
				printk("if_stopped: mv_eth_free_tx_queue: break on !sk\n");
#endif				
			break;
		}
		tx = &(mvEthHw.TXqueue[queue][TXskbIndex]);	/* No write to tx here */
		if (tx->command_status & 0x80000000)
		{
			if (if_stopped)
			{
#ifdef ETH_DBG_TRACE
				printk("mv_eth_free_tx_queue: break on owenership\n");
#endif
				
#ifdef DESCRIPTORS_IN_CACHED_MEM
				mvOsCacheLineFlushInv((MV_ULONG)(tx));
#endif
				mdelay (1);
#ifdef ETH_DBG_TRACE
				if ((tx->command_status & 0x80000000) == 0)
				{

					printk("mv_eth_free_tx_queue: second check succeeded!\n");
					
				}
#endif				
			}
#ifdef DESCRIPTORS_IN_CACHED_MEM
			mvOsCacheLineFlushInv((MV_ULONG)(tx));
#endif
			break;
		}
		vid = (int)(tx->count.tx.reserved);
		ASSERT(vid != 0);

		if (vid == 0)
		{
			panic("something went wrong with VID in TX END\n");
		}
#ifdef ETH_DBG_ECHO		
		printk("Freeing TX packet for [%d]\n",vid);
#endif

#ifdef ETH_DBG_TRACE
		if (if_stopped)
			printk("if_stopped, Freeing TX packet for [%d]\n",vid);
#endif
		/* we need the dev (and the VID in tx descriptor - only for counters */
		dev = mvBindings[vid]->dev;
		priv = dev->priv;

#ifdef DESCRIPTORS_IN_CACHED_MEM
		mvOsCacheLineFlushInv((MV_ULONG)(tx));
#endif
		if (tx->command_status & 0x40)
		{
			priv->stat.tx_fifo_errors++;
		}
		dev_kfree_skb_irq(sk);
		mvEthHw.TXskbuff[queue][TXskbIndex] = 0;
		TXskbIndex++;
		if (TXskbIndex == Q_INDEX_LIMIT)
		{
			TXskbIndex = 0;
		}
		freed_skbs++;
		priv->stat.tx_packets++;
	}
	mvEthHw.TXskbIndex[queue] = TXskbIndex;

	spin_unlock(&lock);
#ifdef ETH_DBG_TRACE	
	if (if_stopped)
	{
		printk("mvEthHw.TXskbuff[queue][mvEthHw.TXindex[queue]] %d\n", mvEthHw.TXskbuff[queue][mvEthHw.TXindex[queue]] );
	}
#endif	
	if (mvEthHw.TXskbuff[queue][mvEthHw.TXindex[queue]] == 0)
	{
		/* there is room in the Tx queue for more SKBs */
		for (i = 1; i <= mvUnmGetNumOfVlans(); i++)
		{
#ifdef ETH_DBG_TRACE					
			if (if_stopped)
			{
				printk("netif_queue_stopped(mvBindings[%d]->dev) %d, mvBindings[%d]->boLinkUp %d\n", i,
					   netif_queue_stopped(mvBindings[i]->dev), i,mvBindings[i]->boLinkUp);
			}
#endif		
			
			if (netif_queue_stopped(mvBindings[i]->dev)
				&& (mvBindings[i]->boLinkUp))
			{
				ulong flags;
				local_irq_save(flags);	
				netif_wake_queue(mvBindings[i]->dev);
#ifdef ETH_DBG_TRACE
				printk("mv_eth_free_tx_queue: netif_wake_queue vid= %d\n", i);
#endif				
				if_stopped = 0;
				local_irq_restore(flags);
			}			
		}
	}
#ifndef USE_NAPI	
	eth_int_mask = MV_REG_READ(MV_ETH_IMR);	/* Enhance to shadow this register */
	MV_REG_WRITE(MV_ETH_IMR, (eth_int_mask | 0x00000008));
#endif	
	return freed_skbs;
}

/*
 * -------------------NAPI poll function------------------------------------------------------
 */

#ifdef USE_NAPI
static int mv_eth_rx_poll(struct net_device *dev, int *budget)
{
	int tx_work_done = mv_eth_free_tx_queue(MV_ETH_TX_QUEUE);
	int rx_work_done = mv_eth_receive_queue(min(*budget,dev->quota));
	*budget -= rx_work_done;
	dev->quota -= rx_work_done;
	
	if (((rx_work_done == 0) && (tx_work_done == 0)) || !netif_running(dev)) 
	{
		u32 eth_int_mask;
 		ulong flags;
		local_irq_save(flags);
#ifdef ETH_DBG_INFO
		printk("mv_eth_rx_poll %s: Rx complete\n", dev->name);
#endif
		netif_rx_complete(dev);

		/*
			Enable interrupts
		*/
		/* here we open the rx and tx interrupt back again */
		eth_int_mask = MV_REG_READ(MV_ETH_IMR);	/* Enhance to shadow this register */
		MV_REG_WRITE(MV_ETH_IMR, (eth_int_mask | 0x00000109));		
		local_irq_restore(flags);
		return 0;
	}
#ifdef ETH_DBG_INFO
	printk("mv_eth_rx_poll %s: more Rx to do.\n", dev->name);
#endif
	return 1;
}
#endif


/*
 * ----------------------------------------------------------------------------
 */

static irqreturn_t mv_eth_int_handler(s32 irq, void * dev_id, struct pt_regs *regs)
{
	u32 eth_int_cause;
	u32 eth_int_mask;
#ifdef USE_NAPI
	struct net_device *dev = (struct net_device *)dev_id;
	u32 sched_poll = 0;
#endif

#ifdef ETH_DBG_INFO
	printk("mv_eth_int_handler\n");
#endif

	eth_int_cause = MV_REG_READ(MV_ETH_ICR);
	eth_int_mask = MV_REG_READ(MV_ETH_IMR);	/* Enhance to shadow this register */

#ifdef ETH_DBG_INFO
	printk("mv_eth_int_handler: irq=%d, cause=0x%08x, mask=0x%08x, irsr=0x%08x\n",
		   irq, eth_int_cause, eth_int_mask, MV_REG_READ(MV_ETH_IRSR));
	printk("MV_ETH_CTDP0 %x\n", MV_REG_READ(MV_ETH_CTDP0));
#endif
	if (!eth_int_cause)
	{
		return IRQ_NONE;
	}
	
	if (eth_int_cause & 0x8)
	{ /* TxDone */
		eth_int_mask &= ~0x8;

#ifdef USE_NAPI	
		sched_poll |= 0x2;
#else	
		MV_REG_WRITE(MV_ETH_IMR, eth_int_mask);
		schedule_work(&txComplete);
		MV_REG_WRITE(MV_ETH_ICR, ~0x8);
#endif		
		
	}

	if (eth_int_cause & 0x1)
	{ /* RxBuffer */
		eth_int_mask &= ~0x1;
		
#ifdef USE_NAPI
		sched_poll = 0x1;
#else
		MV_REG_WRITE(MV_ETH_IMR, eth_int_mask);
		schedule_work(&rxComplete);
		MV_REG_WRITE(MV_ETH_ICR, ~0x101);
#endif
		
	}

	if (eth_int_cause & 0x100)
	{ /* RxError */
		eth_int_mask &= ~0x101;	/* clear pending Rx buffers */
#ifdef USE_NAPI
		sched_poll = 0x1;
#else
		MV_REG_WRITE(MV_ETH_IMR, eth_int_mask);
		schedule_work(&rxComplete);
		MV_REG_WRITE(MV_ETH_ICR, ~0x101);
#endif
		
	}
#ifdef USE_NAPI
	if (sched_poll)
	{	
		if (netif_rx_schedule_prep(dev))
		{
			MV_REG_WRITE(MV_ETH_IMR, eth_int_mask);
			if (sched_poll & 0x2)
				MV_REG_WRITE(MV_ETH_ICR, ~0x8);
			if (sched_poll & 0x1)
				MV_REG_WRITE(MV_ETH_ICR, ~0x101);
			__netif_rx_schedule(dev);
		}
#ifdef ETH_DBG_WARNINIG
		else
		{
			if(netif_running(dev)) 
				printk("mv_eth_int_handler: interrupt while in polling list\n");
		}
#endif		
	}
#endif
	return IRQ_HANDLED;
}

/*********************************************************************/
/* MII/SMI interface                                                 */
/*********************************************************************/

/*****************************************************************************
*
* int etherReadMIIReg (unsigned int portNumber , unsigned int MIIReg,
* unsigned int* value)
*
* Description
* This function will access the MII registers and will read the value of
* the MII register , and will retrieve the value in the pointer.
* Inputs
* portNumber - one of the 2 possiable Ethernet ports (0-1).
* MIIReg - the MII register offset.
* Outputs
* value - pointer to unsigned int which will receive the value.
* Returns Value
* 1 if success.
* 0 if fail to make the assignment.
* Error types (and exceptions if exist)
*/

static int
etherReadMIIReg(unsigned int portNumber, unsigned int MIIReg,
				unsigned int *value)
{
	SMI_REG smiReg;
	unsigned int phyAddr;
	unsigned int timeOut = 1000;
	int i;

	phyAddr = PHY_ADD0 + portNumber;

	/* first check that it is not busy */
	smiReg = MV_REG_READ(MV_ETH_SMIR);
	while (smiReg & SMI_BUSY)
	{
		if (timeOut-- < 1)
		{
#ifdef ETH_DBG_WARNING
			printk("TimeOut Passed Phy is busy\n");
#endif
			return 0;
		}
		for (i = 0; i < 1000; i++) ;
		smiReg = MV_REG_READ(MV_ETH_SMIR);
	}
	/* not busy */

	MV_REG_WRITE(MV_ETH_SMIR,
				 (SMI_OP_CODE_BIT_READ << 26) | (MIIReg << 21) | (phyAddr <<
																  16));

	timeOut = 1000;		/* initialize the time out var again */

	for (i = 0; i < 1000; i++) ;
	smiReg = MV_REG_READ(MV_ETH_SMIR);
	while (!(smiReg & READ_VALID))
	{
		for (i = 0; i < 1000; i++) ;
		smiReg = MV_REG_READ(MV_ETH_SMIR);
		if (timeOut-- < 1)
		{
#ifdef ETH_DBG_WARNING
			printk("TimeOut Passed Read is not valid\n");
#endif
			return 0;
		}
	}

	*value = (unsigned int) (smiReg & 0xffff);

	return 1;
}


/*****************************************************************************
* 
* int etherWriteMIIReg (unsigned int portNumber , unsigned int MIIReg,
* unsigned int value)
* 
* Description
* This function will access the MII registers and will write the value
* to the MII register.
* Inputs
* portNumber - one of the 2 possiable Ethernet ports (0-1).
* MIIReg - the MII register offset.
* value -the value that will be written.
* Outputs
* Returns Value
* 1 if success.
* 0 if fail to make the assignment.
* Error types (and exceptions if exist)
*/

static int
etherWriteMIIReg(unsigned int portNumber, unsigned int MIIReg,
				 unsigned int value)
{
	SMI_REG smiReg;
	unsigned int phyAddr;
	unsigned int timeOut = 10;	/* in 100MS units */
	int i;

	/* first check that it is not busy */
	(unsigned int) smiReg = MV_REG_READ(MV_ETH_SMIR);
	if (smiReg & SMI_BUSY)
	{
		for (i = 0; i < 10000000; i++) ;
		do
		{
			(unsigned int) smiReg = MV_REG_READ(MV_ETH_SMIR);
			if (timeOut-- < 1)
			{
#ifdef ETH_DBG_WARNING
				printk("TimeOut Passed Phy is busy\n");
#endif
				return 0;
			}
		} while (smiReg & SMI_BUSY);
	}
	/* not busy */

	phyAddr = PHY_ADD0 + portNumber;

	smiReg = 0;		/* make sure no garbage value in reserved bits */
	smiReg = smiReg | (phyAddr << 16) | (SMI_OP_CODE_BIT_WRITE << 26) |
			 (MIIReg << 21) | (value & 0xffff);

	MV_REG_WRITE(MV_ETH_SMIR, *((unsigned int *) &smiReg));

	return(1);
}


/*********************************************************************/
/* port manipulation                                                 */
/*********************************************************************/

/*
 * GT_STATUS  mv_eth_remove_port_from_vlan 
 *
 * remove a port from a given vlan.
 * this function will remove the port from the vlan, update its trailer/header
 * if this is the last port in this vlan we will signal the link of this vid to be down
 * the interface will stop its RUNNING flag in this case
 *
 * Inputs:
 * port - the port number to remove.
 * vid  - the vid to remove the port from.
 * newPortsBitMask - the new bit mask of this vid after a logical removal of this port.
 *
 * Outputs:
 * GT_STATUS - operation status.
 */
static GT_STATUS  mv_eth_remove_port_from_vlan( unsigned int port, MV_UNM_VID vid, int newPortsBitMask) {

#ifdef ETH_DBG_TRACE
	printk("mv_eth_remove_port\n");
#endif


	if ( vid < 1 || vid > mvUnmGetNumOfVlans() )
	{
		ASSERT(0); /*  that's a bug */
		return GT_FAIL;
	}
	ASSERT(vid <= mvUnmGetNumOfVlans() );
	ASSERT(vid>0); /*  ypu cannot remove from vid zero */
	ASSERT(mvBindings[vid]->dev);

	/* if the new bit mask is zero it means we have left without any ports in this vlan,
	 * and the link state should be updated (to DOWN)
	 */   
	if ( newPortsBitMask != 0 )
	{ /*  not last port in this vlan */
#if defined (HEADERS)
		mvBindings[vid]->header[1] = (unsigned char)newPortsBitMask;
#elif defined (TRAILERS)
		mvBindings[vid]->trailer[1] = (unsigned char)newPortsBitMask;
#else 
#error "no trailers and no headers?"
#endif
	} else
	{ /*  last port in this vlan */
#if defined (HEADERS)
		mvBindings[vid]->header[0] = 0;
		mvBindings[vid]->header[1] = 0;
#elif defined (TRAILERS)
		mvBindings[vid]->trailer[0] = 0;
		mvBindings[vid]->trailer[1] = 0;
#else 
#error "no trailers and no headers?"
#endif   
		netif_stop_queue(mvBindings[vid]->dev);
		(mvBindings[vid]->dev)->flags &= ~IFF_RUNNING;
		netif_carrier_off(mvBindings[vid]->dev);    
		mvBindings[vid]->boLinkUp = false;
	}
	return GT_OK;
}

/*
 * GT_STATUS  mv_eth_add_port_to_vlan 
 *
 * add a port to a given vlan.
 * this function will add the port to the vlan, update its trailer/header
 * if this is the first port in this vlan we will signal the link of this vid to be up
 * the interface will start and its RUNNING flag will be set in this case
 *
 * Inputs:
 * port - the port number to add.
 * vid  - the vid to add the port to.
 * newPortsBitMask - the new bit mask of this vid after a logical removal of this port.
 * numOfPorts - the number of ports in the vlan after a logical add of this (being added) port
 *
 * Outputs:
 * GT_STATUS - operation status.
 */
static GT_STATUS  mv_eth_add_port_to_vlan( unsigned int port, MV_UNM_VID vid, int newPortsBitMask, int numOfPorts) {

#ifdef ETH_DBG_TRACE
	printk("mv_eth_add_port\n");
#endif

	ASSERT(vid);
	ASSERT(mvBindings[vid]->dev);

	if ( vid < 1 || vid > mvUnmGetNumOfVlans() )
	{
		ASSERT(0); /*  that's a bug */
		return GT_FAIL;
	}

	/* if number of ports is exactly one, then this port (being added now) is the fisrt
	 * port in this vlan, and the link state should be updated
	 */
	if (numOfPorts != 1 )
	{ /*  not the first port, just update trailer/header */
#if defined (HEADERS)
		mvBindings[vid]->header[1] = (unsigned char)newPortsBitMask;
#elif defined (TRAILERS)
		mvBindings[vid]->trailer[1] = (unsigned char)newPortsBitMask;
#else 
#error "no trailers and no headers?"
#endif
	} else
	{ /*  first port in this vlan */
#if defined (HEADERS)
		mvBindings[vid]->header[0] = 0;
		mvBindings[vid]->header[1] = (unsigned char)newPortsBitMask;
#elif defined (TRAILERS)
		mvBindings[vid]->trailer[0] = (1<<7); /*  INGRESS_TRAILER_OVERRIDE */
		mvBindings[vid]->trailer[1] = (unsigned char)newPortsBitMask;
#else 
#error "no trailers and no headers?"
#endif   
		mvBindings[vid]->boLinkUp = true;
		netif_carrier_on(mvBindings[vid]->dev);  
		netif_start_queue(mvBindings[vid]->dev);
		(mvBindings[vid]->dev)->flags |= IFF_RUNNING;
	}
	return GT_OK;
}

GT_STATUS mvDisassociatePort(int qdPort, int fromVlanId, int newPortsBitMask) {
	return mv_eth_remove_port_from_vlan(qdPort, fromVlanId, newPortsBitMask);
}
GT_STATUS mvAssociatePort(int qdPort, int toVlanId, int newPortsBitMask, int numOfPorts) {
	return mv_eth_add_port_to_vlan( qdPort, toVlanId, newPortsBitMask, numOfPorts);
}


/*********************************************************************/
/* module start                                                      */
/*********************************************************************/
extern int qdEntryPoint(void);
extern void qdExitPoint(void);
extern int qdModuleStart(void);

/* 
 * int mv_eth_start(void) 
 *
 * the start function is to enable to split the real initialization of QD and networking 
 * from the module initialization (either threw kernel boot or threw insmod command).
 */
int mv_eth_start(void) {
	int cards = 0;
	int status = GT_OK;
	int iNumOfVlans,i;
	BINDING * pBinding;
	struct net_device * pNetDev = NULL;

#ifdef ETH_DBG_TRACE
	printk("mv_eth_start\n");
#endif

	memset(mvBindings, 0, sizeof(mvBindings));
	memset(&mvEthHw, 0, sizeof(mvEthHw));

	if (qdModuleStart() != 0)
	{
#ifdef ETH_DBG_ERROR
		printk("Error in QD init\n");
#endif
	}

	if ((status = mvUnmInitialize()) != GT_OK)
	{
#ifdef ETH_DBG_ERROR
		printk("Error - NetConfig is invalid, can't start network device\n");
#endif
		return -ENODEV;
	}

#ifdef ETH_DBG_INFO
	printk("mv_eth_start UNM is initialized\n");
#endif

	mvEthHw.hwState = HW_UNKNOWN;

	iNumOfVlans = mvUnmGetNumOfVlans();
#ifdef ETH_DBG_INFO
	printk ("mv_eth init: iNumOfVlans = %d\n", iNumOfVlans);
	mv_nc_printConf();
#endif

	/* When the first interface (represented by network device) is
	   started, it creates and registers all interfaces.  Note that
	   VID 0 is reserved */
	for (i = 1; i <= iNumOfVlans; i++)
	{
		/* allocate and clean data structures */
		pBinding = (BINDING*)kmalloc(sizeof(BINDING), GFP_KERNEL);
		/* setup the reguar Ethernet stuff with the kernel */
		pNetDev = alloc_etherdev(sizeof(gt_eth_priv));
		if (!pNetDev || !pBinding)
		{
			panic("mv_eth_module_init : kmalloc failed\n");
		}
		memset(pBinding, 0, sizeof(BINDING));
		/* initialize structs */
		mvBindings[i] = pBinding;
		pBinding->pmvEthHw = &mvEthHw;
		pBinding->dev = pNetDev;   

		pNetDev->base_addr = PLATFORM_GTREG_BASE + FAST_ETH_MAC_OFFSET;
		pNetDev->init = mv_eth_init;
		/* init device mac addr */		
		if (overEthAddr)
		{
			mv_eth_convert_str_to_mac(CONFIG_ETH_0_MACADDR, pNetDev->dev_addr);			
		}
		else
		{
#ifdef ETH_DBG_WARN
			if (mvMacAddr[0] == 0xFF)
			{
				printk("mv_eth_start: MAC address is not set for eth%d!\n", i-1);	
			}
			else
#endif			
			{	
				memcpy(pNetDev->dev_addr, mvMacAddr, 6);
				
			}
		}
		pNetDev->dev_addr[5] += (i - 1);
#ifdef ETH_DBG_INFO
		printk("mv_eth_start: set MAC address to %02x:%02x:%02x:%02x:%02x:%02x\n",
			   pNetDev->dev_addr[0],
			   pNetDev->dev_addr[1],
			   pNetDev->dev_addr[2],
			   pNetDev->dev_addr[3],
			   pNetDev->dev_addr[4],
			   pNetDev->dev_addr[5]);
#endif				  

		if (register_netdev(mvBindings[i]->dev) == 0)
		{
			cards++;
			printk("if %s registered\n",mvBindings[i]->dev->name);
		}
		else
		{
			printk("if %s register failed!\n", mvBindings[i]->dev->name);
		}
	}

	/* CleanUP - deregister net devices */
	if (cards < iNumOfVlans)
	{
		panic("mv_eth_module_init: unable to register network interface\n");
	}

#ifdef ETH_DBG_TRACE
	printk("mv_eth module inited with %d devices\n", cards);
#endif
	return (cards > 0)? 0: -ENODEV;
}


/*********************************************************************/
/* module initialization and deinitialization                        */
/*********************************************************************/
int __init mv_eth_module_init(void) {
#ifdef ETH_DBG_INFO
	printk("mv_eth module init\n");
#endif

	/* init the QD hidden module */
	qdEntryPoint();
	mv_eth_start();
	return 0;
}

void
mv_eth_module_exit(void)
{
	/* exit for the QD hidden module */
	qdExitPoint();
#ifdef ETH_DBG_INFO
	printk("mv_eth module exited\n");
#endif

}

module_init(mv_eth_module_init);
module_exit(mv_eth_module_exit);

MODULE_LICENSE("GPL");
MODULE_AUTHOR("Marvell");
MODULE_DESCRIPTION("Ethernet driver for UniMAC");
