/*******************************************************************************
Copyright (C) Marvell International Ltd. and its affiliates

This software file (the "File") is owned and distributed by Marvell 
International Ltd. and/or its affiliates ("Marvell") under the following
alternative licensing terms.  Once you have made an election to distribute the
File under one of the following license alternatives, please (i) delete this
introductory statement regarding license alternatives, (ii) delete the two
license alternatives that you have not elected to use and (iii) preserve the
Marvell copyright notice above.


********************************************************************************
Marvell GPL License Option

If you received this File from Marvell, you may opt to use, redistribute and/or 
modify this File in accordance with the terms and conditions of the General 
Public License Version 2, June 1991 (the "GPL License"), a copy of which is 
available along with the File in the license.txt file or by writing to the Free 
Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 or 
on the worldwide web at http://www.gnu.org/licenses/gpl.txt. 

THE FILE IS DISTRIBUTED AS-IS, WITHOUT WARRANTY OF ANY KIND, AND THE IMPLIED 
WARRANTIES OF MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE ARE EXPRESSLY 
DISCLAIMED.  The GPL License provides additional details about this warranty 
disclaimer.
*******************************************************************************/

#include <linux/kernel.h>
#include <linux/version.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/skbuff.h>
#include <linux/pci.h>
#include <linux/ip.h>
#include <linux/in.h>

#include "mvOs.h"
#include "mvDebug.h"
#include "mvSysHwConfig.h"
#include "mvEth.h"
#include "mvEthPhy.h"
#include "msApi.h"
#include "dbg-trace.h"
#ifdef INCLUDE_MULTI_QUEUE
#include "mvEthPolicy.h"
#endif


/* RD2-5181L-FE board/switch ports mapping */
#define FE_QD_PORT_CPU  5
#define FE_QD_PORT_0    0
#define FE_QD_PORT_1    1
#define FE_QD_PORT_2    2
#define FE_QD_PORT_3    3
#define FE_QD_PORT_4    4

/* RD2-5181L-GE board/switch ports mapping */
#define GE_QD_PORT_CPU  3
#define GE_QD_PORT_0    2
#define GE_QD_PORT_1    1
#define GE_QD_PORT_2    0
#define GE_QD_PORT_3    7
#define GE_QD_PORT_4    5

/* run time detection GE/FE */
int QD_PORT_CPU;
int QD_PORT_0;
int QD_PORT_1;
int QD_PORT_2;
int QD_PORT_3;
int QD_PORT_4;

/* prioritization for incoming traffic (lower number = lower priority) */
#define MV_VOIP_PRIO 3
#define MV_VIDEO_PRIO  2

/* helpers for VLAN tag handling */
#define MV_GTW_PORT_VLAN_ID(grp,port)  ((grp)+(port)+1)
#define MV_GTW_GROUP_VLAN_ID(grp)      (((grp)+1)<<8)
#define MV_GTW_VLANID_TO_GROUP(vlanid) ((((vlanid) & 0xf00) >> 8)-1)
#define MV_GTW_VLANID_TO_PORT(vlanid)  (((vlanid) & 0xf)-1)

#ifdef INCLUDE_MULTI_QUEUE
#define EGIGA_TXQ_MASK	     (BIT0)
#define EGIGA_RXQ_MASK	     (BIT9|BIT8|BIT7|BIT6|BIT5|BIT4|BIT3|BIT2)
#define EGIGA_RXQ_RES_MASK   (BIT18|BIT17|BIT16|BIT15|BIT14|BIT13|BIT12|BIT11)
#else
#define EGIGA_TXQ_MASK	     (BIT0)
#define EGIGA_RXQ_MASK       (BIT2)
#define EGIGA_RXQ_RES_MASK   (BIT11)
#endif

#define EGIGA_PICR_MASK      (BIT1|EGIGA_RXQ_MASK|EGIGA_RXQ_RES_MASK)
#define EGIGA_PICER_MASK     (EGIGA_TXQ_MASK)
#define EGIGA_DEF_PORT	    0
#define QD_MIN_ETH_PACKET_LEN 60
#define QD_VLANTAG_SIZE 4
#define MV_MTU MV_ALIGN_UP(1500 + 2 + 4 + ETH_HLEN + 4, 32)  /* 2(HW hdr) 4(QD extra) 14(MAC hdr) 4(CRC) */

/* number of descriptors per queue */
int gbe_desc_num_per_rxq[MV_ETH_RX_Q_NUM] = {
EGIGA_NUM_OF_RX_DESCR*2,
#ifdef INCLUDE_MULTI_QUEUE
EGIGA_NUM_OF_RX_DESCR,EGIGA_NUM_OF_RX_DESCR,EGIGA_NUM_OF_RX_DESCR,EGIGA_NUM_OF_RX_DESCR,
EGIGA_NUM_OF_RX_DESCR,EGIGA_NUM_OF_RX_DESCR,EGIGA_NUM_OF_RX_DESCR*2,
#endif
};
int gbe_desc_num_per_txq[MV_ETH_TX_Q_NUM] = {
/*EGIGA_NUM_OF_TX_DESCR*2*/2000, /* should be large enough to enhance ipsec support */
};

/* debug control */
#define GTW_DEBUG
#undef GTW_DEBUG
#define GTW_DBG_OFF     0x0000
#define GTW_DBG_RX      0x0001
#define GTW_DBG_TX      0x0002
#define GTW_DBG_RX_FILL 0x0004
#define GTW_DBG_TX_DONE 0x0008
#define GTW_DBG_LOAD    0x0010
#define GTW_DBG_IOCTL   0x0020
#define GTW_DBG_INT     0x0040
#define GTW_DBG_STATS   0x0080
#define GTW_DBG_MCAST   0x0100
#define GTW_DBG_VLAN    0x0200
#define GTW_DBG_IGMP    0x0400
#define GTW_DBG_MACADDR 0x0800
#define GTW_DBG_ALL     0xffff
#ifdef GTW_DEBUG
# define GTW_DBG(FLG, X) if( (mv_gtw_dbg & (FLG)) == (FLG) ) printk X
#else
# define GTW_DBG(FLG, X)
#endif
unsigned int mv_gtw_dbg = GTW_DBG_ALL;

/* gigabit statistics */
#ifdef CONFIG_EGIGA_STATIS
#define EGIGA_STATISTICS
#else
#undef EGIGA_STATISTICS
#endif
#define EGIGA_STAT_OFF     0x0000
#define EGIGA_STAT_RX      0x0001
#define EGIGA_STAT_TX      0x0002
#define EGIGA_STAT_RX_FILL 0x0004
#define EGIGA_STAT_TX_DONE 0x0008
#define EGIGA_STAT_LOAD    0x0010
#define EGIGA_STAT_IOCTL   0x0020
#define EGIGA_STAT_INT     0x0040
#define EGIGA_STAT_POLL    0x0080
#define EGIGA_STAT_ALL     0xffff
#ifdef EGIGA_STATISTICS
# define EGIGA_STAT(FLG, CODE) if( (mv_gtw_stat & (FLG)) == (FLG) ) CODE;
#else
# define EGIGA_STAT(FLG, CODE)
#endif
unsigned int mv_gtw_stat =  EGIGA_STAT_ALL;

typedef struct _egiga_statistics
{
    unsigned int irq_total, irq_none;
    unsigned int poll_events, poll_complete;
    unsigned int rx_events, rx_hal_ok[MV_ETH_RX_Q_NUM],
    rx_hal_no_resource[MV_ETH_RX_Q_NUM],rx_hal_no_more[MV_ETH_RX_Q_NUM],
    rx_hal_error[MV_ETH_RX_Q_NUM],rx_hal_invalid_skb[MV_ETH_RX_Q_NUM],
    rx_hal_bad_stat[MV_ETH_RX_Q_NUM],rx_netif_drop[MV_ETH_RX_Q_NUM];
    unsigned int tx_done_events,tx_done_hal_ok[MV_ETH_TX_Q_NUM],
    tx_done_hal_invalid_skb[MV_ETH_TX_Q_NUM],tx_done_hal_bad_stat[MV_ETH_TX_Q_NUM],
    tx_done_hal_still_tx[MV_ETH_TX_Q_NUM],tx_done_hal_no_more[MV_ETH_TX_Q_NUM],
    tx_done_hal_unrecognize[MV_ETH_TX_Q_NUM],tx_done_max[MV_ETH_TX_Q_NUM],
    tx_done_min[MV_ETH_TX_Q_NUM],tx_done_netif_wake[MV_ETH_TX_Q_NUM];
    unsigned int rx_fill_events[MV_ETH_RX_Q_NUM],rx_fill_alloc_skb_fail[MV_ETH_RX_Q_NUM],
    rx_fill_hal_ok[MV_ETH_RX_Q_NUM],rx_fill_hal_full[MV_ETH_RX_Q_NUM],
    rx_fill_hal_error[MV_ETH_RX_Q_NUM],rx_fill_timeout_events;
    unsigned int tx_events,tx_hal_ok[MV_ETH_TX_Q_NUM],tx_hal_no_resource[MV_ETH_TX_Q_NUM],
    tx_hal_error[MV_ETH_TX_Q_NUM],tx_hal_unrecognize[MV_ETH_TX_Q_NUM],tx_netif_stop[MV_ETH_TX_Q_NUM],
    tx_timeout;
} egiga_statistics;

typedef struct _egiga_priv
{
    void* hal_priv;
    void* rx_policy_priv;
    unsigned int rxq_count[MV_ETH_RX_Q_NUM], txq_count[MV_ETH_TX_Q_NUM];
    MV_BUF_INFO tx_buf_info_arr[MAX_SKB_FRAGS+3];
    MV_PKT_INFO tx_pkt_info;
#ifdef EGIGA_STATISTICS
    egiga_statistics egiga_stat;
#endif
    unsigned int rx_coal, tx_coal, rxcause, txcause, rxmask, txmask;
    spinlock_t lock;
    unsigned int netif_stopped;
    struct timer_list rx_fill_timer;
    unsigned rx_fill_flag;
} egiga_priv; 

struct mv_vlan_cfg {
    char name[IFNAMSIZ];
    unsigned int ports_mask;
    unsigned short vlan_grp_id;
    MV_8 *macaddr; /* e.g. "00:11:22:33:44:55" */
};

typedef struct _mv_priv {
    struct net_device *net_dev;			/* back reference to the net_device */
    struct mv_vlan_cfg *vlan_cfg;		/* reference to entry in net config table */
    struct net_device_stats net_dev_stat;	/* statistic counters */
}mv_priv;

/* Network interfaces configuration */
#if 1
#define MV_NUM_OF_IFS  2

#define MV_IF0_NAME          "eth0" /* WAN */
#define MV_IF0_MAC_ADDR      "00:11:66:11:66:11"
#define MV_IF0_VLAN_PORTS_GE ((1<<GE_QD_PORT_0))
#define MV_IF0_VLAN_PORTS_FE ((1<<FE_QD_PORT_0))

#define MV_IF1_NAME          "eth1" /* LAN */
#define MV_IF1_MAC_ADDR      "00:22:77:22:77:22"
#define MV_IF1_VLAN_PORTS_GE ((1<<GE_QD_PORT_1)|(1<<GE_QD_PORT_2)|(1<<GE_QD_PORT_3)|(1<<GE_QD_PORT_4))
#define MV_IF1_VLAN_PORTS_FE ((1<<FE_QD_PORT_1)|(1<<FE_QD_PORT_2)|(1<<FE_QD_PORT_3)|(1<<FE_QD_PORT_4))

#else
#define MV_NUM_OF_IFS  3

#define MV_IF0_NAME          "eth0" /* WAN */
#define MV_IF0_MAC_ADDR      "00:11:66:11:66:11"
#define MV_IF0_VLAN_PORTS_GE ((1<<GE_QD_PORT_0))
#define MV_IF0_VLAN_PORTS_FE ((1<<FE_QD_PORT_0))

#define MV_IF1_NAME          "eth1" /* LAN1 */
#define MV_IF1_MAC_ADDR      "00:22:77:22:77:22"
#define MV_IF1_VLAN_PORTS_GE ((1<<GE_QD_PORT_1)|(1<<GE_QD_PORT_2))
#define MV_IF1_VLAN_PORTS_FE ((1<<FE_QD_PORT_1)|(1<<FE_QD_PORT_2))

#define MV_IF2_NAME          "eth2" /* LAN2 */
#define MV_IF2_MAC_ADDR      "00:33:88:33:88:33"
#define MV_IF2_VLAN_PORTS_GE ((1<<GE_QD_PORT_3)|(1<<GE_QD_PORT_4))
#define MV_IF2_VLAN_PORTS_FE ((1<<FE_QD_PORT_3)|(1<<FE_QD_PORT_4))

#endif


/* globals variables */
static mv_priv mv_privs[MV_NUM_OF_IFS];
static egiga_priv gbe_dev;
static GT_QD_DEV qddev, *qd_dev = NULL;
static GT_SYS_CONFIG qd_cfg;
static struct net_device *main_net_dev = NULL;
struct mv_vlan_cfg net_cfg[MV_NUM_OF_IFS];
static unsigned char zero_pad[QD_MIN_ETH_PACKET_LEN] = {0};
extern unsigned int overEthAddr;
static char boot_mac_addr[32] = {0};

/* functions prototype */
static int __init mv_gtw_init_module( void );
static void __init mv_gtw_exit_module( void );
static int init_switch(void);
static int init_gigabit(void);
static int mv_gtw_start( struct net_device *dev );
static int mv_gtw_stop( struct net_device *dev );
static int mv_gtw_tx( struct sk_buff *skb, struct net_device *dev );
static unsigned int mv_gtw_tx_done(void);
static unsigned int mv_gtw_rx(unsigned int work_to_do);
static unsigned int mv_gtw_rx_fill(unsigned int queue, int count);
static void mv_gtw_tx_timeout( struct net_device *dev );
static void mv_gtw_rx_fill_on_timeout(unsigned long data);
static int mv_gtw_poll( struct net_device *dev, int *budget );
static irqreturn_t mv_gtw_interrupt_handler( int rq , void *dev_id , struct pt_regs *regs );
GT_BOOL ReadMiiWrap(GT_QD_DEV* dev, unsigned int portNumber, unsigned int MIIReg, unsigned int* value);
GT_BOOL WriteMiiWrap(GT_QD_DEV* dev, unsigned int portNumber, unsigned int MIIReg, unsigned int data);
static int mv_gtw_set_port_based_vlan(unsigned int ports_mask);
static int mv_gtw_set_vlan_in_vtu(unsigned short vlan_id,unsigned int ports_mask);
static int mv_gtw_set_qos_in_switch(void);
static int mv_gtw_set_mac_addr( struct net_device *dev, void *addr );
int mv_gtw_set_mac_addr_to_switch(unsigned char *mac_addr, unsigned char db, unsigned int ports_mask, unsigned char op);
static void mv_gtw_set_multicast_list(struct net_device *dev);
static struct net_device_stats* mv_gtw_get_stats( struct net_device *dev );
static struct net_device* mv_gtw_main_net_dev_get(void);
void print_mv_gtw_stat(void);
static int init_config(void);
void print_qd_port_counters(unsigned int port);
void print_qd_atu(int db);
static void mv_gtw_convert_str_to_mac(char *source , char *dest);
static unsigned int mv_gtw_str_to_hex(char ch);
#ifdef CONFIG_MV_GTW_QOS_VOIP
int mv_gtw_qos_tos_quota = -1;
extern int MAX_SOFTIRQ_RESTART;
int mv_gtw_qos_tos_enable(void);
int mv_gtw_qos_tos_disable(void);
EXPORT_SYMBOL(mv_gtw_qos_tos_enable);
EXPORT_SYMBOL(mv_gtw_qos_tos_disable);
#endif
#ifdef CONFIG_MV_GTW_IGMP
extern int mv_gtw_igmp_snoop_init(void);
extern int mv_gtw_igmp_snoop_exit(void);
extern int mv_gtw_igmp_snoop_process(struct sk_buff* skb, unsigned char port, unsigned char vlan_dbnum);
#endif

module_init(mv_gtw_init_module);
module_exit(mv_gtw_exit_module);
MODULE_DESCRIPTION("Marvell Gateway Driver - www.marvell.com");
MODULE_AUTHOR("Tzachi Perelstein <tzachi@marvell.com>");
MODULE_LICENSE("GPL");

extern GT_STATUS hwReadPortReg(IN GT_QD_DEV *dev,IN GT_U8 portNum, IN GT_U8 regAddr, OUT GT_U16 *data);
static unsigned short mv_gtw_read_switch_reg(unsigned char port, unsigned char reg)
{
    unsigned short data=0;
    hwReadPortReg(qd_dev,port,reg,&data);
    return data;
}

static int __init mv_gtw_init_module( void ) 
{
    unsigned int i;
    struct net_device *dev;
    struct mv_vlan_cfg *nc;

    printk("Marvell Gateway Driver:\n");

    if(init_config()) {
	printk("failed to init config\n");
	return -ENODEV;
    }

    if(init_switch()) {
	printk("failed to init switch\n");
	return -ENODEV;
    }

    if(init_gigabit()) {
	printk("failed to init gigabit ethernet\n");
	return -ENODEV;
    }

    /* load net_devices */
    printk("loading network interfaces: ");
    for(i=0, nc=net_cfg; i<MV_NUM_OF_IFS; i++,nc++) {
        dev = alloc_netdev(0, nc->name, ether_setup);
	mv_privs[i].net_dev = dev;	/* back reference to the net_device */
	mv_privs[i].vlan_cfg = nc;	/* reference to entry in net config table */
	memset(&(mv_privs[i].net_dev_stat), 0, sizeof(struct net_device_stats));/* statistic counters */
	dev->priv = &mv_privs[i];
        dev->irq = ETH_PORT0_IRQ_NUM;
	mv_gtw_convert_str_to_mac(nc->macaddr,dev->dev_addr);
	dev->open = mv_gtw_start;
        dev->stop = mv_gtw_stop;
	dev->hard_start_xmit = mv_gtw_tx;
        dev->tx_timeout = mv_gtw_tx_timeout;
	dev->watchdog_timeo = 5*HZ;
        dev->tx_queue_len = gbe_desc_num_per_txq[EGIGA_DEF_TXQ];
	dev->set_mac_address = mv_gtw_set_mac_addr;
	dev->set_multicast_list = mv_gtw_set_multicast_list;
	dev->poll = &mv_gtw_poll;
        dev->weight = 64;
	dev->get_stats = mv_gtw_get_stats;
#ifdef TX_CSUM_OFFLOAD
    dev->features = NETIF_F_SG | NETIF_F_IP_CSUM;
#endif
	if(register_netdev(dev)) {
	    printk(KERN_ERR "failed to register %s\n",dev->name);
	}
	else {
	    printk("%s ",dev->name);
    	}
    }
    main_net_dev = NULL;
    printk("\n");

    /* connect to port interrupt line */
    if( request_irq(dev->irq, mv_gtw_interrupt_handler, (SA_INTERRUPT | SA_SAMPLE_RANDOM), "mv_gateway", NULL ) ) {
        printk(KERN_ERR "failed to assign irq%d\n", dev->irq);
        dev->irq = 0;
    }

    /* unmask interrupts */
    MV_REG_WRITE( ETH_INTR_MASK_REG(EGIGA_DEF_PORT), EGIGA_PICR_MASK );
    MV_REG_WRITE( ETH_INTR_MASK_EXT_REG(EGIGA_DEF_PORT), EGIGA_PICER_MASK );
    gbe_dev.rxmask = EGIGA_PICR_MASK;
    gbe_dev.txmask = EGIGA_PICER_MASK;

#ifdef CONFIG_MV_GTW_IGMP
    /* Initialize the IGMP snooping handler */
    if(mv_gtw_igmp_snoop_init()) {
        printk("failed to init IGMP snooping handler\n");
    }
#endif 

    return 0;
}

static void __init mv_gtw_exit_module(void) 
{
    printk("Removing Marvell Gateway Driver: not implemented\n");
    /* 2do: currently this module is statically linked */

#ifdef CONFIG_MV_GTW_IGMP
    /* Release the IGMP snooping handler */
    if (mv_gtw_igmp_snoop_exit())
	printk("failed to exit IGMP snooping handler\n");
#endif      
}

static struct net_device* mv_gtw_main_net_dev_get(void)
{
    int i;

    for(i=0; i<MV_NUM_OF_IFS; i++) {
        if(netif_running(mv_privs[i].net_dev)) {
            return mv_privs[i].net_dev;
	}
    }
    return NULL;
}

static int mv_gtw_start( struct net_device *dev ) 
{
    mv_priv *mvp = (mv_priv *)dev->priv;
    struct mv_vlan_cfg *vlan_cfg = mvp->vlan_cfg;
    unsigned char broadcast[6] = {0xff,0xff,0xff,0xff,0xff,0xff};

    printk("mv_gateway: starting %s\n",dev->name);

    /* start upper layer */
    netif_carrier_on(dev);
    netif_wake_queue(dev);
    netif_poll_enable(dev);

    /* Add our MAC addr to the VALN DB at switch level to forward packets with this DA */
    /* to CPU port by using the tunneling feature. GbE is always in promisc mode.      */
    mv_gtw_set_mac_addr_to_switch(dev->dev_addr,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),(1<<QD_PORT_CPU),1);

    /* We also need to allow L2 broadcasts comming up for this interface */
    mv_gtw_set_mac_addr_to_switch(broadcast,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),vlan_cfg->ports_mask|(1<<QD_PORT_CPU),1);

#ifdef CONFIG_MV_GTW_QOS
    /* In order to coexist with QoS filtering at gigabit level, we must have a match between  */
    /* incomming packet DA and the mac addr (bits 47:4) set in gigabit level. Since different */
    /* interfaces has different MAC addresses, only the WAN interface benefits from QoS.      */
    if(strcmp(dev->name,CONFIG_MV_GTW_QOS_NET_IF)==0) {
        GTW_DBG(GTW_DBG_MACADDR, ("mv_gateway: %s add %02x %02x %02x %02x %02x %02x also to gigabit level\n", dev->name,
	*(dev->dev_addr), *(dev->dev_addr+1), *(dev->dev_addr+2), *(dev->dev_addr+3), *(dev->dev_addr+4), *(dev->dev_addr+5)));
	mvEthMacAddrSet(gbe_dev.hal_priv, dev->dev_addr, EGIGA_DEF_RXQ);	
    }
#endif
    
    main_net_dev = mv_gtw_main_net_dev_get();

    return 0;
}

static int mv_gtw_stop( struct net_device *dev )
{
    mv_priv *mvp = (mv_priv *)dev->priv;
    struct mv_vlan_cfg *vlan_cfg = mvp->vlan_cfg;
    unsigned char broadcast[6] = {0xff,0xff,0xff,0xff,0xff,0xff};

    printk("mv_gateway: stoping %s\n",dev->name);

    /* stop upper layer */
    netif_poll_disable(dev);
    netif_carrier_off(dev);
    netif_stop_queue(dev);

    /* delete our mac addr and broadcast addr from this DB at switch level */
    mv_gtw_set_mac_addr_to_switch(dev->dev_addr,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),(1<<QD_PORT_CPU),0);
    mv_gtw_set_mac_addr_to_switch(broadcast,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),vlan_cfg->ports_mask|(1<<QD_PORT_CPU),0);

#ifdef CONFIG_MV_GTW_QOS
    /* delete also from gigabit level */
    if(strcmp(dev->name,CONFIG_MV_GTW_QOS_NET_IF)==0) {
        GTW_DBG(GTW_DBG_MACADDR, ("mv_gateway: %s delete %02x %02x %02x %02x %02x %02x also from gigabit level\n", dev->name,
	*(dev->dev_addr), *(dev->dev_addr+1), *(dev->dev_addr+2), *(dev->dev_addr+3), *(dev->dev_addr+4), *(dev->dev_addr+5)));
	mvEthMacAddrSet(gbe_dev.hal_priv, dev->dev_addr, -1);
    }
#endif

    main_net_dev = mv_gtw_main_net_dev_get();

    return 0;
}

static void mv_gtw_stop_netif(void)
{
    int i;
    unsigned long flags;
    local_irq_save(flags);
    GTW_DBG( GTW_DBG_TX | GTW_DBG_TX_DONE, ("mv_gateway: stopping netif transmission\n"));
    EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_netif_stop[queue]++) );
    printk(KERN_INFO "mv_gateway: stopping netif(s) transmission (gbe ring full)\n");
    for(i=0; i<MV_NUM_OF_IFS; i++)
	netif_stop_queue(mv_privs[i].net_dev);
    gbe_dev.netif_stopped = 1;
    local_irq_restore(flags);
}

static void mv_gtw_wakeup_netif(void)
{
    int i;
    unsigned long flags;
    local_irq_save(flags);
    GTW_DBG( GTW_DBG_TX | GTW_DBG_TX_DONE, ("mv_gateway: waking up netif transmission\n"));
    EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_netif_wake[queue]++) );
    printk(KERN_INFO "mv_gateway: waking up netif transmission\n");
    for(i=0; i<MV_NUM_OF_IFS; i++)
	if((netif_queue_stopped(mv_privs[i].net_dev)) && (mv_privs[i].net_dev->flags & IFF_UP))
	    netif_wake_queue(mv_privs[i].net_dev);
    gbe_dev.netif_stopped = 0;
    local_irq_restore(flags);
}

static int init_config(void)
{
    struct mv_vlan_cfg *nc;
    int i;
    MV_U32 board_id = mvBoardIdGet();

    /* switch ports on-board mapping */
    switch(board_id) {
	case RD_88F5181L_VOIP_GE:
	    printk("Detected RD_88F5181L_VOIP_GE\n");
	    QD_PORT_CPU = GE_QD_PORT_CPU;
	    QD_PORT_0   = GE_QD_PORT_0;
	    QD_PORT_1   = GE_QD_PORT_1;
	    QD_PORT_2   = GE_QD_PORT_2;
	    QD_PORT_3   = GE_QD_PORT_3;
	    QD_PORT_4   = GE_QD_PORT_4;
	    break;
	case RD_88F5181L_VOIP_FE:
	    printk("Detected RD_88F5181L_VOIP_FE\n");
	    QD_PORT_CPU = FE_QD_PORT_CPU;
	    QD_PORT_0   = FE_QD_PORT_0;
	    QD_PORT_1   = FE_QD_PORT_1;
	    QD_PORT_2   = FE_QD_PORT_2;
	    QD_PORT_3   = FE_QD_PORT_3;
	    QD_PORT_4   = FE_QD_PORT_4;
	    break;
	default:
	    printk("Unsupported platform.\n");
	    return -1;
    }

#if defined(TX_CSUM_OFFLOAD) && defined(RX_CSUM_OFFLOAD)
    printk("L3/L4 Checksum offload enabled\n");
#else
#if defined(RX_CSUM_OFFLOAD)
    printk("L3/L4 Receive checksum offload enabled\n");
#endif
#if defined(TX_CSUM_OFFLOAD)
    printk("L3/L4 Transmit checksum offload enabled\n");
#endif
#endif

#ifdef INCLUDE_MULTI_QUEUE
    printk("GbE Multi-queue support ( ");
    for(i=0;i<MV_ETH_RX_Q_NUM;i++)
	printk("rxq%d=%d ",i,gbe_desc_num_per_rxq[i]);
    for(i=0;i<MV_ETH_TX_Q_NUM;i++)
	printk("txq%d=%d ",i,gbe_desc_num_per_txq[i]);
    printk(")\n");
#endif

#ifdef CONFIG_MV_GTW_QOS_VOIP
    printk("VoIP QoS support (ToS %s)\n", CONFIG_MV_GTW_QOS_VOIP_TOS);
#endif

#ifdef CONFIG_MV_GTW_QOS_VIDEO
    printk("Video routing QoS support (ToS %s)\n", CONFIG_MV_GTW_QOS_VIDEO_TOS);
#endif

#ifdef CONFIG_MV_GTW_IGMP
    printk("L2 IGMP support\n");
#endif

#ifdef EGIGA_STATISTICS
    printk("Statistics enabled\n");
#endif

#ifdef MV_GTW_DEBUG
    printk("Debug messages enabled\n");
#endif

#ifdef CONFIG_MV_GTW_PROC
    printk("Proc tool API enabled\n");
#endif

    /* build the net config table */
    memset(net_cfg,0xff,sizeof(net_cfg));;
    for(i=0, nc=net_cfg; i<MV_NUM_OF_IFS; i++,nc++) {

	nc->vlan_grp_id = MV_GTW_GROUP_VLAN_ID(i);

	if(i==0){
	    char tmp_addr[MAX_ADDR_LEN];
	    sprintf(nc->name,MV_IF0_NAME);
	    mvEthMacAddrGet(EGIGA_DEF_PORT, tmp_addr);
	    if(overEthAddr) {
		nc->macaddr = MV_IF0_MAC_ADDR;
		printk("Overriding %s boot MAC address to %s\n",nc->name,nc->macaddr);
	    }
	    else {
		sprintf(boot_mac_addr, "%02x:%02x:%02x:%02x:%02x:%02x",
		    tmp_addr[0],tmp_addr[1],tmp_addr[2],tmp_addr[3],tmp_addr[4],tmp_addr[5]);
		nc->macaddr = boot_mac_addr;
		printk("Keeping boot MAC address %s for %s\n",nc->macaddr,nc->name);
	    }
	    switch(board_id) {
		case RD_88F5181L_VOIP_GE:
		    nc->ports_mask = MV_IF0_VLAN_PORTS_GE;
		    break;
		case RD_88F5181L_VOIP_FE:
		    nc->ports_mask = MV_IF0_VLAN_PORTS_FE;
		    break;
		default:
		    printk("mv_gateway failed to set VLAN ports for %s.\n",nc->name);
		    return -1;
	    }
	}
	else if(i==1){
	    sprintf(nc->name,MV_IF1_NAME);
	    nc->macaddr = MV_IF1_MAC_ADDR;
	    switch(board_id) {
		case RD_88F5181L_VOIP_GE:
		    nc->ports_mask = MV_IF1_VLAN_PORTS_GE;
		    break;
		case RD_88F5181L_VOIP_FE:
		    nc->ports_mask = MV_IF1_VLAN_PORTS_FE;
		    break;
		default:
		    printk("mv_gateway failed to set VLAN ports for %s.\n",nc->name);
		    return -1;
	    }
	}
#if (MV_NUM_OF_IFS==3)
	else if(i==2){
	    sprintf(nc->name,MV_IF2_NAME);
	    nc->macaddr = MV_IF2_MAC_ADDR;
	    switch(board_id) {
		case RD_88F5181L_VOIP_GE:
		    nc->ports_mask = MV_IF2_VLAN_PORTS_GE;
		    break;
		case RD_88F5181L_VOIP_FE:
		    nc->ports_mask = MV_IF2_VLAN_PORTS_FE;
		    break;
		default:
		    printk("mv_gateway failed to set VLAN ports for %s.\n",nc->name);
		    return -1;
	    }
	}
#endif
	else{
	    printk("mv_gateway failed to configure network interface #%d\n",i);
	}
	
	printk("%s: mac_addr %s, group-id 0x%03x, group-members are ", nc->name,nc->macaddr,nc->vlan_grp_id);
	if(nc->ports_mask & (1<<QD_PORT_CPU))
	    printk("port-CPU ");
	if(nc->ports_mask & (1<<QD_PORT_0))
	    printk("port-0 ");
	if(nc->ports_mask & (1<<QD_PORT_1))
	    printk("port-1 ");
	if(nc->ports_mask & (1<<QD_PORT_2))
	    printk("port-2 ");
	if(nc->ports_mask & (1<<QD_PORT_3))
	    printk("port-3 ");
	if(nc->ports_mask & (1<<QD_PORT_4))
	    printk("port-4 ");
	printk("\n");
    }

    return 0;
}
static int init_switch(void)
{
    unsigned int i, p;
    unsigned char cnt;
    GT_LPORT port_list[MAX_SWITCH_PORTS];
    struct mv_vlan_cfg *nc;
    MV_U32 board_id = mvBoardIdGet();

    printk("init switch layer... ");

    memset((char*)&qd_cfg,0,sizeof(GT_SYS_CONFIG));

    /* init config structure for qd package */
    qd_cfg.BSPFunctions.readMii   = ReadMiiWrap;
    qd_cfg.BSPFunctions.writeMii  = WriteMiiWrap;
    qd_cfg.BSPFunctions.semCreate = NULL;
    qd_cfg.BSPFunctions.semDelete = NULL;
    qd_cfg.BSPFunctions.semTake   = NULL;
    qd_cfg.BSPFunctions.semGive   = NULL;
    qd_cfg.initPorts = GT_TRUE;
    qd_cfg.cpuPortNum = QD_PORT_CPU;
    if(board_id == RD_88F5181L_VOIP_GE) {
        qd_cfg.mode.scanMode = SMI_MANUAL_MODE;
    }

    /* load switch sw package */
    if( qdLoadDriver(&qd_cfg, &qddev) != GT_OK) {
	printk("qdLoadDriver failed\n");
        return -1;
    }
    qd_dev = &qddev;
    GTW_DBG( GTW_DBG_LOAD, ("qd_dev->numOfPorts=%d\n",qd_dev->numOfPorts));

    /* clear the zero pad buffer */
    memset(zero_pad,0,QD_MIN_ETH_PACKET_LEN);

    /* disable all ports */
    for(p=0; p<qd_dev->numOfPorts; p++) {
	gstpSetPortState(qd_dev, p, GT_PORT_DISABLE);
    }

    /* special board settings */
    switch(board_id) {

	case RD_88F5181L_VOIP_GE:
	    /* check switch id first */
	    if((qd_dev->deviceId != GT_88E6131) && (qd_dev->deviceId != GT_88E6108))
		printk("Unsupported switch id 0x%x. Trying anyway...\n",qd_dev->deviceId);
	    /* enable external ports */
	    GTW_DBG( GTW_DBG_LOAD, ("enable phy polling for external ports\n"));
	    if(gsysSetPPUEn(qd_dev, GT_TRUE) != GT_OK) {
		printk("gsysSetPPUEn failed\n");
		return -1;
	    }
	    /* set cpu-port with ingress double-tag mode */
	    GTW_DBG( GTW_DBG_LOAD, ("cpu port ingress double-tag mode\n"));
	    if(gprtSetDoubleTag(qd_dev, QD_PORT_CPU, GT_TRUE) != GT_OK) {
	        printk("gprtSetDoubleTag failed\n");
	        return -1;
	    }
	    /* set cpu-port with egrees add-tag mode */
	    GTW_DBG( GTW_DBG_LOAD, ("cpu port egrees add-tag mode\n"));
	    if(gprtSetEgressMode(qd_dev, QD_PORT_CPU, GT_ADD_TAG) != GT_OK) {
		printk("gprtSetEgressMode failed\n");
		return -1;
	    }
	    /* config the switch to use the double tag data (relevant to cpu-port only) */
	    GTW_DBG( GTW_DBG_LOAD, ("use double-tag and remove\n"));
	    if(gsysSetUseDoubleTagData(qd_dev,GT_TRUE) != GT_OK) {
	        printk("gsysSetUseDoubleTagData failed\n");
	        return -1;
	    }
	    /* set cpu-port with 802.1q secured mode */
	    GTW_DBG( GTW_DBG_LOAD, ("cpu port-based 802.1q secure mode\n"));
	    if(gvlnSetPortVlanDot1qMode(qd_dev,QD_PORT_CPU,GT_SECURE) != GT_OK) {
		printk("gvlnSetPortVlanDot1qMode failed\n"); 
	        return -1;
	    }
	    break;

	case RD_88F5181L_VOIP_FE:
	    /* check switch id first */
	    if((qd_dev->deviceId != GT_88E6065) && (qd_dev->deviceId != GT_88E6061))
		printk("Unsupported switch id 0x%x. Trying anyway...\n",qd_dev->deviceId);
	    /* set CPU port number */
            if(gsysSetCPUPort(qd_dev, QD_PORT_CPU) != GT_OK) {
	        printk("gsysSetCPUPort failed\n");
	        return -1;
	    }
	    if(qd_dev->deviceId == GT_88E6065) {
		/* Set Provider tag id = 0x8100 to support ingress double tagging */
		if(gprtSetProviderTag(qd_dev, QD_PORT_CPU, 0x8100) != GT_OK) {
		    printk("gprtSetProviderTag failed\n");
		    return -1;
		}
		/* set cpu-port with egrees add-tag mode */
		GTW_DBG( GTW_DBG_LOAD, ("cpu port egrees add-tag mode\n"));
		if(gprtSetEgressMode(qd_dev, QD_PORT_CPU, GT_ADD_TAG) != GT_OK) {
		    printk("gprtSetEgressMode add-tag failed\n");
		    return -1;
		}
	        /* set cpu-port with 802.1q secured mode */
	        GTW_DBG( GTW_DBG_LOAD, ("cpu port-based 802.1q secure mode\n"));
	        if(gvlnSetPortVlanDot1qMode(qd_dev,QD_PORT_CPU,GT_SECURE) != GT_OK) {
	            printk("gvlnSetPortVlanDot1qMode failed\n"); 
	            return -1;
	        }
		if(gstatsFlushAll(qd_dev) != GT_OK)
		    printk("gstatsFlushAll failed\n");
	    }
	    else if(qd_dev->deviceId == GT_88E6061) {
		/* set cpu-port with egrees single tag mode (no support for double tagging) */
		GTW_DBG( GTW_DBG_LOAD, ("cpu port egrees single tag mode\n"));
		if(gprtSetEgressMode(qd_dev, QD_PORT_CPU, GT_TAGGED_EGRESS) != GT_OK) {
		    printk("gprtSetEgressMode single-tag failed\n");
		    return -1;
		}
		/* set all other ports to remove the vlan tag on egress */
		for(i=0; i<qd_dev->numOfPorts; i++) {
		    if(i != QD_PORT_CPU) {
			GTW_DBG( GTW_DBG_LOAD, ("port %d remove tag on egrees\n",i));
			if(gprtSetEgressMode(qd_dev, i, GT_UNTAGGED_EGRESS) != GT_OK) {
			    printk("gprtSetEgressMode remove-tag failed\n");
			    return -1;
			}
		    }
		}
	        /* set cpu-port with 802.1q secured mode */
	        GTW_DBG( GTW_DBG_LOAD, ("cpu port-based 802.1q secure mode\n"));
	        if(gvlnSetPortVlanDot1qMode(qd_dev,QD_PORT_CPU,GT_SECURE) != GT_OK) {
	            printk("gvlnSetPortVlanDot1qMode secure mode failed\n"); 
	            return -1;
	        }
		/* set all other ports with 802.1q fall back mode */
		for(i=0; i<qd_dev->numOfPorts; i++) {
		    if(i != QD_PORT_CPU) {
			if(gvlnSetPortVlanDot1qMode(qd_dev,i,GT_FALLBACK) != GT_OK) {
			    printk("gvlnSetPortVlanDot1qMode fall back failed\n"); 
			    return -1;
			}
		    }
		}
	    }
	    /* init counters */
	    if(gprtClearAllCtr(qd_dev) != GT_OK)
	        printk("gprtClearAllCtr failed\n"); 
	    if(gprtSetCtrMode(qd_dev, GT_CTR_ALL) != GT_OK)
	        printk("gprtSetCtrMode failed\n"); 
	    break;

	default:
	    printk("Error: unsupported board id 0x%x\n",board_id);
	    return -1;
    }

    /* set priorities rules */
    for(i=0; i<qd_dev->numOfPorts; i++) {
        /* default port priority to queue zero */
	if(gcosSetPortDefaultTc(qd_dev, i, 0) != GT_OK)
	    printk("gcosSetPortDefaultTc failed (port %d)\n", i);
        /* enable IP TOS Prio */
	if(gqosIpPrioMapEn(qd_dev, i, GT_TRUE) != GT_OK)
	    printk("gqosIpPrioMapEn failed (port %d)\n",i); 
	/* set IP QoS */
	if(gqosSetPrioMapRule(qd_dev, i, GT_FALSE) != GT_OK)
	    printk("gqosSetPrioMapRule failed (port %d)\n",i); 
        /* disable Vlan QoS Prio */
	if(gqosUserPrioMapEn(qd_dev, i, GT_FALSE) != GT_OK)
	    printk("gqosUserPrioMapEn failed (port %d)\n",i); 
        /* Force flow control for all ports */
	if(gprtSetForceFc(qd_dev, i, GT_FALSE) != GT_OK)
	    printk("gprtSetForceFc failed (port %d)\n",i); 
    }

    /* The switch CPU port is not part of the VLAN, but rather connected by tunneling to each */
    /* of the VLAN's ports. Our MAC addr will be added during start operation to the VALN DB  */
    /* at switch level to forward packets with this DA to CPU port.                           */
    GTW_DBG( GTW_DBG_LOAD, ("Enabling Tunneling on ports: "));
    for(i=0; i<qd_dev->numOfPorts; i++) {
	if(i != QD_PORT_CPU) {
	    if(gprtSetVlanTunnel(qd_dev, i, GT_TRUE) != GT_OK) {
		printk("gprtSetVlanTunnel failed (port %d)\n",i); 
		return -1;
	    }
	    else {
		GTW_DBG( GTW_DBG_LOAD, ("%d ",i));
	    }
	}
    }
    GTW_DBG( GTW_DBG_LOAD, ("\n"));

    /* configure ports (excluding CPU port) for each network interface (VLAN): */
    for(i=0, nc=net_cfg; i<MV_NUM_OF_IFS; i++,nc++) {
	GTW_DBG( GTW_DBG_LOAD, ("%s configuration\n", nc->name));

	/* set port's defaul private vlan id and database number (DB per group): */
	for(p=0; p<qd_dev->numOfPorts; p++) {
	    if( MV_BIT_CHECK(nc->ports_mask, p) && (p != QD_PORT_CPU) ) {
		GTW_DBG( GTW_DBG_LOAD, ("port %d default private vlan id: 0x%x\n", p, MV_GTW_PORT_VLAN_ID(nc->vlan_grp_id,p)));
		if( gvlnSetPortVid(qd_dev, p, MV_GTW_PORT_VLAN_ID(nc->vlan_grp_id,p)) != GT_OK ) {
			printk("gvlnSetPortVid failed"); 
			return -1;
		}
		if( gvlnSetPortVlanDBNum(qd_dev, p, MV_GTW_VLANID_TO_GROUP(nc->vlan_grp_id)) != GT_OK) {
		    printk("gvlnSetPortVlanDBNum failed\n");
		    return -1;
		}
	    }
	}

	/* set port's port-based vlan (CPU port is not part of VLAN) */
        if(mv_gtw_set_port_based_vlan(nc->ports_mask & ~(1<<QD_PORT_CPU)) != 0) {
	    printk("mv_gtw_set_port_based_vlan failed\n");
	}

        /* set vtu with group vlan id (used in tx) */
        if(mv_gtw_set_vlan_in_vtu(nc->vlan_grp_id,nc->ports_mask|(1<<QD_PORT_CPU)) != 0) {
	    printk("mv_gtw_set_vlan_in_vtu failed\n");
	}

        /* set vtu with each port private vlan id (used in rx) */
 	for(p=0; p<qd_dev->numOfPorts; p++) {
	    if(MV_BIT_CHECK(nc->ports_mask, p) && (p!=QD_PORT_CPU)) {
		if(mv_gtw_set_vlan_in_vtu(MV_GTW_PORT_VLAN_ID(nc->vlan_grp_id,p),nc->ports_mask & ~(1<<QD_PORT_CPU)) != 0) {
		    printk("mv_gtw_set_vlan_in_vtu failed\n");
		}
	    }
	}
    }

    /* set cpu-port with port-based vlan to all other ports */
    GTW_DBG( GTW_DBG_LOAD, ("cpu port-based vlan:"));
    for(p=0,cnt=0; p<qd_dev->numOfPorts; p++) {
        if(p != QD_PORT_CPU) {
	    GTW_DBG( GTW_DBG_LOAD, ("%d ",p));
            port_list[cnt] = p;
            cnt++;
        }
    }
    GTW_DBG( GTW_DBG_LOAD, ("\n"));
    if( gvlnSetPortVlanPorts(qd_dev, QD_PORT_CPU, port_list, cnt) != GT_OK) {
        printk("gvlnSetPortVlanPorts failed\n"); 
        return -1;
    }


#ifdef CONFIG_MV_GTW_QOS
    mv_gtw_set_qos_in_switch();
#endif

    if(gfdbFlush(qd_dev,GT_FLUSH_ALL) != GT_OK) {
	printk("gfdbFlush failed\n");
    }

#if 0
    /* rate limit simulation - 32M egress on WAN port */
    if(grcSetEgressRate(qd_dev,QD_PORT_0,GT_32M) != GT_OK)
	printk("mv_gateway failed to simulate 32M upstream on WAN port\n");
#endif

    /* done! enable all ports */
    GTW_DBG( GTW_DBG_LOAD, ("enabling: ports "));
    for(p=0; p<qd_dev->numOfPorts; p++) {
	GTW_DBG( GTW_DBG_LOAD, ("%d ",p));
	if(gstpSetPortState(qd_dev, p, GT_PORT_FORWARDING) != GT_OK) {
	    printk("gstpSetPortState failed\n");
	}
    }
    GTW_DBG( GTW_DBG_LOAD, ("\n"));

#if 0
for(i=0; i<qd_dev->numOfPorts; i++) {
    printk("port %d reg 0x4: 0x%x\n",i,mv_gtw_read_switch_reg(i, 0x4));
    printk("port %d reg 0x6: 0x%x\n",i,mv_gtw_read_switch_reg(i, 0x6));
    printk("port %d reg 0x8: 0x%x\n",i,mv_gtw_read_switch_reg(i, 0x8));
}
#endif

    printk("done\n");

    return 0;
}

static int mv_gtw_set_qos_in_switch(void)
{
    /* The ToS value must be represented as follow: "0xVV;0xYY..."        */
    unsigned char tos = 0;
    char *str;
    int i;

    /* Clear all */
    for(i=0; i<=0x40; i++) {
        if(gcosSetDscp2Tc(qd_dev, i, 0) != GT_OK) {
	    printk("gcosSetDscp2Tc failed\n");
	}
    }

#ifdef CONFIG_MV_GTW_QOS_VOIP
    /* VoIP support only one ToS value */
    str = CONFIG_MV_GTW_QOS_VOIP_TOS;
    tos = (mv_gtw_str_to_hex(str[2])<<4) + (mv_gtw_str_to_hex(str[3]));
    if(gcosSetDscp2Tc(qd_dev, tos>>2, MV_VOIP_PRIO) != GT_OK) {
	printk("gcosSetDscp2Tc failed\n");
    }
    GTW_DBG( GTW_DBG_LOAD, ("VoIP ToS 0x%x goes to queue %d\n",tos,MV_VOIP_PRIO));
#endif
#ifdef CONFIG_MV_GTW_QOS_VIDEO
    /* Video support multiple ToS values */
    for(i=0;;i+=5) {
	str = CONFIG_MV_GTW_QOS_VIDEO_TOS + i;
	tos = (mv_gtw_str_to_hex(str[2])<<4) + (mv_gtw_str_to_hex(str[3]));
	if(gcosSetDscp2Tc(qd_dev, tos>>2, MV_VIDEO_PRIO) != GT_OK) {
	   printk("gcosSetDscp2Tc failed\n");
	}
	GTW_DBG( GTW_DBG_LOAD, ("Video ToS 0x%x goes to queue %d\n",tos,MV_VIDEO_PRIO));
	if(str[4] != ';')
	    break;
    }
#endif
#if 0
    /* ARP queueing */
#endif
}

static int init_gigabit(void)
{
    MV_ETH_PORT_INIT hal_init_struct;
    int i;
    MV_STATUS status;

    printk("init gigabit layer... ");

    mvEthInit();

    hal_init_struct.maxRxPktSize = MV_MTU;
    hal_init_struct.rxDefQ = EGIGA_DEF_RXQ; 
    memcpy(hal_init_struct.rxDescrNum,  gbe_desc_num_per_rxq, sizeof(gbe_desc_num_per_rxq));
    memcpy(hal_init_struct.txDescrNum,  gbe_desc_num_per_txq, sizeof(gbe_desc_num_per_txq));
    gbe_dev.hal_priv = mvEthPortInit(EGIGA_DEF_PORT, &hal_init_struct);
    if(!gbe_dev.hal_priv) {
        printk(KERN_ERR "failed to init gigabit hal\n");
	return -1;
    }
    spin_lock_init( &gbe_dev.lock );
    gbe_dev.netif_stopped = 0;
    memset( &gbe_dev.rx_fill_timer, 0, sizeof(struct timer_list) );
    gbe_dev.rx_fill_timer.function = mv_gtw_rx_fill_on_timeout;
    gbe_dev.rx_fill_timer.data = -1;
    gbe_dev.rx_fill_flag = 0;

#ifdef INCLUDE_MULTI_QUEUE
    /* init rx policy */
    gbe_dev.rx_policy_priv = mvEthRxPolicyInit(EGIGA_DEF_PORT, EGIGA_RX_QUEUE_QUOTA, MV_ETH_PRIO_FIXED);
    if(!gbe_dev.rx_policy_priv) {
        printk(KERN_ERR "failed to init rx policy\n");
    }
#ifdef CONFIG_MV_GTW_QOS_VOIP
    /* set VoIP packets (marked with vlan-prio 3) to gigabit queue 3 */
    if(mvEthVlanPrioRxQueue(gbe_dev.hal_priv,(MV_VOIP_PRIO<<1),MV_VOIP_PRIO) != MV_OK) {
        printk(KERN_ERR "failed to priorities GbE VoIP VlanPrio=%d (queue %d)\n",(MV_VOIP_PRIO<<1),MV_VOIP_PRIO);
    }
#endif
#ifdef CONFIG_MV_GTW_QOS_VIDEO
    /* set Video packets (marked with vlan-prio 2) to gigabit queue 2 */
    if(mvEthVlanPrioRxQueue(gbe_dev.hal_priv,(MV_VIDEO_PRIO<<1),MV_VIDEO_PRIO) != MV_OK) {
        printk(KERN_ERR "failed to priorities GbE Video VlanPrio=%d (queue %d)\n",(MV_VIDEO_PRIO<<1),MV_VIDEO_PRIO);
    }
#endif
#if 0
    /* set ARP packets (marked with vlan-prio 1) to gigabit queue 1 */
    if(mvEthVlanPrioRxQueue(gbe_dev.hal_priv,(MV_ARP_PRIO<<1),MV_ARP_PRIO) != MV_OK) {
        printk(KERN_ERR "failed to priorities VlanPrio=%d (queue %d)\n",(MV_ARP_PRIO << 1), MV_ARP_PRIO);
    }
#endif
#endif /* INCLUDE_MULTI_QUEUE */

    /* set promisc mode in gigabit level */
    GTW_DBG( GTW_DBG_LOAD, ("Gigabit in promiscuos mode\n"));
    mvEthRxFilterModeSet(gbe_dev.hal_priv, 1);

    /* fill rx ring with buffers */
    for(i = 0; i < MV_ETH_RX_Q_NUM; i++) {
    	mv_gtw_rx_fill(i, gbe_desc_num_per_rxq[i]);
    }

    /* start the hal - rx/tx activity */
    status = mvEthPortEnable( gbe_dev.hal_priv );
    if( (status != MV_OK) && (status != MV_NOT_READY)){
         printk(KERN_ERR "ethPortEnable failed");
	 return -1;
    }

    /* set tx/rx coalescing mechanism */
    gbe_dev.tx_coal = mvEthTxCoalSet( gbe_dev.hal_priv, EGIGA_TX_COAL );
    gbe_dev.rx_coal = mvEthRxCoalSet( gbe_dev.hal_priv, EGIGA_RX_COAL );

    /* clear and mask interrupts */
    MV_REG_WRITE( ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT), 0 );
    MV_REG_WRITE( ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT), 0 );
    gbe_dev.rxcause = 0;
    gbe_dev.txcause = 0;
    MV_REG_WRITE( ETH_INTR_MASK_REG(EGIGA_DEF_PORT), 0 );
    MV_REG_WRITE( ETH_INTR_MASK_EXT_REG(EGIGA_DEF_PORT), 0 );
    gbe_dev.rxmask = 0;
    gbe_dev.txmask = 0;

    printk("done\n");
    return 0;
}


static int mv_gtw_tx( struct sk_buff *skb , struct net_device *dev )
{
    mv_priv *mvp = dev->priv;
    unsigned short grp_vlan_id = mvp->vlan_cfg->vlan_grp_id; 
    struct net_device_stats *stats = &(mvp->net_dev_stat);
    unsigned long flags;
    MV_STATUS status;
    int ret = 0, i, queue;
    unsigned int vlan_tag;

    if( netif_queue_stopped( dev ) ) {
        printk( KERN_ERR "%s: transmitting while stopped\n", dev->name );
        return 1;
    }

#if LINUX_VERSION_CODE >= KERNEL_VERSION(2,6,9)
    local_irq_save(flags);
    if(unlikely(!spin_trylock(&gbe_dev.lock))) {
    	/* Collision - tell upper layer to requeue */
        local_irq_restore(flags);
        return NETDEV_TX_LOCKED;
    }
#else
    spin_lock_irqsave( &gbe_dev.lock, flags );
#endif

    GTW_DBG( GTW_DBG_TX, ("%s: tx, #%d frag(s), csum by %s\n",
             dev->name, skb_shinfo(skb)->nr_frags+1, (skb->ip_summed==CHECKSUM_HW)?"HW":"CPU") );
    EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_events++) );

    /* basic init of pkt_info. first cell in buf_info_arr is left for header prepending if necessary */
    gbe_dev.tx_pkt_info.osInfo = (MV_ULONG)skb;
    gbe_dev.tx_pkt_info.pktSize = skb->len;
    gbe_dev.tx_pkt_info.pFrags = &gbe_dev.tx_buf_info_arr[0];
    gbe_dev.tx_pkt_info.numFrags = 0;
    gbe_dev.tx_pkt_info.status = 0;
    
    /* see if this is a single/multiple buffered skb */
    if(skb_shinfo(skb)->nr_frags == 0) {
        gbe_dev.tx_pkt_info.pFrags->bufVirtPtr = skb->data;
        gbe_dev.tx_pkt_info.pFrags->bufSize = skb->len;
        gbe_dev.tx_pkt_info.numFrags = 1;
    }
    else {
        MV_BUF_INFO *p_buf_info = gbe_dev.tx_pkt_info.pFrags;

        /* first skb fragment */
        p_buf_info->bufSize = skb_headlen(skb);
        p_buf_info->bufVirtPtr = skb->data;
        p_buf_info++;

        /* now handle all other skb fragments */
        for ( i = 0; i < skb_shinfo(skb)->nr_frags; i++ ) {

            skb_frag_t *frag = &skb_shinfo(skb)->frags[i];

            p_buf_info->bufSize = frag->size;
            p_buf_info->bufVirtPtr = page_address(frag->page) + frag->page_offset;
            p_buf_info++;
        }

        gbe_dev.tx_pkt_info.numFrags = skb_shinfo(skb)->nr_frags + 1;
    }

#ifdef TX_CSUM_OFFLOAD
    /* if HW is suppose to offload layer4 checksum, set some bits in the first buf_info command */
    if(skb->ip_summed == CHECKSUM_HW) {
        GTW_DBG( GTW_DBG_TX, ("%s: tx csum offload\n", dev->name) );
        /*EGIGA_STAT( EGIGA_STAT_TX, Add counter here );*/
        gbe_dev.tx_pkt_info.status =
        ETH_TX_IP_NO_FRAG |           /* we do not handle fragmented IP packets. add check inside iph!! */
        ETH_TX_VLAN_TAGGED_FRAME_MASK |                             /* mark the VLAN tag that we insert */
	((skb->nh.iph->ihl) << ETH_TX_IP_HEADER_LEN_OFFSET) |                            /* 32bit units */
        ((skb->nh.iph->protocol == IPPROTO_TCP) ? ETH_TX_L4_TCP_TYPE : ETH_TX_L4_UDP_TYPE) | /* TCP/UDP */
        ETH_TX_GENERATE_L4_CHKSUM_MASK |                                /* generate layer4 csum command */
        ETH_TX_GENERATE_IP_CHKSUM_BIT;                              /* generate IP csum (already done?) */
    }
    else {
        GTW_DBG( GTW_DBG_TX, ("%s: no tx csum offload\n", dev->name) );
        /*EGIGA_STAT( EGIGA_STAT_TX, Add counter here );*/
        gbe_dev.tx_pkt_info.status = 0x5 << ETH_TX_IP_HEADER_LEN_OFFSET; /* Errata BTS #50 */
    }
#endif

    queue = EGIGA_DEF_TXQ;

    /* add zero pad for small packets */
    if(unlikely(gbe_dev.tx_pkt_info.pktSize < QD_MIN_ETH_PACKET_LEN)) {
        /* add zero pad in last cell of packet in buf_info_arr */
        GTW_DBG(GTW_DBG_TX,("add zero pad\n"));
        gbe_dev.tx_buf_info_arr[gbe_dev.tx_pkt_info.numFrags].bufVirtPtr = zero_pad;
        gbe_dev.tx_buf_info_arr[gbe_dev.tx_pkt_info.numFrags].bufSize = QD_MIN_ETH_PACKET_LEN - gbe_dev.tx_pkt_info.pktSize;
        gbe_dev.tx_pkt_info.pktSize += QD_MIN_ETH_PACKET_LEN - gbe_dev.tx_pkt_info.pktSize;
        gbe_dev.tx_pkt_info.numFrags++;
    }

    /* add vlan tag by breaking first frag to three parts: */
    /* (1) DA+SA in skb (2) the new tag (3) rest of skb    */
    if(unlikely(gbe_dev.tx_pkt_info.pFrags->bufSize < 12)) {
        /* never happens, but for completeness */
        printk(KERN_ERR "mv_gtw cannot transmit first frag when len < 12\n");
    }
    else {
        MV_BUF_INFO *p_buf_info_first = gbe_dev.tx_pkt_info.pFrags;
	MV_BUF_INFO *p_buf_info_last = gbe_dev.tx_pkt_info.pFrags + gbe_dev.tx_pkt_info.numFrags - 1;

	/* make room for two cells */
        while(p_buf_info_last != p_buf_info_first) {
	    *(p_buf_info_last+2) = *p_buf_info_last;
	    p_buf_info_last--;
	}

	/* the second half of the original first frag */
	(p_buf_info_first+2)->bufSize = p_buf_info_first->bufSize - 12;
	(p_buf_info_first+2)->bufVirtPtr = p_buf_info_first->bufVirtPtr + 12;

	/* the tag (safe on stack) */
	vlan_tag = cpu_to_be32((0x8100 << 16) | grp_vlan_id);
        (p_buf_info_first+1)->bufVirtPtr = (unsigned char*)&vlan_tag;
	(p_buf_info_first+1)->bufSize = QD_VLANTAG_SIZE;

	/* the first half of the original first frag */
        p_buf_info_first->bufSize = 12;

	/* count the new frags */
	gbe_dev.tx_pkt_info.numFrags += 2;
        gbe_dev.tx_pkt_info.pktSize += QD_VLANTAG_SIZE;
    }

    GTW_DBG(GTW_DBG_VLAN, ("%s TX: q=%d, vid=0x%x\n",dev->name,queue,grp_vlan_id));

#if 0
{
unsigned char *data = gbe_dev.tx_pkt_info.pFrags->bufVirtPtr;
printk("TX: [DA = %02x %02x %02x %02x %02x %02x] [SA = %02x %02x %02x %02x %02x %02x] ", 
*(data), *(data+1), *(data+2), *(data+3), *(data+4), *(data+5),
*(data+6), *(data+7), *(data+8), *(data+9), *(data+10), *(data+11));
data = (gbe_dev.tx_pkt_info.pFrags+1)->bufVirtPtr;
printk("[VLAN = %02x %02x %02x %02x] ", *(data), *(data+1), *(data+2), *(data+3));
data = (gbe_dev.tx_pkt_info.pFrags+2)->bufVirtPtr;
printk("[TYPE = %02x %02x %02x %02x]\n", *(data), *(data+1), *(data+2), *(data+3));
}
#endif

    /* now send the packet */
    status = mvEthPortTx( gbe_dev.hal_priv, queue, &gbe_dev.tx_pkt_info );

    /* check status */
    if(likely(status == MV_OK)) {
        stats->tx_bytes += skb->len;
        stats->tx_packets ++;
        dev->trans_start = jiffies;
        gbe_dev.txq_count[queue]++;
        GTW_DBG( GTW_DBG_TX, ("ok (%d); ", gbe_dev.txq_count[queue]) );
        EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_hal_ok[queue]++) );
    }
    else {
        /* tx failed. higher layers will free the skb */
        ret = 1;
        stats->tx_dropped++;

        if(status == MV_NO_RESOURCE) {
            /* it must not happen because we call to netif_stop_queue in advance. */
            GTW_DBG( GTW_DBG_TX, ("%s: queue is full, stop transmit (should never happen here)\n", dev->name) );
	    mv_gtw_stop_netif();
            EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_hal_no_resource[queue]++) );
            EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_netif_stop[queue]++) );
        }
        else if(status == MV_ERROR) {
            printk( KERN_ERR "%s: error on transmit\n", dev->name );
            EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_hal_error[queue]++) );
        }
        else {
            printk( KERN_ERR "%s: unrecognize status on transmit\n", dev->name );
            EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_hal_unrecognize[queue]++) );
        }
    }

    /* if number of available descriptors left is less than MAX_SKB_FRAGS stop the stack. */
    if(unlikely((mvEthTxResourceGet(gbe_dev.hal_priv, queue) - skb_shinfo(skb)->nr_frags) <= MAX_SKB_FRAGS)) {
        GTW_DBG( GTW_DBG_TX, ("%s: stopping network tx interface\n", dev->name) );
	mv_gtw_stop_netif();
    }

    spin_unlock_irqrestore( &gbe_dev.lock, flags );

    return ret;
}


static unsigned int mv_gtw_tx_done(void)
{
    MV_PKT_INFO pkt_info;
    unsigned int count = 0;
    MV_STATUS status;
    unsigned int queue = 0;

    EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_events++) );

    /* read cause once more and clear tx bits */
    gbe_dev.txcause |= MV_REG_READ(ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT));
    MV_REG_WRITE(ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT),gbe_dev.txcause & (~EGIGA_TXQ_MASK) );
    gbe_dev.txcause &= EGIGA_TXQ_MASK;

    /* release the transmitted packets */
    while( 1 ) {

        if(unlikely(!gbe_dev.txcause))
	    break;

        queue = EGIGA_DEF_TXQ;

        /* get a packet */  
        status = mvEthPortTxDone( gbe_dev.hal_priv, queue, &pkt_info );

	if(likely(status == MV_OK)) {

	    gbe_dev.txq_count[queue]--;

	    /* validate skb */
	    if(unlikely(!(pkt_info.osInfo))) {
		EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_invalid_skb[queue]++) );
		continue;
	    }

	    /* handle tx error */
	    if(unlikely(pkt_info.status & (ETH_ERROR_SUMMARY_BIT))) {
	        GTW_DBG( GTW_DBG_TX_DONE, ("bad tx-done status\n") );
		EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_bad_stat[queue]++) );
	    }

	    /* release the skb */
	    dev_kfree_skb( (struct sk_buff *)pkt_info.osInfo );
	    count++;
	    EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_ok[queue]++) );
	    EGIGA_STAT( EGIGA_STAT_TX_DONE, if(gbe_dev.egiga_stat.tx_done_max[queue] < count) gbe_dev.egiga_stat.tx_done_max[queue] = count );
	    EGIGA_STAT( EGIGA_STAT_TX_DONE, if(gbe_dev.egiga_stat.tx_done_min[queue] > count) gbe_dev.egiga_stat.tx_done_min[queue] = count );
	}
	else {
	    if( status == MV_EMPTY ) {
	    	/* no more work */
	    	GTW_DBG( GTW_DBG_TX_DONE, ("no more work ") );
	    	EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_no_more[queue]++) );
	    }
	    else if( status == MV_NOT_FOUND ) {
	    	/* hw still in tx */
	    	GTW_DBG( GTW_DBG_TX_DONE, ("hw still in tx ") );
	    	EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_still_tx[queue]++) );
	    }
	    else {
	    	printk( KERN_ERR "unrecognize status on tx done\n");
	    	EGIGA_STAT( EGIGA_STAT_TX_DONE, (gbe_dev.egiga_stat.tx_done_hal_unrecognize[queue]++) );
	    }

	    gbe_dev.txcause &= ~ETH_CAUSE_TX_BUF_MASK(queue);
	    break;
    	}
    }

    GTW_DBG( GTW_DBG_TX_DONE, ("tx-done %d (%d)\n", count, gbe_dev.txq_count[queue]) );

    return count;
}

static void mv_gtw_tx_timeout( struct net_device *dev ) 
{
    EGIGA_STAT( EGIGA_STAT_TX, (gbe_dev.egiga_stat.tx_timeout++) );
    printk( KERN_INFO "%s: tx timeout\n", dev->name );
}

#ifdef RX_CSUM_OFFLOAD
static MV_STATUS mv_gtw_rx_csum_offload(MV_PKT_INFO *pkt_info)
{
    if( (pkt_info->pktSize > RX_CSUM_MIN_BYTE_COUNT)   && /* Minimum        */
        (pkt_info->status & ETH_RX_IP_FRAME_TYPE_MASK) && /* IPv4 packet    */
        (pkt_info->status & ETH_RX_IP_HEADER_OK_MASK)  && /* IP header OK   */
        (!(pkt_info->fragIP))                          && /* non frag IP    */
        (!(pkt_info->status & ETH_RX_L4_OTHER_TYPE))   && /* L4 is TCP/UDP  */
        (pkt_info->status & ETH_RX_L4_CHECKSUM_OK_MASK) ) /* L4 checksum OK */
            return MV_OK;

    if(!(pkt_info->pktSize > RX_CSUM_MIN_BYTE_COUNT))
        GTW_DBG( GTW_DBG_RX, ("Byte count smaller than %d\n", RX_CSUM_MIN_BYTE_COUNT) );
    if(!(pkt_info->status & ETH_RX_IP_FRAME_TYPE_MASK))
        GTW_DBG( GTW_DBG_RX, ("Unknown L3 protocol\n") );
    if(!(pkt_info->status & ETH_RX_IP_HEADER_OK_MASK))
        GTW_DBG( GTW_DBG_RX, ("Bad IP csum\n") );
    if(pkt_info->fragIP)
        GTW_DBG( GTW_DBG_RX, ("Fragmented IP\n") );
    if(pkt_info->status & ETH_RX_L4_OTHER_TYPE)
        GTW_DBG( GTW_DBG_RX, ("Unknown L4 protocol\n") );
    if(!(pkt_info->status & ETH_RX_L4_CHECKSUM_OK_MASK))
        GTW_DBG( GTW_DBG_RX, ("Bad L4 csum\n") );

    return MV_FAIL;
}
#endif

static unsigned int mv_gtw_rx(unsigned int work_to_do)
{
    mv_priv *mvp;
    struct net_device_stats *stats;
    struct net_device *dev=NULL;
    unsigned int vlan_tag;
    struct sk_buff *skb;
    MV_PKT_INFO pkt_info;
    int work_done = 0;
    MV_STATUS status;
    unsigned int queue = 0;
    unsigned int done_per_q[MV_ETH_RX_Q_NUM] = {0};

#ifdef INCLUDE_MULTI_QUEUE
    unsigned int temp = MV_REG_READ(ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT));
    gbe_dev.rxcause |= temp & EGIGA_RXQ_MASK;
    gbe_dev.rxcause |= (temp & EGIGA_RXQ_RES_MASK) >> (ETH_CAUSE_RX_ERROR_OFFSET - ETH_CAUSE_RX_READY_OFFSET);
    MV_REG_WRITE(ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT), ~(gbe_dev.rxcause | (gbe_dev.rxcause << (ETH_CAUSE_RX_ERROR_OFFSET - ETH_CAUSE_RX_READY_OFFSET)) ) );
    GTW_DBG( GTW_DBG_RX,("%s: cause = 0x%08x\n\n", dev->name, gbe_dev.rxcause) );
#endif /* INCLUDE_MULTI_QUEUE */

    EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_events++) );

#ifdef CONFIG_MV_GTW_QOS_VOIP
    if(mv_gtw_qos_tos_quota != -1) {
        work_to_do = (work_to_do<mv_gtw_qos_tos_quota) ? work_to_do : mv_gtw_qos_tos_quota;
    }
#endif

    GTW_DBG( GTW_DBG_RX, ("^rx work_to_do %d\n",work_to_do) );

    while( work_done < work_to_do ) {

#ifdef INCLUDE_MULTI_QUEUE
        if(!gbe_dev.rxcause)
            break;
        if(gbe_dev.rx_policy_priv)
	    queue = mvEthRxPolicyGet(gbe_dev.rx_policy_priv, gbe_dev.rxcause);
        else
	    queue = EGIGA_DEF_RXQ;
#else
        queue = EGIGA_DEF_RXQ;
#endif /* INCLUDE_MULTI_QUEUE */

        /* get rx packet */ 
	status = mvEthPortRx( gbe_dev.hal_priv, queue, &pkt_info );

        /* check status */
	if(likely(status == MV_OK)) {
	    work_done++;
	    done_per_q[queue]++;
	    gbe_dev.rxq_count[queue]--;
	    EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_ok[queue]++) );
	} else{ 
	    if( status == MV_NO_RESOURCE ) {
	    	/* no buffers for rx */
	    	GTW_DBG( GTW_DBG_RX, ("rx no resource ") );
	    	EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_no_resource[queue]++) );
		TRC_REC("rx no resource\n");
	    } 
	    else if( status == MV_NO_MORE ) {
	    	/* no more rx packets ready */
	    	GTW_DBG( GTW_DBG_RX, ("rx no more ") );
	    	EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_no_more[queue]++) );
		TRC_REC("rx no more\n");
	    } 
	    else {
	    	printk( KERN_ERR "undefined status on rx poll\n");
	    	EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_error[queue]++) );
		TRC_REC("rx undefined status\n");
	    }

#ifdef INCLUDE_MULTI_QUEUE
	    gbe_dev.rxcause &= ~ETH_CAUSE_RX_READY_MASK(queue);
	    continue;
#else
	    break;
#endif
	}

	/* validate skb */ 
	if(unlikely(!(pkt_info.osInfo))) {
	    printk( KERN_ERR "error in rx\n");
	    EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_invalid_skb[queue]++) );
	    continue;
	}

	skb = (struct sk_buff *)( pkt_info.osInfo );

	/* handle rx error */
	if(unlikely(pkt_info.status & (ETH_ERROR_SUMMARY_MASK))) {
            unsigned int err = pkt_info.status & ETH_RX_ERROR_CODE_MASK;
	    TRC_REC("rx general error\n");
            /* RX resource error is likely to happen when receiving packets, which are     */
            /* longer then the Rx buffer size, and they are spreading on multiple buffers. */
            /* Rx resource error - No descriptor in the middle of a frame.                 */
	    if( err == ETH_RX_RESOURCE_ERROR ) {
	        GTW_DBG( GTW_DBG_RX, ("bad rx status %08x, (resource error)",(unsigned int)pkt_info.status));
            }
	    else if( err == ETH_RX_OVERRUN_ERROR ) {
		GTW_DBG( GTW_DBG_RX, ("bad rx status %08x, (overrun error)",(unsigned int)pkt_info.status));
            }
	    else {
		printk( KERN_INFO "bad rx status %08x, ",(unsigned int)pkt_info.status );
	    	if( err == ETH_RX_MAX_FRAME_LEN_ERROR )
	        	printk( KERN_INFO "(max frame length error)" );
	    	else if( err == ETH_RX_CRC_ERROR )
	        	printk( KERN_INFO "(crc error)" );
	    	else
	        	printk( KERN_INFO "(unknown error)" );
	    	printk( KERN_INFO "\n" );
	    }
	    
	    dev_kfree_skb( skb );
	    EGIGA_STAT( EGIGA_STAT_RX, (gbe_dev.egiga_stat.rx_hal_bad_stat[queue]++) );
	    continue;
	}

	/* good rx */
        GTW_DBG( GTW_DBG_RX, ("good rx (skb=%p skb->data=%p skb->len=%d)\n", skb, skb->data, skb->len) );

#if 0
printk("RX: [DA = %02x %02x %02x %02x %02x %02x] [SA = %02x %02x %02x %02x %02x %02x] [VLAN = %02x %02x %02x %02x] [TYPE = %02x %02x %02x %02x]\n", 
*(skb->data), *(skb->data+1), *(skb->data+2), *(skb->data+3), *(skb->data+4), *(skb->data+5),
*(skb->data+6), *(skb->data+7), *(skb->data+8), *(skb->data+9), *(skb->data+10), *(skb->data+11),
*(skb->data+12), *(skb->data+13), *(skb->data+14), *(skb->data+15),
*(skb->data+16), *(skb->data+17), *(skb->data+18), *(skb->data+19));
#endif

#ifdef RX_CSUM_OFFLOAD
        /* checksum offload */
        if( mv_gtw_rx_csum_offload( &pkt_info ) == MV_OK ) {

            GTW_DBG( GTW_DBG_RX, ("%s: rx csum offload ok\n", dev->name) );
            /* EGIGA_STAT( EGIGA_STAT_RX, Add counter here) */

            skb->ip_summed = CHECKSUM_UNNECESSARY;

            /* Is this necessary? */
            skb->csum = htons((pkt_info.status & ETH_RX_L4_CHECKSUM_MASK) >> ETH_RX_L4_CHECKSUM_OFFSET);
        }
        else {
            GTW_DBG( GTW_DBG_RX, ("%s: rx csum offload failed\n", dev->name) );
            /* EGIGA_STAT( EGIGA_STAT_RX, Add counter here) */
            skb->ip_summed = CHECKSUM_NONE;
        }
#else
        skb->ip_summed = CHECKSUM_NONE;
#endif

	/* find our net_device by the vid group id from vlan-tag */
	vlan_tag = ((skb->data[14]<<8) | skb->data[15]);
	mvp = &mv_privs[MV_GTW_VLANID_TO_GROUP(vlan_tag)];
	GTW_DBG( GTW_DBG_RX, ("vid = 0x%x map to dev %s\n",vlan_tag,mvp->net_dev->name) );
	dev = mvp->net_dev;
	stats = &mvp->net_dev_stat;
	stats->rx_packets++;
	stats->rx_bytes += pkt_info.pktSize;
	skb->dev = dev;
	GTW_DBG( GTW_DBG_VLAN, ("%s RX: q=%d, vlan_tag=0x%x, prio=%d\n",dev->name,queue,vlan_tag,(skb->data[14] >> 5)));

	/* set skb length to the total size minus 2 bytes shifted by GbE to align IP header minus 4 byte CRC */
	skb_put(skb, pkt_info.pktSize - 4 - 2);

	/* remove the vlan tag */
	memmove(skb->data+QD_VLANTAG_SIZE,skb->data,12);
	skb_pull(skb, QD_VLANTAG_SIZE);

	skb->protocol = eth_type_trans(skb,dev); 

#ifdef CONFIG_MV_GTW_IGMP
	if( (MV_GTW_VLANID_TO_GROUP(vlan_tag) != 0) && 
	    (ntohs(skb->protocol) == ETH_P_IP) && 
	    (((struct iphdr*)(skb->data))->protocol == IPPROTO_IGMP)) {		
		GTW_DBG( GTW_DBG_IGMP, ("invoke IGMP snooping handler for IGMP message"));
		mv_gtw_igmp_snoop_process(skb, (unsigned char) MV_GTW_VLANID_TO_PORT(vlan_tag), 
					    (unsigned char) MV_GTW_VLANID_TO_GROUP(vlan_tag));		
	}
#endif

	status = netif_receive_skb( skb );
        EGIGA_STAT( EGIGA_STAT_RX, if(status) (gbe_dev.egiga_stat.rx_netif_drop[queue]++) );
    }

    GTW_DBG( GTW_DBG_RX, ("\nwork_done %d (%d)", work_done, gbe_dev.rxq_count[queue]) );

    /* refill rx ring with new buffers */
    for(queue = 0; queue < MV_ETH_RX_Q_NUM; queue++) {
	if(done_per_q[queue] > 0) {
	    TRC_REC("^rx receive %d packets from queue %d\n",done_per_q[queue],queue);
	    mv_gtw_rx_fill( queue, gbe_desc_num_per_rxq[queue] );
	}
    }

    return(work_done);
}



static unsigned int mv_gtw_rx_fill(unsigned int queue, int total)
{
    MV_PKT_INFO pkt_info;
    MV_BUF_INFO bufInfo;
    struct sk_buff *skb;
    unsigned int count = 0, buf_size;
    MV_STATUS status;
    int alloc_skb_failed = 0;

    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_events[queue]++) );

    while( total-- ) {

        /* allocate a buffer */
	buf_size = MV_MTU + 32 /* 32(extra for cache prefetch) */ + 8 /* +8 to align on 8B */;

        skb = dev_alloc_skb( buf_size ); 
	if(unlikely(!skb)) {
	    GTW_DBG( GTW_DBG_RX_FILL, ("rx_fill failed alloc skb\n") );
	    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_alloc_skb_fail[queue]++) );
	    alloc_skb_failed = 1;
	    break;
	}

	/* align the buffer on 8B */
	if( (unsigned long)(skb->data) & 0x7 ) {
	    skb_reserve( skb, 8 - ((unsigned long)(skb->data) & 0x7) );
	}

        bufInfo.bufVirtPtr = skb->data;
        bufInfo.bufSize = MV_MTU;
        pkt_info.osInfo = (MV_ULONG)skb;
        pkt_info.pFrags = &bufInfo;
	pkt_info.pktSize = MV_MTU; /* how much to invalidate */

	/* skip on first 2B (HW header) */
	skb_reserve( skb, 2 );
	skb->len = 0;

	/* give the buffer to hal */
	status = mvEthPortRxDone( gbe_dev.hal_priv, queue, &pkt_info );
	
	if(likely(status == MV_OK)) {
	    count++;
	    gbe_dev.rxq_count[queue]++;
	    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_hal_ok[queue]++) );	    
	}
	else if( status == MV_FULL ) {
	    /* the ring is full */
	    count++;
	    gbe_dev.rxq_count[queue]++;
	    GTW_DBG( GTW_DBG_RX_FILL, ("rxq full\n") );
	    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_hal_full[queue]++) );
	    if( gbe_dev.rxq_count[queue] != gbe_desc_num_per_rxq[queue])
	        printk( KERN_ERR "Q %d: error in status fill (%d != %d)\n", queue, gbe_dev.rxq_count[queue], gbe_desc_num_per_rxq[queue]);
	    break;
	} 
	else {
	    printk( KERN_ERR "Q %d: error in rx-fill\n", queue );
	    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_hal_error[queue]++) );
	    break;
	}
    }

    /* if allocation failed and the number of rx buffers in the ring is less than */
    /* half of the ring size, then set a timer to try again later.                */
    if(unlikely(alloc_skb_failed && (gbe_dev.rxq_count[queue] < (gbe_desc_num_per_rxq[queue]/2)))) {
        if( gbe_dev.rx_fill_flag == 0 ) {
	    printk( KERN_INFO "Q %d: set rx timeout to allocate skb\n", queue );
	    gbe_dev.rx_fill_timer.expires = jiffies + (HZ/10); /*100ms*/
	    add_timer( &gbe_dev.rx_fill_timer );
	    gbe_dev.rx_fill_flag = 1;
	}
    }

    GTW_DBG( GTW_DBG_RX_FILL, ("rx fill %d (total %d)", count, gbe_dev.rxq_count[queue]) );
    
    return count;
}

static void mv_gtw_rx_fill_on_timeout( unsigned long data ) 
{
    unsigned int queue;

    GTW_DBG( GTW_DBG_RX_FILL, ("rx_fill_on_timeout") );
    EGIGA_STAT( EGIGA_STAT_RX_FILL, (gbe_dev.egiga_stat.rx_fill_timeout_events++) );
   
    gbe_dev.rx_fill_flag = 0;
    for(queue = 0; queue < MV_ETH_RX_Q_NUM; queue++)
    	mv_gtw_rx_fill(queue, gbe_desc_num_per_rxq[queue]);
}

static int mv_gtw_poll( struct net_device *dev, int *budget )
{
    int rx_work_done;
    int tx_work_done;
    unsigned long flags;

    EGIGA_STAT( EGIGA_STAT_INT, (gbe_dev.egiga_stat.poll_events++) );

    tx_work_done = mv_gtw_tx_done();

    /* it transmission was previously stopped, now it can be restarted. */
    if(tx_work_done &&  gbe_dev.netif_stopped)
	mv_gtw_wakeup_netif();

    rx_work_done = mv_gtw_rx( min(*budget,dev->quota) );

    *budget -= rx_work_done;
    dev->quota -= rx_work_done;

    GTW_DBG( GTW_DBG_RX, ("poll work done: tx-%d rx-%d\n",tx_work_done,rx_work_done) );
    TRC_REC("^mv_gtw_poll work done: tx-%d rx-%d\n",tx_work_done,rx_work_done);

    if(((tx_work_done==0) && (rx_work_done==0)) || (!netif_running(dev))) {
	    local_irq_save(flags);
	    TRC_REC("^mv_gtw_poll remove from napi\n");
            netif_rx_complete(dev);
	    EGIGA_STAT( EGIGA_STAT_INT, (gbe_dev.egiga_stat.poll_complete++) );
	    /* unmask interrupts */
	    MV_REG_WRITE( ETH_INTR_MASK_REG(EGIGA_DEF_PORT), EGIGA_PICR_MASK );
	    MV_REG_WRITE( ETH_INTR_MASK_EXT_REG(EGIGA_DEF_PORT), EGIGA_PICER_MASK );
	    gbe_dev.rxmask = EGIGA_PICR_MASK;
	    gbe_dev.txmask = EGIGA_PICER_MASK;
	    GTW_DBG( GTW_DBG_RX, ("unmask\n") );
	    local_irq_restore(flags);
            return 0;
    }

    return 1;
}

static irqreturn_t mv_gtw_interrupt_handler( int irq , void *dev_id , struct pt_regs *regs )
{
    unsigned int picr, picer=0;

    spin_lock( &gbe_dev.lock );
    EGIGA_STAT( EGIGA_STAT_INT, (gbe_dev.egiga_stat.irq_total++) );

    picr = MV_REG_READ( ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT) );
    GTW_DBG( GTW_DBG_INT, ("%s [picr %08x]",__FUNCTION__,picr) );
    TRC_REC("^mv_gtw_interrupt_handler\n");

    if(unlikely(!picr)) {
        EGIGA_STAT( EGIGA_STAT_INT, (gbe_dev.egiga_stat.irq_none++) );
	spin_unlock( &gbe_dev.lock );
        return IRQ_NONE;
    }
    MV_REG_WRITE( ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT), ~picr );

    if(picr & BIT1) {
	picer = MV_REG_READ( ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT) );
	if(picer)
	    MV_REG_WRITE( ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT), ~picer );
    }

    /* schedule the first net_device to do the work out of interrupt context (NAPI) */
    if(netif_rx_schedule_prep(main_net_dev)) {
        TRC_REC("^sched napi\n");

	/* mask cause */
        gbe_dev.rxmask = 0;
        MV_REG_WRITE( ETH_INTR_MASK_REG(EGIGA_DEF_PORT), 0 );

	/* save rx cause and clear */
        gbe_dev.rxcause |= (picr&EGIGA_RXQ_MASK) | ((picr&EGIGA_RXQ_RES_MASK) >> (ETH_CAUSE_RX_ERROR_OFFSET-ETH_CAUSE_RX_READY_OFFSET));
	MV_REG_WRITE( ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT), 0 );

	/* mask tx-event */
        gbe_dev.txmask = 0;
        MV_REG_WRITE( ETH_INTR_MASK_EXT_REG(EGIGA_DEF_PORT), 0 );

	/* save tx cause and clear */
        gbe_dev.txcause |= picer & EGIGA_TXQ_MASK;
	MV_REG_WRITE( ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT), 0 );

        /* schedule the work (rx+txdone) out of interrupt contxet */
        __netif_rx_schedule(main_net_dev);
    }
    else {
        if(unlikely(netif_running(main_net_dev))) {
	    printk("rx interrupt while in polling list\n");
	    printk("rx-cause=0x%08x\n",MV_REG_READ(ETH_INTR_CAUSE_REG(EGIGA_DEF_PORT)));
	    printk("rx-mask =0x%08x\n",MV_REG_READ(ETH_INTR_MASK_REG(EGIGA_DEF_PORT)));
	    printk("tx-cause=0x%08x\n",MV_REG_READ(ETH_INTR_CAUSE_EXT_REG(EGIGA_DEF_PORT)));
	    printk("tx-mask =0x%08x\n",MV_REG_READ(ETH_INTR_MASK_EXT_REG(EGIGA_DEF_PORT)));
	}
    }

    spin_unlock( &gbe_dev.lock );
    return IRQ_HANDLED;
}

static struct net_device_stats* mv_gtw_get_stats( struct net_device *dev )
{
    return &(((mv_priv *)dev->priv)->net_dev_stat);
}

static int mv_gtw_set_mac_addr( struct net_device *dev, void *p )
{
    mv_priv *mvp = (mv_priv *)dev->priv;
    struct mv_vlan_cfg *vlan_cfg = mvp->vlan_cfg;

    struct sockaddr *addr = p;
    if(!is_valid_ether_addr(addr->sa_data))
	return -EADDRNOTAVAIL;

    /* remove old mac addr from VLAN DB */
    mv_gtw_set_mac_addr_to_switch(dev->dev_addr,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),(1<<QD_PORT_CPU),0);

    memcpy(dev->dev_addr, addr->sa_data, 6);

    /* add new mac addr to VLAN DB */
    mv_gtw_set_mac_addr_to_switch(dev->dev_addr,MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id),(1<<QD_PORT_CPU),1);

    printk("mv_gateway: %s change mac address to %02x:%02x:%02x:%02x:%02x:%02x\n", dev->name,
	*(dev->dev_addr), *(dev->dev_addr+1), *(dev->dev_addr+2), *(dev->dev_addr+3), *(dev->dev_addr+4), *(dev->dev_addr+5));

#ifdef CONFIG_MV_GTW_QOS
    /* In order to coexist with QoS filtering at gigabit level, we must have a match between  */
    /* incomming packet DA and the mac addr (bits 47:4) set in gigabit level. Since different */
    /* interfaces has different MAC addresses, only the WAN interface benefits from QoS.      */
    if(strcmp(dev->name,CONFIG_MV_GTW_QOS_NET_IF)==0) {
        GTW_DBG(GTW_DBG_MACADDR, ("mv_gateway: %s change to %02x %02x %02x %02x %02x %02x also in gigabit level\n", dev->name,
	*(dev->dev_addr), *(dev->dev_addr+1), *(dev->dev_addr+2), *(dev->dev_addr+3), *(dev->dev_addr+4), *(dev->dev_addr+5)));
	mvEthMacAddrSet(gbe_dev.hal_priv, dev->dev_addr, EGIGA_DEF_RXQ);	
    }
#endif

    return 0;
}

int mv_gtw_set_mac_addr_to_switch(unsigned char *mac_addr, unsigned char db, unsigned int ports_mask, unsigned char op)
{
    GT_ATU_ENTRY mac_entry;
    struct mv_vlan_cfg *nc;

    /* validate db with VLAN id */
    nc = &net_cfg[db];
    if(MV_GTW_VLANID_TO_GROUP(nc->vlan_grp_id) != db) {
        printk("mv_gtw_set_mac_addr_to_switch (invalid db)\n");
	return -1;
    }

    memset(&mac_entry,0,sizeof(GT_ATU_ENTRY));

    mac_entry.trunkMember = GT_FALSE;
    mac_entry.prio = 0;
    mac_entry.exPrio.useMacFPri = GT_FALSE;
    mac_entry.exPrio.macFPri = 0;
    mac_entry.exPrio.macQPri = 0;
    mac_entry.DBNum = db;
    mac_entry.portVec = ports_mask;
    memcpy(mac_entry.macAddr.arEther,mac_addr,6);

    if(is_multicast_ether_addr(mac_addr))
	mac_entry.entryState.mcEntryState = GT_MC_STATIC;
    else
	mac_entry.entryState.ucEntryState = GT_UC_NO_PRI_STATIC;

    printk("mv_gateway: %s db%d port-mask=0x%x, %02x:%02x:%02x:%02x:%02x:%02x ", 
	nc->name, db, (unsigned int)mac_entry.portVec,
	mac_entry.macAddr.arEther[0],mac_entry.macAddr.arEther[1],mac_entry.macAddr.arEther[2],
	mac_entry.macAddr.arEther[3],mac_entry.macAddr.arEther[4],mac_entry.macAddr.arEther[5]);

    if((op == 0) || (mac_entry.portVec == 0)) { 
        if(gfdbDelAtuEntry(qd_dev, &mac_entry) != GT_OK) {
	    printk("gfdbDelAtuEntry failed\n");
	    return -1;
        }
	printk("deleted\n");
    }
    else {
        if(gfdbAddMacEntry(qd_dev, &mac_entry) != GT_OK) {
	    printk("gfdbAddMacEntry failed\n");
	    return -1;
        }
	printk("added\n");
    }

    return 0;
}

static void mv_gtw_set_multicast_list(struct net_device *dev) 
{
    struct dev_mc_list *curr_addr = dev->mc_list;
    mv_priv *mvp = (mv_priv *)dev->priv;
    struct mv_vlan_cfg *vlan_cfg = mvp->vlan_cfg;
    int i;

    disable_irq(ETH_PORT0_IRQ_NUM);
    if((dev->flags & IFF_PROMISC) || (dev->flags & IFF_ALLMULTI)) { 
	/* promiscuous mode - connect the CPU port to the VLAN (port based + 802.1q) */
	if(dev->flags & IFF_PROMISC)
	    printk("mv_gateway: setting promiscuous mode\n");
	if(dev->flags & IFF_ALLMULTI)
	    printk("mv_gateway: setting multicast promiscuous mode\n");
	mv_gtw_set_port_based_vlan(vlan_cfg->ports_mask|(1<<QD_PORT_CPU));
	for(i=0; i<qd_dev->numOfPorts; i++) {
	    if(MV_BIT_CHECK(vlan_cfg->ports_mask, i) && (i!=QD_PORT_CPU)) {
		if(mv_gtw_set_vlan_in_vtu(MV_GTW_PORT_VLAN_ID(vlan_cfg->vlan_grp_id,i),vlan_cfg->ports_mask|(1<<QD_PORT_CPU)) != 0) {
		    printk("mv_gtw_set_vlan_in_vtu failed\n");
		}
	   }
	}
    }
    else {
	/* does not in promiscuous or allmulti mode - disconnect the CPU port to the VLAN (port based + 802.1q) */
	mv_gtw_set_port_based_vlan(vlan_cfg->ports_mask&(~(1<<QD_PORT_CPU)));
	for(i=0; i<qd_dev->numOfPorts; i++) {
	    if(MV_BIT_CHECK(vlan_cfg->ports_mask, i) && (i!=QD_PORT_CPU)) {
		if(mv_gtw_set_vlan_in_vtu(MV_GTW_PORT_VLAN_ID(vlan_cfg->vlan_grp_id,i),vlan_cfg->ports_mask&(~(1<<QD_PORT_CPU))) != 0) {
		    printk("mv_gtw_set_vlan_in_vtu failed\n");
		}
	   }
	}
	if(dev->mc_count) { 
	    /* accept specific multicasts */
	    for(i=0; i<dev->mc_count; i++, curr_addr = curr_addr->next) {
	        if (!curr_addr)
		    break;
		mv_gtw_set_mac_addr_to_switch(curr_addr->dmi_addr, MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id), (1<<QD_PORT_CPU), 1);
	    }
	}
        else { 
	    /* does not accept any multicasts */ 
	    for(i=0; i<dev->mc_count; i++, curr_addr = curr_addr->next) {
	        if (!curr_addr)
		    break;
		mv_gtw_set_mac_addr_to_switch(curr_addr->dmi_addr, MV_GTW_VLANID_TO_GROUP(vlan_cfg->vlan_grp_id), (1<<QD_PORT_CPU), 0);
	    }
	}
    }
    enable_irq(ETH_PORT0_IRQ_NUM);
}

GT_BOOL ReadMiiWrap(GT_QD_DEV* dev, unsigned int portNumber, unsigned int MIIReg, unsigned int* value)
{
    if(mvEthPhyRegRead(portNumber, MIIReg , (unsigned short *)value) == MV_OK)
        return GT_TRUE;
    return GT_FALSE;
}
GT_BOOL WriteMiiWrap(GT_QD_DEV* dev, unsigned int portNumber, unsigned int MIIReg, unsigned int data)
{
    /*printk("MII write: port=0x%x, reg=%d, data=0x%x\n",portNumber,MIIReg,data);*/
    if (mvEthPhyRegWrite(portNumber, MIIReg ,data) == MV_OK)
        return GT_TRUE;
    return GT_FALSE;
}
static int mv_gtw_set_port_based_vlan(unsigned int ports_mask)
{
    unsigned int p, pl;
    unsigned char cnt;
    GT_LPORT port_list[MAX_SWITCH_PORTS];

    for(p=0; p<qd_dev->numOfPorts; p++) {
	if( MV_BIT_CHECK(ports_mask, p) && (p != QD_PORT_CPU) ) {
	    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("port based vlan, port %d: ",p));
	    for(pl=0,cnt=0; pl<qd_dev->numOfPorts; pl++) {
		if( MV_BIT_CHECK(ports_mask, pl) && (pl != p) ) {
		    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("%d ",pl));
		    port_list[cnt] = pl;
                    cnt++;
                }
	    }
            if( gvlnSetPortVlanPorts(qd_dev, p, port_list, cnt) != GT_OK) {
	        printk("gvlnSetPortVlanPorts failed\n");
                return -1;
            }
	    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("\n"));
        }
    }
    return 0;
}
static int mv_gtw_set_vlan_in_vtu(unsigned short vlan_id,unsigned int ports_mask)
{
    GT_VTU_ENTRY vtu_entry;
    unsigned int p;

    vtu_entry.vid = vlan_id;
    vtu_entry.DBNum = MV_GTW_VLANID_TO_GROUP(vlan_id);
    vtu_entry.vidPriOverride = GT_FALSE;
    vtu_entry.vidPriority = 0;
    vtu_entry.vidExInfo.useVIDFPri = GT_FALSE;
    vtu_entry.vidExInfo.vidFPri = 0;
    vtu_entry.vidExInfo.useVIDQPri = GT_FALSE;
    vtu_entry.vidExInfo.vidQPri = 0;
    vtu_entry.vidExInfo.vidNRateLimit = GT_FALSE;
    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("vtu entry: vid=0x%x, port ", vtu_entry.vid));
    for(p=0; p<qd_dev->numOfPorts; p++) {
        if(MV_BIT_CHECK(ports_mask, p)) {
	    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("%d ", p));
	    if(qd_dev->deviceId == GT_88E6061) {
		/* for 6061 device, no double/provider tag controlling on ingress. */
		/* therefore, we need to strip the tag on egress on all ports except cpu port */
		if(p == QD_PORT_CPU)
		    vtu_entry.vtuData.memberTagP[p] = MEMBER_EGRESS_TAGGED; /* in addition to egress tag mode, this is required for promisc mode */
		else
		    vtu_entry.vtuData.memberTagP[p] = MEMBER_EGRESS_UNTAGGED;
	    }
	    else {
		vtu_entry.vtuData.memberTagP[p] = MEMBER_EGRESS_UNMODIFIED;
	    }
	}
	else {
	    vtu_entry.vtuData.memberTagP[p] = NOT_A_MEMBER;
	}
	vtu_entry.vtuData.portStateP[p] = 0;
    }
    if(gvtuAddEntry(qd_dev, &vtu_entry) != GT_OK) {
        printk("gvtuAddEntry failed\n");
        return -1;
    }

    GTW_DBG( GTW_DBG_LOAD|GTW_DBG_MCAST|GTW_DBG_VLAN, ("\n"));
    return 0;
}

#ifdef CONFIG_MV_GTW_QOS_VOIP
int mv_gtw_qos_tos_enable(void)
{
    printk("mv_gateway: VoIP QoS ON\n");
    netdev_max_backlog = 15;
    MAX_SOFTIRQ_RESTART = 1;
    mv_gtw_qos_tos_quota = 5;
    TRC_REC("^QoS ON sched-iter=%d quota=%d netdev_max_backlog=%d\n",MAX_SOFTIRQ_RESTART,mv_gtw_qos_tos_quota,netdev_max_backlog);
    return 0;
}
int mv_gtw_qos_tos_disable(void)
{
    printk("mv_gateway VoIP QoS OFF\n");
    netdev_max_backlog = 300;
    MAX_SOFTIRQ_RESTART = 10;
    mv_gtw_qos_tos_quota = -1;
    TRC_REC("^QoS OFF sched-iter=%d quota=%d\n",MAX_SOFTIRQ_RESTART,mv_gtw_qos_tos_quota);
    return 0;
}
#endif /*CONFIG_MV_GTW_QOS*/

#ifdef CONFIG_MV_GTW_IGMP
int mv_gtw_enable_igmp(void)
{
    unsigned char p;

    GTW_DBG( GTW_DBG_IGMP, ("enabling L2 IGMP snooping\n"));

    /* enable IGMP snoop on all ports (except cpu port) */
    for(p=0; p<qd_dev->numOfPorts; p++) {
	if(p != QD_PORT_CPU) {
	    if(gprtSetIGMPSnoop(qd_dev, p, GT_TRUE) != GT_OK) {
		printk("gprtSetIGMPSnoop failed\n");
		return -1;
	    }
	}
    }
    return -1;
}
#endif

static void mv_gtw_convert_str_to_mac( char *source , char *dest ) 
{
    dest[0] = (mv_gtw_str_to_hex( source[0] ) << 4) + mv_gtw_str_to_hex( source[1] );
    dest[1] = (mv_gtw_str_to_hex( source[3] ) << 4) + mv_gtw_str_to_hex( source[4] );
    dest[2] = (mv_gtw_str_to_hex( source[6] ) << 4) + mv_gtw_str_to_hex( source[7] );
    dest[3] = (mv_gtw_str_to_hex( source[9] ) << 4) + mv_gtw_str_to_hex( source[10] );
    dest[4] = (mv_gtw_str_to_hex( source[12] ) << 4) + mv_gtw_str_to_hex( source[13] );
    dest[5] = (mv_gtw_str_to_hex( source[15] ) << 4) + mv_gtw_str_to_hex( source[16] );
}
static unsigned int mv_gtw_str_to_hex( char ch ) 
{
    if( (ch >= '0') && (ch <= '9') )
        return( ch - '0' );

    if( (ch >= 'a') && (ch <= 'f') )
	return( ch - 'a' + 10 );

    if( (ch >= 'A') && (ch <= 'F') )
	return( ch - 'A' + 10 );

    return 0;
}

#define STAT_PER_Q(qnum,x)  for(queue = 0; queue < qnum; queue++) \
				printk("%10u ",x[queue]); \
			    printk("\n");

void print_mv_gtw_stat(void)
{
#ifndef EGIGA_STATISTICS
  printk(" Error: driver is compiled without statistics support!! \n");
  return;
#else
    egiga_statistics *stat = &gbe_dev.egiga_stat;
    unsigned int queue;

    printk("QUEUS:.........................");
    for(queue = 0; queue < MV_ETH_RX_Q_NUM; queue++) 
	printk( "%10d ",queue);
    printk("\n");

  if( mv_gtw_stat & EGIGA_STAT_INT ) {
      printk( "\n====================================================\n" );
      printk( "interrupt statistics" );
      printk( "\n-------------------------------\n" );
      printk( "irq_total.....................%10u\n", stat->irq_total );
      printk( "irq_none......................%10u\n", stat->irq_none );
  }
  if( mv_gtw_stat & EGIGA_STAT_POLL ) {
      printk( "\n====================================================\n" );
      printk( "polling statistics" );
      printk( "\n-------------------------------\n" );
      printk( "poll_events...................%10u\n", stat->poll_events );
      printk( "poll_complete.................%10u\n", stat->poll_complete );
  }
  if( mv_gtw_stat & EGIGA_STAT_RX ) {
      printk( "\n====================================================\n" );
      printk( "rx statistics" );
      printk( "\n-------------------------------\n" );
      printk( "rx_events................%10u\n", stat->rx_events );
      printk( "rx_hal_ok................");STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_ok);
      printk( "rx_hal_no_resource.......");STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_no_resource );
      printk( "rx_hal_no_more..........."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_no_more );
      printk( "rx_hal_error............."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_error );
      printk( "rx_hal_invalid_skb......."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_invalid_skb );
      printk( "rx_hal_bad_stat.........."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_hal_bad_stat );
      printk( "rx_netif_drop............"); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_netif_drop );
  }
  if( mv_gtw_stat & EGIGA_STAT_RX_FILL ) {
      printk( "\n====================================================\n" );
      printk( "rx fill statistics" );
      printk( "\n-------------------------------\n" );
      printk( "rx_fill_events................"); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_fill_events );
      printk( "rx_fill_alloc_skb_fail........"); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_fill_alloc_skb_fail );
      printk( "rx_fill_hal_ok................"); STAT_PER_Q(MV_ETH_RX_Q_NUM,stat->rx_fill_hal_ok);
      printk( "rx_fill_hal_full.............."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_fill_hal_full );
      printk( "rx_fill_hal_error............."); STAT_PER_Q(MV_ETH_RX_Q_NUM, stat->rx_fill_hal_error );
      printk( "rx_fill_timeout_events........%10u\n", stat->rx_fill_timeout_events );
      printk( "rx buffer size................%10u\n", MV_MTU);
  }
  if( mv_gtw_stat & EGIGA_STAT_TX ) {
      printk( "\n====================================================\n" );
      printk( "tx statistics" );
      printk( "\n-------------------------------\n" );
      printk( "tx_events.....................%10u\n", stat->tx_events );
      printk( "tx_hal_ok.....................");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_hal_ok);
      printk( "tx_hal_no_resource............");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_hal_no_resource );
      printk( "tx_hal_no_error...............");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_hal_error );
      printk( "tx_hal_unrecognize............");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_hal_unrecognize );
      printk( "tx_netif_stop.................");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_netif_stop );
      printk( "tx_timeout....................%10u\n", stat->tx_timeout );
  }
  if( mv_gtw_stat & EGIGA_STAT_TX_DONE ) {
      printk( "\n====================================================\n" );
      printk( "tx-done statistics" );
      printk( "\n-------------------------------\n" );
      printk( "tx_done_events................%10u\n", stat->tx_done_events );
      printk( "tx_done_hal_ok................");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_ok);
      printk( "tx_done_hal_invalid_skb.......");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_invalid_skb );
      printk( "tx_done_hal_bad_stat..........");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_bad_stat );
      printk( "tx_done_hal_still_tx..........");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_still_tx );
      printk( "tx_done_hal_no_more...........");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_no_more );
      printk( "tx_done_hal_unrecognize.......");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_hal_unrecognize );
      printk( "tx_done_max...................");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_max );
      printk( "tx_done_min...................");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_min );
      printk( "tx_done_netif_wake............");STAT_PER_Q(MV_ETH_TX_Q_NUM, stat->tx_done_netif_wake );
  }

  memset( stat, 0, sizeof(egiga_statistics) );

  printk("QD CPU_PORT statistics: \n");
  print_qd_port_counters(QD_PORT_CPU);
#endif /*EGIGA_STATISTICS*/
}

void print_qd_port_counters(unsigned int port)
{
    GT_STATS_COUNTER_SET3 counters;

    printk("Getting QD counters for port %d.\n", port);

    if(gstatsGetPortAllCounters3(qd_dev,port,&counters) != GT_OK) {
    	printk("gstatsGetPortAllCounters3 failed");
	return;
    }

    printk("InGoodOctetsLo  %lu    ", counters.InGoodOctetsLo);
    printk("InGoodOctetsHi  %lu   \n", counters.InGoodOctetsHi);
    printk("InBadOctets     %lu    ", counters.InBadOctets);
    printk("OutFCSErr       %lu   \n", counters.OutFCSErr);
    printk("Deferred        %lu   \n", counters.Deferred);
    printk("InBroadcasts    %lu    ", counters.InBroadcasts);
    printk("InMulticasts    %lu   \n", counters.InMulticasts);
    printk("Octets64        %lu    ", counters.Octets64);
    printk("Octets127       %lu   \n", counters.Octets127);
    printk("Octets255       %lu    ", counters.Octets255);
    printk("Octets511       %lu   \n", counters.Octets511);
    printk("Octets1023      %lu    ", counters.Octets1023);
    printk("OctetsMax       %lu   \n", counters.OctetsMax);
    printk("OutOctetsLo     %lu    ", counters.OutOctetsLo);
    printk("OutOctetsHi     %lu   \n", counters.OutOctetsHi);
    printk("OutUnicasts     %lu    ", counters.OutOctetsHi);
    printk("Excessive       %lu   \n", counters.Excessive);
    printk("OutMulticasts   %lu    ", counters.OutMulticasts);
    printk("OutBroadcasts   %lu   \n", counters.OutBroadcasts);
    printk("Single          %lu    ", counters.OutBroadcasts);
    printk("OutPause        %lu   \n", counters.OutPause);
    printk("InPause         %lu    ", counters.InPause);
    printk("Multiple        %lu   \n", counters.InPause);
    printk("Undersize       %lu    ", counters.Undersize);
    printk("Fragments       %lu   \n", counters.Fragments);
    printk("Oversize        %lu    ", counters.Oversize);
    printk("Jabber          %lu   \n", counters.Jabber);
    printk("InMACRcvErr     %lu    ", counters.InMACRcvErr);
    printk("InFCSErr        %lu   \n", counters.InFCSErr);
    printk("Collisions      %lu    ", counters.Collisions);
    printk("Late            %lu   \n", counters.Late);

    return;
}

void print_qd_atu(int db)
{
    GT_ATU_ENTRY atu_entry;
    memset(&atu_entry,0,sizeof(GT_ATU_ENTRY));
    atu_entry.DBNum = db;
    printk("ATU DB%d display:\n",db);
    while(gfdbGetAtuEntryNext(qd_dev,&atu_entry) == GT_OK) {
	printk("%02x:%02x:%02x:%02x:%02x:%02x, PortVec %#x, Pri %#x, State %#x\n",
	atu_entry.macAddr.arEther[0],atu_entry.macAddr.arEther[1],
	atu_entry.macAddr.arEther[2],atu_entry.macAddr.arEther[3],
	atu_entry.macAddr.arEther[4],atu_entry.macAddr.arEther[5],
	atu_entry.portVec,atu_entry.prio,atu_entry.entryState.mcEntryState);
    }
}


