//**************************************************************************
//
//	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		confutil.c
//
//	Abstract:
//		System configuration Access Functions.
//
//	FUNCTIONS:	TBD.
//
//	COMMENTS: 	N/A
//
//	HISTORY:
//	08/07/01	Louis Tsai add new profile processing function
//	07/26/01	Meiji Chang Remove unused code.
//	06/03/01	Louis Tsai add Update_Flash_File() to sync flash disk
//	04/24/01	Louis Tsai updated r/w lock & tab/space issue in  relative function
//	04/11/01	Louis Tsai fix bug in Conf_Lock_File()-forgot to closedir!!!!
//	04/04/01	Louis Tsai modified Conf_Remove_Section & Conf_Remove_Field
//			can return Error_Not_Found
//	03/26/01	Louis Tsai add  Conf_Case_Sensitive & Conf_Lock_File() & Conf_Unlock_File()
//	03/23/01	Louis Tsai transfer to library
//      03/22/01	Louis Tsai added GetPrivateProfileString() & WritePrivateProfileString()
//	02/21/01	Meiji Chang created
//
//**************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <unistd.h>
#include <time.h>
#include <ctype.h>
#include "Util.h"
#include "file_lock.h"
#include "shm_util.h"
#include "nas_lib_common.h"

static int case_flag=0;

typedef struct LOCKDATA
{
	char type;	// r-> read lock; w-> write lock
	int pid;	// which pid owned this lock
	int count;	// number of read; 
	time_t time_out;// time to force unlock
	
} LOCKDATA; 

/* ------------------------------------------------------------------------- */
/*         		Function Implementation                              */
/* ------------------------------------------------------------------------- */

// Check if a string is ended with new line...
// Return:	1 - true
//		0 - false
int string_ended_with_EOL(char *str)
{
	if (str[strlen(str) - 1] == '\n') return 1;
	return 0;
}



// Kill the new line character on the end of the string...
// Return:	1 - new line killed
//		0 - no new line on the tail
int strip_EOL(char *str)
{
	if (str[strlen(str) - 1] == '\n') {
		str[strlen(str) - 1] = '\0';
		if (str[strlen(str) - 1] == '\r')
			str[strlen(str) - 1] = '\0';
		return 1;
	}
	return 0;
}

// Concat two string and produce a new string...
// Return:
//		pointer to the newly allocated concatened string
// 		NULL - if memory allocation fail
char* combine_string(char *str1, char *str2)
{
	int size = strlen(str1) + strlen(str2) + 1;
	char *p = (char *) malloc(size);
	if (NULL != p) {
		strcpy(p, str1);
		strcat(p, str2);
	}
	return p;
}

// Read from a file stream and skip until a new line character was found...
// Return: 	None
void skip_to_EOL(FILE *fp)
{
	char buf[BUF_SIZE];

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		if (string_ended_with_EOL(buf))
			return;
	}
}


// Skip the white space (SPACE & TAB) characters from the start of a string
// Return the string started from the first non-white-space character
char *skip_white_space(char *str)
{
	char *p = str;
	while (*p == ' ' || *p == '\t') p++;
	return p;
}

char *Skip_White_Space_Ex(char *str)
{
	int i, len;
	char *p = str;
	while (*p == ' ' || *p == '\t') p++;
	len = strlen(p);
	for(i = len-1; i >= 0; i--)
		if(*(p+i) == ' ' || *(p+i) == '\t')
			*(p+i) = '\0';
	return p;
}


// Strip off the white space characters appened on the end of a string.
// Return the original string ended with non-white-space characters
#if 0
char *stripe_white_space_tail(char *str)
{
	if (strlen(str) > 0) {
		char *p = str + strlen(str) - 1;
		while (*p == ' ' || *p == '\t') p--;
		*(p+1) = '\0';
	}
	return str;
}
#endif

// Search a string for the offset of a specified character...
// Return:	non-negative - offset in the string that contains the specified character
//		-1 - the specified charater was not found in the string
int find_char_in_string(char *q, char delimiter)
{
	int i = 0;
	while (q[i] != '\0') {
		if ((delimiter == SPACE) || (delimiter == TAB) ) {
			if ((q[i] == ' ') || (q[i] == '\t')) {
				if ((q[i+1]== ' ') || (q[i+1] == '\t') ) {
					while ( (q[i+1]== ' ') || (q[i+1] == '\t')) 
						i++;
				}
				return i;
			}
			else
				i++;
		}
		else {
			if (q[i] == delimiter) return i;
			i++;
		 }
	}
	return -1;	// not found...
}


//
char *get_string_until_ch(char *str, char ch, char *buf, int buf_size)
{
	int i;

	str = skip_white_space(str);
	if(ch == ']') i = (int)(strrchr(str,']') - str) ;
	else i = find_char_in_string(str, ch);
	if (i < 0) // no CH was found in this line
		return NULL;

	if (i >= buf_size) i = buf_size - 1;
	strncpy(buf, str, i);
	buf[i] = '\0';

	stripe_white_space_tail(buf);

	return buf;
}

// Check if a string is started with comment characters...
// Return:	1 - true
//		0 - false
int is_commented_string(char *str)
{
	char *q = skip_white_space(str);
	if (*q == '#' || *q == ';') return 1;
	return 0;
}



char *get_one_line_of_string(FILE *fp)
{
	static char *extra_buf = NULL;
//	static char buf[BUF_SIZE];
	static char buf[4096];

	if (extra_buf != NULL) {	// free previously allocated buffer first...
		free(extra_buf);
		extra_buf = NULL;
	}

	if (fgets(buf, sizeof(buf), fp) == NULL) return NULL;		// EOF
	if (strip_EOL(buf)) 		// got the whole line
		return buf;

	// original buffer not large enough, try to allocate more...
	if ((extra_buf = strdup(buf)) == NULL) {	// memory allocation error...
		skip_to_EOL(fp);		// skip the remains of this line.
		return buf;			// should we skip to EOL??
	}

	while (fgets(buf, sizeof(buf), fp) != NULL) {
		char *p = combine_string(extra_buf, buf);
		if (NULL == p) {		// memory allocation error...
			skip_to_EOL(fp);		// skip the remains of this line.
			return extra_buf;		// should we skip to EOL??
		}
		free(extra_buf);
		extra_buf = p;

		if (strip_EOL(extra_buf)) 	// found EOL...
			break;
	}

	return extra_buf;
}


int get_section_name(char *str, char *section_name, int buf_size)
{
	char *q = skip_white_space(str);
	if (*q == '[') {	//start of section name...
		if (get_string_until_ch(q+1, ']', section_name, buf_size) != NULL)
			return SUCCESS;
	}
	return ERROR_FAIL;
}


int get_field_name(char *str, char *field_name, int buf_size)
{
	if (!is_commented_string(str)) {
		char *q = skip_white_space(str);
		if (*q == '[') // start of section name...
			return ERROR_FAIL;
		if (get_string_until_ch(q, '=', field_name, buf_size) != NULL)
			return SUCCESS;
	}
	return ERROR_FAIL;
}


#ifdef XXX
//-------------------------------------------------------------
// Compare two string with case-sensitive or non case-sensitive
// case_flag = 0 ,case insensitive
//           = 1 ,case sensitive
// return value is the same as strcmp()
//--------------------------------------------------------------

int compare_string(char* str1,char* str2)
{
	if(str1 == NULL || str2 == NULL)
		return -1;
	if( case_flag )
		return(strcmp(str1,str2));
	else
		return(strcasecmp(str1,str2));
}
#endif

//-------------------------------------------------------------------
// Compare two string with case-sensitive and aware of multibytes
// return value is the same as strcmp()
//--------------------------------------------------------------
int compare_string(char* str1, char* str2)
{
	return(mb_strcmp(str1, str2));
}

//-------------------------------------------------------------------
// Compare two string with non case-sensitive and aware of multibytes
// return value is the same as strcmp()
//--------------------------------------------------------------
int compare_case_string(char* str1, char* str2)
{
	//return(mb_strcasecmp(str1, str2));
	return(strcasecmp(str1, str2));
}

static int get_file_name(char *in_str,char *out_str,int buf_size)
{
	char *tmp_str;
	tmp_str=strrchr(in_str,(int)'/');
	if(tmp_str == NULL) {
		if(strlen(in_str)>buf_size) {
			strncpy(out_str,in_str,buf_size-1);
			out_str[buf_size]='\0';
		}
		else
			strcpy(out_str,in_str);
	}
	else {
		if((strlen(tmp_str+1)+1)>buf_size) {
			strncpy(out_str,tmp_str+1,buf_size-1);
			out_str[buf_size]='\0';
		}
		else
			strcpy(out_str,tmp_str+1);
	}
	return 0;
}

char *String_To_Upper_Case(char *src)
{
	int i;

	for (i = 0; src[i] != '\0'; i++) {
		src[i] = toupper(src[i]);
	}

	return src;
}

BOOL is_pid_alive(int pid)
{
	DIR *dp;	
	char tmp_str[20];
	BOOL ret = TRUE;
	sprintf(tmp_str,"/proc/%d",pid);
	
	if ((dp=opendir(tmp_str)) != NULL )
		closedir(dp);
	else
		ret = FALSE;
	
	return ret;
	
}

/* ------------------------------------------------------------------------- */
/* Before Opening Configuration File, process have to check wehther the file */
/* be locked by others 					                     */
/* ------------------------------------------------------------------------- */
int Conf_Lock_File( char *lock_file_name,char type,int time_to_live)
{
	char *tmp_str;
	int file_desc;
	int return_value = SUCCESS;
   	char tmp_filename[FILE_NAME_MAX_LENGTH];
   	DIR *dp;
   	FILE *fp;

   	get_file_name(lock_file_name,tmp_filename,FILE_NAME_MAX_LENGTH);
   	
   	tmp_str = (char *) malloc( strlen(LOCK_FILE_PATH)+strlen(tmp_filename)+strlen(".lck")+2);
	if (tmp_str == NULL ) return ERROR_OUT_OF_MEMORY;
   	
   	sprintf(tmp_str,"%s/%s.lck",LOCK_FILE_PATH,tmp_filename);
	
   	if ((dp=opendir(LOCK_FILE_PATH)) == NULL ) {
    		mkdir(LOCK_FILE_PATH,777);
    	}
    	else
    		closedir(dp);
	
    	file_desc = open(tmp_str, O_RDWR | O_CREAT | O_EXCL, 0777);
   	if (file_desc == -1) {
   		if((fp = fopen(tmp_str,"rb")) != NULL) {
 
   			LOCKDATA m_lck;
   			time_t current_time;
   			int t_status = 0, r_ok = 0;
   			
   			current_time = time((time_t *)0);
   			fread((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   			fclose(fp);

                        /* add by Kent 2001/01/02 */
			if (is_pid_alive(m_lck.pid)==FALSE)
                                t_status=1;
                        /* end */

   			if(current_time > m_lck.time_out) t_status = 1; //lock time out
   			if((toupper(m_lck.type) == 'R') && (toupper(m_lck.type) == toupper(type))) r_ok = 1;// check share_read
   				
   			if ( t_status > 0 ) { 
   				if((fp = fopen(tmp_str,"wb")) != NULL) {
   					m_lck.type 	= type;
					/* add by Kent 2001/01/02 */
					m_lck.pid=getpid();
					/* end */
   					m_lck.count	= 1;
   					m_lck.time_out= time((time_t *)0) + time_to_live ;
   					fwrite((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   					fclose(fp);	
   				}
   				else
					return_value = ERROR_LOCK_FILE;
   			}
   			else if (r_ok == 1) {
   				if((fp = fopen(tmp_str,"wb")) != NULL) {
   					m_lck.count	= m_lck.count + 1;
   					m_lck.time_out= time((time_t *)0) + time_to_live ;
   					fwrite((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   					fclose(fp);	
   				}
   				else
					return_value = ERROR_LOCK_FILE;
   			}
   			else {
   				if (toupper(type) == 'R') 
   					return_value = ERROR_READ_LOCK;
   				else 
   					return_value = ERROR_WRITE_LOCK;
   			}
   				
   		}
   		else 
   			return_value = ERROR_LOCK_FILE;
  
   	}
	else {
   		close(file_desc);
   		if((fp = fopen(tmp_str,"wb")) != NULL) {
   			LOCKDATA m_lck;
   			m_lck.type 	= type;
			/* add by Kent 2001/01/02 */
			m_lck.pid=getpid();
			/* end */
   			m_lck.count	= 1;
   			m_lck.time_out 	= time((time_t *)0) + time_to_live ;
   			fwrite((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   			fclose(fp);	
   		}
   		else
			return_value = ERROR_LOCK_FILE;
   	}
   	
   	if(return_value == ERROR_LOCK_FILE )
   		unlink(tmp_str);
   		
   	free(tmp_str);
   	return return_value;
}

int Conf_Unlock_File( char *lock_file_name, char type)
{
 	char *tmp_str;
   	char tmp_filename[FILE_NAME_MAX_LENGTH];
   	FILE *fp;
   	get_file_name(lock_file_name,tmp_filename,FILE_NAME_MAX_LENGTH);
   	tmp_str = (char *) malloc( strlen(LOCK_FILE_PATH)+strlen(tmp_filename)+strlen(".lck")+2);
	if (tmp_str == NULL ) return ERROR_OUT_OF_MEMORY;
	
   	sprintf(tmp_str,"%s/%s.lck",LOCK_FILE_PATH,tmp_filename);
   	if((fp = fopen(tmp_str,"rb")) != NULL) {
   			LOCKDATA m_lck;
   			fread((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   			fclose(fp);
   			if((toupper(type) == 'W') && (toupper(m_lck.type) == 'W'))
   				unlink(tmp_str);
   			if((toupper(type) == 'R') && (toupper(m_lck.type) == 'R')) {
   				if(m_lck.count > 1 ) {
   					if((fp = fopen(tmp_str,"wb")) != NULL) {
   						m_lck.count = m_lck.count -1;
   						fwrite((void *)&m_lck,sizeof(LOCKDATA),1,fp);
   						fclose(fp);
   					}
   					else
   						unlink(tmp_str);
   				}
   				else
   					unlink(tmp_str);
   			}
   	}
   	else
   		unlink(tmp_str);
 	
 	free(tmp_str);
   	return 0;
}

int Open_File(char *path, int mode, FILE **pfp)
{
	char	mod[2], type;

	mod[1]=0x0;
	switch (mode)
	{
		case O_RDONLY:
			type=mod[0]='r';
			break;
		case O_WRONLY:
			type=mod[0]='w';
			break;
	}
	if (NAS_File_Lock(path, type)==0)
		return ERROR_OPEN_FILE;
	if ((*pfp=fopen(path, mod))==NULL)
	{
		NAS_File_Unlock(path);
		return ERROR_OPEN_FILE;
	}
	return SUCCESS;
/*	int result = SUCCESS;
	int retry = LOCK_RETRY_TIMES;
	char *mod = "r";
	char type ;
	switch (mode) {
		case O_RDONLY:
			mod = "r";
			type = 'r';
			break;
		case O_WRONLY:
			mod = "w";
			type = 'w';
			break;
	}
	while(retry-- > 0) {
      		if ((result=Conf_Lock_File(path,type,LOCK_TIME_OUT))<0) {
          		if (retry > 0)
             		sleep(1);
      		}
      		else 
      			break;
   	}
	
	if (result == SUCCESS) {
		*pfp = fopen(path, mod);
		if (NULL == *pfp) {
			result = ERROR_OPEN_FILE;
			Conf_Unlock_File(path,type);
		}
	}
	
	return result;*/
}

int Close_File(char *path, int mode, FILE *pfp)
{
	fclose(pfp);
	NAS_File_Unlock(path);
	return SUCCESS;
/*	char *mod = "r";
	char type ;
	switch (mode) {
		case O_RDONLY:
			mod = "r";
			type = 'r';
			break;
		case O_WRONLY:
			mod = "w";
			type = 'w';
			break;
	}

	fclose(pfp);
	return Conf_Unlock_File(path,type);*/
}

/*
int Conf_Lock_File( char *lock_file_name, char type)
{
	int retry = LOCK_TIMEOUT;
   	int lock_type;
   	int fd;

   	fd = open(lock_file_name,O_CREAT | O_RDWR);
   	if (fd <= 0)	
        	return ERROR_OPEN_FILE;

   	if (type == 'R' || type == 'r')
      		lock_type = LOCK_SH | LOCK_NB;
   	else
      		lock_type = LOCK_EX | LOCK_NB;

   	while(retry-- > 0) {
      		if (flock(fd,lock_type) == -1) {
          		if (retry > 0)
             		sleep(1);
      		}
      		else
          		return fd;
   	}
	if (type == 'R' || type == 'r') {
   		close(fd);
   		return ERROR_READ_LOCK;
   	}
   	else {
   		close(fd);
   		return ERROR_WRITE_LOCK;
	}
}

int Conf_Unlock_File(int fd)
{
   flock(fd,LOCK_UN);
   close(fd);
   return SUCCESS;
}
*/

//----------------------------------------------------------------------------------------
// get each individual field(substring) from input string, if out_str_buf == NULL
// ,return required length of this field.
// 2001/4/24	Louis Tsai - if delimiter = SPACE or TAB, regard continueous TAB or SPACE 
//			     as a single seperator. 
//----------------------------------------------------------------------------------------

int Get_String_Field(char *in_str,int field_no,char delimiter,char *out_str_buf,int buf_size)
{
	int field_count=0,s_pos=0,t_pos=0;// s_pos: start_position, t_pos: tail position
	char *tmp_str;
	
	in_str = skip_white_space(in_str);
	tmp_str = in_str;

	while( (t_pos = find_char_in_string(tmp_str,delimiter)) != -1) {
		t_pos = t_pos + s_pos;
		field_count++;
		if( field_count != field_no ) {
			s_pos = t_pos + 1;
			tmp_str = in_str+s_pos;
		}
		else break;
	}
	// last field
	if(t_pos == -1 ) {
		if (field_no == field_count+1 )
			t_pos = strlen(in_str)+1;
		else if ( field_no > field_count+1 )
			return ERROR_NOT_EXISTED;
	}

	// return required buffer length
	if ( out_str_buf == NULL) return (t_pos - s_pos+1);

	if ( buf_size < (t_pos - s_pos)+1) {
		strncpy(out_str_buf,in_str+s_pos,buf_size-1);
		out_str_buf[buf_size-1]='\0';
		return ERROR_BUFFER_TOO_SMALL;
	}
	else {
		strncpy(out_str_buf,in_str+s_pos,t_pos - s_pos);
		out_str_buf[t_pos-s_pos] = '\0';
		stripe_white_space_tail(out_str_buf);
		if(out_str_buf[strlen(out_str_buf)-1] == '\n') 
			out_str_buf[strlen(out_str_buf)-1] = '\0';
		return SUCCESS;
	}
}

//--------------------------------------------------------------------------
// set each individual field(substring) from input string, if new_str_buf == NULL
// ,return required length of new string.
// 2001/4/24	Louis Tsai - if delimiter = SPACE or TAB, regard continueous TAB or SPACE 
//			     as a single seperator. 
//--------------------------------------------------------------------------

int Set_String_Field(char *in_str,int field_no,char delimiter,char *field_value,char *new_str_buf,int buf_size)
{
	int field_count=0,s_pos=0,t_pos=0;// s_pos: start_position, t_pos: tail position
	char *tmp_str;
	int len;
	in_str = skip_white_space(in_str);
	tmp_str = in_str;
	while( (t_pos = find_char_in_string(tmp_str,delimiter)) != -1) {
		t_pos = t_pos + s_pos;
		field_count++;
		if( field_count != field_no ) {
			s_pos = t_pos + 1;
			tmp_str = in_str+s_pos;
		}
		else break;
	}
	// last field
	if(t_pos == -1 ) {
		if (field_no == field_count+1 )
			t_pos = strlen(in_str)+1 ;
		else if ( field_no > field_count+1 )
			return ERROR_NOT_EXISTED;
	}

	// return required new string length
	len = strlen(in_str)-(t_pos - s_pos)+strlen(field_value)+1;
	if ( new_str_buf == NULL) return (len);

	// new string buffer less than required
	if ( buf_size< len ) {
		strncpy(new_str_buf,in_str,buf_size-1);
		new_str_buf[buf_size-1]='\0';
		return ERROR_BUFFER_TOO_SMALL;
	}
	else {
		strncpy(new_str_buf,in_str,s_pos);
		new_str_buf[s_pos] = '\0';
		strcat(new_str_buf,field_value);
		if (t_pos != strlen(in_str)+1)
			strcat(new_str_buf,in_str+t_pos);

		stripe_white_space_tail(new_str_buf);
		return SUCCESS;
	}
}

//--------------------------------------------------------------------------
// Add specific field from a string ,if field_no = 0, we assigned it
// become last field. if new_str == NULL,return required length of new string.
// 2001/4/24	Louis Tsai - if delimiter = SPACE or TAB, regard continueous TAB or SPACE 
//			     as a single seperator. 
//--------------------------------------------------------------------------

int Insert_String_Field(char *in_str,int field_no,char delimiter,char *field_value,char *new_str_buf,int buf_size)
{
	int field_count=0,s_pos=0,t_pos=0;// s_pos: start_position, t_pos: tail position
	char *tmp_str;
	int len;
	
	in_str = skip_white_space(in_str);
	// append to last field
	if ( field_no == 0) {
		len = strlen(in_str)+strlen(field_value)+2;// old_str+delimiter+field_value
		if ( new_str_buf == NULL) return (len);
		if ( buf_size < len ) {
			strncpy(new_str_buf,in_str,buf_size-1);
			new_str_buf[buf_size-1]='\0';
			return ERROR_BUFFER_TOO_SMALL;
		}
		else
			if(strcmp(in_str,"")!=0)
				sprintf(new_str_buf,"%s%c%s",in_str,delimiter,field_value);
			else
				sprintf(new_str_buf,"%s",field_value);
		return SUCCESS;
	}


	// append to some field

	tmp_str = in_str;
	while( (t_pos = find_char_in_string(tmp_str,delimiter)) != -1) {
		t_pos = t_pos + s_pos;
		field_count++;
		if( field_count != field_no ) {
			s_pos = t_pos + 1;
			tmp_str = in_str+s_pos;
		}
		else break;
	}

	// last field situation
	if(t_pos == -1 ) {
		if (field_no == field_count+1 )
			t_pos = strlen(in_str)+1 ;
		else if ( field_no > field_count+1 )
			return ERROR_NOT_EXISTED;
	}

	// return required new string length
	len = strlen(in_str)+strlen(field_value)+2;
	if ( new_str_buf == NULL) return (len);

	// new string buffer less than required
	if ( buf_size < len ) {
		strncpy(new_str_buf,in_str,buf_size-1);
		new_str_buf[buf_size-1]='\0';
		return ERROR_BUFFER_TOO_SMALL;
	}
	else {
		strncpy(new_str_buf,in_str,s_pos);
		new_str_buf[s_pos] = delimiter;
		new_str_buf[s_pos+1] = '\0';
		strcat(new_str_buf,field_value);
		strcat(new_str_buf,in_str+s_pos);
		stripe_white_space_tail(new_str_buf);
		return SUCCESS;
	}
}
//--------------------------------------------------------------------------
// remove specific field from a string
// 2001/4/24	Louis Tsai - if delimiter = SPACE or TAB, regard continueous TAB or SPACE 
//			     as a single seperator. 
//--------------------------------------------------------------------------

int Remove_String_Field(char *in_str,int field_no,char delimiter)
{
	int field_count=0,s_pos=0,t_pos=0;// s_pos: start_position, t_pos: tail position
	char *tmp_str;
	
	in_str = skip_white_space(in_str);
	tmp_str = in_str;
	while( (t_pos = find_char_in_string(tmp_str,delimiter)) != -1) {
		t_pos = t_pos + s_pos;
		field_count++;
		if( field_count != field_no ) {
			s_pos = t_pos + 1;
			tmp_str = in_str+s_pos;
		}
		else break;
	}
	// last field
	if(t_pos == -1 ) {
		if (field_no == field_count+1 ) {
			 if (s_pos == 0)
			 	in_str[s_pos]='\0';
			 else
			 	in_str[s_pos-1]='\0';
			 return SUCCESS;
		}
		else if ( field_no > field_count+1 )
			return ERROR_NOT_EXISTED;
	}

	in_str[s_pos]='\0';
	strcat(in_str,in_str+t_pos+1);
	return SUCCESS;
}

char *Parse_String_Field(char *str, char *delimiters)
{
	static char *current_source = NULL;
	static int last_modified_pos = -1;
	static char last_char_to_restore = 0;

	char *p = str;

	if (NULL != str) {
		current_source = str;
		last_modified_pos = -1;
	}
	else
	if (NULL != current_source) {
		if (last_modified_pos >= 0) {
			current_source[last_modified_pos] = last_char_to_restore;
			p = &(current_source[last_modified_pos + 1]);
			last_modified_pos = -1;
		}
		else
			p = NULL;		// has reached EOS...
	}

	if (NULL != p) {
		char *q = p;
		while (*q != '\0') {
			int i;
			for (i = 0; delimiters[i] != '\0'; i++) {
				if (*q == delimiters[i]) {
					last_char_to_restore = *q;
					last_modified_pos = q - current_source;
					*q = '\0';
					return p;
//					break;
				}
			}
//			if ('\0' == *q) break;
			q++;
		}
//		if (strlen(p) == 0) 		// NO MORE FIELD...
//			p = NULL;
	}

	return p;
}

//----------------------------------------------------------------------------//
// SYSTEM file (as group,passwd...,etc) process function                      //
//----------------------------------------------------------------------------//

//----------------------------------------------------------------------------//
// Get one string include specific field from input sysfile                   //
//----------------------------------------------------------------------------//

int SYS_Get_One_String(char *sysfile,int field_no,char delimiter,char *s_value,char *out_str_buf,int buf_size)
{
	FILE *fp = NULL;
	int i = 0;
	int found_flag = 0;
	char *p;

	if ((i = Open_File(sysfile,O_RDONLY, &fp)) != SUCCESS)
		return i;
	
	rewind(fp);
	if (NULL != fp) {
		while ((p = get_one_line_of_string(fp)) != NULL) {
			int len;
			if( (len = Get_String_Field(p,field_no,delimiter,NULL,0)) >0) {
				char tmp_str[len];
				Get_String_Field(p,field_no,delimiter,tmp_str,len);
				if (compare_case_string(s_value,tmp_str) == 0) {
					found_flag = 1;
					break;
				}
			}
		}

		Close_File(sysfile,O_RDONLY, fp);
		
		if ( found_flag == 0) return ERROR_NOT_EXISTED;
		//return required buf size
		if ( out_str_buf == NULL ) return (strlen(p)+1);

		// check if return buf overflow
		if ( buf_size < strlen(p)+1) {
			strncpy(out_str_buf,p,buf_size-1);
			out_str_buf[buf_size-1] = '\0';
			return ERROR_BUFFER_TOO_SMALL;
		}
		else strcpy(out_str_buf,p);
		return SUCCESS;
	}
	return ERROR_OPEN_FILE;
}

//----------------------------------------------------------------------------//
// Set one string include specific field from input sysfile                   //
//----------------------------------------------------------------------------//
int SYS_Set_One_String(char *sysfile,int field_no,char delimiter,char *s_value,char *set_str)
{
	FILE *fp = NULL,*tmpfp = NULL;
	int i = 0;
	int found_flag = 0;
	char *p;

	if ((i = Open_File(sysfile,O_RDONLY, &fp)) != SUCCESS) 
		return i;
	
	
	rewind(fp);
	tmpfp = tmpfile();
	if (NULL != fp) {
		while ((p = get_one_line_of_string(fp)) != NULL) {
			int len;
			if( (len = Get_String_Field(p,field_no,delimiter,NULL,0)) >0) {
				char tmp_str[len];
				Get_String_Field(p,field_no,delimiter,tmp_str,len);
				if (compare_case_string(s_value,tmp_str) == 0) {
					found_flag = 1;
					fprintf(tmpfp,"%s\n",set_str);
					continue;
				}
				//fputs(p,tmpfp);
				//johnshiau 2001-11-23: modified for avoiding that don't write into file.
				if(fputs(p, tmpfp) < 0) {
					Close_File(sysfile,O_RDONLY,fp);
					return ERROR_WRITE_FILE;
				}
				//end
				fputs("\n", tmpfp);
			}
		}
	}
	else {
		Close_File(sysfile,O_RDONLY,fp);
		return ERROR_OPEN_FILE;
	}
	
	Close_File(sysfile,O_RDONLY,fp);
	
	if(found_flag) {
		if ((i = Open_File(sysfile,O_WRONLY, &fp)) == SUCCESS) {
			rewind(tmpfp);
			while ((p = get_one_line_of_string(tmpfp)) != NULL) {
				fputs(p, fp);
				fputs("\n", fp);
			}
			Close_File(sysfile,O_WRONLY,fp);
			fclose(tmpfp);
			return SUCCESS;
		}
		else {
			fclose(tmpfp);
			return i;
		}
		
	}
	else {
		fclose(tmpfp);
		return ERROR_NOT_EXISTED;
	}
}

///////////////////////////////////////////////////////////////////////////////////

void Conf_Case_Sensitive(BOOL flag)
{
	case_flag = flag;
}


int Conf_Get_Field(char *conf_file, char *section, char *field, char *result, int buf_size)
{
//	char *in_section = NULL;
	FILE *fp = NULL;
//	char section_name[SECTION_NAME_MAX_LENGTH] = "";
	char *section_name;
	char *p, *q;
	int section_found = 0;
	int i = 0;
	int	ret;

	/* use share memory to speed up	*/
	if (!strcmp(conf_file, "/etc/config/uLinux.conf"))
	{
		if (!strcasecmp(section, "System") || !strcasecmp(section, "Storage") || !strcasecmp(section, "Misc"))
		{
			ret=SHM_Get_Field(section, field, result, buf_size);
			if (ret!=ERROR_NOT_FOUND)
				return ret;
		}

	}

	if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
		return i;
	
	if (EMPTY_STRING(section))
		section_found = 1;

	while ((p = get_one_line_of_string(fp)) != NULL) {
		q = skip_white_space(p);

		if (*q == '[')
		{	//start of section name...
			section_name=q+1;
			p=section_name+strlen(section_name)-1;
			while (*p!=']')
			{
				if (p==section_name)
					break;
				p--;
			}
			if (p==section_name)
			{
				if (section_found)
					goto this_is_item;
				continue;
			}
			else
			{
				if (section_found)
					break;
			}
			*p=0x0;
			section_name=skip_white_space(section_name);
			section_name=stripe_white_space_tail(section_name);
			if (compare_case_string(section_name, section)==0)
				section_found=1;

/*			if (get_string_until_ch(q+1, ']', section_name, sizeof(section_name)) != NULL) {
				if (section_found) 		// already scan the whole section...
					break;
				if (compare_case_string(section_name, section) == 0)
					section_found = 1;
			}*/
		}
		else
		if (section_found) {
this_is_item:
			if (is_commented_string(q))
				continue;
			else
			{	// not a section declaration, should be field data...
/*				char buf[FIELD_NAME_MAX_LENGTH] = "";

				if (get_string_until_ch(q, '=', buf, sizeof(buf)) == NULL)
					continue;
*/
				char	*item, *value;

				item=p=q;
//				if (*item!=*field)
//					continue;
				while (*p!='=')
				{
					if (*p==0x0)
						break;
					p++;
				}
				if (*p==0x0)
					continue;
				*p=0x0;
				value=p+1;
				item=stripe_white_space_tail(item);

				if (compare_case_string(field, item)==0)
				{
					value=skip_white_space(value);
					value=stripe_white_space_tail(value);
					p=value+strlen(value)-1;
					while (*p==0x0a || *p==0x0d)
					{
						if (p==value)
							break;
						p--;
					}
					if (*p==0x0a || *p==0x0d)
						*p=0x0;
					*(p+1)=0x0;
					if (result==NULL)
					{
						Close_File(conf_file, O_RDONLY, fp);
						return (strlen(value)+1);
					}
					if (buf_size<strlen(value)+1)
					{
						Close_File(conf_file, O_RDONLY, fp);
						return ERROR_BUFFER_TOO_SMALL;
					}
					strcpy(result, value);
					Close_File(conf_file, O_RDONLY, fp);
					return SUCCESS;
				}

/*				if (compare_case_string(field, buf) == 0) {
					int j;
					q = q + find_char_in_string(q, '=') + 1;	// skip to the next character after '='
					q = skip_white_space(q);
					if (NULL != result) {
						result[0]='\0'; //empty string
						j = strlen(q);
						if (j+1 > buf_size) {
							j = buf_size - 1;
							strncpy(result, q, j);
							result[j] = '\0';
							stripe_white_space_tail(result);
							Close_File(conf_file,O_RDONLY,fp);
							return ERROR_BUFFER_TOO_SMALL;
						}
						else {
							strncpy(result, q, j);
							result[j] = '\0';
							stripe_white_space_tail(result);
							Close_File(conf_file,O_RDONLY,fp);
							return SUCCESS;
						}

					}
					else { 
						Close_File(conf_file,O_RDONLY,fp);
						return (strlen(q)+1); //return required buffer length
					}
				}
*/
			}
		}
	}
	Close_File(conf_file,O_RDONLY,fp);
	return ERROR_NOT_FOUND;
}



int Conf_Enum_Section(char *conf_file, char *section_name, int buf_size)
{
	static FILE *current_fp = NULL;
	static char tmp_conf_file[FILE_NAME_MAX_LENGTH];
	FILE *fp = NULL;
	char *p;
	int i;

	if (conf_file != NULL) {

		if (NULL != current_fp) {
			Close_File(tmp_conf_file,O_RDONLY,current_fp);		
			current_fp = NULL;
		}

		if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
			return i;
		strcpy(tmp_conf_file,conf_file);
		current_fp = fp;
		rewind(fp);
	}
	if (NULL != current_fp) {
		while ((p = get_one_line_of_string(current_fp)) != NULL) {
			if (get_section_name(p, section_name, buf_size) == SUCCESS) {
				return SUCCESS;
			}
		}
	}

	if (NULL != current_fp) {
		Close_File(tmp_conf_file,O_RDONLY,current_fp);
		current_fp = NULL;
	}

	return ERROR_FAIL;
}

int Conf_Remove_Section(char *conf_file, char *section_name)
{
	FILE *fp = NULL, *tmpfp = NULL;
	char buf[BUF_SIZE];
	char *p;
	int i, dirty = 0, result = ERROR_FAIL;

	if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
		return i;
	
	rewind(fp);
	tmpfp = tmpfile();		// temp file for buffer

	if (NULL != fp) {
		int section_found = 0;
		if (EMPTY_STRING(section_name)) section_found = 1;
		while ((p = get_one_line_of_string(fp)) != NULL) {
			if (get_section_name(p, buf, sizeof(buf)) == SUCCESS) {
				if (compare_case_string(section_name, buf)== 0)
					section_found = 1;
				else
					section_found = 0;
			}
			if (section_found) {
				dirty++;
				continue;
			}
			//fputs(p,tmpfp);
			//johnshiau 2001-11-23: modified for avoiding that don't write into file.
			if(fputs(p, tmpfp) < 0) {
				Close_File(conf_file,O_RDONLY,fp);
				if (tmpfp) fclose(tmpfp);
				return ERROR_WRITE_FILE;
			}
			//end
			fputs("\n", tmpfp);
		}

		Close_File(conf_file,O_RDONLY,fp);

		if (dirty) {
			//  Copy_File(fp, tmpfp);
			if ((i = Open_File(conf_file,O_WRONLY, &fp)) == SUCCESS) {
				rewind(tmpfp);
				while ((i = fread(buf, 1, sizeof(buf), tmpfp)) > 0) {
					fwrite(buf, 1, i, fp);
				}
				Close_File(conf_file,O_WRONLY,fp);
				result = SUCCESS;
			}
			else
				result = i;
				
		}
		else
			result = ERROR_NOT_FOUND;
		
	}
	if (tmpfp) fclose(tmpfp);

	return result;
}

int Conf_Remove_Field(char *conf_file, char *section_name, char *field_name)
{
	FILE *fp = NULL, *tmpfp = NULL;
	char buf[BUF_SIZE];
	char *p;
	int i, dirty = 0, result = ERROR_FAIL;


	if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
		return i;
	
	rewind(fp);
	tmpfp = tmpfile();		// temp file for buffer

	if (NULL != fp) {
		int section_found = 0;
		if (EMPTY_STRING(section_name)) section_found = 1;
		while ((p = get_one_line_of_string(fp)) != NULL) {
			if (get_section_name(p, buf, sizeof(buf)) == SUCCESS) {
				if (compare_case_string(section_name, buf) == 0)
					section_found = 1;
				else
					section_found = 0;
			}
			else
			if (section_found) {
				if (get_field_name(p, buf, sizeof(buf)) == SUCCESS) {
					if (compare_case_string(field_name, buf) == 0) {
						dirty++;
						continue;	// don't copy this line...
					}
				}
			}
			//fputs(p,tmpfp);
			//johnshiau 2001-11-23: modified for avoiding that don't write into file.
			if(fputs(p, tmpfp) < 0) {
				Close_File(conf_file,O_RDONLY,fp);
				if (tmpfp) fclose(tmpfp);
				return ERROR_WRITE_FILE;
			}
			//end
			fputs("\n", tmpfp);
		}

		Close_File(conf_file,O_RDONLY,fp);

		if (dirty) {
			//  Copy_File(fp, tmpfp);
			if ((i = Open_File(conf_file,O_WRONLY, &fp)) == SUCCESS) {
				rewind(tmpfp);
				while ((i = fread(buf, 1, sizeof(buf), tmpfp)) > 0) {
					fwrite(buf, 1, i, fp);
				}
				Close_File(conf_file,O_WRONLY,fp);
				result = SUCCESS;
			}
			else
				result = i;
		}
		else
			result = ERROR_NOT_FOUND;
	}
	if (tmpfp) fclose(tmpfp);
	return result;
}

int Conf_Set_Field(char *conf_file, char *section_name, char *field_name, char *field_data)
{
	FILE *fp = NULL, *tmpfp = NULL;
	char buf[BUF_SIZE];
	char *p;
	int i, dirty = 0, result = ERROR_FAIL;
	char	tmpfile[256], *ptr;
	char	line[BUF_SIZE];
	BOOL line_complete;

	if ((i = Open_File(conf_file,O_RDONLY, &fp)) != SUCCESS)
		return i;

	ptr=get_filename(conf_file);
	sprintf(tmpfile, "/tmp/%s.bak", ptr);
	tmpfp=fopen(tmpfile, "w+");

	if (NULL != fp)
	{
		int section_found = 0, field_found = 0;
		if (EMPTY_STRING(section_name))
			section_found = 1;

		while (!feof(fp))
		{
			if (fgets(line, BUF_SIZE, fp)==NULL)
				continue;
			if (line[strlen(line)-1] == 0x0a || line[strlen(line)-1] == 0x0d) {
				line[strlen(line)-1]=0x0;
				line_complete = TRUE;
			}
			else line_complete = FALSE;

			p=line;
			if (get_section_name(p, buf, sizeof(buf)) == SUCCESS) {
				if (compare_case_string(section_name, buf) == 0)
					section_found = 1;
				else {
					if (section_found && (!dirty)) {	// no existing field in section
						dirty++;
						fprintf(tmpfp, "%s = %s\n", field_name, field_data);
					}
					section_found = 0;
				}
			}
			else if (section_found) {
				if (get_field_name(p, buf, sizeof(buf)) == SUCCESS) {
					if (compare_case_string(field_name,buf) == 0) {	// field name match...
						dirty++;
						field_found = 1;
						fprintf(tmpfp, "%s = %s\n", field_name, field_data);
						continue;	// don't copy this line...
					}
					else
						field_found = 0;
				}
				else if (field_found) // the rest of the section's data (too much for the line buffer)
					continue;
			}

			//fputs(p,tmpfp);
			//johnshiau 2001-11-23: modified for avoiding that don't write into file.
			if(fputs(p, tmpfp) < 0) {
				Close_File(conf_file,O_RDONLY,fp);
				if (tmpfp) fclose(tmpfp);
				return ERROR_WRITE_FILE;
			}
			//end
			if (line_complete) fputs("\n", tmpfp);
		}
		if (!dirty) {
			if (!section_found)
				fprintf(tmpfp, "[%s]\n", section_name);
			fprintf(tmpfp, "%s = %s\n", field_name, field_data);
			dirty++;
		}
		Close_File(conf_file,O_RDONLY,fp);
		if (dirty) {
			if ((i = Open_File(conf_file,O_WRONLY, &fp)) == SUCCESS) {
				/* critical section	*/
				rewind(tmpfp);
				while ((i = fread(buf, 1, sizeof(buf), tmpfp)) > 0) {
					fwrite(buf, 1, i, fp);
				}

			        /* set to share memory to become NAS cache      */
			        if (!strcmp(conf_file, "/etc/config/uLinux.conf"))
			        {
			                if (!strcasecmp(section_name, "System") || !strcasecmp(section_name, "Storage") || !strcmp(section_name, "Misc"))
                        			SHM_Set_Field(section_name, field_name, field_data);
			        }

				Close_File(conf_file,O_WRONLY,fp);
				result = SUCCESS;
			}
			else
				result = i;
		}
	}
	if (tmpfp) {
		fclose(tmpfp);
		unlink(tmpfile);
	}

	return result;
}


// The return value is the number of characters copied to the buffer, not including the
// terminating null character.
int GetPrivateProfileString(
  char* lpAppName,        // section name
  char* lpKeyName,        // key name
  char* lpDefault,        // default string
  char* lpReturnedString,  // destination buffer
  int   nSize,              // size of destination buffer
  char* lpFileName        // initialization file name
)
{

	if (Conf_Get_Field(lpFileName, lpAppName, lpKeyName, lpReturnedString, nSize) == ERROR_NOT_FOUND) {
		if (strlen(lpDefault)<nSize)
			strncpy(lpReturnedString,lpDefault,nSize);
		else {
			strncpy(lpReturnedString,lpDefault,nSize-1);
			lpDefault[nSize] = '\0';
		}
	}

	return strlen(lpReturnedString);
}

BOOL WritePrivateProfileString(
  char *lpAppName,  // section name
  char *lpKeyName,  // key name
  char *lpString,   // string to add
  char *lpFileName  // initialization file
)
{
	if (Conf_Set_Field(lpFileName, lpAppName, lpKeyName, lpString)==SUCCESS)
		return TRUE;
	else
		return FALSE;
}

// Get configuration from system default config file. If there is no data, the
// default string will be returned.
// Return:
// 	The return value is the number of characters copied to the buffer, not including the
// 	terminating null character.
int GetProfileString(char *lpAppName, char *lpKeyName, char * lpDefault, char * lpReturnedString, int nSize)
{
	return (GetPrivateProfileString(lpAppName, lpKeyName, lpDefault, lpReturnedString,
		nSize, DEFAULT_CFG_FILE));
}

BOOL WriteProfileString(char * lpAppName, char * lpKeyName, char * lpString)
{
	return (WritePrivateProfileString(lpAppName, lpKeyName, lpString, DEFAULT_CFG_FILE));
}

//Get configuration from system NetLinks.conf config file. If there is no data, the
// default string will be returned.
// Return:
// 	The return value is the number of characters copied to the buffer, not including the
// 	terminating null character.
// added by YF

int GetProfileStringNetLinks(char *lpAppName, char *lpKeyName, char * lpDefault, char * lpReturnedString, int nSize)
{
	return (GetPrivateProfileString(lpAppName, lpKeyName, lpDefault, lpReturnedString,
		nSize, NETLINKS_CFG_FILE));
}

BOOL WriteProfileStringNetLinks(char * lpAppName, char * lpKeyName, char * lpString)
{
	return (WritePrivateProfileString(lpAppName, lpKeyName, lpString, NETLINKS_CFG_FILE));
}

//----------------------------------- NasWare 2.0 start -------------------------------------------//

/*---------- Profile handle function ----------*/

int  Get_Profile_String(char *lpAppName, char *lpKeyName, char * lpDefault, char * lpReturnedString, int nSize)
{
	return GetProfileString(lpAppName,lpKeyName,lpDefault,lpReturnedString,nSize);
}

BOOL Set_Profile_String(char * lpAppName, char * lpKeyName, char * lpString)
{
	return WriteProfileString(lpAppName,lpKeyName,lpString);	
}

BOOL Get_Profile_Boolean(char *lpAppName, char *lpKeyName,BOOL default_value)
{
/* JohnShiau 2001-9-27: modified for better compatibility
	char m_default[10],ReturnedString[128];
	int ret;
	
	sprintf(m_default,"%d",default_value);
	
	ret = Get_Profile_String(lpAppName,lpKeyName,m_default,ReturnedString,128);
	if(ret<0) ret = default_value;
	else {
		if( (strcasecmp(ReturnedString,"TRUE") == 0) || (strcasecmp(ReturnedString,"1") == 0) )
			ret = TRUE;
		else 
			ret = FALSE;
	}
	return ret;
*/	
	char str[128];
	Get_Profile_String(lpAppName,lpKeyName,BOOL2STR(default_value),str,sizeof(str));
	return STR2BOOL(str);
}

BOOL Set_Profile_Boolean(char *lpAppName, char *lpKeyName,BOOL bValue)
{ 
/* JohnShiau 2001-9-27: modified For better compatibility
	char m_value[5];
	sprintf(m_value,"%d",bValue);
	
	return Set_Profile_String(lpAppName,lpKeyName,m_value);
*/
	return Set_Profile_String(lpAppName,lpKeyName,BOOL2STR(bValue));
}

int  Get_Profile_Integer(char *lpAppName, char *lpKeyName,int default_value)
{
/* JohnShiau 2001-9-27: modified for better compatibility
	char m_default[64],ReturnedString[128];
	int ret;
	
	sprintf(m_default,"%d",default_value);
	
	ret = Get_Profile_String(lpAppName,lpKeyName,m_default,ReturnedString,128);
	if(ret>=0)
		sscanf(ReturnedString,"%d",&ret);
	else
		ret = default_value;
	
	return ret;	
*/
	char val[64],str[128];
	sprintf(val,"%d",default_value);
	Get_Profile_String(lpAppName, lpKeyName, val, str, sizeof(str));
	return atoi(str);
}

BOOL  Set_Profile_Integer(char *lpAppName, char *lpKeyName,int nValue)
{
	char m_value[64];
	sprintf(m_value,"%d",nValue);
	return Set_Profile_String(lpAppName,lpKeyName,m_value);
}


/*------ Private profile handle function --------*/

int  Get_Private_Profile_String( char* lpAppName,char* lpKeyName,char* lpDefault,char* lpReturnedString,int nSize,char* lpFileName)
{
	return GetPrivateProfileString(lpAppName,lpKeyName,lpDefault,lpReturnedString,nSize,lpFileName);
}

BOOL Set_Private_Profile_String(char *lpAppName,char *lpKeyName,char *lpString,char *lpFileName)
{
	return WritePrivateProfileString(lpAppName,lpKeyName,lpString,lpFileName);	
}

BOOL Get_Private_Profile_Boolean(char *lpAppName, char *lpKeyName,BOOL default_value,char *lpFileName)
{
/* Meiji 2001-09-26: modified for better compatibility.
	char m_default[10],ReturnedString[128];
	int ret;
	
	sprintf(m_default,"%d",default_value);
	
	ret = Get_Private_Profile_String(lpAppName,lpKeyName,m_default,ReturnedString,128,lpFileName);
	if(ret<0) ret = default_value;
	else {
		if( (strcasecmp(ReturnedString,"TRUE") == 0) || (strcasecmp(ReturnedString,"1") == 0) )
			ret = TRUE;
		else 
			ret = FALSE;
	}
	return ret;	
*/
	char str[128];
	Get_Private_Profile_String(lpAppName, lpKeyName, BOOL2STR(default_value), str, sizeof(str), lpFileName);
	return STR2BOOL(str);
}

BOOL Set_Private_Profile_Boolean(char *lpAppName, char *lpKeyName,BOOL bValue,char *lpFileName)
{
/* Meiji 2001-09-26: modified for better compatibility.
	char m_value[5];
	sprintf(m_value,"%d",bValue);

	return Set_Private_Profile_String(lpAppName,lpKeyName,m_value,lpFileName);
*/
	return Set_Private_Profile_String(lpAppName, lpKeyName, BOOL2STR(bValue), lpFileName);
}

int  Get_Private_Profile_Integer(char *lpAppName, char *lpKeyName,int default_value,char *lpFileName)
{
/* JohnShiau 2001-9-27: modified for better compatibility
	char m_default[64],ReturnedString[128];
	int ret;
	
	sprintf(m_default,"%d",default_value);
	
	ret = Get_Private_Profile_String(lpAppName,lpKeyName,m_default,ReturnedString,128,lpFileName);
	if(ret>=0)
		sscanf(ReturnedString,"%d",&ret);
	else
		ret = default_value;
		
	return ret;
*/		
	char val[64],str[128];
	sprintf(val,"%d",default_value);
	Get_Private_Profile_String(lpAppName,lpKeyName,val,str,sizeof(str),lpFileName);
	return atoi(str);
}

BOOL  Set_Private_Profile_Integer(char *lpAppName, char *lpKeyName,int nValue,char *lpFileName)
{
	char m_value[64];
	sprintf(m_value,"%d",nValue);
	
	return Set_Private_Profile_String(lpAppName,lpKeyName,m_value,lpFileName);
}

char *File_Fgets(FILE *fptr)
{
	char	*ptr;
	int	ssize=0, cnt=0;

	if ((ptr=(char *)calloc(1, 1024))==NULL)
		return NULL;
	while (1)
	{
		fgets(ptr+ssize-cnt, 1024, fptr);
		if (feof(fptr))
		{
			free(ptr);
			return NULL;
		}
		if (ptr[strlen(ptr)-1]==0x0a)
		{
//			ptr[strlen(ptr)-1]=0x0;
			break;
		}
		ssize+=1024;
		cnt++;
		realloc(ptr, ssize+1024);
		if (ptr==NULL)
			return NULL;
		memset(ptr+ssize, 0, 1024);
	}
	return ptr;
}
