//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		raid.c
//
//	Abstract: 
//		 raid relative fuction for storage library
//
//	HISTORY:
//		2002/09/26	Catherine Shen -- modify mdx->mdxx
//		2002/08/15	Catherine Shen -- remove the code that assumes
//					drive number never exceeds 4
//		2002/06/11	Catherine Shen -- modify "Get_All_RaidDev_Status"
//					to take "resync" percentage back w/ a
//					bias "RAID_RESYNC_PER_BASE".
//		2002/05/24	Catherine Shen -- modify "Get_All_RaidDev_Status"
//					to solve a bug that some disks are not in
//					the drive lists
//		2002/05/20	Catherine Shen -- modify "Get_All_RaidDev_Status"
//					so that the active_drive_no lists the
//					drive in the exact sequence of the drives
//					in the status file
//		2002/03/11	Catherine Shen -- take "resync" as normal
//		2002/02/20	Catherine Shen -- support hot-spare
//					add 'Create_NAS_RaidDev_Ex()',
//					modified 'Get_Raid_Level()'
//		2001/5/21	Louis Tsai created 
//		2001/6/1	Louis Tsai self test ok
//
//**************************************************************************
#include <sys/stat.h>
#include <fcntl.h>
#include <linux/raid/md_u.h>	
#include <sys/ioctl.h>
#include <linux/hdreg.h>
#include <linux/major.h>
#include "storage.h"
#include "storage_err.h"
#include "file_lock.h"

int Get_RaidDev_Count(void)
{	
	FILE *fp;
    	char buff[512];
    	int  raid_no=0;
// 2002/08/15 Catherine ==>
	char *p;
	int raidno;
// <== 2002/08/15 Catherine

	if (NAS_File_Lock(RAID_CONF_FILE, 'r')==0)
		return 0;
    	fp = fopen(RAID_CONF_FILE,"r");
    	if (fp==NULL )
	{
		NAS_File_Unlock(RAID_CONF_FILE);
		return 0;
	}

   	 while(1) {
      		if (fgets(buff,sizeof(buff),fp)==NULL)  break;   /*- EOF or Error -*/
      		if ( buff[0]=='#')  continue;        /*- skip comment line -*/
/* 2002/08/15 Catherine ==>
      		if (strstr(buff,"raiddev") == NULL ||
      		    strstr(buff,"md4") != NULL ||
      		    strstr(buff,"md5") != NULL )
      			continue;
*/
		p = strstr(buff, "raiddev");
		if (p==NULL) continue; // not a RAID device string

		p = strstr(buff, "md");
		if (p==NULL) continue; // invalid RAID name

		raidno = atoi(p+2);
		// a swap RAID
		if (raidno >= Get_Swap_RAID_No_Base())
			continue;
// <== 2002/08/15 Catherine
      		
      		raid_no++;
    	}

    	fclose(fp);
	NAS_File_Unlock(RAID_CONF_FILE);
    	return raid_no;	
}



int Get_All_RaidDev_Conf(RAID_DEVICE *rd_list, int list_cnt)
{
 	char line[BUF_SIZE],value[BUF_SIZE],device_name[BUF_SIZE];
    	FILE *fp;
    	int  ret=0, raid_cnt=0, drive_no=0, swap_md_flag=0, err_flag=0;
    	int  raid_disk_cnt=0,spare_disk_cnt=0;
    	RAID_DEVICE *c_raid_ptr;
    	char tmp_no[10];
	int drive_cnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);

	if(rd_list == NULL)
		return Get_RaidDev_Count();

	if (NAS_File_Lock(RAID_CONF_FILE, 'r')==0)
		return ERROR_LOCK_FILE;
    	fp = fopen(RAID_CONF_FILE,"r");
    	if (fp==NULL)
	{
		NAS_File_Unlock(RAID_CONF_FILE);
		return ERROR_OPEN_FILE;
	}

    	while(1) {
    		if(swap_md_flag == 1) {
			swap_md_flag = 0;
			--raid_cnt;	
		}
      		if (fgets(line,sizeof(line),fp)==NULL) break;
		Get_String_Field(line,1,SPACE,value,BUF_SIZE);
		if(strcasecmp(value,"raiddev") == 0 ) {
			char *p=strstr(line, "md");
			int raid_no = (p==NULL) ? 0 : atoi(p+2);

			if (raid_no >= Get_Swap_RAID_No_Base()) {
				swap_md_flag = 1;
			}
			if (err_flag > 0 && raid_disk_cnt <= err_flag)
				raid_cnt--;

			c_raid_ptr = rd_list + raid_cnt;
			Get_String_Field(line,2,SPACE,c_raid_ptr->name,HD_DEVICE_NAME_LENGTH);

			raid_cnt++;
			raid_disk_cnt=0;
			spare_disk_cnt=0;
			err_flag = 0;

			if(raid_cnt>list_cnt) {
				ret = ERROR_BUFFER_TOO_SMALL;
				break;		
			}
			continue;	
		}
		if(strcasecmp(value,"raid-level") == 0 ) {
			Get_String_Field(line,2,SPACE,c_raid_ptr->raid_level,7);
			continue;		
		}
		if(strcasecmp(value,"nr-raid-disks") == 0) {
			Get_String_Field(line,2,SPACE,tmp_no,5);
			c_raid_ptr->data_drive_cnt = atoi(tmp_no);
			continue;
		}		
		if(strcasecmp(value,"nr-spare-disks") == 0){
			Get_String_Field(line,2,SPACE,tmp_no,5);
			c_raid_ptr->spare_drive_cnt = atoi(tmp_no);
			continue;
		}
		if(strcasecmp(value,"chunk-size") == 0) continue;
		if(strcasecmp(value,"persistent-superblock") == 0) continue;
		if(strcasecmp(value,"device") == 0) {
			Get_String_Field(line,2,SPACE,device_name,BUF_SIZE);
			drive_no = Get_Drive_No(device_name);
			if (fgets(line,sizeof(line),fp)==NULL) {
				ret = ERROR_FAIL;
				break;
			}
			Get_String_Field(line,1,SPACE,value,BUF_SIZE);
			if(strcasecmp(value,"raid-disk") == 0) {
				if (Get_System_HD_Type() == SCSI && 
					drive_no > drive_cnt)
				{ // scsi max drive number always equals to the drive count
					err_flag++;
				}
				c_raid_ptr->data_drive_no[raid_disk_cnt++] = drive_no;
				if(raid_disk_cnt >c_raid_ptr->data_drive_cnt) {
					ret = ERROR_FAIL;
					break;		
				} 
				continue;	
			}

			if(strcasecmp(value,"spare-disk") == 0) {
				if (Get_System_HD_Type() == SCSI && 
					drive_no > drive_cnt)
				{ // scsi max drive number always equals to the drive count
					err_flag++;
				}
				c_raid_ptr->spare_drive_no[spare_disk_cnt++] = drive_no;
				if(spare_disk_cnt >c_raid_ptr->spare_drive_cnt) {
					ret = ERROR_FAIL;
					break;		
				} 
				continue;	
			}
		
		}
      	}
      	fclose(fp);
	NAS_File_Unlock(RAID_CONF_FILE);
      	
 	if(ret < 0)
   		return ret;
   	else
   		return raid_cnt;
   		
   		
   	
}

//===========================================================//
//check if some hardrive belong to some raid, if TRUE,put raid
//device name into raid_device_name buffer. if null buffer, 
//just return TRUE or FALSE.
//===========================================================//

BOOL Is_Raid_HD(int drive_no,char *raid_device_name,int buf_size)
{
	int rd_cnt,ret;
	RAID_DEVICE *rd=NULL;
	int i,j;
	int drvcnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	int swap_raid_cnt = drvcnt /2;

	if (drvcnt <=1 ) return FALSE;

	//allocat memory
	rd_cnt = Get_RaidDev_Count();
	if (rd_cnt <=0) return FALSE;

	rd = calloc(rd_cnt+swap_raid_cnt,sizeof(RAID_DEVICE));
	if (rd == NULL) return FALSE;

	ret = Get_All_RaidDev_Conf(rd,rd_cnt+swap_raid_cnt);

	for(i=0;i<rd_cnt;i++) {
		// check raid data disk
		for(j=0;j<rd[i].data_drive_cnt;j++) {
			if(drive_no == rd[i].data_drive_no[j]) {
				if(raid_device_name != NULL) {
					if(buf_size<strlen(rd[i].name)) {
						if (rd!=NULL) free(rd);
						return ERROR_BUFFER_TOO_SMALL;
					}
					strcpy(raid_device_name,rd[i].name);
				}
				if (rd!=NULL) free(rd);
				return TRUE;
			}
		}

		// check raid spare disk
		for(j=0;j<rd[i].spare_drive_cnt;j++) {
			if(drive_no == rd[i].spare_drive_no[j]) {
				if(raid_device_name != NULL) {
					if(buf_size<strlen(rd[i].name)) {
						if (rd!=NULL) free(rd);
						return ERROR_BUFFER_TOO_SMALL;
					}
					strcpy(raid_device_name,rd[i].name);
				}
				if (rd!=NULL) free(rd);
				return TRUE;
			}
		}	
	}

	if (rd!=NULL) free(rd);

	return FALSE;
}


int Get_First_Available_RaidName(char *rd_name,int rd_name_len)
{
	int rd_cnt,ret;
	int i,j;
	char tmp_rd_name[HD_DEVICE_NAME_LENGTH];
	
	rd_cnt = Get_RaidDev_Count();
	
		
	if(rd_cnt > 0) {
		RAID_DEVICE rd[rd_cnt+2];
		ret = Get_All_RaidDev_Conf(rd,rd_cnt+2);
		if (ret<0) return ret;
		
		for(i=0;i<MAX_RAID_NO;i++) {
			int rep_flag = 0;
			sprintf(tmp_rd_name,"/dev/md%02d",i);
			for(j=0;j<rd_cnt;j++) {
				if(strcmp(tmp_rd_name,rd[j].name) == 0) {
					rep_flag =1;
					break;
				}
			}
			
			if (rep_flag == 0) {
				if (rd_name_len < strlen(tmp_rd_name)) {
					ret = ERROR_BUFFER_TOO_SMALL;
					break;	
				}
				
				strcpy(rd_name,tmp_rd_name);
				ret = SUCCESS;
				break;
			}
		}
	}
	else {
		strcpy(tmp_rd_name,"/dev/md00");
		if (rd_name_len < strlen(tmp_rd_name))
			ret = ERROR_BUFFER_TOO_SMALL;
		else {
			strcpy(rd_name,tmp_rd_name);
			ret = SUCCESS;	
		}
	}
	
	return ret;
}

//===========================================================//
//Add one raid array in /etc/raidtab!Before use this function
//,must use Get_First_Available_RaidName() to get correct raid
// device name!
//===========================================================//

int Create_RaidDev(RAID_DEVICE *rd_ptr)
{
	int  i;
    	char buff[1024],d_name[100], tmp_str[HD_DEVICE_NAME_LENGTH];
	FILE *fp;
    	
    	if ( rd_ptr->data_drive_cnt <=0 )     return RAID_DATA_ERROR;
    
    	if ( Get_RaidDev_Count() >= MAX_RAID_NO)  return RAID_OVER_MAX_NO;
    	
    	if(Is_RaidDev_Exist(rd_ptr->name)) return ERROR_RAIDDEV_ALREADY_EXIST;
    	/*-- fill the device name --*/
    	sprintf(buff,"raiddev %s\n",rd_ptr->name);

    	/*-- fill raid level --*/
    	sprintf(d_name,"\traid-level\t%s\n",rd_ptr->raid_level);
      	strcat(buff,d_name);

    	/*-- fill data disk No. ---*/
    	sprintf(d_name,"\tnr-raid-disks\t%d\n",rd_ptr->data_drive_cnt);
      	strcat(buff,d_name);

    	/*-- fill spare disk No. ---*/
    	sprintf(d_name,"\tnr-spare-disks\t%d\n",rd_ptr->spare_drive_cnt);
      	strcat(buff,d_name);

   	/*-- fill chunk size ---*/
    	sprintf(d_name,"\tchunk-size\t4\n");
     	strcat(buff,d_name);

   	/*-- fill super-block ---*/
    	sprintf(d_name,"\tpersistent-superblock\t1\n");
      	strcat(buff,d_name);

   	/*-- fill data HD Name ---*/
   	for (i=0; i<rd_ptr->data_drive_cnt; i++) {
   		if(Is_Raid_HD(rd_ptr->data_drive_no[i],NULL,0)) return RAID_HD_EXIST;
       		Get_Partition_Name(rd_ptr->data_drive_no[i],DATA_PART,tmp_str);
       		sprintf(d_name,"\tdevice\t%s\n\traid-disk\t%d\n",tmp_str,i);
       		strcat(buff,d_name);
   	}

   	/*-- fill spare HD Name ---*/
   	for (i=0; i<rd_ptr->spare_drive_cnt; i++) {
   		if(Is_Raid_HD(rd_ptr->spare_drive_no[i],NULL,0)) return RAID_HD_EXIST;
       		Get_Partition_Name(rd_ptr->spare_drive_no[i],DATA_PART,tmp_str);
       		sprintf(d_name,"\tdevice\t%s\n\tspare-disk\t%d\n",tmp_str,i);
       		strcat(buff,d_name);
   	}
	
	if (NAS_File_Lock(RAID_CONF_FILE, 'w')==0)
		return ERROR_LOCK_FILE;
	if((fp=fopen(RAID_CONF_FILE,"a+"))!=NULL) {
		fputs(buff,fp);
		fclose(fp);
	}
	else
	{
		NAS_File_Unlock(RAID_CONF_FILE);
		return ERROR_FAIL;
	}
	NAS_File_Unlock(RAID_CONF_FILE);
	return SUCCESS;
}

int Create_Raid01Dev()
{
    	char buff[1024],d_name[100];
	FILE *fp;
    	
    	/*-- fill the device name --*/
    	sprintf(buff,"raiddev %s\n","/dev/md2");

    	/*-- fill raid level --*/
    	sprintf(d_name,"\traid-level\t%s\n","1");
      	strcat(buff,d_name);

    	/*-- fill data disk No. ---*/
    	sprintf(d_name,"\tnr-raid-disks\t%d\n",2);
      	strcat(buff,d_name);

    	/*-- fill spare disk No. ---*/
    	sprintf(d_name,"\tnr-spare-disks\t%d\n",0);
      	strcat(buff,d_name);

   	/*-- fill chunk size ---*/
    	sprintf(d_name,"\tchunk-size\t4\n");
     	strcat(buff,d_name);

   	/*-- fill super-block ---*/
    	sprintf(d_name,"\tpersistent-superblock\t1\n");
      	strcat(buff,d_name);

   	/*-- fill data HD Name ---*/
   
       	sprintf(d_name,"\tdevice\t%s\n\traid-disk\t%d\n","/dev/md0",0);
       	strcat(buff,d_name);
	sprintf(d_name,"\tdevice\t%s\n\traid-disk\t%d\n","/dev/md1",1);
       	strcat(buff,d_name);
	
	
        if (NAS_File_Lock(RAID_CONF_FILE, 'w')==0)
                return ERROR_LOCK_FILE;
	if((fp=fopen(RAID_CONF_FILE,"a+"))!=NULL) {
		fputs(buff,fp);
		fclose(fp);
	}
	else
	{
		NAS_File_Unlock(RAID_CONF_FILE);
		return ERROR_FAIL;
	}
	NAS_File_Unlock(RAID_CONF_FILE);
	return SUCCESS;
}

int Delete_RaidDev(char *rd_name)
{
	char line[BUF_SIZE],value[BUF_SIZE],device_name[HD_DEVICE_NAME_LENGTH];
	char buf[BUF_SIZE];
    	FILE *fp=NULL,*tmpfp = NULL;
    	int found_flag = 0;
    	int dirty = 0,ret =0 ;
 
	
	if (NAS_File_Lock(RAID_CONF_FILE, 'r')==0)
		return ERROR_LOCK_FILE;
    	fp = fopen(RAID_CONF_FILE,"r");
    	if (fp==NULL)
	{
		NAS_File_Unlock(RAID_CONF_FILE);
		return ERROR_OPEN_FILE;
	}

    	
	tmpfp = tmpfile();		// temp file for buffer
	rewind(tmpfp);

    	while(1) {
      		if (fgets(line,sizeof(line),fp)==NULL) break;
		Get_String_Field(line,1,SPACE,value,BUF_SIZE);
		if(strcasecmp(value,"raiddev") == 0 ) {
			Get_String_Field(line,2,SPACE,device_name,HD_DEVICE_NAME_LENGTH);
			if(strcasecmp(device_name,rd_name) == 0 ) {
				found_flag = 1;
				dirty = 1;
			}
			else
				found_flag = 0;	
		}
		if (!found_flag) fputs(line,tmpfp);
		
      	}
      	fclose(fp);
	NAS_File_Unlock(RAID_CONF_FILE);
      	
      	if(dirty) {
      		int i;

		if (NAS_File_Lock(RAID_CONF_FILE, 'w')==0) {
			fclose(tmpfp);
			return ERROR_LOCK_FILE;
		}
      		if((fp = fopen(RAID_CONF_FILE,"w"))!=NULL){
			rewind(tmpfp);
			while ((i = fread(buf, 1, sizeof(buf), tmpfp)) > 0) {
				fwrite(buf, 1, i, fp);
			}
			fclose(fp);
			ret = SUCCESS;
		}
		else
		{
			ret = ERROR_OPEN_FILE;
		}
		NAS_File_Unlock(RAID_CONF_FILE);
      		
      	}
      	else
      		ret = RAID_DEVICE_NO;
     
      	fclose(tmpfp);
      	return ret;
      	
}

int Get_One_RaidDev_Status(char *rd_name,RAID_STATUS *sts_rd)
{
	RAID_STATUS rd_list[MAX_DRIVE_NO];
	int ret,i;
	int found_flag=0;
	ret = Get_All_RaidDev_Status(rd_list,MAX_DRIVE_NO);
	
	if(ret>=0){
		int myraidno, raidno, hdrlen = strlen("/dev/md");
		char *p;

		p = rd_name + hdrlen;
		if (p)
			myraidno = atoi(p);

		if (myraidno<0 || myraidno>MAX_RAID_NO)
			return ERROR_NOT_FOUND;

		for(i=0;i<ret;i++) {
			p = rd_list[i].name + hdrlen;
			raidno = atoi(p);

			if (raidno == myraidno) {
				*sts_rd=rd_list[i];
				found_flag=1;
				break;	
			}
		}	
	}
	else
		return ERROR_FAIL; 
	if(found_flag==0)
		return ERROR_NOT_FOUND;
	else
		return SUCCESS;
}

int Get_All_RaidDev_Status(RAID_STATUS *sts_rd_ptr,int rd_list_cnt)
{
	int tab_rd_cnt=0,sts_rd_cnt=0;
	int act_hd_cnt=0,cfg_hd_cnt=0;
	int i,j,k,hd_end_flag=0,swap_md_flag=0;
	int ret;
	char line[512],md_line[512],value[BUF_SIZE];
	char *str1;
	int md_start=0,table_end_flag=0,is_md_line=0;
	char *str_s, *str_e;  // Catherine 2003/08/25
	int func_hd_cnt=0, need_hd_cnt=0; // Catherine 2003/08/25
	
	tab_rd_cnt = Get_RaidDev_Count();
	
	if(tab_rd_cnt > 0) {
		RAID_DEVICE tab_rd[tab_rd_cnt+2];
		FILE *fp;
		ret = Get_All_RaidDev_Conf(tab_rd,tab_rd_cnt+2);
		
		fp = fopen(RAID_STAT_FILE,"r");
		if (fp==NULL) return ERROR_OPEN_FILE;
		
		//ignore first two line
		fgets(line,sizeof(line),fp);
		fgets(line,sizeof(line),fp);	
		
    		while(fgets(line,sizeof(line),fp)!=NULL) {
      			
      			if (strstr(line,"unused")!=NULL) table_end_flag=1;
      			
      			if(Get_String_Field(line,1,':',value,BUF_SIZE)>=0) {

      				if(strstr(value,"md")!=NULL)
      					is_md_line=1;
      				else
      					is_md_line=0;
      			}
      			else
      				is_md_line=0;
      				
      			if( is_md_line && md_start == 0){ 
      				strcpy(md_line,"");
      				line[strlen(line)-1]='\0';
				strcat(md_line," ");
				strcat(md_line,line);
				md_start = 1;
      				continue;
      			}
      			
      			if(is_md_line==0 && md_start == 1 && !table_end_flag){ 
      				line[strlen(line)-1]='\0';
				strcat(md_line," ");
				strcat(md_line,line);
      				continue;
      			}
	
      			if((is_md_line==1 || table_end_flag==1)&& md_start == 1){ 
      				int raid_no;
      				act_hd_cnt=0;
      				cfg_hd_cnt=0;
				need_hd_cnt=0;
				func_hd_cnt=0;
				//get raid name
				Get_String_Field(md_line,1,':',value,BUF_SIZE);

				raid_no = atoi(strstr(value, "md") + 2);
				if (raid_no >= Get_Swap_RAID_No_Base())
					swap_md_flag = 1;

				sprintf(sts_rd_ptr[sts_rd_cnt].name,"/dev/md%02d", raid_no);
				for(i=1;i>=0;i++) {
					if(Get_String_Field(md_line,i,SPACE,value,BUF_SIZE)>=0) {
						if(strcasecmp(value,"blocks") == 0) {
							hd_end_flag = i-2;
							break;
					
						}
					}
				}	

				//get active hd
				memset((void*)sts_rd_ptr[sts_rd_cnt].active_drive_no,
					0, MAX_DRIVE_NO*sizeof(int));

				for(i=5;i<=hd_end_flag;i++) {
					char tmp_dev_name[20],*tmp_ptr;
					char tmp_order[10], *tmp_ptr_2;
					int drive_order;

					Get_String_Field(md_line,i,SPACE,value,BUF_SIZE);
					if(strstr(value,"(F)") != NULL ) continue;
					tmp_ptr = strchr(value,'[');
					tmp_ptr_2 = strchr(value, ']');
					memset(tmp_order, 0, 10*sizeof(char));
					strncpy(tmp_order, tmp_ptr+1, (int)(tmp_ptr_2-tmp_ptr-1));
					*tmp_ptr = '\0';
					sprintf(tmp_dev_name,"/dev/%s",value);
					//sts_rd_ptr[sts_rd_cnt].active_drive_no[act_hd_cnt++] = Get_Drive_No(tmp_dev_name);	
					drive_order = atoi(tmp_order);
					sts_rd_ptr[sts_rd_cnt].active_drive_no[drive_order] = Get_Drive_No(tmp_dev_name);
					act_hd_cnt++;
				}		
				sts_rd_ptr[sts_rd_cnt].active_drive_cnt = act_hd_cnt;

				// remove the "0"s
				k = 0;
				for(i=0; i<MAX_DRIVE_NO; i++) {
					if (i>act_hd_cnt) break;

					if (sts_rd_ptr[sts_rd_cnt].active_drive_no[i]==0) {
						if (k==0) k = i;

						for(j=k+1; j<MAX_DRIVE_NO; j++) {
							if(sts_rd_ptr[sts_rd_cnt].active_drive_no[j]!=0) {
								k = j;
								break;
							}
						}
						if (j<MAX_DRIVE_NO) {
							sts_rd_ptr[sts_rd_cnt].active_drive_no[i] =
								sts_rd_ptr[sts_rd_cnt].active_drive_no[j];
							sts_rd_ptr[sts_rd_cnt].active_drive_no[j] = 0;
						}
					}
				}
// Catherine 2003/08/25 ==>
				for (i = 3; Get_String_Field(md_line, hd_end_flag+i, SPACE, value, BUF_SIZE)>=0; i++)
				{
    					if (value[0] == '[') {
    						str_s = strchr(value, '/');
    						*str_s = 0x0;
    						need_hd_cnt = atoi(value+1);

	    					str_e = strchr(++str_s, ']');
	    					*str_e = 0x0;
	    					func_hd_cnt = atoi(str_s);
	    					break;
	    				}
	    			}
// <== Catherine 2003/08/25
				//get sync or recovery percentage
				if ( (str1=strstr(md_line,"resync")) != NULL) {
       					str_e = strstr(str1,"%");
       					if(str_e != NULL ) {
       						*str_e='\0';
       						str_s = strstr(str1,"="); 
       						str_s++;
       						sts_rd_ptr[sts_rd_cnt].resync_per=atoi(str_s)+RAID_RESYNC_PER_BASE;
       					}
// Catherine 2003/08/25 ==>
       					else if (need_hd_cnt > func_hd_cnt)
	    				{ // rebuild
	    					sts_rd_ptr[sts_rd_cnt].resync_per = 0;
	    				}
// <== Catherine 2003/08/25
	    				else
       						sts_rd_ptr[sts_rd_cnt].resync_per=RAID_RESYNC_PER_BASE;
    				}
    				else
				if ( (str1=strstr(md_line,"recovery")) != NULL) {
         				str_e = strstr(str1,"%");
         				*str_e='\0';
         				str_s = strstr(str1,"="); str_s++;
					if(atof(str_s)==0.0)
						sts_rd_ptr[sts_rd_cnt].resync_per = 101;
		    			else
         					sts_rd_ptr[sts_rd_cnt].resync_per = atoi(str_s);
    				}
// Catherine 2003/08/25 ==>
    				else if	(act_hd_cnt >= need_hd_cnt && need_hd_cnt > func_hd_cnt)
    				{ // delay rebuild, not degrade
					sts_rd_ptr[sts_rd_cnt].resync_per = 0;
    				}
// <== Catherine 2003/08/25
    				else sts_rd_ptr[sts_rd_cnt].resync_per = -1;		
				// put config hd in structure
				for(i=0;i<tab_rd_cnt;i++) {
					if(strcasecmp(tab_rd[i].name,sts_rd_ptr[sts_rd_cnt].name) == 0) {
						for(j=0;j<tab_rd[i].data_drive_cnt;j++) 
							sts_rd_ptr[sts_rd_cnt].config_drive_no[cfg_hd_cnt++]=tab_rd[i].data_drive_no[j];
					
						for(j=0;j<tab_rd[i].spare_drive_cnt;j++)
							sts_rd_ptr[sts_rd_cnt].config_drive_no[cfg_hd_cnt++]=tab_rd[i].spare_drive_no[j];
					}
				}
				sts_rd_ptr[sts_rd_cnt].config_drive_cnt = cfg_hd_cnt;				
				++sts_rd_cnt;
				if(swap_md_flag == 1) { 
					sts_rd_cnt--;
					swap_md_flag = 0;
				}
				if(table_end_flag) 
					break;
				else
					strcpy(md_line,line);
			}
		}
		ret = sts_rd_cnt;
		fclose(fp);
	}
	else
		ret = 0;

	return sts_rd_cnt;
}

BOOL Is_Active_Raid_HD(int drive_no, char *rd_name,int buf_size)
{
	int max_active_cnt,i,j,ret;
// 2002/08/15 Catherine ==>
	int drvcnt=Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	int swapcnt = drvcnt/2;
//<== Catherine

	max_active_cnt = Get_RaidDev_Count();
	if(max_active_cnt<=0) 
		return FALSE;
	else {
		RAID_STATUS sts_rd[max_active_cnt+swapcnt];
		ret = Get_All_RaidDev_Status(sts_rd,max_active_cnt+swapcnt);
		if(ret>0) {
			for(i=0;i<ret;i++) {
				for(j=0;j<sts_rd[i].active_drive_cnt;j++) {
					if(drive_no == sts_rd[i].active_drive_no[j]) {
						if(strlen(sts_rd[i].name) < buf_size )
							strcpy(rd_name,sts_rd[i].name);
						return TRUE;
					}
				}
			}
		}
		return FALSE;
	}
}


int Get_Raid_Level(int drive_no)//-1:linear
{
	int rd_cnt,ret, raidlevel = ERROR_NOT_RAID_DEVICE;
	RAID_DEVICE *rd;
	int i,j;
	int drvcnt=Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	int swapcnt = drvcnt/2;

	if (drvcnt <= 1) return raidlevel;

	//allocat memory
	rd_cnt = Get_RaidDev_Count();
	if (rd_cnt <= 0) return raidlevel;

	rd = calloc(rd_cnt+swapcnt,sizeof(RAID_DEVICE));
	if (rd == NULL) return ERROR_OUT_OF_MEMORY;

	ret = Get_All_RaidDev_Conf(rd,rd_cnt+swapcnt);
	for(i=0;i<rd_cnt;i++) {
		// check raid data disk
		for(j=0;j<rd[i].data_drive_cnt;j++) {
			if(drive_no == rd[i].data_drive_no[j]) {			
				if(strcasecmp(rd[i].raid_level,"linear")==0)
					raidlevel = -1;
				else
					raidlevel = atoi(rd[i].raid_level);
			}
		}
		// check raid spare disk
		for(j=0;j<rd[i].spare_drive_cnt;j++) {
			if(drive_no == rd[i].spare_drive_no[j]) {
				if(strcasecmp(rd[i].raid_level,"linear")==0)
					raidlevel = -1;
				else
					raidlevel = atoi(rd[i].raid_level);
			}
		}	
	}
	free(rd);
	return raidlevel ;
	
}

BOOL Is_RaidDev_Exist(char *raid_name)
{
	int rd_cnt,ret;
	RAID_DEVICE *rd;
	int i;
	int drvcnt=Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	int swapcnt = drvcnt/2;

	if (drvcnt <= 1) return FALSE;

	//allocat memory
	rd_cnt = Get_RaidDev_Count();
	if(rd_cnt == 0) return FALSE;
	
	rd = calloc(rd_cnt+swapcnt,sizeof(RAID_DEVICE));
	if (rd == NULL) return FALSE;

	ret = Get_All_RaidDev_Conf(rd,rd_cnt+swapcnt);
	for(i=0;i<rd_cnt;i++) {
		if(strcasecmp(rd[i].name,raid_name) == 0) {
			free(rd);
			return TRUE;
		}
	}
	free(rd);
	return FALSE;
}

BOOL Is_RaidDev_Active(char *raid_name)
{
	int drvcnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);
	int rd_cnt;
	RAID_STATUS rd[drvcnt];
	int i;
	int myraidno, raidno, hdrlen = strlen("/dev/md");
	char *p;

	// Catherine 2003/04/09
	if (raid_name == NULL || (strlen(raid_name) <= hdrlen) ||
		strncmp(raid_name, "/dev/md", hdrlen))
	{
#ifdef DEBUG
		printf("Is_RaidDev_Active(%s) : either NULL poiter or not a RAID name!\n",
			(raid_name == NULL) ? "NULL" : raid_name);
#endif
		return FALSE;
	}

	//allocat memory
	rd_cnt = Get_All_RaidDev_Status(rd, drvcnt);
	if(rd_cnt <= 0) return FALSE;

	p = raid_name + hdrlen;
	if (p)
		myraidno = atoi(p);

	if (myraidno<0 || myraidno>MAX_RAID_NO)
		return FALSE;

	for(i=0;i<rd_cnt;i++) {
		p = rd[i].name + hdrlen;
		raidno = atoi(p);

		if (raidno == myraidno)
			return TRUE;
	}
	return FALSE;
}

int Make_Raid(char *raid_name)
{
	char cmd_line[BUF_SIZE];
	
	sprintf(cmd_line,"%s --really-force %s > /dev/null",CMD_MKRAID,raid_name);
	system(cmd_line);
	sleep(3);
		return SUCCESS;
}

int Start_Raid(char *raid_name)
{
	char cmd_line[BUF_SIZE], devname[BUF_SIZE];
	RAID_DEVICE rdlist[MAX_RAID_NO];
	int rdcnt, i, index = -1;

	if(!Is_RaidDev_Exist(raid_name)) return ERROR_NO_RAIDDEV;
	
	// try raidstart first
	sprintf(cmd_line,"%s %s > /dev/null",CMD_RAIDSTART,raid_name);
	system(cmd_line);
	sleep(1);
	if(Is_RaidDev_Active(raid_name)) 
		return SUCCESS;
	
	// raidstart fails --> try mdadm to run the md device
	rdcnt = Get_All_RaidDev_Conf(rdlist, MAX_RAID_NO);
	for (i=0; i<rdcnt; i++) {
		if (!strcmp(raid_name, rdlist[i].name)) {
			index = i;
			break;
		}
	}
	if (index < 0) return ERROR_FAIL;	// should not happen?
	
	sprintf(cmd_line, "/sbin/mdadm --assemble -R %s", raid_name);
	
	for (i=0; i<rdlist[index].data_drive_cnt; i++) {
  		Get_Partition_Name(rdlist[index].data_drive_no[i], DATA_PART, devname);
  		if (Is_HD_Exist(devname))
  			sprintf(cmd_line, "%s %s", cmd_line, devname);
  	}
	for (i=0; i<rdlist[index].spare_drive_cnt; i++) {
  		Get_Partition_Name(rdlist[index].spare_drive_no[i], DATA_PART, devname);
  		if (Is_HD_Exist(devname))
  			sprintf(cmd_line, "%s %s", cmd_line, devname);
  	}
	system(cmd_line);
	sleep(1);
	if(Is_RaidDev_Active(raid_name)) 
		return SUCCESS;

	return ERROR_FAIL;
}	

int Raid_Hot_Add_Memeber_Disk(char *raid_name,char *device_name)
{
	if(Is_RaidDev_Active(raid_name)) {
		char cmd_line[BUF_SIZE],md_name[128];
		sprintf(cmd_line,"%s %s %s > /dev/null",CMD_RAIDHOTADD,raid_name,device_name);
		system(cmd_line);
	
		if(!Is_Active_Raid_HD(Get_Drive_No(device_name),md_name,128)) 
			return ERROR_FAIL;
		else	
			return SUCCESS;
	}
	return ERROR_FAIL;
}
	
int Stop_Raid(char *raid_name)
{
	char cmd_line[BUF_SIZE];
	int ret;
	
	if(!Is_RaidDev_Exist(raid_name)) return ERROR_NO_RAIDDEV;
	if(Is_Mounted(raid_name)) {
		if((ret=Unmount_Partition(raid_name)) ==ERROR_FAIL)
			return ERROR_FAIL;
	}
	
	sprintf(cmd_line,"%s %s > /dev/null",CMD_RAIDSTOP,raid_name);
	system(cmd_line);
	
	if(Is_RaidDev_Active(raid_name)) 
		return ERROR_FAIL;
	else	
		return SUCCESS;
	
	
}


int Create_NAS_RaidDev(char *rd_name,int *drive_no_list,int list_cnt,int raid_level)
{
	
	RAID_DEVICE rd;
	int i,ret;
	
	if(raid_level != 4) { // != raid0+1
		strcpy(rd.name,rd_name);
		rd.data_drive_cnt = list_cnt;
	
		for(i=0;i<list_cnt;i++)
			rd.data_drive_no[i] = drive_no_list[i];
	
		if(raid_level == -1 )
			strcpy(rd.raid_level,"linear");
		else
			sprintf(rd.raid_level,"%d",raid_level);
		
		rd.spare_drive_cnt = 0;
			
		ret = Create_RaidDev(&rd);	
	}
	else {
		int  rd0_drv_no_list1[2],rd0_drv_no_list2[2];
		
		rd0_drv_no_list1[0]= drive_no_list[0];
		rd0_drv_no_list1[1]= drive_no_list[1];
		rd0_drv_no_list2[0]= drive_no_list[2];
		rd0_drv_no_list2[1]= drive_no_list[3];
		
		//crate raid0 device first  
		sprintf(rd.raid_level,"%d",0);
		
		strcpy(rd.name,"/dev/md0");
		rd.data_drive_cnt = 2;
	
		for(i=0;i<2;i++)
			rd.data_drive_no[i] = rd0_drv_no_list1[i];
		
		rd.spare_drive_cnt = 0;
			
		ret = Create_RaidDev(&rd);
		
		strcpy(rd.name,"/dev/md1");
		rd.data_drive_cnt = 2;
	
		for(i=0;i<2;i++)
			rd.data_drive_no[i] = rd0_drv_no_list2[i];
		
		rd.spare_drive_cnt = 0;
			
		ret = Create_RaidDev(&rd);		
		
		//create raid1 device
		strcpy(rd.name,"/dev/md2");
		ret = Create_Raid01Dev();
			
	}
	return ret;	
}

/**********************************************************************
** Create_NAS_RaidDev_Ex
** Description:	Add a RAID volumes w/ spare disk support if level=1 or 5
** Output:	the index of the new volume record or the error number
**		of adding the record
***********************************************************************/
int Create_NAS_RaidDev_Ex
(char *rd_name,int *drive_no_list,int list_cnt,int *spare_drive_list,int spare_list_cnt,int raid_level)
{
	RAID_DEVICE rd;
	int i,ret;
#ifdef DEBUG
	printf("Create_NAS_RaidDev_Ex start...\n");
#endif
	if(raid_level != 4) { // != raid0+1
		strcpy(rd.name,rd_name);
		rd.data_drive_cnt = list_cnt;

		for(i=0;i<list_cnt;i++)
			rd.data_drive_no[i] = drive_no_list[i];

		if(raid_level == -1 )
			strcpy(rd.raid_level,"linear");
		else
			sprintf(rd.raid_level,"%d",raid_level);

		rd.spare_drive_cnt = spare_list_cnt;
		for(i=0;i<spare_list_cnt;i++)
			rd.spare_drive_no[i] = spare_drive_list[i];
#ifdef DEBUG
		printf("before Create_RaidDev....\n");
#endif
		ret = Create_RaidDev(&rd);
	}
	else {
		int  rd0_drv_no_list1[2],rd0_drv_no_list2[2];

		rd0_drv_no_list1[0]= drive_no_list[0];
		rd0_drv_no_list1[1]= drive_no_list[1];
		rd0_drv_no_list2[0]= drive_no_list[2];
		rd0_drv_no_list2[1]= drive_no_list[3];

		//crate raid0 device first  
		sprintf(rd.raid_level,"%d",0);

		strcpy(rd.name,"/dev/md0");
		rd.data_drive_cnt = 2;

		for(i=0;i<2;i++)
			rd.data_drive_no[i] = rd0_drv_no_list1[i];

		rd.spare_drive_cnt = 0;

		ret = Create_RaidDev(&rd);

		strcpy(rd.name,"/dev/md1");
		rd.data_drive_cnt = 2;

		for(i=0;i<2;i++)
			rd.data_drive_no[i] = rd0_drv_no_list2[i];

		rd.spare_drive_cnt = 0;

		ret = Create_RaidDev(&rd);		
		
		//create raid1 device
		strcpy(rd.name,"/dev/md2");
		ret = Create_Raid01Dev();
			
	}
#ifdef DEBUG
	printf("Create_NAS_RaidDev_Ex finish...\n");
#endif
	return ret;	
}

// RAID is during the process of "resync" or "recovery"
BOOL Is_Raid_Busy(char* raid_name)
{
	int tab_rd_cnt;
	char *mdname;
	char line[512],value[BUF_SIZE];
	char *str1;
	int md_start=0,table_end_flag=0,is_my_md_line=0, is_md_line=0;

	mdname = raid_name;
	while((str1 = strchr(mdname, '/'))!=NULL)
		mdname = str1+1;
#ifdef DEBUG
	printf("raid name = %s, md name = %s\n",
		raid_name, mdname);
#endif
	tab_rd_cnt = Get_RaidDev_Count();
	
	if(tab_rd_cnt > 0) {
		FILE *fp;
		
		fp = fopen(RAID_STAT_FILE,"r");
		if (fp==NULL) {
#ifdef DEBUG
			printf("Fail to open %s\n", RAID_STAT_FILE);
#endif
			return FALSE;
		}
		
		//ignore first two line
		fgets(line,sizeof(line),fp);
		fgets(line,sizeof(line),fp);	
		
    		while(fgets(line,sizeof(line),fp)!=NULL) {
#ifdef DEBUG
			printf("line : %s\n", line);
#endif
      			if (strstr(line,"unused")!=NULL) // over
				table_end_flag=1;
      			
      			if(Get_String_Field(line,1,':',value,BUF_SIZE)>=0) {
				if(strstr(value, "md")!=NULL) {
					is_md_line = 1;
      					if(strstr(value,mdname)!=NULL) {
#ifdef DEBUG
						printf("Found!\n");
#endif
      						is_my_md_line = 1;
					}
					else
						is_my_md_line = 0;
				}
      				else {
					is_md_line=0;
				}
      			}
      			else {
				is_md_line=0;
			}
      				
      			if( is_md_line && is_my_md_line && md_start == 0){ 
				//finally the first line of my RAID
				md_start = 1;
      				continue;
      			}

			if (is_md_line && (!is_my_md_line) && md_start) {
#ifdef DEBUG
				printf("Opportunity is gone....\n");
#endif
				break;
			}
	
      			if((!is_md_line) && (is_my_md_line==1 || table_end_flag==1)&& md_start == 1){ 
				if ( ((str1=strstr(line,"resync")) != NULL) ||
					((str1=strstr(line,"recovery")) != NULL)) {
					fclose(fp);
					return TRUE;
    				}
			}
		}
		fclose(fp);
	}

	return FALSE;
}

//============================================================
// Set_Raid_Device_Faulty
// output : SUCCESS or ERROR_FAIL
//============================================================
int Set_Raid_Device_Faulty(char* md_name, char* bad_devname)
{
	int md_fd;
	struct stat s;
	int ret;

	md_fd = open(md_name, O_RDWR);
	if ( md_fd < 0 ) {
#ifdef DEBUG
		printf("Set_Raid_Device_Faulty : fails to open %s\n",
			md_name);
#endif
		return ERROR_FAIL;
	} else {
		stat (bad_devname, &s);	
		ret = ioctl (md_fd,SET_DISK_FAULTY, (unsigned long)s.st_rdev);
		sleep(1);
		if (ret!=0) // failed
			ret = ERROR_FAIL;
	}

	close(md_fd);
	return ret;
}
