//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		nasdisk.c
//
//	Abstract: 
//		NAS disk operation function
//
//	HISTORY:
//		2002/08/15	Catherine Shen -- remove the code that assumes
//					drive number never exceeds 4
//		2002/03/28	Catherine Shen -- move "is_an_inactive_raid_hd"
//					into "Get_NAS_Disks_Info()"
//					to avoid CGI time-out when RAID recovering
//		2002/03/22	Catherine Shen -- create a specific size of parition 3
//					for test
//		2002/03/21	Catherine Shen -- speed up "Get_NAS_Disks_Info()"
//					a little bit
//		2002/03/06	Catherine Shen -- for "raidstart" fails
//					when one of the mirror devices unplugged.
//		2001/6/4	Louis Tsai created	
//		2001/6/5	self test ok!
//
//**************************************************************************
#include <sys/stat.h>  // for open()
#include <assert.h>
#include <fcntl.h>     // for open()
#include <sys/types.h>	//for kill
#include <signal.h>		//for kill
#include <time.h>
#include <dirent.h>
#include <sys/wait.h>
#include <sys/vfs.h>
#include <sys/mount.h>
#include "file_lock.h"
#include "cfg_system.h"
#include "storage.h"
#include "storage_err.h"
#include "nas_quota.h"
#include "cfg_nic.h"
#include "cfg_rsync.h"
#include "eventlog.h"
#include "qnap_pic_user.h"

#define MAPPING		"/etc/config/usb_share_mapping"

char *RW_BK_MP[]={"0",".hda.conf",".hdb.conf",".hdc.conf",".hdd.conf",".hde.conf",".hdf.conf",".hdg.conf",".hdh.conf"};

static char dev_map[]={'z', 'a', 'b', 'd', 'k', 'r'};

/////////////////////////////////////////////////////////////////////////////
// Set_HD_RW_Error
// Description :	mark a disk as having r/w error
// Output :		TRUE -- success / FALSE -- failure
// Catherine Aug. 2004
/////////////////////////////////////////////////////////////////////////////
extern void start_vol_quota(int vol, int check);

BOOL Set_HD_RW_Error(int drvno)
{
	char hiddenfile[BUF_SIZE];
	int ret=0;
	Get_Hidden_Conf_File_Name(drvno, hiddenfile);
	ret = WritePrivateProfileString("", "hw_error", "TRUE", hiddenfile);
	Create_RW_Err_In_Ram(drvno);
	return ret;
}

/////////////////////////////////////////////////////////////////////////////
// Get_HD_RW_Error
// Description :	has this disk been marked "r/w error"?
// Output :		TRUE -- has error / FALSE -- not marked
// Catherine Aug. 2004
/////////////////////////////////////////////////////////////////////////////
BOOL Get_HD_RW_Error(int drvno)
{
	char hiddenfile[BUF_SIZE], result[BUF_SIZE];

	Get_Hidden_Conf_File_Name(drvno, hiddenfile);
	GetPrivateProfileString("", "hw_error", "FALSE", result, sizeof(result), hiddenfile);
	if (!strcasecmp(result, "TRUE")) return TRUE;
	return FALSE;
}

int Create_NAS_Partitions(int drive_no)
{
//Ricky
//	PARTITION_DATA size[3];
	PARTITION_DATA size[4];
//End
	char device_name[HD_DEVICE_NAME_LENGTH];
	int ret = 1;
//retry = 20;
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	//prepare partition data
	size[0].part_no = 1;
	size[0].is_boot = 1;
	size[0].sys_id = 0x83;//ROOT PART
//Ricky
//	size[0].capacity_val = 200;
	size[0].capacity_val = 256;
//End
	size[0].capacity_format = SIZE_M;
	
	size[1].part_no = 2;
	size[1].is_boot = 0;
	size[1].sys_id = 0x82;//SWAP PART
//Ricky
//	size[1].capacity_val = 64;
	size[1].capacity_val = 512;
//End
	size[1].capacity_format = SIZE_M;
	
//Ricky
	size[2].part_no = 3;//DATA PART
	size[2].is_boot = 0;
	size[2].sys_id = 0x83;
	size[2].capacity_val = sg_readcap_M(device_name)-256-512-100;
	size[2].capacity_format = SIZE_M;

//	size[3].part_no = 4; //DATA PART
//End
	size[3].part_no = 4; 
	size[3].is_boot = 0;
	size[3].sys_id = 0x83;
	size[3].capacity_val = 0;
	size[3].capacity_format = SIZE_M;

	// Catherine 2002/03/06 ===>
//	if (Get_Raid_Level(drive_no)>-2) // RAID device
//		size[2].sys_id = 0xfd; // Linux raid autodetect
//	else
//		size[2].sys_id = 0x83; // Linux
	// <=== Catherine 2002/03/06


	// Catherine 2002/03/22 modified ==>
	if(Get_Profile_Boolean(NASCONF_STORAGE_SECTION,NASCONF_DEBUG,FALSE)) {
		int disk_size;
		disk_size = Get_Profile_Integer(NASCONF_STORAGE_SECTION, "NAS DEBUG DISK SIZE", 100);
		size[3].capacity_val = disk_size;
	}
	// <== Catherine 2002/03/22 modified
	
// I don't why it need retry, Ricky remarked 2005/11/10
//	while(retry--) {
		ret = Delete_NAS_Partitions(drive_no);
//		if(ret == SUCCESS ) break;
//		sleep(drive_no);
		
//	}
//	retry = 20;
//	while(retry--) {
		ret = Create_Partitions(device_name,size,4);
//		if(ret == SUCCESS ) break;
//		sleep(drive_no);
//	}
//End
	
	return ret;
}

int Create_NAS_Partitions_Ex(char* dev_name)
{
	PARTITION_DATA size[4];
	int ret = 1,retry = 20;
	
	//prepare partition data
	size[0].part_no = 1;
	size[0].is_boot = 1;
	size[0].sys_id = 0x83;//ROOT PART
	size[0].capacity_val = 256;
	size[0].capacity_format = SIZE_M;
	
	size[1].part_no = 2;
	size[1].is_boot = 0;
	size[1].sys_id = 0x82;//SWAP PART
	size[1].capacity_val = 512;
	size[1].capacity_format = SIZE_M;
	
	size[2].part_no = 3;//DATA PART
	size[2].is_boot = 0;
	size[2].sys_id = 0x83;
	size[2].capacity_val = sg_readcap_M(dev_name)-256-512-100;
	size[2].capacity_format = SIZE_M;

	size[3].part_no = 4; 
	size[3].is_boot = 0;
	size[3].sys_id = 0x83;
	size[3].capacity_val = 0;
	size[3].capacity_format = SIZE_M;


	if(Get_Profile_Boolean(NASCONF_STORAGE_SECTION,NASCONF_DEBUG,FALSE)) {
		int disk_size;
		disk_size = Get_Profile_Integer(NASCONF_STORAGE_SECTION, "NAS DEBUG DISK SIZE", 100);
		size[3].capacity_val = disk_size;
	}
	
	while(retry--) {
		ret = Delete_NAS_Partitions_Ex(dev_name);
		if(ret == SUCCESS ) break;
		sleep(1);
		
	}
	retry = 20;
	while(retry--) {
		ret = Create_Partitions(dev_name,size,4);
		if(ret == SUCCESS ) break;
		sleep(1);
	}
	
	return ret;
}

int Delete_NAS_Partitions(int drive_no)
{
	char device_name[HD_DEVICE_NAME_LENGTH];
	char cmd_line_str[MAX_CMDLINE_LENGTH];
	int ret;
	
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	Stop_Swapping(drive_no);
	sprintf(cmd_line_str,"%s -d all %s > /dev/null",CMD_FDISK,device_name);
	ret = system(cmd_line_str);
	
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
	
	return ret;
}

int Delete_NAS_Partitions_Ex(char* dev_name)
{
	char cmd_line_str[MAX_CMDLINE_LENGTH];
	int ret;
	
	Stop_Swapping_Ex(dev_name);
	sprintf(cmd_line_str,"%s -d all %s > /dev/null",CMD_FDISK,dev_name);
	ret = system(cmd_line_str);
	
	if (ret != 0) return ERROR_FAIL;
	return SUCCESS;
	
	return ret;
}

//just format singal root or data partition on each drive
int Format_NAS_Partition(int drive_no,int which_part,int fs_type)
{
	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH];
	int ret;
	
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	Get_Partition_Name(drive_no,which_part,part_name);
	
	ret = Format_Partition(part_name,FALSE,fs_type);

	return ret;	
}

int Format_NAS_Partition_Ex(char* dev_name,int which_part,int fs_type)
{
	char part_name[HD_DEVICE_NAME_LENGTH];
	
	Get_Partition_Name_Ex(dev_name,which_part,part_name);
	return Format_Partition(part_name,FALSE,fs_type);
}

int Mount_NAS_Partition(int drive_no,int which_part,int fs_type)
{
	char mp_name[MP_LENGTH];
	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH];
	int ret;
	
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	Get_Partition_Name(drive_no,which_part,part_name);
	Get_MP_String(drive_no,which_part,mp_name);
	
	ret = Mount_Partition(part_name,mp_name,fs_type);
	
	return ret;
}

int Mount_NAS_Partition_Ex(char* dev_name,int which_part,int fs_type)
{
	char mp_name[MP_LENGTH];
//	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH];
	
	Get_Partition_Name_Ex(dev_name,which_part,part_name);
	Get_MP_String(dev_name[7]-'a'+1,which_part,mp_name);
	
	return Mount_Partition(part_name,mp_name,fs_type);
}

int Unmount_NAS_Partition(int drive_no,int which_part)
{
	
	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH];
	int ret;
//	char tmp_str[256];
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	Get_Partition_Name(drive_no,which_part,part_name);
	
	ret = Unmount_Partition(part_name);
	return ret;	
}

int Unmount_NAS_Partition_Ex(char* dev_name,int which_part)
{
	char part_name[HD_DEVICE_NAME_LENGTH];
	
	Get_Partition_Name_Ex(dev_name,which_part,part_name);
	return Unmount_Partition(part_name);
}

long long Get_NAS_Partition_Free_Size(int drive_no, int which_part){
	char mp_name[MP_LENGTH];
	char device_name[HD_DEVICE_NAME_LENGTH];
	int ret;
	
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	if (ret <=0) return ERROR_Get_HDev_Name;
	
	Get_MP_String(drive_no,which_part,mp_name);

	return Get_Free_Disk_Size(mp_name);
}

long long Get_NAS_Data_Free_Size(int drive_no){
	return Get_NAS_Partition_Free_Size(drive_no, DATA_PART);
}

int create_swap_raidev(RAID_DEVICE *rd_ptr)
{
	int  i;
    	char buff[1024],d_name[100], tmp_str[HD_DEVICE_NAME_LENGTH];
	FILE *fp;
    	
    	/*-- fill the device name --*/
    	sprintf(buff,"raiddev %s\n",rd_ptr->name);

    	/*-- fill raid level --*/
    	sprintf(d_name,"\traid-level\t%s\n",rd_ptr->raid_level);
      	strcat(buff,d_name);

    	/*-- fill data disk No. ---*/
    	sprintf(d_name,"\tnr-raid-disks\t%d\n",rd_ptr->data_drive_cnt);
      	strcat(buff,d_name);

    	/*-- fill spare disk No. ---*/
    	sprintf(d_name,"\tnr-spare-disks\t%d\n",rd_ptr->spare_drive_cnt);
      	strcat(buff,d_name);

   	/*-- fill chunk size ---*/
    	sprintf(d_name,"\tchunk-size\t4\n");
     	strcat(buff,d_name);

   	/*-- fill super-block ---*/
    	sprintf(d_name,"\tpersistent-superblock\t1\n");
      	strcat(buff,d_name);

   	/*-- fill data HD Name ---*/
   	for (i=0; i<rd_ptr->data_drive_cnt; i++) {
       		Get_Partition_Name(rd_ptr->data_drive_no[i],SWAP_PART,tmp_str);
       		sprintf(d_name,"\tdevice\t%s\n\traid-disk\t%d\n",tmp_str,i);
       		strcat(buff,d_name);
   	}

   	/*-- fill spare HD Name ---*/
   	for (i=0; i<rd_ptr->spare_drive_cnt; i++) {
       		Get_Partition_Name(rd_ptr->spare_drive_no[i],SWAP_PART,tmp_str);
       		sprintf(d_name,"\tdevice\t%s\n\tspare-disk\t%d\n",tmp_str,i);
       		strcat(buff,d_name);
   	}
	
	if((fp=fopen(RAID_CONF_FILE,"a+"))!=NULL) {
		fputs(buff,fp);
		fclose(fp);
	}
	else
		return ERROR_FAIL;
	
	return SUCCESS;
}

int Init_Swapping(int drive_no)
{
  	char part_name[PARTITION_NAME_LENGTH], cmd_line[BUF_SIZE];
 	int  ret;

  	ret=Get_Partition_Name(drive_no,SWAP_PART,part_name);
  	if (ret<0) return ERROR_FAIL;
  	
  	//check partition swap on, if yes, stop it
  	if(SYS_Get_One_String(SWAPON_FILE,1,SPACE,part_name,NULL,0)>=0) 
  		Stop_Swapping(drive_no);
	
  	// format swap partition
  	sprintf(cmd_line, "%s %s >/dev/null",CMD_MKSWAP ,part_name);
  	ret = system(cmd_line);
	if(ret!=0) return ERROR_FAIL;
	return SUCCESS;	
}

//==============================================================//
//	Making a swap partition raid and avoid crashing O.S while
//	plug out a using swap partition!
//	2001/10/24	Louis Tsai
//==============================================================//

int Start_Swapping(int drive_no)
{
 	char cmd_line[256], swapdev[HD_DEVICE_NAME_LENGTH];
 	int ok_raid = 0;
 	int partner_no = (drive_no % 2 == 0) ? (drive_no-1) : (drive_no + 1);

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	if (Verify_Hidden_Conf_File(drive_no) != SUCCESS )
		return ERROR_FAIL;

	if (Verify_Hidden_Conf_File(partner_no) == SUCCESS )
 		ok_raid = 1;

	Stop_Swapping(drive_no);
	Stop_Swapping(partner_no);

/* 	if ( ok_raid == 1 ) {
 		RAID_DEVICE rd;

		sprintf(swapdev, "/dev/md%02d", Get_Swap_RAID_No_Base() + (drive_no-1)/2);

 		//set swap raid in /etc/raidtab
 		strcpy(rd.name, swapdev);
		rd.data_drive_cnt = 2;
	
		if (drive_no > partner_no) {
			rd.data_drive_no[0] = partner_no;
			rd.data_drive_no[1] = drive_no;
		}
		else {
			rd.data_drive_no[1] = partner_no;
			rd.data_drive_no[0] = drive_no;
		}

		sprintf(rd.raid_level,"%d",1);
		rd.spare_drive_cnt = 0;
		
		create_swap_raidev(&rd);
		
 		sprintf(cmd_line, "%s --really-force %s > /dev/null", CMD_MKRAID, rd.name);
 		system(cmd_line);
 		sleep(1);
 	}
 	else Get_Partition_Name(drive_no, SWAP_PART, swapdev);
*/ 
	Get_Partition_Name(drive_no, SWAP_PART, swapdev);
	sprintf(cmd_line,"%s %s > /dev/null", CMD_MKSWAP, swapdev);
	system(cmd_line);
	sleep(1);
	sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPON, swapdev);
	system(cmd_line);
  	
  	return SUCCESS;
}

int Start_Swapping_Ex(int drive_no, char* dev_name)
{
 	char cmd_line[256], swapdev[HD_DEVICE_NAME_LENGTH];
 	int ok_raid = 0;
 	int partner_no = (drive_no % 2 == 0) ? (drive_no-1) : (drive_no + 1);

	if (drive_no > MAX_DRIVE_NO)
		return ERROR_INVALID_PARAM;

	if (Verify_Hidden_Conf_File(drive_no) != SUCCESS )
		return ERROR_FAIL;

	if (Verify_Hidden_Conf_File(partner_no) == SUCCESS )
 		ok_raid = 1;

	Stop_Swapping_Ex(dev_name);
	Stop_Swapping(partner_no);

	Get_Partition_Name_Ex(dev_name, SWAP_PART, swapdev);
	sprintf(cmd_line,"%s %s > /dev/null", CMD_MKSWAP, swapdev);
	system(cmd_line);
	sleep(1);
	sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPON, swapdev);
	system(cmd_line);
  	
  	return SUCCESS;
}


int Stop_Swapping(int drive_no)
{
	char swapdev[HD_DEVICE_NAME_LENGTH];
 	char cmd_line[256];
// 	int partner_no = (drive_no % 2 == 0) ? (drive_no-1) : (drive_no + 1);
//	char partname[HD_DEVICE_NAME_LENGTH];
//	BOOL is_partner_exist = FALSE;

/*	sprintf(swapdev, "/dev/md%02d", Get_Swap_RAID_No_Base() + (drive_no-1)/2);

	if (SYS_Get_One_String(SWAPON_FILE,1, SPACE, swapdev, NULL, 0)>=0) {
		// stop swap
		sprintf(cmd_line, "%s %s > /dev/null", CMD_SWAPOFF, swapdev);
 		system(cmd_line);

		// if a disk is gone, set swap raid faulty
		if (!Is_HD_ReadWrite_Safe(drive_no)) {
			Get_Partition_Name(drive_no, SWAP_PART, partname);
			Set_Raid_Device_Faulty(swapdev, partname);
		}

		is_partner_exist = Is_HD_ReadWrite_Safe(partner_no);
		Get_Partition_Name(partner_no, SWAP_PART, partname);

		if (!is_partner_exist) {
			Set_Raid_Device_Faulty(swapdev, partname);
		}

		// stop swap raid
 		sprintf(cmd_line, "%s %s > /dev/null", CMD_RAIDSTOP, swapdev);
 		system(cmd_line);

		// remove swap raid record
		Delete_RaidDev(swapdev);

		// start the other swap device (Catherine 2003/06/10)
		if (is_partner_exist) {
		 	sprintf(cmd_line,"%s %s > /dev/null", CMD_MKSWAP, partname);
		 	system(cmd_line);
		 	sleep(1);
		 	sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPON, partname);
		 	system(cmd_line);
		 }
	}
	else {
		Get_Partition_Name(drive_no, SWAP_PART, swapdev);
		if (SYS_Get_One_String(SWAPON_FILE,1, SPACE, swapdev, NULL, 0)>=0) {
			sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPOFF, swapdev);
 			system(cmd_line);
 		}
 	}*/
	Get_Partition_Name(drive_no, SWAP_PART, swapdev);
	if (SYS_Get_One_String(SWAPON_FILE,1, SPACE, swapdev, NULL, 0)>=0) {
		sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPOFF, swapdev);
		system(cmd_line);
	}

	return SUCCESS;
}

int Stop_Swapping_Ex(char* dev_name)
{
	char swapdev[HD_DEVICE_NAME_LENGTH];
 	char cmd_line[256];

	Get_Partition_Name_Ex(dev_name, SWAP_PART, swapdev);
	if (SYS_Get_One_String(SWAPON_FILE,1, SPACE, swapdev, NULL, 0)>=0) {
		sprintf(cmd_line,"%s %s > /dev/null", CMD_SWAPOFF, swapdev);
		system(cmd_line);
	}

	return SUCCESS;
}

int Clear_MBR(int drive_no)
{
	int  fd, ret;
  	char hd_name[21], buff[SECTOR_SIZE];

  	if ( (ret=Get_HD_DevName(drive_no, hd_name)) != SUCCESS )  return ret;
  		fd = open(hd_name, O_RDWR);
  		if (fd==-1) return ERROR_FAIL;

 	//---- Clear buff to zero
  	memset(buff, 0, sizeof(buff));
  	ret = write(fd, buff, sizeof(buff));
  	if ( ret == -1)
	{
		close(fd);
		return ERROR_FAIL;
	}

  	close (fd);
  	sync();
  	return SUCCESS;
}

int Clear_MBR_Ex(char* dev_name)
{
	int  fd, ret;
  	char buff[SECTOR_SIZE];

  	fd = open(dev_name, O_RDWR);
  	if (fd==-1) return ERROR_FAIL;

 	//---- Clear buff to zero
  	memset(buff, 0, sizeof(buff));
  	ret = write(fd, buff, sizeof(buff));
  	if ( ret == -1)
	{
		close(fd);
		return ERROR_FAIL;
	}

  	close (fd);
  	sync();
  	return SUCCESS;
}

static void Recover_Config(){
	system("/bin/rm /etc/config -rf");
        mkdir("/etc/config", 0777);
        system("/bin/cp /etc/default_config/* /etc/config");
}

static int set_fisrt_init_flag(char *dev_name){
        int fd;
        char item[64];

        sprintf(item, "%s4", dev_name);
        if((fd = open(item, O_WRONLY)) < 0){
                return ERROR_FAIL;
//          printf("open failed \n");
        }
        else{
                lseek(fd, (70 * 1024 * 1024) + 16, SEEK_SET);
                write(fd, "2", 5);
                close(fd);
        }
        return 0;
}

//===================================================/
//Init NAS disk include following step:
// 1. Umount all partition on disk
// 2. Stop swaping
// 3. create_partition(root,swap,data)
// 4. format each partition
// 5. Init swap(Init_Swapping(),Start_Swapping())
// 6. mount root part and create .conf on it
//===================================================/

int Init_NAS_Disk(int drive_no)
{
	struct stat status;
	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH];
	int fs_no;
	int ret;
	
	//Check HD is exist
	ret = Get_HD_DevName(drive_no,device_name);
	
	//Umount all partition on disk drive

        Get_Partition_Name(drive_no,DATA_PART,part_name);//DATA
        if(Is_Mounted(part_name))  {
                ret = Unmount_Partition(part_name);
		if(ret<0){
//			Set_Disk_Status(drive_no, S_READY);
			Start_Services();
			return ERROR_FORMAT_FAIL;
		}
        }

	Get_Partition_Name(drive_no,ROOT_PART,part_name);//ROOT 
	
	if(Is_Mounted(part_name)) {
		if(Is_HD_Valid(drive_no)){
			goto valid;
		}
		else{
	                if(!lstat("/etc/config", &status)){
        	                if(S_ISLNK(status.st_mode)){
					// /etc/config is a link
					Recover_Config();
				}
                	}
			ret = Unmount_Partition(part_name);
			if(ret<0){
				if(Mount_NAS_Partition(drive_no,DATA_PART,fs_no)==SUCCESS)
					Set_Disk_Status(drive_no, S_READY);
				return ERROR_FORMAT_FAIL;
			}
			goto invalid;
		}
	}
	else{
		if(!lstat("/etc/config", &status)){
			if(S_ISLNK(status.st_mode)){
				// /etc/config is a link
				Recover_Config();
			}
		}
		goto invalid;
	}

invalid:
	//stop swapping
	Stop_Swapping(drive_no);	
	//Clear_MBR
	Clear_MBR(drive_no);
	//create NAS spec partition
	Create_NAS_Partitions(drive_no);
	//format each partition
	fs_no = Get_FileSystem_Type();
	
	Format_NAS_Partition(drive_no,ROOT_PART,fs_no);
	
	//Mount root part & create .conf file 
	Mount_NAS_Partition(drive_no,ROOT_PART,fs_no);
	if((ret = Create_Hidden_Conf_File(drive_no))<0) return ret;
	
	//initialize swap
	if((ret = Start_Swapping(drive_no))<0) return ret;
	set_fisrt_init_flag(device_name);

valid:
	return SUCCESS;
}

int Init_NAS_Disk_Ex(char* dev_name)
{
//	struct stat status;
//	char device_name[HD_DEVICE_NAME_LENGTH];
	char part_name[HD_DEVICE_NAME_LENGTH], share_name[HD_DEVICE_NAME_LENGTH];
	int fs_no;
	int ret;
	
	//Check HD is exist
	//ret = Get_HD_DevName(drive_no,device_name);
	
	Stop_Services();
	//Umount all partition on disk drive

        Get_Partition_Name_Ex(dev_name,DATA_PART,part_name);//DATA
        if(Is_Mounted(part_name))  {
                ret = Unmount_Partition(part_name);
		if(ret<0){
                        Set_Disk_Status(Get_DriveNo_By_DevName(dev_name), S_READY);
                        return ERROR_FORMAT_FAIL;
                }
        }
	Get_Partition_Name_Ex(dev_name,ROOT_PART,part_name);//ROOT 
	
	Start_Services();

	if(Is_Mounted(part_name)) {
		if(Is_HD_Valid(dev_name[7]-'a'+1)){
			goto valid;
		}
		else{
	                /*if(!lstat("/etc/config", &status)){
        	                if(S_ISLNK(status.st_mode)){
					// /etc/config is a link
					Recover_Config();
				}
                	}*/
			ret = Unmount_Partition(part_name);
			if(ret<0){
				int drv=Get_DriveNo_By_DevName(dev_name);
				if(Mount_NAS_Partition(drv,DATA_PART,fs_no)==SUCCESS)
                                	Set_Disk_Status(drv, S_READY);
                                return ERROR_FORMAT_FAIL;
                        }
			goto invalid;
		}
	}
	else{
		/*if(!lstat("/etc/config", &status)){
			if(S_ISLNK(status.st_mode)){
				// /etc/config is a link
				Recover_Config();
			}
		}*/
		goto invalid;
	}

invalid:
//	remove_mapping(dev_name);
        Get_Share_Name(dev_name, share_name, sizeof(share_name));
        Remove_NAS_Share(share_name);
	//stop swapping
//	Stop_Swapping_Ex(dev_name);	
	//Clear_MBR
	Clear_MBR_Ex(dev_name);
	//create NAS spec partition
	Create_NAS_Partitions_Ex(dev_name);
	//format each partition
	fs_no = Get_FileSystem_Type();
	
	Format_NAS_Partition_Ex(dev_name,ROOT_PART,fs_no);
	
	//Mount root part & create .conf file 
	Mount_NAS_Partition_Ex(dev_name,ROOT_PART,fs_no);
	if((ret = Create_Hidden_Conf_File_Ex(dev_name[7]-'a'+1, dev_name))<0) return ret;
	
	//initialize swap
//	if((ret = Start_Swapping_Ex(drive_no, dev_name))<0) return ret;

valid:
	return SUCCESS;
}

int Get_Disk_Status(int drive_no)
{
        char disk_section[20],tmp_str[10];
	int ret=0;
        sprintf(disk_section,"VOLUME %d",drive_no);

        ret = Conf_Get_Field(STORAGE_CONF,disk_section,STATUS_FIELD,tmp_str,10);
        if(ret<0)
		return ret;
        else
                return (atoi(tmp_str));
}


int Set_Disk_Status(int drive_no,int status)
{
        char vol_section[20],tmp_str[10];
        int ret=0;
//	int retstatus;

        sprintf(vol_section,"VOLUME %d",drive_no);
        sprintf(tmp_str,"%d",status);
// Catherine 2002/03/26 ==>
//      return (Conf_Set_Field(STORAGE_CONF,vol_section,"status",tmp_str));

        ret = Conf_Set_Field(STORAGE_CONF,vol_section,STATUS_FIELD,tmp_str);
/*        retstatus = Get_Disk_Status(drive_no);
        while (retstatus!=status && i++<100) {
                sleep(1);
                ret = Conf_Set_Field(STORAGE_CONF,vol_section,"status",tmp_str);
                retstatus = Get_Disk_Status(drive_no);
        }
*/        return ret;
// <== Catherine 2002/03/26
}

int Set_Disk_Status_Ex(char* dev_name,int status)
{
        char vol_section[20];
        int drive_no;

	if((drive_no = Get_DriveNo_By_DevName(dev_name)) < 0)
		return -1;

        sprintf(vol_section,"VOLUME %d",drive_no);
	return Set_Private_Profile_Integer(vol_section, STATUS_FIELD, status, STORAGE_CONF);
}

int Get_QRAID1_Disk_Status()
{
	return Get_Profile_Integer("QRAID1", STATUS_FIELD, S_NOT_EXIST);
}

int Set_QRAID1_Disk_Status(int status)
{
	return Set_Profile_Integer("QRAID1", STATUS_FIELD, status);
}

int Create_Disk_Storage_Conf(){
	int ret=0;
	char str[256];
// device_name[256];;
	char buf[10];
	FILE *fp;
//        int drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,NASCONF_DRIVE_NO_FIELD,0);

#ifdef  STORAGE_FILE_LOCK
        if (NAS_File_Lock(STORAGE_CONF, 'w')==0)
                return ERROR_OPEN_FILE;
#endif
        //remove original storage.conf & create new
        unlink(STORAGE_CONF);
        if((fp = fopen(STORAGE_CONF,"w+")) != NULL)
                fclose(fp);
        else {
#ifdef  STORAGE_FILE_LOCK
                NAS_File_Unlock(STORAGE_CONF);
#endif
                return ERROR_FAIL;
        }
	sprintf(buf, "%d", 1);
	WritePrivateProfileString(STORAGE_GLOBAL_SECTION, AVAIL_FIELD, buf, STORAGE_CONF);

//	while((ret = Get_Partition_Name(i,DATA_PART,device_name))>=0) {
//                if(i>=drive_no) break;
	sprintf(str, "VOLUME 1");
	WritePrivateProfileString(str, STATUS_FIELD, "-1", STORAGE_CONF);
	WritePrivateProfileString(str, DEVICE_FIELD, "/dev/sda", STORAGE_CONF);
	WritePrivateProfileString(str, NO_FIELD, "1", STORAGE_CONF);
	WritePrivateProfileString(str, RAID_FIELD, "", STORAGE_CONF);
//		i++;
//	}
	return ret;
}

int Have_Qnap_Flag(char *dev_name){
	int fd;
	char item[16];
	char buf[8];
	
	sprintf(item, "%s4", dev_name);
	if((fd = open(item, O_RDONLY)) < 0){
		return 0;
	}
	else{
    		lseek(fd, 70 * 1024 * 1024, SEEK_SET);
    		read(fd, buf, 8);
    		close(fd);
	}
	if(strncmp(buf, "QNAP", 4))
		return 0;
	else
		return 1;
}

static int set_qnap_flag(char *dev_name){
	int fd;
	char item[64];
	
	sprintf(item, "%s4", dev_name);
	if((fd = open(item, O_WRONLY)) < 0){
		return ERROR_FAIL;
//	    printf("open failed \n");
	}
	else{
    		lseek(fd, 70 * 1024 * 1024, SEEK_SET);
    		write(fd, "QNAP", 5);
    		close(fd);
	}
	return 0;
}

int Check_Default_Share(int drive_no){
	int i=0, ret=0;
#ifndef def_share_cnt
	#define DEF_SHARE_CNT	5
#endif
	char *def_share[DEF_SHARE_CNT] = {"Public", "Qusb", "Qweb", "Qmultimedia", "Qdownload"};
	char path[256], mp_name[256];

	Get_MP_String(drive_no,DATA_PART,mp_name);

	for(i=0;i<DEF_SHARE_CNT;i++){
		sprintf(path, "%s/%s", mp_name, def_share[i]);		
		if(Is_Dir_Exist(path))
			ret=1;
//		else{
//			char msg[512];
//			sprintf(msg, "The system default share folder(%s) is not exist. It's suggested reformatting your SATA disk.", def_share[i]);
//			Write_Log(msg, EVENTLOG_WARNING_TYPE);
//		}
	}
	return ret;
}

int Init_NAS_Single_Disk(int drive_no){

	int ret;
	char item[64], buf[32];
	char cmd[128];	
	int status = Get_Disk_Status(drive_no);

	// 2005.12.21, Johnson Cheng
	// Set format flag to 1 to enable Recover button
	Set_Profile_Integer("QRAID1", "SATA Format", 1);//Shone recovered 2006,01,10

	Set_Disk_Status(drive_no,S_INITIALIZING);
        sprintf(item,"Drive %d",drive_no);
	GetProfileString("Storage", item, "sda", buf, sizeof(buf));
	sprintf(item, "/dev/%s3", buf);
	HD_Reset_Format_Progress(item);

//	Set_Disk_Status(drive_no,S_UNINITIALIZE);
	Stop_User_Quota(drive_no);
	Stop_Services();
        if((ret=Init_NAS_Disk(drive_no))<0){
		Set_Disk_Status(drive_no,status);
		return ret;
	}

	Set_Disk_Status(drive_no,S_FORMATTING);

	sprintf(cmd, "/bin/cp /etc/default_config/smb.conf /etc/config/");
	system(cmd);
	sprintf(cmd, "/sbin/daemon_mgr hotswap stop /sbin/hotswap; /sbin/daemon_mgr hotswap start \"/sbin/hotswap -d\"");
        system(cmd);
        if((ret = Format_NAS_Partition(drive_no,DATA_PART,Get_FileSystem_Type()))<0){
                Set_Disk_Status(drive_no,S_NOT_MOUNTED); 
                return ret;
        }

        if((ret=Mount_NAS_Partition(drive_no,DATA_PART,Get_FileSystem_Type()))<0){
                Set_Disk_Status(drive_no,S_NOT_MOUNTED); // Catherine 2002/03/28
                return ret;
        }

	sprintf(item, "/dev/%s", buf);
	set_qnap_flag(item);
	WritePrivateProfileString("VOLUME 1", DEVICE_FIELD, item, STORAGE_CONF);
	sprintf(item, "%d", 1);
	WritePrivateProfileString("VOLUME 1", NO_FIELD, item, STORAGE_CONF);
	WritePrivateProfileString("VOLUME 1", RAID_FIELD, "", STORAGE_CONF);

	// Add default shares for internal SATA disk 2005,12,15 Shone
        Create_Default_Share("Qmultimedia",SHARE_ADMINISTRATOR);
        Create_Default_Share("Qdownload",SHARE_ADMINISTRATOR);
        Create_Default_Share("Qweb",SHARE_ADMINISTRATOR);
        Create_Default_Share("Qusb", SHARE_EVERYONE);
        Create_Default_Share("Public",SHARE_EVERYONE_GUEST);

	Start_Services();

        if (drive_no == 1)
		start_vol_quota(drive_no, 1);

        Set_Disk_Status(drive_no,S_READY);

        return SUCCESS;

}

int Is_QRAID1_running()
{
	char dev_name[32];
	if(GetProfileString("QRAID1", "device name", "", dev_name, sizeof(dev_name)) <= 0)
		return FALSE;
	else
		return TRUE;
}


int Init_NAS_QRAID1_Disk(int drive_no)
{
	int ret;
	char buf[32], dev_name[32], serial_num[64];
//	char vendor[32], model[32], revision[32];

	// 2005.11.11, Johnson Cheng marked
	// Web CGI will check this condition
	/*if(Is_QRAID1_running())
	{
		//Set_Disk_Status(drive_no,S_UNKNOW_ERROR);
		return S_UNKNOW_ERROR;
	}*/
	if( S_READY != Get_Private_Profile_Integer("VOLUME 1", "status", -1, STORAGE_CONF))
		return S_SATA_NOT_READY;

        sprintf(buf,"VOLUME %d",drive_no);
	if(GetPrivateProfileString(buf, DEVICE_FIELD, "", dev_name, sizeof(dev_name), STORAGE_CONF) <= 0)
	{
		//Set_QRAID1_Disk_Status(S_UNKNOW_ERROR);
		return S_UNKNOW_ERROR;
	}

	WriteProfileString("QRAID1", "device name", dev_name);

	if(!Have_Qnap_Flag(dev_name))
	{
		char share_name[128];
		//Init_NAS_Single_Disk(drive_no);
		Set_QRAID1_Disk_Status(S_INITIALIZING);
	
		GetPrivateProfileString(buf, SHARE_FIELD, "none", share_name, sizeof(share_name), STORAGE_CONF);

		sprintf(buf, "%s3", dev_name);
		HD_Reset_Format_Progress(buf);
        	if((ret=Init_NAS_Disk_Ex(dev_name))<0){
			Set_QRAID1_Disk_Status(S_UNINITIALIZE);
			WriteProfileString("QRAID1", "device name", "");
			return ret;
		}

		Set_QRAID1_Disk_Status(S_FORMATTING);

        	if((ret = Format_NAS_Partition_Ex(dev_name,DATA_PART,Get_FileSystem_Type()))<0){
                	Set_QRAID1_Disk_Status(S_NOT_MOUNTED); 
			WriteProfileString("QRAID1", "device name", "");
	                return ret;
       		}

        	if((ret=Mount_NAS_Partition_Ex(dev_name,DATA_PART,Get_FileSystem_Type()))<0){
                	Set_QRAID1_Disk_Status(S_NOT_MOUNTED);
			WriteProfileString("QRAID1", "device name", "");
                	return ret;
        	}

		Remove_NAS_Share(share_name);
		set_qnap_flag(dev_name);
	}
	else
	{
		char part_name[HD_DEVICE_NAME_LENGTH];
		Get_Partition_Name_Ex(dev_name,ROOT_PART,part_name);
		if (!Is_Mounted(part_name))
		{
			if((ret = Mount_NAS_Partition_Ex(dev_name,ROOT_PART,EXT3_FS)) < 0)
			{
                		Set_QRAID1_Disk_Status(S_NOT_MOUNTED); 
				return ret;
			}
		}

		Get_Partition_Name_Ex(dev_name,DATA_PART,part_name);
		if (!Is_Mounted(part_name))
		{
        		if((ret = Mount_NAS_Partition_Ex(dev_name,DATA_PART,EXT3_FS)) < 0)
			{
                		Set_QRAID1_Disk_Status(S_NOT_MOUNTED); 
				return ret;
			}
		}
	}
	Set_Profile_Integer("QRAID1", "SATA Format", 0);
	Set_Profile_Integer("QRAID1", "Progress", 0);
        Set_QRAID1_Disk_Status(S_QRAID1);

	{
		char fs_buf[2], disk_section[16];
		char mnt_dir[64], size_buf[64];
		sprintf(disk_section,"VOLUME %d",drive_no);
		sprintf(fs_buf, "%d", FS_EXT3);
		WritePrivateProfileString(disk_section, FS_FIELD, fs_buf, STORAGE_CONF);
		Get_QNAP_Mount_Name(dev_name,mnt_dir,DATA_PART);
		sprintf(size_buf, "%lld", Get_Free_Disk_Size(mnt_dir));
		WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, size_buf, STORAGE_CONF);
	}
	
	// Save model name and serial number of SATA HD
	//Get_Sg_Info("/dev/sda", vendor, model, revision);
	Get_Sg_Serial_Number("/dev/sda", serial_num);
	if(Get_Hidden_Conf_File_Name(dev_name[7]-'a'+1, buf) < 0)
	{
		Set_QRAID1_Disk_Status(S_UNKNOW_ERROR);
		return S_UNKNOW_ERROR;
	}
	//Conf_Set_Field(buf, "", "model", model);
	Conf_Set_Field(buf, "", "serial number", serial_num);

	Start_QRAID1(dev_name, 1);

	return 0;
}

int Recover_NAS_QRAID1_Disk(int drive_no)
{
	int ret;
	char buf[256], dev_name[32], serial_num[64];
	char cmd[512], root_part[64], data_part[64];
//	char vendor[32], model[32], revision[32];

	if( S_READY != Get_Private_Profile_Integer("VOLUME 1", "status", -1, STORAGE_CONF))
		return S_SATA_NOT_READY;

        sprintf(buf,"VOLUME %d",drive_no);
	if(GetPrivateProfileString(buf, DEVICE_FIELD, "", dev_name, sizeof(dev_name), STORAGE_CONF) <= 0)
	{
		//Set_QRAID1_Disk_Status(S_UNKNOW_ERROR);
		return S_UNKNOW_ERROR;
	}

	Set_Profile_Integer("QRAID1", "SATA Format", 0);
	WriteProfileString("QRAID1", "device name", dev_name);
	Set_Profile_Integer("QRAID1", "Progress", 0);
        Set_QRAID1_Disk_Status(S_QRAID1_RECOVER);
	Get_MP_String(dev_name[7]-'a'+1, ROOT_PART, root_part);
	sprintf(buf, "%s%s", root_part, "/.config/uLinux.conf");
	Set_Private_Profile_Integer("QRAID1", "SATA Format", 0, buf);
	Set_Private_Profile_String("QRAID1", "device name", dev_name, buf);
	Set_Private_Profile_Integer("QRAID1", "Progress", 0, buf);
	Set_Private_Profile_Integer("QRAID1", STATUS_FIELD, S_QRAID1_RECOVER, buf);

	{
		char fs_buf[2], disk_section[16];
		char mnt_dir[64], size_buf[64];
		sprintf(disk_section,"VOLUME %d",drive_no);
		sprintf(fs_buf, "%d", FS_EXT3);
		WritePrivateProfileString(disk_section, FS_FIELD, fs_buf, STORAGE_CONF);
		Get_QNAP_Mount_Name(dev_name,mnt_dir,DATA_PART);
		sprintf(size_buf, "%lld", Get_Free_Disk_Size(mnt_dir));
		WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, size_buf, STORAGE_CONF);
	}
	
	// Save model name and serial number of SATA HD
	//Get_Sg_Info("/dev/sda", vendor, model, revision);
	Get_Sg_Serial_Number("/dev/sda", serial_num);
	if(Get_Hidden_Conf_File_Name(dev_name[7]-'a'+1, buf) < 0)
	{
		Set_QRAID1_Disk_Status(S_UNKNOW_ERROR);
		return S_UNKNOW_ERROR;
	}

	// do recovering
	//Get_MP_String(dev_name[7]-'a'+1, ROOT_PART, root_part);
	Get_MP_String(dev_name[7]-'a'+1, DATA_PART, data_part);
	sprintf(cmd, "/usr/bin/rsync -a --qnap-mode=1 --delete --ignore-errors --timeout=30 --exclude-from=/etc/config/Qraid1_conf_exclude.conf %s/.config/ /mnt/HDA_ROOT/.config > /dev/null 2>&1", root_part);
	system(cmd);
	sprintf(cmd, "/usr/bin/rsync -a --qnap-mode=1 --delete --ignore-errors --timeout=30 %s/.logs/ /mnt/HDA_ROOT/.logs > /dev/null 2>&1", root_part);
	system(cmd);
	sprintf(cmd, "/usr/bin/rsync -a --qnap-mode=1 --progress --delete --ignore-errors --timeout=30 --exclude-from=/etc/config/Qraid1_share_exclude.conf %s/ /share/HDA_DATA > /dev/null 2>&1", data_part);
	Write_Log("Q-RAID 1 recovery started.", EVENTLOG_INFORMATION_TYPE);
	ret = system(cmd) >> 8;
	if(ret == 12)	// RERR_STREAMIO
	{
		Write_Log("Q-RAID 1 recovery failed, maybe the capacity of internal disk is full.", EVENTLOG_ERROR_TYPE);
        	Set_QRAID1_Disk_Status(S_QRAID1_FAILED);
	}
	else
	{	// Assume Recover success
		Write_Log("Q-RAID 1 recovery finished successfully.", EVENTLOG_INFORMATION_TYPE);

		// Bug 2463. 2006.01.13, Johnson Cheng
		// Save serial number when recover finished
		//Conf_Set_Field(buf, "", "model", model);
		Conf_Set_Field(buf, "", "serial number", serial_num);

	    	Start_QRAID1(dev_name, 0);
	}

	return 0;
}


int do_qraid1_action(char* dev_name, int plugged_type)
{
	int ret;
	char buf[32], filename[32], serial_num[32];
	char part_name[HD_DEVICE_NAME_LENGTH];
//	char vendor[32], model[32], revision[32];

	if(plugged_type == PLUG_IN)
	{
		if(Is_QRAID1_running()) {
                	//Set_Disk_Status_Ex(dev_name,S_QRAID1_RUN); 
			return S_QRAID1_RUN;
		}

		WriteProfileString("QRAID1", "device name", dev_name);

		Get_Partition_Name_Ex(dev_name,ROOT_PART,part_name);
		if (!Is_Mounted(part_name))
		{
			if((ret = Mount_NAS_Partition_Ex(dev_name,ROOT_PART,EXT3_FS)) < 0) {
                		Set_QRAID1_Disk_Status(S_NOT_MOUNTED); 
				return ret;
			}
		}

		Get_Partition_Name_Ex(dev_name,DATA_PART,part_name);
		if (!Is_Mounted(part_name))
		{
        		if((ret = Mount_NAS_Partition_Ex(dev_name,DATA_PART,EXT3_FS)) < 0) {
                		Set_QRAID1_Disk_Status(S_NOT_MOUNTED); 
				return ret;
			}
		}
		if(Get_Profile_Integer("QRAID1","SATA Format",0)) {
                	//Set_QRAID1_Disk_Status(S_QRAID1_UNMATCH); 
			return S_QRAID1_UNMATCH;
		}

		if(Get_Hidden_Conf_File_Name(dev_name[7]-'a'+1, filename) < 0) {
			Set_QRAID1_Disk_Status(S_UNKNOW_ERROR);
			return S_UNKNOW_ERROR;
		}
		//Get_Sg_Info("/dev/sda", vendor, model, revision);
		//Conf_Get_Field(filename, "", "model", buf, sizeof(buf));
		//if(strcmp(Skip_White_Space_Ex(model), buf))
		//	return -1;
		Get_Sg_Serial_Number("/dev/sda", serial_num);
		Conf_Get_Field(filename, "", "serial number", buf, sizeof(buf));
		if(strcmp(Skip_White_Space_Ex(serial_num), buf)) {
                	Set_QRAID1_Disk_Status(S_QRAID1_UNMATCH); 
			return S_QRAID1_UNMATCH;
		}

		if( S_READY != Get_Private_Profile_Integer("VOLUME 1", "status", -1, STORAGE_CONF))
		{
			Set_QRAID1_Disk_Status(S_SATA_NOT_READY);
			return S_SATA_NOT_READY;
		}

		Set_Profile_Integer("QRAID1", "Progress", 0);
        	Set_QRAID1_Disk_Status(S_QRAID1);
		Start_QRAID1(dev_name, 1);

		// Set QRAID1 status in uLinux
		//WriteProfileString("QRAID1", "device name", dev_name);
	}
	else
	{	// PLUG_OUT
		if(Is_QRAID1_running())
		{
			GetProfileString("QRAID1", "device name", "", buf, sizeof(buf));
			if(!strcasecmp(dev_name, buf))
				Stop_QRAID1();
		}

		Unmount_NAS_Partition_Ex(dev_name, ROOT_PART);
		Unmount_NAS_Partition_Ex(dev_name, DATA_PART);
	}
	return 0;
}

// Catherine 2002/03/21 for speeding up Get_NAS_Disks_Info
// is a sound RAID disk (root mounted) but not active in the RAID array
// eq.
//	if (Is_Raid_HD(drive_no, rd_name, rd_name_len)) {
//		Get_Partition_Name(i+1,ROOT_PART,part_name);
//		if(!Is_RaidDev_Active(rd_name)) continue;
//   		if(!Is_Active_Raid_HD(i+1,rd_name,32) && Is_Mounted(part_name))
//   			disk_info_list[i].status = S_RW_ERROR;
//   	}
//
BOOL is_an_inactive_raid_hd(int drive_no)
{
	RAID_DEVICE rd[MAX_RAID_NO];
	RAID_STATUS sts_rd[MAX_RAID_NO];
	int rdcnt, active_rdcnt;
	char part_name[HD_DEVICE_NAME_LENGTH];
	int i, j, found;
	
	rdcnt = Get_All_RaidDev_Conf(rd,MAX_RAID_NO);
	if (rdcnt<=0)
		return FALSE; // no RAID configed.

	found=0;
	for(i=0;i<rdcnt && found==0;i++) {
		// check raid data disk
		for(j=0;j<rd[i].data_drive_cnt;j++) {
			if(drive_no == rd[i].data_drive_no[j]) {
				found=1;
				break;
			}
		}

		// check raid spare disk
		for(j=0;j<rd[i].spare_drive_cnt && found==0;j++) {
			if(drive_no == rd[i].spare_drive_no[j]) {
				found=1;
				break;
			}
		}	
	}

	if (!found) return FALSE; // not a RAID disk

	active_rdcnt = Get_All_RaidDev_Status(sts_rd,MAX_RAID_NO);
	if (active_rdcnt<=0)
		return FALSE; // no active RAID found

	for(i=0;i<active_rdcnt;i++) {
		for(j=0;j<sts_rd[i].active_drive_cnt;j++) {
			if(drive_no == sts_rd[i].active_drive_no[j]) {
				return FALSE; // is an active disk in RAID
			}
		}
	}

	Get_Partition_Name(drive_no,ROOT_PART,part_name);
	if (Is_Mounted(part_name))
		return TRUE; // HD is OK but not active in RAID

	return FALSE;

}

int Get_NAS_Disks_Info(DISK_INFO *disk_info_list,int buf_list_cnt)
{
	int i,drive_no;
	char devname[HD_DEVICE_NAME_LENGTH], raidname[HD_DEVICE_NAME_LENGTH];
	RAID_DEVICE rd[MAX_RAID_NO];
	RAID_STATUS sts_rd[MAX_RAID_NO];
	int rdcnt, active_rdcnt;
	char part_name[HD_DEVICE_NAME_LENGTH];
	int j,	k, found, raidfound;
	FILE *fp;

	drive_no = Get_Profile_Integer(NASCONF_STORAGE_SECTION,
		NASCONF_DRIVE_NO_FIELD,0);

	rdcnt = Get_All_RaidDev_Conf(rd,MAX_RAID_NO); // Catherine 2002/03/28
	active_rdcnt = Get_All_RaidDev_Status(sts_rd,MAX_RAID_NO); // Catherine 2002/03/28

	for(i=0;i<drive_no;i++) {
		if(i>buf_list_cnt) break;
		if(Get_HD_DevName(i+1,devname)>0) {
			Get_Disk_Info(devname,&disk_info_list[i]);

			// check if the disk is a RAID device but not active
			if(disk_info_list[i].status == S_READY ||
			   disk_info_list[i].status == S_UNKNOW_ERROR ){
			   	// Catherine Aug. 2004 : is this disk marked as "r/w error" once?
				if (Get_HD_RW_Error(i+1))
					disk_info_list[i].status = S_RW_ERROR;

				// Catherine 2002/03/28 ==>
				if (rdcnt>0 && active_rdcnt>0) {
					found=0;
					for(k=0;k<rdcnt && found==0;k++) {
						// check raid data disk
						for(j=0;j<rd[k].data_drive_cnt;j++) {
							if(i+1 == rd[k].data_drive_no[j]) {
								strcpy(raidname, rd[k].name);
								found=1;
								break;
							}
						}
				
						// check raid spare disk
						for(j=0;j<rd[k].spare_drive_cnt && found==0;j++) {
							if(i+1 == rd[k].spare_drive_no[j]) {
								strcpy(raidname, rd[k].name);
								found=1; // disk is in a RAID array
								break;
							}
						}	
					}
					if (found==0) // not a RAID disk
						continue;

					found=0;
					raidfound = 0;
					for(k=0;k<active_rdcnt && found==0;k++) {
						if (strcmp(sts_rd[k].name, raidname))
							continue;

						raidfound++;
						for(j=0;j<sts_rd[k].active_drive_cnt;j++) {
							if(i+1 == sts_rd[k].active_drive_no[j]) {
								found=1; // is an active disk in RAID
								break;
							}
						}
						break;
					}
					if ((raidfound ^ found) == 0 || disk_info_list[i].status == S_RW_ERROR)
						continue;

					// RAID device is active, and so is the drive
					Get_Partition_Name(i+1,ROOT_PART,part_name);
					fp = fopen("/tmp/hotswap_status", "r");
					if (Is_Mounted(part_name) && fp==NULL ) { // HD is OK but not active in RAID //2004/09/14,albert add
						disk_info_list[i].status = S_RW_ERROR;

						// Catherine Aug. 2004 : keep record of error
						Set_HD_RW_Error(i+1);
					}
					if (fp!=NULL)
						fclose(fp);
				}
				// <== Catherine 2002/03/28
			}
		}
		else
			break;
	}
	return i;
}

// Catherine 2003/08/19 ==>
// do similar jobs as Get_Disk_Info w/o the hidden config file check
int get_disk_status(char* device_name, DISK_INFO* disk_info)
{
  	FILE *fp;
	char buf[BUF_SIZE],*tmp_ptr;
	int hd_no = Get_Drive_No(device_name);

  	if(!Is_HD_ReadWrite_Safe(hd_no)){
  		strcpy(disk_info->model_name,"--");
  		disk_info->capacity = 0;
  		disk_info->status   = S_NOT_EXIST;
  		return SUCCESS;
  	}

	tmp_ptr = strrchr(device_name,'/')+1;

	sprintf(buf,"/proc/ide/%s/model",tmp_ptr);                                           

	if ((fp = fopen(buf, "r"))!=NULL)
	{
		if (fgets(buf, sizeof(buf), fp) != NULL)
			strcpy(disk_info->model_name,buf);
		fclose(fp);
	}

	sprintf(buf,"/proc/ide/%s/capacity",tmp_ptr);                                           

	if ((fp = fopen(buf, "r"))!=NULL)
	{
		if (fgets(buf, sizeof(buf), fp) != NULL)
			disk_info->capacity = (double) 0.5*atol(buf);
		fclose(fp);
	}

  	disk_info->status = S_READY;
  	return SUCCESS;
}
// <== Catherine 2003/08/19

// Catherine 2003/06/10
// Catherine 2003/08/19 :
//	stop calling Get_Disk_Info because "Verify_Hidden_Conf_File" makes
//	all drives awake;
int Get_One_NAS_Disk_Info(int driveno, DISK_INFO *disk_info)
{
	int ret, drive_cnt, j, k, found, raidfound, rdcnt, active_rdcnt;
	char devname[HD_DEVICE_NAME_LENGTH], raidname[HD_DEVICE_NAME_LENGTH];
	RAID_DEVICE rd[MAX_RAID_NO];
	RAID_STATUS sts_rd[MAX_RAID_NO];
	char part_name[HD_DEVICE_NAME_LENGTH];

	drive_cnt = Get_Profile_Integer(NASCONF_STORAGE_SECTION,
		NASCONF_DRIVE_NO_FIELD, 0);
	if (drive_cnt<=0) {
#ifdef DEBUG
		printf("Get_One_NAS_Disk_Info(%d) : "
			"fails to get total drive no, err=%d\n",
			driveno, drive_cnt);
#endif
		return ERROR_FAIL;
	}

	if (driveno <= 0 || driveno > drive_cnt || disk_info == NULL) {
#ifdef DEBUG
		printf("Get_One_NAS_Disk_Info(%d) : invalid parameter\n", driveno);
#endif
		return ERROR_INVALID_PARAM;
	}

	rdcnt = Get_All_RaidDev_Conf(rd,MAX_RAID_NO);
	active_rdcnt = Get_All_RaidDev_Status(sts_rd,MAX_RAID_NO);

	if ((ret = Get_HD_DevName(driveno,devname))>0) {
		ret = SUCCESS;

		get_disk_status(devname, disk_info); // Catherine 2003/08/19
		// check if the disk is a RAID device but not active
		if(disk_info->status == S_READY ||
			disk_info->status == S_UNKNOW_ERROR )
		{
			// Catherine Aug. 2004 : is this disk marked as "r/w error" once?
			// Ricky Jan. 10 2005 : check the backup config in the ram disk first.
			if(Is_RW_Err_In_Ram(driveno)){
				if (Get_HD_RW_Error(driveno)){
					disk_info->status = S_RW_ERROR;
					printf("S_RW_ERROR.\n");
				}
				else
					Create_RW_Err_In_Ram(driveno);
			}
		
			if (rdcnt>0 && active_rdcnt>0) {
				found = 0;
				for(k=0; k<rdcnt && found==0; k++) {
					// check raid data disk
					for(j=0; j<rd[k].data_drive_cnt; j++) {
						if(driveno == rd[k].data_drive_no[j]) {
							strcpy(raidname, rd[k].name);
							found=1;
							break;
						}
					}

					// check raid spare disk
					for(j=0; j<rd[k].spare_drive_cnt && found==0; j++) {
						if(driveno == rd[k].spare_drive_no[j]) {
							strcpy(raidname, rd[k].name);
							found=1; // disk is in a RAID array
							break;
						}
					}	
				}

				if (found==0) // not a RAID disk
					return SUCCESS;

				found=0;
				raidfound = 0;
				for(k=0; k<active_rdcnt && found==0; k++) {
					if (strcmp(sts_rd[k].name, raidname))
						continue;

					raidfound++;
					for(j=0;j<sts_rd[k].active_drive_cnt;j++) {
						if(driveno == sts_rd[k].active_drive_no[j]) {
							found=1; // is an active disk in RAID
							break;
						}
					}
					break;
				}
				// Catherine Aug. 2004 : if r/w error, return as well
				if ((raidfound ^ found) == 0 || disk_info->status == S_RW_ERROR)
					return SUCCESS;

				// RAID device is active, and so is the drive
				Get_Partition_Name(driveno,ROOT_PART,part_name);
				if (Is_Mounted(part_name)) { // HD is OK but not active in RAID
					disk_info->status = S_RW_ERROR;

					// Catherine Aug. 2004 : keep record of error
					Set_HD_RW_Error(driveno);
				}
			}
		}
	}

	return ret;
}

int Create_RW_Err_In_Ram(int drvno){
	char src[1024],dest[1024],filename[1024],cmd[1024];
	strcpy(filename,RW_BK_MP[drvno]);
	sprintf(dest, "/mnt/config/%s",filename);
	Get_Hidden_Conf_File_Name(drvno, src);
	sprintf(cmd,"rm -rf %s",dest);
	system(cmd);
	sprintf(cmd,"yes|cp -rf %s %s",src, dest);
	system(cmd);
	return 0;
}

BOOL Is_RW_Err_In_Ram(int drvno){
	char path[1024],filename[1024], result[1024];
	strcpy(filename,RW_BK_MP[drvno]);
	sprintf(path, "/mnt/config/%s",filename);
	GetPrivateProfileString("", "hw_error", "FALSE", result, sizeof(result),path);
	if (!strcasecmp(result, "TRUE")) return TRUE;
	return FALSE;
}


/********************************Ricky add*****************************************************/

int Get_Share_Name(char *dev, char *share_name, int size){
	char disk_section[32], buf[32];
	int idx=1, avail_disk=0;
	
	GetPrivateProfileString(STORAGE_GLOBAL_SECTION, AVAIL_FIELD, "1", buf, sizeof(buf), STORAGE_CONF);

	avail_disk=atoi(buf);
	for(;idx<avail_disk+1;idx++){
		sprintf(disk_section, "VOLUME %d", idx);
		GetPrivateProfileString(disk_section, DEVICE_FIELD, "", buf, sizeof(buf), STORAGE_CONF);
		if(!strcmp(dev, buf)){
			GetPrivateProfileString(disk_section, SHARE_FIELD, "", share_name, size, STORAGE_CONF);
			return SUCCESS;
		}
	}
	strncpy(share_name, "none", size);
	return ERROR_FAIL;
}

int Stop_And_Umount_By_Dev(char *dev_name){
	char cmd_line[MAX_CMDLINE_LENGTH], mnt_dir[HD_DEVICE_NAME_LENGTH];
	int ret=0, ret1=0;
	char share_name[64];

	Get_QNAP_Mount_Name(dev_name,mnt_dir,ROOT_PART);
	sprintf(cmd_line, "/bin/umount %s1 1>/dev/null 2>/dev/null", dev_name);
//	printf("%s\n", cmd_line);
	ret = system(cmd_line);
	if(ret!=0){
		sprintf(cmd_line, "/bin/umount %s1 1>/dev/null 2>/dev/null", mnt_dir);
		ret = system(cmd_line);
	}
	sprintf(cmd_line, "/bin/umount %s 1>/dev/null 2>/dev/null", dev_name);
	ret1 = system(cmd_line);
	if(ret1!=0){
		sprintf(cmd_line, "/bin/umount %s 1>/dev/null 2>/dev/null", mnt_dir);
		ret1 = system(cmd_line);
	}
	remove_mapping(dev_name);
	// add by Kent 2005/11/25
	// due to when Eject invalid Q-Raid1, have to umount data partition
	Unmount_NAS_Partition_Ex(dev_name, DATA_PART);
	// end
	Get_Share_Name(dev_name, share_name, sizeof(share_name));
	Remove_NAS_Share(share_name);
//	printf("Remove Share: %s\n", share_name);
	if(ret1==0 || ret==0)
		return 0;
	else
		return ERROR_FAIL;
}
/*
void Stop_And_Umount_Host(int drive_no){
	char dev[HD_DEVICE_NAME_LENGTH];
	int idx=0;
        Stop_Services();
	for(;idx<4;idx++){
		sprintf(dev, "/dev/sd%c", dev_map[drive_no]+idx);
		Stop_And_Umount_By_Dev(dev);
	}
        Start_Services();
}
*/

void Get_Qnap_Disk_Name(int drive_no, char *dev_name){
	sprintf(dev_name, "/dev/sd%c", dev_map[drive_no]);
	return;
}

void Get_QNAP_Mount_Name(char *dev_name, char *mount_name, int part){

        if(strstr(dev_name, "sda")){
		if(part==ROOT_PART)
			sprintf(mount_name, "/mnt/HDA_ROOT");
		else
                	sprintf(mount_name, "/share/HDA_DATA");
	}
        else{
		if(Have_Qnap_Flag(dev_name))
			Get_MP_String(dev_name[7] - 'a' + 1, part, mount_name);
		else{
			char dev[4];
			strncpy(dev, dev_name+5,3);
			dev[3]='\0';
			sprintf(mount_name, "/share/external/%s", dev);
		}
 //       		sprintf(mount_name, "/mnt/%s", dev_name+5);
	}
}

int Mount_QNAP_Disk_By_Name(char *part_name, int part, int fs){
        int ret=0;
	DIR *dir;
        char cmd_line[MAX_CMDLINE_LENGTH], mount_name[PARTITION_NAME_LENGTH];

	Get_QNAP_Mount_Name(part_name, mount_name, part);
	
	if(!(dir=opendir(mount_name)))
                mkdir(mount_name,S_IRUSR | S_IWUSR | S_IXUSR |S_IRGRP | S_IWGRP | S_IXGRP |S_IROTH | S_IWOTH | S_IXOTH);

	if (dir!=NULL)
	        closedir(dir);

        switch(part){
                case ROOT_PART:
			if(fs==FS_EXT3)
                        	sprintf(cmd_line,"%s -t ext3 %s1 %s 2>/dev/null 1>/dev/null",CMD_MOUNT, part_name, mount_name);
			else if(fs==FS_FAT32)
				sprintf(cmd_line,"%s -t vfat -o utf8,umask=0000,shortname=mixed %s1 %s 2>/dev/null 1>/dev/null",CMD_MOUNT, part_name, mount_name);
			else
				sprintf(cmd_line,"%s %s1 %s 2>/dev/null 1>/dev/null",CMD_MOUNT, part_name, mount_name);
                        break;
                case DATA_PART:
                        sprintf(cmd_line,"%s -t ext3 %s2 %s 2>/dev/null 1>/dev/null",CMD_MOUNT, part_name, mount_name);
                        break;
                default:
                        fprintf(stderr, "Error given part no.");
                        return ERROR_FAIL;
        }
        ret = system(cmd_line);
        return ret;
}

int Mount_QNAP_Disk(int disk_no, int part, int fs){
	int ret=0;
	char part_name[PARTITION_NAME_LENGTH], disk_section[PARTITION_NAME_LENGTH];

	sprintf(disk_section, "VOLUME %d", disk_no);
	GetPrivateProfileString(disk_section, DEVICE_FIELD, "", part_name, sizeof(part_name), STORAGE_CONF);
	
	ret = Mount_QNAP_Disk_By_Name(part_name, part, fs);
	return ret;
}

int Get_Disk_Conf_path(int disk_no, char *path){
	char disk_section[PARTITION_NAME_LENGTH], dev_name[PARTITION_NAME_LENGTH];
	char mount_name[PARTITION_NAME_LENGTH];
	int ret=0;
	
	sprintf(disk_section, "VOLUME %d", disk_no);
	ret = GetPrivateProfileString(disk_section, DEVICE_FIELD, "", dev_name, sizeof(dev_name), STORAGE_CONF);

	Get_QNAP_Mount_Name(dev_name, mount_name, ROOT_PART);
	sprintf(path, "%s%s", mount_name, HD_HIDDEN_CONF);
	return ret;
}

int Mount_Disk(char *part_name, int fs){
	char cmd[MAX_CMDLINE_LENGTH];
	char mnt_dir[HD_DEVICE_NAME_LENGTH], dev[HD_DEVICE_NAME_LENGTH];
	DIR *dir;

	strncpy(dev, part_name+5, 3);
	dev[3]='\0';

//	sprintf(mnt_dir, "/mnt/%s", dev);
	sprintf(mnt_dir, "/share/external/%s", dev);
	if(fs==FS_UNKNOWN)
		sprintf(cmd, "%s %s %s", CMD_MOUNT, part_name, mnt_dir);

	if(!(dir=opendir(mnt_dir)))
                mkdir(mnt_dir,S_IRUSR | S_IWUSR | S_IXUSR |S_IRGRP | S_IWGRP | S_IXGRP |S_IROTH | S_IWOTH | S_IXOTH);
	if (dir!=NULL)
	        closedir(dir);
	
	return system(cmd);
}

int Verify_Disk_Conf_By_Name(char *dev_name){
	char conf_file[HD_DEVICE_NAME_LENGTH], m_name[HD_DEVICE_NAME_LENGTH];
	int ret = 0;
	char mount_name[HD_DEVICE_NAME_LENGTH];
	
	Get_QNAP_Mount_Name(dev_name, mount_name, ROOT_PART);
	
	sprintf(conf_file, "%s%s", mount_name, HD_HIDDEN_CONF);
	
	ret = Conf_Get_Field(conf_file, "", "hd_name", m_name, sizeof(m_name));

        if (ret <0) {
                return ERROR_FAIL;
        }
	return ret;
}

int Verify_Disk_Conf(int disk_no){
	char conf_file[HD_DEVICE_NAME_LENGTH], m_name[HD_DEVICE_NAME_LENGTH];
	int ret = 0;
	ret = Get_Disk_Conf_path(disk_no, conf_file);

	if (ret <0) {
                return ERROR_FAIL;
        }

	ret = Conf_Get_Field(conf_file, "", "hd_name", m_name, sizeof(m_name));
	
	if (ret <0) {
                return ERROR_FAIL;
        }
	return ret;
}

int get_fs_mounted(char *dev_name){
	FILE *fp;
	char buf[256], *p;
	int ret=FS_UNKNOWN;
	fp=popen(CMD_MOUNT, "r");	

	while(fgets(buf, sizeof(buf), fp) != NULL){
		if(!strncmp(dev_name, buf, 9)){
			p=strstr(buf,"type");
			p=p+5;
			*(p+5)='\0';
			if(strstr(p, "ext2"))
				ret = FS_EXT2;
			else if(strstr(p, "ext3"))
				ret = FS_EXT3;
			else if(strstr(p, "msdos"))
				ret = FS_MSDOS;
			else if(strstr(p, "vfat"))
				ret = FS_FAT;
			else if(strstr(p, "ntfs"))
				ret = FS_NTFS;
			else
				ret = FS_UNKNOWN;
		}
	}
	if (fp!=NULL)
		pclose(fp);
	if(ret == FS_MSDOS || ret == FS_FAT){
		char cmd[64], mnt_dir[64];
		Get_QNAP_Mount_Name(dev_name, mnt_dir, ROOT_PART);
		sprintf(cmd, "/bin/umount %s 1>/dev/null 2>/dev/null", mnt_dir);
		system(cmd);
//Try to remount because moust mount with umask when file system is fat
		sprintf(cmd, "%s %s %s -o utf8,umask=0000,shortname=mixed 1>/dev/null 2>/dev/null", CMD_MOUNT, dev_name, mnt_dir);
		system(cmd);
	}
	if(ret == FS_NTFS){
		char cmd[64], mnt_dir[64];
		Get_QNAP_Mount_Name(dev_name, mnt_dir, ROOT_PART);
		sprintf(cmd, "/bin/umount %s 1>/dev/null 2>/dev/null", mnt_dir);
		system(cmd);
		sprintf(cmd, "%s %s %s -o utf8,umask=0200 1>/dev/null 2>/dev/null", CMD_MOUNT, dev_name, mnt_dir);
		system(cmd);
	}
		
	return ret;
}

int Get_Disk_FS_Type_By_Name(char *dev_name, int *part_no){
	char part_name[PARTITION_NAME_LENGTH], mnt_dir[PARTITION_NAME_LENGTH];
	char cmd[MAX_CMDLINE_LENGTH];
	int fs=FS_UNKNOWN;

	sprintf(part_name, "%s1", dev_name);
	Get_QNAP_Mount_Name(dev_name, mnt_dir, ROOT_PART);
	
	if(!Is_Mounted(part_name)){
		DIR *dir;
		if(!(dir=opendir(mnt_dir)))
                mkdir(mnt_dir,S_IRUSR | S_IWUSR | S_IXUSR |S_IRGRP | S_IWGRP | S_IXGRP |S_IROTH | S_IWOTH | S_IXOTH);

		if (dir!=NULL)
	     	   	closedir(dir);
		sprintf(cmd, "%s %s %s", CMD_MOUNT, part_name, mnt_dir);
        	system(cmd);
	}
	
	if(Is_Mounted(part_name))
		fs = get_fs_mounted(part_name);
	if(fs==FS_UNKNOWN){
		sprintf(part_name, "%s ", dev_name);
		*part_no=0;
		sprintf(cmd, "%s %s %s", CMD_MOUNT, dev_name, mnt_dir);
		system(cmd);
		//The first partition is unknown check whole disk type
	       if(!Is_Mounted(part_name)){
        	        sprintf(cmd, "%s %s %s", CMD_MOUNT, part_name, mnt_dir);
                	system(cmd);
        	}

		if(Is_Mounted(dev_name))
			fs = get_fs_mounted(part_name);
	}
	else{
		if(!Is_Mounted(part_name)){
			Mount_Disk(part_name, FS_UNKNOWN);
		}
		*part_no=1;
	}
        return fs;
}

int Get_Disk_FS_Type(int disk_no, int *part_no){

	char disk_section[PARTITION_NAME_LENGTH], buf[PARTITION_NAME_LENGTH];
	char dev_name[HD_DEVICE_NAME_LENGTH], part_name[PARTITION_NAME_LENGTH];
	int fs;
	
	sprintf(disk_section, "VOLUME %d", disk_no);
	GetPrivateProfileString(disk_section, DEVICE_FIELD, "", buf, sizeof(buf), STORAGE_CONF);
	sprintf(dev_name, "%s1", buf);
	
	if(!Is_Mounted(dev_name))
		fs = Get_FStype(dev_name);
	else
		fs = get_fs_mounted(dev_name);
	sprintf(part_name, "%s ", dev_name);
	if(fs==FS_UNKNOWN){
		*part_no = 0;
                //The first partition is unknown check whole disk type
                if(!Is_Mounted(dev_name))
                        fs = Get_FStype(dev_name);
                else
                        fs = get_fs_mounted(part_name);
        }
	else
		*part_no = 1;
	return fs;
}

int Is_QNAP_Disk(char *dev_name){

	char part_name[PARTITION_NAME_LENGTH];
	sprintf(part_name, "%s1", dev_name);
	if(!Is_Mounted(part_name)){
//		printf("Try to mount\n");
		if(Mount_QNAP_Disk_By_Name(dev_name, ROOT_PART, FS_EXT3)==0){
			if(Verify_Disk_Conf_By_Name(dev_name)<0)
				return 0;
			else
				return 1;
		}
		else
			return 0;
	}
	else{
//		printf("Mounted\n");
		if(Verify_Disk_Conf_By_Name(dev_name)<0){
			return 0;
		}
		else{
			return 1;
		}
	}
	fprintf(stderr, "Something wrong here..!\n");
        return 0;
}
/*
BOOL Is_USB_Host_Exist(int drive_no){
	int fd=0, size=0;
	BOOL ret=TRUE;
        char buffer[BUF_SIZE];
	switch(drive_no){
		case 1:
		case 2:
		case 3:
			fd = open(USB_PROC, O_RDONLY);
			size = read(fd, buffer, sizeof(buffer));
			if(fd<0){
				fprintf(stderr, "Error open device (%d).", drive_no);
				ret=FALSE;
				close(fd);
				break;
			}
			close(fd);
			//Read 2 4 6 bits
			if(buffer[drive_no*2]!='0'){
				ret=TRUE;
			}
			else
				ret=FALSE;
			break;
		default:
			fprintf(stderr, "Error given drive number (%d).", drive_no);
			break;
	}	
	return ret;
}
*/
int Get_HD_Type(int disk_no){
	if(disk_no==1 || disk_no==2)
		return SATA;
	else if(disk_no>3 && disk_no<25)
		return USB;
	else
		return ERROR_FAIL;
}
/*
int remove_entry_from_list(TS_DISK_INFO_P *disk_list, TS_DISK_INFO_P node, char *device_name){
//	if(*disk_list){
	if(node){
		TS_DISK_INFO_P tmp=NULL, pre_tmp=NULL;
		for(tmp=node;tmp!=NULL;pre_tmp=tmp,tmp=tmp->next){
			if(!strcmp(tmp->device_name, device_name)){
				pre_tmp->next=tmp->next;
				free(tmp);
				break;
			}
		}
	}
//	else{
//		*disk_list=(*disk_list)->next;
//	}
	return 0;
}
*/
void remove_entry_from_list(TS_DISK_INFO_P *list, char *dev_name){
        if(*list){
                TS_DISK_INFO_P tmp = NULL, pre_tmp = NULL;
                for(tmp=*list;tmp!=NULL;pre_tmp=tmp,tmp=tmp->next){
                        if(!strcmp(tmp->device_name, dev_name)){
				if(pre_tmp)
                                	pre_tmp->next=tmp->next;
				else
					*list = tmp->next;
                                free(tmp);
                                break;
                        }
                }

        }
}

void add_entry_to_list(TS_DISK_INFO_P *list, TS_DISK_INFO_P node){

        if(*list){
                TS_DISK_INFO_P tmp = NULL, pre_tmp = NULL;

                for(tmp=*list, pre_tmp=tmp; tmp!=NULL; pre_tmp=tmp, tmp=tmp->next){
                        if(!strcmp(tmp->device_name, node->device_name))
                                return;
                }
                node->next=NULL;
                pre_tmp->next=node;
//              new->next=NULL;
        }
        else{
                node->next=NULL;
                *list=node;
        }
}

/*
void add_entry_to_list(TS_DISK_INFO_P *disk_list, TS_DISK_INFO_P node, TS_DISK_INFO_P new){

	if(node){
		TS_DISK_INFO_P tmp = NULL, pre_tmp = NULL;
	
		for(tmp=node, pre_tmp=tmp; tmp!=NULL; pre_tmp=tmp, tmp=tmp->next){
			if(!strcmp(tmp->device_name, new->device_name))
				return;
		}
		new->next=pre_tmp->next;
		pre_tmp->next=new;
//		new->next=NULL;
	}
	else{
		new->next=NULL;
		*disk_list=new;
	}
	
	return;
}
*/
int show_list(TS_DISK_INFO_P disk_list){
	TS_DISK_INFO_P tmp = NULL;
	for(tmp=disk_list; tmp!=NULL; tmp=tmp->next){
		printf("Device: %s\n", tmp->device_name);
		printf("Vendor: %s\n", tmp->vendor);
        	printf("Model: %s\n", tmp->model);
		printf("Capacity: %lld\n", tmp->capacity);
		printf("Type: %s\n", tmp->type==SATA?"SATA":"USB");
		printf("File System: %d\n", tmp->sys_id);
		printf("Partition Number: %d\n\n", tmp->part_no);
	}
	return 0;
}

void free_list(TS_DISK_INFO_P *disk_list, TS_DISK_INFO_P node){

	if(*disk_list){
		TS_DISK_INFO_P tmp = NULL;
		for(tmp=node;tmp!=NULL;tmp=tmp->next)
			free(tmp);
	}
	else{
	}
}

#define MAX_PORT        7
static int hub_port[MAX_PORT+1] = { USB_PORT, USB_HUB_PORT1, USB_HUB_PORT2, USB_HUB_PORT3,
                        USB_HUB_PORT4, USB_HUB_PORT5, USB_HUB_PORT6, USB_HUB_PORT7};
char hub_msg[MAX_PORT+1][20] = { "USB Storage(No hub)", "Hub port 1", "Hub port 2", "Hub port 3", "Hub port 4", "Hub port 5", "Hub port 6", "Hub port 7"};

//Get speed field by usb host number
int get_ts_speed_field(int usb_host){
        char *delim = " ", *p;
        int fd=0, size=0, idx=0;
        char buf[128];

        fd=open(USB_PROC_FOR_HOTSWAP, O_RDONLY);
        if(fd<0)
                return ERROR_FAIL;
        size = read(fd, buf, sizeof(buf));
        close(fd);
        buf[size]='\0';
        if((p=strtok(buf, delim))==NULL)
                return ERROR_FAIL;
        while(((p=strtok(NULL, delim))!=NULL) && idx<(usb_host+4)){
                idx++;
        }
        return atox(p);
}

int Get_TS_Speed(char *dev_name){
        int host=0, value=0;
        switch(*(dev_name+7)){
                case 'd':
                case 'e':
                case 'f':
                case 'g':
                case 'h':
                case 'i':
                case 'j':
                        host=1;
                        break;
                case 'k':
                case 'l':
                case 'm':
                case 'n':
                case 'o':
                case 'p':
                case 'q':
                        host=2;
                        break;
                case 'r':
                case 's':
                case 't':
                case 'u':
                case 'v':
                case 'w':
                case 'x':
                default:
                        host=3;
                        break;
        }
        value = get_ts_speed_field(host);

        if(value&hub_port[0]){
                return USB20;
        }
        else{
                if(value & (hub_port[*(dev_name+7)-dev_map[host+2]+1])){
                        return USB20;
                }
        }
	printf("USB 1.0\n");
        return USB10;
}

int check_usb(char *dev_name){
	FILE *fp = NULL;
	char cmd[MAX_CMDLINE_LENGTH], mnt_dir[BUF_SIZE], dev[BUF_SIZE], buf[256];
	DIR *dir=NULL;
	int	ret=S_READY;

	char part_name[BUF_SIZE];

	if(!Have_Qnap_Flag(dev_name)){
		strncpy(dev, dev_name+5, 3);
        	dev[3]='\0';
//        sprintf(mnt_dir, "/mnt/%s", dev);
		sprintf(mnt_dir, "/share/external/%s", dev);
	
		if(!(dir=opendir(mnt_dir)))
        	        mkdir(mnt_dir,S_IRUSR | S_IWUSR | S_IXUSR |S_IRGRP | S_IWGRP | S_IXGRP |S_IROTH | S_IWOTH | S_IXOTH);

		if (dir!=NULL)
		        closedir(dir);

		sprintf(part_name, "%s1", dev_name);	
//	sprintf(cmd, "%s %s %s -o utf8,umask=0000,shortname=mixed 2>&1", CMD_MOUNT, part_name, mnt_dir);
		sprintf(cmd, "%s %s %s 2>&1", CMD_MOUNT, part_name, mnt_dir);
		system(cmd);
	
		sprintf(cmd, "%s %s %s 2>&1", CMD_MOUNT, dev_name, mnt_dir);
//	sprintf(cmd, "%s %s %s -o utf8,umask=0000 2>&1", CMD_MOUNT, dev_name, mnt_dir);
//	printf("cmd:%s\n", cmd);
		fp = popen(cmd, "r");
		if(fp==NULL){
                //Nothing
	        }
		else
		if (fgets(buf, sizeof(buf), fp)!=NULL){
			if(strstr(buf, "No medium found"))
				ret=S_NO_MEDIUM_FOUND;
			else if(strstr(buf, "not a valid block device"))
				ret=S_NOT_EXIST;
			if(fp!=NULL)
				pclose(fp);
		}
	}
	return ret;
}

int Get_TS_Disk_Info(TS_DISK_INFO_P disk_info, char *dev_name){
	
	char sg_dev[BUF_SIZE], cmd[MAX_CMDLINE_LENGTH];
	char buf[BUF_SIZE];
	char hd_sec[BUF_SIZE];
	FILE *fp;
	int index=1;
	int idx=(int)(*(dev_name+7)-'a');
//	int status=0;
	
	sprintf(disk_info->device_name, "%s", dev_name);
	sprintf(sg_dev, "/dev/sg%d", idx);
	disk_info->type = Get_HD_Type(idx);
	sprintf(cmd, "/sbin/get_sg_info %s", sg_dev);
	fp=popen(cmd, "r");
	if(fp==NULL){
		//Nothing
	}
	else
	if(fgets(buf, sizeof(buf), fp)!=NULL){
		char *delim="$", *p;

		if((p=strtok(buf, delim)))
			sprintf(disk_info->vendor, "%s", p);
		if((p=strtok(NULL, delim)))
			sprintf(disk_info->model, "%s", p);
		if((p=strtok(NULL, delim)))
			disk_info->capacity=atol(p);
	}
	if (fp!=NULL)
		pclose(fp);
//	if(disk_info->capacity<10000)
	disk_info->qnap_disk=0;
//	else
//		disk_info->qnap_disk = Is_QNAP_Disk(dev_name);
	if(strcmp(dev_name, "/dev/sda")&&strcmp(dev_name, "/dev/sdb")&&strcmp(dev_name, "/dev/sdc"))
		disk_info->type=USB;
	else
		disk_info->type=SATA;
		
	disk_info->speed = Get_TS_Speed(dev_name);
//printf("Speed = %d.........%s\n", disk_info->speed, dev_name);
	if(disk_info->speed != USB10 && disk_info->speed != USB20)
		disk_info->speed = USB20;

//	disk_info->speed = Get_TS_USB_Disk_Speed_Type(dev_name);
	/*if(disk_info->part_no==0){
		if(Is_Mounted(dev_name)){
			char mnt_dir[PARTITION_NAME_LENGTH];
			Get_QNAP_Mount_Name(dev_name, mnt_dir, DATA_PART);
			disk_info->free_capacity = Get_Free_Disk_Size(mnt_dir);
			disk_info->status = S_READY;
		}
		else{
		//	disk_info->status = S_NOT_MOUNTED;
			disk_info->capacity = 0;
                        disk_info->free_capacity = 0;
			disk_info->status = status;
		}
	}
	else{
		sprintf(part, "%s1", dev_name);
		if(Is_Mounted(part)){
			char mnt_dir[PARTITION_NAME_LENGTH];
			Get_QNAP_Mount_Name(dev_name, mnt_dir, DATA_PART);
B
			disk_info->status = status;
		}
		else{
		//	disk_info->status = S_NOT_MOUNTED;
			disk_info->capacity = 0;
			disk_info->free_capacity = 0;
			disk_info->status = status;
		}
	}*/	
	for(index=1;index<10;index++){
		sprintf(hd_sec, "%s%d", SHARE_USB_PREFIX, index);
		GetPrivateProfileString("", hd_sec, "none", buf, sizeof(buf), MAPPING);
		if(!strcmp(buf, dev_name)){
			sprintf(disk_info->share, "%s", hd_sec);
			break;
		}
	}
	if(!strcmp(dev_name, "/dev/sdb")){
		strcpy(disk_info->share, SHARE_ESATA);
	}
	else{
		if(index>9)
			sprintf(disk_info->share, "none");
	}
	if(disk_info->status == S_READY && strcmp(disk_info->share, "none")){
		if(!Have_Qnap_Flag(dev_name))
			Create_Storage_Share(disk_info->share, SHARE_ADMINISTRATOR, dev_name);
		else{
			//QNAP RAID1 Mode
		}
	}
	return 0;
}

int add_mapping(char *dev){
	int idx=1;
	char hd_sec[BUF_SIZE], buf[BUF_SIZE];
	if(!strcmp(dev, "/dev/sda") || !strcmp(dev, "/dev/sdb"))
		return -1;
	for(idx=1;idx<10;idx++){
                sprintf(hd_sec, "%s%d", SHARE_USB_PREFIX, idx);
                GetPrivateProfileString("", hd_sec, "none", buf, sizeof(buf), MAPPING);
                if(!strcmp(buf, dev))
                        return idx;
        }
	for(idx=1;idx<10;idx++){
		sprintf(hd_sec, "%s%d", SHARE_USB_PREFIX, idx);
		GetPrivateProfileString("", hd_sec, "none", buf, sizeof(buf), MAPPING);
		if(!strncmp(buf, "none", 4)){
			WritePrivateProfileString("", hd_sec, dev, MAPPING);
			return idx;
		}
	}
	return -1;
}

int remove_mapping(char *dev){
	int idx=0;
	char hd_sec[BUF_SIZE],buf[BUF_SIZE];
	for(idx=1;idx<10;idx++){
                sprintf(hd_sec, "%s%d", SHARE_USB_PREFIX, idx);
                GetPrivateProfileString("", hd_sec, "none", buf, sizeof(buf), MAPPING);
//		printf("%s\n", buf);
                if(!strcmp(buf,dev)){
			WritePrivateProfileString("", hd_sec, "none", MAPPING);
                        return idx;
		}
        }
	return 0;
}

int Try_Host(int drive_no){
        int fd=0, idx=0, ret=0;
        char dev_name[BUF_SIZE];
//      printf("Try Host: %p\n", branch_device_array);
        if(drive_no>2 && drive_no<6){
                for(idx=0;idx<4;idx++){
                        sprintf(dev_name, "/dev/sd%c", dev_map[drive_no]+idx);
//                      printf("%s\n", dev_name);
                        fd=open(dev_name, O_RDONLY);
                        if(fd>=0){
                                ret=1;
                        }
                        close(fd);
                }
        }
        else
                fprintf(stderr, "Error given drive number (%d).", drive_no);
        return ret;
}

/*
int Get_Disk_List_Info(TS_DISK_INFO **disk_list){
	int	drive_cnt = 0, i=0, hd_exist_flag=0;
	int 	cnt=0, idx=0, fd=0;	
	char	dev_name[BUF_SIZE];
	
	drive_cnt = Get_Profile_Integer("Storage","Disk Drive Number",0);

	for (i=1;i<drive_cnt+1;i++){
		usleep(10000);
	
		hd_exist_flag = Get_HD_Exist(i);
		if(hd_exist_flag){
//		printf("%d HD exist\n", i);
			if(i==1 || i==2){
				TS_DISK_INFO_P tmp = (TS_DISK_INFO_P)malloc(sizeof(TS_DISK_INFO));

				sprintf(tmp->device_name, "/dev/sd%c", 'a'+(i-1));
				Get_TS_Disk_Info(tmp, tmp->device_name);
				tmp->type=SATA;				
				add_entry_to_list(disk_list, *disk_list, tmp);
				cnt++;
			}
			else if(i==3 || i==4 || i==5){ //Scan each 4 slot of host
				sleep(5);
				Try_Host(i);
				for(idx=0;idx<4;idx++){
        	                	sprintf(dev_name, "/dev/sd%c", dev_map[i]+idx);
                	        	fd=open(dev_name, O_RDONLY);
                        		if(fd>=0){
						TS_DISK_INFO_P tmp = (TS_DISK_INFO_P)malloc(sizeof(TS_DISK_INFO));
//				printf("%s Open success\n", dev_name);
						add_mapping(dev_name);
                        		        sprintf(tmp->device_name, "%s", dev_name);
						Get_TS_Disk_Info(tmp, tmp->device_name);
						tmp->type=USB;
						add_entry_to_list(disk_list, *disk_list, tmp);
                        		        cnt++;
                        		}
					else
						remove_mapping(dev_name);
                        		close(fd);
                		}
			}
		}
	}
	Write_Disk_Info_To_Config(*disk_list);
	free_list(disk_list, *disk_list);
	return cnt;
}
*/
int Write_One_Disk_Info_To_Config(TS_DISK_INFO_P disk_list, int disk_no){
	char disk_section[BUF_SIZE], buf[BUF_SIZE];
	sprintf(disk_section, "VOLUME %d", disk_no);
	sprintf(buf, "%d", disk_no);
	WritePrivateProfileString(disk_section, NO_FIELD, buf, STORAGE_CONF);
	WritePrivateProfileString(disk_section, DEVICE_FIELD, disk_list->device_name, STORAGE_CONF);
	WritePrivateProfileString(disk_section, VENDOR_FIELD, disk_list->vendor, STORAGE_CONF);
        WritePrivateProfileString(disk_section, MODEL_FIELD, disk_list->model, STORAGE_CONF);
        sprintf(buf, "%lld", disk_list->capacity);
        WritePrivateProfileString(disk_section, CAPACITY_FIELD, buf, STORAGE_CONF);
        sprintf(buf, "%lld", disk_list->free_capacity);
        WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, buf, STORAGE_CONF);       
	sprintf(buf, "%d", disk_list->type);
	WritePrivateProfileString(disk_section, TYPE_FIELD, buf, STORAGE_CONF);
	sprintf(buf, "%d", disk_list->qnap_disk);
        WritePrivateProfileString(disk_section, QNAPDISK_FIELD, buf, STORAGE_CONF);
        sprintf(buf, "%d", disk_list->sys_id);
        WritePrivateProfileString(disk_section, FS_FIELD, buf, STORAGE_CONF);
        sprintf(buf, "%d", disk_list->part_no);
        WritePrivateProfileString(disk_section, PART_NO_FIELD, buf, STORAGE_CONF);
        sprintf(buf, "%d", disk_list->status);
        WritePrivateProfileString(disk_section, STATUS_FIELD, buf, STORAGE_CONF);
        sprintf(buf, "%d", disk_list->speed);
        WritePrivateProfileString(disk_section, SPEED_FIELD, buf, STORAGE_CONF);
        WritePrivateProfileString(disk_section, SHARE_FIELD, disk_list->share, STORAGE_CONF);
        WritePrivateProfileString(disk_section, RAID_FIELD, "", STORAGE_CONF);
	
	return 0;
}

int Write_Disk_Info_To_Config(TS_DISK_INFO_P disk_list){
	TS_DISK_INFO_P tmp=NULL;
	char disk_section[BUF_SIZE], buf[BUF_SIZE];
	int i=1;

	system("/bin/rm /etc/config/storage.conf -rf");
	system("/bin/touch /etc/config/storage.conf");

//	p();
	tmp=disk_list;
	while(tmp!=NULL){
		sprintf(disk_section, "VOLUME %d",i);
		sprintf(buf, "%d", i);
		WritePrivateProfileString(disk_section, NO_FIELD, buf, STORAGE_CONF); 
		WritePrivateProfileString(disk_section, DEVICE_FIELD, tmp->device_name, STORAGE_CONF);
		WritePrivateProfileString(disk_section, VENDOR_FIELD, tmp->vendor, STORAGE_CONF);
		WritePrivateProfileString(disk_section, MODEL_FIELD, tmp->model, STORAGE_CONF);
		sprintf(buf, "%lld", tmp->capacity);
		WritePrivateProfileString(disk_section, CAPACITY_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%lld", tmp->free_capacity);
		WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%d", tmp->type);
		WritePrivateProfileString(disk_section, TYPE_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%d", tmp->qnap_disk);
		WritePrivateProfileString(disk_section, QNAPDISK_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%d", tmp->sys_id);
		WritePrivateProfileString(disk_section, FS_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%d", tmp->part_no);
		WritePrivateProfileString(disk_section, PART_NO_FIELD, buf, STORAGE_CONF);
		sprintf(buf, "%d", tmp->status);
		WritePrivateProfileString(disk_section, STATUS_FIELD, buf, STORAGE_CONF);
		
		sprintf(buf, "%d", tmp->speed);
		WritePrivateProfileString(disk_section, SPEED_FIELD, buf, STORAGE_CONF);
	
		WritePrivateProfileString(disk_section, SHARE_FIELD, tmp->share, STORAGE_CONF);
		WritePrivateProfileString(disk_section, RAID_FIELD, "", STORAGE_CONF);
		
		i++;
		tmp=tmp->next;
	}
	sprintf(buf, "%d", i-1);
	WritePrivateProfileString(STORAGE_GLOBAL_SECTION, AVAIL_FIELD, buf, STORAGE_CONF);

//	v();
	return 0;
}

int get_ts_disk_size(TS_DISK_INFO_P disk_info, int qnap_flag)
{
    if(qnap_flag)
    {
	char buf[32];
	sprintf(buf, "%s3", disk_info->device_name);
	if(Is_Mounted(buf)){
		char mnt_dir[PARTITION_NAME_LENGTH];
		Get_QNAP_Mount_Name(disk_info->device_name, mnt_dir, DATA_PART);
		disk_info->free_capacity = Get_Free_Disk_Size(mnt_dir);
		return 0;
	}
	else
	{
//		disk_info->capacity = 0;
                disk_info->free_capacity = 0;
		return -1;
	}
    }
    else
    {
	if(disk_info->part_no==0){
		if(Is_Mounted(disk_info->device_name)){
			char mnt_dir[PARTITION_NAME_LENGTH];
			Get_QNAP_Mount_Name(disk_info->device_name, mnt_dir, DATA_PART);
			disk_info->free_capacity = Get_Free_Disk_Size(mnt_dir);
			return 0;
		}
		else{
		//	disk_info->status = S_NOT_MOUNTED;
//			disk_info->capacity = 0;
                        disk_info->free_capacity = 0;
			return -1;
		}
	}
	else{
		char part[BUF_SIZE];
		sprintf(part, "%s1", disk_info->device_name);
		if(Is_Mounted(part)){
			char mnt_dir[PARTITION_NAME_LENGTH];
			Get_QNAP_Mount_Name(disk_info->device_name, mnt_dir, DATA_PART);
//			printf("mnt_dir: %s\n", mnt_dir);
			disk_info->free_capacity = Get_Free_Disk_Size(mnt_dir);
			return 0;
		}
		else{
		//	disk_info->status = S_NOT_MOUNTED;
//			disk_info->capacity = 0;
			disk_info->free_capacity = 0;
			return -1;
		}
	}	
    }
}

void Add_Dev(char *dev_name){
	TS_DISK_INFO_P disk_list=NULL, tmp=NULL;

	Get_Disk_Info_From_Config(&disk_list);
	tmp = (TS_DISK_INFO_P)malloc(sizeof(TS_DISK_INFO));
	strcpy(tmp->device_name, dev_name);

	if(Have_Qnap_Flag(dev_name))
	{
		do_qraid1_action(dev_name, PLUG_IN);
		add_mapping(dev_name);
		if(get_ts_disk_size(tmp, 1) == 0){
			tmp->sys_id = FS_EXT3;
			tmp->status = S_READY;
		}
		else	// -1
			tmp->status = S_NOT_MOUNTED;
	}
	else {
		int status;
		add_mapping(dev_name);
		status = check_usb(dev_name);
		if(status != S_NOT_MOUNTED && status!=S_NO_MEDIUM_FOUND)
			tmp->sys_id = Get_Disk_FS_Type_By_Name(dev_name, &tmp->part_no);
		else
			tmp->sys_id = FS_UNKNOWN;

		if(get_ts_disk_size(tmp, 0) == 0)
		{
			// add by Kent 2005/11/25
			if (tmp->sys_id==FS_UNKNOWN)
				tmp->status = S_NOT_MOUNTED;
			else
				tmp->status = S_READY;
		}
		else	// -1
		{
			// add by Kent 2005/11/25
			if (tmp->sys_id==FS_UNKNOWN)
				tmp->status = S_NOT_MOUNTED;
			else
				tmp->status = status;
		}
	}

	Get_TS_Disk_Info(tmp, tmp->device_name);
	add_entry_to_list(&disk_list, tmp);
//	show_list(disk_list);
	
	Write_Disk_Info_To_Config(disk_list);
//	free(tmp);
	free_list(&disk_list, disk_list);

	system("/etc/init.d/rsyncd_srv.sh > /dev/null 2>&1");
}

int Remove_Dev(char *dev_name){
	int ret = 0;
	TS_DISK_INFO_P disk_list=NULL;
	char qraid1_dev_name[32];

	GetProfileString("QRAID1", "device name", "", qraid1_dev_name, sizeof(qraid1_dev_name));
	if(!strcasecmp(dev_name, qraid1_dev_name)){
		ret = do_qraid1_action(dev_name, PLUG_OUT);
		remove_mapping(dev_name);
	}
	else {
        	ret = Stop_And_Umount_By_Dev(dev_name);
		remove_mapping(dev_name);
	}

	Get_Disk_Info_From_Config(&disk_list);
	remove_entry_from_list(&disk_list, dev_name);
	Write_Disk_Info_To_Config(disk_list);

	system("/etc/init.d/rsyncd_srv.sh > /dev/null 2>&1");
	return ret;
}

int Get_Disk_Info_From_Config(TS_DISK_INFO_P *disk_list){
	
	char disk_section[BUF_SIZE], buf[BUF_SIZE];
	int idx=1, avail_disk=1;
	TS_DISK_INFO_P tmp=NULL;
	GetPrivateProfileString(STORAGE_GLOBAL_SECTION, AVAIL_FIELD, "1", buf, sizeof(buf), STORAGE_CONF);
	
	avail_disk=atoi(buf);
	for(idx=1;idx<avail_disk+1;idx++){
		int status;
		tmp = (TS_DISK_INFO_P)malloc(sizeof(TS_DISK_INFO));
		sprintf(disk_section, "VOLUME %d", idx);
		GetPrivateProfileString(disk_section, NO_FIELD, "", buf, sizeof(buf), STORAGE_CONF);
		tmp->no = atoi(buf);
		GetPrivateProfileString(disk_section, DEVICE_FIELD, "", tmp->device_name, sizeof(tmp->device_name), STORAGE_CONF);	
		GetPrivateProfileString(disk_section, VENDOR_FIELD, "", tmp->vendor, sizeof(tmp->vendor), STORAGE_CONF);
		GetPrivateProfileString(disk_section, MODEL_FIELD, "", tmp->model, sizeof(tmp->model), STORAGE_CONF);
		GetPrivateProfileString(disk_section, CAPACITY_FIELD, "0", buf, sizeof(buf), STORAGE_CONF);
		tmp->capacity = atol(buf);

		GetPrivateProfileString(disk_section, PART_NO_FIELD, "0", buf, sizeof(buf), STORAGE_CONF);
		tmp->part_no = atoi(buf);
		status = Get_Disk_Status(idx);
		GetPrivateProfileString(disk_section, F_CAPACITY_FIELD, "0", buf, sizeof(buf), STORAGE_CONF);

		if(status==S_FORMATTING){
			tmp->free_capacity = atol(buf);
		}
		else{
			if(Have_Qnap_Flag(tmp->device_name)){
				get_ts_disk_size(tmp,1);
			}
			else{
				get_ts_disk_size(tmp,0);
			}
			if(tmp->free_capacity != atol(buf)){
				sprintf(buf, "%lld", tmp->free_capacity);
				WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, buf, STORAGE_CONF); 
			}
		}
//add_entry_to_list(disk_list, tmp);
		GetPrivateProfileString(disk_section, TYPE_FIELD, "1", buf, sizeof(buf), STORAGE_CONF);
		tmp->type=atoi(buf);
		GetPrivateProfileString(disk_section, QNAPDISK_FIELD, "0", buf, sizeof(buf), STORAGE_CONF);
		tmp->qnap_disk=atoi(buf);
		GetPrivateProfileString(disk_section, FS_FIELD, "8", buf, sizeof(buf), STORAGE_CONF);
		tmp->sys_id=atoi(buf);

		GetPrivateProfileString(disk_section, PART_NO_FIELD, "1",buf, sizeof(buf), STORAGE_CONF);
		tmp->part_no = atoi(buf);
		GetPrivateProfileString(disk_section, STATUS_FIELD, "-4",buf, sizeof(buf), STORAGE_CONF);
		tmp->status = atoi(buf);
		GetPrivateProfileString(disk_section, SPEED_FIELD, "1",buf, sizeof(buf), STORAGE_CONF);
		tmp->speed = atoi(buf);
		if(tmp->speed != USB10 && tmp->speed != USB20){
			tmp->speed=USB20;
			sprintf(buf, "%d", tmp->speed);
			WritePrivateProfileString(disk_section, SPEED_FIELD, buf, STORAGE_CONF);
		}
		
		GetPrivateProfileString(disk_section, SHARE_FIELD, "none",tmp->share, sizeof(tmp->share), STORAGE_CONF);
		tmp->next=NULL;
		add_entry_to_list(disk_list, tmp);
	}
	return avail_disk;
}

int Get_DriveNo_By_DevName(char* dev_nameP)
{
	int i, volume;
	char buf[32], dev_name[32];

	volume = Get_Private_Profile_Integer("Global", "Available Disk", 0, STORAGE_CONF);
	for (i = 1; i <= volume; i++)
	{ 
        	sprintf(buf,"VOLUME %d",i);
		GetPrivateProfileString(buf, DEVICE_FIELD, "", dev_name, sizeof(dev_name), STORAGE_CONF);
		if(!strcasecmp(dev_nameP, dev_name))
			return i;
	
	}

	return -1;
}

int Set_HD_Exist(int drive_no, int exist){
	char hd_sec_name[20], buf[4];
	int ret=0;
	sprintf(hd_sec_name,"Drive %d EXIST",drive_no);
	sprintf(buf, "%d", exist);
	ret = WriteProfileString("Disk", hd_sec_name, buf);
	return ret;
}

int Get_HD_Exist(int drive_no){
	char hd_sec_name[20], buf[4];
        sprintf(hd_sec_name,"Drive %d EXIST",drive_no);
	GetProfileString("Disk", hd_sec_name, "0", buf, sizeof(buf));
	return atoi(buf);
}

int Main_Check_Disk(){
	int ret=0;
	char part_name[64];
	int status = Get_Disk_Status(1);
	Set_Disk_Status(1, S_SCANNING);
	Get_Partition_Name(1, DATA_PART, part_name);
	HD_Reset_Check_Progress(part_name);
	Stop_Services();
	if(Is_Mounted(part_name))
		ret = Unmount_NAS_Partition(1, DATA_PART);
	Start_Services();
	if(ret<0){
		Mount_NAS_Partition(1, DATA_PART, EXT3_FS);	
		Set_Disk_Status(1, status);
		return ERROR_CANT_UMOUNT;
	}
	ret = Scan_Partition(part_name, CHECK_WITH_REPAIR, EXT3_FS);
	if(ret!=0){
		if(Mount_NAS_Partition(1, DATA_PART, EXT3_FS)==SUCCESS)
                	Set_Disk_Status(1, S_READY);
		else
			Set_Disk_Status(1, S_NOT_MOUNTED);
		return ERROR_FAIL;
	}
	else{
		if(Mount_NAS_Partition(1, DATA_PART, EXT3_FS)==SUCCESS)
			Set_Disk_Status(1, S_READY);
		else
                	Set_Disk_Status(1, S_NOT_MOUNTED);

		return SUCCESS;
	}
}

int Format_QNAP_Disk(int disk_no, int fs){
	char dev_name[HD_DEVICE_NAME_LENGTH], disk_section[HD_DEVICE_NAME_LENGTH];;
	char cmd[256], mkfs_cmd[256], part_name[HD_DEVICE_NAME_LENGTH],item[64];
	int ret=0, fs_no;
	char share_name[256];
	int status = Get_Disk_Status(disk_no);

	Set_Disk_Status(disk_no, S_FORMATTING);
	sprintf(disk_section, "VOLUME %d", disk_no);
	GetPrivateProfileString(disk_section, DEVICE_FIELD, "",dev_name, sizeof(dev_name), STORAGE_CONF);
	sprintf(item,"%s1",dev_name);
	HD_Reset_Format_Progress(item);//Shone added and modified 2005,10,27
	sprintf(part_name, "%s", dev_name);
	strcat(part_name,"1");

	Stop_Services();
	
	GetPrivateProfileString(disk_section, SHARE_FIELD, "none", share_name, sizeof(share_name), STORAGE_CONF);

        Remove_NAS_Share(share_name);
	if(!Have_Qnap_Flag(dev_name)){
		if(Is_Mounted(dev_name)){
			fs_no = get_fs_mounted(dev_name);
			ret = Stop_And_Umount_By_Dev(dev_name);
			if(ret<0)
				goto start_service;
		}
		if(Is_Mounted(part_name)){
			fs_no = get_fs_mounted(dev_name);
			ret = Stop_And_Umount_By_Dev(part_name);
			if(ret<0)
                                goto start_service;
		}
	}
	else{
		char qnap_part[32];
		sprintf(qnap_part, "%s1", dev_name);
		if(Is_Mounted(qnap_part)){
			ret = Unmount_NAS_Partition_Ex(dev_name, ROOT_PART);
			if(ret<0)
                                goto start_service;
		}
		sprintf(qnap_part, "%s3", dev_name);
		if(Is_Mounted(qnap_part)){
                	ret = Unmount_NAS_Partition_Ex(dev_name, DATA_PART);
			if(ret<0)
                                goto start_service;
		}
	}
start_service:
	Start_Services();
	if(ret<0){
//		if(!Have_Qnap_Flag(dev_name)){
//		}
//		else{
//		}
		Set_Disk_Status(disk_no, status);
		return ERROR_CANT_UMOUNT;
	}
	sprintf(cmd, "%s -d all %s", CMD_FDISK, dev_name);
	ret = system(cmd);
	if(ret!=0){
		Set_Disk_Status(disk_no, S_NOT_MOUNTED);
		return ERROR_FORMAT_FAIL;
	}
	
	switch(fs){
	case FS_MSDOS:
	case FS_FAT16:
	case FS_FAT32:
		sprintf(cmd, "%s -n 1 -z c %s", CMD_FDISK, dev_name);
		sprintf(mkfs_cmd, "/sbin/mkfs.vfat -F 32 %s", part_name);
		break;
	case FS_EXT2:
		sprintf(cmd, "%s -n 1 -z 83 %s", CMD_FDISK, dev_name);
                sprintf(mkfs_cmd, "%s -m0 -b %d -F %s 2>/dev/null 1>/dev/null", CMD_MKE2FS, FS_BLOCK_SIZE, part_name);
		break;
	case FS_EXT3:
		sprintf(cmd, "%s -n 1 -z 83 %s", CMD_FDISK, dev_name);
		sprintf(mkfs_cmd, "%s -j -m0 -b %d -F %s 2>/dev/null 1>/dev/null", CMD_MKE2FS, FS_BLOCK_SIZE, part_name);
		break;
	default:
		fprintf(stderr, "Error file system type(%d)\n", fs);
		ret = ERROR_FAIL;
		goto error;
	}

	ret = system(cmd);
	if(ret!=0){
		ret = ERROR_FORMAT_FAIL;
		goto error;
	}
	ret = system(mkfs_cmd);
	if(ret!=0){
		ret = ERROR_FORMAT_FAIL;
		goto error;
	}
	
error:
	ret = Mount_QNAP_Disk(disk_no, ROOT_PART, fs);
	if(ret==0){
		char buf[32];
		sprintf(buf, "%d", fs);
		WritePrivateProfileString(disk_section, FS_FIELD, buf, STORAGE_CONF);
		Set_Disk_Status(disk_no, S_READY);
		{
			char mnt_dir[32];
			Get_QNAP_Mount_Name(dev_name,mnt_dir,ROOT_PART);
			sprintf(buf, "%lld", Get_Free_Disk_Size(mnt_dir));
			WritePrivateProfileString(disk_section, F_CAPACITY_FIELD, buf, STORAGE_CONF);
			Create_Storage_Share(share_name, SHARE_EVERYONE, dev_name);
		}
	}
	else
		Set_Disk_Status(disk_no, S_NOT_MOUNTED);
	
	return ret;
}

int set_mode(char *dev_name, int mode){
	char mnt_dir[HD_DEVICE_NAME_LENGTH];
	char path[128], buf[BUF_SIZE];
	int fd;

	Get_QNAP_Mount_Name(dev_name,mnt_dir,ROOT_PART);
	sprintf(path, "%s%s", mnt_dir, HD_HIDDEN_CONF);
	fd=open(path, O_CREAT|O_RDWR);
	close(fd);
	sprintf(buf, "%d", mode);
	WritePrivateProfileString("", MODE_FIELD, buf, path);
	
	return 0;
}

int get_mode(char *dev_name){
	char mnt_dir[HD_DEVICE_NAME_LENGTH];
	char path[128], buf[BUF_SIZE];
	int fd;

	Get_QNAP_Mount_Name(dev_name,mnt_dir,ROOT_PART);
	sprintf(path, "%s%s", mnt_dir, HD_HIDDEN_CONF);	
//	printf("%s\n", path);
	fd=open(path, O_CREAT|O_RDWR);
	close(fd);
	GetPrivateProfileString("", MODE_FIELD, "1", buf, sizeof(buf), path);
	WritePrivateProfileString("", MODE_FIELD, buf, path);
	
	return atoi(buf);
}

/* Compare vendor and model to get host, channel, id, and lun*/
int TS_Get_Sg_Map(char *vendor, char *model, int* host, int* channel, int* id, int* lun){
	FILE *fp = NULL;
        char buf[BUF_SIZE];
	
	if ((fp = fopen("/proc/scsi/scsi", "r")) == NULL) {
		return ERROR_OPEN_FILE;
	}

	while (fgets(buf, sizeof(buf), fp) != NULL) {

		char tmp1[BUF_SIZE], tmp2[BUF_SIZE];

//printf("%s %s -- line %d\n", __FILE__, __FUNCTION__, __LINE__);
		if (Get_String_Field(buf, 1, ' ', tmp1, sizeof(tmp1)) != SUCCESS ||
                        strcasecmp(tmp1, "Host:") != 0)	{
			 continue;
		}
		
		if (host!=NULL && Get_String_Field(buf, 2, ' ', tmp1, sizeof(tmp1)) == SUCCESS &&
                        strlen(tmp1) > 4 && strncasecmp(tmp1, "scsi", 4) == 0){
                        *host = atoi((char*)(tmp1+4));

//		printf("Host: %d\n", *host);
                }
		if (channel!=NULL && Get_String_Field(buf, 4, ' ', tmp1, sizeof(tmp1)) == SUCCESS){
                        *channel = atoi(tmp1);
//printf("Channel: %d\n", *channel);
                }
                if (id != NULL && Get_String_Field(buf, 6, ' ', tmp1, sizeof(tmp1)) == SUCCESS){
                        *id = atoi(tmp1);
//printf("id: %d\n", *id);
                }
                if (lun != NULL && Get_String_Field(buf, 8, ' ', tmp1, sizeof(tmp1)) == SUCCESS){
//printf("lun: %d\n", *lun);
                        *lun= atoi(tmp1);
                }		

		if (fgets(buf, sizeof(buf), fp) == NULL){
			break;
		}

		{
			strncpy(tmp1, buf+10, 9);
			*(tmp1+10)='\0';
			strncpy(tmp2, buf+26, 17);
			*(tmp2+17)='\0';
			if(strstr(tmp1, vendor) && strstr(tmp2, model)){
//printf("Vendor: %s\n Model: %s host: %d, Channel: %d, id: %d, lun: %d\n", tmp1, tmp2, *host, *channel, *id, *lun);
				fclose(fp);
				return SUCCESS;
			}
			continue;
		}
	}
	fclose(fp);
	return ERROR_FAIL;
}

void disconnect_samba_share_by_name(char *dev_name){
	char share_dir[PARTITION_NAME_LENGTH];
	char disk_section[32], buf[32], dev_buf[HD_DEVICE_NAME_LENGTH];
	int idx=0, avail=0;

	GetPrivateProfileString(STORAGE_GLOBAL_SECTION, AVAIL_FIELD, "1", buf, sizeof(buf), STORAGE_CONF);
	avail = atoi(buf);
	for(idx=1;idx<avail+1;idx++){
		sprintf(disk_section, "VOLUME %d", idx);
		GetPrivateProfileString(disk_section, DEVICE_FIELD, "", dev_buf, sizeof(dev_buf), STORAGE_CONF);
//		printf("%s %s\n", dev_buf, dev_name);
		if(!strcmp(dev_buf, dev_name)){
			GetPrivateProfileString(disk_section, SHARE_FIELD, "", share_dir, sizeof(share_dir), STORAGE_CONF);
			break;
		}
	}
	if(idx>avail)
		strcpy(share_dir, "");
	Remove_Connection_For_NAS_Share(share_dir);
}

int sata_action(int host, int plugged_type){
	char dev_name[HD_DEVICE_NAME_LENGTH];
	switch(host){
	case 1:
		sprintf(dev_name, "/dev/sda");
		break;
	case 2:	
		sprintf(dev_name, "/dev/sdb");
		break;
	default:
		break;
	}
	if(plugged_type==PLUG_IN){
		disk_opt(host, "add");
		Add_Dev(dev_name);
	}
	else{
		//Need to stop some services
		disconnect_samba_share_by_name(dev_name);
                system("/etc/init.d/ftp.sh stop 1>/dev/null 2>/dev/null");
		//Stop_And_Umount_By_Dev(dev_name);
		Remove_Dev(dev_name);
		disk_opt(host, "remove");
		system("/etc/init.d/ftp.sh start 1>/dev/null 2>/dev/null");
	}
	return 0;
}

static void Dev_Action(char *dev_name, int plugged_type){
	if(plugged_type==PLUG_IN)
		Add_Dev(dev_name);
	else{
		//Need to stop some services
		disconnect_samba_share_by_name(dev_name);
		system("/etc/init.d/ftp.sh stop 1>/dev/null 2>/dev/null");
		//Stop_And_Umount_By_Dev(dev_name);
		Remove_Dev(dev_name);
		system("/etc/init.d/ftp.sh start 1>/dev/null 2>/dev/null");
	}
}
static int action(int port, int hex_value, int hex_value_of_change, int plugged_type){
	
	int idx=0, idx1=0;
	char dev_name[HD_DEVICE_NAME_LENGTH];
	for(idx=0; idx<MAX_PORT+1; idx++){
		if(hex_value_of_change&hub_port[idx]){
			if(idx==0){
				if(plugged_type == PLUG_OUT){
					sprintf(dev_name, "/dev/sd%c", dev_map[port+2]);
					disconnect_samba_share_by_name(dev_name);
                                        system("/etc/init.d/ftp.sh stop 1>/dev/null 2>/dev/null");
				}
				for(idx1=0;idx1<5;idx1++){
					sprintf(dev_name, "/dev/sd%c", dev_map[port+2]+idx1);	
					if(plugged_type == PLUG_IN){
						if(check_usb(dev_name)!=S_NOT_EXIST){
							printf("%s %s %s has been %s\n", port==FRONT?"Front:":port==B_UPPER?"Back upper":"Back lower", hub_msg[idx], dev_name, plugged_type==PLUG_IN?"plug-in":"plug-out");
							Dev_Action(dev_name, plugged_type);
						}
					}
					else{
        				        //Stop_And_Umount_By_Dev(dev_name);
                				Remove_Dev(dev_name);
					}
				}
				if(plugged_type == PLUG_OUT)
					system("/etc/init.d/ftp.sh start 1>/dev/null 2>/dev/null");
			}
			else{
				sprintf(dev_name, "/dev/sd%c", dev_map[port+2]+(idx-1));
				Dev_Action(dev_name, plugged_type);
			printf("%s %s %s has been %s\n", port==FRONT?"Front:":port==B_UPPER?"Back upper":"Back lower", hub_msg[idx], dev_name, plugged_type==PLUG_IN?"plug-in":"plug-out");
			}
		}
	}
	return 0;
}

int Is_USB_Host_Exist(int usb_host_no){
	int fd=0, size=0, idx=0;
	char buf[32];
	char *delim = " ";
	int value=0, ret=0;
	char *p=NULL;
	switch(usb_host_no){
	case 1:
	case 2:
	case 3:
		fd=open(USB_PROC_FOR_HOTSWAP, O_RDONLY);
		if(fd<0)
                        return ERROR_FAIL;
		size = read(fd, buf, sizeof(buf));
		close(fd);
		buf[size]='\0';
//		printf("Buffer = %s, size= %d\n", buf, size);
		if((p=strtok(buf, delim))==NULL)
                        return ERROR_FAIL;
		while(idx<usb_host_no+2 && (p=strtok(NULL, delim)))
			idx++;
		value= atox(p);
//		printf("value : %x\n", value);
		if(value&USB_PORT){
			ret=1;
		}
		else
			ret=0;
		break;
	default:
		//Something wrong here
		break;
	}
	return ret;
}

int Plug_Dev_Action(int port, int host, int plugged_type){
	
	int fd=0, size=0, idx=0;
	char buf[32];
	char *delim = " ";
	char *p=NULL;
	int value=0, changed_value=0;

	switch(host){
	case 1:
	case 2:
		sata_action(host, plugged_type);
		break;
	case 3:
	case 4:
	case 5:
		fd=open(USB_PROC, O_RDONLY);
		if(fd<0)
			return ERROR_FAIL;
		size = read(fd, buf, sizeof(buf));
		close(fd);
		buf[size]='\0';
//		printf("Buffer = %s, size= %d\n", buf, size);
		if((p=strtok(buf, delim))==NULL)	//get first field
			return ERROR_FAIL;
		if(host==3)
			changed_value = atox(p);	//transfer first field
		while(idx<host && (p=strtok(NULL, delim))){
			//Get the changed devices
			if((idx+4)==host)
				changed_value = atox(p);	//transfer second or third field
			idx++;
		}
		//Get the ready devices
		value= atox(p);	//Not implement. no meaning
		
		action(port, value, changed_value, plugged_type);
		break;
	default:
		//Something wrong here
		break;
	}
	
	return 0;
}

int TS_Eject_Disk(int disk_no){
        char disk_section[64], dev_name[HD_DEVICE_NAME_LENGTH];
        int ret = -1;

        sprintf(disk_section, "VOLUME %d", disk_no);
        GetPrivateProfileString(disk_section, DEVICE_FIELD, "",  dev_name, sizeof(dev_name), STORAGE_CONF);
        if(!strcmp(dev_name, "/dev/sda")){
                ret = sata_action(1, PLUG_OUT);
        }
        else if(!strcmp(dev_name, "/dev/sdb")){
                ret = sata_action(2, PLUG_OUT);
        }
	else {
		if(!strcmp(dev_name, "/dev/sdd") && Get_Profile_Integer("System", "usb copying", 0) )
			system("kill -9 `cat /var/run/rsync-usbcopy.pid`");
//		disconnect_samba_share_by_name(dev_name);
		system("/etc/init.d/smb.sh stop 1>/dev/null 2>/dev/null");
		system("/etc/init.d/ftp.sh stop 1>/dev/null 2>/dev/null");
                //ret = Stop_And_Umount_By_Dev(dev_name);
                ret = Remove_Dev(dev_name);
                system("/etc/init.d/ftp.sh start 1>/dev/null 2>/dev/null");
	}
	Pic_Send_Message(QNAP_PIC_USB_COPY, QNAP_PIC_EVENT_OFF);
        return ret;
}

int Stop_Format(int vol_no){
	char cmd[128];
	int retry=20;
	int ret=256, ret1=256;
	char disk_section[256], share_name[256];
	
	Set_Disk_Status(vol_no, S_STOPPING);

//I can't use the absoluted path of fdisk and mke2fs to stop those process
	while(retry>0 && ret!=0 && ret1!=0){
		ret = system("/sbin/pidof fdisk 1>/dev/null 2>/dev/null");
		ret1 = system("/sbin/pidof mke2fs 1>/dev/null 2>/dev/null");
		retry--;
//		printf("retry = %d, ret = %d\n", retry, ret);
		sleep(1);
	}
	sprintf(cmd, "/bin/kill -9 `/sbin/pidof fdisk` 2>/dev/null 1>/dev/null");
	system(cmd);
	sleep(1);
	system(cmd);
	sleep(1);
	sprintf(cmd, "/bin/kill -9 `/sbin/pidof mke2fs` 2>/dev/null 1>/dev/null");
	ret = system(cmd);
	if(vol_no == 1){
		sprintf(cmd, "/sbin/write_log \"[SATA Disk] User stop formatting.\" 2");
	}
	else{
		sprintf(disk_section, "VOLUME %d", vol_no);
		GetPrivateProfileString(disk_section, SHARE_FIELD, "", share_name, sizeof(share_name), STORAGE_CONF);
		sprintf(cmd, "/sbin/write_log \"[%s] User stop formatting.\" 2", share_name);
	}
	system(cmd);
	return ret;
}

int Stop_Check(int vol_no){
	char cmd[128];
        int retry=20;
        int ret=256;
	char disk_section[256], share_name[256];

	Set_Disk_Status(vol_no, S_STOPPING);

	while(retry>0 && ret!=0){
                ret = system("/sbin/pidof e2fsck 1>/dev/null 2>/dev/null");
                retry--;
//                printf("retry = %d, ret = %d\n", retry, ret);
                sleep(1);
        }
	sprintf(cmd, "/bin/kill -9 `/sbin/pidof e2fsck` 2>/dev/null 1>/dev/null");
	
	ret = system(cmd);
	if(vol_no == 1){
		sprintf(cmd, "/sbin/write_log \"[SATA Disk] User stop exmining.\" 2");
	}
	else{
		sprintf(disk_section, "VOLUME %d", vol_no);
        	GetPrivateProfileString(disk_section, SHARE_FIELD, "", share_name, sizeof(share_name), STORAGE_CONF);
		sprintf(cmd, "/sbin/write_log \"[%s] User stop exmining.\" 2", share_name);
	}
	system(cmd);
	return ret;
}

//This AP do not try to mount HD to solve HD standby problem
long Get_Available_Disk_Size( char *filesys_path )
{
        struct statfs fsd;
        long usage_size;

        if (statfs(filesys_path,&fsd) == 0){
                double tmp_bsize, tmp_bavail;
                tmp_bavail = fsd.f_bavail;    /** free blocks for non-superuser **/
                tmp_bsize  = fsd.f_bsize;

                usage_size = (tmp_bavail * tmp_bsize)/ 1024;

        }
        return usage_size;
}


void Get_TS_SATA_Model(char *model, int size){
	char vendor[32], product[32], rev[32];
	char buf[256];

	vendor[0]=0x0;
	product[0]=0x0;
	rev[0]=0x0;
	Get_Sg_Info("/dev/sg0", vendor, product, rev);
	sprintf(buf, "%s%s%s", vendor, product, rev);
	if(size>=sizeof(buf))
		sprintf(model, "%s", buf);
	else
		strncpy(model, buf, size);
}

int Get_Badblocks_Status(int dev_no)
{
	int ret;
	ret=Get_Private_Profile_Integer(BAD_BLACK_SECTION, "Status", BADBLOCKS_STOP, BADBLOCKS_CONF_PATH);
	return ret;
}

int Get_Badblocks_Progress(int dev_no)
{
	int ret;
	ret=Get_Private_Profile_Integer(BAD_BLACK_SECTION, "Progress", 0, BADBLOCKS_CONF_PATH);
	return ret;
}

int Get_Badblocks_Num(int dev_no)
{
	int ret;
	ret=Get_Private_Profile_Integer(BAD_BLACK_SECTION, "BB_count", 0, BADBLOCKS_CONF_PATH);
	return ret;
}

int Set_Badblocks_Integer(int dev_no, char *field, int value)
{
	int ret;
	ret=Set_Private_Profile_Integer(BAD_BLACK_SECTION, field, value, BADBLOCKS_CONF_PATH);
	return ret;
}

int Set_Badblocks_Count(int dev_no, char *field, int value)
{
	int ret;
	char log_path[256];
	char buf[256]="A";
	buf[0]='A'+dev_no-1;
	sprintf(log_path, BADBLOCKS_LOG_PATH, buf);
	Set_Private_Profile_Integer("Bad Block", field, value, log_path);
	return ret;
}

int Badblocks_Start(int dev_no)
{
	int status;
	char str[256]="/sbin/badblocks -s /dev/sda";
	
	if (Get_Disk_Status(1) == 0) {
		status=Get_Badblocks_Status(1);
		if(status==0) {
			Set_Private_Profile_Integer(BAD_BLACK_SECTION, "Status", BADBLOCKS_SCANNING, BADBLOCKS_CONF_PATH);
			system(str);
		}
	}
	else
		return -1;
	
	return SUCCESS;
}

int Badblocks_Stop(int dev_no)
{
	int pid=0;
//	int count=0;
//	char buffer[256]="";

	pid = Get_Private_Profile_Integer(BAD_BLACK_SECTION, "Pid", 0, BADBLOCKS_CONF_PATH);
	kill(pid, SIGUSR1);
	Set_Private_Profile_Integer(BAD_BLACK_SECTION, "Status", BADBLOCKS_STOP, BADBLOCKS_CONF_PATH);
	
	/*count=Get_Badblocks_Num();
	
	if (count > 0) {
		sprintf(buffer, "Disk bad blocks scanning stopped. [ %d ] bad block(s) found.", count);
		Write_Log(buffer, EVENTLOG_ERROR_TYPE);
	}
	else {
		sprintf(buffer, "Disk bad blocks scanning stopped. No bad blocks are found.");
		Write_Log(buffer, EVENTLOG_INFORMATION_TYPE);
	}*/
	
	return SUCCESS;
}

