//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		quota.c
//
//	Abstract:
//		Quota Functions.
//
//	FUNCTIONS:	TBD.
//
//
//	HISTORY:
//
//	10/08/01	Kevin Liao created
//	07/17/02	Catherine -- NASWare 3.0
//	09/12/02	Catherine -- add a function, get_quota_format,
//				to change quota format if necessary
//
//**************************************************************************

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <grp.h>   //for getgrnam
#include <sys/types.h>
#include <ctype.h>
#include "uLinux.h"
#include "nas_quota.h"
#include "naslvm.h"	// LVM Support
#include "NAS.h"

// Catherine 2002/09/12
int get_quota_format(char* format, int buf_size)
{
	return Get_Profile_String("System",
		"Quota Type", "vfsold",
		format, buf_size);
}

// Catherine added 2002/07/17 common functions ==>
int get_volume_mp_string(int vol_no, char* mp_str)
{
	int ret;
	LVM_VOLUME_CONF vol;

	if ((ret = Get_One_LVM_Volume_Conf(vol_no, &vol)) == SUCCESS)
		ret = Get_MP_String(vol.drive_no_list[0], DATA_PART, mp_str);

	return ret;
}

BOOL is_volume_exist(int vol_no)
{
	LVM_VOLUME_CONF vol;

	if (Get_One_LVM_Volume_Conf(vol_no, &vol) == SUCCESS)
		return TRUE;

	return FALSE;
}

double convert_to_MB(char* str)
{
	char *p=str;
	char number[BUF_SIZE], unit[10];
	double result=0;

	if (p==NULL || *p=='\0') return FALSE;

	while (p && *p) {
		if (!isdigit((int)(*p)))
			break;
		p++;
	}
	strncpy(number, str, (int)(p-str));
	number[(int)(p-str)]='\0';
	if (*p) {
		strcpy(unit, p);
		Remove_Blank_From_String(unit);
		if (unit[strlen(unit)-1] == '*')
			unit[strlen(unit)-1] = 0x0;
	}
	else strcpy(unit, "");

	if (strcmp(unit, "") == 0) // no unit
		result = atof(number) / 1024;
	else if (strcasecmp(unit, "T")==0)
		result = atof(number) * 1048576;
	else if (strcasecmp(unit, "G")==0)
		result = atof(number) * 1024;
	else if (strcasecmp(unit, "M")==0)
		result = atof(number);

	return result;
}
// <== Catherine added 2002/07/17 common functions

//Get volume number from device
int get_volume_from_dev(char* dev)
{
	LVM_VOLUME_CONF *vol_list=NULL;
	int i, cnt, vol_no;

	cnt = Get_All_LVM_Volumes_Conf(&vol_list);
	for(i=0;i<cnt;i++){
		if((!strcmp(vol_list[i].device_name, dev)) ||
			(!strcmp(vol_list[i].lv_name, dev)))
		{
			vol_no = vol_list[i].vol_no;
			Release_List(vol_list);
			return vol_no;
		}
	}
	if (cnt>0) Release_List(vol_list);
	return ERROR_FAIL;
}

int get_system_group_id(char* name)
{
	struct group* grp;
	if (!(grp = getgrnam(name)))
		return ERROR_BAD_USERNAME;
	return grp->gr_gid;
}

int Stop_User_Quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	int ret = SUCCESS;

	if (volume == 0){
		if(system("/sbin/quotaoff -au 2> /dev/null"))
			ret = ERROR_FAIL;
	}else {
		if ((ret = get_volume_mp_string(volume, tmp))==SUCCESS) {
			sprintf(buf, "/sbin/quotaoff -u %s 2> /dev/null", tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;
}

int Start_User_Quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	int ret = SUCCESS;

	if (volume == 0){
		if(system("/sbin/quotaon -au 2> /dev/null"))
			ret = ERROR_FAIL;
	}else {
		if((ret = get_volume_mp_string(volume, tmp)) == SUCCESS){
			sprintf(buf, "/sbin/quotaon -u %s 2> /dev/null", tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;
}

int Stop_Group_Quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	int ret = SUCCESS;

	if (volume == 0){
		if(system("/sbin/quotaoff -ag 2> /dev/null"))
			ret = ERROR_FAIL;
	}else {
		if((ret = get_volume_mp_string(volume, tmp)) == SUCCESS){
			sprintf(buf, "/sbin/quotaoff -g %s 2> /dev/null", tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;
}

int Start_Group_Quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	int ret = SUCCESS;

	if (volume == 0){
		if(system("/sbin/quotaon -ag 2> /dev/null"))
			ret = ERROR_FAIL;
	}else {
		if((ret = get_volume_mp_string(volume, tmp)) == SUCCESS){
			sprintf(buf, "/sbin/quotaon -g %s 2> /dev/null", tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;
}

int Check_User_Quota(int volume, int create)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], param[BUF_SIZE] = "-uM";
	int ret = SUCCESS;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if (create != 0)
		strcat(param, "c");
	if (volume == 0){
		sprintf(buf, "/sbin/quotacheck %s -F %s -a 2> /dev/null", param, qformat);
		if(system(buf))
			ret = ERROR_FAIL;
	}else {
		if((ret = get_volume_mp_string(volume, tmp)) == SUCCESS){
			sprintf(buf, "/sbin/quotacheck %s -F %s %s 2> /dev/null", param, qformat, tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;
}

int Check_Group_Quota(int volume, int create)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], param[BUF_SIZE] = "-gM";
	int ret = SUCCESS;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if (create != 0)
		strcat(param, "c");
	if (volume == 0){
		sprintf(buf, "/sbin/quotacheck %s -F %s -a 2> /dev/null", param, qformat);
		if (system(buf))
			ret = ERROR_FAIL;
	}else {
		if((ret = get_volume_mp_string(volume, tmp)) == SUCCESS){
			sprintf(buf, "/sbin/quotacheck %s -F %s %s 2> /dev/null", param, qformat, tmp);
			if(system(buf))
				ret = ERROR_FAIL;
		}
	}
	return ret;	
}

int m_datacmp(const void *a, const void *b)
{
	QUOTA_STATUS_USER *c,*d;
	LVM_VOLUME_CONF aconf, bconf;

	c = (QUOTA_STATUS_USER *) a;
	d = (QUOTA_STATUS_USER *) b;
/* Catherine 2003/04/15
	if (c->vol > d->vol)
		return 1;
	else if (c->vol == d->vol)
		return 0;
	else
		return -1;	
*/
	
	Get_One_LVM_Volume_Conf(c->vol, &aconf);
	Get_One_LVM_Volume_Conf(d->vol, &bconf);

	if(aconf.raid_level > bconf.raid_level ) return 1;
	if(aconf.raid_level < bconf.raid_level ) return -1;
	if(aconf.drive_no_list[0] > bconf.drive_no_list[0] ) return 1;
	if(aconf.drive_no_list[0] < bconf.drive_no_list[0] ) return -1;
	if(aconf.drive_no_list[0] == bconf.drive_no_list[0] ) return 0;

	return -1;
}

int m_vol_datacmp(const void *a, const void *b)
{
	QUOTA_STATUS_VOLUME *c,*d;

	c = (QUOTA_STATUS_VOLUME *) a;
	d = (QUOTA_STATUS_VOLUME *) b;
	return strcasecmp(c->name, d->name);
}

/*
#include <stdarg.h>
void DebugOutput(char *msg, ...)
{
	FILE *dout;
	char buf[2048];
	va_list args;

	va_start(args, msg);
	vsnprintf(buf, sizeof(buf)-1, msg, args);
	va_end(args);

	dout = fopen("/home/httpd/cgi-bin/cgi.debug", "a");
	fputs(buf, dout);
	fclose(dout);
}
*/

BOOL is_unit(char* pstr)
{
	int len;
	char c;

	if (pstr == NULL) return FALSE;

	if ((len = strlen(pstr))>2 || len == 0) return FALSE;

	c = *pstr;
	if (c != 'K' && c != 'M' && c !='G')
		return FALSE;
	if (len == 2) {
		c = *(pstr+1);
		if (c != '*') return FALSE;
	}
	return TRUE;
}

BOOL is_numeric_string(char* str)
{
	int i;
	int len = strlen(str);
	char* p = str;

	if (str == NULL) return FALSE;

	if ( p[len-1] == '*' ) len--;

	for (i=0; i<len; i++) {
		if (strlen(p)<=2 && is_unit(p))
			return TRUE;

		if (!isdigit(*p++))
			return FALSE;
	}
	return TRUE;
}

int Get_One_Volume_User_Quota(int volume, QUOTA_STATUS_VOLUME *qsv, int count)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE] , mp[BUF_SIZE];
	FILE *fp;
	int tmp_cnt = 0, found = 0, ret;
	char qformat[BUF_SIZE];
// Catherine 2003/05/27 for user name might include blank characters ==>
	char *sp = NULL;
// <== Catherine 2003/05/27

	get_quota_format(qformat, BUF_SIZE);

	if((ret = get_volume_mp_string(volume, mp)) != SUCCESS)
		return ret;

	sprintf(tmp, "/sbin/repquota -F %s -uv %s > %s", qformat, mp, SZ_VOLQUOTAREPORT_PATH);
	system(tmp);
	if (((fp = fopen(SZ_VOLQUOTAREPORT_PATH, "r")) == NULL) && errno == ENOENT)
		return ERROR_OPEN_FILE;
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (strstr(buf, "---")) {
			found = 1;
			break;
		}
	}
	if (found == 0) {
		fclose(fp);
		return 0;
	}

	// vfsold -> vfsv0 format change
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (buf[0] == '\n')
			continue;
		if (strncasecmp(buf, "Statistics:", 11)==0)
			break;

		if ((tmp_cnt + 1 > count) && (qsv != NULL)) {
			fclose(fp);
			return ERROR_BUFFER_TOO_SMALL;
		}
// Catherine 2003/05/27 for user name might include blank characters ==>
		if ((sp = strstr(buf, "--")) == NULL &&
			(sp = strstr(buf, "+-")) == NULL)
			continue;

		// tmp buffer size enough?
		if ((int)(sp - buf) > sizeof(tmp) - 1)
			continue;

		strncpy(tmp, buf, (int)(sp - buf));
		tmp[(int)(sp-buf)] = 0x0;
		Remove_Blank_From_String(tmp);
		if (Get_Exact_System_User_Name(tmp) < 0)
			continue;

		if (strcasecmp(tmp, "root") == 0)
			continue;
		if (qsv != NULL) {
			char *limit_start = sp + 2;

			while (*limit_start == ' ') limit_start++;

			strcpy(qsv[tmp_cnt].name, tmp);
			Get_String_Field(limit_start, 1, ' ', tmp, sizeof(tmp));
			qsv[tmp_cnt].b_used = convert_to_MB(tmp);
			Get_String_Field(limit_start, 2, ' ', tmp, sizeof(tmp));
			qsv[tmp_cnt].b_limit = convert_to_MB(tmp);
		}
		tmp_cnt++;
	}

	fclose(fp);
	if (qsv != NULL)
		qsort((void *) qsv, tmp_cnt, sizeof(QUOTA_STATUS_VOLUME), m_vol_datacmp);
	return tmp_cnt;
}

int Get_One_Volume_Group_Quota(int volume, QUOTA_STATUS_VOLUME *qsv, int count)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], mp[BUF_SIZE];
	FILE *fp;
	int tmp_cnt = 0, found = 0, ret;
	char qformat[BUF_SIZE];
// Catherine 2003/05/27 for user name might include blank characters ==>
	char *sp = NULL;
// <== Catherine 2003/05/27

	get_quota_format(qformat, BUF_SIZE);

	if((ret = get_volume_mp_string(volume, mp)) != SUCCESS)
		return ret;

	sprintf(tmp, "/sbin/repquota -F %s -gv %s > %s", qformat, mp, SZ_VOLQUOTAREPORT_PATH);
	system(tmp);
	if (((fp = fopen(SZ_VOLQUOTAREPORT_PATH, "r")) == NULL) && errno == ENOENT)
		return ERROR_OPEN_FILE;
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (strstr(buf, "---")) {
			found = 1;
			break;
		}
	}
	if (found == 0) {
		fclose(fp);
		return 0;
	}

	// vfsold -> vfsv0 format changes
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (buf[0] == '\n')
			continue;
		if (strncasecmp(buf, "Statistics:", 11)==0)
			break;

		if ((tmp_cnt + 1 > count) && (qsv != NULL)) {
			fclose(fp);
			return ERROR_BUFFER_TOO_SMALL;
		}

// Catherine 2003/05/27 for user name might include blank characters ==>
		if ((sp = strstr(buf, "--")) == NULL &&
			(sp = strstr(buf, "+-")) == NULL)
			continue;

		// tmp buffer size enough?
		if ((int)(sp - buf) > sizeof(tmp) - 1)
			continue;

		strncpy(tmp, buf, (int)(sp - buf));
		tmp[(int)(sp-buf)] = 0x0;
		Remove_Blank_From_String(tmp);

		if (Get_Exact_System_Group_Name(tmp) < 0)
			continue;
// <== Catherine 2003/05/27

		if (qsv != NULL) {
// Catherine 2003/05/27 for user name might include blank characters ==>
			char *limit_start = sp + 2;

			strcpy(qsv[tmp_cnt].name, tmp);

			while (*limit_start == ' ') limit_start++;

			Get_String_Field(limit_start, 1, ' ', tmp, sizeof(tmp));
			qsv[tmp_cnt].b_used = convert_to_MB(tmp);
			Get_String_Field(limit_start, 2, ' ', tmp, sizeof(tmp));
			qsv[tmp_cnt].b_limit = convert_to_MB(tmp);
// <== Catherine 2003/05/27		
		}
		tmp_cnt++;
	}
	fclose(fp);
	if (qsv != NULL)
		qsort((void *) qsv, tmp_cnt, sizeof(QUOTA_STATUS_VOLUME), m_vol_datacmp);
	return tmp_cnt;
}

int Get_One_User_Quota(char *user_name, QUOTA_STATUS_USER *qsu, int count)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE] ,uid[128];
	FILE *fp;
	int tmp_cnt = 0, found = 0;
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);

	if (user_name == NULL)
		return ERROR_NOT_EXISTED;

	//Get_Exact_NAS_User_Name(user_name);
	if (!Is_System_User(user_name))
		return ERROR_NOT_EXISTED;

	Get_Exact_NAS_User_Name(user_name);

	sprintf(uid, "%d", (Get_System_User_ID(user_name)));	
	sprintf(tmp, "/sbin/quota -F %s -uv \"%s\" > %s", qformat, uid, SZ_USRQUOTAREPORT_PATH);
	system(tmp);

	if (((fp = fopen(SZ_USRQUOTAREPORT_PATH, "r")) == NULL) && errno == ENOENT)
		return ERROR_OPEN_FILE;

	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (strstr(buf, "Filesystem")) {
			found = 1;
			break;
		}
	}
	if (found == 0) {
		fclose(fp);
		return 0;
	}
	found = 0;
	// vfsold -> vfsv0 format changes
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (buf[0] == '\n')
			continue;

		if ((tmp_cnt + 1 > count) && (qsu != NULL)) {
			fclose(fp);
			return ERROR_BUFFER_TOO_SMALL;
		}

		if (Get_String_Field(buf, 1, ' ', tmp, sizeof(tmp))<0)
			continue;

		if (found>0 &&
			is_numeric_string(tmp))
		{
			if (count>0 && qsu != NULL) {
				// the previous line contains only the volume device name
				// this line contains only the numberic parts
				qsu[tmp_cnt].b_used = convert_to_MB(tmp);
				Get_String_Field(buf, 2, ' ', tmp, sizeof(tmp));
				qsu[tmp_cnt].b_limit = convert_to_MB(tmp);
			}
		}
		else {
			found++;
			if (count>0 && qsu != NULL) {
				//qsu[tmp_cnt].vol = get_volume_from_dev(tmp);
				qsu[tmp_cnt].vol = Get_Drive_No(tmp);
				if (qsu[tmp_cnt].vol < 0) {
					found = 0;
					continue;
				}
			}
			if (Get_String_Field(buf, 2, ' ', tmp, sizeof(tmp))<0)
				continue;
			if (count>0 && qsu != NULL) {
				qsu[tmp_cnt].b_used = convert_to_MB(tmp);
				Get_String_Field(buf, 3, ' ', tmp, sizeof(tmp));
				qsu[tmp_cnt].b_limit = convert_to_MB(tmp);
			}
		}
		found = 0;
		tmp_cnt++;
	}
	fclose(fp);

	if (count>0 && qsu != NULL)
		qsort((void *) qsu, tmp_cnt, sizeof(QUOTA_STATUS_USER), m_datacmp);
	return tmp_cnt;
}

int Get_One_Group_Quota(char *grp_name, QUOTA_STATUS_USER *qsu, int count)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], gid[128];
	FILE *fp;
	int tmp_cnt = 0, found = 0;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if (grp_name == NULL)
		return ERROR_NOT_EXISTED;
	//Get_Exact_NAS_Group_Name(grp_name);
	if (!Is_System_Group(grp_name))
		return ERROR_NOT_EXISTED;
	Get_Exact_NAS_Group_Name(grp_name);
	sprintf(gid, "%d", (get_system_group_id(grp_name)));	
	sprintf(tmp, "/sbin/quota -F %s -gvs \"%s\" > %s", qformat, gid, SZ_GRPQUOTAREPORT_PATH);
	system(tmp);
	if (((fp = fopen(SZ_GRPQUOTAREPORT_PATH, "r")) == NULL) && errno == ENOENT)
		return ERROR_OPEN_FILE;
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (strstr(buf, "Filesystem")) {
			found = 1;
			break;
		}
	}
	if (found == 0)
		return 0;

	// vfsold -> vfsv0 format changes
	while (fgets(buf, BUF_SIZE, fp) != NULL) {
		if (buf[0] == '\n')
			continue;
		if ((tmp_cnt + 1 > count) && (qsu != NULL)) {
			fclose(fp);
			return ERROR_BUFFER_TOO_SMALL;
		}
		if (qsu != NULL) {
			Get_String_Field(buf, 1, ' ', tmp, sizeof(tmp));
			qsu[tmp_cnt].vol = get_volume_from_dev(tmp);
			Get_String_Field(buf, 2, ' ', tmp, sizeof(tmp));
			qsu[tmp_cnt].b_used = convert_to_MB(tmp);
			Get_String_Field(buf, 3, ' ', tmp, sizeof(tmp));
			qsu[tmp_cnt].b_limit = convert_to_MB(tmp);
		}
		tmp_cnt++;
	}
	fclose(fp);
	if (qsu != NULL)
		qsort((void *) qsu, tmp_cnt, sizeof(QUOTA_STATUS_USER), m_datacmp);
	return tmp_cnt;
}

int Delete_One_User_Quota(char *user_name)
{
	if (user_name == NULL)
		return -1;
	//Get_Exact_NAS_User_Name(user_name);
	if (!Is_System_User(user_name))
		return ERROR_NOT_EXISTED;
	return Conf_Remove_Section(SZ_USRQUOTACONF_PATH, user_name);
}

int Delete_One_Group_Quota(char *grp_name)
{
	if (grp_name == NULL)
		return -1;
	//Get_Exact_NAS_Group_Name(grp_name);
	if (!Is_System_Group(grp_name))
		return ERROR_NOT_EXISTED;
	return Conf_Remove_Section(SZ_GRPQUOTACONF_PATH, grp_name);
}

int Set_One_User_Quota(char *user_name, unsigned long quota_size, int volume, int write)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], uid[128];
	unsigned long quota = 0;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if (user_name == NULL || quota_size < 0)
		return -1;
	//Get_Exact_NAS_User_Name(user_name);
	if (!Is_System_User(user_name))
		return ERROR_NOT_EXISTED;
	if (quota_size > 0)
		quota = quota_size * 1024;
	if (write == 1) {
		sprintf(tmp, "%lu", quota);
		Conf_Set_Field(SZ_USRQUOTACONF_PATH, user_name, "Size", tmp);
	}
	Get_Exact_NAS_User_Name(user_name);	
	sprintf(uid, "%d", (Get_System_User_ID(user_name)));	
	if (volume == 0)
		sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" %lu %lu 0 0 -a 2> /dev/null", qformat, uid, quota, quota);
	else {
		if(get_volume_mp_string(volume, buf)== SUCCESS){
			sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" %lu %lu 0 0 %s 2> /dev/null", qformat, uid, quota, quota, buf);	
		}		
	}
	if(system(tmp))
		return ERROR_FAIL;
	return 0;
}

int Set_One_Group_Quota(char *grp_name, unsigned long quota_size, int volume, int write)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], gid[128];
	unsigned long quota = 0;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine
	if (grp_name == NULL || quota_size < 0)
		return -1;
	//Get_Exact_NAS_Group_Name(grp_name);
	if (!Is_System_Group(grp_name))
		return ERROR_NOT_EXISTED;
	if (quota_size > 0)
		quota = quota_size * 1024;
	if (write == 1) {
		sprintf(tmp, "%lu", quota);
		Conf_Set_Field(SZ_GRPQUOTACONF_PATH, grp_name, "Size", tmp);
	}
	Get_Exact_NAS_Group_Name(grp_name);
	sprintf(gid, "%d", (get_system_group_id(grp_name)));
	if (volume == 0)
		sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" %lu %lu 0 0 -a 2> /dev/null", qformat, gid, quota, quota);
	else {
		if(get_volume_mp_string(volume, buf)== SUCCESS){
			sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" %lu %lu 0 0 %s 2> /dev/null", qformat, gid, quota, quota, buf);		
		}
	}
	if(system(tmp))
		return ERROR_FAIL;
	return SUCCESS;
}

int set_all_user_quota(unsigned long quota_size, int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE] , uid[128];
	char user_list[MAX_USER_NUMBER][USER_NAME_LENGTH];
	int i, j;
	unsigned long quota;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if(volume != 0){
		if(get_volume_mp_string(volume, buf)!= SUCCESS)	
			return  -1;		
	}	
	i = Get_NAS_User_List(user_list, MAX_USER_NUMBER);
	if (i == ERROR_BUFFER_TOO_SMALL)
		return -1;
	if (i < MAX_USER_NUMBER)
		strcpy(user_list[i++], USER_GUEST);
	for (j = 0; j < i; j++) {
		//Get_Exact_NAS_User_Name(user_list[j]);
		sprintf(uid, "%d", Get_System_User_ID(user_list[j]));	
		if (volume == 0) {
			Conf_Remove_Section(SZ_USRQUOTACONF_PATH, user_list[j]);
			quota = quota_size * 1024;
			sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" %lu %lu 0 0 -a 2> /dev/null", qformat, uid, quota, quota);
		}
		else {
			if (Conf_Get_Field(SZ_USRQUOTACONF_PATH, user_list[j], "Size", tmp, sizeof(tmp)) == SUCCESS)
				quota = atol(tmp);
			else
				quota = quota_size * 1024;
			sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" %lu %lu 0 0 %s 2> /dev/null", qformat, uid, quota, quota, buf);
		}
		//Ricky modify 2005/1/17 skip setting user quota fail and write event log.
		if(system(tmp)){
			char log[1024];
			sprintf(log,"Setting user quota fail with user %s",user_list[j]);
			write_log(log,1);
		}
		//End
//			return ERROR_FAIL;
	}
	return SUCCESS;
}

int set_all_group_quota(unsigned long quota_size, int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], gid[128];
	char grp_list[MAX_GROUP_NUMBER][GROUP_NAME_LENGTH];
	int i, j;
	unsigned long quota;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if(volume != 0){
		if(get_volume_mp_string(volume, buf) != SUCCESS)	
			return  -1;		
	}	
	i = Get_NAS_Group_List(grp_list, MAX_GROUP_NUMBER);
	if (i == ERROR_BUFFER_TOO_SMALL)
		return -1;
	if (i < MAX_GROUP_NUMBER)
		strcpy(grp_list[i++], GROUP_GUEST);
	for (j = 0; j < i; j++) {
		//Get_Exact_NAS_Group_Name(grp_list[j]);
		sprintf(gid, "%d", get_system_group_id(grp_list[j]));
		if (volume == 0) {
			Conf_Remove_Section(SZ_GRPQUOTACONF_PATH, grp_list[j]);
			quota = quota_size * 1024;
			sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" %lu %lu 0 0 -a 2> /dev/null", qformat, gid, quota, quota);
		}
		else {
			if (Conf_Get_Field(SZ_GRPQUOTACONF_PATH, grp_list[j], "Size", tmp, sizeof(tmp)) == SUCCESS)
				quota = atol(tmp);
			else
				quota = quota_size * 1024;	
			sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" %lu %lu 0 0 %s 2> /dev/null", qformat, gid, quota, quota, buf);
			
		}
		if(system(tmp))
			return ERROR_FAIL;
	}
	return SUCCESS;
}

int clear_all_user_quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], uid[128];
	char user_list[MAX_USER_NUMBER][USER_NAME_LENGTH];
	int i, j;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if(volume != 0){
		if(get_volume_mp_string(volume, buf) != SUCCESS)	
			return  -1;		
	}	
	i = Get_NAS_User_List(user_list, MAX_USER_NUMBER);
	if (i == ERROR_BUFFER_TOO_SMALL)
		return -1;
	if (i < MAX_USER_NUMBER)
		strcpy(user_list[i++], USER_GUEST);
	for (j = 0; j < i; j++) {
		//Get_Exact_NAS_User_Name(user_list[j]);
		sprintf(uid, "%d", Get_System_User_ID(user_list[j]));	
		if (volume == 0)
			sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" 0 0 0 0 -a 2> /dev/null", qformat, uid);
		else
			sprintf(tmp, "/sbin/setquota -u -F %s \"%s\" 0 0 0 0 %s 2> /dev/null", qformat, uid, buf);

		system(tmp);
		Conf_Remove_Section(SZ_USRQUOTACONF_PATH, user_list[j]);
	}
	return SUCCESS;
}

int clear_all_group_quota(int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE], gid[128];
	char grp_list[MAX_GROUP_NUMBER][GROUP_NAME_LENGTH];
	int i, j;
// Catherine 2002/09/12 ==>
	char qformat[BUF_SIZE];

	get_quota_format(qformat, BUF_SIZE);
// <== Catherine

	if(volume != 0){
		if(get_volume_mp_string(volume, buf) != SUCCESS)	
			return  -1;		
	}	
	i = Get_NAS_Group_List(grp_list, MAX_GROUP_NUMBER);
	if (i == ERROR_BUFFER_TOO_SMALL)
		return -1;
	if (i < MAX_GROUP_NUMBER)
		strcpy(grp_list[i++], GROUP_GUEST);
	for (j = 0; j < i; j++) {
		//Get_Exact_NAS_Group_Name(grp_list[j]);
		sprintf(gid, "%d", (get_system_group_id(grp_list[j])));
		if (volume == 0)
			sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" 0 0 0 0 -a 2> /dev/null", qformat, gid);
		else 
			sprintf(tmp, "/sbin/setquota -g -F %s \"%s\" 0 0 0 0 %s 2> /dev/null", qformat, gid, buf);

		system(tmp);
		Conf_Remove_Section(SZ_GRPQUOTACONF_PATH, grp_list[j]);
	}
	return SUCCESS;
}

BOOL Is_User_Quota_Enabled(unsigned long *quota_size)
{
	char buf[BUF_SIZE];
	BOOL ret;

	ret = Get_Private_Profile_Boolean("", SZ_USERENABLE_FIELD, 0, SZ_USRQUOTACONF_PATH );
	if (ret && quota_size != NULL) {
		if(GetPrivateProfileString("", SZ_USERSIZE_FIELD, "1000000", buf, sizeof(buf), SZ_USRQUOTACONF_PATH) > 0)
			*quota_size = atol(buf) / 1024;
	}
	return ret;
}

BOOL Is_Group_Quota_Enabled(unsigned long *quota_size)
{
	char buf[BUF_SIZE];
	BOOL ret;
	
	ret = Get_Private_Profile_Boolean("", SZ_GROUPENABLE_FIELD, 0, SZ_GRPQUOTACONF_PATH );
	if (ret && quota_size != NULL) {
		if(GetPrivateProfileString("", SZ_GROUPSIZE_FIELD, "1000000", buf, sizeof(buf), SZ_GRPQUOTACONF_PATH ) > 0)
			*quota_size = atol(buf) / 1024;
	}
	return ret;
}

BOOL Is_User_Quota_Enabled_Disk(unsigned long *quota_size, int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	BOOL ret;
	
	GetPrivateProfileString("", SZ_SAMEQUOTA_FIELD, "TRUE", tmp, sizeof(tmp), SZ_USRQUOTACONF_PATH);
	if (strcasecmp(tmp, "TRUE") == 0)
		return Is_User_Quota_Enabled(quota_size);

	if (!is_volume_exist(volume))
		return FALSE;

	sprintf(buf, "volume%d %s", volume, SZ_USERENABLE_FIELD);
	
	ret=Get_Private_Profile_Boolean("", buf, 0,SZ_USRQUOTACONF_PATH);
	if (ret && quota_size != NULL) {
		sprintf(tmp, "volume%d %s", volume, SZ_GROUPSIZE_FIELD);
		if(GetPrivateProfileString("", tmp, "1000000", buf, sizeof(buf), SZ_USRQUOTACONF_PATH ) > 0)
		*quota_size = atol(buf) / 1024;
	}
	return ret;
}

BOOL Is_Group_Quota_Enabled_Disk(unsigned long *quota_size, int volume)
{
	char buf[BUF_SIZE], tmp[BUF_SIZE];
	BOOL ret;
	
	GetPrivateProfileString("", SZ_SAMEQUOTA_FIELD, "TRUE", tmp, sizeof(tmp), SZ_GRPQUOTACONF_PATH);
	if (strcasecmp(tmp, "TRUE") == 0)
		return Is_Group_Quota_Enabled(quota_size);

	if (!is_volume_exist(volume))
		return FALSE;
	
	sprintf(buf, "volume%d %s", volume, SZ_GROUPENABLE_FIELD);
	
	ret=Get_Private_Profile_Boolean("", buf, 0,SZ_GRPQUOTACONF_PATH);
	if (ret && quota_size != NULL) {
		sprintf(tmp, "volume%d %s", volume, SZ_GROUPSIZE_FIELD);
		if(GetPrivateProfileString("", tmp, "1000000", buf, sizeof(buf), SZ_GRPQUOTACONF_PATH ) > 0)
		*quota_size = atol(buf) / 1024;
	}
	return ret;
}

int Set_User_Quota(int bEnable, unsigned long quota_size)
{
	unsigned long quota;
	char buf[BUF_SIZE];

	if (bEnable && quota_size <= 0)
		return -1;
		
	if(!Set_Private_Profile_Boolean("", SZ_USERENABLE_FIELD, bEnable, SZ_USRQUOTACONF_PATH))
		return -1;
	if (bEnable) {
		quota = quota_size * 1024;
		sprintf(buf, "%lu", quota);
		Conf_Set_Field(SZ_USRQUOTACONF_PATH, "", SZ_USERSIZE_FIELD, buf);
	}
	return SUCCESS;
}

int Set_User_Quota_Disk(int bEnable, unsigned long quota_size, int volume)
{
	unsigned long quota;
	char buf[BUF_SIZE], tmp[BUF_SIZE] = "";
	
	GetPrivateProfileString("", SZ_SAMEQUOTA_FIELD, "TRUE", tmp, sizeof(tmp), SZ_USRQUOTACONF_PATH);
	if (strcasecmp(tmp, "TRUE") == 0)
		return Set_User_Quota(bEnable, quota_size);
	if (bEnable && quota_size <= 0)
		return -1;

	if (!is_volume_exist(volume))
		return -1;
	
	sprintf(buf, "volume%d %s", volume, SZ_USERENABLE_FIELD);
	if(!Set_Private_Profile_Boolean("", buf, bEnable, SZ_USRQUOTACONF_PATH))
		return -1;
	if (bEnable) {
		sprintf(buf, "volume%d %s", volume, SZ_USERSIZE_FIELD);
		quota = quota_size * 1024;
		sprintf(tmp, "%lu", quota);
		Conf_Set_Field(SZ_USRQUOTACONF_PATH, "", buf, tmp);
	}
	return SUCCESS;
}

int Set_Group_Quota(int bEnable, unsigned long quota_size)
{
	unsigned long quota;
	char buf[BUF_SIZE];

	if (bEnable && quota_size <= 0)
		return -1;
	if(!Set_Private_Profile_Boolean("", SZ_GROUPENABLE_FIELD, bEnable, SZ_GRPQUOTACONF_PATH))
		return -1;
	if (bEnable) {
		quota = quota_size * 1024;
		sprintf(buf, "%lu", quota);
		Conf_Set_Field(SZ_GRPQUOTACONF_PATH, "", SZ_GROUPSIZE_FIELD, buf);
	}
	return SUCCESS;
}

int Set_Group_Quota_Disk(int bEnable, unsigned long quota_size, int volume)
{
	unsigned long quota;
	char buf[BUF_SIZE], tmp[BUF_SIZE] = "";
	
	GetPrivateProfileString("", SZ_SAMEQUOTA_FIELD, "TRUE", tmp, sizeof(tmp), SZ_GRPQUOTACONF_PATH);
	if (strcasecmp(tmp, "TRUE") == 0)
		return Set_Group_Quota(bEnable, quota_size);
	if (bEnable && quota_size <= 0)
		return -1;

	if (!is_volume_exist(volume))
		return -1;
	
	sprintf(buf, "volume%d %s", volume, SZ_GROUPENABLE_FIELD);
	if(!Set_Private_Profile_Boolean("", tmp, bEnable, SZ_GRPQUOTACONF_PATH))
		return -1;
	if (bEnable) {
		sprintf(buf, "volume%d %s", volume, SZ_GROUPSIZE_FIELD);
		quota = quota_size * 1024;
		sprintf(tmp, "%lu", quota);
		Conf_Set_Field(SZ_GRPQUOTACONF_PATH, "", buf, tmp);
	}
	return SUCCESS;
}

int Enable_User_Quota()
{
	unsigned long quota;

	if (Is_User_Quota_Enabled(&quota))
		set_all_user_quota(quota, 0);
	else
		clear_all_user_quota(0);
	return SUCCESS;
}

int Enable_User_Quota_Disk(int volume)
{
	unsigned long quota;

	if (Is_User_Quota_Enabled_Disk(&quota, volume))
		set_all_user_quota(quota, volume);
	else
		clear_all_user_quota(volume);
	return SUCCESS;
}

int Enable_Group_Quota()
{
	unsigned long quota;

	if (Is_Group_Quota_Enabled(&quota))
		set_all_group_quota(quota, 0);
	else
		clear_all_group_quota(0);
	return SUCCESS;
}

int Enable_Group_Quota_Disk(int volume)
{
	unsigned long quota;

	if (Is_Group_Quota_Enabled_Disk(&quota, volume))
		set_all_group_quota(quota, volume);
	else
		clear_all_group_quota(volume);
	return SUCCESS;
}

//Write event log funtion Ricky add 2005/1/17
int write_log(char *statement,int type){

	char cmd[1024];
	sprintf(cmd,"%s \"%s\" %d", "/sbin/write_log",statement,type);
	return system(cmd);
}
//End
