//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_ups.c
//
//	Abstract: 
//		System configuration Access Functions for UPS section in uLinux.conf.
//
//	FUNCTIONS:	Setting and getting UPS configuration from uLinux.
//		public function:
//			
//
//	COMMENTS: 	It must working with util.c
//
//	HISTORY:
//		V1.0 2002/03/26		Tiger Fu created
//
//**************************************************************************
#include	<stdio.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<string.h>

#include	<Util.h>
#include	<cfg_nic.h>

#include	"cfg_ups.h"

#define		APCACCESS_CMD		"/sbin/apcaccess"
#define		UPS_SHELL_SCRIPT	"/etc/init.d/ups.sh"
#define		SNMP_UPS_SHELL_SCRIPT	"/etc/init.d/snmp_ups.sh"

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
BOOL Is_UPS_Support (void)
{
	return Get_Profile_Boolean (UPS_SECTION, UPS_SUPPORT, FALSE);
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
BOOL Is_UPS_Enable (void)
{
	return Get_Profile_Boolean (UPS_SECTION, UPS_ENABLE, FALSE);
}

BOOL Has_Serial_UPS_Support() {
	return Get_Profile_Boolean (UPS_SECTION, UPS_SERIAL_UPS, FALSE);
}

BOOL Has_SNMP_UPS_Support() {
	return Get_Profile_Boolean (UPS_SECTION, UPS_SNMP_UPS, FALSE);
}

BOOL Has_USB_UPS_Support() {
	return Get_Profile_Boolean (UPS_SECTION, UPS_USB_UPS, FALSE);
}

void Set_Daemon_Type(int type) {
	char	buf[80];
	int	ret = 0;
	
	Get_Profile_String (UPS_SECTION, UPS_APCUPSD, "", buf, sizeof (buf));
	if ( !strcasecmp(buf, "True") ) {
		switch (type){
			case	2:	//SNMP_DAEMON:
				ret = rw_apcupsd_conf("UPSCABLE", "ether", 1);
				ret |= rw_apcupsd_conf("UPSTYPE", "snmp", 1);
				//ret |= rw_apcupsd_conf("DEVICE", "", 1); ???
				//This should be modified if we use apcupsd for SNMP in the future
				Set_Profile_String (UPS_SECTION, UPS_USE_APCUPSD, "False");
				break;
			case	1:	//SERIAL_DAEMON:
				ret = rw_apcupsd_conf("UPSCABLE", "940-0020B", 1);
				ret |= rw_apcupsd_conf("UPSTYPE", "apcsmart", 1);
				//ret |= rw_apcupsd_conf("DEVICE", "/dev/ttyS0", 1);
				//This should be modified if we use apcupsd for SERIAL in the future
				Set_Profile_String (UPS_SECTION, UPS_USE_APCUPSD, "False");
				break;
			case	3:	//USB_DAEMON:
				ret = rw_apcupsd_conf("UPSCABLE", "usb", 1);
				ret |= rw_apcupsd_conf("UPSTYPE", "usb", 1);
				ret |= rw_apcupsd_conf("DEVICE", "/dev/usb/hid/hiddev[0-15]", 1);
				Set_Profile_String (UPS_SECTION, UPS_USE_APCUPSD, "TRUE");
				break;
		}
	}
	Set_Profile_Integer (UPS_SECTION, UPS_DAEMON_TYPE, type);
}

BOOL Set_UPS_IP(char* ip) {
	return Set_Profile_String(UPS_SECTION, UPS_UPS_IP, ip);  
}

int Get_Daemon_Type() {
	return Get_Profile_Integer(UPS_SECTION, UPS_DAEMON_TYPE, 1); 
}


//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int S_Enable_UPS (BOOL Enable, char* cmd)
{
	int ret = 0;
	char command[256];
	memset (command, '\0', sizeof (command));

	if (Enable == TRUE)
	{
  		Set_Profile_String (UPS_SECTION, UPS_ENABLE, "TRUE");
		sprintf (command, "%s start 2> /dev/null 1> /dev/null", cmd);	
		system (command);
	}
	else 
	{
		Set_Profile_String (UPS_SECTION, UPS_ENABLE, "FALSE");
		sprintf (command, "%s stop 2> /dev/null 1> /dev/null", cmd);
		system (command);
	}

	return ret;
}

int Enable_UPS (BOOL Enable) {
	char command[256];
	memset (command, '\0', sizeof (command));
	strcpy (command, UPS_SHELL_SCRIPT);
	return S_Enable_UPS(Enable, command);  
}

int Enable_SNMP_UPS (BOOL Enable) {
	char command[256];
	memset (command, '\0', sizeof (command));
	strcpy (command, SNMP_UPS_SHELL_SCRIPT);
	return S_Enable_UPS(Enable, command);  
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int Get_UPS_Info (struct UPS_Information * UPS_info)
{
	char buf[80];

	UPS_info->Support = Is_UPS_Support ();				//  support
	UPS_info->Enable = Is_UPS_Enable ();				//  enable

	Get_Profile_String (UPS_SECTION, UPS_TYPE, "", buf, sizeof (buf));
	strcpy (UPS_info->UPS_Type, buf);
	UPS_info->UPS_Shutdown_In = Get_Profile_Integer (UPS_SECTION, UPS_SHUTDOWN_IN, 2);
	Get_Profile_String (UPS_SECTION, UPS_VALID_COM, "1", buf, sizeof (buf));
	strcpy (UPS_info->UPS_Valid_COM, buf);
	UPS_info->AC_Power = Get_Power_Status ();
	UPS_info->Battery_Power = Get_Battery_Status ();
	UPS_info->Daemon_Type = Get_Daemon_Type ();
	
	Get_Profile_String(UPS_SECTION, UPS_UPS_IP, "0.0.0.0", buf, sizeof(buf));
	strcpy(UPS_info->UPS_IP, buf);
	
	return 0;
}

//**************************************************************************
//	Function:	Set_UPS_Type
//	Description:	Set ups type
//			
//	Parameter:	char *
//	Return value:	TRUE	-> success
//			FALSE	-> fail
//
BOOL Set_UPS_Type (char * type)
{
	return Set_Profile_String (UPS_SECTION, UPS_TYPE, type);
}

//**************************************************************************
//	Function:	Set_UPS_Shutdown_In
//	Description:	
//			
//	Parameter:	int
//	Return value:	
//
int Set_UPS_Shutdown_In (int shutdown_time)
{
	char	buf[80];
	int	ret = 0;
	
	Get_Profile_String (UPS_SECTION, UPS_APCUPSD, "", buf, sizeof (buf));
	if ( !strcasecmp(buf, "True") ) {
		char	val[32];

		sprintf(val, "%d", shutdown_time);
		ret = rw_apcupsd_conf("BATTERYLEVEL", "0", 1);	//always 0
		ret |= rw_apcupsd_conf("MINUTES", val, 1);
		ret |= rw_apcupsd_conf("TIMEOUT", "0", 1);	//always 0
		if ( ret )
			return ret;
	}
	
	return Set_Profile_Integer (UPS_SECTION, UPS_SHUTDOWN_IN, shutdown_time);
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int Get_Power_Status (void)
{
	char	buf[80];
	char	val[128];
	int	ret = 0;
	
	Get_Profile_String (UPS_SECTION, UPS_USE_APCUPSD, "False", buf, sizeof (buf));
	if ( !strcasecmp(buf, "True") ) {
		if ( get_apcaccess_status("STATUS", val) == -1 )
			return UNKNOWN_STATUS;
		if ( !strncmp (val, "ONLINE", 6) )
			ret = POWER_OK;
		else if ( !strncmp (val, "ONBATT", 6) )
			ret = POWER_FAIL;
		else
			ret = UNKNOWN_STATUS;
	} else {
		Get_Profile_String (UPS_SECTION, UPS_AC_POWER, "OK", buf, sizeof (buf));
		if (!strcmp (buf, "OK"))
			ret = POWER_OK;
		else if (!strcmp (buf, "FAIL"))
			ret = POWER_FAIL;
		else
			ret = UNKNOWN_STATUS;
	}
	return ret;
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int Set_Power_Status (int status)
{
	if (status == POWER_OK)
		return Set_Profile_String (UPS_SECTION, UPS_AC_POWER, "OK");
	else if (status == POWER_FAIL)
		return Set_Profile_String (UPS_SECTION, UPS_AC_POWER, "FAIL");
	else
		return Set_Profile_String (UPS_SECTION, UPS_AC_POWER, "UNKNOWN");
	
		
	return 0;
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int Get_Battery_Status (void)
{
	char	buf[80];
	char	val[128];
	char	tmp[16];
	int	ret = 0;
	int	threshold_val = 0;
	
	Get_Profile_String (UPS_SECTION, UPS_USE_APCUPSD, "False", buf, sizeof (buf));
	if ( !strcasecmp(buf, "True") ) {
		if ( get_apcaccess_status("BCHARGE", val) == -1 )
			return UNKNOWN_STATUS;
		strncpy(tmp, val, 5);
		tmp[5] = 0;
		threshold_val = atoi(tmp);
		if ( threshold_val >= 15 && threshold_val <= 100)
			ret = BATTERY_OK;
		else if ( threshold_val < 15 && threshold_val >= 0 )
			ret = BATTERY_LOW;
		else
			ret = UNKNOWN_STATUS;
	} else {
		Get_Profile_String (UPS_SECTION, UPS_BATTERY_POWER, "OK", buf, sizeof (buf));
		if (!strcmp (buf, "OK"))
			ret = BATTERY_OK;
		else if (!strcmp (buf, "LOW"))
			ret = BATTERY_LOW;
		else 
			ret = UNKNOWN_STATUS;
	}
	return ret;
}

//**************************************************************************
//	Function: 	
//	Description:	
//			
//	Parameter:	
//	Return value:	
//
int Set_Battery_Status (int status)
{
	if (status == BATTERY_OK)
		return Set_Profile_String (UPS_SECTION, UPS_BATTERY_POWER, "OK");
	else if (status == BATTERY_LOW)
		return Set_Profile_String (UPS_SECTION, UPS_BATTERY_POWER, "LOW");
	else
		return Set_Profile_String (UPS_SECTION, UPS_BATTERY_POWER, "UNKNOWN");
	return 0;
}

int get_apcaccess_status(char* field, char* value)
{
	char		buf[256];
	char		line[256];
	FILE		*fp, *tmpfp;
	int		field_len = strlen(field);
	int		res = -1;
	
	sprintf(buf, "%s 2>/dev/null", APCACCESS_CMD);
	fp = popen(buf, "r");
	if ( !fp )
		return -1;
	tmpfp = tmpfile();

	while (1)
	{
		fgets(line, 256, fp);
		fputs(line, tmpfp);
		line[0] = 0;
		if ( feof(fp) )
			break;
	}
	
	pclose(fp);
	rewind(tmpfp);

	while (1)
	{
		fgets(line, 256, tmpfp);

		/* the output looks like this...
		APC      : 001,040,0980
		DATE     : Fri Oct 03 10:45:30 CST 2003
		HOSTNAME : upsHost
		RELEASE  : 3.10.6
		VERSION  : 3.10.6 (12 September 2003) redhat
		UPSNAME  : CS350
		CABLE    : USB Cable
		MODEL    : APC CS 350
		UPSMODE  : Stand Alone
		STARTTIME: Thu Oct 02 09:32:53 CST 2003
		STATUS   : ONBATT
		*/

		if ( !strncmp(line, field, field_len) ) {
			strcpy(value, &line[11]);
			res = 0;
			break;
		}

		if ( feof(tmpfp) )
			break;
	}
	fclose(tmpfp);

	return res;
}

int rw_apcupsd_conf (char * field, char *val, int func)
{
	FILE	*fp = NULL;
	char	line[1024];
	int	field_len = strlen(field);
	
	if ( func == 0 ) {	//read
		fp = fopen("/etc/apcupsd/apcupsd.conf", "r");
	} else if ( func == 1 ){//write
		fp = fopen("/etc/apcupsd/apcupsd.conf", "r+");
	}
	if ( !fp )
		return -1;

	while (1)
	{
		int	size = 0;

		fgets(line, 1024, fp);
		size = strlen(line);
		if ( !strncasecmp(line, field, field_len) ) {
			if ( func == 0 ) {
				strcpy(val, &line[field_len+1]);
			} else if ( func == 1 ) {
				sprintf(&line[field_len+1], "%s\n%c", val, 0);
				fseek(fp, -size, SEEK_CUR);
				fputs(line, fp);
			}
			break;
		}
		line[0] = 0;
		if ( feof(fp) )
			break;
	}
	if ( fp )
		fclose(fp);

	return 0;
}

