//==========================================================================
//
//      Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//      FILE:
//              cgi.c
//
//      Abstract:
//              cgi library
//
//      FUNCTIONS:
//
//      COMMENTS:       N/A
//
//      HISTORY:
//              2001/08/01      Kent create
//              2001/08/15      Add NIC_Restart();
//				add NIC_Is_Started(int type);
//				add NIC_Get_Dev(char *dev, int len, int type);
//				add NIC_Get_HWADDR(char *hwaddr, int addr_len, int type);
//		2001/09/03	add pppoe function
//		2001/09/05	add pppx device function (for pppoe)
//		2001/11/06	add pptp functions
//
//		2002/02/19  Add by Egbert Chen
//			Add wireless LAN functions to set the parameters
//		
//
//==========================================================================
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/param.h>
#include <sys/socket.h>
#include <sys/ioctl.h>
#include <fcntl.h>

#include <net/if.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netinet/if_ether.h>
#include <netdb.h>
#include <unistd.h>

/* include NAS library */
#include <NAS.h>
#include <Util.h>
#include <hwmon.h>
#include <cfg_system.h>

/* include local file */
#include <cfg_nic.h>

//Shone added for checking ethernet conection status 2005,04,18
typedef unsigned long u64;
typedef unsigned int u32;
typedef unsigned short u16;
typedef unsigned char u8;
//typedefs above are needed in linux/ethtool.h
#include <linux/sockios.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#define __MII_STATUS__		1
#define	MAX_IF_NUMBER		20
//Shone added end

/* define in lib/config/cfg_samba.c */
extern int set_samba_WINS_server(char *ipaddr);

//===================================================
//	check if nic interface is up or down?
//===================================================
int NIC_Check_Status(int type)
{
	char			dev[20];

/* modify by Kent 2002/09/09	*/
        if (	type==NIC_WAN_DEFAULT || type==NIC_LAN01_DEFAULT || type==NIC_LAN02_DEFAULT ||
		type==NIC_LAN03_DEFAULT || type==NIC_LAN04_DEFAULT)
                NIC_Get_Default_Dev(dev, NIC_DEV_LEN, type);
        else
                NIC_Get_Dev(dev, NIC_DEV_LEN, type);
/* end	*/
	return NIC_Check_Status_By_Dev(dev);
}

int NIC_Check_Status_By_Dev(char *dev)
{
	struct ifreq	ifr;
	int		skfd;

	skfd=socket(AF_INET, SOCK_DGRAM, 0);
	if (skfd<0)
		return NIC_FAIL;
        strcpy(ifr.ifr_name, dev);
        if (ioctl(skfd, SIOCGIFFLAGS, &ifr) < 0) {
                close(skfd);
                return (-1);
        }
        close(skfd);
        if (ifr.ifr_flags & IFF_UP)
                return NIC_SUCCESS;
        else
                return NIC_FAIL;	
}

//=====================================================
//	debug NIC_INFO
//=====================================================
void NIC_Debug_Info(NIC_INFO *p)
{
	printf("dev=%s<br>\n", p->dev);
	printf("hdaddr=%s<br>\n", p->hwaddr);
	printf("ipaddr=%s<br>\n", p->ipaddr);
	printf("netmask=%s<br>\n", p->netmask);
	printf("gateway=%s<br>\n", p->gateway);
	printf("broadcast=%s<br>\n", p->broadcast);
	printf("dstaddr=%s<br>\n", p->dstaddr);
}

//=====================================================
//      debug NIC_NAS_INFO
//=====================================================
void NIC_Debug_NAS_Info(NIC_NAS_INFO *p)
{
        printf("dev=%s<br>\n", p->ifconfig.dev);
        printf("hdaddr=%s<br>\n", p->ifconfig.hwaddr);
        printf("ipaddr=%s<br>\n", p->ifconfig.ipaddr);
        printf("netmask=%s<br>\n", p->ifconfig.netmask);
        printf("gateway=%s<br>\n", p->ifconfig.gateway);
        printf("broadcast=%s<br>\n", p->ifconfig.broadcast);
	printf("dstaddr=%s<br>\n", p->ifconfig.dstaddr);
	printf("usage=%d<br>\n", p->usage);
}

//=====================================================
//      debug NIC_TCPIP
//=====================================================
void NIC_Debug_Tcpip(NIC_TCPIP *p)
{
	printf("dns=%s<br>\n", p->dns);
	printf("wan=%s<br>\n", p->wan);
	printf("lan=%s<br>\n", p->lan01);
}

//==================================================
//	reset to default 
//	note: this function can auto detect one nic or two nic
//	mode=1 => restart thttpd & network
//	mode=0 => do nothing
//==================================================
void NIC_Reset(int mode)
{
	int		num;
	NIC_NAS_INFO	nic_nas_info;

	num = NIC_Count_Interface();
	switch (num)
	{
		case 3:
			if (NIC_Is_Bonding_Support())
			{
                                NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN01);
                                nic_nas_info.usage=NIC_USE_DHCP;
                                strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.1");
                                strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
                                strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
                                strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
                                nic_nas_info.bonding_type=BONDING_LOAD_BALANCE;
                                NIC_Set_Info_To_uLinux(&nic_nas_info);
                                WriteProfileString(nic_nas_info.ifconfig.dev, SZ_CONFIGURED_FIELD, "FALSE");

                                NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN02);
                                nic_nas_info.usage=NIC_USE_DHCP;
                                strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.1");
                                strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
                                strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
                                strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
                                nic_nas_info.bonding_type=BONDING_LOAD_BALANCE;
                                NIC_Set_Info_To_uLinux(&nic_nas_info);

				if (NIC_Count_Bonding_Interface() == 3) {
	                                NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN03);
	                                nic_nas_info.usage=NIC_USE_DHCP;
	                                strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.1");
	                                strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
	                                strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
	                                strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
	                                nic_nas_info.bonding_type=BONDING_LOAD_BALANCE;
	                                NIC_Set_Info_To_uLinux(&nic_nas_info);
	                        }
	                        else if (NIC_Is_Switch_Support()) {
	                                NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN03);
	                                nic_nas_info.usage=NIC_USE_DHCP;
	                                strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.2");
	                                strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
	                                strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
	                                strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
	                                nic_nas_info.bonding_type=BONDING_STANDALONE;
					NIC_Set_Info_To_uLinux(&nic_nas_info);
	                        }
				break;

			}
			break;
		case 2:
			/* reset wan device */
			if (NIC_Is_Bonding_Support())
			{
				NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN01);
				nic_nas_info.usage=NIC_USE_DHCP;
				strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.1");
				strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
				strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
				strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
				nic_nas_info.bonding_type=BONDING_LOAD_BALANCE;
				NIC_Set_Info_To_uLinux(&nic_nas_info);
				WriteProfileString(nic_nas_info.ifconfig.dev, SZ_CONFIGURED_FIELD, "FALSE");

				NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN02);
				nic_nas_info.usage=NIC_USE_DHCP;
				strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.0.1");
				strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
				strcpy(nic_nas_info.ifconfig.broadcast, "192.168.0.255");
				strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
				nic_nas_info.bonding_type=BONDING_LOAD_BALANCE;
				NIC_Set_Info_To_uLinux(&nic_nas_info);
				Set_Web_Access(SYSTEM_WAN_ACCESS, 1);
				break;
			}
			else
			{
				NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_WAN);
				nic_nas_info.usage=NIC_USE_DHCP;
				strcpy(nic_nas_info.ifconfig.ipaddr, "10.0.0.1");
				strcpy(nic_nas_info.ifconfig.netmask, "255.0.0.0");
				strcpy(nic_nas_info.ifconfig.broadcast, "10.255.255.255");
				strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
				NIC_Set_Info_To_uLinux(&nic_nas_info);
				WriteProfileString(nic_nas_info.ifconfig.dev, SZ_CONFIGURED_FIELD, "FALSE");
				/* reset lan device */
				NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN01);
				nic_nas_info.usage=NIC_USE_STATIC;
				strcpy(nic_nas_info.ifconfig.ipaddr, "192.168.1.254");
				strcpy(nic_nas_info.ifconfig.netmask, "255.255.255.0");
				strcpy(nic_nas_info.ifconfig.broadcast, "192.168.1.255");
				strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
				NIC_Set_Info_To_uLinux(&nic_nas_info);
				Set_Web_Access(SYSTEM_WAN_ACCESS, 1);
				break;
			}
		case 1:
			NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_LAN01);
			nic_nas_info.usage=NIC_USE_DHCP;
			strcpy(nic_nas_info.ifconfig.ipaddr, "169.254.100.100");
			strcpy(nic_nas_info.ifconfig.netmask, "255.255.0.0");
			strcpy(nic_nas_info.ifconfig.broadcast, "169.254.255.255");
			strcpy(nic_nas_info.ifconfig.gateway, "0.0.0.0");
			NIC_Set_Info_To_uLinux(&nic_nas_info);
			WriteProfileString(nic_nas_info.ifconfig.dev, SZ_CONFIGURED_FIELD, "FALSE");
			Set_Web_Access(SYSTEM_LAN_ACCESS, 1);
			break;
	}
	Set_Web_Access_Port(80);
	if (mode)
	{
		// 2005.11.26, Johnson Cheng
		// Don't need restart thttpd even if IP address is changed.
		//Web_Server_Restart();
		NIC_Restart();
	}
	return;
}

//=====================================================
//      get default gateway from dev
//=====================================================
int get_default_gw(char *dev, struct sockaddr_in *gw)
{
        char			str[256];
        FILE			*fptr;
        struct sockaddr_in	sin;
        int			cnt=0;
	char			route_file[]={"/proc/net/route"};
	char			seps[] = " \t\n";
	char			*token;

        fptr=fopen(route_file, "r");
        fgets(str, 256, fptr);
        while (fgets(str, 256, fptr) != NULL)
	{
                token = strtok(str, seps);
                if(token == NULL)
                        continue;

		/* check if our device name? */
                if(strcmp(token, dev))
                        continue;
                token = strtok(NULL, seps);
		/* check if destinition is 0.0.0.0 */
                sin.sin_addr.s_addr = (uint32_t)strtoul(token, NULL, 16);
		if (sin.sin_addr.s_addr!=0)
			continue;

		/* get default gw address */
                token = strtok(NULL, seps);
                sin.sin_addr.s_addr = (uint32_t)strtoul(token, NULL, 16);
                gw->sin_addr.s_addr = sin.sin_addr.s_addr;
                cnt++;
        }
        fclose(fptr);
	if (cnt==0)
		return NIC_FAIL;
	else
		return NIC_SUCCESS;
}

int NIC_Info_Comp(NIC_NAS_INFO *info1, NIC_NAS_INFO *info2)
{
        if (	!strcmp(info1->ifconfig.dev, info2->ifconfig.dev) &&
		!strcmp(info1->ifconfig.hwaddr, info2->ifconfig.hwaddr) &&
		!strcmp(info1->ifconfig.ipaddr, info2->ifconfig.ipaddr) &&
		!strcmp(info1->ifconfig.netmask, info2->ifconfig.netmask) &&
		!strcmp(info1->ifconfig.gateway, info2->ifconfig.gateway) &&
		!strcmp(info1->ifconfig.broadcast, info2->ifconfig.broadcast) &&
		!strcmp(info1->ifconfig.dstaddr, info2->ifconfig.dstaddr) &&
		info1->usage==info2->usage)
                return 0;
        else
                return 1;
}

//=====================================================
//      get network info by using ioctl
//=====================================================
int NIC_Get_Info(NIC_INFO *pif, int type)
{
        char 			*inbuf=NULL;
        struct ifconf		ifc;
        struct ifreq		ifreq, *ifrp;
        struct ifreq		ifrm;
        int			i, siz, len=8192;
        char			ifrbuf[8192];
        unsigned char	*ptr;
        struct sockaddr_in	*sin;
        int			count = 0;
	int			s;
	char			name[30];
	struct sockaddr_in	netmask;
	struct sockaddr_in	broadcast;
	struct sockaddr_in	dstaddr;
	struct sockaddr_in	gw;
	char			dev[NIC_DEV_LEN];

	/* reset to default */
	strcpy(pif->ipaddr, "");
	strcpy(pif->netmask, "");
	strcpy(pif->gateway, "");
	strcpy(pif->broadcast, "");
	strcpy(pif->dstaddr, "");

/*	NIC_Get_Dev(dev, NIC_DEV_LEN, type);
	if (type==NIC_WAN_DEV && !strcmp(dev, "ppp0"))
		GetProfileString("Network", SZ_WAN_FIELD, "eth0", dev, NIC_DEV_LEN);
	if (type==NIC_LAN01_DEV && !strcmp(dev, "br0"))
		GetProfileString("Network", SZ_LAN_FIELD, "eth1", dev, NIC_DEV_LEN);*/
/* modify by Kent 2002/09/09	*/
        if (	type==NIC_WAN_DEFAULT || type==NIC_LAN01_DEFAULT || type==NIC_LAN02_DEFAULT ||
		type==NIC_LAN03_DEFAULT || type==NIC_LAN04_DEFAULT)
                NIC_Get_Default_Dev(dev, NIC_DEV_LEN, type);
        else
                NIC_Get_Dev(dev, NIC_DEV_LEN, type);
/* end	*/
	strcpy(pif->dev, dev);
        memset(&ifrm, 0, sizeof(struct ifreq));
        strcpy(ifrm.ifr_name, pif->dev);

	s=socket(AF_INET, SOCK_DGRAM, 0);
        if (s<0)
                return NIC_FAIL;
        while (1)
	{
                ifc.ifc_len=len;
                ifc.ifc_buf=inbuf=calloc(1, len);
                if (inbuf==NULL) {
                        close(s);
                        return NIC_FAIL;
                }

                if (ioctl(s, SIOCGIFCONF, &ifc)<0)
			goto nic_get_info_fail;
                if ((ifc.ifc_len+sizeof(ifreq))<len)
                        break;
                len *= 2;
        }
        ifrp = ifc.ifc_req;
        ifreq.ifr_name[0] = '\0';
        for (i = 0; i < ifc.ifc_len; )
	{
                ifrp = (struct ifreq *)((caddr_t)ifc.ifc_req + i);
                memcpy(ifrbuf, ifrp, sizeof(*ifrp));

		/* glibc>2.2.3 not support */
//                siz = SA_LEN(&(((struct ifreq *)ifrbuf)->ifr_addr));
//                if (siz < sizeof(ifrp->ifr_addr))
                        siz = sizeof(ifrp->ifr_addr);
                siz += sizeof(ifrp->ifr_name);
                i += siz;
                /* avoid alignment issue */
                if (sizeof(ifrbuf) < siz)
			goto nic_get_info_fail;
                memcpy(ifrbuf, ifrp, siz);
                ifrp = (struct ifreq *)ifrbuf;

                if (strncmp(ifrm.ifr_name, ifrp->ifr_name, sizeof(ifrp->ifr_name)))
                        continue;
                strncpy(name, ifrp->ifr_name, sizeof(ifrp->ifr_name));
                sin = (struct sockaddr_in *)&ifrp->ifr_addr;
                //----------------------------------------------------
                // IP Address
                //----------------------------------------------------
                sprintf(pif->ipaddr, "%s", inet_ntoa(sin->sin_addr));

		//=================================================
		// Broadcast Address
		//=================================================
                if (ioctl(s, SIOCGIFBRDADDR, (caddr_t)ifrp) < 0)
			goto nic_get_info_fail;
                broadcast.sin_addr = ((struct sockaddr_in *)&(ifrp->ifr_addr))->sin_addr;
                sprintf(pif->broadcast, "%s", inet_ntoa(broadcast.sin_addr));

                //====================================================
		// Netmask address
		//====================================================
                if (ioctl(s, SIOCGIFNETMASK, (caddr_t)ifrp) < 0)
			goto nic_get_info_fail;
                netmask.sin_addr = ((struct sockaddr_in *)&(ifrp->ifr_addr))->sin_addr;
                sprintf(pif->netmask, "%s", inet_ntoa(netmask.sin_addr));

		//=====================================================
		// Destnition address ( for pppx device use )
		// if not pppx device, the dstaddr will be equal to ipaddr
		//=====================================================
                if (ioctl(s, SIOCGIFDSTADDR, (caddr_t)ifrp) < 0)
                        goto nic_get_info_fail;
                dstaddr.sin_addr = ((struct sockaddr_in *)&(ifrp->ifr_addr))->sin_addr;
                sprintf(pif->dstaddr, "%s", inet_ntoa(dstaddr.sin_addr));

                //----------------------------------------------------
                // Gateway Address
                //----------------------------------------------------
                if (get_default_gw(ifrp->ifr_name, &gw)==NIC_SUCCESS)
                        sprintf(pif->gateway, "%s", inet_ntoa(gw.sin_addr));
		else
			sprintf(pif->gateway, "0.0.0.0");

                count++;
        }
        // there is no need to enumerate if in order to find MAC address
        //      because sometimes cable is not connected...
        // you can force to call ioctl iff device driver is up
        strcpy(ifrp->ifr_name, dev);
        if (ioctl(s, SIOCGIFHWADDR, (caddr_t)ifrp) < 0)
		goto nic_get_info_fail;
        //----------------------------------------------------
        // MAC Address
        //----------------------------------------------------
        ptr = &(ifrp->ifr_hwaddr.sa_data[0]);
        sprintf(pif->hwaddr, "%.2X:%.2X:%.2X:%.2X:%.2X:%.2X", ptr[0], ptr[1], ptr[2], ptr[3], ptr[4], ptr[5]);
        free(inbuf);
        if (ioctl(s, SIOCGIFFLAGS, &ifrm) < 0) {
        	close(s);
                return NIC_FAIL;
        }
	close(s);
        if (!( ifrm.ifr_flags & 1))
                return NIC_FAIL;
        return NIC_SUCCESS;

nic_get_info_fail:
	free(inbuf);
	close(s);
	return NIC_FAIL;
}

//=====================================================
//      get network section from uLinux.conf
//=====================================================
int NIC_Get_Tcpip(NIC_TCPIP *tcpip)
{
        GetProfileString("Network", SZ_DNS_FIELD, "0.0.0.0", tcpip->dns, NIC_IPADDR_LEN);
        GetProfileString("Network", SZ_WINS_FIELD, "0.0.0.0", tcpip->wins, NIC_IPADDR_LEN);
        GetProfileString("Network", SZ_WAN_FIELD, "eth0", tcpip->wan, NIC_DEV_LEN);
        GetProfileString("Network", SZ_LAN01_FIELD, "eth0", tcpip->lan01, NIC_DEV_LEN);
        GetProfileString("Network", SZ_LAN02_FIELD, "eth1", tcpip->lan02, NIC_DEV_LEN);
        GetProfileString("Network", SZ_LAN03_FIELD, "eth2", tcpip->lan03, NIC_DEV_LEN);
	GetProfileString("Network", SZ_LAN04_FIELD, "eth3", tcpip->lan04, NIC_DEV_LEN);
	return NIC_SUCCESS;
}

//=====================================================
//      get device from your type
//	if the len if less than NIC_DEV_LEN, it will return NIC_FAIL
//	type =	NIC_WAN
//		NIC_LAN01
//=====================================================
int NIC_Get_Dev(char *dev, int len, int type)
{
	NIC_TCPIP	tcpip;
	char		usage[80];
// Catherine for switch begin ==>
	BOOL		is_switch_support;
	int		bond_cnt, count_nic;

	is_switch_support = Get_Profile_Boolean("Network", "Switch Support", FALSE);
	count_nic = NIC_Count_Interface();
	bond_cnt = NIC_Count_Bonding_Interface();
// <== Catherine for switch

        if (len<NIC_DEV_LEN)
                return NIC_FAIL;
        NIC_Get_Tcpip(&tcpip);
        GetProfileString(tcpip.wan, SZ_USAGE_FIELD, "", usage, 80);

        // get wan device
        if ((type==NIC_WAN || type==NIC_WAN_DEFAULT) && NIC_Is_WAN_Use_Tmp_DHCP())
                strcpy(dev, tcpip.wan);
        else
// Catherine for switch begin ==>
	if (type == NIC_LAN03 && is_switch_support && count_nic > bond_cnt) {
		strcpy(dev, tcpip.lan03);
	}
	else
// <== Catherine for switch
	if (NIC_Is_Bonding_Support() && NIC_Get_Bonding_Type()!=BONDING_STANDALONE)
		strcpy(dev, BONDING_DEV);
	else
        if (!strcmp(usage, "PPPOE") && type==NIC_WAN)
                strcpy(dev, "ppp0");
        else
        if (!strcmp(usage, "PPTP") && type==NIC_WAN)
                strcpy(dev, "ppp0");
        else
        if (type==NIC_WAN || type==NIC_WAN_DEFAULT)
                strcpy(dev, tcpip.wan);
        else
        // get lan device
        if (    (type==NIC_LAN01 || type==NIC_LAN01_DEFAULT) &&
                Is_WLAN_Supported() && Is_WLAN_Card_Exist() && Is_WLAN_Enabled()) {
                strcpy(dev, BRIDGE_DEV);
	} else
        if (	type==NIC_LAN01 || type==NIC_LAN01_DEFAULT ) { 
                strcpy(dev, tcpip.lan01);
	} else
        if (	type==NIC_LAN02 || type==NIC_LAN02_DEFAULT ) {
                strcpy(dev, tcpip.lan02);
	} else
        if (	type==NIC_LAN03 || type==NIC_LAN03_DEFAULT ) {
                strcpy(dev, tcpip.lan03);
	} else
        if (	type==NIC_LAN04 || type==NIC_LAN04_DEFAULT ) {
                strcpy(dev, tcpip.lan04);
	}
        else
	{
		strcpy(dev, "errordev");
                return NIC_FAIL;
	}
        return NIC_SUCCESS;
}

int NIC_Get_Default_Dev(char *dev, int len, int type)
{
        NIC_TCPIP       tcpip;
        char            usage[80];

        if (len<NIC_DEV_LEN)
                return NIC_FAIL;
        NIC_Get_Tcpip(&tcpip);
        GetProfileString(tcpip.wan, SZ_USAGE_FIELD, "", usage, 80);

        if (type==NIC_WAN || type==NIC_WAN_DEFAULT)
                strcpy(dev, tcpip.wan);
        else
        if (type==NIC_LAN01 || type==NIC_LAN01_DEFAULT)
                strcpy(dev, tcpip.lan01);
	else
	if (type==NIC_LAN02 || type==NIC_LAN02_DEFAULT)
		strcpy(dev, tcpip.lan02);
	else
	if (type==NIC_LAN03 || type==NIC_LAN03_DEFAULT)
		strcpy(dev, tcpip.lan03);
	else
	if (type==NIC_LAN04 || type==NIC_LAN04_DEFAULT)
		strcpy(dev, tcpip.lan04);
        else
	{
		strcpy(dev, "errordefaultdev");
                return NIC_FAIL;
	}
        return NIC_SUCCESS;
}

//=====================================================
//      set network section into uLinux.conf
//=====================================================
int NIC_Set_Tcpip(NIC_TCPIP *tcpip)
{
	int		ret;
//	FILE		*fptr;
//	NIC_NAS_INFO	nic_nas_info;

	/* write dns */
	NIC_Set_DNS(tcpip->dns);
//	ret=WriteProfileString("Network", SZ_DNS_FIELD, tcpip->dns);
//	NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_WAN);
//	if (nic_nas_info.usage==NIC_USE_STATIC)
//	{	/* use static IP			*/
//		/* so only select fix DNS address	*/
//		fptr=fopen(DNS_CONF_FILE, "wb");
//		fprintf(fptr, "nameserver %s", tcpip->dns);
//		fclose(fptr);
//	}
//	else
//	{
//		fptr=fopen(DNS_CONF_FILE, "a+");
//	}

	/* write wins */
	ret=WriteProfileString("Network", SZ_WINS_FIELD, tcpip->wins);
	if (!ret)
		return NIC_FAIL;

	/* write wan */
        ret=WriteProfileString("Network", SZ_WAN_FIELD, tcpip->wan);
        if (!ret)
                return NIC_FAIL;

	/* write lan */
        ret=WriteProfileString("Network", SZ_LAN01_FIELD, tcpip->lan01);
        if (!ret)
                return NIC_FAIL;
	return NIC_SUCCESS;
}

//=================================================================
//      base on the type to get device information from uLinux.conf
//=================================================================
int NIC_Get_Info_From_uLinux(NIC_NAS_INFO *nic_nas_info, int type)
{
        char		buf[NIC_LINE];

	NIC_Get_Default_Dev(nic_nas_info->ifconfig.dev, NIC_DEV_LEN, type);
	
	GetProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "", buf, NIC_LINE);
	if (!strcmp(buf, "DHCP"))
		nic_nas_info->usage=NIC_USE_DHCP;
	else
	if (!strcmp(buf, "STATIC"))
		nic_nas_info->usage=NIC_USE_STATIC;
	else
	if (!strcmp(buf, "PPPOE"))
		nic_nas_info->usage=NIC_USE_PPPOE;
	else
	if (!strcmp(buf, "PPTP"))
		nic_nas_info->usage=NIC_USE_PPTP;
	else
		nic_nas_info->usage=NIC_USE_OTHER;

        GetProfileString(nic_nas_info->ifconfig.dev, SZ_HWADDR_FIELD, "", nic_nas_info->ifconfig.hwaddr, NIC_HWADDR_LEN);
        GetProfileString(nic_nas_info->ifconfig.dev, SZ_IPADDR_FIELD, "", nic_nas_info->ifconfig.ipaddr, NIC_IPADDR_LEN);
        GetProfileString(nic_nas_info->ifconfig.dev, SZ_NETMASK_FIELD, "", nic_nas_info->ifconfig.netmask, NIC_IPADDR_LEN);
        GetProfileString(nic_nas_info->ifconfig.dev, SZ_GATEWAY_FIELD, "", nic_nas_info->ifconfig.gateway, NIC_IPADDR_LEN);
        GetProfileString(nic_nas_info->ifconfig.dev, SZ_BROADCAST_FIELD, "", nic_nas_info->ifconfig.broadcast, NIC_IPADDR_LEN);
        if (NIC_Is_Bonding_Device(nic_nas_info->ifconfig.dev))
		nic_nas_info->bonding_type=NIC_Get_Bonding_Type();
	else
		nic_nas_info->bonding_type=BONDING_STANDALONE;
	return NIC_SUCCESS;
}

//=====================================================
//      set device information to uLinux.conf
//=====================================================
int NIC_Set_Info_To_uLinux(NIC_NAS_INFO *nic_nas_info)
{
	int		ret, restart=0, type=NIC_LAN01;
	NIC_NAS_INFO	tmp_info;
	NIC_TCPIP	tcpip;

	NIC_Get_Tcpip(&tcpip);
	if (!strcmp(tcpip.wan, nic_nas_info->ifconfig.dev))
		type=NIC_WAN_DEFAULT;
	else
	if (!strcmp(tcpip.lan01, nic_nas_info->ifconfig.dev))
		type=NIC_LAN01_DEFAULT;
	else
	if (!strcmp(tcpip.lan02, nic_nas_info->ifconfig.dev))
		type=NIC_LAN02_DEFAULT;
	else
	if (!strcmp(tcpip.lan03, nic_nas_info->ifconfig.dev))
		type=NIC_LAN03_DEFAULT;
	else
	if (!strcmp(tcpip.lan04, nic_nas_info->ifconfig.dev))
		type=NIC_LAN04_DEFAULT;

	NIC_Get_Info_From_uLinux(&tmp_info, type);
	/* two usages are not the same, need to restart */
	if (tmp_info.usage!=nic_nas_info->usage)
		restart=1;
	else
	if (tmp_info.usage==NIC_USE_STATIC)
	{
		if (	!strcmp(tmp_info.ifconfig.ipaddr, nic_nas_info->ifconfig.ipaddr) &&
			!strcmp(tmp_info.ifconfig.gateway, nic_nas_info->ifconfig.gateway) &&
			!strcmp(tmp_info.ifconfig.netmask, nic_nas_info->ifconfig.netmask))
			restart=0;
		else
			restart=1;
	}

        // set bonding type
	if (NIC_Is_Bonding_Support() && NIC_Is_Bonding_Device(nic_nas_info->ifconfig.dev))
	{
		if (NIC_Get_Bonding_Type()!=nic_nas_info->bonding_type)
			restart=1;
	        NIC_Set_Bonding_Type(nic_nas_info->bonding_type);
	}
 
	if (nic_nas_info->usage==NIC_USE_DHCP)
	{
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "DHCP");
		if (!ret)
			return NIC_FAIL;
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "TRUE");
		if (!ret)
			return NIC_FAIL;
	}
	else
	if (nic_nas_info->usage==NIC_USE_STATIC)
	{
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "STATIC");
                if (!ret)
                        return NIC_FAIL;
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                if (!ret)
                        return NIC_FAIL;
	}
	else
	if (nic_nas_info->usage==NIC_USE_PPPOE)
	{
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "PPPOE");
                if (!ret)
                        return NIC_FAIL;
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                if (!ret)
                        return NIC_FAIL;
	}
	else
	if (nic_nas_info->usage==NIC_USE_PPTP)
	{
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "PPTP");
		if (!ret)
			return NIC_FAIL;
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
		if (!ret)
			return NIC_FAIL;
	}
	else
	{
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "OTHER");
                if (!ret)
                        return NIC_FAIL;
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                if (!ret)
                        return NIC_FAIL;
	}
	ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_IPADDR_FIELD, nic_nas_info->ifconfig.ipaddr);
        if (!ret)
                return NIC_FAIL;
	ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_NETMASK_FIELD, nic_nas_info->ifconfig.netmask);
        if (!ret)
                return NIC_FAIL;
	ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_GATEWAY_FIELD, nic_nas_info->ifconfig.gateway);
        if (!ret)
                return NIC_FAIL;
	strcpy(nic_nas_info->ifconfig.broadcast, "0.0.0.0");
        if (strcasecmp(nic_nas_info->ifconfig.broadcast,"0.0.0.0") == 0 )
	{
                ULONG bin_ip,bin_netmask,bin_broadcast;
                struct in_addr in;
                bin_ip = inet_addr ( nic_nas_info->ifconfig.ipaddr);
                bin_netmask = inet_addr (nic_nas_info->ifconfig.netmask);
                bin_broadcast = (bin_ip & bin_netmask) | (~bin_netmask);
                in.s_addr = bin_broadcast;

                sprintf(nic_nas_info->ifconfig.broadcast, "%s", (char *)inet_ntoa(in));
        }
        ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_BROADCAST_FIELD, nic_nas_info->ifconfig.broadcast);
        if (!ret)
                return NIC_FAIL;
        // set configured flag for ethx to TRUE, thereafter it will follow the definition now.
        WriteProfileString(nic_nas_info->ifconfig.dev, SZ_CONFIGURED_FIELD, "TRUE");

	// check if bonding support ?
	// if yes, duplicate BONDING01 to BONDING02
/*	if (NIC_Is_Bonding_Support())
	{
		NIC_Bonding_Cfg_Copy(nic_nas_info);
	}
*/
	// The code above is not compatible with standalone network settings, we should not use that "if" statement
	if ( nic_nas_info->bonding_type == BONDING_FAIL_OVER || nic_nas_info->bonding_type == BONDING_LOAD_BALANCE )
		NIC_Bonding_Cfg_Copy(nic_nas_info);

	if (restart==1)
		return NIC_RESTART;
	return NIC_SUCCESS;
}

//=====================================================
//      this is not export function
//	for NIC_Gen_Ifinfo() function used
//	this function will create a temp file
//=====================================================
void nic_write_tmp(char *filename, int type)
{
	FILE		*fptr;
	NIC_NAS_INFO	nic_nas_info;

	NIC_Get_Info_From_uLinux(&nic_nas_info, type);

	fptr=fopen(filename, "w");
	fprintf(fptr, "DEV=%s\n", nic_nas_info.ifconfig.dev);
	if (nic_nas_info.usage==NIC_USE_STATIC)
	        fprintf(fptr, "USAGE=STATIC\n");
	else
	if (nic_nas_info.usage==NIC_USE_DHCP)
		fprintf(fptr, "USAGE=DHCP\n");
	else
	if (nic_nas_info.usage==NIC_USE_PPPOE)
		fprintf(fptr, "USAGE=PPPOE\n");
	else
	if (nic_nas_info.usage==NIC_USE_PPTP)
		fprintf(fptr, "USAGE=PPTP\n");
	else
	if (nic_nas_info.usage==NIC_USE_OTHER)
		fprintf(fptr, "USAGE=OTHER\n");
	fprintf(fptr, "HWADDR=%s\n", nic_nas_info.ifconfig.hwaddr);
	fprintf(fptr, "IPADDR=%s\n", nic_nas_info.ifconfig.ipaddr);
	fprintf(fptr, "NETMASK=%s\n", nic_nas_info.ifconfig.netmask);
	fprintf(fptr, "GATEWAY=%s\n", nic_nas_info.ifconfig.gateway);
	fprintf(fptr, "BROADCAST=%s\n", nic_nas_info.ifconfig.broadcast);
	fclose(fptr);
	return;
}

//=====================================================
//      generate the nic device information in /tmp
//	for network.sh used
//=====================================================
void NIC_Gen_Ifinfo()
{
        NIC_TCPIP       tcpip;
	FILE		*fptr;
	char		buf[80];
	int		num;

	system("/bin/rm /tmp/nic* -rf 2>/dev/null 1>/dev/null");
        NIC_Get_Tcpip(&tcpip);
        GetProfileString("Network", SZ_NIC_NUM_FIELD, "1", buf, 80);
        num=atoi(buf);

	/* write nic device file */
	fptr=fopen(NIC_TMP_FILE, "w");
	if (num==1)
		fprintf(fptr, "LAN01=%s\n", tcpip.lan01);
	else
	if (num==3 && NIC_Is_Bonding_Support())
	{
		fprintf(fptr, "LAN01=%s\n", tcpip.lan01);
		fprintf(fptr, "LAN02=%s\n", tcpip.lan02);
		fprintf(fptr, "LAN03=%s\n", tcpip.lan03);
	}
	else
	if (num==2 && NIC_Is_Bonding_Support())
	{
		fprintf(fptr, "LAN01=%s\n", tcpip.lan01);
		fprintf(fptr, "LAN02=%s\n", tcpip.lan02);
	}
	else
	if (num==2)
	{
		fprintf(fptr, "LAN01=%s\n", tcpip.lan01);
		fprintf(fptr, "WAN=%s\n", tcpip.wan);
	}
	fclose(fptr);

	/* generate nic ifconfig information */
	if (num==1)
		nic_write_tmp(NIC01_TMP_FILE, NIC_LAN01);
	else
	if (num==3 && NIC_Is_Bonding_Support())
	{
		nic_write_tmp(NIC01_TMP_FILE, NIC_LAN01);
		nic_write_tmp(NIC02_TMP_FILE, NIC_LAN02);
		nic_write_tmp(NIC03_TMP_FILE, NIC_LAN03);
	}
	else
	if (num==2 && NIC_Is_Bonding_Support())
	{
		nic_write_tmp(NIC01_TMP_FILE, NIC_LAN01);
		nic_write_tmp(NIC02_TMP_FILE, NIC_LAN02);
	}
	else
	if (num==2)
	{
		nic_write_tmp(NIC01_TMP_FILE, NIC_WAN);
		nic_write_tmp(NIC02_TMP_FILE, NIC_LAN01);
	}

	/* write hwaddr in uLinux.conf */
//	NIC_Set_HWADDR(NIC_WAN);
//	NIC_Set_HWADDR(NIC_LAN01);
	return;
}

//=====================================================
//      get nic device number
//=====================================================
int NIC_Count_Interface()
{
	int	cnt;
	char	buf[80];

	GetProfileString("Network", SZ_NIC_NUM_FIELD, "1", buf, 80);
        cnt=atoi(buf);
	return cnt;
}

//===================================================
//	read dns address from /etc/resolv.conf
//===================================================
int NIC_Get_Resolv(char *ipaddr, int len)
{
	char	buf[NIC_LINE], *ptr;
	FILE	*fptr;
	int	find=0;

	if (len<NIC_IPADDR_LEN)
		return NIC_FAIL;
	fptr=fopen(DNS_CONF_FILE, "r");
	while (!feof(fptr))
	{
		if (fgets(buf, NIC_LINE, fptr)!=NULL)
		{
			if (strstr(buf, "nameserver"))
			{
				ptr=strtok(buf, " \t\n");
				ptr=strtok(NULL, " \t\n");
				strcpy(ipaddr, ptr);
				find=1;
				break;
			}
		}
	}
	fclose(fptr);
	if (find==0)
		NIC_Get_DNS(ipaddr, len);
	return NIC_SUCCESS;
}

int NIC_Get_Resolv2(char *ipaddr1, int len1, char *ipaddr2, int len2)
{
        char    buf[NIC_LINE], *ptr;
        FILE    *fptr;
        int     find=0;

        if (len1<NIC_IPADDR_LEN || len2<NIC_IPADDR_LEN)
                return NIC_FAIL;
        if ((fptr=fopen(DNS_CONF_FILE, "r"))==NULL)
	{
		strcpy(ipaddr1, "0.0.0.0");
		strcpy(ipaddr2, "0.0.0.0");
		return NIC_SUCCESS;
	}
        while (!feof(fptr))
        {
                if (fgets(buf, NIC_LINE, fptr)!=NULL)
                {
                        if (strstr(buf, "nameserver"))
                        {
                                ptr=strtok(buf, " \t\n");
                                ptr=strtok(NULL, " \t\n");
				find++;
				if (find==1)
	                                strcpy(ipaddr1, ptr);
				else
				if (find==2)
				{
					strcpy(ipaddr2, ptr);
					break;
				}
                        }
                }
        }
        fclose(fptr);
        if (find==0)
	{
                NIC_Get_DNS(ipaddr1, len1);
		strcpy(ipaddr2, "0.0.0.0");
	}
	else
	if (find==1)
		strcpy(ipaddr2, "0.0.0.0");
	return NIC_SUCCESS;
}

//====================================================
//	create /etc/resolv.conf from uLinux.conf
//====================================================
int NIC_Create_Resolv()
{
	FILE		*fptr;
	char		dns[NIC_MAC_DNS_COUNT][NIC_IPADDR_LEN];
	NIC_NAS_INFO	nic_nas_info;
	int		i, dns_cnt = 0;

        NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_WAN);
	dns_cnt = NIC_Get_DNS_Ex((char*)dns, NIC_IPADDR_LEN, NIC_MAC_DNS_COUNT);
        if (nic_nas_info.usage==NIC_USE_STATIC)
        {       /* use static IP                        */
                fptr=fopen(DNS_CONF_FILE, "wb");
                for (i = 0; i<dns_cnt; i++) {
                	if(strcmp(dns[i], "0.0.0.0"))
                		fprintf(fptr, "nameserver %s\n", dns[i]);
                }
                fclose(fptr);
        }
        else
        {
		fptr=fopen(DNS_CONF_FILE, "a+");
                for (i = 0; i<dns_cnt; i++) {
			fprintf(fptr, "nameserver %s\n", dns[i]);
		}
		fclose(fptr);
        }

	return NIC_SUCCESS;
}

//=====================================================
//      set dns into uLinux.conf and /etc/resolv.conf
//=====================================================
int NIC_Set_DNS(char *ipaddr)
{
	int		ret;

        ret=WriteProfileString("Network", SZ_DNS_FIELD, ipaddr);
        if (!ret)
                return NIC_FAIL;
	NIC_Create_Resolv();
	return NIC_SUCCESS;
}

//=====================================================
//      get dns from uLinux.conf
//=====================================================
int NIC_Get_DNS(char *ipaddr, int len)
{
	char	buf[NIC_IPADDR_LEN];

	GetProfileString("Network", SZ_DNS_FIELD, "0.0.0.0", buf, NIC_IPADDR_LEN);
	if ((strlen(buf)+1)>len)
		return NIC_FAIL;
	strcpy(ipaddr, buf);
	return NIC_SUCCESS;
}

//=====================================================
//      set dns into uLinux.conf and /etc/resolv.conf -- added by YFHuang 20040421
//=====================================================
int NIC_Set_DNS_Ex(int index, char *ipaddr)
{
	int	ret;
	char	buf[BUF_SIZE];

        if (index > NIC_MAC_DNS_COUNT) return NIC_FAIL;

        sprintf(buf, "Domain Name Server %d", index);
		
        ret = WriteProfileString("Network", buf, ipaddr);
        
        if (!ret)
                return NIC_FAIL; 
	
	return NIC_SUCCESS;
}

//=====================================================
//      set dns into uLinux.conf for DHCP Server -- added by YFHuang 20040421
//=====================================================
int NIC_Set_DNS_DHCP(int index, char *ipaddr) 
{
	int	ret;
	char	buf[BUF_SIZE];


	if (index > NIC_MAC_DNS_COUNT) return NIC_FAIL;
	
	if (NIC_Is_Support_Router())
        	ret=WriteProfileString("Network", SZ_DNS_FIELD, ipaddr);
        else {
        	if (index > NIC_MAC_DNS_COUNT) return NIC_FAIL;

        	sprintf(buf, "Domain Name Server %d", index);
		
        	ret = WriteProfileString(DHCP_SECTION, buf, ipaddr);
        }

        if (!ret)
                return NIC_FAIL;
	
	return NIC_SUCCESS;
}

//=====================================================
//      get dns from uLinux.conf
//=====================================================
int NIC_Get_DNS_Ex(char *ipaddr_list, int buf_size, int array_size)
{
	char	buf[NIC_IPADDR_LEN];
	int	i, cnt=0;

	if (ipaddr_list == NULL || array_size <= 0 || buf_size <=0)
		return NIC_FAIL;

	if (NIC_Is_Support_Router()) {
		GetProfileString("Network", SZ_DNS_FIELD, "0.0.0.0", buf, NIC_IPADDR_LEN);
		if ((strlen(buf)+1)>buf_size)
			return NIC_FAIL;

		memcpy(ipaddr_list, buf, (strlen(buf)+1) * sizeof(char));
		cnt++;
	}
	else {//if (NIC_Is_DHCP_Server_Enabled()) { // Catherine -- may be removed later???
		char field_name[BUF_SIZE];

		for (i=0; i<NIC_MAC_DNS_COUNT && i<array_size; i++) {
			sprintf(field_name, "Domain Name Server %d", i+1);
			GetProfileString("Network", field_name, "0.0.0.0",
						buf, NIC_IPADDR_LEN);

			if ((strlen(buf)+1)>buf_size)
				return NIC_FAIL;

			memcpy(ipaddr_list + i * buf_size * sizeof(char), buf,
				(strlen(buf)+1) * sizeof(char));
			cnt++;
		}
	}

	return cnt;
}

int NIC_Is_WINS_Enabled()
{
	char	buf[NIC_IPADDR_LEN];

	GetProfileString("Network", SZ_WINS_ENABLE_FIELD, "False", buf, NIC_IPADDR_LEN);
	if (!strcasecmp(buf, "TRUE"))
		return 1;
	else
		return 0;
}

int NIC_Is_Support_NAT()
{
        char    buf[80];

        GetProfileString("NAT", "Support", "False", buf, 80);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

int NIC_Is_Support_Router()
{
	int	num;
	num=NIC_Count_Interface();
	if (!NIC_Is_Bonding_Support() && num==2 && NIC_Is_Support_NAT())
		return 1;
	else
		return 0;
}

int NIC_Enable_WINS(int bEnable)
{
	if (bEnable)
		WriteProfileString("Network", SZ_WINS_ENABLE_FIELD, "TRUE");
	else {
		WriteProfileString("Network", SZ_WINS_ENABLE_FIELD, "False");
		set_samba_WINS_server("");
	}
	return NIC_SUCCESS;
}

//=====================================================
//      set wins into uLinux.conf and smb.conf
//=====================================================
int NIC_Set_WINS(char *ipaddr)
{
        int             ret;

        ret=WriteProfileString("Network", SZ_WINS_FIELD, ipaddr);
        if (!ret)
                return NIC_FAIL;

	if (NIC_Is_WINS_Enabled())
	{
		if (!strcmp(ipaddr, "0.0.0.0"))
			ipaddr[0]=0x0;
		set_samba_WINS_server(ipaddr);
	}
	else
		set_samba_WINS_server("");
	/* do we need to restart samba? */
	/* might cause all connections broke */
        return NIC_SUCCESS;
}

//=====================================================
//      get wins from uLinux.conf
//=====================================================
int NIC_Get_WINS(char *ipaddr, int len)
{
        char    buf[NIC_IPADDR_LEN];

        GetProfileString("Network", SZ_WINS_FIELD, "0.0.0.0", buf, NIC_IPADDR_LEN);
        if ((strlen(buf)+1)>len)
                return NIC_FAIL;
        strcpy(ipaddr, buf);
	return NIC_SUCCESS;
}

int NIC_Set_WINS_Support(int bEnable)
{
        int     ret;

        if (bEnable)
                ret=WritePrivateProfileString("global", "wins support", "yes", SAMBA_CONF_PATH);
        else
                ret=WritePrivateProfileString("global", "wins support", "no", SAMBA_CONF_PATH);
        ret=WriteProfileString("Network", SZ_WINS_SUPPORT_FIELD, BOOL2STR(bEnable));
        // restart samba to make the change effective...
        return ret;
}

int NIC_Get_WINS_Support()
{
        int     ret;
        char    buf[80];

        ret=GetProfileString("Network", SZ_WINS_SUPPORT_FIELD, "FALSE", buf, 80);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

//=====================================================
//      does the nic start
//	you can specify type which you want to get
//	like NIC_WAN or NIC_LAN01
//=====================================================
int NIC_Is_Started(int type)
{
        int		ret;
	char		command[NIC_LINE];
	char		dev[NIC_DEV_LEN];

        if (	type==NIC_WAN_DEFAULT || type==NIC_LAN01_DEFAULT || type==NIC_LAN02_DEFAULT ||
		type==NIC_LAN03_DEFAULT || type==NIC_LAN04_DEFAULT)
                NIC_Get_Default_Dev(dev, NIC_DEV_LEN, type);
        else
                NIC_Get_Dev(dev, NIC_DEV_LEN, type);
	sprintf(command, "/sbin/ifconfig %s 2>/dev/null 1>/dev/null", dev);
	ret=system(command);
        return (ret==0);
}

//=====================================================
//      restart network
//=====================================================
int NIC_Restart()
{
	int	ret;
	
	ret=system("/sbin/setcfg eth0 MTU 1500"); //add by KenChen 20060330
	sleep(1);
	ret=system("/etc/init.d/network.sh restart lcdrefresh 2>/dev/null 1>/dev/null");
	return ret;
}

//=====================================================
//      get nic hardware address
//	you can specify type which you want to get
//	link NIC_WAN or NIC_LAN01
//=====================================================
/*int NIC_Get_HWADDR(char *hwaddr, int addr_len, int type)
{
	NIC_INFO	nic_info;

	if (addr_len<NIC_HWADDR_LEN)
		return NIC_FAIL;
	if (type==NIC_WAN)
		type=NIC_WAN_DEV;
	NIC_Get_Info(&nic_info, type);
        strcpy(hwaddr, nic_info.hwaddr);
        return NIC_SUCCESS;
}*/
int NIC_Get_HWADDR(char *hwaddr, int addr_len, int type)
{
        NIC_INFO        nic_info;

        hwaddr[0]=0x0;
        if (addr_len<NIC_HWADDR_LEN)
                return NIC_FAIL;

        NIC_Get_Info(&nic_info, type);
        strcpy(hwaddr, nic_info.hwaddr);
        return NIC_SUCCESS;
}

int NIC_Get_Default_HWADDR(char *hwaddr, int addr_len, int type)
{
        NIC_TCPIP       tcpip;
        NIC_INFO        nic_info;

        hwaddr[0]=0x0;
        if (addr_len<NIC_HWADDR_LEN)
                return NIC_FAIL;
        NIC_Get_Tcpip(&tcpip);
        if (type==NIC_WAN || type==NIC_WAN_DEFAULT)
                GetProfileString(tcpip.wan, SZ_HWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
        else
        if (type==NIC_LAN01 || type==NIC_LAN01_DEFAULT)
                GetProfileString(tcpip.lan01, SZ_HWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
	else
	if (type==NIC_LAN02 || type==NIC_LAN02_DEFAULT)
		GetProfileString(tcpip.lan02, SZ_HWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
	else
	if (type==NIC_LAN03 || type==NIC_LAN03_DEFAULT)
		GetProfileString(tcpip.lan03, SZ_HWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
	else
	if (type==NIC_LAN04 || type==NIC_LAN04_DEFAULT)
		GetProfileString(tcpip.lan04, SZ_HWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
        if (!strcmp(hwaddr, "00:00:00:00:00:00") || hwaddr[0]==0x0)
        {
                if (type==NIC_WAN)
                        type=NIC_WAN_DEFAULT;
		else
                if (type==NIC_LAN01)
                        type=NIC_LAN01_DEFAULT;
		else
		if (type==NIC_LAN02)
			type=NIC_LAN02_DEFAULT;
		else
		if (type==NIC_LAN03)
			type=NIC_LAN03_DEFAULT;
		else
		if (type==NIC_LAN04)
			type=NIC_LAN04_DEFAULT;
                NIC_Get_Info(&nic_info, type);
                strcpy(hwaddr, nic_info.hwaddr);
                if (type==NIC_WAN_DEFAULT)
                        WriteProfileString(tcpip.wan, SZ_HWADDR_FIELD, nic_info.hwaddr);
                else
                if (type==NIC_LAN01_DEFAULT)
                        WriteProfileString(tcpip.lan01, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN02_DEFAULT)
			WriteProfileString(tcpip.lan02, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN03_DEFAULT)
			WriteProfileString(tcpip.lan03, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN04_DEFAULT)
			WriteProfileString(tcpip.lan04, SZ_HWADDR_FIELD, nic_info.hwaddr);
        }
        return NIC_SUCCESS;
}

//--------------------------------------------------------
//	this function set the real hwaddr in uLinux.conf
//	just set it in the first time
//--------------------------------------------------------
int NIC_Set_HWADDR(int type)
{
        NIC_TCPIP       tcpip;
        NIC_INFO        nic_info;
        char            buf[100];

        buf[0]=0x0;
        NIC_Get_Default_HWADDR(buf, 100, type);
        if (!strcmp(buf, "00:00:00:00:00:00") || buf[0]==0x0)
        {
                NIC_Get_Tcpip(&tcpip);
                if (type==NIC_WAN)
                        type=NIC_WAN_DEFAULT;
		else
                if (type==NIC_LAN01)
                        type=NIC_LAN01_DEFAULT;
		else
		if (type==NIC_LAN02)
			type=NIC_LAN02_DEFAULT;
		else
		if (type==NIC_LAN03)
			type=NIC_LAN03_DEFAULT;
		else
		if (type==NIC_LAN04)
			type=NIC_LAN04_DEFAULT;
                NIC_Get_Info(&nic_info, type);
                if (type==NIC_WAN || NIC_WAN_DEFAULT)
                        WriteProfileString(tcpip.wan, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
                if (type==NIC_LAN01 || type==NIC_LAN01_DEFAULT)
                        WriteProfileString(tcpip.lan01, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN02 || type==NIC_LAN02_DEFAULT)
			WriteProfileString(tcpip.lan02, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN03 || type==NIC_LAN03_DEFAULT)
			WriteProfileString(tcpip.lan03, SZ_HWADDR_FIELD, nic_info.hwaddr);
		else
		if (type==NIC_LAN04 || type==NIC_LAN04_DEFAULT)
			WriteProfileString(tcpip.lan04, SZ_HWADDR_FIELD, nic_info.hwaddr);
        }
        return NIC_SUCCESS;
}

//===================================================================
//
//	below functions are about pppoe setting in uLinux.conf
//
//==================================================================

//===================================================
//	write to pap-secrets and chap-srcrets for pppd
//===================================================
int nic_write_pppd(char *user, char *passwd)
{
	FILE		*fptr;
	NIC_NAS_INFO	nic_nas_info;

	NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_WAN_DEFAULT);
	if (nic_nas_info.usage==NIC_USE_PPPOE)
	{
		fptr=fopen(PPPD_PAP_CONF, "wb");
		fprintf(fptr, "\"%s\"	*	\"%s\"\n", user, passwd);
		fclose(fptr);
        	fptr=fopen(PPPD_CHAP_CONF, "wb");
	        fprintf(fptr, "\"%s\"   *       \"%s\"\n", user, passwd);
	        fclose(fptr);
	}
	return NIC_SUCCESS;
}

int NIC_Is_Support_PPPOE()
{
	char	buf[256];

        GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_SUPPORT_FIELD, "False", buf, 256);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

//========================================================
//	write pppoe section to uLinux.conf
//========================================================
int write_tag_to_conf(char *filename, char *tag, char *val)
{
	char	command[1024], tmp[255]={"tmpXXXXXX"}, tmpfile[255], backup[1024];
	FILE	*fptr, *fptr1;
	char	line[1024], *name=NULL, *value=NULL;

	mktemp(tmp);
	sprintf(tmpfile, "/tmp/%s", tmp);
	sprintf(command, "/bin/cp %s %s 2>/dev/null 1>/dev/null", filename, tmpfile);
	system(command);
	if ((fptr=fopen(tmpfile, "rb"))==NULL)
		return NIC_FAIL;
	fptr1=fopen(filename, "w");
	while (!feof(fptr))
	{
		if (fgets(line, 1024, fptr)!=NULL)
		{
			strcpy(backup, line);
			name=strtok(line, "=");
			value=strtok(NULL, "\t\n");
			if (!strcmp(tag, name))
			{	/* match tag */
				fprintf(fptr1, "%s=%s\n", name, val);
			}
			else
			{
				fprintf(fptr1, "%s", backup);
			}
		}
	}
	fclose(fptr);
	fclose(fptr1);
	sprintf(command, "/bin/rm %s -f 2>/dev/null 1>/dev/null", tmpfile);
	system(command);
	return NIC_SUCCESS;
}

int NIC_Set_PPPOE(PPPOE *pppoe)
{
	char	buf[255];

	/* write connect_on_boot */
        if (pppoe->connect_on_boot==PPPOE_CONNECT_BOOT)
                WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_CONNECT_FIELD, "TRUE");
        else
                WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_CONNECT_FIELD, "FALSE");

	/* write idle time */
        sprintf(buf, "%d", pppoe->idle);
        WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_IDLE_TIME_FIELD, buf);

	/********************************/
	/* write keep_alive		*/
	/* also write to pppoe.conf	*/
	/********************************/
        if (pppoe->keep_alive)
	{
		WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_KEEP_ALIVE_FIELD, "TRUE");
		write_tag_to_conf(PPPOE_CONF, "DEMAND", "no");
	}
        else
	{
		WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_KEEP_ALIVE_FIELD, "FALSE");
		write_tag_to_conf(PPPOE_CONF, "DEMAND", buf);
	}

	/* fail to use dhcp */
	if (pppoe->fail_use_dhcp)
	{
		WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_FAIL_USE_DHCP_FIELD, "TRUE");
	}
	else
	{
		WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_FAIL_USE_DHCP_FIELD, "FALSE");
	}

	/****************************************************************/
	/* write username & pwd 					*/
	/* also write to pppoe.conf & pap-secrets & chap-secrets	*/
	/****************************************************************/
	WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_USERNAME_FIELD, pppoe->username);
	WriteProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_PASSWORD_FIELD, pppoe->passwd);
	sprintf(buf, "'%s'", pppoe->username);
	write_tag_to_conf(PPPOE_CONF, "USER", buf);
	nic_write_pppd(pppoe->username, pppoe->passwd);
	return NIC_SUCCESS;
}

//=======================================================
//	read pppoe section from uLinux.conf
//=======================================================
int NIC_Get_PPPOE(PPPOE *pppoe)
{
	char	buf[256];

	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_USERNAME_FIELD, "", pppoe->username, 256);
	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_PASSWORD_FIELD, "", pppoe->passwd, 256);
	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_CONNECT_FIELD, "FALSE", buf, 256);
	if (!strcasecmp(buf, "TRUE"))
		pppoe->connect_on_boot=1;
	else
		pppoe->connect_on_boot=0;
	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_KEEP_ALIVE_FIELD, "", buf, 256);
	if (!strcasecmp(buf, "TRUE"))
		pppoe->keep_alive=1;
	else
		pppoe->keep_alive=0;
	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_IDLE_TIME_FIELD, "300", buf, 256);
	pppoe->idle=atoi(buf);
	GetProfileString(SZ_PPPOE_SECTION, SZ_PPPOE_FAIL_USE_DHCP_FIELD, "FALSE", buf, 256);
	if (!strcasecmp(buf, "TRUE"))
		pppoe->fail_use_dhcp=1;
	else
		pppoe->fail_use_dhcp=0;
	return NIC_SUCCESS;
}

int NIC_Get_PPPOE_Latest_Error()
{
        FILE    *fptr;
        char    buf[80];
        int     status;

        if ((fptr=fopen("/var/pppd.error", "rb"))!=NULL)
        {
                fgets(buf, 80, fptr);
                status=atoi(buf);
                fclose(fptr);
                return status;
        }
        else
                return NIC_SUCCESS;

}

int NIC_Get_PPPOE_Status()
{
//	int		pppoe, pppd, adsl_connect;
	int		pppoe;
        NIC_INFO        nic_info;
        char            dev[NIC_DEV_LEN];
        char            command[255];

	if (NIC_Is_WAN_Use_Tmp_DHCP())
		return PPPOE_STATUS_DHCP;

	pppoe=system("/sbin/pidof pppoe 2>/dev/null 1>/dev/null");
	NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
	NIC_Get_Info(&nic_info, NIC_WAN);
	sprintf(command, "/sbin/ifconfig ppp0 2>/dev/null 1>/dev/null");
	if (system(command)!=0)
		return PPPOE_STATUS_DISABLE;
	else
	if (nic_info.ipaddr[0]==0x0)
		return PPPOE_STATUS_CONNECTING;
	else
	if (!strcmp(nic_info.ipaddr, "10.112.112.112"))
	{
		if (pppoe==0)
			return PPPOE_STATUS_CONNECTING;
		else
			return PPPOE_STATUS_CONNECT;
	}
	else
		return PPPOE_STATUS_READY;
/*
	pppoe=system("/sbin/pidof adsl-connect 2>/dev/null 1>/dev/null");
	adsl_connect=system("/sbin/pidof pppoe 2>/dev/null 1>/dev/null");
	pppd=system("/sbin/pidof pppd 2>/dev/null 1>/dev/null");
	if (adsl_connect==0 && pppoe!=0)
		return PPPOE_STATUS_CONNECTING;
	else
	if (adsl_connect!=0 && pppd!=0)
		return PPPOE_STATUS_DISABLE;
	else
		return PPPOE_STATUS_CONNECT;*/
}

int NIC_Restart_PPPOE()
{
//	system("/usr/sbin/adsl-stop 2>/dev/null 1>/dev/null");
//	system("/usr/sbin/adsl-start 2>/dev/null 1>/dev/null");
	system("/etc/init.d/network.sh start wan 2>/dev/null 1>/dev/null");
	return NIC_SUCCESS;
}

int NIC_Start_PPPOE()
{
	if (NIC_Get_PPPOE_Status()==PPPOE_STATUS_READY)
		return NIC_SUCCESS;
	NIC_Restart_PPPOE();
	return NIC_SUCCESS;
}

int NIC_Stop_PPPOE()
{
//	system("/usr/sbin/adsl-stop 2>/dev/null 1>/dev/null");
	system("/etc/init.d/network.sh stop wan 2>/dev/null 1>/dev/null");
	return NIC_SUCCESS;
}

//===============================================================
// below functions are PPTP API
//================================================================
int nic_write_pppd_options(char *user, char *passwd)
{
        FILE    	*fptr;
	NIC_NAS_INFO	nic_nas_info;

	NIC_Get_Info_From_uLinux(&nic_nas_info, NIC_WAN_DEFAULT);
	if (nic_nas_info.usage==NIC_USE_PPTP)
	{
		/* write to /ppp/pap-secrets */
        	fptr=fopen(PPPD_PAP_CONF, "wb");
	        fprintf(fptr, "\"%s\"   *       \"%s\"\n", user, passwd);
        	fclose(fptr);

		/* write to /ppp/chap-secrets */
        	fptr=fopen(PPPD_CHAP_CONF, "wb");
	        fprintf(fptr, "\"%s\"   *       \"%s\"\n", user, passwd);
        	fclose(fptr);

		/* write to /ppp/options */
		fptr=fopen(PPPD_OPTIONS_CONF, "wb");
		fprintf(fptr, "name \"%s\"\n", user);
		fprintf(fptr, "noauth\n");
		fprintf(fptr, "noipdefault\n");
		fprintf(fptr, "usepeerdns\n");
		fprintf(fptr, "defaultroute\n");
		fprintf(fptr, "mtu 1492\n");
		fprintf(fptr, "mru 1492\n");
		fclose(fptr);
	}
        return NIC_SUCCESS;
}

int NIC_Is_Support_PPTP()
{
        char    buf[256];

        GetProfileString(SZ_PPPOE_SECTION, SZ_PPTP_SUPPORT_FIELD, "False", buf, 256);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

//=======================================================
//      read pptp section from uLinux.conf
//=======================================================
int NIC_Get_PPTP(PPTP *pptp)
{
        char    buf[256];

        GetProfileString(SZ_PPTP_SECTION, SZ_PPTP_USERNAME_FIELD, "", pptp->username, 256);
        GetProfileString(SZ_PPTP_SECTION, SZ_PPTP_PASSWORD_FIELD, "", pptp->passwd, 256);
	GetProfileString(SZ_PPTP_SECTION, SZ_PPTP_SERVER_FIELD, "0.0.0.0", pptp->serveraddr, NIC_IPADDR_LEN);
        GetProfileString(SZ_PPTP_SECTION, SZ_PPPOE_KEEP_ALIVE_FIELD, "", buf, 256);
        if (!strcasecmp(buf, "TRUE"))
                pptp->keep_alive=1;
        else
                pptp->keep_alive=0;
        GetProfileString(SZ_PPTP_SECTION, SZ_PPTP_IDLE_TIME_FIELD, "300", buf, 256);
        pptp->idle=atoi(buf);
        GetProfileString(SZ_PPTP_SECTION, SZ_PPTP_FAIL_USE_DHCP_FIELD, "FALSE", buf, 256);
	if (!strcasecmp(buf, "TRUE"))
		pptp->fail_use_dhcp=1;
	else
	        pptp->fail_use_dhcp=0;
        return NIC_SUCCESS;
}

//====================================================================
//      src = 0.1.2.3
//      pos is below
//      0 : ip1
//      1 : ip2
//      2 : ip3
//      3 : ip4
//====================================================================
char *nic_parser_ipaddr(char *dest, const char *src, int pos)
{
        char *ptr;

        strcpy(dest, src);
        ptr=strtok(dest, ".");
        if (pos==0)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==1)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==2)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==3)
                return ptr;
        return NULL;
}

int NIC_Set_PPTP(PPTP *pptp)
{
        char    buf[255];
	char	local[256];
	int	ip[4];

        /* write idle time */
        sprintf(buf, "%d", pptp->idle);
        WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_IDLE_TIME_FIELD, buf);

        /********************************/
        /* write keep_alive             */
	/* also write to pppoe.conf     */
        /********************************/
        if (pptp->keep_alive)
                WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_KEEP_ALIVE_FIELD, "TRUE");
        else
                WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_KEEP_ALIVE_FIELD, "FALSE");

        /************************************************************************/
        /* write username & pwd                                         	*/
        /* also write to pppoe.conf & pap-secrets & chap-secrets &options       */
        /************************************************************************/
        WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_USERNAME_FIELD, pptp->username);
        WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_PASSWORD_FIELD, pptp->passwd);
        sprintf(buf, "'%s'", pptp->username);
        nic_write_pppd_options(pptp->username, pptp->passwd);

	/* write server address */
	WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_SERVER_FIELD, pptp->serveraddr);

	/* write local address */
	ip[0]=atoi(nic_parser_ipaddr(local, pptp->serveraddr, 0));
	ip[1]=atoi(nic_parser_ipaddr(local, pptp->serveraddr, 1));
	ip[2]=atoi(nic_parser_ipaddr(local, pptp->serveraddr, 2));
	ip[3]=atoi(nic_parser_ipaddr(local, pptp->serveraddr, 3));
	if (ip[3]!=141)
		ip[3]=141;
	else
		ip[3]=142;
	sprintf(local, "%d.%d.%d.%d", ip[0], ip[1], ip[2], ip[3]);
	WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_LOCAL_FIELD, local);
	if (pptp->fail_use_dhcp)
		WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_FAIL_USE_DHCP_FIELD, "TRUE");
	else
		WriteProfileString(SZ_PPTP_SECTION, SZ_PPTP_FAIL_USE_DHCP_FIELD, "FALSE");
	return NIC_SUCCESS;
}

int NIC_Get_PPTP_Status()
{
        int             pptp;
        NIC_INFO        nic_info;
        char            dev[NIC_DEV_LEN];
        char            command[255];

	if (NIC_Is_WAN_Use_Tmp_DHCP())
		return PPTP_STATUS_DHCP;

        pptp=system("/sbin/pidof pptp 2>/dev/null 1>/dev/null");
        NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
        NIC_Get_Info(&nic_info, NIC_WAN);
        sprintf(command, "/sbin/ifconfig ppp0 2>/dev/null 1>/dev/null");
        if (system(command)!=0)
                return PPTP_STATUS_DISABLE;
        else
        if (nic_info.ipaddr[0]==0x0)
                return PPTP_STATUS_CONNECTING;
        else
                return PPTP_STATUS_READY;
}

int NIC_Restart_PPTP()
{
//        system("/etc/init.d/pptp.sh stop 2>/dev/null 1>/dev/null");
//        system("/etc/init.d/pptp.sh start 2>/dev/null 1>/dev/null");
	system("/etc/init.d/network.sh start wan 2>/dev/null 1>/dev/null");
        return NIC_SUCCESS;
}

int NIC_Start_PPTP()
{
        if (NIC_Get_PPTP_Status()==PPTP_STATUS_READY)
                return NIC_SUCCESS;
        NIC_Restart_PPTP();
        return NIC_SUCCESS;
}

int NIC_Stop_PPTP()
{
//        system("/usr/sbin/ppto.sh stop 2>/dev/null 1>/dev/null");
	system("/etc/init.d/network.sh stop wan 2>/dev/null 1>/dev/null");
        return NIC_SUCCESS;
}

//================================================
//	below is DHCP functions
//=================================================
int NIC_Restart_DHCP(int type)
{
	if (type==NIC_WAN || NIC_WAN_DEFAULT)
		system("/etc/init.d/network.sh start wan 2>/dev/null 1>/dev/null");
	else
		system("/etc/init.d/network.sh start lan 2>/dev/null 1>/dev/null");
	return NIC_SUCCESS;
}

int NIC_Start_DHCP(int type)
{
	if (type==NIC_WAN || NIC_WAN_DEFAULT)
		system("/etc/init.d/network.sh start wan 2>/dev/null 1>/dev/null");
	else
		system("/etc/init.d/network.sh start lan 2>/dev/null 1>/dev/null");
	return NIC_SUCCESS;
}

int NIC_Stop_DHCP(int type)
{
	if (type==NIC_WAN || NIC_WAN_DEFAULT)
		system("/etc/init.d/network.sh stop wan 2>/dev/null 1>/dev/null");
	else
		system("/etc/init.d/network.sh stop lan 2>/dev/null 1>/dev/null");
	return NIC_SUCCESS;
}

int NIC_Is_WAN_Use_Tmp_DHCP()
{
	char	buf[256];

	GetProfileString("Network", SZ_WAN_DHCP_FIELD, "FALSE", buf, sizeof(buf));
	if (!strcasecmp(buf, "TRUE"))
		return 1;
	else
		return 0;
}

int NIC_Get_DHCP_Status(int type)
{
	NIC_INFO	nic_info;
	NIC_NAS_INFO	nic_nas_info;

	NIC_Get_Info_From_uLinux(&nic_nas_info, type);
	NIC_Get_Info(&nic_info, type);

	if (nic_nas_info.usage!=NIC_USE_DHCP)
		return DHCP_STATUS_DISABLE;
	if (nic_nas_info.usage==NIC_USE_DHCP && (!strcmp(nic_info.ipaddr, "127.0.0.1") || nic_info.ipaddr[0]==0x0))
		return DHCP_STATUS_CONNECTING;
	return DHCP_STATUS_READY;
}

//Shone ported from check_ethernet_link() in NasWare4/NasDriver/nasmgr/app/hwmond/hwmond.c 2005,04,18
int eth_port_link(type)
{
	char iflist[MAX_IF_NUMBER][20];
	//int bonding_cnt=0, bonding_nic[3]={0, 0, 0};
	int ret, linkup,eth_port=-1;	
	struct ifreq ifr;
	int fd;	
	struct ethtool_value ethval;
	char tmp1[BUF_SIZE], tmp2[BUF_SIZE];
#if __MII_STATUS__
	struct mii_ioctl_data *miiptr;
	int mii_find=0;
#endif	

	switch(type)
	{
	   case NIC_WAN :
	   case NIC_WAN_DEFAULT:
	   case NIC_LAN01:
	   case NIC_LAN01_DEFAULT:
	   	    eth_port=0;
	           break;
	           
	   case NIC_LAN02:
	   case NIC_LAN02_DEFAULT:
	   	    eth_port=1;
	           break;
	           
	   case NIC_LAN03:
	   case NIC_LAN03_DEFAULT:
	   	    eth_port=2;
	           break; 	           	           		   	
	}   
	
	fd = socket(AF_INET, SOCK_DGRAM, 0);
	if (fd < 0) 
	{
		perror("Cannot get control socket");
		return -1;
	}

	if (eth_port<NIC_Count_Interface())
	{
		linkup = 0;
		mii_find=0;
		sprintf(tmp1, "interface%d", eth_port+1);
		Conf_Get_Field("/etc/config/uLinux.conf", "network", tmp1, tmp2, sizeof(tmp2));
		strcpy(iflist[type], tmp2);
		if (strstr(iflist[type], "eth"))
		  {		
			strcpy(ifr.ifr_name, iflist[type]);
#if __MII_STATUS__
			// MII link status		
			miiptr = (struct mii_ioctl_data *)&ifr.ifr_data;

                	/*
                	 * We cannot assume that SIOCGMIIPHY will also read a
                 	* register; not all network drivers (e.g., e100)
                 	* support that.
                 	*/

			if ((ret=ioctl(fd, SIOCGMIIPHY, &ifr)) == 0)
		   	{
				miiptr->reg_num = MII_BMSR;
				if (ioctl(fd, SIOCGMIIREG, &ifr) == 0)
				{
					linkup = (miiptr->val_out & BMSR_LSTATUS) ? 1:0;
					mii_find=1;

				}
		   	}	
#endif	
		   
                	/* try SIOCETHTOOL ioctl, some drivers cache ETHTOOL_GLINK */
                	/* for a period of time so we attempt to get link status   */
                	/* from it last if the above MII ioctls fail...            */
			if (mii_find==0)
		   	{

				ifr.ifr_data = (char*) &ethval;
				ethval.cmd = ETHTOOL_GLINK;
				ret = ioctl(fd, SIOCETHTOOL, &ifr);
				if (ret!=0) 
				{
					//continue;
					return 0;
				}
				linkup = (ethval.data) ? 1 : 0;
		    	}				
		}	
		return linkup;
	}	
	return 0;	
}

int NIC_Get_Status(int type)
{
	NIC_INFO	nic_info;
	int		status;
	NIC_NAS_INFO	nic_nas_info;
	char                      tmp[80];

	NIC_Get_Info_From_uLinux(&nic_nas_info, type);
	NIC_Get_Info(&nic_info, type);

	/* check if connecting, must check pppoe & pptp & pptp */ 
	if (nic_nas_info.usage==NIC_USE_DHCP && (!strcmp(nic_info.ipaddr, "127.0.0.1") || nic_info.ipaddr[0]==0x0))
		return NIC_CONNECTING;

	status=NIC_Get_PPPOE_Status();
	if (status==PPPOE_STATUS_CONNECTING)
		return NIC_CONNECTING;

	status=NIC_Get_PPTP_Status();
	if (status==PPTP_STATUS_CONNECTING)
		return NIC_CONNECTING;

	if (!strcmp(nic_info.ipaddr, "127.0.0.1") || !strcmp(nic_info.ipaddr, "10.112.112.112"))
		return NIC_DISCONNECT;
	if (nic_info.ipaddr[0]==0x0)//This one seems to be unworkable, not knowing why
		return NIC_DISCONNECT;
	Conf_Get_Field("/etc/config/uLinux.conf","System","Model",tmp,80);
	//NAS-2100 has only one lan port, and it involves something called MII that I don't understand, and eth_port_link() will return 0
	// for the MII linkage check while the network is linking somehow, so I'll turn this checkment off for NAS-2100 before I got the clue.
	//Shone 2005,04,28			
	/*if ((eth_port_link(type)==0) && (strcmp(tmp,"NAS-2100") && strcmp(tmp, "ND-23000")) ) //Shone added to have further check of connection status 2005,04,18	
		return NIC_DISCONNECT;	*/
	return NIC_CONNECT;
}

//===========================================
//	Setup Virtual HWADDR
//	return:
//		NIC_FAIL => input error
//		NIC_SUCCESS => success
//============================================
int NIC_Set_VHWADDR(char *hwaddr, int type)
{
	NIC_TCPIP	tcpip;
	int		i;

	for (i=0;i<17;i++)
	{
		if (i==2 || i==5 || i==8 || i==11 || i==14)
		{
			if (hwaddr[i]!=':')
				return NIC_FAIL;
		}
		else
		{
			if (	(hwaddr[i]>='0' && hwaddr[i]<='9') ||
				(hwaddr[i]>='a' && hwaddr[i]<='f') ||
				(hwaddr[i]>='A' && hwaddr[i]<='F'))
			{
			}
			else
				return NIC_FAIL;
		}
	}
	NIC_Get_Tcpip(&tcpip);
	if (type==NIC_WAN || type==NIC_WAN_DEFAULT)
		WriteProfileString(tcpip.wan, SZ_VHWADDR_FIELD, hwaddr);
	if (type==NIC_LAN01)
		WriteProfileString(tcpip.lan01, SZ_VHWADDR_FIELD, hwaddr);
	return NIC_SUCCESS;
}

//=====================================================
//      get nic hardware address
//      you can specify type which you want to get
//      link NIC_WAN or NIC_LAN01
//=====================================================
int NIC_Get_VHWADDR(char *hwaddr, int addr_len, int type)
{
	NIC_TCPIP	tcpip;

        if (addr_len<NIC_HWADDR_LEN)
                return NIC_FAIL;
	NIC_Get_Tcpip(&tcpip);
	if (type==NIC_WAN)
		GetProfileString(tcpip.wan, SZ_VHWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
        if (type==NIC_LAN01)
		GetProfileString(tcpip.lan01, SZ_VHWADDR_FIELD, "00:00:00:00:00:00", hwaddr, addr_len);
        return NIC_SUCCESS;
}

int NIC_Enable_VHWADDR(int bEnable, int type)
{
	NIC_TCPIP	tcpip;

	NIC_Get_Tcpip(&tcpip);
	if (type==NIC_WAN)
	{
		if (bEnable)
			WriteProfileString(tcpip.wan, SZ_VHWADDR_ENABLE_FIELD, "TRUE");
		else
			WriteProfileString(tcpip.wan, SZ_VHWADDR_ENABLE_FIELD, "FALSE");
	}
	else
	{
		if (bEnable)
			WriteProfileString(tcpip.lan01, SZ_VHWADDR_ENABLE_FIELD, "TRUE");
		else
			WriteProfileString(tcpip.lan01, SZ_VHWADDR_ENABLE_FIELD, "FALSE");
	}
	return NIC_SUCCESS;
}

//  add by Tiger
int NIC_Is_VHWADDR_Enabled(int type)
{
	char	buf[NIC_IPADDR_LEN];
	NIC_TCPIP	tcpip;

	NIC_Get_Tcpip(&tcpip);

	if (type == NIC_WAN)
		GetProfileString(tcpip.wan, SZ_VHWADDR_ENABLE_FIELD, "False", buf, NIC_IPADDR_LEN);
	else
		GetProfileString(tcpip.lan01, SZ_VHWADDR_ENABLE_FIELD, "False", buf, NIC_IPADDR_LEN);

	if (!strcasecmp(buf, "TRUE"))
		return 1;
	else
		return 0;
}

int NIC_Is_DHCP_Server_Enabled()
{
	char	buf[BUF_SIZE];

	if (GetProfileString(DHCP_SECTION, "Enable", "False", buf, BUF_SIZE) < 0)
		return 0;
	if (!strcasecmp(buf, "TRUE"))
		return 1;
	else
		return 0;
}

int str_to_dec (char * str)
{
	int len = strlen (str), msb = 0, lsb = 0;
	
	if (len < 2)
		return 0;

	if (str[0] == ':')
		return 0;
	else
	{
		if (str[0] >= '0' && str[0] <= '9')
			msb = (int)str[0] - 48;
		else if (str[0] >= 'A' && str[0] <= 'F')
			msb = (int)str[0] - 55;
		else if (str[0] >= 'a' && str[0] <= 'f')
			msb = (int)str[0] - 87;
	}

	if (str[1] == ':')
		return 0;
	else
	{
		if (str[1] >= '0' && str[1] <= '9')
			lsb = (int)str[1] - 48;
		else if (str[1] >= 'A' && str[1] <= 'F')
			lsb = (int)str[1] - 55;
		else if (str[1] >= 'a' && str[1] <= 'f')
			lsb = (int)str[1] - 87;
	}
	return msb * 16 + lsb;
}

int NIC_Parse_HWADDR (char * mac_addr, int index) //  return value is 10'
{
	char * ptr = mac_addr;

	if (!strcmp (mac_addr, ""))
		return 0;

	while (--index)
	{
		ptr = (char *)strchr (ptr, ':');
		if (ptr)
			ptr += 1;
		else
			return 0;
	}
	return str_to_dec (ptr);
}

/////////////////////////////////////////////////////////////////////////////
//	Add by Egbert Chen in 2002/02/19

//===========================================================================
//	int Is_WLAN_Supported();
//		No parameter
//		return	WLAN_UNSUPPORTED:0 No wireless LAN supported
//				WLAN_SUPPORTED:1 support wireless LAN
//===========================================================================	
int Is_WLAN_Supported()
{
	char szBuffer[8]={0};

	GetProfileString(WLAN_SECTION, WLAN_KEY_SUPPORT, "FALSE", szBuffer, 8);

	return IS_TRUE(szBuffer);
}

//===========================================================================
//	int IsWLANEnabled();
//		No parameter
//		return	WLAN_DISABLED:0  Wireless LAN function is disabled
//				WLAN_ENABLED:1  Wireless LAN function is enabled
//===========================================================================	
int Is_WLAN_Enabled()
{
	char szBuffer[8]={0};

	if(!Is_WLAN_Supported())
		return FALSE;

	GetProfileString(WLAN_SECTION, WLAN_KEY_ENABLE, "TRUE", szBuffer, 8);

	return IS_TRUE(szBuffer);
}

//===========================================================================
//	int Get_WLAN_Data(PWLAN pwlan);
//		parameters:
//			pwlan, type: PWLAN; point to WLAN structure to get the data of wlan; 
//		return	0 failure
//				1 success
//===========================================================================	
int Get_WLAN_Data(PWLAN pwlan)
{
	char	szBuffer[WLAN_STRBUFF_SIZE]={0};
	int		i;
	char	szValue[8]={0};

	if(pwlan==NULL)
		return FALSE;

	pwlan->nEnabled=Is_WLAN_Enabled();

	GetProfileString(WLAN_SECTION, WLAN_KEY_SSID, "", szBuffer, WLAN_STRBUFF_SIZE);
	strcpy(pwlan->szSSID, szBuffer);

	GetProfileString(WLAN_SECTION, WLAN_KEY_CHANNELID, "6", szValue, 8);
	pwlan->nChannelID=atoi(szValue);

	GetProfileString(WLAN_SECTION, WLAN_KEY_RTSTHRESHOLD, "2347", szValue, 8);
	pwlan->nRTSThreshold=atoi(szValue);

	GetProfileString(WLAN_SECTION, WLAN_KEY_FRAGMENTTHRESHOLD, "2346", szValue, 8);
	pwlan->nFragmentThreshold=atoi(szValue);

	GetProfileString(WLAN_SECTION, WLAN_KEY_WEPENCYPTION, "0", szValue, 8);
	pwlan->nWEPEncyption=atoi(szValue);

	GetProfileString(WLAN_SECTION, WLAN_KEY_WEPDEFAULTKEYID, "0", szValue, 8);
	pwlan->nWEPDefaultKeyID=atoi(szValue);

	for(i=0; i< 4; i++)
	{
		sprintf(szValue, "%s%d", WLAN_KEY_KEYVALUE, i);
		GetProfileString(WLAN_SECTION, szValue, "", szBuffer, WLAN_STRBUFF_SIZE);
		strcpy(pwlan->szKeyValue[i], szBuffer);
	}
	return TRUE;
}

//===========================================================================
//	int Set_WLAN_Data(PWLAN pwlan);
//		parameters:
//			pwlan, type: PWLAN; point to WLAN structure to set the data of wlan; 
//		return	0 failure
//				1 success
//===========================================================================	
int Set_WLAN_Data(PWLAN pwlan)
{
	int		i;
	char	szValue[8]={0};

	if(pwlan==NULL)
		return FALSE;
	if(pwlan->nEnabled)
	{
		strcpy(szValue, "TRUE");
	}
	else
	{
		strcpy(szValue, "FALSE");
	}
	WriteProfileString(WLAN_SECTION, WLAN_KEY_ENABLE, szValue);

	WriteProfileString(WLAN_SECTION, WLAN_KEY_SSID, pwlan->szSSID);

	Int2Str(pwlan->nChannelID, szValue, 8, Dec);
	WriteProfileString(WLAN_SECTION, WLAN_KEY_CHANNELID, szValue);

	Int2Str(pwlan->nRTSThreshold, szValue, 8, Dec);
	WriteProfileString(WLAN_SECTION, WLAN_KEY_RTSTHRESHOLD, szValue);
	pwlan->nRTSThreshold=atoi(szValue);

	Int2Str(pwlan->nFragmentThreshold, szValue, 8, Dec);
	WriteProfileString(WLAN_SECTION, WLAN_KEY_FRAGMENTTHRESHOLD, szValue);
	pwlan->nFragmentThreshold=atoi(szValue);

	Int2Str(pwlan->nWEPEncyption, szValue, 8, Dec);
	WriteProfileString(WLAN_SECTION, WLAN_KEY_WEPENCYPTION, szValue);
	pwlan->nWEPEncyption=atoi(szValue);

	Int2Str(pwlan->nWEPDefaultKeyID, szValue, 8, Dec);
	WriteProfileString(WLAN_SECTION, WLAN_KEY_WEPDEFAULTKEYID, szValue);
	pwlan->nWEPDefaultKeyID=atoi(szValue);

	for(i=0; i< 4; i++)
	{
		sprintf(szValue, "%s%d", WLAN_KEY_KEYVALUE, i);
		WriteProfileString(WLAN_SECTION, szValue, pwlan->szKeyValue[i]);
	}
	return TRUE;
}

//===========================================================================
//	int Int2Str(int value, char* buff, int buff_size, CodeType type);
//		cover integer(32bits) to null-end string
//		parameters:
//			value, type: int; the integer value that you want to conver; 
//			buff, type: char*; null-end string to store the converd number; 
//			buff_size, type: int; above parameter's "buff" size; 
//			type, type: CodeType; what base you want to conver. there are 4
//					code type: Bin, Oct, Dec, Hex; 
//		return	0 failure
//				1 success
//===========================================================================	

int Int2Str(int value, char* buff, int buff_size, CodeType type)
{
	int base, remainder, quotient = value, i=0;
    char tbuff[33]={0};

	tbuff[32]='\0';
	if(value==0)
	{
	  	strcpy(buff, "0");
		return 1;
	}
	switch(type)
	{
	case Bin:
		base=2;
		for(i=31; i>=0 && quotient; i--)
		{
			remainder=quotient%base;
			quotient/=base;
			tbuff[i]='0'+remainder;
		}
		break;
	case Oct:
		base=8;
		for(i=31; i>=0 && quotient; i--)
		{
			remainder=quotient%base;
			quotient/=base;
			tbuff[i]='0'+remainder;
		}
		break;
	case Dec:
		base=10;
		for(i=31; i>=0 && quotient; i--)
		{
			remainder=quotient%base;
			quotient/=base;
			tbuff[i]='0'+remainder;
		}
		break;
	case Hex:
		base=16;
		for(i=31; i>=0 && quotient; i--)
		{
			remainder=quotient%base;
			quotient/=base;
			if(remainder<10)
			{
				tbuff[i]='0'+remainder;
			}
			else
			{
				tbuff[i]='A'+(remainder-10);
			}
		}
		break;
	}
    strncpy(buff, tbuff+(i+1), buff_size-1);
	return 1;
}

//===========================================================================
//	int Is_WLAN_Card_Exist();
//		No parameter
//		return	0 No wireless LAN card found
//				1 Wireless LAN card found
//===========================================================================	
int Is_WLAN_Card_Exist()
{
	int hHwmon, result=0;

	hHwmon = open(HWMON_DEV, O_RDWR);
	if ( hHwmon < 0 )
	{
		fprintf(stderr, "fail to open %s\n", HWMON_DEV);
		return 0;
	}

	if( ioctl(hHwmon, IOCTL_HWMON_IS_WLANCARD_EXIST, &result)<0)
	{
		fprintf(stderr, "fail to call ioctl in IsWLANCardExist()\n");
		close(hHwmon);
		return 0;
	}
	close(hHwmon);
	return result;
}
//
/////////////////////////////////////////////////////////////////////////////

// bonding API
int NIC_Is_Bonding_Support()
{
        int     ret;
        char    buf[80];

        ret=GetProfileString("Network", SZ_BONDING_SUPPORT_FIELD, "FALSE", buf, 80);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

int NIC_Get_Bonding_Interface(BONDING *bonding)
{
        int     ret;
        char    buf[80];
	int	cnt=0;

	cnt = NIC_Count_Bonding_Interface();
        ret=GetProfileString("Network", SZ_BONDING01_FIELD, "Interface1", buf, 80);
        GetProfileString("Network", buf, "eth0", bonding->nic01, NIC_DEV_LEN);
        ret=GetProfileString("Network", SZ_BONDING02_FIELD, "Interface2", buf, 80);
        GetProfileString("Network", buf, "eth1", bonding->nic02, NIC_DEV_LEN);

	if (cnt==3)
	{
		ret=GetProfileString("Network", SZ_BONDING03_FIELD, "Interface3", buf, 80);
		GetProfileString("Network", buf, "eth2", bonding->nic03, NIC_DEV_LEN);
	}
        return NIC_SUCCESS;
}

int NIC_Get_Bonding_Type()
{
        int     type1=BONDING_STANDALONE, type2=BONDING_STANDALONE;
        BONDING bonding;
        char    buf[80];

        NIC_Get_Bonding_Interface(&bonding);
        GetProfileString(bonding.nic01, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE, buf, 80);
        if (!strcasecmp(buf, BONDING_VALUE_LOAD_BALANCE))
                type1=BONDING_LOAD_BALANCE;
        else
        if (!strcasecmp(buf, BONDING_VALUE_FAIL_OVER))
                type1=BONDING_FAIL_OVER;

        GetProfileString(bonding.nic02, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE, buf, 80);
        if (!strcasecmp(buf, BONDING_VALUE_LOAD_BALANCE))
                type2=BONDING_LOAD_BALANCE;
        else
        if (!strcasecmp(buf, BONDING_VALUE_FAIL_OVER))
                type2=BONDING_FAIL_OVER;

        if (type1!=type2)
                NIC_Set_Bonding_Type(type1);
        return type1;
}

int NIC_Set_Bonding_Type(int bonding_type)
{
        int     ret;
        BONDING bonding;
	int	cnt=0;

	cnt = NIC_Count_Bonding_Interface();

        NIC_Get_Bonding_Interface(&bonding);
        if (bonding_type==BONDING_FAIL_OVER)
        {
                ret=WriteProfileString(bonding.nic01, SZ_BONDING_FIELD, BONDING_VALUE_FAIL_OVER);
                ret=WriteProfileString(bonding.nic02, SZ_BONDING_FIELD, BONDING_VALUE_FAIL_OVER);
		if (cnt==3)
			ret=WriteProfileString(bonding.nic03, SZ_BONDING_FIELD, BONDING_VALUE_FAIL_OVER);
        }
        else
        if (bonding_type==BONDING_LOAD_BALANCE)
        {
                ret=WriteProfileString(bonding.nic01, SZ_BONDING_FIELD, BONDING_VALUE_LOAD_BALANCE);
                ret=WriteProfileString(bonding.nic02, SZ_BONDING_FIELD, BONDING_VALUE_LOAD_BALANCE);
		if (cnt==3)
			ret=WriteProfileString(bonding.nic03, SZ_BONDING_FIELD, BONDING_VALUE_LOAD_BALANCE);
        }
        else
        {
                ret=WriteProfileString(bonding.nic01, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE);
                ret=WriteProfileString(bonding.nic02, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE);
		if (cnt==3)
			ret=WriteProfileString(bonding.nic03, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE);
        }
        return NIC_SUCCESS;
}

int NIC_Bonding_Cfg_Copy(NIC_NAS_INFO *nic_nas_info)
{
	char	buf[80];
	BONDING	bonding;
	int	cnt;

	cnt = NIC_Count_Bonding_Interface();

	NIC_Get_Bonding_Interface(&bonding);
	GetProfileString(bonding.nic01, SZ_CONFIGURED_FIELD, "FALSE", buf, 80);
	WriteProfileString(bonding.nic02, SZ_CONFIGURED_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_CONFIGURED_FIELD, buf);

	GetProfileString(bonding.nic01, SZ_USE_DHCP_FIELD, "FALSE", buf, 80);
	WriteProfileString(bonding.nic02, SZ_USE_DHCP_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_USE_DHCP_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_USAGE_FIELD, "DHCP", buf, 80);
        WriteProfileString(bonding.nic02, SZ_USAGE_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_USAGE_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_IPADDR_FIELD, "192.168.0.1", buf, 80);
        WriteProfileString(bonding.nic02, SZ_IPADDR_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_IPADDR_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_NETMASK_FIELD, "192.168.0.255", buf, 80);
        WriteProfileString(bonding.nic02, SZ_NETMASK_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_NETMASK_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_GATEWAY_FIELD, "0.0.0.0", buf, 80);
        WriteProfileString(bonding.nic02, SZ_GATEWAY_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_GATEWAY_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_BROADCAST_FIELD, "192.168.0.255", buf, 80);
        WriteProfileString(bonding.nic02, SZ_BROADCAST_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_BROADCAST_FIELD, buf);

        GetProfileString(bonding.nic01, SZ_BONDING_FIELD, "FAIL OVER", buf, 80);
        WriteProfileString(bonding.nic02, SZ_BONDING_FIELD, buf);
	if (cnt==3)
		WriteProfileString(bonding.nic03, SZ_BONDING_FIELD, buf);

	return NIC_SUCCESS;
}

// Catherine 2003/08/05 ==>
int	NIC_Count_Bonding_Interface()
{
	return (Get_Profile_Integer("Network", "Bonding Interface Number", NIC_Count_Interface()));
}

int     NIC_Is_Switch_Support()
{
	return (int)(Get_Profile_Boolean("Network", "Switch Support", FALSE));
}

int	NIC_Is_Bonding_Device(char* nic_name)
{
	BONDING bond;
	int bond_dev_cnt = NIC_Count_Bonding_Interface();

	NIC_Get_Bonding_Interface(&bond);
	if (strcmp(nic_name, bond.nic01) == 0)
		return TRUE;
	if (strcmp(nic_name, bond.nic02) == 0)
		return TRUE;
	if (bond_dev_cnt == 3 && strcmp(nic_name, bond.nic03) == 0)
		return TRUE;
	return FALSE;
}

// <== Catherine 2003/08/05


int	NLK_Get_Link_Cnt() // YF
{
	int	cnt;
	char	buf[80];

	Conf_Get_Field(NETLINKS_CONF_FILE,GLOBAL_SECTION, "Logical Link Number", buf, 80);
        cnt=atoi(buf);
	return cnt;
}

int	NLK_Get_Nic_Cnt()  //YF
{
	int	cnt;
	char	buf[80];

	Conf_Get_Field(NETLINKS_CONF_FILE,GLOBAL_SECTION, "Physical Interface Number", buf, 80);
        cnt=atoi(buf);
	return cnt;
}

int     NLK_Get_Link_Name(char *link_section, char *name)  //YF
{
	char	buf[32];
	//char	link_section[80];
	//sprintf(link_section, "link%d", link_num);

	if(Conf_Get_Field(NETLINKS_CONF_FILE, link_section, "Name", buf, 32)==ERROR_NOT_FOUND) return NLK_FAIL;
	
	strcpy(name, buf);
	
	return NLK_SUCCESS;
}

int	NLK_Get_Link_Nic_Cnt(int link_num) // YF
{
	int	cnt=0;
	char	buf[80];
	char	link_section[80];
	
	sprintf(link_section, "link%d", link_num);
	
	//GetProfileStringNetLinks(link_section, "Physical Interface Number", "0", buf, 80);
	if(Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "Physical Interface Number", buf, 80)==ERROR_NOT_FOUND)
		return 0;    	

      	cnt=atoi(buf);
    	
	return cnt;
}


int    NLK_Get_Available_NIC_cnt()
{
	int lnk_cnt = 0;
	int nic_cnt = 0;
	int assigned_nic_cnt = 0;
	int available_nic_cnt = 0;
	int size = 1, i = 0;
        char (*link)[32]; //, buf[BUF_SIZE];
	char *ptr;
	char Gateway[256];
	
	nic_cnt = NLK_Get_Nic_Cnt();
	if(nic_cnt == 0) return 0;

	lnk_cnt = NLK_Get_Link_Cnt();
	if(lnk_cnt == 0) return nic_cnt;
	
	NIC_Get_Default_Link(Gateway);	
	
	if (!strchr(Gateway, ',')){
		if((link = (char (*)[32]) malloc((size+1)*32)) == NULL) {
			return 0;
		}	
		strcpy(*link, Gateway);
        }
	else{
		while (Get_String_Field(Gateway, size, ',', NULL, 0) >= 0)
			size++;
		if((link = (char (*)[32]) malloc(((--size)+1)*32)) == NULL) {
		return 0;
		}
		size = 1;
		while (Get_String_Field(Gateway, size , ',', *(link+i), sizeof(*(link+i))) >= 0){
			size++;
			i++;
		}
		--size;	
	}
	
	for(i = 0; i < size; i++)
	{
		ptr=*(link+i);

		if (*ptr==' ')
			ptr++;
			
		assigned_nic_cnt = assigned_nic_cnt + NLK_Get_Link_Nic_Cnt(atoi(&ptr[4])); 
	}
	
//	for(i = 0; i < lnk_cnt; i++)
//	{
//		assigned_nic_cnt = assigned_nic_cnt + NLK_Get_Link_Nic_Cnt(i); 
//	}
	
	available_nic_cnt = nic_cnt - assigned_nic_cnt;
	
	return available_nic_cnt;
}

int    *NLK_Get_Available_NIC()
{
	int *lan;
	int nic_cnt = 0;
	int available_nic_cnt = 0;
	int j = 0 , k = 0, cnt = 0;
	int brk = 0;
	int size = 1, i = 0;
        char (*link)[32]; //, buf[BUF_SIZE];
	char *ptr;
	char Gateway[256];
	
	NIC_Get_Default_Link(Gateway);	
	
	if (!strchr(Gateway, ',')){
		if((link = (char (*)[32]) malloc((size+1)*32)) == NULL) {
			return NULL;
		}	
		strcpy(*link, Gateway);
        }
	else{
		while (Get_String_Field(Gateway, size, ',', NULL, 0) >= 0)
			size++;
		if((link = (char (*)[32]) malloc(((--size)+1)*32)) == NULL) {
		return NULL;
		}
		size = 1;
		while (Get_String_Field(Gateway, size , ',', *(link+i), sizeof(*(link+i))) >= 0){
			size++;
			i++;
		}
		--size;	
	}
	
	available_nic_cnt = NLK_Get_Available_NIC_cnt();
	nic_cnt = NLK_Get_Nic_Cnt();
	
	lan = (int *) calloc(available_nic_cnt, sizeof(int));
	
	for(i = 0; i < nic_cnt; i++)
	{
		int *lnk_lan = NULL;
		int lnk_nic_cnt = 0;
		brk = 0;
		//lnk_cnt = NLK_Get_Link_Cnt();
		
		for(j = 0; j < size; j++)
		{
			ptr=*(link+j);
	
			if (*ptr==' ')
				ptr++;
			
			lnk_nic_cnt = 0;
			lnk_lan = NLK_Get_Link_Assigned_NIC(atoi(&ptr[4]));  //bug
			lnk_nic_cnt = NLK_Get_Link_Nic_Cnt(atoi(&ptr[4]));
			
			for(k = 0; k < lnk_nic_cnt; k++)
			{
				if(i==lnk_lan[k]) 
				{
					brk = 1;
					break;
				}
			}
			
			free(lnk_lan);
			lnk_lan = NULL;
			
			if(brk) break;
		}
		

		if(!brk){
			lan[cnt++] = i;
		}
	}
	
	return lan;
}

int	*NLK_Get_Link_Assigned_NIC(int link_num)
{
	int 	*lan;
	int 	nic_cnt = 0, i=0;
	char	buf[80];
	char	link_section[80];
	char    field[80];
//	char	cmd[200];
	
	sprintf(link_section, "link%d", link_num);

	nic_cnt = NLK_Get_Link_Nic_Cnt(link_num);
	
	if(nic_cnt == 0) return NULL;
	
	lan = (int *)calloc(nic_cnt, sizeof(int));
	
	for(i = 0; i<nic_cnt; i++)
	{	
		sprintf(field, "Interface%d", i+1);
		//GetProfileStringNetLinks(link_section, field, "", buf, 80);
		Conf_Get_Field(NETLINKS_CONF_FILE,link_section, field, buf, 80); 	
		lan[i] = atoi((char *)&buf[3]);
	}
	
	return lan;
}

int NLK_Get_Link_Num(char *name)
{
/*	int lnk_cnt = 0;
	char lnk_name[32];
	int i = 0;
	int rlt = NLK_FAIL;
	FILE *fp;
	
	fp = fopen("/tmp/tmp2.log", "w");
	
	lnk_cnt = NLK_Get_Link_Cnt();
	
	while(rlt==NLK_FAIL)
	{	
		rlt = NLK_Get_Link_Name(lnk_name, i);
		//fprintf(fp,"result = %d %s %d\n", NLK_Get_Link_Name(lnk_name, i), name, i);
		if(rlt==NLK_SUCCESS)
		{
			if(!strcmp(name, lnk_name))
			{
				return i;
				break;
			}
		}
		i++;
	}
	
	fclose(fp);
	*/
	
	int size = 1, i = 0;
        char (*link)[32]; //, buf[BUF_SIZE];
	char *ptr;
	char Gateway[256];
	char lnk_name[32];
	
	NIC_Get_Default_Link(Gateway);	
	
	if (!strchr(Gateway, ',')){
		if((link = (char (*)[32]) malloc((size+1)*32)) == NULL) {
		return -1;
		}	
		strcpy(*link, Gateway);
        }
	else{
		while (Get_String_Field(Gateway, size, ',', NULL, 0) >= 0)
			size++;
		if((link = (char (*)[32]) malloc(((--size)+1)*32)) == NULL) {
		return -1;
		}
		size = 1;
		while (Get_String_Field(Gateway, size , ',', *(link+i), sizeof(*(link+i))) >= 0){
			size++;
			i++;
		}
		--size;	
	}
	
	for (i = 0; i<size; i++){
		ptr=*(link+i);
	
		if (*ptr==' ')
			ptr++;
			
		NLK_Get_Link_Name(ptr, lnk_name);
		if(!strcmp(name, lnk_name))
		{
			return atoi(&ptr[4]);
		}
	}
	
	return -1;
}

//=================================================================
//      base on the type to get device information from NetLinks.conf
//=================================================================
int NIC_Get_Info_From_NetLinks(NIC_NAS_INFO *nic_nas_info, int link_num)
{
        char		buf[NIC_LINE];
        char		link_section[80];
        
	sprintf(link_section, "link%d", link_num);
	//NIC_Get_Default_Dev(nic_nas_info->ifconfig.dev, NIC_DEV_LEN, type);
	
	//GetProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "", buf, NIC_LINE);
	Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "Usage", buf, NIC_LINE); 
	if (!strcmp(buf, "DHCP"))
		nic_nas_info->usage=NIC_USE_DHCP;
	else
	if (!strcmp(buf, "STATIC"))
		nic_nas_info->usage=NIC_USE_STATIC;
	else
	if (!strcmp(buf, "PPPOE"))
		nic_nas_info->usage=NIC_USE_PPPOE;
	else
	if (!strcmp(buf, "PPTP"))
		nic_nas_info->usage=NIC_USE_PPTP;
	else
		nic_nas_info->usage=NIC_USE_OTHER;
	
        Conf_Get_Field(NETLINKS_CONF_FILE, link_section, SZ_HWADDR_FIELD, nic_nas_info->ifconfig.hwaddr, NIC_HWADDR_LEN);
        Conf_Get_Field(NETLINKS_CONF_FILE, link_section, SZ_IPADDR_FIELD, nic_nas_info->ifconfig.ipaddr, NIC_IPADDR_LEN);
        Conf_Get_Field(NETLINKS_CONF_FILE, link_section, SZ_NETMASK_FIELD, nic_nas_info->ifconfig.netmask, NIC_IPADDR_LEN);
        Conf_Get_Field(NETLINKS_CONF_FILE, link_section, SZ_GATEWAY_FIELD, nic_nas_info->ifconfig.gateway, NIC_IPADDR_LEN);
        Conf_Get_Field(NETLINKS_CONF_FILE, link_section, SZ_BROADCAST_FIELD, nic_nas_info->ifconfig.broadcast, NIC_IPADDR_LEN);
        //if (NIC_Is_Bonding_Device(nic_nas_info->ifconfig.dev))
        Conf_Get_Field(NETLINKS_CONF_FILE, "global", "Bonding Type", buf, NIC_LINE);
        
        if(strcmp(buf, "LOAD BALANCE"))
        	nic_nas_info->bonding_type=BONDING_LOAD_BALANCE;
        else if(strcmp(buf, "FAIL OVER"))
        	nic_nas_info->bonding_type=BONDING_FAIL_OVER;
	else
		nic_nas_info->bonding_type=BONDING_STANDALONE;
	return NIC_SUCCESS;
}

int NIC_Get_Bonding_Type_NetLinks()
{
        int     type1=BONDING_STANDALONE;
        //BONDING bonding;
        char    buf[80];

        //NIC_Get_Bonding_Interface(&bonding);
        Conf_Get_Field(NETLINKS_CONF_FILE, "global" ,"Bonding Type", buf, 80);
        //GetProfileString(bonding.nic01, SZ_BONDING_FIELD, BONDING_VALUE_STANDALONE, buf, 80);
        if (!strcasecmp(buf, BONDING_VALUE_LOAD_BALANCE))
                type1=BONDING_LOAD_BALANCE;
        else
        if (!strcasecmp(buf, BONDING_VALUE_FAIL_OVER))
                type1=BONDING_FAIL_OVER;

        return type1;
}

int NIC_Set_Bonding_Type_NetLinks(int bonding_type)
{
 //       int     ret;
 //       BONDING bonding;
//	int	cnt=0;

//	cnt = NLK_Get_Nic_Cnt();

//        NIC_Get_Bonding_Interface(&bonding);
        if (bonding_type==BONDING_FAIL_OVER)
        {
		Conf_Set_Field(NETLINKS_CONF_FILE, "global", "Bonding Type", BONDING_VALUE_FAIL_OVER);
        }
        else
        if (bonding_type==BONDING_LOAD_BALANCE)
        {
        	Conf_Set_Field(NETLINKS_CONF_FILE, "global", "Bonding Type", BONDING_VALUE_LOAD_BALANCE);
        }
        else
        {
        	Conf_Set_Field(NETLINKS_CONF_FILE, "global", "Bonding Type", BONDING_VALUE_STANDALONE);
        }
        return NIC_SUCCESS;
}

// bonding API //YFHuang 20040602
int NIC_Is_Bonding_Support_Netlinks()
{
        int     ret;
        char    buf[80];
	
        ret=Conf_Get_Field(NETLINKS_CONF_FILE, "global", "Always Bonding", buf, 80);
        if (!strcasecmp(buf, "TRUE"))
                return 1;
        else
                return 0;
}

//=====================================================
//      set device information to NetLinks.conf
//=====================================================
int NIC_Set_Info_To_NetLinks(NIC_NAS_INFO *nic_nas_info, int link_num)
{
	int		ret, restart=0;//, type=NIC_LAN01;
	NIC_NAS_INFO	tmp_info;
	char		link_name[80];
	
	sprintf(link_name, "link%d", link_num);
/*	NIC_TCPIP	tcpip;

	NIC_Get_Tcpip(&tcpip);
	if (!strcmp(tcpip.wan, nic_nas_info->ifconfig.dev))
		type=NIC_WAN_DEFAULT;
	else
	if (!strcmp(tcpip.lan01, nic_nas_info->ifconfig.dev))
		type=NIC_LAN01_DEFAULT;
	else
	if (!strcmp(tcpip.lan02, nic_nas_info->ifconfig.dev))
		type=NIC_LAN02_DEFAULT;
	else
	if (!strcmp(tcpip.lan03, nic_nas_info->ifconfig.dev))
		type=NIC_LAN03_DEFAULT;
	else
	if (!strcmp(tcpip.lan04, nic_nas_info->ifconfig.dev))
		type=NIC_LAN04_DEFAULT;
*/
	NIC_Get_Info_From_NetLinks(&tmp_info, link_num);
	/* two usages are not the same, need to restart */
	if (tmp_info.usage!=nic_nas_info->usage)
		restart=1;
	else
	if (tmp_info.usage==NIC_USE_STATIC)
	{
		if (	!strcmp(tmp_info.ifconfig.ipaddr, nic_nas_info->ifconfig.ipaddr) &&
			!strcmp(tmp_info.ifconfig.gateway, nic_nas_info->ifconfig.gateway) &&
			!strcmp(tmp_info.ifconfig.netmask, nic_nas_info->ifconfig.netmask))
			restart=0;
		else
			restart=1;
	}

        // set bonding type
/*	if (NIC_Is_Bonding_Support_NetLinks() && NIC_Is_Bonding_Device(nic_nas_info->ifconfig.dev))
	{
		if (NIC_Get_Bonding_Type_NetLinks()!=nic_nas_info->bonding_type)
			restart=1;
	        NIC_Set_Bonding_Type_NetLinks(nic_nas_info->bonding_type);
	}
*/ 
	if (nic_nas_info->usage==NIC_USE_DHCP)
	{
		ret=WriteProfileStringNetLinks(link_name, SZ_USAGE_FIELD, "DHCP");
		if (!ret)
			return NIC_FAIL;
		//ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "TRUE");
		//if (!ret)
		//	return NIC_FAIL;
	}
	else
	if (nic_nas_info->usage==NIC_USE_STATIC)
	{
                ret=WriteProfileStringNetLinks(link_name, SZ_USAGE_FIELD, "STATIC");
                if (!ret)
                        return NIC_FAIL;
               // ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                //if (!ret)
                //        return NIC_FAIL;
	}
/*	else
	if (nic_nas_info->usage==NIC_USE_PPPOE)
	{
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "PPPOE");
                if (!ret)
                        return NIC_FAIL;
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                if (!ret)
                        return NIC_FAIL;
	}
	else
	if (nic_nas_info->usage==NIC_USE_PPTP)
	{
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "PPTP");
		if (!ret)
			return NIC_FAIL;
		ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
		if (!ret)
			return NIC_FAIL;
	}
	else
	{
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USAGE_FIELD, "OTHER");
                if (!ret)
                        return NIC_FAIL;
                ret=WriteProfileString(nic_nas_info->ifconfig.dev, SZ_USE_DHCP_FIELD, "FALSE");
                if (!ret)
                        return NIC_FAIL;
	} */
	ret=WriteProfileStringNetLinks(link_name, SZ_IPADDR_FIELD, nic_nas_info->ifconfig.ipaddr);
        if (!ret)
                return NIC_FAIL;
	ret=WriteProfileStringNetLinks(link_name, SZ_NETMASK_FIELD, nic_nas_info->ifconfig.netmask);
        if (!ret)
                return NIC_FAIL;
	ret=WriteProfileStringNetLinks(link_name, SZ_GATEWAY_FIELD, nic_nas_info->ifconfig.gateway);
        if (!ret)
                return NIC_FAIL;
	strcpy(nic_nas_info->ifconfig.broadcast, "0.0.0.0");
        if (strcasecmp(nic_nas_info->ifconfig.broadcast,"0.0.0.0") == 0 )
	{
                ULONG bin_ip,bin_netmask,bin_broadcast;
                struct in_addr in;
                bin_ip = inet_addr ( nic_nas_info->ifconfig.ipaddr);
                bin_netmask = inet_addr (nic_nas_info->ifconfig.netmask);
                bin_broadcast = (bin_ip & bin_netmask) | (~bin_netmask);
                in.s_addr = bin_broadcast;

                sprintf(nic_nas_info->ifconfig.broadcast, "%s", (char *)inet_ntoa(in));
        }
        ret=WriteProfileStringNetLinks(link_name, SZ_BROADCAST_FIELD, nic_nas_info->ifconfig.broadcast);
        
        if (!ret)
                return NIC_FAIL;
        // set configured flag for ethx to TRUE, thereafter it will follow the definition now.
        //WriteProfileString(nic_nas_info->ifconfig.dev, SZ_CONFIGURED_FIELD, "TRUE");

	// check if bonding support ?
	// if yes, duplicate BONDING01 to BONDING02
	/*if (NIC_Is_Bonding_Support_NetLinks())
	{
		NIC_Bonding_Cfg_Copy(nic_nas_info);
	}*/

	if (restart==1)
		return NIC_RESTART;
	return NIC_SUCCESS;
}

int	NIC_Get_Switch_Speed(char *speed)
{
	char    buf[80];
	
        Conf_Get_Field(NETLINKS_CONF_FILE, "global", "Switch Speed", buf, 80);
        
        strcpy(speed, buf);
        
        return NIC_SUCCESS;
}

int     NIC_Set_Switch_Speed(char *speed)
{
	Conf_Set_Field(NETLINKS_CONF_FILE, "global", "Switch Speed", speed);
	
	return NIC_SUCCESS;
}

int	NIC_Get_Default_Link(char *Gateway)
{
	char    buf[256];
	
	Conf_Get_Field(NETLINKS_CONF_FILE, "global", "Selected Default Gateway", buf, 256);
	strcpy(Gateway, buf);
	
	return NIC_SUCCESS;
}

int	NIC_Set_Default_Link(char *Gateway)
{
	Conf_Set_Field(NETLINKS_CONF_FILE, "global", "Selected Default Gateway", Gateway);
	
	return NIC_SUCCESS;
}

int	NIC_Get_New_Link_Num()
{
	int size = 1, i = 0, brk = 1, j = 1;
        char (*link)[32]; //, buf[BUF_SIZE];
	char *ptr;
	char Gateway[256];
	
	NIC_Get_Default_Link(Gateway);	
	
	if (!strchr(Gateway, ',')){
		if((link = (char (*)[32]) malloc((size+1)*32)) == NULL) {
		return -1;
		}	
		strcpy(*link, Gateway);
        }
	else{
		while (Get_String_Field(Gateway, size, ',', NULL, 0) >= 0)
			size++;
		if((link = (char (*)[32]) malloc(((--size)+1)*32)) == NULL) {
		return -1;
		}
		size = 1;
		while (Get_String_Field(Gateway, size , ',', *(link+i), sizeof(*(link+i))) >= 0){
			size++;
			i++;
		}
		--size;	
	}
	
	while(brk)
	{
		brk = 0;
		for (i = 0; i<size; i++){
			ptr=*(link+i);
		
			if (*ptr==' ')
				ptr++;
				
			if(j == atoi(&ptr[4]))
			{
			   brk = 1;
			   break;
			}
		}
		j++;
	}
	
	j--;
	return j;
}
