//==========================================================================
//
//      Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//      FILE:
//              cfg_fw.c
//
//      Abstract:
//              Firewall configuration setup function 
//		(include firewall rule and virtual server configuration)
//		functions:
//			static routing table
//			virtual server
//			special application
//			url filter
//			advance firewall rule
//			virtual DMZ
//			one-to-one address mapping
//
//      FUNCTIONS:	see cfg_fw.h
//
//      COMMENTS:       N/A
//
//      HISTORY:
//              2001/07/18      Kent create
//		2001/08/08	add DMZ function
//		2001/08/15	add filter function
//		2001/08/22	add output to /dev/null
//		2001/08/22	add routing table function
//		2001/08/28	add special application function
//		2001/08/29	add icmp request function
//		2001/08/30	add POSTROUTING in virtual server
//		2001/09/22	add ULOG in all functions
//				the ULOG must be supported by kernel
//				use define ULOG to control if we support user space log function
//		2001/09/23	add conf header
//		2001/09/24	add lan-to-lan setting (url filter & rule & web access)
//		2001/09/25	add one-to-one address mapping
//		2001/09/26	fix rename duplicate bug
//		2001/10/18	modify special application
//				add auto trigger support
//		2001/10/19	move trigger to PREROUTING
//				modify trigger device to LAN01
//		2001/10/21	modify run_special_application
//				change to no destinition and no interface
//				source=anywhere, destinition=anywhere
//		2001/11/09	add fw_set_web_access() function
//				let lan can always access web control
//		2001/11/26	add reset function (ex: FW_Reset_xxxxxxxxx())
//		2001/12/18	add string filter (need kernel support)
//
//==========================================================================
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <unistd.h>

/* include NAS lib */
#include <Util.h>
#include <cfg_system.h>
#include <cfg_nic.h>

/* include local file */
#include "cfg_fw.h"

//#define _DEBUG		1
#define ULOG			1

/* NAS directory for iptables */
char	iptables[]={"/sbin/iptables"};
char	devnull[]={"2>/dev/null 1>/dev/null"};

/* ip_forward file */
char	ip_forward[]={"/proc/sys/net/ipv4/ip_forward"};

int	fw_count_vserver();
int	fw_count_rule();
void	fw_write_special_application(FW_SA *sa);

/****************************************************************/
/*	below functions are about ip address functions		*/
/****************************************************************/

//============================================================
//	check if ipaddr can be resolved 	
// 	the result will store in name	
// 	return : 				
//		success -> FW_SUCCESS		
//		fail	-> FW_FAIL		
// 	ex: 	
//		ipaddr=bbs.ntu.edu.tw		
//		the name will be 140.112.1.6	
//===========================================================
int FW_Resolve_Addr(char *ipaddr, char *name, int len)
{
        struct hostent *hp;
        struct in_addr **hptr;

        hp=gethostbyname(ipaddr);
        if (hp==NULL)
                return FW_FAIL;
        hptr=(struct in_addr **)hp->h_addr_list;
        if (strlen(inet_ntoa(**hptr))>len)
                return FW_FAIL;
        strcpy(name, inet_ntoa(**hptr));
        return FW_SUCCESS;
}

//==============================================================
//	check if the addr is IP address or domain name address	
//	return :							
//		IP address : FW_ADDR_IP					
//		domain name : FW_ADDR_DOMAIN_NAME			
//================================================================
int FW_Check_Addr(const char *addr)
{
        const char	*ptr;
	int		dot=0;

        ptr=addr;
        while (*ptr!=0x0)
        {
                if (*ptr=='.' || *ptr=='/' || (*ptr>='0' && *ptr<='9'))
		{
                        ptr++;
			dot++;
		}
                else
                        return FW_ADDR_DOMAIN_NAME;
        }
	if (dot==3)
	        return FW_ADDR_IP;
	else
		return FW_ADDR_ERROR;
}

/****************************************************************/
/*	nat function (standard functions by NAS)		*/
/****************************************************************/

//=================================================
//	update vserver conf file
//	code is not ready now...
//=================================================
int fw_update_vserver()
{
	FILE			*fptr;
	FW_HEADER		header;
	FW_VIRTUAL_SERVER	*vserver=NULL, tmp_vserver, *start=NULL;
	int			ret, count=0;

	if ((fptr=fopen(FW_VSERVER_CONF, "r"))!=NULL)
	{
		fread(&header, 1, sizeof(FW_HEADER), fptr);
		if (header.version!=FW_VERSION || header.bodysize!=sizeof(FW_VSERVER_CONF))
		{
        		while (!feof(fptr))
        		{
                		ret=fread(&tmp_vserver, 1, header.bodysize, fptr);
                		if (ret==header.bodysize)
                		{
                        		if (count==0)
                        		{
                                		vserver=calloc(1, header.bodysize);
                                		start=vserver;
                                		count++;
                        		}
                        		else
                        		{
                                		vserver->next=calloc(1, header.bodysize);
                                		vserver=(FW_VIRTUAL_SERVER *)vserver->next;
                        		}
                        		memcpy(vserver, &tmp_vserver, header.bodysize);
                		}
        		}
		}
		fclose(fptr);
	}
	return FW_SUCCESS;
}

int fw_update_special_application()
{
	FILE		*fptr;
	FW_HEADER	header;
	FW_SA		*sa=NULL, *tmp_sa=NULL, *sa_default;

	if ((fptr=fopen(FW_SA_CONF, "r"))!=NULL)
	{
		if (fread(&header, 1, sizeof(FW_HEADER), fptr)<sizeof(FW_HEADER))
		{
			fclose(fptr);
			unlink(FW_SA_CONF);
			return FW_SUCCESS;
		}
		fclose(fptr);
		if (header.version==1)
		{	/* need to upgrade to new version	*/
			/* new version is version 2		*/

			/* do we have default special application need to copy?	*/
			sa_default=FW_Get_Special_Application(1);

			/* old version => set default is enable */
			sa=FW_Get_Special_Application(0);
			tmp_sa=sa;
			while (tmp_sa!=NULL)
			{
				tmp_sa->enable=1;
				tmp_sa=(FW_SA *)tmp_sa->next;
			}

			/* merge two linked list	*/
			if (sa_default!=NULL)
			{
				tmp_sa=sa_default;
				while (tmp_sa!=NULL)
				{
					if (tmp_sa->next==NULL)
						break;
					tmp_sa=(FW_SA *)tmp_sa->next;
				}
				tmp_sa->next=(struct FW_SA *)sa;
				sa=sa_default;
			}

			fw_write_special_application(sa);
			FW_Free_Special_Application(sa);
		}
	}
	return FW_SUCCESS;
}

//==============================================================
//	update all firewall conf file
//==============================================================
int fw_update_conf()
{
//	fw_update_vserver();
	fw_update_special_application();
	return FW_SUCCESS;
}

//================================================================
//	check if NAT is enable?
//	return :
//		enable -> FW_SUCCESS
//		disable -> FW_FAIL
//================================================================
int FW_Is_NAT_Enable()
{
        char    buf[80];

        GetProfileString(SZ_NAT_SECTION, SZ_ENABLE_FIELD, "FALSE", buf, sizeof(buf));
        if (!strcasecmp(buf, "TRUE"))
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

//=============================================================================
//	this function can auto detect to enable/disable nat function
//	use this function to start "ALL" nat function
//	arg "trigger" is for special application used
//	and the keypoint is make router to auto select destinition ip address
//	see application trigger(cgi-bin/firewall/trigger.c) for detail
//============================================================================
int FW_Begin_NAT(FW_TRIGGER *trigger)
{
	fw_update_conf();
	if (FW_Is_NAT_Enable()==FW_SUCCESS)
		FW_Start_NAT(trigger);
	else
		FW_Stop_NAT();
	return FW_SUCCESS;
}

//===================================================================
//	this function do two things
//	one is enable NAT
//	second is start all nat rule
//	see FW_Begin_NAT(FW_TRIGGER *) for trigger detail
//===================================================================
int FW_Start_NAT(FW_TRIGGER *trigger)
{
	FW_Enable_NAT(1);
	FW_Refresh_NAT(trigger);
	return FW_SUCCESS;
}

//====================================================================
//	1.disable NAT
//	2.refresh NAT tables
//====================================================================
int FW_Stop_NAT()
{
	FW_Enable_NAT(0);
	FW_Refresh_NAT(NULL);
	return FW_SUCCESS;
}

//===================================================================
//	set nat enable/disable to uLinux.conf
//===================================================================
int FW_Set_NAT(int bEnable)
{
	int	ret, restart=0;

	ret=FW_Is_NAT_Enable();
	if (ret==FW_SUCCESS && bEnable==0)
		restart=1;
	else
	if (ret==FW_FAIL && bEnable==1)
		restart=1;

        /* write to uLinux.conf */
        if (bEnable)
                ret=WriteProfileString(SZ_NAT_SECTION, SZ_ENABLE_FIELD, "TRUE");
        else
                ret=WriteProfileString(SZ_NAT_SECTION, SZ_ENABLE_FIELD, "FALSE");

	/* need to restart nat? */
	if (restart)
		FW_Begin_NAT(NULL);
	return FW_SUCCESS;
}

int fw_flush_normal_table()
{
	char	command[FW_LINE];

        /* flush normal table */
        sprintf(command, "%s -F", iptables);
        system(command);
	if (system(command)!=0)
		return FW_FAIL;
	return FW_SUCCESS;
}

int fw_flush_postrouting()
{
	char	command[FW_LINE], dev[NIC_DEV_LEN];

        /* clear iptables POSTROUTING */
        sprintf(command, "%s -t nat -F POSTROUTING %s", iptables, devnull);
#ifdef _DEBUG
        printf("command=%s<br>\n", command);
#endif
        if (system(command)!=0)
                return FW_FAIL;

        /* set nat tables POSTROUTING to MASQ */
	if (FW_Is_NAT_Enable()==FW_SUCCESS)
	{
	        NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
        	sprintf(command, "%s -t nat -A POSTROUTING -o %s -j MASQUERADE %s", iptables, dev, devnull);
#ifdef _DEBUG
	        printf("command=%s<br>\n", command);
#endif
        	if (system(command)!=0)
                	return FW_FAIL;
	}
        return FW_SUCCESS;
}

//==================================================================
//	enable nat function, this function deal with two things
//	one is enable ip_forward
//	second is enable MASQ
//==================================================================
int FW_Enable_NAT(int bEnable)
{
	char	command[FW_LINE];
//	char	dev[NIC_DEV_LEN];

	/* enable ip forwarding */
	sprintf(command, "echo 1 > %s", ip_forward);
#ifdef _DEBUG
        printf("command=%s<br>\n", command);
#endif
        if (system(command)!=0)
                return FW_FAIL;

	/* not enable nat */
	if (bEnable==0)
	{
		/* flush all nat table */
		sprintf(command, "%s -t nat -F", iptables);
		system(command);
		return FW_SUCCESS;
	}

	fw_flush_normal_table();
	fw_flush_postrouting();

	/* flush normal table */
//	sprintf(command, "%s -F", iptables);
//	system(command);

        /* clear iptables POSTROUTING */
/*        sprintf(command, "%s -t nat -F POSTROUTING %s", iptables, devnull);
#ifdef _DEBUG
        printf("command=%s<br>\n", command);
#endif
        if (system(command)!=0)
                return FW_FAIL;*/

        /* set nat tables POSTROUTING to MASQ */
/*	NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
        sprintf(command, "%s -t nat -A POSTROUTING -o %s -j MASQUERADE %s", iptables, dev, devnull);
#ifdef _DEBUG
        printf("command=%s<br>\n", command);
#endif
        if (system(command)!=0)
                return FW_FAIL;*/
	return FW_SUCCESS;
}

//==============================================
//	restart nat
//	1.stop NAT
//	2.start NAT
//==============================================
int FW_Restart_NAT()
{
	if (FW_Stop_NAT()==FW_FAIL)
		return FW_FAIL;
	if (FW_Start_NAT(NULL)==FW_FAIL)
		return FW_FAIL;
	return FW_SUCCESS;
}

//==========================================================
//	auto check web access enable/disable
//	if disable, then add a rule to disable it
//	two type:
//		wan port : SYSTEM_WAN_ACCESS
//		lan port : SYSTEM_LAN_ACCESS
//==========================================================
int FW_Add_Web_Access_Rule(int type)
{
	int		ret, port;
	char		command[FW_LINE];
	NIC_INFO	nic_info;

	ret=Get_Web_Access(type);
	if (ret==0)
	{	/* disable web access, add one rule to enable */
		NIC_Get_Info(&nic_info, NIC_WAN);
		port=Get_Web_Access_Port();
//		sprintf(command, "%s -t nat -A PREROUTING -p tcp -d %s --dport %d -j DROP %s",
//					iptables, nic_info.ipaddr, port, devnull);

		/* should be worked fine under nat enable/disable */
		sprintf(command, "%s -A INPUT -p tcp -d %s --dport %d -j DROP %s",
				iptables, nic_info.ipaddr, port, devnull);
#ifdef _DEBUG
		printf("command=%s<br>\n", command);
#endif
#ifndef _DEBUG
		system(command);
#endif
	}
	return FW_SUCCESS;
}

//=========================================================================
//	we always make lan have the ability to access ROUTER web control
//=========================================================================
int fw_set_web_access(int bEnable)
{
        int             port;
        char            command[FW_LINE];
        NIC_INFO        nic_info;

	NIC_Get_Info(&nic_info, NIC_LAN01);
	port=Get_Web_Access_Port();

	/* always add in first rule */
	sprintf(command, "%s -t nat -I PREROUTING 1 -p tcp -d %s --dport %d -j %s %s",
			iptables, nic_info.ipaddr, port, (bEnable)?"ACCEPT":"DROP", devnull);

#ifdef _DEBUG
                printf("command=%s<br>\n", command);
#endif
#ifndef _DEBUG
                system(command);
#endif
        return FW_SUCCESS;
}

//=========================================================
//	refresh all iptables rules
//=========================================================
int FW_Refresh_NAT(FW_TRIGGER *trigger)
{
        char    command[FW_LINE];
	int	bnat;
	FW_SA	*sa, *tmp;

	fw_flush_normal_table();
	fw_flush_postrouting();

        /* clear nat table PREROUTING */
        sprintf(command, "%s -t nat -F PREROUTING %s", iptables, devnull);
#ifdef _DEBUG
        printf("%s<br>\n", command);
#endif
#ifndef _DEBUG
        system(command);
#endif

        /* start rule, vserver, special application, dmz and web access rule */
	if (FW_Is_NAT_Enable()==FW_SUCCESS)
		bnat=1;
	else
		bnat=0;

	/* if trigger==NULL, set special application destination to 0.0.0.0 */
	sa=FW_Get_Special_Application(0);
	tmp=sa;
	while (tmp!=NULL)
	{
		strcpy(tmp->ipaddr, "0.0.0.0");
		tmp=(FW_SA *)tmp->next;
	}
	FW_Free_Special_Application(sa);

	/* start to refresh all rules */
	if (bnat)
	{
		fw_set_web_access(1);
	        FW_Start_Rule(bnat);
		FW_Start_Filter(bnat);
	        FW_Start_Vserver();
		FW_Start_Special_Application(trigger);
		FW_Start_Mapping();
		FW_Start_Vdmz();
		FW_Start_String_Filter(bnat);
		FW_Add_Web_Access_Rule(SYSTEM_WAN_ACCESS);
		FW_Add_ICMP_Rule();
	}
	else
	{
		fw_set_web_access(1);
		FW_Start_Rule(bnat);
		FW_Start_Filter(bnat);
		FW_Add_Web_Access_Rule(SYSTEM_WAN_ACCESS);
		FW_Add_ICMP_Rule();
		FW_Start_String_Filter(bnat);
	}
	FW_Start_Routing_Table();
	return FW_SUCCESS;
}

//=============================================================
//	icmp function
//=============================================================

//================================================================
//      check if NAT is enable?
//      return :
//              enable -> FW_SUCCESS
//              disable -> FW_FAIL
//================================================================
int FW_Is_ICMP_Request()
{
        char    buf[80];

        GetProfileString(SZ_NAT_SECTION, SZ_ICMP_REQUEST_FIELD, "TRUE", buf, sizeof(buf));
        if (!strcasecmp(buf, "TRUE"))
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

//===================================================================
//      set icmp request enable/disable to uLinux.conf
//===================================================================
int FW_Set_ICMP_Request(int bEnable)
{
        int     ret;

        /* write to uLinux.conf */
        if (bEnable)
                ret=WriteProfileString(SZ_NAT_SECTION, SZ_ICMP_REQUEST_FIELD, "TRUE");
        else
                ret=WriteProfileString(SZ_NAT_SECTION, SZ_ICMP_REQUEST_FIELD, "FALSE");

        return FW_SUCCESS;
}

//==========================================================
//      auto check icmp echo-request enable/disable
//      if disable, then flash the icmp-chain
//	if enable, add icmp-chain and rule
//==========================================================
int FW_Add_ICMP_Rule()
{
	char	command[FW_LINE], dev[NIC_DEV_LEN];
	int	ret;

	if (FW_Is_ICMP_Request()==FW_FAIL)
	{
		NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
		sprintf(command, "%s -N %s %s",
				iptables,
				icmp_chain,
				devnull);
		ret=system(command);
		if (ret!=0)
		{
			sprintf(command, "%s -F %s %s", iptables, icmp_chain, devnull);
			system(command);
		}
		sprintf(command, "%s -A %s -i %s -p ICMP --icmp-type echo-request -j DROP %s",
				iptables,
				icmp_chain,
				dev,
				devnull);
		system(command);
		sprintf(command, "%s -A INPUT -p icmp -j %s %s",
				iptables,
				icmp_chain,
				devnull);
		system(command);
	}
	else
	{	/* flush icmp-chain */
		sprintf(command, "%s -F %s %s", iptables, icmp_chain, devnull);
		system(command);
	}
	return FW_SUCCESS;
}

//====================================================
//	write header for firewall config file
//	include all functions
//====================================================
int fw_write_header(FILE *fptr, int bodysize, int total)
{
	FW_HEADER	header;

	header.version=FW_VERSION;
	header.bodysize=bodysize;
	header.total=total;
	header.reserved=0;
	if (fwrite(&header, 1, sizeof(FW_HEADER), fptr)!=sizeof(FW_HEADER))
		return FW_FAIL;
	return FW_SUCCESS;
}

//==============================================================
//	below virtual DMZ function
//==============================================================

//===============================================================
//	get all dmz from config file
//	return :
//		NULL=>no dmz
//		pointer to FW_VIRTUAL_DMZ
//================================================================
FW_VIRTUAL_DMZ *FW_Get_Vdmz()
{
        FILE                    *fptr;
        FW_VIRTUAL_DMZ		*start=NULL, *vdmz=NULL, tmp_vdmz;
        int                     v_size, ret, count=0;
	FW_HEADER		header;

        if ((fptr=fopen(FW_VDMZ_CONF, "r"))==NULL)
                return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_VIRTUAL_DMZ);
        while (!feof(fptr))
        {
                ret=fread(&tmp_vdmz, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                vdmz=calloc(1, v_size);
                                start=vdmz;
                                count++;
                        }
                        else
                        {
                                vdmz->next=calloc(1, v_size);
                                vdmz=(FW_VIRTUAL_DMZ *)vdmz->next;
                        }
                        memcpy(vdmz, &tmp_vdmz, v_size);
                }
        }
        fclose(fptr);
        return start;
}

//==================================================================
//	free memory which be pointed vdmz
//==================================================================
void FW_Free_Vdmz(FW_VIRTUAL_DMZ *vdmz)
{
        FW_VIRTUAL_DMZ		*tmp;

        tmp=vdmz;
        while (vdmz!=NULL)
        {
                tmp=(FW_VIRTUAL_DMZ *)vdmz->next;
                free(vdmz);
                vdmz=tmp;
        }
        return;
}

//=====================================================
//	compare two DMZ
//	return:
//		0: the same
//		1: different
//=====================================================
int fw_comp_dmz(FW_VIRTUAL_DMZ *vdmz1, FW_VIRTUAL_DMZ *vdmz2)
{
        if (!strcmp(vdmz1->addr, vdmz2->addr) && vdmz1->ulog==vdmz2->ulog)
                return 0;
        else
                return 1;
}

int FW_Modify_Vdmz(FW_VIRTUAL_DMZ *new)
{
	FW_VIRTUAL_DMZ		*vdmz=NULL;
	int			v_size;
	FILE			*fptr;

        vdmz=FW_Get_Vdmz();
	if (vdmz!=NULL)
	{
		if (!fw_comp_dmz(vdmz, new))
			return FW_SUCCESS;
	}
	fptr=fopen(FW_VDMZ_CONF, "w");
	fw_write_header(fptr, sizeof(FW_VIRTUAL_DMZ), 1);
        v_size=sizeof(FW_VIRTUAL_DMZ);
	new->next=NULL;
	fwrite(new, 1, v_size, fptr);
	fclose(fptr);
	FW_Free_Vdmz(vdmz);
	FW_Refresh_NAT(NULL);
	return FW_SUCCESS;
}

int FW_Set_DMZ(int bEnable)
{
	int	ret, restart=0;

        ret=FW_Is_DMZ_Enable();
        if (ret==FW_SUCCESS && bEnable==0)
                restart=1;
        else
        if (ret==FW_FAIL && bEnable==1)
                restart=1;

        /* write to uLinux.conf */
	if (bEnable)
		ret=WriteProfileString(SZ_NAT_SECTION, SZ_DMZ_FIELD, "TRUE");
	else
		ret=WriteProfileString(SZ_NAT_SECTION, SZ_DMZ_FIELD, "FALSE");
        if (!ret)
		return FW_FAIL;
	if (restart)
		FW_Begin_NAT(NULL);
	return FW_SUCCESS;
}

int FW_Is_DMZ_Enable()
{
	char	buf[80];

	GetProfileString(SZ_NAT_SECTION, SZ_DMZ_FIELD, "FALSE", buf, sizeof(buf));
	if (!strcasecmp(buf, "TRUE"))
		return FW_SUCCESS;
	else
		return FW_FAIL;
}

int fw_run_vdmz(FW_VIRTUAL_DMZ *vdmz)
{
	char		command[FW_LINE], dev[NIC_DEV_LEN];
	int		ret=0;
	NIC_INFO	nic_info;

	if (FW_Is_DMZ_Enable()==FW_FAIL)
		return FW_SUCCESS;
	NIC_Get_Info(&nic_info, NIC_WAN);
        if (vdmz!=NULL)
        {
		NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
#ifndef ULOG
		sprintf(command, "%s -t nat -A PREROUTING -i %s -d %s -j DNAT --to %s %s",
			iptables,
			dev,
			nic_info.ipaddr,
			vdmz->addr,
			devnull);
                ret=system(command);
#endif
#ifdef ULOG
		sprintf(command, "%s -t nat -N vdmz %s",
			iptables,
			devnull);
		ret=system(command);
		if (ret!=0)
		{
			sprintf(command, "%s -t nat -F vdmz %s",
				iptables,
				devnull);
			system(command);
		}
		sprintf(command, "%s -t nat -A PREROUTING -i %s -d %s -j vdmz %s",
			iptables,
			dev,
			nic_info.ipaddr,
			devnull);
		system(command);
		if (vdmz->ulog)
		{
			sprintf(command, "%s -t nat -A vdmz -j ULOG --ulog-nlgroup 32 --ulog-prefix \"virtual DMZ\" %s",
				iptables,
				devnull);
			ret=system(command);
		}
		sprintf(command, "%s -t nat -A vdmz -j DNAT --to %s %s",
			iptables,
			vdmz->addr,
			devnull);
		system(command);
#endif
        }
        if (ret==0)
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

int FW_Start_Vdmz()
{
        FW_VIRTUAL_DMZ		*vdmz, *tmp;

        vdmz=FW_Get_Vdmz();
        tmp=vdmz;
        while (tmp!=NULL)
        {
                fw_run_vdmz(tmp);
                tmp=(FW_VIRTUAL_DMZ *)tmp->next;
        }
        FW_Free_Vdmz(vdmz);
	return FW_SUCCESS;
}

//======================================================================
//	firewall virtual server function
//======================================================================
int fw_count_vserver()
{
	FILE		*fptr;
	FW_HEADER	header;

        if ((fptr=fopen(FW_VSERVER_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
	fclose(fptr);
	return header.total;
}

int fw_run_vserver(FW_VIRTUAL_SERVER *vserver, int action, int position, int num)
{
	char		protocol[20], command[FW_LINE];
	int		ret=0, total=0;
	NIC_INFO        nic_info;
	char		dev[NIC_DEV_LEN];

	NIC_Get_Info(&nic_info, NIC_WAN);
	if (vserver!=NULL)
	{
                if (vserver->protocol==FW_PROTOCOL_TCP)
                        strcpy(protocol, "tcp");
                else
                if (vserver->protocol==FW_PROTOCOL_UDP)
                        strcpy(protocol, "udp");
                else
                        strcpy(protocol, "tcp");
		if (action==FW_IPTABLES_APPEND)
		{
#ifndef ULOG
			sprintf(command, "%s -t nat -A PREROUTING -p %s -d %s --dport %d -j DNAT --to %s:%d %s",
                        	iptables,
				protocol, 
				nic_info.ipaddr, 
				vserver->src_port, 
				vserver->addr, 
				vserver->dest_port,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
			/* create new chain */
			sprintf(command, "%s -t nat -N vserver%d %s",
				iptables,
				num,
				devnull);
			ret=system(command);
			if (ret!=0)
			{	/* chain have already existed, so clear chain */
				sprintf(command, "%s -t nat -F vserver%d %s",
					iptables,
					num,
					devnull);
				ret=system(command);
			}
			/* add vserver and jump to chain */
			sprintf(command, "%s -t nat -A PREROUTING -p %s -d %s --dport %d -j vserver%d %s",
				iptables,
				protocol,
				nic_info.ipaddr,
				vserver->src_port,
				num,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
			/* add rules in new chain */
			if (vserver->ulog)
			{
				sprintf(command, "%s -t nat -A vserver%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"virual server\" %s",
					iptables,
					num,
					devnull);
				ret=system(command);
				if (ret!=0)
					return FW_FAIL;
			}
			sprintf(command, "%s -t nat -A vserver%d -p %s -j DNAT --to %s:%d %s",
				iptables,
				num,
				protocol,
				vserver->addr,
				vserver->dest_port,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
		}
		else
		/* now is never used		*/
		/* code is not ready for ULOG	*/
		if (action==FW_IPTABLES_REPLACE)
		{
			total=fw_count_rule();
			sprintf(command, "%s -t nat -R PREROUTING %d -p %s -d %s --dport %d -j DNAT --to %s:%d %s",
				iptables,
				position+total,
				protocol,
				nic_info.ipaddr,
				vserver->src_port,
				vserver->addr,
				vserver->dest_port,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
		}

		/****************************************************************/
		/* check if we let virtual IP to access our virtual server ? 	*/
		/* if yes, add a rule in POSTROUTING				*/
		/****************************************************************/
		if (vserver->nat_access==1)
		{
			NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_LAN01);
			if (Is_WLAN_Supported() && Is_WLAN_Card_Exist() && Is_WLAN_Enabled())
						strcpy(dev, BRIDGE_DEV);
			sprintf(command, "%s -t nat -A POSTROUTING -o %s -p %s --dport %d -j MASQUERADE %s",
				iptables,
				dev,
				protocol,
				vserver->dest_port,
				devnull);
			system(command);
			if (ret!=0)
				return FW_FAIL;
		}
	}
	if (ret==0)
		return FW_SUCCESS;
	else
		return FW_FAIL;
}

int fw_remove_vserver(int position)
{
	char	command[FW_LINE];
	int	ret=0, num;

	num=fw_count_rule();
	sprintf(command, "%s -t nat -D PREROUTING %d %s", iptables, position+num, devnull);
#ifdef _DEBUG
	printf("%s<br>\n", command);
#endif
#ifndef _DEBUG
	ret=system(command);
#endif
	return ret;
}

int FW_Start_Vserver()
{
	int			ret=FW_SUCCESS;
	FW_VIRTUAL_SERVER	*vserver, *tmp;
	int			num=0;

	vserver=FW_Get_Vserver();
	tmp=vserver;
	while (tmp!=NULL)
	{
		fw_run_vserver(tmp, FW_IPTABLES_APPEND, 0, num);
		tmp=(FW_VIRTUAL_SERVER *)tmp->next;
		num++;
	}
	FW_Free_Vserver(vserver);
	return ret;
}

FW_VIRTUAL_SERVER *FW_Get_Vserver()
{
	FILE			*fptr;
	FW_VIRTUAL_SERVER 	*start=NULL, *vserver=NULL, tmp_vserver;
	int			v_size, ret, count=0;
	FW_HEADER		header;

	if ((fptr=fopen(FW_VSERVER_CONF, "r"))==NULL)
		return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
	v_size=sizeof(FW_VIRTUAL_SERVER);
	while (!feof(fptr))
	{
		ret=fread(&tmp_vserver, 1, v_size, fptr);
		if (ret==v_size)
		{
			if (count==0)
			{
				vserver=calloc(1, v_size);
				start=vserver;
				count++;
			}
			else
			{
				vserver->next=calloc(1, v_size);
				vserver=(FW_VIRTUAL_SERVER *)vserver->next;
			}
			memcpy(vserver, &tmp_vserver, v_size);
		}
	}
	fclose(fptr);
	return start;
}

void fw_write_vserver(FW_VIRTUAL_SERVER *vserver)
{
	FW_VIRTUAL_SERVER	*tmp;
	int			v_size;
	FILE			*fptr;
	int			total=0;

	tmp=vserver;
	while (tmp!=NULL)
	{
		tmp=(FW_VIRTUAL_SERVER *)tmp->next;
		total++;
	}
	if ((fptr=fopen(FW_VSERVER_CONF, "w"))==NULL)
		return;
	fw_write_header(fptr, sizeof(FW_VIRTUAL_SERVER), total);
	v_size=sizeof(FW_VIRTUAL_SERVER);
	tmp=vserver;
	while (tmp!=NULL)
	{
		fwrite(tmp, 1, v_size, fptr);
		tmp=(FW_VIRTUAL_SERVER *)tmp->next;
	}
	fclose(fptr);
}

void FW_Free_Vserver(FW_VIRTUAL_SERVER *vserver)
{
	FW_VIRTUAL_SERVER	*tmp;

	tmp=vserver;
	while (vserver!=NULL)
	{
                tmp=(FW_VIRTUAL_SERVER *)vserver->next;
                free(vserver);
                vserver=tmp;
	}
	return;
}

int fw_comp_vserver(FW_VIRTUAL_SERVER *vserver1, FW_VIRTUAL_SERVER *vserver2)
{
	if (!strcmp(	vserver1->addr, vserver2->addr) && 
			vserver1->protocol==vserver2->protocol && 
			vserver1->src_port==vserver2->src_port &&
			vserver1->dest_port==vserver2->dest_port &&
			vserver1->nat_access==vserver2->nat_access &&
			vserver1->ulog==vserver2->ulog)
		return 0;
	else
		return 1;
}

int FW_Add_Vserver(FW_VIRTUAL_SERVER *new)
{
	FW_VIRTUAL_SERVER 	*vserver=NULL, *tmp=NULL;
	int			v_size, count=1;
	NIC_INFO		nic_info;

	NIC_Get_Info(&nic_info, NIC_WAN);
	vserver=FW_Get_Vserver();
	v_size=sizeof(FW_VIRTUAL_SERVER);
	if (vserver==NULL)
	{
		vserver=calloc(1, v_size);
		memcpy(vserver, new, v_size);
		vserver->next=NULL;
	}
	else
	{
		tmp=vserver;
		while (tmp!=NULL)
		{
//			if (!strcmp(nic_info.ipaddr, tmp->addr))
//				fw_run_vserver(tmp, FW_IPTABLES_REPLACE, count);
			if (!fw_comp_vserver(tmp, new))
			{
				FW_Free_Vserver(vserver);
				return FW_VSERVER_DUPLICATE;
			}
			if (tmp->next==NULL)
				break;
			else
				tmp=(FW_VIRTUAL_SERVER *)tmp->next;
			count++;
		}

		tmp->next=calloc(1, v_size);
		tmp=(FW_VIRTUAL_SERVER *)tmp->next;
		memcpy(tmp, new, v_size);
		tmp->next=NULL;
	}
        /* can we insert? */
        if (fw_run_vserver(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM)!=FW_SUCCESS)
        {
	        FW_Free_Vserver(vserver);
                return FW_VSERVER_ADD_FAIL;
        }
	fw_write_vserver(vserver);
	FW_Free_Vserver(vserver);
	FW_Refresh_NAT(NULL);
	return FW_SUCCESS;
}

int FW_Refresh_Vserver()
{
	FW_VIRTUAL_SERVER	*vserver, *tmp;
	int			v_size, count=1;
	NIC_INFO		nic_info;
	int			num=0;

	NIC_Get_Info(&nic_info, NIC_WAN);
	vserver=FW_Get_Vserver();
	v_size=sizeof(FW_VIRTUAL_SERVER);
	if (vserver!=NULL)
	{
		tmp=vserver;
		while (tmp!=NULL)
		{
			if (!strcmp(nic_info.ipaddr, tmp->addr))
				fw_run_vserver(tmp, FW_IPTABLES_REPLACE, count, num);
			tmp=(FW_VIRTUAL_SERVER *)tmp->next;
			count++;
			num++;
		}
	}
	return FW_SUCCESS;
}

int FW_Del_Vserver(FW_VIRTUAL_SERVER *del)
{
	FW_VIRTUAL_SERVER	*vserver, *tmp, *parent;
	int			v_size, count=1;
	NIC_INFO		nic_info;

	NIC_Get_Info(&nic_info, NIC_WAN);
	vserver=FW_Get_Vserver();
	v_size=sizeof(FW_VIRTUAL_SERVER);
	if (vserver==NULL)
		return FW_VSERVER_NOT_FOUND;
	else
	{
		tmp=vserver;
		parent=vserver;
		while (tmp!=NULL)
		{
//			if (!strcmp(nic_info.ipaddr, tmp->addr))
//				fw_run_vserver(tmp, FW_IPTABLES_REPLACE, count);
			if (!fw_comp_vserver(tmp, del))
			{
				if (tmp==vserver)
				{
					vserver=(FW_VIRTUAL_SERVER *)vserver->next;
					free(tmp);
				}
				else
				{
					parent->next=tmp->next;
					free(tmp);
				}
				fw_write_vserver(vserver);
				FW_Free_Vserver(vserver);
				fw_remove_vserver(count);
				FW_Refresh_NAT(NULL);
				return FW_SUCCESS;
				
			}
			parent=tmp;
			tmp=(FW_VIRTUAL_SERVER *)tmp->next;
			count++;
		}
	}
	FW_Free_Vserver(vserver);
	return FW_VSERVER_NOT_FOUND;
}

int FW_Ren_Vserver(FW_VIRTUAL_SERVER *new, FW_VIRTUAL_SERVER *old)
{
        FW_VIRTUAL_SERVER       *vserver, *tmp, *check;
        int                     v_size, count=1;
	NIC_INFO		nic_info;

	if (!fw_comp_vserver(new, old))
		return FW_SUCCESS;
	NIC_Get_Info(&nic_info, NIC_WAN);
        vserver=FW_Get_Vserver();
        v_size=sizeof(FW_VIRTUAL_SERVER);
        if (vserver==NULL)
                return FW_VSERVER_NOT_FOUND;
        else
        {
		/* replace old to new vserver */
                tmp=vserver;
                while (tmp!=NULL)
                {
//                        if (!strcmp(nic_info.ipaddr, tmp->addr))
//                                fw_run_vserver(tmp, FW_IPTABLES_REPLACE, count);
                        if (!fw_comp_vserver(tmp, old))
                        {
                                /* check if duplicate */
                                check=vserver;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if (!fw_comp_vserver(check, new))
						{
							FW_Free_Vserver(vserver);
							return FW_VSERVER_DUPLICATE;
						}
                                        }
                                        check=(FW_VIRTUAL_SERVER *)check->next;
                                }

				/* copy new to tmp */
				new->next=tmp->next;
				memcpy(tmp, new, sizeof(FW_VIRTUAL_SERVER));

				/* can we insert */
				if (fw_run_vserver(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM)!=FW_SUCCESS)
				{
					FW_Free_Vserver(vserver);
					return FW_VSERVER_REN_FAIL;
				}

				/* replace to iptables */
                                fw_write_vserver(vserver);
                                FW_Free_Vserver(vserver);
				FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_VIRTUAL_SERVER *)tmp->next;
                        count++;
                }
	}
	return FW_VSERVER_NOT_FOUND;
}

//==================================================================================
//	firewall rule function
//==================================================================================
int fw_count_rule()
{
        FILE            *fptr;
        FW_HEADER       header;

        if ((fptr=fopen(FW_RULE_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        fclose(fptr);
        return header.total;
}

int fw_rule_valid(FW_RULE *rule)
{
        char    buf[ADDR_LEN];

        /* check if we can resolve the ip? */
	if (rule->src_flag!=FW_FLAG_ANY)
	{
		if (rule->src_type!=FW_ADDR_IP)
		        if (FW_Check_Addr(rule->src_addr)==FW_ADDR_DOMAIN_NAME)
        		{
                		if (FW_Resolve_Addr(rule->src_addr, buf, ADDR_LEN)!=FW_SUCCESS)
                        		return FW_RESOLVE_FAIL;
		        }
	}
	if (rule->dest_flag!=FW_FLAG_ANY)
	{
		if (rule->dst_type!=FW_ADDR_IP)
		        if (FW_Check_Addr(rule->dest_addr)==FW_ADDR_DOMAIN_NAME)
        		{
                		if (FW_Resolve_Addr(rule->dest_addr, buf, ADDR_LEN)!=FW_SUCCESS)
                        		return FW_RESOLVE_FAIL;
		        }
	}
        return FW_SUCCESS;
}

int fw_count_valid_rule()
{
	FW_RULE		*rule, *tmp;
	int		count=0;

	rule=FW_Get_Rule();
	tmp=rule;
	while (tmp!=NULL)
	{
		if (tmp->valid==FW_VALID)
			count++;
		tmp=(FW_RULE *)tmp->next;
	}
	FW_Free_Rule(rule);
	return count;
}

FW_RULE *FW_Get_Rule()
{
        FILE                    *fptr;
        FW_RULE			*start=NULL, *rule=NULL, tmp_rule;
        int                     r_size, ret, count=0;
	FW_HEADER		header;

        if ((fptr=fopen(FW_RULE_CONF, "r"))==NULL)
                return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
        r_size=sizeof(FW_RULE);
        while (!feof(fptr))
        {
                ret=fread(&tmp_rule, 1, r_size, fptr);
                if (ret==r_size)
                {
                        if (count==0)
                        {
                                rule=calloc(1, r_size);
                                start=rule;
                                count++;
                        }
                        else
                        {
                                rule->next=calloc(1, r_size);
                                rule=(FW_RULE *)rule->next;
                        }
                        memcpy(rule, &tmp_rule, r_size);
                }
        }
        fclose(fptr);
        return start;
}

int fw_run_rule(FW_RULE *rule, int action, int position, int num, int bnat)
{
        char    	command[FW_LINE]={""}, sport[10]={""}, dport[10]={""}, protocol[10]={""};
        int     	ret=0;
	NIC_INFO	nic_info;
        char            nattable[256]={""}, which_table[256]={""};

        if (bnat)
        {
                sprintf(nattable, "-t nat");
                sprintf(which_table, "PREROUTING");
        }
        else
        {
                sprintf(nattable, " ");
                sprintf(which_table, "FORWARD");
        }
	NIC_Get_Info(&nic_info, NIC_WAN);
        if (rule!=NULL)
        {
		if (rule->src_port==0)
			sport[0]=0x0;
		else
			sprintf(sport, "--sport %d", rule->src_port);
		if (rule->dest_port==0)
			dport[0]=0x0;
		else
			sprintf(dport, "--dport %d", rule->dest_port);

		if (rule->protocol==FW_PROTOCOL_ALL)
			strcpy(protocol, "");
		else
		if (rule->protocol==FW_PROTOCOL_TCP)
			strcpy(protocol, "-p tcp");
		else
		if (rule->protocol==FW_PROTOCOL_UDP)
			strcpy(protocol, "-p udp");

                if (action==FW_IPTABLES_INSERT)
		{
#ifndef ULOG
			if (rule->protocol==FW_PROTOCOL_TCP || rule->protocol==FW_PROTOCOL_ALL)
			{
				sprintf(command, "%s %s -I %s %d %s -s %s %s %s -d %s %s %s -j %s %s",
					iptables,
					nattable,
					which_table,
					position,
//					protocol,
					"-p tcp",
					(rule->src_flag==FW_FLAG_NOT)?"!":"",
					(rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
					sport,
					(rule->dest_flag==FW_FLAG_NOT)?"!":"",
					(rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
					dport,
					(rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
					devnull);
				ret=system(command);
				if (ret!=0)
					return FW_FAIL;
			}
			if (rule->protocol==FW_PROTOCOL_UDP || rule->protocol==FW_PROTOCOL_ALL)
			{
                                sprintf(command, "%s %s -I %s %d %s -s %s %s %s -d %s %s %s -j %s %s",
                                        iptables,
                                        nattable,
                                        which_table,
                                        position,
//                                      protocol,
                                        "-p udp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        (rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
			}
#endif
#ifdef ULOG
			sprintf(command, "%s %s -N rule%d %s",
				iptables,
				nattable,
				num,
				devnull);
			ret=system(command);
			if (ret!=0)
			{
				sprintf(command, "%s %s -F rule%d %s",
					iptables,
					nattable,
					num,
					devnull);
				system(command);
			}
			if (rule->protocol==FW_PROTOCOL_TCP || rule->protocol==FW_PROTOCOL_ALL)
			{
				sprintf(command, "%s %s -I %s %d %s -s %s %s %s -d %s %s %s -j rule%d %s",
					iptables,
					nattable,
					which_table,
					position,
//					protocol,
					"-p tcp",
					(rule->src_flag==FW_FLAG_NOT)?"!":"",
					(rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
					sport,
					(rule->dest_flag==FW_FLAG_NOT)?"!":"",
					(rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
					dport,
					num,
					devnull);
				ret=system(command);
				if (ret!=0)
					return FW_FAIL;
			}
                        if (rule->protocol==FW_PROTOCOL_UDP || rule->protocol==FW_PROTOCOL_ALL)
                        {
                                sprintf(command, "%s %s -I %s %d %s -s %s %s %s -d %s %s %s -j rule%d %s",
                                        iptables,
                                        nattable,
                                        which_table,
                                        position,
//                                      protocol,
                                        "-p udp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        num,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        /* add rules in new chain */
                        if (rule->ulog)
                        {
                                sprintf(command, "%s %s -A rule%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"firewall\" %s",
                                        iptables,
					nattable,
                                        num,
                                        devnull);
                                ret=system(command);
				if (ret!=0)
					return FW_FAIL;
                        }
			sprintf(command, "%s %s -A rule%d -j %s %s",
					iptables,
					nattable,
					num,
					(rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
					devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
		}
		else
		if (action==FW_IPTABLES_APPEND)
		{
#ifndef ULOG
                        if (rule->protocol==FW_PROTOCOL_TCP || rule->protocol==FW_PROTOCOL_ALL)
                        {
                                sprintf(command, "%s %s -A %s %s -s %s %s %s -d %s %s %s -j %s %s",
                                        iptables,
                                        nattable,
                                        which_table,
//                                      protocol,
                                        "-p tcp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        (rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        if (rule->protocol==FW_PROTOCOL_UDP || rule->protocol==FW_PROTOCOL_ALL)
                        {
                                sprintf(command, "%s %s -A %s %s -s %s %s %s -d %s %s %s -j %s %s",
                                        iptables,
                                        nattable,
                                        which_table,
//                                      protocol,
                                        "-p udp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        (rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
#endif
#ifdef ULOG
                        sprintf(command, "%s %s -N rule%d %s",
                                iptables,
                                nattable,
                                num,
                                devnull);
                        ret=system(command);
                        if (ret!=0)
                        {
                                sprintf(command, "%s %s -F rule%d %s",
                                        iptables,
                                        nattable,
                                        num,
                                        devnull);
                                system(command);
                        }
                        if (rule->protocol==FW_PROTOCOL_TCP || rule->protocol==FW_PROTOCOL_ALL)
                        {
                                sprintf(command, "%s %s -A %s %s -s %s %s %s -d %s %s %s -j rule%d %s",
                                        iptables,
                                        nattable,
                                        which_table,
//                                      protocol,
                                        "-p tcp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        num,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        if (rule->protocol==FW_PROTOCOL_UDP || rule->protocol==FW_PROTOCOL_ALL)
                        {
                                sprintf(command, "%s %s -A %s %s -s %s %s %s -d %s %s %s -j rule%d %s",
                                        iptables,
                                        nattable,
                                        which_table,
//                                      protocol,
                                        "-p udp",
                                        (rule->src_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
                                        sport,
                                        (rule->dest_flag==FW_FLAG_NOT)?"!":"",
                                        (rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
                                        dport,
                                        num,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        /* add rules in new chain */
                        if (rule->ulog)
                        {
                                sprintf(command, "%s %s -A rule%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"firewall\" %s",
                                        iptables,
                                        nattable,
                                        num,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        sprintf(command, "%s %s -A rule%d -j %s %s",
                                        iptables,
                                        nattable,
                                        num,
                                        (rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
                                        devnull);
                        ret=system(command);
                        if (ret!=0)
                                return FW_FAIL;
#endif
		}
                else
		/* now is never used 		*/
		/* code is not ready for ULOG	*/
                if (action==FW_IPTABLES_REPLACE)
		{
			sprintf(command, "%s %s -R %s %d %s -s %s %s %s -d %s %s %s -j %s %s",
				iptables,
				nattable,
				which_table,
				position,
				(rule->src_port!=0 || rule->dest_port!=0)?"-p tcp":"",
				(rule->src_flag==FW_FLAG_NOT)?"!":"",
				(rule->src_flag==FW_FLAG_ANY)?"any/0":rule->src_addr,
				sport,
				(rule->dest_flag==FW_FLAG_NOT)?"!":"",
				(rule->dest_flag==FW_FLAG_ANY)?"any/0":rule->dest_addr,
				dport,
				(rule->action==FW_ACCEPT)?"ACCEPT":"DROP",
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
		}
        }
	if (ret==0)
		return FW_SUCCESS;
	else
		return FW_FAIL;
}

int fw_remove_rule(int position)
{
        char    command[FW_LINE];
        int     ret=0;

        sprintf(command, "%s -t nat -D PREROUTING %d %s", iptables, position, devnull);
#ifdef _DEBUG
        printf("%s<br>\n", command);
#endif
#ifndef _DEBUG
        ret=system(command);
#endif
        return ret;
}

void fw_write_rule(FW_RULE *rule)
{
        FW_RULE			*tmp;
        int                     r_size;
        FILE                    *fptr;
	int			total=0;

	tmp=rule;
	while (tmp!=NULL)
	{
		tmp=(FW_RULE *)tmp->next;
		total++;
	}
        if ((fptr=fopen(FW_RULE_CONF, "w"))==NULL)
                return;
	fw_write_header(fptr, sizeof(FW_RULE), total);
        r_size=sizeof(FW_RULE);
        tmp=rule;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, r_size, fptr);
                tmp=(FW_RULE *)tmp->next;
        }
        fclose(fptr);
}

void FW_Free_Rule(FW_RULE *rule)
{
        FW_RULE			*tmp;

        tmp=rule;
        while (rule!=NULL)
        {
                tmp=(FW_RULE *)rule->next;
                free(rule);
                rule=tmp;
        }
        return;
}

int fw_comp_rule(FW_RULE *rule1, FW_RULE *rule2)
{
        if (	!strcmp(rule1->src_addr, rule2->src_addr) &&
		!strcmp(rule1->dest_addr, rule2->dest_addr) &&
		rule1->src_flag==rule2->src_flag &&
		rule1->dest_flag==rule2->dest_flag &&
		rule1->protocol==rule2->protocol &&
		rule1->src_port==rule2->src_port &&
		rule1->dest_port==rule2->dest_port &&
		rule1->action==rule2->action &&
		rule1->ulog==rule2->ulog)
                return 0;
        else
                return 1;
}

int FW_Add_Rule(FW_RULE *new)
{
        FW_RULE			*rule=NULL, *tmp=NULL;
        int                     r_size, cnt=1, ret, bnat;

        rule=FW_Get_Rule();
        r_size=sizeof(FW_RULE);
        if (rule==NULL)
        {
                rule=calloc(1, r_size);
                memcpy(rule, new, r_size);
		rule->next=NULL;
        }
        else
        {
		new->valid=FW_VALID;
                tmp=rule;
                while (tmp!=NULL)
                {
                        if (!fw_comp_rule(tmp, new))
                        {
                                FW_Free_Rule(rule);
                                return FW_RULE_DUPLICATE;
                        }

			ret=fw_rule_valid(tmp);
			if (ret==FW_SUCCESS)
				tmp->valid=FW_VALID;
			else
				tmp->valid=FW_INVALID;

			if (tmp->valid==FW_VALID)
				cnt++;
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_RULE *)tmp->next;
                }

		/* check if the rule valid? */
		if (fw_rule_valid(new)!=FW_SUCCESS)
			return FW_RESOLVE_FAIL;

		/* add to linked list */
                tmp->next=calloc(1, r_size);
                tmp=(FW_RULE *)tmp->next;
                memcpy(tmp, new, r_size);
		tmp->next=NULL;
        }
	/* can we insert to iptables? */
        if (FW_Is_NAT_Enable()==FW_SUCCESS)
        	bnat=1;
        else
                bnat=0;
        if (fw_run_rule(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM, bnat)==FW_FAIL)
        {
                FW_Free_Rule(rule);
                return FW_RULE_ADD_FAIL;
        }

	/* write to rule file */
        fw_write_rule(rule);

	/* free memory */
        FW_Free_Rule(rule);
	FW_Refresh_NAT(NULL);
        return FW_SUCCESS;
}

int FW_Del_Rule(FW_RULE *del)
{
        FW_RULE			*rule, *tmp, *parent;
        int                     count=1;

        rule=FW_Get_Rule();
        if (rule==NULL)
                return FW_RULE_NOT_FOUND;
        else
        {
                tmp=rule;
                parent=rule;
                while (tmp!=NULL)
                {
                        if (!fw_comp_rule(tmp, del))
                        {
                                if (tmp==rule)
                                {
                                        rule=(FW_RULE *)rule->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_rule(rule);
                                FW_Free_Rule(rule);
				fw_remove_rule(count);
				FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_RULE *)tmp->next;
			count++;
                }
        }
        FW_Free_Rule(rule);
	FW_Refresh_NAT(NULL);
	return FW_RULE_NOT_FOUND;
}

int FW_Ren_Rule(FW_RULE *new, FW_RULE *old)
{
        FW_RULE                 *rule, *tmp, *check;
	int			bnat;

	if (!fw_comp_rule(new, old))
		return FW_SUCCESS;
        rule=FW_Get_Rule();
        if (rule==NULL)
                return FW_RULE_NOT_FOUND;
        else
        {
                tmp=rule;
                while (tmp!=NULL)
                {
                        if (!fw_comp_rule(tmp, old))
                        {
                                /* check if duplicate */
                                check=rule;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if (!fw_comp_rule(check, new))
                                                {
                                                        FW_Free_Rule(rule);
                                                        return FW_RULE_DUPLICATE;
                                                }
                                        }
                                        check=(FW_RULE *)check->next;
                                }

				/* copy new to tmp */
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_RULE));
				if (fw_rule_valid(new)==FW_SUCCESS)
					new->valid=FW_VALID;
				else
					new->valid=FW_INVALID;

		                /* can we insert to iptables? */
                		if (FW_Is_NAT_Enable()==FW_SUCCESS)
		                        bnat=1;
                		else
		                        bnat=0;
                		if (fw_run_rule(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM, bnat)==FW_FAIL)
		                {
                		        FW_Free_Rule(rule);
		                        return FW_RULE_REN_FAIL;
                		}
				/* replace to iptables */
                                fw_write_rule(rule);
                                FW_Free_Rule(rule);
				FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_RULE *)tmp->next;
                }
        }
        FW_Free_Rule(rule);
	FW_Refresh_NAT(NULL);
        return FW_RULE_NOT_FOUND;
}

int FW_Move_Rule(FW_RULE *select, int action)
{
	FW_RULE		*rule, *ptr, tmp, *parent;
	int		cnt=1, total, r_size;

	total=fw_count_rule();
	if (total<2)
		return FW_SUCCESS;
	rule=FW_Get_Rule();
	parent=ptr=rule;
	r_size=sizeof(FW_RULE);
	while (ptr!=NULL)
	{
		/* find match rule */
		if (!fw_comp_rule(ptr, select))
		{
			if (cnt==total && action==FW_RULE_MOVE_DOWN)
				break;
			else
			if (cnt==1 && action==FW_RULE_MOVE_UP)
				break;
			if (action==FW_RULE_MOVE_UP)
			{
				memcpy(&tmp, ptr, r_size);
				memcpy(ptr, parent, r_size);
				memcpy(parent, &tmp, r_size);
				parent->next=(struct FW_RULE *)ptr;
				ptr->next=tmp.next;
				
			}
			else
			if (action==FW_RULE_MOVE_DOWN)
			{
				if (cnt!=1)
					parent->next=ptr->next;
				parent=(FW_RULE *)parent->next;
				ptr->next=parent->next;
				parent->next=(struct FW_RULE*)ptr;
				if (cnt==1)
					rule=parent;
			}
			/* replace to iptables */
			fw_write_rule(rule);
			break;
		}
		parent=ptr;
		ptr=(FW_RULE *)ptr->next;
		cnt++;
	}
	FW_Free_Rule(rule);
	FW_Refresh_NAT(NULL);
	return FW_SUCCESS;
}

int FW_Start_Rule(int bnat)
{
	/* why the cnt==2 ?? 					*/
	/* the first rule is reserved to fw_set_web_access() 	*/
	/* this action make lan can always access web control	*/
        int                     cnt=2, ret, num=0;
        FW_RULE			*rule, *tmp;

        rule=FW_Get_Rule();
        tmp=rule;
        while (tmp!=NULL)
        {
		/* check if the rule is valid? */
		ret=fw_rule_valid(tmp);
		if (ret!=FW_SUCCESS)
			tmp->valid=FW_INVALID;
		else
			tmp->valid=FW_VALID;

		/* insert into iptables */
		if (tmp->valid==FW_VALID)
		{
//                	if (fw_run_rule(tmp, FW_IPTABLES_INSERT, cnt, num, bnat)==FW_SUCCESS)
			if (fw_run_rule(tmp, FW_IPTABLES_APPEND, cnt, num, bnat)==FW_SUCCESS)
			{
				cnt++;
				num++;
			}
			else
				tmp->valid=FW_INVALID;
		}
                tmp=(FW_RULE *)tmp->next;
        }
        FW_Free_Rule(rule);
        return FW_SUCCESS;
}

//======================================================================
//      firewall filter function
//======================================================================
int fw_count_filter()
{
        FILE            *fptr;
        FW_HEADER       header;

        if ((fptr=fopen(FW_FILTER_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        fclose(fptr);
        return header.total;
}

int fw_filter_valid(FW_FILTER *filter)
{
        char    buf[ADDR_LEN];

        /* check if we can resolve the ip? */
//        if (FW_Check_Addr(filter->url)==FW_ADDR_DOMAIN_NAME)
//        {
                if (FW_Resolve_Addr(filter->url, buf, ADDR_LEN)!=FW_SUCCESS)
                        return FW_RESOLVE_FAIL;
		else
			return FW_SUCCESS;
//        }
//        return FW_FAIL;
}

int fw_run_filter(FW_FILTER *filter, int action, int position, int num, int bnat)
{
        char            command[FW_LINE];
        int             ret=0;
	char		nattable[256]={""}, which_table[256]={""};

	if (bnat)
	{
		sprintf(nattable, "-t nat");
		sprintf(which_table, "PREROUTING");
	}
	else
	{
		sprintf(nattable, " ");
		sprintf(which_table, "FORWARD");
	}
        if (filter!=NULL)
        {
		/* now is never used		*/
		/* code is not ready for ULOG	*/
                if (action==FW_IPTABLES_INSERT)
                {
                        sprintf(command, "%s %s -I %s %d -d %s -j DROP %s",
                                iptables,
				nattable,
				which_table,
                                position,
                                filter->url,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
                }
                else
		/* now is never used		*/
		/* code is not ready for ULOG	*/
                if (action==FW_IPTABLES_REPLACE)
                {
                        sprintf(command, "%s %s -R %s %d -d %s -j DROP %s",
                                iptables,
				nattable,
				which_table,
                                position,
                                filter->url,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
                }
		else
		if (action==FW_IPTABLES_APPEND)
		{
#ifndef ULOG
			sprintf(command, "%s %s -A %s -d %s -j DROP %s",
				iptables,
				nattable,
				which_table
				filter->url,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
                        if (num==0)
                        {       /* create new filter chain      */
                                sprintf(command, "%s %s -N filter %s",
                                        iptables,
					nattable,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                {
                                        sprintf(command, "%s %s -F filter %s",
                                                iptables,
						nattable,
                                                devnull);
					system(command);
                                }
                                sprintf(command, "%s %s -A filter -j ULOG --ulog-nlgroup 32 --ulog-prefix \"filter\" %s",
                                        iptables,
					nattable,
                                        devnull);
                                ret=system(command);
                                sprintf(command, "%s %s -A filter -j DROP %s",
                                        iptables,
					nattable,
                                        devnull);
                                system(command);
				if (ret!=0)
					return FW_FAIL;
                        }
                        if (filter->ulog)
                        {
                                sprintf(command, "%s %s -A %s -d %s -j filter %s",
                                        iptables,
					nattable,
					which_table,
                                        filter->url,
                                        devnull);
                                system(command);
				if (ret!=0)
					return FW_FAIL;
                        }
                        else
                        {
                                sprintf(command, "%s %s -A %s -d %s -j DROP %s",
                                        iptables,
					nattable,
					which_table,
                                        filter->url,
                                        devnull);
                                ret=system(command);
				if (ret!=0)
					return FW_FAIL;
                        }
#endif
		}
        }
        if (ret==0)
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

int fw_remove_filter(int position)
{
        char    command[FW_LINE];
        int     ret=0, num;

        num=fw_count_valid_rule();
        sprintf(command, "%s -t nat -D PREROUTING %d %s", iptables, position+num, devnull);
#ifdef _DEBUG
        printf("%s<br>\n", command);
#endif
#ifndef _DEBUG
        ret=system(command);
#endif
        return ret;
}

FW_FILTER *FW_Get_Filter()
{
        FILE		*fptr;
        FW_FILTER	*start=NULL, *filter=NULL, tmp_filter;
        int             v_size, ret, count=0;
	FW_HEADER	header;

        if ((fptr=fopen(FW_FILTER_CONF, "r"))==NULL)
                return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_FILTER);
        while (!feof(fptr))
        {
                ret=fread(&tmp_filter, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                filter=calloc(1, v_size);
                                start=filter;
                                count++;
                        }
                        else
                        {
                                filter->next=calloc(1, v_size);
                                filter=(FW_FILTER *)filter->next;
                        }
                        memcpy(filter, &tmp_filter, v_size);
                }
        }
        fclose(fptr);
        return start;
}

void fw_write_filter(FW_FILTER *filter)
{
        FW_FILTER	*tmp;
        int             v_size;
        FILE		*fptr;
	int		total=0;

	tmp=filter;
	while (tmp!=NULL)
	{
		tmp=(FW_FILTER *)tmp->next;
		total++;
	}
        if ((fptr=fopen(FW_FILTER_CONF, "w"))==NULL)
                return;
	fw_write_header(fptr, sizeof(FW_FILTER), total);
        v_size=sizeof(FW_FILTER);
        tmp=filter;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, v_size, fptr);
                tmp=(FW_FILTER *)tmp->next;
        }
        fclose(fptr);
}

void FW_Free_Filter(FW_FILTER *filter)
{
        FW_FILTER	*tmp;

        tmp=filter;
        while (filter!=NULL)
        {
                tmp=(FW_FILTER *)filter->next;
                free(filter);
                filter=tmp;
        }
        return;
}

int fw_comp_filter(FW_FILTER *filter1, FW_FILTER *filter2)
{
        if (!strcmp(filter1->url, filter2->url) && filter1->ulog==filter2->ulog)
                return 0;
        else
                return 1;
}

int FW_Add_Filter(FW_FILTER *new)
{
        FW_FILTER	*filter=NULL, *tmp=NULL;
        int		v_size, count=1, num;
	int		bnat;

        filter=FW_Get_Filter();
        v_size=sizeof(FW_FILTER);
        if (filter==NULL)
        {
		if (fw_filter_valid(new)==FW_SUCCESS)
		{
			new->valid=FW_VALID;
	                filter=calloc(1, v_size);
        	        memcpy(filter, new, v_size);
			filter->next=NULL;
		}
		else
			return FW_FILTER_ADD_FAIL;
        }
        else
        {
                tmp=filter;
                while (tmp!=NULL)
                {
                        if (!fw_comp_filter(tmp, new))
                        {
                                FW_Free_Filter(filter);
                                return FW_FILTER_DUPLICATE;
                        }
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_FILTER *)tmp->next;
                        count++;
                }
		if (fw_filter_valid(new)==FW_SUCCESS)
			new->valid=FW_VALID;
		else
			new->valid=FW_INVALID;
		num=fw_count_valid_rule();

                tmp->next=calloc(1, v_size);
                tmp=(FW_FILTER *)tmp->next;
                memcpy(tmp, new, v_size);
		tmp->next=NULL;
        }
	/* can we insert to iptables? */
        if (FW_Is_NAT_Enable()==FW_SUCCESS)
        	bnat=1;
        else
                bnat=0;
        if (fw_run_filter(tmp, FW_IPTABLES_APPEND, 0, FW_TEST_NUM, bnat)==FW_FAIL)
        {
                FW_Free_Filter(filter);
                return FW_FILTER_ADD_FAIL;
        }

        fw_write_filter(filter);
        FW_Free_Filter(filter);
        FW_Refresh_NAT(NULL);
        return FW_SUCCESS;
}

int FW_Del_Filter(FW_FILTER *del)
{
        FW_FILTER	*filter, *tmp, *parent;
        int             v_size, count=1;

        filter=FW_Get_Filter();
        v_size=sizeof(FW_FILTER);
        if (filter==NULL)
                return FW_FILTER_NOT_FOUND;
        else
        {
                tmp=filter;
                parent=filter;
                while (tmp!=NULL)
                {
                        if (!fw_comp_filter(tmp, del))
                        {
                                if (tmp==filter)
                                {
                                        filter=(FW_FILTER *)filter->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_filter(filter);
                                FW_Free_Filter(filter);
                                fw_remove_filter(count);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_FILTER *)tmp->next;
                        count++;
                }
        }
        FW_Free_Filter(filter);
        return FW_FILTER_NOT_FOUND;
}

int FW_Ren_Filter(FW_FILTER *new, FW_FILTER *old)
{
        FW_FILTER	*filter, *tmp, *check;
        int             v_size, count=1, num;
	int		bnat;

	if (!fw_comp_filter(new, old))
		return FW_SUCCESS;
        filter=FW_Get_Filter();
        v_size=sizeof(FW_FILTER);
        if (filter==NULL)
                return FW_FILTER_NOT_FOUND;
        else
        {
                /* replace old to new vserver */
                tmp=filter;
                while (tmp!=NULL)
                {
                        if (!fw_comp_filter(tmp, old))
                        {
                                /* check if duplicate */
                                check=filter;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if (!fw_comp_filter(check, new))
                                                {
                                                        FW_Free_Filter(filter);
                                                        return FW_FILTER_DUPLICATE;
                                                }
                                        }
                                        check=(FW_FILTER *)check->next;
                                }

                                /* copy new to tmp */
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_FILTER));

		                /* can we insert to iptables? */
                		if (FW_Is_NAT_Enable()==FW_SUCCESS)
		                        bnat=1;
                		else
		                        bnat=0;
                		if (fw_run_filter(tmp, FW_IPTABLES_APPEND, 0, FW_TEST_NUM, bnat)==FW_FAIL)
		                {
                		        FW_Free_Filter(filter);
		                        return FW_FILTER_REN_FAIL;
                		}

                                /* replace to iptables */
				num=fw_count_valid_rule();
//                                fw_run_filter(tmp, FW_IPTABLES_REPLACE, num+count);
                                fw_write_filter(filter);
                                FW_Free_Filter(filter);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_FILTER *)tmp->next;
			if (tmp->valid==FW_VALID)
	                        count++;
                }
        }
        return FW_FILTER_NOT_FOUND;
}

int FW_Start_Filter(int bnat)
{
        int		ret=FW_SUCCESS;
        FW_FILTER	*filter, *tmp;
	int 		num=0;

        filter=FW_Get_Filter();
        tmp=filter;
        while (tmp!=NULL)
        {
                fw_run_filter(tmp, FW_IPTABLES_APPEND, 0, num, bnat);
		num++;
                tmp=(FW_FILTER *)tmp->next;
        }
        FW_Free_Filter(filter);
        return ret;
}

//==============================================
//	routing table  function
//==============================================
int fw_count_routing_table()
{
        FILE            *fptr;
        FW_HEADER       header;

        if ((fptr=fopen(FW_ROUTING_TABLE_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        fclose(fptr);
        return header.total;

/*        FW_ROUTING_TABLE	*table, *tmp;
        int			count=0;

        table=FW_Get_Routing_Table();
        tmp=table;
        while (tmp!=NULL)
        {
                tmp=(FW_ROUTING_TABLE *)tmp->next;
                count++;
        }
        FW_Free_Routing_Table(table);
        return count;*/
}

FW_ROUTING_TABLE *FW_Get_Routing_Table()
{
        FILE            	*fptr;
        FW_ROUTING_TABLE	*start=NULL, *table=NULL, tmp_table;
        int             	v_size, ret, count=0;
	FW_HEADER		header;

        if ((fptr=fopen(FW_ROUTING_TABLE_CONF, "r"))==NULL)
                return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_ROUTING_TABLE);
        while (!feof(fptr))
        {
                ret=fread(&tmp_table, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                table=calloc(1, v_size);
                                start=table;
                                count++;
                        }
                        else
                        {
                                table->next=calloc(1, v_size);
                                table=(FW_ROUTING_TABLE *)table->next;
                        }
                        memcpy(table, &tmp_table, v_size);
                }
        }
        fclose(fptr);
        return start;
}

void FW_Free_Routing_Table(FW_ROUTING_TABLE *table)
{
        FW_ROUTING_TABLE	*tmp;

        tmp=table;
        while (table!=NULL)
        {
                tmp=(FW_ROUTING_TABLE *)table->next;
                free(table);
                table=tmp;
        }
        return;
}

int fw_comp_routing_table(FW_ROUTING_TABLE *table1, FW_ROUTING_TABLE *table2)
{
        if (	!strcmp(table1->dest_addr, table2->dest_addr) && !strcmp(table1->netmask, table2->netmask) &&
		!strcmp(table1->gw, table2->gw) && !strcmp(table1->interface, table2->interface) &&
		table1->metric==table2->metric)
                return 0;
        else
                return 1;
}

void fw_write_routing_table(FW_ROUTING_TABLE *table)
{
        FW_ROUTING_TABLE	*tmp;
        int             	v_size;
        FILE            	*fptr;
	int			total=0;

	tmp=table;
	while (tmp!=NULL)
	{
		tmp=(FW_ROUTING_TABLE *)tmp->next;
		total++;
	}
        if ((fptr=fopen(FW_ROUTING_TABLE_CONF, "w"))==NULL)
                return;
	fw_write_header(fptr, sizeof(FW_ROUTING_TABLE), total);
        v_size=sizeof(FW_ROUTING_TABLE);
        tmp=table;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, v_size, fptr);
                tmp=(FW_ROUTING_TABLE *)tmp->next;
        }
        fclose(fptr);
}

int fw_run_routing_table(FW_ROUTING_TABLE *table, int type)
{
	char	command[FW_LINE];
	int	ret;

	sprintf(command, "/sbin/route %s -net %s netmask %s dev %s gw %s metric %d %s",
			(type==ADD_ROUTING_TABLE)?"add":"del",
			table->dest_addr,
			table->netmask,
			table->interface,
			table->gw,
			table->metric,
			devnull);
#ifdef _DEBUG
	printf("command = %s<br>", command);
#endif
	ret=system(command);
#ifdef _DEBUG
	printf("ret=%d<br>", ret);
#endif
	if (ret==0)
		return FW_SUCCESS;
	else
		return FW_FAIL;
}

int FW_Start_Routing_Table()
{
        int                     ret=FW_SUCCESS;
        FW_ROUTING_TABLE        *table=NULL, *tmp=NULL;
        int                     num=0;

        table=FW_Get_Routing_Table();
        tmp=table;
        while (tmp!=NULL)
        {
                fw_run_routing_table(tmp, ADD_ROUTING_TABLE);
                tmp=(FW_ROUTING_TABLE *)tmp->next;
                num++;
        }
        FW_Free_Routing_Table(table);
        return ret;
}

int FW_Add_Routing_Table(FW_ROUTING_TABLE *new)
{
        FW_ROUTING_TABLE	*table=NULL, *tmp=NULL;
        int             	v_size, ret;

	new->user_define=1;
        table=FW_Get_Routing_Table();
        v_size=sizeof(FW_ROUTING_TABLE);
        if (table==NULL)
        {
                table=calloc(1, v_size);
                memcpy(table, new, v_size);
                table->next=NULL;
		ret=fw_run_routing_table(table, ADD_ROUTING_TABLE);
        }
        else
        {
                tmp=table;
                while (tmp!=NULL)
                {
                        if (!fw_comp_routing_table(tmp, new))
                        {
                                FW_Free_Routing_Table(table);
                                return FW_TABLE_DUPLICATE;
                        }
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_ROUTING_TABLE *)tmp->next;
                }
                tmp->next=calloc(1, v_size);
                tmp=(FW_ROUTING_TABLE *)tmp->next;
                memcpy(tmp, new, v_size);
                tmp->next=NULL;
		ret=fw_run_routing_table(tmp, ADD_ROUTING_TABLE);
        }
	if (ret==FW_SUCCESS)
	        fw_write_routing_table(table);
        FW_Free_Routing_Table(table);
	return ret;
}

int FW_Del_Routing_Table(FW_ROUTING_TABLE *del)
{
        FW_ROUTING_TABLE	*table, *tmp, *parent;
        int             	v_size;

        table=FW_Get_Routing_Table();
        v_size=sizeof(FW_ROUTING_TABLE);
        if (table==NULL)
                return FW_TABLE_NOT_FOUND;
        else
        {
                tmp=table;
                parent=table;
                while (tmp!=NULL)
                {
                        if (!fw_comp_routing_table(tmp, del))
                        {
				fw_run_routing_table(del, DEL_ROUTING_TABLE);
                                if (tmp==table)
                                {
                                        table=(FW_ROUTING_TABLE *)table->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_routing_table(table);
                                FW_Free_Routing_Table(table);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_ROUTING_TABLE *)tmp->next;
                }
        }
        FW_Free_Routing_Table(table);
        return FW_TABLE_NOT_FOUND;
}

int is_user_define_table(FW_ROUTING_TABLE *list, FW_ROUTING_TABLE *item)
{
        FW_ROUTING_TABLE        *tmp;

        if (list==NULL)
                return 0;
        tmp=list;
        while (tmp!=NULL)
        {
                if (    !strcmp(tmp->dest_addr, item->dest_addr) && !strcmp(tmp->netmask, item->netmask) &&
                        !strcmp(tmp->gw, item->gw) && !strcmp(tmp->interface, item->interface) &&
                        tmp->metric==item->metric)
                        return 1;
                tmp=(FW_ROUTING_TABLE *)tmp->next;
        }
        return 0;
}

FW_ROUTING_TABLE *FW_Get_Default_Routing_Table()
{
        char                    str[256];
        FILE                    *fptr;
        struct sockaddr_in      sin;
        int                     cnt=0;
        char                    route_file[]={"/proc/net/route"};
        char                    seps[] = " \t\n";
        char                    *token;
	FW_ROUTING_TABLE	*table=NULL, *ptable=NULL;
	int			vsize, flags;
	FW_ROUTING_TABLE	*list, tmptable;

	list=FW_Get_Routing_Table();
        fptr=fopen(route_file, "r");
        fgets(str, 256, fptr);
	vsize=sizeof(FW_ROUTING_TABLE);
        while (fgets(str, 256, fptr) != NULL)
        {
                token = strtok(str, seps);
                if(token == NULL)
                        continue;

                /* get device name */
                strcpy(tmptable.interface, token);

		/* get dest addr */
                token = strtok(NULL, seps);
                sin.sin_addr.s_addr = (uint32_t)strtoul(token, NULL, 16);
		strcpy(tmptable.dest_addr, inet_ntoa(sin.sin_addr));

                /* get gw address */
                token = strtok(NULL, seps);
                sin.sin_addr.s_addr = (uint32_t)strtoul(token, NULL, 16);
		strcpy(tmptable.gw, inet_ntoa(sin.sin_addr));

		/* get metric */
		token = strtok(NULL, seps);
		flags=atoi(token);
//		if (flags==2)
		if ((flags%2)==0)
			continue;

		token = strtok(NULL, seps);
		token = strtok(NULL, seps);
		token = strtok(NULL, seps);
		tmptable.metric=atoi(token);

		/* get mask */
		token = strtok(NULL, seps);
		sin.sin_addr.s_addr = (uint32_t)strtoul(token, NULL, 16);
		strcpy(tmptable.netmask, inet_ntoa(sin.sin_addr));
                cnt++;

                if (table==NULL)
                {
                        table=calloc(1, vsize);
                        ptable=table;
                }
                else
                {
                        ptable->next=calloc(1, vsize);
                        ptable=(FW_ROUTING_TABLE *)ptable->next;
                }

		memcpy(ptable, &tmptable, vsize);
		if (is_user_define_table(list, ptable))
			ptable->user_define=1;
		else
			ptable->user_define=0;
		ptable->next=NULL;
        }
        fclose(fptr);
	FW_Free_Routing_Table(list);
	if (cnt==0)
		return NULL;
	else
		return table;
}

int FW_Ren_Routing_Table(FW_ROUTING_TABLE *new, FW_ROUTING_TABLE *old)
{
        FW_ROUTING_TABLE	*table, *tmp, *check;
        int             	v_size;

	if (!fw_comp_routing_table(new, old))
		return FW_SUCCESS;
        table=FW_Get_Routing_Table();
        v_size=sizeof(FW_ROUTING_TABLE);
        if (table==NULL)
                return FW_TABLE_NOT_FOUND;
        else
        {
                tmp=table;
                while (tmp!=NULL)
                {
                        if (!fw_comp_routing_table(tmp, old))
                        {
                                /* check if duplicate */
                                check=table;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if (!fw_comp_routing_table(check, new))
                                                {
                                                        FW_Free_Routing_Table(table);
                                                        return FW_TABLE_DUPLICATE;
                                                }
                                        }
                                        check=(FW_ROUTING_TABLE *)check->next;
                                }

				if (fw_run_routing_table(new, ADD_ROUTING_TABLE)!=FW_SUCCESS)
				{
					FW_Free_Routing_Table(table);
					return FW_TABLE_REN_FAIL;
				}
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_ROUTING_TABLE));

                                fw_write_routing_table(table);
                                FW_Free_Routing_Table(table);
				fw_run_routing_table(old, DEL_ROUTING_TABLE);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_ROUTING_TABLE *)tmp->next;
                }
        }
	return FW_SUCCESS;
}
//========================================================
//	special application function
//========================================================
int fw_count_special_application()
{
        FILE            *fptr;
        FW_HEADER       header;

        if ((fptr=fopen(FW_SA_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        fclose(fptr);
        return header.total;
/*        FW_SA			*sa, *tmp;
        int                     count=0;

        sa=FW_Get_Special_Application(0);
        tmp=sa;
        while (tmp!=NULL)
        {
                tmp=(FW_SA *)tmp->next;
                count++;
        }
        FW_Free_Special_Application(sa);
        return count;*/
}

FW_SA *FW_Get_Special_Application(int type)
{
        FILE                    *fptr;
        FW_SA			*start=NULL, *sa=NULL, tmp_sa;
        int                     v_size, ret, count=0;
	FW_HEADER		header;

	if (type==0)
	{
	        if ((fptr=fopen(FW_SA_CONF, "r"))==NULL)
        	        return NULL;
	}
	else
	if (type==1)
	{
		if ((fptr=fopen(FW_SA_DEFAULT_CONF, "r"))==NULL)
			return NULL;
	}
	else
		return NULL;
	fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_SA);
        while (!feof(fptr))
        {
                ret=fread(&tmp_sa, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                sa=calloc(1, v_size);
                                start=sa;
                                count++;
                        }
                        else
                        {
                                sa->next=calloc(1, v_size);
                                sa=(FW_SA *)sa->next;
                        }
                        memcpy(sa, &tmp_sa, v_size);
                }
        }
        fclose(fptr);
        return start;
}

void FW_Free_Special_Application(FW_SA *sa)
{
        FW_SA		*tmp;

        tmp=sa;
        while (sa!=NULL)
        {
                tmp=(FW_SA *)sa->next;
                free(sa);
                sa=tmp;
        }
        return;
}

int fw_comp_fw_port(FW_PORT *port1, FW_PORT *port2)
{
	if (port1->start==0 && port1->end==0 && port2->start==0 && port2->end==0)
		return 0;
	if (port1->protocol==port2->protocol && port1->start==port2->start && port1->end==port2->end)
		return 0;
	else
		return 1;
}

void FW_Debug_SA(FW_SA *sa)
{
	printf("name=%s<br>", sa->name);
	printf("ipaddr=%s<br>", sa->ipaddr);
	printf("range[0].protocol=%d<br>", sa->range[0].protocol);
	printf("range[0].start=%d<br>", sa->range[0].start);
	printf("range[0].end=%d<br>", sa->range[0].end);
        printf("range[1].protocol=%d<br>", sa->range[1].protocol);
        printf("range[1].start=%d<br>", sa->range[1].start);
        printf("range[1].end=%d<br>", sa->range[1].end);
        printf("range[2].protocol=%d<br>", sa->range[2].protocol);
        printf("range[2].start=%d<br>", sa->range[2].start);
        printf("range[2].end=%d<br>", sa->range[2].end);
	printf("enalbe=%d<br>", sa->enable);
}

int fw_comp_special_application(FW_SA *sa1, FW_SA *sa2)
{
        if (    !strcmp(sa1->name, sa2->name) &&
                !fw_comp_fw_port(&(sa1->range[0]), &(sa2->range[0])) &&
		!fw_comp_fw_port(&(sa1->range[1]), &(sa2->range[1])) &&
		!fw_comp_fw_port(&(sa1->range[2]), &(sa2->range[2])) &&
		sa1->enable==sa2->enable &&
		sa1->ulog==sa2->ulog)
                return 0;
        else
	if (!strcmp(sa1->name, sa2->name))
		return FW_SA_NAME_SAME;
	else
	if (	!fw_comp_fw_port(&(sa1->range[0]), &(sa2->range[0])) &&
		!fw_comp_fw_port(&(sa1->range[1]), &(sa2->range[1])) &&
		!fw_comp_fw_port(&(sa1->range[2]), &(sa2->range[2])))
		return FW_SA_PORT_SAME;
	else
                return 1;
}

void fw_write_special_application(FW_SA *sa)
{
        FW_SA			*tmp;
        int                     v_size;
        FILE                    *fptr;
	int			total=0;

	tmp=sa;
	while (tmp!=NULL)
	{
		tmp=(FW_SA *)tmp->next;
		total++;
	}
        if ((fptr=fopen(FW_SA_CONF, "w"))==NULL)
                return;
	fw_write_header(fptr, sizeof(FW_SA), total);
        v_size=sizeof(FW_SA);
        tmp=sa;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, v_size, fptr);
                tmp=(FW_SA *)tmp->next;
        }
        fclose(fptr);
}

//====================================================================
//      src = 0.1.2.3
//      pos is below
//      0 : ip1
//      1 : ip2
//      2 : ip3
//      3 : ip4
//====================================================================
char *parser_ipaddr(char *dest, const char *src, int pos)
{
        char *ptr;

        strcpy(dest, src);
        ptr=strtok(dest, ".");
        if (pos==0)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==1)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==2)
                return ptr;
        ptr=strtok(NULL, ".");
        if (pos==3)
                return ptr;
        return NULL;
}

void get_netgroup(char *netgroup)
{
	NIC_INFO	nic_info;
	int		ip[4], netmask[4], i, mask=0;
	char		buf[256];

	NIC_Get_Info(&nic_info, NIC_LAN01);
	for (i=0;i<4;i++)
	{
		ip[i]=atoi(parser_ipaddr(buf, nic_info.ipaddr, i));
		netmask[i]=atoi(parser_ipaddr(buf, nic_info.netmask, i));
		switch (netmask[i])
		{
			case 255:
				mask+=8;
				break;
			case 254:
				mask+=7;
				break;
			case 252:
				mask+=6;
				break;
			case 248:
				mask+=5;
				break;
			case 240:
				mask+=4;
				break;
			case 224:
				mask+=3;
				break;
			case 192:
				mask+=2;
				break;
			case 64:
				mask+=1;
				break;
		}
	}
	sprintf(netgroup, "%s/%d", nic_info.ipaddr, mask);
}

int fw_run_special_application(FW_SA *sa, int num, FW_TRIGGER *trigger)
{
        char    	command[FW_LINE]={""}, dev[NIC_DEV_LEN], netgroup[256]={"0.0.0.0"};
        int    		ret=0, i, tmp;
        NIC_INFO	nic_info;
	char		landev[NIC_DEV_LEN];

	if (sa==NULL)
		return FW_SUCCESS;
	if (sa->enable==0)
		return FW_SUCCESS;
        NIC_Get_Info(&nic_info, NIC_WAN);
	NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
	NIC_Get_Dev(landev, NIC_DEV_LEN, NIC_LAN01);
	if (Is_WLAN_Supported() && Is_WLAN_Card_Exist() && Is_WLAN_Enabled())
		strcpy(landev, BRIDGE_DEV);
	for (i=0;i<3;i++)
	{
		if (sa->range[i].start==0 && sa->range[i].end==0)
			continue;
#ifdef ULOG
		tmp=num*3+i;
		/* create trigger chain and flush it */
		sprintf(command, "%s -t nat -N trigger%d %s",
			iptables,
			tmp,
			devnull);
		ret=system(command);
		if (ret!=0)
		{
			sprintf(command, "%s -t nat -F trigger%d %s",
				iptables,
				tmp,
				devnull);
			system(command);
		}
		sprintf(command, "%s -t nat -A trigger%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"trigger\" %s",
			iptables,
			tmp,
			devnull);
		ret=system(command);

		get_netgroup(netgroup);
		if (sa->range[i].protocol==FW_PROTOCOL_ALL || sa->range[i].protocol==FW_PROTOCOL_TCP)
		{
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp -s %s --dport %d:%d -j trigger%d %s",
				iptables,
				landev,
				netgroup,
				sa->range[i].start,
				sa->range[i].end,
				tmp,
				devnull);
			ret=system(command);
		}
		if (sa->range[i].protocol==FW_PROTOCOL_ALL || sa->range[i].protocol==FW_PROTOCOL_UDP)
                {
                        sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp -s %s --dport %d:%d -j trigger%d %s",
                                iptables,
                                landev,
                                netgroup,
                                sa->range[i].start,
                                sa->range[i].end,
                                tmp,
                                devnull);
                        ret=system(command);
                }


		/* create new chain and flush new chain	*/
		sprintf(command, "%s -t nat -N sa%d %s",
			iptables,
			tmp,
			devnull);
		ret=system(command);
		if (ret!=0)
		{
			sprintf(command, "%s -t nat -F sa%d %s",
				iptables,
				tmp,
				devnull);
			system(command);
		}
		if (sa->ulog)
		{
			sprintf(command, "%s -t nat -A sa%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"special application\" %s",
				iptables,
				tmp,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
		}

		/* do we trigger it? */
		if (trigger!=NULL)
		{
			if (trigger->enable==1)
			{
				sprintf(command, "%s -t nat -A sa%d -j DNAT --to %s %s",
					iptables,
					tmp,
					trigger->ipaddr,
					devnull);
				system(command);
			}
		}
		/* now destinition is for trigger used, so mark it */
/*		sprintf(command, "%s -t nat -A sa%d -j DNAT --to %s %s",
			iptables,
			tmp,
			sa->ipaddr,
			devnull);
		ret=system(command);
		if (ret!=0)
			return FW_FAIL;*/
#endif
		if (sa->range[i].protocol==FW_PROTOCOL_ALL)
		{
#ifndef ULOG
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp -d %s --dport %d:%d -j DNAT --to %s %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp --dport %d:%d -j DNAT --to %s %s",
				iptables,
				dev,
//				nic_info.ipaddr,
				sa->range[i].start,
				sa->range[i].end,
				sa->ipaddr,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp -d %s --dport %d:%d -j DNAT --to %s %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp --dport %d:%d -j DNAT --to %s %s",
                                iptables,
				dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                sa->ipaddr,
                                devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
//			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp -d %s --dport %d:%d -j sa%d %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp --dport %d:%d -j sa%d %s",
				iptables,
				dev,
//				nic_info.ipaddr,
				sa->range[i].start,
				sa->range[i].end,
				tmp,
				devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
//                        sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp -d %s --dport %d:%d -j sa%d %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp --dport %d:%d -j sa%d %s",
                                iptables,
                                dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                tmp,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
		}
		else
		if (sa->range[i].protocol==FW_PROTOCOL_TCP)
		{
#ifndef ULOG
//			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp -d %s --dport %d:%d -j DNAT --to %s %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp --dport %d:%d -j DNAT --to %s %s",
                                iptables,
				dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                sa->ipaddr,
                                devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
//			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp -d %s --dport %d:%d -j sa%d %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p tcp --dport %d:%d -j sa%d %s",
                                iptables,
                                dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                tmp,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
		}
		else
		if (sa->range[i].protocol==FW_PROTOCOL_UDP)
		{
#ifndef ULOG
//			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp -d %s --dport %d:%d -j DNAT --to %s %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp --dport %d:%d -j DNAT --to %s %s",
                                iptables,
				dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                sa->ipaddr,
                                devnull);
			ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
//			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp -d %s --dport %d:%d -j sa%d %s",
			sprintf(command, "%s -t nat -A PREROUTING -i %s -p udp --dport %d:%d -j sa%d %s",
                                iptables,
                                dev,
//                                nic_info.ipaddr,
                                sa->range[i].start,
                                sa->range[i].end,
                                tmp,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
		}
	}
        if (ret==0)
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

int FW_Add_Special_Application(FW_SA *new)
{
        FW_SA        		*sa=NULL, *tmp=NULL;
        int                     v_size, ret=FW_SUCCESS, retcode;

        sa=FW_Get_Special_Application(0);
        v_size=sizeof(FW_SA);
        if (sa==NULL)
        {
                sa=calloc(1, v_size);
                memcpy(sa, new, v_size);
                sa->next=NULL;
        }
        else
        {
                tmp=sa;
                while (tmp!=NULL)
                {
                        if ((retcode=fw_comp_special_application(tmp, new))!=1)
                        {
                                FW_Free_Special_Application(sa);
				if (retcode==0)
	                                return FW_SA_DUPLICATE;
				else
					return retcode;
                        }
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_SA *)tmp->next;
                }
                tmp->next=calloc(1, v_size);
                tmp=(FW_SA *)tmp->next;
                memcpy(tmp, new, v_size);
                tmp->next=NULL;
        }
        /* can we insert? */
        if (fw_run_special_application(new, FW_TEST_NUM, NULL)==FW_FAIL)
        {
         	FW_Free_Special_Application(sa);
                return FW_SA_ADD_FAIL;
        }
        if (ret==FW_SUCCESS)
                fw_write_special_application(sa);
        FW_Free_Special_Application(sa);
	FW_Refresh_NAT(NULL);
        return ret;
}

int FW_Del_Special_Application(FW_SA *del)
{
        FW_SA       	*sa, *tmp, *parent;
        int             v_size, count=1;

        sa=FW_Get_Special_Application(0);
        v_size=sizeof(FW_SA);
        if (sa==NULL)
                return FW_SA_NOT_FOUND;
        else
        {
                tmp=sa;
                parent=sa;
                while (tmp!=NULL)
                {
                        if (fw_comp_special_application(tmp, del)==0)
                        {
                                if (tmp==sa)
                                {
                                        sa=(FW_SA *)sa->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_special_application(sa);
                                FW_Free_Special_Application(sa);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_SA *)tmp->next;
                        count++;
                }
        }
        FW_Free_Special_Application(sa);
        return FW_SA_NOT_FOUND;
}

int FW_Ren_Special_Application(FW_SA *new, FW_SA *old)
{
        FW_SA       	*sa, *tmp, *check;
        int             v_size, ret;

	if (fw_comp_special_application(new, old)==0)
		return FW_SUCCESS;
        sa=FW_Get_Special_Application(0);
        v_size=sizeof(FW_SA);
        if (sa==NULL)
                return FW_SA_NOT_FOUND;
        else
        {
                /* replace old to new vserver */
                tmp=sa;
                while (tmp!=NULL)
                {
                        if (fw_comp_special_application(tmp, old)==0)
                        {
                                /* check if duplicate */
                                check=sa;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if ((ret=fw_comp_special_application(check, new))!=1)
                                                {
                                                        FW_Free_Special_Application(sa);
							if (ret==0)
	                                                        return FW_SA_DUPLICATE;
							else
								return ret;
                                                }
                                        }
                                        check=(FW_SA *)check->next;
                                }
                                /* copy new to tmp */
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_SA));

	        	        /* can we insert? */
        	        	if (fw_run_special_application(new, FW_TEST_NUM, NULL)==FW_FAIL)
		                {
                		        FW_Free_Special_Application(sa);
		                        return FW_SA_REN_FAIL;
                		}

                                /* replace to iptables */
                                fw_write_special_application(sa);
                                FW_Free_Special_Application(sa);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_SA *)tmp->next;
                }
        }
	return FW_SA_NOT_FOUND;
}

int FW_Check_Special_Application(int protocol, int port, char *dst_ipaddr)
{
	FW_SA		*sa, *tmp;
	int		i;

	sa=FW_Get_Special_Application(0);
	tmp=sa;
	while (tmp!=NULL)
	{
		for (i=0;i<3;i++)
		{
			if (	port>=tmp->range[i].start && port<=tmp->range[i].end &&
				(protocol==tmp->range[i].protocol || tmp->range[i].protocol==FW_PROTOCOL_ALL))
			{
				if (!strcmp(dst_ipaddr, tmp->ipaddr))
				{
					FW_Free_Special_Application(sa);
					return FW_SUCCESS;
				}
				else
				{
					strcpy(tmp->ipaddr, dst_ipaddr);
					fw_write_special_application(sa);
					FW_Free_Special_Application(sa);
					return FW_FAIL;
				}
			}
		}
		tmp=(FW_SA *)tmp->next;
	}
	FW_Free_Special_Application(sa);
	return FW_SUCCESS;
}

int FW_Start_Special_Application(FW_TRIGGER *trigger)
{
        FW_SA			*sa, *tmp;
	int			num=0;

        sa=FW_Get_Special_Application(0);
        tmp=sa;
        while (tmp!=NULL)
        {
		fw_run_special_application(tmp, num, trigger);
		num++;
                tmp=(FW_SA *)tmp->next;
        }
        FW_Free_Special_Application(sa);
        return FW_SUCCESS;
}

//========================================================
//	one-to-one address mapping
//========================================================
int fw_run_mapping(FW_MAPPING *mapping, int action, int position, int num);
int fw_count_mapping()
{
        FILE            *fptr;
        FW_HEADER       header;

        if ((fptr=fopen(FW_MAPPING_CONF, "r"))==NULL)
                return 0;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        fclose(fptr);
        return header.total;
}

FW_MAPPING *FW_Get_Mapping()
{
        FILE                    *fptr;
        FW_MAPPING		*start=NULL, *mapping=NULL, tmp_mapping;
        int                     v_size, ret, count=0;
        FW_HEADER               header;

        if ((fptr=fopen(FW_MAPPING_CONF, "r"))==NULL)
                return NULL;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_MAPPING);
        while (!feof(fptr))
        {
                ret=fread(&tmp_mapping, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                mapping=calloc(1, v_size);
                                start=mapping;
                                count++;
                        }
                        else
                        {
                                mapping->next=calloc(1, v_size);
                                mapping=(FW_MAPPING *)mapping->next;
                        }
                        memcpy(mapping, &tmp_mapping, v_size);
                }
        }
        fclose(fptr);
        return start;
}

void FW_Free_Mapping(FW_MAPPING *mapping)
{
        FW_MAPPING		*tmp;

        tmp=mapping;
        while (mapping!=NULL)
        {
                tmp=(FW_MAPPING *)mapping->next;
                free(mapping);
                mapping=tmp;
        }
        return;
}

void fw_write_mapping(FW_MAPPING *mapping)
{
        FW_MAPPING       	*tmp;
        int                     v_size;
        FILE                    *fptr;
        int                     total=0;

        tmp=mapping;
        while (tmp!=NULL)
        {
                tmp=(FW_MAPPING *)tmp->next;
                total++;
        }
        if ((fptr=fopen(FW_MAPPING_CONF, "w"))==NULL)
                return;
        fw_write_header(fptr, sizeof(FW_MAPPING), total);
        v_size=sizeof(FW_MAPPING);
        tmp=mapping;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, v_size, fptr);
                tmp=(FW_MAPPING *)tmp->next;
        }
        fclose(fptr);
}

int fw_comp_mapping(FW_MAPPING *mapping1, FW_MAPPING *mapping2)
{
	if (	!strcmp(mapping1->srcaddr, mapping2->srcaddr) &&
		!strcmp(mapping1->dstaddr, mapping2->dstaddr) &&
		mapping1->ulog==mapping2->ulog)
                return 0;
        else
                return 1;
}

int FW_Add_Mapping(FW_MAPPING *new)
{
        FW_MAPPING		*mapping=NULL, *tmp=NULL;
        int                     v_size, count=1;

        mapping=FW_Get_Mapping();
        v_size=sizeof(FW_MAPPING);
        if (mapping==NULL)
        {
                mapping=calloc(1, v_size);
                memcpy(mapping, new, v_size);
                mapping->next=NULL;
        }
        else
        {
                tmp=mapping;
                while (tmp!=NULL)
                {
                        if (!fw_comp_mapping(tmp, new))
                        {
                                FW_Free_Mapping(mapping);
                                return FW_MAPPING_DUPLICATE;
                        }
			/* source ip address can not be the same */
			if (!strcmp(tmp->srcaddr, new->srcaddr))
			{
				FW_Free_Mapping(mapping);
				return FW_MAPPING_SRC_DUPLICATE;
			}
			if (!strcmp(tmp->dstaddr, new->dstaddr))
			{
				FW_Free_Mapping(mapping);
				return FW_MAPPING_DST_DUPLICATE;
			}
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_MAPPING *)tmp->next;
                        count++;
                }
                tmp->next=calloc(1, v_size);
                tmp=(FW_MAPPING *)tmp->next;
                memcpy(tmp, new, v_size);
                tmp->next=NULL;
        }
	if (fw_run_mapping(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM)==FW_FAIL)
	{
		FW_Free_Mapping(mapping);
		return FW_MAPPING_ADD_FAIL;
	}
        fw_write_mapping(mapping);
        FW_Free_Mapping(mapping);
        FW_Refresh_NAT(NULL);
        return FW_SUCCESS;
}

int FW_Del_Mapping(FW_MAPPING *del)
{
        FW_MAPPING       	*mapping, *tmp, *parent;
        int                     v_size, count=1;
	int			total, i;
	char			ifconfig[]={"/sbin/ifconfig"}, command[1024], dev[NIC_DEV_LEN];

	total=fw_count_mapping();
        mapping=FW_Get_Mapping();
        v_size=sizeof(FW_MAPPING);
        if (mapping==NULL)
                return FW_MAPPING_NOT_FOUND;
        else
        {
                tmp=mapping;
                parent=mapping;
                while (tmp!=NULL)
                {
                        if (!fw_comp_mapping(tmp, del))
                        {
                                if (tmp==mapping)
                                {
                                        mapping=(FW_MAPPING *)mapping->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_mapping(mapping);
                                FW_Free_Mapping(mapping);
				/* del all alias address */
				NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
				for (i=0;i<total;i++)
				{
					sprintf(command, "%s %s:%d 0.0.0.0 %s", ifconfig, dev, i, devnull);
					system(command);
				}
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_MAPPING *)tmp->next;
                        count++;
                }
        }
        FW_Free_Mapping(mapping);
        return FW_MAPPING_NOT_FOUND;
}

int FW_Ren_Mapping(FW_MAPPING *new, FW_MAPPING *old)
{
        FW_MAPPING       	*mapping, *tmp, *check;
        int                     v_size, count=1, count_src, count_dst;

	if (!fw_comp_mapping(new, old))
		return FW_SUCCESS;
        mapping=FW_Get_Mapping();
        v_size=sizeof(FW_MAPPING);
        if (mapping==NULL)
                return FW_MAPPING_NOT_FOUND;
        else
        {
                /* replace old to new vserver */
                tmp=mapping;
                while (tmp!=NULL)
                {
                        if (!fw_comp_mapping(tmp, old))
                        {
				/* check again if srcaddr or dstaddr duplicate */
				check=mapping;
				count_src=count_dst=0;
				while (check!=NULL)
				{
					if (tmp!=check)
					{
						if (!strcmp(check->srcaddr, new->srcaddr))
							count_src++;
						if (!strcmp(check->dstaddr, new->dstaddr))
							count_dst++;
					}
					check=(FW_MAPPING *)check->next;
				}
				if (count_src>0)
				{
					FW_Free_Mapping(mapping);
					return FW_MAPPING_SRC_DUPLICATE;
				}
				else
				if (count_dst>0)
				{
					FW_Free_Mapping(mapping);
					return FW_MAPPING_DST_DUPLICATE;
				}
                                /* copy new to tmp */
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_MAPPING));

				/* can we insert ? */
			        if (fw_run_mapping(new, FW_IPTABLES_APPEND, 0, FW_TEST_NUM)==FW_FAIL)
			        {
			                FW_Free_Mapping(mapping);
			                return FW_MAPPING_REN_FAIL;
			        }

                                /* replace to iptables */
                                fw_write_mapping(mapping);
                                FW_Free_Mapping(mapping);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_MAPPING *)tmp->next;
                        count++;
                }
        }
        return FW_MAPPING_NOT_FOUND;
}

int fw_run_mapping(FW_MAPPING *mapping, int action, int position, int num)
{
        char            command[FW_LINE];
	char		ifconfig[]={"/sbin/ifconfig"}, dev[NIC_DEV_LEN];
        int             ret=0;

        if (mapping!=NULL)
        {
		NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_WAN);
		sprintf(command, "%s %s:%d %s broadcast %s netmask 255.255.255.255 %s", ifconfig, dev, num, mapping->srcaddr, mapping->srcaddr, devnull);
		system(command);
                if (action==FW_IPTABLES_APPEND)
                {
#ifndef ULOG
                        sprintf(command, "%s -t nat -A PREROUTING -d %s -j DNAT --to %s %s",
                                iptables,
                                mapping->srcaddr,
                                mapping->dstaddr,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
#ifdef ULOG
                        /* create new chain */
                        sprintf(command, "%s -t nat -N mapping%d %s",
                                iptables,
                                num,
                                devnull);
                        ret=system(command);
                        if (ret!=0)
                        {       /* chain have already existed, so clear chain */
                                sprintf(command, "%s -t nat -F mapping%d %s",
                                        iptables,
                                        num,
                                        devnull);
                                ret=system(command);
                        }
                        /* add vserver and jump to chain */
                        sprintf(command, "%s -t nat -A PREROUTING -d %s -j mapping%d %s",
                                iptables,
				mapping->srcaddr,
                                num,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
                        /* add rules in new chain */
                        if (mapping->ulog)
                        {
                                sprintf(command, "%s -t nat -A mapping%d -j ULOG --ulog-nlgroup 32 --ulog-prefix \"mapping\" %s",
                                        iptables,
                                        num,
                                        devnull);
                                ret=system(command);
				if (ret!=0)
					return FW_FAIL;
                        }
                        sprintf(command, "%s -t nat -A mapping%d -j DNAT --to %s %s",
                                iptables,
                                num,
                                mapping->dstaddr,
                                devnull);
                        ret=system(command);
			if (ret!=0)
				return FW_FAIL;
#endif
                }
        }
	if (num==FW_TEST_NUM)
	{
		sprintf(command, "%s %s:%d 0.0.0.0 %s", ifconfig, dev, num, devnull);
		system(command);
	}
        if (ret==0)
                return FW_SUCCESS;
        else
                return FW_FAIL;
}



int FW_Start_Mapping()
{
        int                     ret=FW_SUCCESS;
        FW_MAPPING       	*mapping, *tmp;
        int                     num=0;
	char			command[FW_LINE];
	char			dev[NIC_DEV_LEN];

        mapping=FW_Get_Mapping();
        tmp=mapping;

	// add by Kent 2003/02/25
	// make nat client can directly access public IP address
	if (tmp!=NULL)
	{
		NIC_Get_Dev(dev, NIC_DEV_LEN, NIC_LAN01);
		if (Is_WLAN_Supported() && Is_WLAN_Card_Exist() && Is_WLAN_Enabled())
			strcpy(dev, BRIDGE_DEV);
		sprintf(command, "%s -t nat -A POSTROUTING -o %s -p tcp -j MASQUERADE %s",
			iptables,
			dev,
			devnull);
		system(command);
		sprintf(command, "%s -t nat -A POSTROUTING -o %s -p udp -j MASQUERADE %s",
			iptables,
			dev,
			devnull);
		system(command);
	}
	// end

        while (tmp!=NULL)
        {
                fw_run_mapping(tmp, FW_IPTABLES_APPEND, 0, num);
                tmp=(FW_MAPPING *)tmp->next;
                num++;
        }
        FW_Free_Mapping(mapping);
        return ret;
}

void FW_Reset()
{
	if (NIC_Is_Support_Router())
	{
		FW_Reset_Router(0);
		FW_Reset_Vserver(0);
		FW_Reset_Mapping(0);
		FW_Reset_Special_Application(0);
		FW_Reset_Filter(0);
		FW_Reset_Rule(0);
		FW_Reset_Vdmz(0);
		FW_Begin_NAT(NULL);
	}
}

void FW_Reset_Router(int restart)
{
	FW_ROUTING_TABLE	*table=NULL, *ptr;

	table=FW_Get_Routing_Table();
	if (table==NULL)
		return;
	ptr=table;
	while (ptr!=NULL)
	{
		FW_Del_Routing_Table(ptr);
		ptr=(FW_ROUTING_TABLE *)ptr->next;
	}
	FW_Free_Routing_Table(table);
	if (restart)
		FW_Begin_NAT(NULL);
}

void FW_Reset_Vserver(int restart)
{
	FW_VIRTUAL_SERVER	*vserver=NULL, *ptr;

	vserver=FW_Get_Vserver();
	if (vserver==NULL)
		return;
	ptr=vserver;
	while (ptr!=NULL)
	{
		FW_Del_Vserver(ptr);
		ptr=(FW_VIRTUAL_SERVER *)ptr->next;
	}
	FW_Free_Vserver(vserver);
        if (restart)
                FW_Begin_NAT(NULL);
}

void FW_Reset_Mapping(int restart)
{
        FW_MAPPING	*mapping=NULL, *ptr;

        mapping=FW_Get_Mapping();
        if (mapping==NULL)
                return;
        ptr=mapping;
        while (ptr!=NULL)
        {
                FW_Del_Mapping(ptr);
                ptr=(FW_MAPPING *)ptr->next;
        }
        FW_Free_Mapping(mapping);
        if (restart)
                FW_Begin_NAT(NULL);
}

void FW_Reset_Special_Application(int restart)
{
        FW_SA	*sa=NULL, *ptr;

        sa=FW_Get_Special_Application(0);
        if (sa==NULL)
                return;
        ptr=sa;
        while (ptr!=NULL)
        {
                FW_Del_Special_Application(ptr);
                ptr=(FW_SA *)ptr->next;
        }
        FW_Free_Special_Application(sa);
        if (restart)
                FW_Begin_NAT(NULL);
}

void FW_Reset_Filter(int restart)
{
        FW_FILTER	*filter=NULL, *ptr;

        filter=FW_Get_Filter();
        if (filter==NULL)
                return;
        ptr=filter;
        while (ptr!=NULL)
        {
                FW_Del_Filter(ptr);
                ptr=(FW_FILTER *)ptr->next;
        }
        FW_Free_Filter(filter);
        if (restart)
                FW_Begin_NAT(NULL);
}

void FW_Reset_Rule(int restart)
{
        FW_RULE	*rule=NULL, *ptr;

        rule=FW_Get_Rule();
        if (rule==NULL)
                return;
        ptr=rule;
        while (ptr!=NULL)
        {
                FW_Del_Rule(ptr);
                ptr=(FW_RULE *)ptr->next;
        }
        FW_Free_Rule(rule);
        if (restart)
                FW_Begin_NAT(NULL);
}

void FW_Reset_Vdmz(int restart)
{
	FW_VIRTUAL_DMZ	*vdmz;

	vdmz=FW_Get_Vdmz();
	if (vdmz==NULL)
		return;
	strcpy(vdmz->addr, "0.0.0.0");
	vdmz->ulog=0;
	vdmz->klog=0;
	FW_Modify_Vdmz(vdmz);
	FW_Free_Vdmz(vdmz);
	FW_Set_DMZ(0);
        if (restart)
                FW_Begin_NAT(NULL);
}

//==================================================
//	below is string filter functions
//==================================================
int fw_run_string_filter(FW_STRING_FILTER *string, int action, int position, int num, int bnat)
{
        char            command[FW_LINE];
        int             ret=0;
        char            nattable[256]={""}, which_table[256]={""};

	if (bnat)
        {
                sprintf(nattable, " ");
                sprintf(which_table, "FORWARD");
        }
        else
        {
                sprintf(nattable, " ");
                sprintf(which_table, "FORWARD");
        }
        if (string!=NULL)
        {
                /* now is never used            */
                /* code is not ready for ULOG   */
                if (action==FW_IPTABLES_INSERT)
                {
                        sprintf(command, "%s %s -I %s %d -p tcp --sport 80 -m string --string '%s' -j DROP %s",
                                iptables,
                                nattable,
                                which_table,
                                position,
                                string->string,
                                devnull);
                        ret=system(command);
                        if (ret!=0)
                                return FW_FAIL;
                }
                else
                /* now is never used            */
                /* code is not ready for ULOG   */
                if (action==FW_IPTABLES_REPLACE)
                {
                        sprintf(command, "%s %s -R %s %d -p tcp --sport 80 -m string --string '%s' -j DROP %s",
                                iptables,
                                nattable,
                                which_table,
                                position,
                                string->string,
                                devnull);
                        ret=system(command);
                        if (ret!=0)
                                return FW_FAIL;
                }
                else
                if (action==FW_IPTABLES_APPEND)
                {
#ifndef ULOG
                        sprintf(command, "%s %s -A %s -p tcp --sport 80 -m string --string '%s' -j DROP %s",
                                iptables,
                                nattable,
                                which_table
                                string->string,
                                devnull);
                        ret=system(command);
                        if (ret!=0)
                                return FW_FAIL;
#endif
#ifdef ULOG
                        if (num==0)
                        {       /* create new filter chain      */
                                sprintf(command, "%s %s -N str_pkt %s",
                                        iptables,
                                        nattable,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                {
                                        sprintf(command, "%s %s -F str_pkt %s",
                                                iptables,
                                                nattable,
                                                devnull);
                                        system(command);
                                }
                                sprintf(command, "%s %s -A str_pkt -j ULOG --ulog-nlgroup 32 --ulog-prefix \"string\" %s",
                                        iptables,
                                        nattable,
                                        devnull);
                                ret=system(command);
                                sprintf(command, "%s %s -A str_pkt -j DROP %s",
                                        iptables,
                                        nattable,
                                        devnull);
                                system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        if (string->ulog)
                        {
                                sprintf(command, "%s %s -A %s -p tcp --sport 80 -m string --string '%s' -j str_pkt %s",
                                        iptables,
                                        nattable,
                                        which_table,
                                        string->string,
                                        devnull);
                                system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
                        else
                        {
                                sprintf(command, "%s %s -A %s -p tcp --sport 80 -m string --string '%s' -j DROP %s",
                                        iptables,
                                        nattable,
                                        which_table,
                                        string->string,
                                        devnull);
                                ret=system(command);
                                if (ret!=0)
                                        return FW_FAIL;
                        }
#endif
                }
        }
        if (ret==0)
                return FW_SUCCESS;
        else
                return FW_FAIL;
}

FW_STRING_FILTER *FW_Get_String_Filter()
{
        FILE            	*fptr;
        FW_STRING_FILTER	*start=NULL, *string=NULL, tmp_string;
        int             	v_size, ret, count=0;
        FW_HEADER       	header;

        if ((fptr=fopen(FW_STRING_FILTER_CONF, "r"))==NULL)
                return NULL;
        fread(&header, 1, sizeof(FW_HEADER), fptr);
        v_size=sizeof(FW_STRING_FILTER);
        while (!feof(fptr))
        {
                ret=fread(&tmp_string, 1, v_size, fptr);
                if (ret==v_size)
                {
                        if (count==0)
                        {
                                string=calloc(1, v_size);
                                start=string;
                                count++;
                        }
                        else
                        {
                                string->next=calloc(1, v_size);
                                string=(FW_STRING_FILTER *)string->next;
                        }
                        memcpy(string, &tmp_string, v_size);
                }
        }
        fclose(fptr);
        return start;
}

void fw_write_string_filter(FW_STRING_FILTER *string)
{
        FW_STRING_FILTER       	*tmp;
        int             	v_size;
        FILE            	*fptr;
        int             	total=0;

        tmp=string;
        while (tmp!=NULL)                                                                                                  {
                tmp=(FW_STRING_FILTER *)tmp->next;
                total++;
        }
        if ((fptr=fopen(FW_STRING_FILTER_CONF, "w"))==NULL)
                return;
        fw_write_header(fptr, sizeof(FW_STRING_FILTER), total);
        v_size=sizeof(FW_STRING_FILTER);
        tmp=string;
        while (tmp!=NULL)
        {
                fwrite(tmp, 1, v_size, fptr);
                tmp=(FW_STRING_FILTER *)tmp->next;
        }
        fclose(fptr);
}

void FW_Free_String_Filter(FW_STRING_FILTER *string)
{
        FW_STRING_FILTER	*tmp;

        tmp=string;
        while (string!=NULL)
        {
                tmp=(FW_STRING_FILTER *)string->next;
                free(string);
                string=tmp;
        }
        return;
}

int fw_comp_string_filter(FW_STRING_FILTER *string1, FW_STRING_FILTER *string2)
{
        if (!strcmp(string1->string, string2->string) && string1->ulog==string2->ulog)
                return 0;
        else
                return 1;
}

int FW_Add_String_Filter(FW_STRING_FILTER *new)
{
	FW_STRING_FILTER	*string=NULL, *tmp=NULL;
        int             	v_size, count=1;
        int             	bnat;

        string=FW_Get_String_Filter();
        v_size=sizeof(FW_STRING_FILTER);
        if (string==NULL)
        {
		string=calloc(1, v_size);
                memcpy(string, new, v_size);
                string->next=NULL;
        }
        else
        {
                tmp=string;
                while (tmp!=NULL)
                {
                        if (!fw_comp_string_filter(tmp, new))
                        {
                                FW_Free_String_Filter(string);
                                return FW_STRING_FILTER_DUPLICATE;
                        }
                        if (tmp->next==NULL)
                                break;
                        else
                                tmp=(FW_STRING_FILTER *)tmp->next;
                        count++;
                }

                tmp->next=calloc(1, v_size);
                tmp=(FW_STRING_FILTER *)tmp->next;
                memcpy(tmp, new, v_size);
                tmp->next=NULL;
        }
        /* can we insert to iptables? */
        if (FW_Is_NAT_Enable()==FW_SUCCESS)
                bnat=1;
        else
                bnat=0;
        if (fw_run_string_filter(tmp, FW_IPTABLES_APPEND, 0, FW_TEST_NUM, bnat)==FW_FAIL)
        {
                FW_Free_String_Filter(string);
                return FW_STRING_FILTER_ADD_FAIL;
        }

        fw_write_string_filter(string);
        FW_Free_String_Filter(string);
        FW_Refresh_NAT(NULL);
        return FW_SUCCESS;
}

int FW_Del_String_Filter(FW_STRING_FILTER *del)
{
        FW_STRING_FILTER	*string, *tmp, *parent;
        int             	v_size, count=1;

        string=FW_Get_String_Filter();
        v_size=sizeof(FW_STRING_FILTER);
        if (string==NULL)
                return FW_STRING_FILTER_NOT_FOUND;
        else
        {
                tmp=string;
                parent=string;
                while (tmp!=NULL)
                {
                        if (!fw_comp_string_filter(tmp, del))
                        {
                                if (tmp==string)
                                {
                                        string=(FW_STRING_FILTER *)string->next;
                                        free(tmp);
                                }
                                else
                                {
                                        parent->next=tmp->next;
                                        free(tmp);
                                }
                                fw_write_string_filter(string);
                                FW_Free_String_Filter(string);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        parent=tmp;
                        tmp=(FW_STRING_FILTER *)tmp->next;
                        count++;
                }
        }
        FW_Free_String_Filter(string);
        return FW_STRING_FILTER_NOT_FOUND;
}

int FW_Ren_String_Filter(FW_STRING_FILTER *new, FW_STRING_FILTER *old)
{
        FW_STRING_FILTER	*string, *tmp, *check;
        int             	v_size, count=1;

        if (!fw_comp_string_filter(new, old))
                return FW_SUCCESS;
        string=FW_Get_String_Filter();
        v_size=sizeof(FW_STRING_FILTER);
        if (string==NULL)
                return FW_STRING_FILTER_NOT_FOUND;
        else
        {
                /* replace old to new string filter */
                tmp=string;
                while (tmp!=NULL)
                {
                        if (!fw_comp_string_filter(tmp, old))
                        {
                                /* check if duplicate */
                                check=string;
                                while (check!=NULL)
                                {
                                        if (tmp!=check)
                                        {
                                                if (!fw_comp_string_filter(check, new))
                                                {
                                                        FW_Free_String_Filter(string);
                                                        return FW_STRING_FILTER_DUPLICATE;
                                                }
                                        }
                                        check=(FW_STRING_FILTER *)check->next;
                                }

                                /* copy new to tmp */
                                new->next=tmp->next;
                                memcpy(tmp, new, sizeof(FW_STRING_FILTER));

                                /* replace to iptables */
                                fw_write_string_filter(string);
                                FW_Free_String_Filter(string);
                                FW_Refresh_NAT(NULL);
                                return FW_SUCCESS;

                        }
                        tmp=(FW_STRING_FILTER *)tmp->next;
                        count++;
                }
        }
        return FW_STRING_FILTER_NOT_FOUND;
}

int FW_Start_String_Filter(int bnat)
{
        int             	ret=FW_SUCCESS;
        FW_STRING_FILTER	*string, *tmp;
        int             	num=0;

        string=FW_Get_String_Filter();
        tmp=string;
        while (tmp!=NULL)
        {
                fw_run_string_filter(tmp, FW_IPTABLES_APPEND, 0, num, bnat);
                num++;
                tmp=(FW_STRING_FILTER *)tmp->next;
        }
        FW_Free_String_Filter(string);
        return ret;
}

