//**************************************************************************
//
//	Copyright (c) 2001  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		cfg_dhcp.c
//
//	Abstract: 
//		System configuration Access Functions.
//
//	FUNCTIONS:	Setting and getting the dhcp configuration from uLinux.
//		public function:
//			BOOL Is_DHCP_Enable (void);
//			BOOL Enable_DHCP (int Enable);
//			int Get_DHCP_Range (int pos, char * buf, int size);
//			BOOL Set_DHCP_Range (int pos, char * range);
//			int Get_Reserve_Range (int index, int pos, char * buf, int size);
//			BOOL Set_Reserve_Range (int index, int pos, char * range);
//			int Get_Max_Lease_Time (void);
//			int Get_Max_Lease_Time_Day (void);
//			int Get_Max_Lease_Time_Hour (void);
//			BOOL Set_Max_Lease_Time (int time);
//			BOOL Set_Max_Lease_Time_Day (int time);
//			BOOL Set_Max_Lease_Time_Hour (int time);
//			Device_t Get_DHCP_Device (void);
//			BOOL Set_DHCP_Device (int device);
//if not use new function{
//	if defined use private config file {
//			int GetPrivateProfileInteger (char * section_name, char * key_name,
//							int default_value, char * filename);
//	}
//	else {
//			int GetProfileInteger (char * section_name, char * key_name, 
//							int default_value);
//	}
//
//
//			BOOL GetProfileBOOL (char * section_name, char * key_name, 
//							BOOL default_value);
//}
//		private function:
//			BOOL set_dhcp_profile_bool (char * key_name, BOOL value);
//			int get_dns (char * buf, int size);
//			BOOL set_dns (char * buf);
//
//			int get_ex_dns (char * buf, int size);
//			BOOL set_ex_dns (char * buf);
//
//	COMMENTS: 	It must working with util.c
//
//	HISTORY:
//		        04/21/2004      YFHuang add function
//					get_ex_dns (...)
//					get_ex_dns2 (...)
//					set_ex_dns (...)
//					set_ex_dns2 (...)
//
//		v1.02	05/17/2002	Tiger fu fix bug (PTR 697)
//
//		V1.01	03/05/2002	Tiger Fu add function
//					get_dns2 (...)
//					set_dns2 (...)
//
//		V1.00	07/16/01	Tiger Fu created
//
//**************************************************************************

#include	<stdio.h>
#include	<ctype.h>
#include	<stdlib.h>
#include	<string.h>

#include	"Util.h"
#include	"cfg_nic.h"
#include	"cfg_dns.h"
#include	"cfg_fw.h"

#include	"cfg_dhcp.h"

#define		DHCP_BUF_SIZE		1024
#define		NUMBER_OF_DEVICE	7

//#define		_WRITE_TO_LOG

#ifdef	_WRITE_TO_LOG
  #include	<eventlog.h>
#endif
//  uLinux.conf
//#define		DHCP_SECTION		"DHCP Server"

//  system command
#define		REMOVE			"rm"
#define		LINK_COMMAND		"ln"
#define		DEVICE_NULL		"/dev/null"
#define		DHCPD_SHELL_SCRIPT	"/etc/init.d/dhcpd.sh"

BOOL set_dhcp_profile_bool (char *, BOOL value);
//**************************************************************************
//	Function: 	Is_DHCP_Enable
//	Description:	Check DHCP whether enable or eisable, base on 
//			contents of uLinux.conf.
//	Parameter:	None
//	Return value:	TRUE = DHCP enable | FALSE = DHCP disable
//
BOOL Is_DHCP_Enable (void)
{
#ifdef	_USE_NEW_FUNCTION
	return Get_Profile_Boolean (DHCP_SECTION, "Enable", FALSE);
#else
	return GetProfileBOOL (DHCP_SECTION, "Enable", FALSE);
#endif
}

//**************************************************************************
//	Function: 	Enable_DHCP
//	Description:	Start or stop DHCP serveice
//	Parameter:	int Enable(0=disable, 1=enable).
//	Return value:	
//
BOOL Enable_DHCP (BOOL Enable)
{
	BOOL result = TRUE;
	char command[256];
	memset (command, '\0', sizeof (command));

	switch (Enable)
	{
	case FALSE:
		result = SET_PROFILE_STRING (DHCP_SECTION, "Enable", "FALSE");
		sprintf (command, "%s stop > %s", DHCPD_SHELL_SCRIPT, DEVICE_NULL);
		system (command);
		break;
	case TRUE:
		result = SET_PROFILE_STRING (DHCP_SECTION, "Enable", "TRUE");
		sprintf (command,"%s restart > %s", DHCPD_SHELL_SCRIPT, DEVICE_NULL);
		system (command);
		break;
	}
	return result;
}

//**************************************************************************
//	Function: 	Get_DHCP_Range
//	Description:	Get range setting from uLinux.conf
//	Parameter:	int pos(address of start or end, 0=start, 1=end).
//	Return value:	number of characters copied to the buffer, 
//			-1 = position error, the position only START_POSITION(0) and END_POSITION(1)
//
int Get_DHCP_Range (int pos, char * buf, int size)
{
	int result = -1;
	
	memset (buf, '\0', size);
	
	switch (pos)
	{
	case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
		result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Start Range",
						 "0.0.0.0", buf, size, DHCP_CONF);
#else
		result = GET_PROFILE_STRING (DHCP_SECTION, "Start Range",
						 "0.0.0.0", buf, size);
#endif
		break;
	case END_POSITION:
#ifdef	_USE_PRIVATE_CONF
		result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "End Range",
						"0.0.0.0", buf, size, DHCP_CONF);
#else
		result = GET_PROFILE_STRING (DHCP_SECTION, "End Range",
						"0.0.0.0", buf, size);
#endif
		break;
	}
	return result;
}

//**************************************************************************
//	Function: 	Set_DHCP_Range
//	Description:	Set range into uLinux.conf 
//	Parameter:	char * range(pointer to string of range), 
//			int pos(address of start or end, 0=position of start, 1=position of end).
//	Return value:	
//
BOOL Set_DHCP_Range (int pos, char * range)
{
	int result = FALSE;
	switch (pos)
	{
	case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
		result=SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Start Range", range, DHCP_CONF);
#else
		result=SET_PROFILE_STRING (DHCP_SECTION, "Start Range", range);
#endif
		break;
	case END_POSITION:
#ifdef	_USE_PRIVATE_CONF
		result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "End Range", range, DHCP_CONF);
#else
		result = SET_PROFILE_STRING (DHCP_SECTION, "End Range", range);
#endif
		break;
	}
	return result;
}

//**************************************************************************
//	Function: 	Get_Reserve_Range
//	Description:	Get Reserve range from uLinux.conf
//	Parameter:	int index(1 | 2,"this version just provide 2 range"), 
//			int pos(same as Get_DHCP_Range).
//	Return value:	number of characters copied to the buffer
//			-1=parameter error, index must > 2, pos must > 2
//
int Get_Reserve_Range (int index, int pos, char * buf, int size)
{
	int result = -1;
	
	memset (buf, '\0', size);
	
	switch (index)
	{
	case RESERVE_RANGE_1:
		switch (pos)
		{
		case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
				"Reserve Start Range 1", "0.0.0.0", buf, size, DHCP_CONF);
#else
			result = GET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 1", "0.0.0.0", buf, size);
#endif
			break;
		case END_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
				"Reserve End Range 1", "0.0.0.0", buf, size, DHCP_CONF);
#else
			result = GET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 1", "0.0.0.0", buf, size);
#endif
			break;
		default:
			if (size > sizeof ("0.0.0.0"))
				strcpy (buf, "0.0.0.0");
			else
				strcpy (buf, "0");
			break;
		}
		break;
	case RESERVE_RANGE_2:
		switch (pos)
		{
		case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
				"Reserve Start Range 2", "0.0.0.0", buf, size, DHCP_CONF);
#else
			result = GET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 2", "0.0.0.0", buf, size);
#endif
			break;
		case END_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
				"Reserve End Range 2", "0.0.0.0", buf, size, DHCP_CONF);
#else
			result = GET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 2", "0.0.0.0", buf, size);
#endif
			break;
		default:
			if (size > sizeof ("0.0.0.0"))
				strcpy (buf, "0.0.0.0");
			else
				strcpy (buf, "0");
			break;
		}
		break;
		
	}
	return result;
}

//**************************************************************************
//	Function: 	Set_Reserve_Range
//	Description:	Set Reserve range into uLinux.conf
//	Parameter:	char * range(string of address ex:172.16.21.1),
//			int index(1 | 2,"this version just provide 2 range"), 
//			int pos(same as Get_DHCP_Range).
//	Return value:	
//
BOOL Set_Reserve_Range (int index, int pos, char * range)
{
	int result = FALSE;

	switch (index)
	{
	case RESERVE_RANGE_1:
		switch (pos)
		{
		case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 1", range, DHCP_CONF);
#else
			result = SET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 1", range);
#endif
			break;
		case END_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 1", range, DHCP_CONF);
#else
			result = SET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 1", range);
#endif
			break;
		}
		break;
	case RESERVE_RANGE_2:
		switch (pos)
		{
		case START_POSITION:
#ifdef	_USE_PRIVATE_CONF
			result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 2", range, DHCP_CONF);
#else
			result = SET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve Start Range 2", range);
#endif
			break;
		case 1:
#ifdef	_USE_PRIVATE_CONF
			result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 2", range, DHCP_CONF);
#else
			result = SET_PROFILE_STRING (DHCP_SECTION, 
					"Reserve End Range 2", range);
#endif
			break;
		}
	}
	return result;
}

//**************************************************************************
//	Function: 	Get_Max_Lease_Time_Day
//	Description:	Get maxima lease time from uLinux.conf
//	Parameter:	None
//	Return value:	
//
unsigned long Get_Max_Lease_Time_Day (void)
{
#ifdef	_USE_PRIVATE_CONF
	return GET_PRIVATE_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time Day", 1, DHCP_CONF); //  one day
#else
	return GET_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time Day", 1); //  one day
#endif
}

//**************************************************************************
//	Function: 	Get_Max_Lease_Time_Hour
//	Description:	Get maxima lease hour from uLinux.conf
//	Parameter:	None
//	Return value:	
//
unsigned long Get_Max_Lease_Time_Hour (void)
{
#ifdef	_USE_PRIVATE_CONF
	return GET_PRIVATE_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time Hour", 0, DHCP_CONF); // 0 hour default
#else
	return GET_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time Hour", 0); //  0 hour by default
#endif
}

//**************************************************************************
//	Function: 	Get_Max_Lease_Time
//	Description:	Get maxima lease time from uLinux.conf
//	Parameter:	None
//	Return value:	
//
unsigned long Get_Max_Lease_Time (void)
{
#ifdef	_USE_PRIVATE_CONF
	return GET_PRIVATE_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time", 86400, DHCP_CONF); //  one day
#else
	return GET_PROFILE_INTEGER (DHCP_SECTION, "Max Lease Time", 86400); //  one day
#endif
}

//**************************************************************************
//	Function: 	Set_Max_Lease_Time
//	Description:	Set maxima lease time into uLinux.conf
//	Parameter:	int time(time of maxima lease time in second)
//	Return value:	0 = success, greater than 0 = failure
//
BOOL Set_Max_Lease_Time (unsigned long time)  //in second
{
	char number[32];
	int result = FALSE;
	
	memset (number, '\0', sizeof (number));
	sprintf (number, "%lu", time);
#ifdef	_USE_PRIVATE_CONF
	result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Max Lease Time", number, DHCP_CONF);
#else
	result = SET_PROFILE_STRING (DHCP_SECTION, "Max Lease Time", number);
#endif
	return result;
}

//**************************************************************************
//	Function: 	Set_Max_Lease_Time_Day
//	Description:	Set maxima lease time day into uLinux.conf
//	Parameter:	int time(time of maxima lease time in second)
//	Return value:	0 = success, greater than 0 = failure
//
BOOL Set_Max_Lease_Time_Day (unsigned long time) 
{
	char number[32];
	int result = FALSE;
	
	memset (number, '\0', sizeof (number));
	sprintf (number, "%lu", time);
#ifdef	_USE_PRIVATE_CONF
	result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Max Lease Time Day", number, DHCP_CONF);
#else
	result = SET_PROFILE_STRING (DHCP_SECTION, "Max Lease Time Day", number);
#endif
	return result;
}

//**************************************************************************
//	Function: 	Set_Max_Lease_Time_Hour
//	Description:	Set maxima lease time hour into uLinux.conf
//	Parameter:	int time(time of maxima lease time in second)
//	Return value:	0 = success, greater than 0 = failure
//
BOOL Set_Max_Lease_Time_Hour (unsigned long time)
{
	char number[32];
	int result = FALSE;
	
	memset (number, '\0', sizeof (number));
	sprintf (number, "%lu", time);
#ifdef	_USE_PRIVATE_CONF
	result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Max Lease Time Hour", number, DHCP_CONF);
#else
	result = SET_PROFILE_STRING (DHCP_SECTION, "Max Lease Time Hour", number);
#endif
	return result;
}


//**************************************************************************
//	Function: 	Get_DHCP_Device
//	Description:	Get device(The one will be provided DHCP service) 
//			from uLinux.conf
//	Parameter:	None
//	Return value:	0 represent eth0, 1 represent eth1, -1 fail to get
//
int Get_DHCP_Device (void)		
{
	int device = -1;
	char buf[DHCP_BUF_SIZE];
	memset (buf, '\0', sizeof (buf));

	GET_PROFILE_STRING (DHCP_SECTION, "Device", "eth0", buf, sizeof (buf));

	if (!strcmp (buf, "eth0"))
		device = ETH_0;
	else if (!strcmp (buf, "eth1"))
		device = ETH_1;
	else if (!strcmp (buf, "eth2"))
		device = ETH_2;
	else if (!strcmp (buf, "eth3"))
		device = ETH_3;
	else if (!strcmp (buf, "eth4"))
		device = ETH_4;
	else if (!strcmp (buf, "eth5"))
		device = ETH_5;
	else if (!strcmp (buf, "eth6"))
		device = ETH_6;
	else if (!strcmp (buf, "eth7"))
		device = ETH_7;

	return device;
}

//**************************************************************************
//	Function: 	Set_DHCP_Device
//	Description:	Set device setting into uLinux.conf
//	Parameter:	int device(0 represent eth0, 1 represent eth1, and so on..)
//	Return value:	
//
BOOL Set_DHCP_Device (int device)
{
	char device_buf[6];
	int result = FALSE;
	
	memset (device_buf, '\0', sizeof (device_buf));
	
	if ((int)device <= NUMBER_OF_DEVICE)
	{
		sprintf (device_buf, "eth%d", device);
		result = SET_PROFILE_STRING (DHCP_SECTION, "device", device_buf);
	}
	
	return result;
}

//**************************************************************************
//	Function: 	get_dns
//	Description:	Get Domain Name Server 1 from uLinux.conf
//	Parameter:	None
//	Return value:	number of characters copied to the buffer
//
int get_dns (char * buf, int size)
{
	memset (buf, '\0', size);
#ifdef	_USE_PRIVATE_CONF
	return GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 1", "0.0.0.0", buf, size, DHCP_CONF);
#else
	return GET_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 1", "0.0.0.0", buf, size);
#endif
}

//**************************************************************************
//	Function: 	get_dns2
//	Description:	Get Domain Name Server 2 from uLinux.conf
//	Parameter:	None
//	Return value:	number of characters copied to the buffer
//
int get_dns2 (char * buf, int size)
{
	memset (buf, '\0', size);
#ifdef	_USE_PRIVATE_CONF
	return GET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 2", "0.0.0.0", buf, size, DHCP_CONF);
#else
	return GET_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 2", "0.0.0.0", buf, size);
#endif
}

//**************************************************************************
//	Function: 	set_dns
//	Description:	Set Domain Name Server 1 from uLinux.conf
//	Parameter:	char * buf(contents of DNS)
//	Return value:	greater than 0 = success, less than 0 = failure
//
BOOL set_dns (char * buf)
{
#ifdef	_USE_PRIVATE_CONF
	return SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 1", buf, DHCP_CONF);
#else
	return SET_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 1", buf);
#endif
}

//**************************************************************************
//	Function: 	set_dns2
//	Description:	Set Domain Name Server 2 from uLinux.conf
//	Parameter:	char * buf(contents of DNS)
//	Return value:	greater than 0 = success, less than 0 = failure
//
BOOL set_dns2 (char * buf)
{
#ifdef	_USE_PRIVATE_CONF
	return SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 2", buf, DHCP_CONF);
#else
	return SET_PROFILE_STRING (DHCP_SECTION, "Domain Name Server 2", buf);
#endif
}

// YFHuang 2004/04/21 under coding <begin>
//**************************************************************************
//	Function: 	get_ex_dns
//	Description:	Get external Domain Name Server 1 from uLinux.conf
//	Parameter:	None
//	Return value:	number of characters copied to the buffer
//
int get_ex_dns (char * buf, int size)
{
	memset (buf, '\0', size);
	return GET_PROFILE_STRING ("Network", "Domain Name Server 1", "0.0.0.0", buf, size);
}

//**************************************************************************
//	Function: 	get_ex_dns2
//	Description:	Get external Domain Name Server 2 from uLinux.conf
//	Parameter:	None
//	Return value:	number of characters copied to the buffer
//
int get_ex_dns2 (char * buf, int size)
{
	memset (buf, '\0', size);
	return GET_PROFILE_STRING ("Network", "Domain Name Server 2", "0.0.0.0", buf, size);
}

//**************************************************************************
//	Function: 	set_ex_dns
//	Description:	Set external Domain Name Server 1 from uLinux.conf
//	Parameter:	char * buf(contents of DNS)
//	Return value:	greater than 0 = success, less than 0 = failure
//
BOOL set_ex_dns (char * buf)
{

	return SET_PROFILE_STRING ("Network", "Domain Name Server 1", buf);
}

//**************************************************************************
//	Function: 	set_ex_dns2
//	Description:	Set external Domain Name Server 2 from uLinux.conf
//	Parameter:	char * buf(contents of DNS)
//	Return value:	greater than 0 = success, less than 0 = failure
//
BOOL set_ex_dns2 (char * buf)
{
	return SET_PROFILE_STRING ("Network", "Domain Name Server 2", buf);
}
//YFHuang 2004/04/21 <end>

//**************************************************************************
//	Function: 	GetPrivateProfileInteger
//	Description:	Get type of integer of parameter from uLinux.conf
//	Parameter:	
//			
//	Return value:	Returns specify key's integer value
//
#ifndef	_USE_NEW_FUNCTION

#ifdef	_USE_PRIVATE_CONF
int GetPrivateProfileInteger (char * section_name, char * key_name, 
					int default_value, char * filename)
#else
int GetProfileInteger (char * section_name, char * key_name, int default_value)
#endif
{
	char buf[DHCP_BUF_SIZE], number[32];
	int is_digit = 1, len, value = default_value;
	
	memset (buf, '\0', sizeof (buf));
	memset (number, '\0', sizeof (number));
	
	sprintf (number, "%d", default_value);
#ifdef	_USE_PRIVAGE_FUNCTION
	GET_PRIVATE_PROFILE_STRING (section_name, key_name, number, buf, sizeof (buf), filename);
#else
	GET_PROFILE_STRING (section_name, key_name, number, buf, sizeof (buf));
#endif
	len = strlen (buf);
	while (len--)
	{
		if (!isdigit (buf[len]))
		{
			is_digit = 0;
			break;
		}
	}
	
	if (is_digit)
		value = strtol (buf, (char **)NULL, 10); 

	return value;
}

//**************************************************************************
//	Function: 	GetProfileBOOL
//	Description:	Get type of BOOL of parameter from uLinux.conf
//	Parameter:	
//			
//	Return value:	Returns specify key's boolean value
//
BOOL GetProfileBOOL (char * section_name, char * key_name, BOOL default_value)
{
	char buf[DHCP_BUF_SIZE];
	BOOL value = default_value;
	
	memset (buf, '\0', sizeof (buf));
	
	if (default_value)
		GetProfileString (section_name, key_name, "TRUE", buf, sizeof (buf));
	else
		GetProfileString (section_name, key_name, "FALSE", buf, sizeof (buf));

	if (!strcmp (String_To_Upper_Case (buf), "TRUE"))
		value = TRUE;
	else if (!strcmp (String_To_Upper_Case (buf), "FALSE"))
		value = FALSE;

	return value;
}

#endif

//**************************************************************************
//	Function: 	set_dhcp_profile_bool
//	Description:	
//	Parameter:	
//	Return value:	
//
BOOL set_dhcp_profile_bool (char * key_name, BOOL value)
{
	int result = FALSE;
	switch (value)
	{
	case TRUE:
#ifdef	_USE_PRIVATE_CONF
		result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, key_name, "TRUE", DHCP_CONF);
#else
		result = SET_PROFILE_STRING (DHCP_SECTION, key_name, "TRUE");
#endif
		break;
	case FALSE:
#ifdef	_USE_PRIVATE_CONF
		result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, key_name, "FALSE", DHCP_CONF);
#else
		result = SET_PROFILE_STRING (DHCP_SECTION, key_name, "FALSE");
#endif
		break;
	default:
#ifdef	_USE_PRIVATE_CONF
		result = SET_PRIVATE_PROFILE_STRING (DHCP_SECTION, key_name, "FALSE", DHCP_CONF);
#else
		result = SET_PROFILE_STRING (DHCP_SECTION, key_name, "FALSE");
#endif
		break;
	}
	return result;
}

//**************************************************************************
//	Function: 	Reload_DHCP_Config
//	Description:	
//	Parameter:	
//	Return value:	
//
int Reload_DHCP_Config (void)
{
	struct dhcp_conf conf;
	memset (&conf, '\0', sizeof (conf));
	
	read_dhcp_conf (&conf);

#if 0
printf ("Operate %d\n", conf.nOperate);
printf ("Start Address %s\n", conf.sStartAddress);
printf ("End Address %s\n", conf.sEndAddress);
printf ("Max lease time %d\n", conf.nMax_lease_time_day);
printf ("Max lease time %d\n", conf.nMax_lease_time_hour);
printf ("Local IP = %s\n", conf.sLocalIP);
printf ("Subnet = %s\n", conf.sSubnet);
printf ("Netmask = %s\n", conf.sNetmask);
printf ("Router = %s\n", conf.sRouter);
printf ("Howmany = %d\n", conf.nHowmany);
#endif
	create_dhcpd_conf (conf);
	
	if (conf.nOperate)
	{
		Enable_DHCP (1);
		return 1;
	}
	else
		Enable_DHCP (0);
	return 0;
}
//**************************************************************************
//**************************************************************************
//**************************************************************************
//**************************************************************************

#ifdef	_WRITE_TO_LOG
void write_dhcp_to_log (char * message)
{

	int fd;
	char info_str[256];
	char *ptr[10];

	bzero (info_str, sizeof (info_str));
	strcpy (info_str, message);
	ptr[0] = info_str;
	fd = Open_Event_Log (1);
	Report_Event (fd, EVENTLOG_INFORMATION_TYPE, 1, ptr);
	Close_Event_Log (fd);

}
#endif
//**************************************************************************
//	Function: 	create_dhcp_conf
//	Description:	
//
//	Parameter:	
//	Return value:
//
int create_dhcpd_conf (struct dhcp_conf DHCP_conf)
{
	FILE * fp_dhcpconf = NULL;
	char buf[DHCP_BUF_SIZE];
	struct address_range_link * head = NULL, * link = NULL;
	int result = SUCCESS;

	memset (buf, '\0', sizeof (buf));

	fp_dhcpconf = fopen (DHCPD_CONF, "w");
#ifdef	_WRITE_TO_LOG
	
	{
		char info_str[256];

		bzero (info_str, sizeof (info_str));
		sprintf (info_str, "max-lease-time = %d", DHCP_conf.nMax_lease_time_hour);
		write_dhcp_to_log (info_str);
		
		bzero (info_str, sizeof (info_str));
		sprintf (info_str, "subnet = %s", DHCP_conf.sSubnet);
		write_dhcp_to_log (info_str);
		
		bzero (info_str, sizeof (info_str));
		sprintf (info_str, "netmask = %s", DHCP_conf.sNetmask);
		write_dhcp_to_log (info_str);
		
		bzero (info_str, sizeof (info_str));
		sprintf (info_str, "option routers = %s", DHCP_conf.sRouter);
		write_dhcp_to_log (info_str);
	}
#endif

	if (fp_dhcpconf != NULL)
	{
		sprintf (buf, "max-lease-time %d;\n", 
		(DHCP_conf.nMax_lease_time_day * 24 + DHCP_conf.nMax_lease_time_hour)* 60 * 60);
		if (fputs (buf, fp_dhcpconf) == EOF)
		{
			result = WRITE_CONFIG_FILE_ERROR;
			goto Error;
		}
			
		sprintf (buf, "default-lease-time %d;\n", 
		(DHCP_conf.nMax_lease_time_day * 24 + DHCP_conf.nMax_lease_time_hour) * 60 * 60);	
		if (fputs (buf, fp_dhcpconf) == EOF)
		{
			result = WRITE_CONFIG_FILE_ERROR;
			goto Error;
		}
		
		//fputs ("ddns-update-style ad-hoc;\n", fp_dhcpconf);
		fputs ("\n", fp_dhcpconf);
		
		sprintf (buf, "subnet %s netmask %s {\n", DHCP_conf.sSubnet, DHCP_conf.sNetmask);
		if (fputs (buf, fp_dhcpconf) == EOF)
		{
			result = WRITE_CONFIG_FILE_ERROR;
			goto Error;
		}
		

		convert_range (DHCP_conf.sStartAddress, DHCP_conf.sEndAddress, 
			&DHCP_conf.ReserveRange1, &DHCP_conf.ReserveRange2, &head);
		link = head;
		
		while (link)
		{
#ifdef	_DEBUG
	printf ("start write range<br>\n");
#endif			
			sprintf (buf, "\trange %s %s;\n", link->sStartadd, link->sEndadd);
			if (fputs (buf, fp_dhcpconf) == EOF)
			{
				result = WRITE_CONFIG_FILE_ERROR;
				goto Error;
			}
			link = link->next;
		}
		if (strcmp (DHCP_conf.sRouter, "0.0.0.0") && strcmp (DHCP_conf.sRouter, ""))
		{
			sprintf (buf, "\toption routers %s;\n", DHCP_conf.sRouter);
			if (fputs (buf, fp_dhcpconf) == EOF)
			{
				result = WRITE_CONFIG_FILE_ERROR;
				goto Error;
			}
		}

		{
			FILE * tmpfp = tmpfile ();
			if (tmpfp)
			{
				int set = 0, result = 0;
				char buf1[256], buf2[256];
				bzero (buf1, sizeof (buf1));
				bzero (buf2, sizeof (buf2));

				fprintf (tmpfp, "\toption domain-name-servers ");
				if (NIC_Count_Interface () == 1)
				{
					if (strcmp (DHCP_conf.sDomain_name_server, "0.0.0.0"))
					{
						fprintf (tmpfp, " %s", DHCP_conf.sDomain_name_server);
						set = 1;
					}
					
					if (strcmp (DHCP_conf.sDomain_name_server2, "0.0.0.0"))
					{
						if (set == 1)
							fprintf (tmpfp, ", ");
						fprintf (tmpfp, "%s", DHCP_conf.sDomain_name_server2);
						set = 1;
					}
					fprintf (tmpfp, ";\n");
				}
				else
				{
					if (Is_DNS_Enable ())
					{
						fprintf (tmpfp, "%s", get_local_ip_from_uLinux ());
						set = 1;
					}
					else
					{
						if ((result = nic_dns_setting (buf1, sizeof (buf1), buf2, sizeof (buf2))) != 0)
						{
							fprintf (tmpfp, "%s", buf1);
							if (result == 2)
							{
								fprintf (tmpfp, "%s", ", ");
								fprintf (tmpfp, "%s", buf2);
							}
							set = 1;
						}
					}
					fprintf (tmpfp, ";\n");
				}
				
				if (set == 1)
				{
					rewind (tmpfp);
					if (fgets (buf, sizeof (buf), tmpfp))
					{
						if (fputs (buf, fp_dhcpconf) == EOF)
						{
							result = WRITE_CONFIG_FILE_ERROR;
							fclose (tmpfp);
							goto Error;
						}
					}
				}
					
				fclose (tmpfp);
			}
		}
		
		if (fputs ("}\n", fp_dhcpconf) == EOF)
		{
			result = WRITE_CONFIG_FILE_ERROR;
			goto Error;
		}
		
		if (get_wan_hwaddr ())
		{
			fprintf (fp_dhcpconf, "host myclient {\n\thardware ethernet %s;\n\tdeny booting;\n}\n", get_wan_hwaddr ());
		}
		result = SUCCESS;
	}
	else
		result = OPEN_CONFIG_FILE_ERROR;
Error:
	fclose (fp_dhcpconf);
	release_link (head);
	return result;
}

//**************************************************************************
//	Function: 	read_dhcp_conf
//	Description:	
//
//	Parameter:	
//	Return value:
//
void read_dhcp_conf (struct dhcp_conf * pDHCP_conf)
{
	char net_mask[16], ip[16];
	char buf[32];
	int howmany = 0;
	NIC_NAS_INFO Info_uLinux;
	NIC_INFO Info_NIC;
	//IFInfo Info;
	memset (buf, '\0', sizeof (buf));
	memset (pDHCP_conf, '\0', sizeof (struct dhcp_conf));
	memset (net_mask, '\0', sizeof (net_mask));
	memset (ip, '\0', sizeof (ip));
	memset (&Info_uLinux, '\0', sizeof (Info_uLinux));
	memset (&Info_NIC, '\0', sizeof (Info_NIC));
	
	if (NIC_Get_Info_From_uLinux (&Info_uLinux, NIC_LAN01_DEFAULT) == NIC_FAIL)
	{
		strcpy (Info_uLinux.ifconfig.dev, "");
		strcpy (Info_uLinux.ifconfig.hwaddr, "");
		strcpy (Info_uLinux.ifconfig.ipaddr, "0.0.0.0");
		strcpy (Info_uLinux.ifconfig.netmask, "0.0.0.0");
		strcpy (Info_uLinux.ifconfig.gateway, "0.0.0.0");
		strcpy (Info_uLinux.ifconfig.broadcast, "0.0.0.0");
	}
	
	if (NIC_Get_Info (&Info_NIC, NIC_LAN01) == NIC_FAIL)
		strcpy (Info_NIC.ipaddr, "0.0.0.0");

#ifdef	_DEBUG
	if (Is_DHCP_Enable ())
		printf ("The DHCP server enable is = Enable<br>\n");
	else
		printf ("The DHCP server enable is = Disable<br>\n");
#endif		
	if (Is_DHCP_Enable ())
		pDHCP_conf->nOperate = 1;
	else
		pDHCP_conf->nOperate = 0;
		
	pDHCP_conf->nMax_lease_time_day = Get_Max_Lease_Time_Day();
	pDHCP_conf->nMax_lease_time_hour = Get_Max_Lease_Time_Hour();
#ifdef	_DEBUG
	printf ("get max lease time = %d<br>\n", Get_Max_Lease_Time ());
#endif
	Get_DHCP_Range (START_POSITION, buf, sizeof (buf));
	strcpy (pDHCP_conf->sStartAddress, buf);
#ifdef	_DEBUG
	printf ("get start address = %s<br>\n", buf);
#endif
	Get_DHCP_Range (END_POSITION, buf, sizeof (buf));
	strcpy (pDHCP_conf->sEndAddress, buf);
#ifdef	_DEBUG
	printf ("get end address = %s<br>\n", buf);
#endif			
	Get_Reserve_Range (RESERVE_RANGE_1, START_POSITION, buf, sizeof (buf));
	strcpy (pDHCP_conf->ReserveRange1.sStartadd, buf);
#ifdef	_DEBUG
	printf ("get reserve start address 1 = %s<br>\n", buf);
#endif 
	Get_Reserve_Range (RESERVE_RANGE_1, END_POSITION, buf, sizeof (buf));
	strcpy (pDHCP_conf->ReserveRange1.sEndadd, buf);
#ifdef	_DEBUG
	printf ("get reserve end address 1 = %s<br>\n", buf);
#endif
	Get_Reserve_Range (RESERVE_RANGE_2, START_POSITION, buf, sizeof (buf));
		strcpy (pDHCP_conf->ReserveRange2.sStartadd, buf);
#ifdef	_DEBUG
	printf ("get reserve start address 2 = %s<br>\n", buf);
#endif
	Get_Reserve_Range (RESERVE_RANGE_2, END_POSITION, buf, sizeof (buf));
		strcpy (pDHCP_conf->ReserveRange2.sEndadd, buf);
#ifdef	_DEBUG
	printf ("get reserve end address 2 = %s<br>\n", buf);
#endif
	strcpy (pDHCP_conf->sLocalIP, Info_uLinux.ifconfig.ipaddr);
	strcpy (pDHCP_conf->sNetmask, Info_uLinux.ifconfig.netmask);
		
	//  get subnet
	if (!get_subnet (pDHCP_conf->sLocalIP, pDHCP_conf->sNetmask, buf, sizeof (buf)))
	{
		int idx = 0, jmp = 0;
		char buffer[16];
		memset (buffer, '\0', sizeof (buffer));

		strcpy (pDHCP_conf->sSubnet, buf);
#ifdef	_DEBUG
	printf ("Subnet = %s<br>\n", pDHCP_conf->sSubnet);
#endif
		while (++idx <= 4)
		{
			if (parser_ip (pDHCP_conf->sNetmask, idx) < 255)
			    howmany++;
		}
		idx = 0;
		pDHCP_conf->nHowmany = howmany;
#ifdef	_DEBUG
	printf ("howmany = %d<br>\n", howmany);
#endif
		while (idx < 16)
		{
			if (pDHCP_conf->sLocalIP[idx] == '.')
				jmp++;
			if (jmp >= 4 - howmany)
				break;
			pDHCP_conf->sSubnet_web[idx] = pDHCP_conf->sLocalIP[idx];
			idx++;
		}
	}
	else
	{
		strcpy (pDHCP_conf->sSubnet, "0.0.0.0");
		strcpy (pDHCP_conf->sSubnet_web, "0.0.0.0");
	}
	
	strcpy (pDHCP_conf->sStartAddress, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->sStartAddress,
							howmany));
	strcpy (pDHCP_conf->sEndAddress, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->sEndAddress,
							howmany));
	Set_DHCP_Range (0, pDHCP_conf->sStartAddress);
	Set_DHCP_Range (1, pDHCP_conf->sEndAddress);
	if (strcmp (pDHCP_conf->ReserveRange1.sStartadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange1.sStartadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange1.sStartadd,
							howmany));
		Set_Reserve_Range (0, 0, pDHCP_conf->ReserveRange1.sStartadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange1.sEndadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange1.sEndadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange1.sEndadd,
							howmany));
		Set_Reserve_Range (0, 1, pDHCP_conf->ReserveRange1.sEndadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange2.sStartadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange2.sStartadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange2.sStartadd,
							howmany));
		Set_Reserve_Range (1, 0, pDHCP_conf->ReserveRange2.sStartadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange2.sEndadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange2.sEndadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange2.sEndadd,
							howmany));
		Set_Reserve_Range (1, 1, pDHCP_conf->ReserveRange2.sEndadd);
	}
#ifdef	_DEBUG
	printf ("get local ip address = %s<br>\n", Info_uLinux.ifconfig.ipaddr);
	printf ("get netmask address = %s<br>\n", Info_uLinux.ifconfig.netmask);
	printf ("get subnet address = %s<br>\n", buf);
#endif
	get_dns (buf, sizeof (buf));
	strcpy (pDHCP_conf->sDomain_name_server, buf);
	get_dns2 (buf, sizeof (buf));
	strcpy (pDHCP_conf->sDomain_name_server2, buf);
#ifdef	_DEBUG
	printf ("get dns address = %s<br>\n", buf);
#endif
	if (NIC_Count_Interface()==2 && NIC_Is_Support_Router())
	{
		if (FW_Is_NAT_Enable()==FW_SUCCESS)
			strcpy(pDHCP_conf->sRouter, Info_NIC.ipaddr);
		else
		{
			if (!strcmp(Info_uLinux.ifconfig.gateway, "0.0.0.0"))
				strcpy(pDHCP_conf->sRouter, Info_NIC.ipaddr);
			else
				strcpy(pDHCP_conf->sRouter, Info_uLinux.ifconfig.gateway);
		}
	}
	else
	{
		strcpy(pDHCP_conf->sRouter, Info_NIC.gateway);
	}
#ifdef	_DEBUG
	printf ("get routers address = %s<br>\n", Info.ifconfig.gateway);
#endif
}

//**************************************************************************
//	Function: 	read_dhcp_conf
//	Description:	
//
//	Parameter:	
//	Return value:
//
void read_dhcp_conf_netlinks (struct dhcp_conf * pDHCP_conf, int link_num)
{
	char net_mask[16], ip[16];
	char buf[32];
	int howmany = 0;
	NIC_NAS_INFO Info_uLinux;
	NIC_INFO Info_NIC;
	char link_section[80];
	long lease_time;
	
	sprintf(link_section, "link%d", link_num);
	//IFInfo Info;
	memset (buf, '\0', sizeof (buf));
	memset (pDHCP_conf, '\0', sizeof (struct dhcp_conf));
	memset (net_mask, '\0', sizeof (net_mask));
	memset (ip, '\0', sizeof (ip));
	memset (&Info_uLinux, '\0', sizeof (Info_uLinux));
	memset (&Info_NIC, '\0', sizeof (Info_NIC));
	
	strcpy (Info_uLinux.ifconfig.dev, "");
	strcpy (Info_uLinux.ifconfig.hwaddr, "");
	
	if (Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "IP Address", buf, 80) == ERROR_NOT_FOUND)
		strcpy (Info_uLinux.ifconfig.ipaddr, "0.0.0.0");
	else
		strcpy (Info_uLinux.ifconfig.ipaddr, buf);
	
	if (Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "Subnet Mask", buf, 80) == ERROR_NOT_FOUND)
		strcpy (Info_uLinux.ifconfig.netmask, "0.0.0.0");
	else
		strcpy (Info_uLinux.ifconfig.netmask, buf);
		
	if (Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "Gateway", buf, 80) == ERROR_NOT_FOUND)
		strcpy (Info_uLinux.ifconfig.gateway, "0.0.0.0");
	else
		strcpy (Info_uLinux.ifconfig.gateway, buf);
		
	if (Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "Broadcast", buf, 80) == ERROR_NOT_FOUND)
		strcpy (Info_uLinux.ifconfig.broadcast, "0.0.0.0");
	else
		strcpy (Info_uLinux.ifconfig.broadcast, buf);
		
	
	//if (NIC_Get_Info (&Info_NIC, NIC_LAN01) == NIC_FAIL)
	//	strcpy (Info_NIC.ipaddr, "0.0.0.0");

#ifdef	_DEBUG
	if (Is_DHCP_Enable ())
		printf ("The DHCP server enable is = Enable<br>\n");
	else
		printf ("The DHCP server enable is = Disable<br>\n");
#endif		
	Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "DHCP Server Enable", buf, 80);
	
	if(!strcmp(buf, "TRUE"))
		pDHCP_conf->nOperate = 1;
	else
		pDHCP_conf->nOperate = 0;
	
	Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "DHCP Max Lease Time", buf, 80);
	lease_time = strtol(buf, NULL, 10);
	
	pDHCP_conf->nMax_lease_time_day = (int)(lease_time/86400);
	pDHCP_conf->nMax_lease_time_hour = (int)((lease_time%86400)/360);
	//pDHCP_conf->nMax_lease_time_day = Get_Max_Lease_Time_Day();
	//pDHCP_conf->nMax_lease_time_hour = Get_Max_Lease_Time_Hour();
#ifdef	_DEBUG
	printf ("get max lease time = %d<br>\n", Get_Max_Lease_Time ());
#endif
	Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "DHCP Start Range", buf, 80);
	strcpy (pDHCP_conf->sStartAddress, buf);
#ifdef	_DEBUG
	printf ("get start address = %s<br>\n", buf);
#endif
	Conf_Get_Field(NETLINKS_CONF_FILE,link_section, "DHCP End Range", buf, 80);
	strcpy (pDHCP_conf->sEndAddress, buf);
#ifdef	_DEBUG
	printf ("get end address = %s<br>\n", buf);
#endif			
	//Get_Reserve_Range (RESERVE_RANGE_1, START_POSITION, buf, sizeof (buf));
	//strcpy (pDHCP_conf->ReserveRange1.sStartadd, buf);
#ifdef	_DEBUG
	printf ("get reserve start address 1 = %s<br>\n", buf);
#endif 
	//Get_Reserve_Range (RESERVE_RANGE_1, END_POSITION, buf, sizeof (buf));
	//strcpy (pDHCP_conf->ReserveRange1.sEndadd, buf);
#ifdef	_DEBUG
	printf ("get reserve end address 1 = %s<br>\n", buf);
#endif
	Get_Reserve_Range (RESERVE_RANGE_2, START_POSITION, buf, sizeof (buf));
		strcpy (pDHCP_conf->ReserveRange2.sStartadd, buf);
#ifdef	_DEBUG
	printf ("get reserve start address 2 = %s<br>\n", buf);
#endif
	Get_Reserve_Range (RESERVE_RANGE_2, END_POSITION, buf, sizeof (buf));
		strcpy (pDHCP_conf->ReserveRange2.sEndadd, buf);
#ifdef	_DEBUG
	printf ("get reserve end address 2 = %s<br>\n", buf);
#endif
	strcpy (pDHCP_conf->sLocalIP, Info_uLinux.ifconfig.ipaddr);
	strcpy (pDHCP_conf->sNetmask, Info_uLinux.ifconfig.netmask);
		
	//  get subnet
	if (!get_subnet (pDHCP_conf->sLocalIP, pDHCP_conf->sNetmask, buf, sizeof (buf)))
	{
		int idx = 0, jmp = 0;
		char buffer[16];
		memset (buffer, '\0', sizeof (buffer));

		strcpy (pDHCP_conf->sSubnet, buf);
#ifdef	_DEBUG
	printf ("Subnet = %s<br>\n", pDHCP_conf->sSubnet);
#endif
		while (++idx <= 4)
		{
			if (parser_ip (pDHCP_conf->sNetmask, idx) < 255)
			    howmany++;
		}
		idx = 0;
		pDHCP_conf->nHowmany = howmany;
#ifdef	_DEBUG
	printf ("howmany = %d<br>\n", howmany);
#endif
		while (idx < 16)
		{
			if (pDHCP_conf->sLocalIP[idx] == '.')
				jmp++;
			if (jmp >= 4 - howmany)
				break;
			pDHCP_conf->sSubnet_web[idx] = pDHCP_conf->sLocalIP[idx];
			idx++;
		}
	}
	else
	{
		strcpy (pDHCP_conf->sSubnet, "0.0.0.0");
		strcpy (pDHCP_conf->sSubnet_web, "0.0.0.0");
	}
	
	strcpy (pDHCP_conf->sStartAddress, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->sStartAddress,
							howmany));
	strcpy (pDHCP_conf->sEndAddress, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->sEndAddress,
							howmany));
	//Set_DHCP_Range (0, pDHCP_conf->sStartAddress);
	//Conf_Set_Field(NETLINKS_CONF_FILE,link_section, "DHCP Start Range", pDHCP_conf->sStartAddress);
	//Set_DHCP_Range (1, pDHCP_conf->sEndAddress);
	//Conf_Set_Field(NETLINKS_CONF_FILE,link_section, "DHCP End Range", pDHCP_conf->sEndAddress);
			
	if (strcmp (pDHCP_conf->ReserveRange1.sStartadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange1.sStartadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange1.sStartadd,
							howmany));
		Set_Reserve_Range (0, 0, pDHCP_conf->ReserveRange1.sStartadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange1.sEndadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange1.sEndadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange1.sEndadd,
							howmany));
		Set_Reserve_Range (0, 1, pDHCP_conf->ReserveRange1.sEndadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange2.sStartadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange2.sStartadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange2.sStartadd,
							howmany));
		Set_Reserve_Range (1, 0, pDHCP_conf->ReserveRange2.sStartadd);
	}
	if (strcmp (pDHCP_conf->ReserveRange2.sEndadd, "0.0.0.0"))
	{
		strcpy (pDHCP_conf->ReserveRange2.sEndadd, combine_ip (pDHCP_conf->sLocalIP, 
							pDHCP_conf->ReserveRange2.sEndadd,
							howmany));
		Set_Reserve_Range (1, 1, pDHCP_conf->ReserveRange2.sEndadd);
	}
#ifdef	_DEBUG
	printf ("get local ip address = %s<br>\n", Info_uLinux.ifconfig.ipaddr);
	printf ("get netmask address = %s<br>\n", Info_uLinux.ifconfig.netmask);
	printf ("get subnet address = %s<br>\n", buf);
#endif
	Conf_Get_Field(NETLINKS_CONF_FILE, link_section, "DHCP Primary DNS", buf, 80);
	//get_dns (buf, sizeof (buf));
	strcpy (pDHCP_conf->sDomain_name_server, buf);
	//get_dns2 (buf, sizeof (buf));
	Conf_Get_Field(NETLINKS_CONF_FILE, link_section, "DHCP Secondary DNS", buf, 80);
	strcpy (pDHCP_conf->sDomain_name_server2, buf);

#ifdef	_DEBUG
	printf ("get dns address = %s<br>\n", buf);
#endif
	//if (NIC_Count_Interface()==2 && NIC_Is_Support_Router())
	if (NLK_Get_Link_Nic_Cnt(link_num)==2 && NIC_Is_Support_Router())
	{
		if (FW_Is_NAT_Enable()==FW_SUCCESS)
			strcpy(pDHCP_conf->sRouter, Info_NIC.ipaddr);
		else
		{
			if (!strcmp(Info_uLinux.ifconfig.gateway, "0.0.0.0"))
				strcpy(pDHCP_conf->sRouter, Info_NIC.ipaddr);
			else
				strcpy(pDHCP_conf->sRouter, Info_uLinux.ifconfig.gateway);
		}
	}
	else
	{
		strcpy(pDHCP_conf->sRouter, Info_NIC.gateway);
	}
#ifdef	_DEBUG
	printf ("get routers address = %s<br>\n", Info.ifconfig.gateway);
#endif
}



//**************************************************************************
//	Function: 	combine_ip
//	Description:	
//
//	Parameter:	
//	Return value:
//
char * combine_ip (char * head, char * tail, int pos)
{
	static char result[16];
	int jmp = 0, idx = 0, range_idx = 0;
	
	memset (result, '\0', sizeof (result));

	while (range_idx < 16)
	{
		if (head[range_idx] == '.')
			jmp++;
		if (jmp >= 4 - pos)
			break;
		result[range_idx] = head[range_idx];
		range_idx++;
	}

	jmp = 0;
	while (idx < 16)
	{
		if (tail[idx] == '.')
			jmp++;
		if (jmp >= 4 - pos)
			result[range_idx++] = tail[idx];
		idx++;
	}
	return result;
}

//**************************************************************************
//	Function: 	get_subnet
//	Description:	
//
//	Parameter:	
//	Return value:
//
int get_subnet(char * ip, char *netmask, char * result_buf, int size)
{
	int result = 0, ipaddr[4], netmaskaddr[4], after_operate[4], i = 0;
	char buf[16];
	
	memset (buf, '\0', sizeof (buf));
	memset (result_buf, '\0', size);

	while (i < 4)
	{
		ipaddr[i] = parser_ip (ip, i + 1);
		netmaskaddr[i] = parser_ip (netmask, i + 1);
		after_operate[i] = ipaddr[i] & netmaskaddr[i];
		i++;
	}
	
	sprintf (buf, "%d.%d.%d.%d", after_operate[0], after_operate[1], after_operate[2], after_operate[3]);

	if (strlen (buf) <= size)
		strcpy (result_buf, buf);
	else
		result = -1;
		
	return result;
}

//**************************************************************************
//	Function: 	parser_ip
//	Description:	
//
//	Parameter:	
//	Return value:
//
int parser_ip (char * ip, int index)
{
	char * ptr = ip;

	if (!strcmp (ip, ""))
		return 0;

	while (--index)
	{
		ptr = (char *)strchr (ptr, '.');
		if (ptr)
			ptr += 1;
		else
			return 0;
	}
	return atoi (ptr);
}

//**************************************************************************
//	Function: 	get_default_nameserver
//	Description:	
//
//	Parameter:	
//	Return value:
//
int get_default_nameserver (char * buf, int size)
{
	FILE * fp = NULL;
	char * content_buf = NULL;
	char * ptr = NULL;
	int len = 0, result = 0;
	
	memset (buf, '\0', size);
	
	fp = fopen (RESOLV_CONF, "r");
	if (fp != NULL)
	{
		len = fseek (fp, 0, SEEK_END);
		content_buf = (char *)malloc (sizeof (char) * len);
		if (content_buf != NULL)
		{
			fread (content_buf, len, 1, fp);
			ptr = (char *)strstr (content_buf, "nameserver") + 10;
			if (ptr)
			{
				int idx = 0, idx1 = 0;
				while (isspace (*(ptr + idx++)));

				while (!isspace (*(ptr + idx)))
				{
					if (idx1 < sizeof (content_buf))
					{
						content_buf[idx1] = *(ptr + idx);
					}
					else
					{
						result = -1;  //  size of buffer is too small
						break;
					}
					idx++;
				}
				strcpy (buf, content_buf);
				
			}
			else
				result = -2;  //  can't not find namserver string
			free (content_buf);
		}
		else
			result = -3;  //  allocate memory error
		fclose(fp);
	}
	else 
		result = -4;  //  open file error
		
	return result;
}

//**************************************************************************
//	Function: 	convert_range
//	Description:	
//
//	Parameter:	
//	Return value:
//
int convert_range (char * startaddr, char * endaddr, struct range * ex1, struct range * ex2, struct address_range_link **link)
{
	int result = 0;
	struct address_range_link * link_buf = NULL;
	
	*link = NULL;
	
	*link = (struct address_range_link *)calloc (1, sizeof (struct address_range_link));
	if (!(*link))
	{
		result = -1;
		goto Error;
	}
	(*link)->next = NULL;
#ifdef	_DEBUG
	printf ("start address = %s<br>\n", startaddr);
	printf ("end address = %s<br>\n", endaddr);
	printf ("reserve start address 1 = %s<br>\n", ex1->sStartadd);
	printf ("reserve end address 1 = %s<br>\n", ex1->sEndadd);
	printf ("reserve start address 2 = %s<br>\n", ex2->sStartadd);
	printf ("reserve end address 2 = %s<br>\n", ex2->sEndadd);
	printf ("head->start < start<br>\n");
#endif
	strcpy ((*link)->sStartadd, startaddr);
	if (strcmp (ex1->sStartadd, "0.0.0.0") && strcmp (ex1->sStartadd, ""))
	{
#ifdef	_DEBUG
	printf ("head->end < ex1->start<br>\n");
#endif
		strcpy ((*link)->sEndadd, ex1->sStartadd);
		link_buf = (struct address_range_link *)calloc(1, sizeof (struct address_range_link));
		if (!link_buf)
		{
			result = -2;
			goto Error;
		}
		link_buf->next = NULL;
		add_to_link (link, link_buf);
		strcpy (link_buf->sStartadd, ex1->sEndadd);
		if (strcmp (ex2->sStartadd, "0.0.0.0") && strcmp (ex2->sStartadd, ""))
		{
			struct address_range_link * node = NULL;
#ifdef	_DEBUG
	printf ("head->link->start < ex1->end<br>\n");
#endif
			strcpy (link_buf->sEndadd, ex2->sStartadd);
			node = (struct address_range_link *)calloc(1, sizeof (struct address_range_link));
			if (!node)
			{
				result = -3;
				goto Error;
			}
			node->next = NULL;
			add_to_link (link, node);
#ifdef	_DEBUG
	printf ("head->link->end < ex2->start<br>\n");
	printf ("head->link->link->start < ex2->end<br>\n");
	printf ("head->link->link->end < end<br>\n");
#endif			
			strcpy (node->sStartadd, ex2->sEndadd);
			strcpy (node->sEndadd, endaddr);
		}
		else
		{
#ifdef	_DEBUG
	printf ("head->link->end < end<br>\n");
#endif
			strcpy (link_buf->sEndadd, endaddr);
		}
		
	}
	else
	{
		if (strcmp (ex2->sStartadd, "0.0.0.0") && strcmp (ex2->sStartadd, ""))
		{
#ifdef	_DEBUG
	printf ("head->link->end < ex2->start<br>\n");
#endif
			strcpy ((*link)->sEndadd, ex2->sStartadd);
			link_buf = (struct address_range_link *)calloc(1, sizeof (struct address_range_link));
			if (!link_buf)
			{
				result = -2;
				goto Error;
			}
			add_to_link (link, link_buf);
#ifdef	_DEBUG
	printf ("head->link->start < ex2->end<br>\n");
	printf ("head->link->end < end<br>\n");
#endif
			strcpy (link_buf->sStartadd, ex2->sEndadd);
			strcpy (link_buf->sEndadd, endaddr);
		}
		else
		{
			strcpy ((*link)->sEndadd, endaddr);
		}
	}	
Error:
	return result;
}

//**************************************************************************
//	Function: 	add_to_link
//	Description:	
//
//	Parameter:	
//	Return value:
//
void add_to_link (struct address_range_link ** head, struct address_range_link * node)
{
	struct address_range_link * ptr = *head;
	while (ptr->next != NULL)
		ptr = ptr->next;
	ptr->next = node;
}

//**************************************************************************
//	Function: 	release_link
//	Description:	
//
//	Parameter:	
//	Return value:
//
void release_link (struct address_range_link *head)
{
	struct address_range_link * ptr = head;

	while (ptr)
	{
		ptr=ptr->next;
		free (head);
		head = ptr;
	}
}


int get_max_range (char * natmask, char * ip, int index)
{
	int result = 0;
	int netmask_value = parser_ip (natmask, index);
	int ip_value;
	
	if (netmask_value == 0)
		result = 255;
	else
	{
		ip_value = parser_ip (ip, index);
		result = ip_value & netmask_value;
		result &= 0x000000ff;
		netmask_value = ~netmask_value;
		netmask_value &= 0x000000ff;
		result |= netmask_value;
		result &= 0x000000ff;
		
	}
	
	return result;
}

int get_min_range (char * natmask, char * ip, int index)
{
	int result = 0;
	int netmask_value = parser_ip (natmask, index);
	int ip_value;
	
	if (netmask_value == 0)
		result = 0;
	else
	{
		ip_value = parser_ip (ip, index);

		result = netmask_value & ip_value;
	}
	
	return result;
}

int nic_dns_setting (char * buf1, int size1, char * buf2, int size2)
{
	int result = 0;
	if (NIC_Get_Resolv2 (buf1, size1, buf2, size2) == NIC_SUCCESS)
	{
		if (strcmp (buf1, "0.0.0.0"))
			result++;
		if (strcmp (buf2, "0.0.0.0"))
			result++;
	}
	return result;
}

char * get_local_ip_from_uLinux ()
{
	static char buf[32];
	NIC_NAS_INFO info;
	memset (&info, '\0', sizeof (info));
	
	NIC_Get_Info_From_uLinux (&info, NIC_LAN01);
	strcpy (buf, info.ifconfig.ipaddr);
	
	return buf;
}

char * get_wan_hwaddr ()
{
	static char buf[32];
	memset (buf, '\0', sizeof (buf));
	
	if (NIC_Get_HWADDR (buf, sizeof (buf), NIC_WAN) == NIC_FAIL)
		return NULL;
	return buf;
}

char * get_dhcp_device ()
{
	static char buf[32];
	memset (buf, '\0', sizeof (buf));
	
	if (NIC_Get_Dev (buf, sizeof (buf), NIC_LAN01) == NIC_FAIL)
		return NULL;
	return buf;
}

int get_usage ()
{
	NIC_NAS_INFO info;
	memset (&info, '\0', sizeof (info));
	
	NIC_Get_Info_From_uLinux (&info, NIC_LAN01);
	return info.usage;
}
