//**************************************************************************
//
//	Copyright (c) 2000  ICP Electronics Inc.  All Rights Reserved.
//
//	FILE:
//		user_grp.c
//
//	Abstract:
//		uLinux user/group management library functions.
//
//	HISTORY:
//		05/26/02 Andy Wu -- replace strcpy with strncpy
//		10/17/02 John - add Create_NAS_Mutiple_Users() APIs
//		06/17/02 Catherine Shen -- add/remove Netware user passwords in NAS APIs
//		05/28/02 Kent - add Str_Trans_Cmdstr for command line string transfer
//		05/08/02 Bird - add Check special chars on change smbpasswd
//		04/29/02 Bird - Check special chars on smbpasswd
//		01/09/01 Kent - Modify smbpasswd with "no passwd" for security problem
//		01/08/01 Catherine Shen -- modify 'Change_NAS_User_Password()',
//				'Create_System_User_Ex()'
//		12/28/01 Catherine -- add 'Release_List()',
//				'Get_NAS_User_List_Of_Group_Ex()',
//				'Get_NAS_Group_List_Of_User_Ex()',
//		12/14/01 Catherine Shen -- modified 'User_Belongs_To_Group()',
//				'Is_My_User()', 'Get_NAS_Group_Info()',
//				'Change_File_Attribute()', 'Get_NAS_User_Info()',
//				'Get_System_User_ID()', 'Check_Local_User_Password()'
//		12/12/01 Catherine Shen -- modified Check_System_User_Password();
//				include "user_grp.h";
//				add Create_System_User_Ex(), NAS_Get_User_Type(),
//				Get_NAS_User_List_Ex(), Get_NAS_Group_List_Ex(),
//				& NAS_Get_Group_Type();
//				remove 'compare_string()';
//				replace 'compare_string()' by 'compare_case_string()'
//				from the "Util" library;
//		11/27/01 Catherine Shen - change the command buffer size of
//					Create_System_Group() &
//					Remove_System_Group() &
//					Change_System_Group_Name() &
//					Create_System_User
//		10/04/01 John Shiau - modified compare_string()
//		09/24/01 John Shiau - modified Get_Exact_System_User_Name() & Get_Exact_System_Group_Name()
//		09/13/01 John Shiau - modified Is_System_User() & Is_System_Group() & User_Belongs_To_Group()
//		04/24/01 Louis Tsai - add function to get exactly user name & group,CGI function
//				      should call it before updating system user or group!
//		04/20/01 Louis Tsai - fix remove user & group bug ( remove share privilege)
//		04/17/01 Louis Tsai - add Get_Samba_Status()
//              03/30/01 Louis Tsai - replace all find_conf() & edit_conf to util lib function
//		03/21/01 Louis Tsai - modified Sort_Name_List algorithm to qsort()
//		03/20/01 Louis Tsai - modified Get_All_SubDir() & Is_My_SubDIR() &
//		12/18/00 Meiji Chang - re-formated
//		09/11/00 Nick Ho
//			- add setgrent(),setpwent() before call getgrent() & getpwent()
//			- add endgrent(),endpwent() after call getgrent() & getpwent()
//		09/01/00 Nick Ho - Created
//
//**************************************************************************

#include <stdio.h>
#include <string.h>
#include <pwd.h>          // for getpwname()
#include <shadow.h>
#include <grp.h>          // for getgrgid()
#include <sys/types.h>    // for getpwname()
#include <unistd.h>       // for crypt()
#include <stdlib.h>       // for system()
#include <errno.h>        // for errno
#include <ctype.h>        // for topper()       // 2000/12/7
//#include <syslog.h>       // for temp debug
//- for Get_All_SubDIR() start -
#include <crypt.h>
#include <sys/stat.h>
#include <dirent.h>
//- for Get_All_SubDIR() End   -
#include <ftw.h>

#include "NAS.h"
#include "uLinux.h"
#include "Util.h"
#include "user_grp.h"	// Catherine
#include "naspdc.h"
#include "cfg_nic.h"
#include "nas_quota.h"
#include "naslvm.h"	// Catherine
#include "nas_lib_common.h"
#ifdef NAS_PPC
#include "nas_hash.h"
#endif

static uid_t	current_uid = 0;	// Catherine

#define	CRYPT_SALT_NO		"Icp"

#define	SMBPASSWD_PATH		"/usr/local/samba/bin/smbpasswd"
#define SMBSTATUS_PATH		"/usr/local/samba/bin/smbstatus"
#define SMBREPORT_PATH		"/tmp/smb.status"
/*
#define	USERADD_PATH		"/usr/sbin/useradd"
#define	USERDEL_PATH		"/usr/sbin/userdel"
#define	USERMOD_PATH		"/usr/sbin/usermod"
#define	GROUPADD_PATH		"/usr/sbin/groupadd"
#define	GROUPDEL_PATH		"/usr/sbin/groupdel"*/
#define	GROUPMOD_PATH		"/usr/sbin/groupmod"

#define	ADDUSER_PATH		"/bin/adduser"
#define	DELUSER_PATH		"/bin/deluser"
#define	ADDGROUP_PATH		"/bin/addgroup"
#define	DELGROUP_PATH		"/bin/delgroup"
#define	PASSWD_PATH		"/usr/bin/passwd"

#define NASCRYPT		bigcrypt
extern char *bigcrypt(const char *key, const char *salt);

// 
// check special chars: `"\ in str for system command arguments
// if find, add '\' before special char
// note: ` is special char in system command argument, but ' is not
// not same as c language's special char
//
int Str_Trans_CmdStr(const char *in_str, char *out_str)
//int check_special_chars(const char *in_str, char *out_str)
{
	char	*special_chars = "\\`\"$";
	int i, j, len = strlen(in_str);
	
	for(j = 0, i = 0; i < len; i++) {
		if(strchr(special_chars, in_str[i]))
			out_str[j++] = '\\';
		out_str[j++] = in_str[i];
	}
	out_str[j]=0x0;
	return j;
}

int str2up(char *str)
{
	char	*ptr;

        ptr=str;
        while (*ptr!=0x0)
        {
                if (*ptr>='a' && *ptr<='z')
                        *ptr='A'+*ptr-'a';
                ptr++;
        }
	return 1;
}

//------------------------------------------------------------------
//  Create or modify Netware user password entry.
//  Return:   SUCCESS
//            ERROR_EXEC_FAIL      --> call system function fail
//------------------------------------------------------------------
int change_netware_user_password(char* username, char* passwd)
{
	/*char cmd[BUF_SIZE], check_name[2*USER_NAME_LENGTH];

	Str_Trans_CmdStr(username, check_name);
	sprintf(cmd, "/sbin/nwutils -a \"%s\" \"%s\" > /dev/null",
		check_name, passwd);
	if (!system(cmd))
		return SUCCESS;

	return ERROR_EXEC_FAIL;*/
	return SUCCESS;
}

//------------------------------------------------------------------
//  Remove a Netware user password entry.
//  Return:   SUCCESS
//            ERROR_EXEC_FAIL      --> call system function fail
//------------------------------------------------------------------
int remove_netware_user(char* username)
{
	/*char cmd[BUF_SIZE], check_name[2*USER_NAME_LENGTH];

	Str_Trans_CmdStr(username, check_name);
	sprintf(cmd, "/sbin/nwutils -d \"%s\" > /dev/null", check_name);
	if (!system(cmd))
		return SUCCESS;

	return ERROR_FAIL;*/
	return SUCCESS;
}

// algorithm:
//	console:
//		key=sum(mac_address)
//		pin=version
//		str=crypt(server, key)
//		pwd=crypt(str, pin)
//	reset switch:
//		key=sum(server)
//		pin=version
//		str=crypt(mac_address, key)
//		pwd=crypt(str, pin)
//
int NAS_Get_Serurity_Pwd(char *pwd, int len, char *input_hwaddr, char *input_server, char *input_version, int type)
{
        char    server[64], hwaddr[64], version[64];
#ifdef NAS_PPC
	char	*ptr, buf[1024], buf2[33];
#else
        char    *ptr, *str;
        int     key=0, pin=0, count=0;
	char	salt_key[3], salt_pin[3], *s_key, *s_pin;
#endif

        if (input_server==NULL)
                Get_Server_Name(server, 64);
        else
                strcpy(server, input_server);
	str2up(server);

        if (input_version==NULL)
                Get_System_Version(version, 64);
        else
                strcpy(version, input_version);

        if (input_hwaddr==NULL)
	{
		if (NIC_Count_Interface()==2)
	                NIC_Get_HWADDR(hwaddr, 64, NIC_WAN);
		else
			NIC_Get_HWADDR(hwaddr, 64, NIC_LAN01);
	}
        else
                strcpy(hwaddr, input_hwaddr);
	str2up(hwaddr);

#ifdef NAS_PPC
	sprintf(buf, "QNAP_NAS_%s_%s_%s", server, hwaddr, version);
	NAS_Hash_Hex(buf, buf2, sizeof(buf2), NAS_HASH_MD5);
	strcpy(pwd, buf2);
#else
	if (type==GET_CONSOLE_PWD)
	{
	        ptr=hwaddr;
        	while (*ptr!=0x0)
	        {
			if (*ptr!=':')
				key=key+(int)*ptr;
	                ptr++;
        	}
	}
	else
	{
		ptr=server;
		while (*ptr!=0x0)
		{
			key=key+(int)*ptr;
			ptr++;
		}
	}
        ptr=version;
        while (*ptr!=0x0)
        {
                if (*ptr>='0' && *ptr<='9')
                {
                        count++;
                }
                ptr++;
        }
        if (count<4)
        {
                ptr=version;
                while (*ptr!=0x0)
                {
                        if (*ptr>='0' && *ptr<='9')
                        {
                                count--;
                                pin=pin+(0xffff&((*ptr-'0')<<(count*4)));
                        }
                        ptr++;
                }
        }
        else
                pin=0xFFFF;

	salt_key[0]=0xff00&key;
	salt_key[1]=0xff&key;
	salt_key[2]=0x0;
	s_key=salt_key;
	if (*s_key==0x0)
		s_key++;

	salt_pin[0]=0xff00&pin;
	salt_pin[1]=0xff&key;
	salt_pin[2]=0x0;
	s_pin=salt_pin;
	if (*s_pin==0x0)
		s_pin++;

        if (type==GET_CONSOLE_PWD)
        {
                str=crypt(server, s_key);
                ptr=crypt(str, s_pin);
        }
        else
        if (type==GET_RESET_PWD)
        {
                str=crypt(hwaddr, s_key);
                ptr=crypt(str, s_pin);
        }
        else
                ptr=0x0;
        strcpy(pwd, ptr);
#endif
	ptr=pwd;
	while (*ptr!=0x0)
	{
		if (*ptr=='/' || *ptr=='&' || *ptr=='=' || *ptr=='?' || *ptr==' ')
			*ptr='k';
		else
		if ((*ptr>='0' && *ptr<='9') || (*ptr>='A' && *ptr<='Z') || (*ptr>='a' && *ptr<='z'))
		{
		}
		else
			*ptr='i';
		ptr++;
	}
        return 1;
}

static BOOL is_pdc_name(char *name)
{
	char *tmp_name;
	int ret;

	tmp_name = strdup(name);
	if(tmp_name !=NULL) {
		int prefix_len;
		prefix_len = strlen(PDC_PREFIX_NAME);
		
		if(strlen(name)>=prefix_len) {
			char ch;
			ch = *(tmp_name+prefix_len);
			*(tmp_name+prefix_len) = '\0';
			if(strcmp(PDC_PREFIX_NAME,tmp_name) == 0)
				ret = TRUE;
			else
				ret = FALSE;
		}
		else
			ret = FALSE;
	}
	else
		return FALSE;
	
	free(tmp_name);
	return ret;
}


// Strip off the white space characters appened on the end of a string.
// Return the original string ended with non-white-space characters
#if 0
char *stripe_white_space_tail(char *str)
{
	if (strlen(str) > 0) {
		char *p = str + strlen(str) - 1;
		while (*p == ' ' || *p == '\t') p--;
		*(p+1) = '\0';
	}
	return str;
}
#endif

/* Catherine : this function is implemented in the Util library
int compare_string(char* str1,char* str2)
{
	//john modified 2001-10-04:add to avoid core dumped
	if( str1 == NULL || str2 == NULL)
		return -1;
	//modified end
	if( CASE_FLAG )
		return(strcmp(str1,str2));
	else
		return(strcasecmp(str1,str2));
}
*/

// Catherine -- change owner to administrator (uid == 0)
int chgown(const char* file, const struct stat *sb, int flag)
{
#ifdef CAT_DEBUG
	printf("chgown(file=%s, uid=%d, gid=%d, flag=%X) starts!\n",
		file, (int)sb->st_uid, (int)sb->st_gid, flag);
#endif
	// if sb->st_uid == current_uid, change owner uid to 0 (administrator)
	if (current_uid>0 && sb->st_uid == current_uid) {
#ifdef CAT_DEBUG
		printf("%s owner changed to administrator\n", file);
#endif
		chown(file, 0, sb->st_gid);
	}
	return 0;
}

// Catherine -- set the file owner to administrator when deleting the user
int change_user_owned_files_to_admin(uid_t uid)
{
	int cnt, ret=0, tmp, i;
	LVM_VOLUME_INFO *vol_list = NULL;

#ifdef CAT_DEBUG
	printf("change_user_owned_files_to_admin(%d) starts!\n", uid);
#endif
	if (uid==0) return SUCCESS;

	current_uid = uid;
#ifdef CAT_DEBUG
	printf("\tcurrent_uid=%d!\n", (int)current_uid);
#endif
	cnt = Get_All_LVM_Configured_Vol_For_Share(&vol_list);
	for (i=0; i<cnt; i++) {
		char volmt[BUF_SIZE];

		Get_MP_String(vol_list[i].vol_data.drive_no_list[0], DATA_PART, volmt);
		// go through each path
		tmp = ftw(volmt, chgown, 300);
#ifdef CAT_DEBUG
		printf("\tmount point %d (drvno=%d): %s -- %d\n",
			i+1, vol_list[i].vol_data.drive_no_list[0], volmt, tmp);
#endif
		ret += tmp;
	}

	if (cnt>0 && vol_list) Release_LVM_Info_List(vol_list, cnt);
	return ret;
}

//--------------------------------------------------------------
// get the count of system users (including guest, administrator.....)
//--------------------------------------------------------------
int get_total_user_count()
{
	/*int count=0;
	struct passwd *pwd;

	setpwent();
	while ((pwd = getpwent()) != NULL) {
		if (!Is_Hidden_User(pwd->pw_name))
			count++;
	}
	endpwent();*/

	// 2005.02.04, Johnson Cheng rewrite it
	// getpwent() will get all users including AD users, we just count local users.
	int count=0;
	struct passwd *user;
	FILE *stream;

	stream = fopen("/etc/passwd", "r");
	while ((user = fgetpwent(stream)) != NULL)
	{
		if (!Is_Hidden_User(user->pw_name))
			count++;
	}	
	fclose(stream);

	return count;
}

//--------------------------------------------------------------
// comparisiom function for Sort_Name_List ( qsort() required )
//--------------------------------------------------------------
int m_strcmp(const void *a,const void *b)
{
	return (strcmp((char*) a,(char*) b));
}


//-------------------------------------------------------------
// Sort Area by alphabet
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------

int Sort_Name_List(char **name_list, int name_length, int list_cnt)
{

	char (*ptr)[name_length];

	ptr = (char (*)[name_length])name_list;
	qsort((void *)ptr,list_cnt,name_length,m_strcmp);

	return SUCCESS;
}



//============================================================
//                     EXPORT  FUNCTIONS
//============================================================
int Get_Exact_System_User_Name(char *user_name)
{
	struct passwd *pwd;
	int flag=0;
	setpwent();
	while ((pwd = getpwent()) != NULL) {
		if (compare_case_string(user_name, pwd->pw_name)==0) {
			if (strcmp(user_name, pwd->pw_name))
				strcpy(user_name,pwd->pw_name);
			flag = 1;
			break;
		}
	}
	endpwent();		// close /etc/passwd file

	//John modified 2001-09-24:fix wrong return value
	//if ( flag == 1) return ERROR_NOT_EXISTED;
	if ( flag == 0) return ERROR_NOT_EXISTED;
	return SUCCESS;

}

int Get_Exact_NAS_User_Name(char *user_name)
{
	return 	Get_Exact_System_User_Name(user_name);
}

int Get_Exact_System_Group_Name (char *group_name)
{
	struct group *p;
	int flag=0;
	setgrent();
	while ((p = getgrent()) != NULL) {
		if (compare_case_string(group_name, p->gr_name)==0) {
			strcpy(group_name, p->gr_name);
			flag = 1;
			break;
		}
	}
	endgrent();		// close /etc/group file
	//John modified 2001-09-24:fix wrong return value
	//if ( flag == 1) return ERROR_NOT_EXISTED;
	if ( flag == 0) return ERROR_NOT_EXISTED;
	return SUCCESS;

}

int Get_Exact_NAS_Group_Name (char *group_name)
{
	return Get_Exact_System_Group_Name (group_name) ;
}

//***************************************************************************//

BOOL Is_Reserved_User(char *user_name)
{
	if (compare_case_string(user_name, USER_ADMINISTRATOR) == 0) return TRUE;
	if (compare_case_string(user_name, USER_GUEST) == 0) return TRUE;
	if (compare_case_string(user_name, USER_ROOT) == 0) return TRUE;
	if (compare_case_string(user_name, USER_ANONYMOUS) == 0) return TRUE;
	

	return FALSE;
}

BOOL Is_Hidden_User(char *user_name)
{
//	if (compare_case_string(user_name, USER_ADMINISTRATOR) == 0) return TRUE;
	if (compare_case_string(user_name, USER_GUEST) == 0) return TRUE;
	if (compare_case_string(user_name, USER_ROOT) == 0) return TRUE;
	if (compare_case_string(user_name, USER_ANONYMOUS) == 0) return TRUE;

	return FALSE;
}

//*------------------------------------------------------------------
//*- Check the specified user exists in /etc/passwd file or not
//*- Return: TRUE --> User already exist
//*-         FALSE --> User not exist
//*------------------------------------------------------------------
BOOL Is_System_User(char *user_name)
{
	struct passwd *pwd;
	BOOL bRet = FALSE;
 	//john modified 2001-09-13:add condition.
	if(user_name != NULL){
	//modified end. 
		setpwent();
		while ((pwd = getpwent()) != NULL) {
			if (compare_case_string(user_name, pwd->pw_name)==0) {
				bRet = TRUE;
				break;
			}
		}
		endpwent();		// close /etc/passwd file
	}
	return bRet;
}


//*------------------------------------------------------------------
//*- Check the specified group exists in /etc/group file or not
//*- Return: 1 --> group already exist
//*-         0 --> group not exist
//*------------------------------------------------------------------
BOOL Is_System_Group(char *group_name)
{
	struct group *p;
	BOOL bRet = FALSE;

 	//john modified 2001-09-13:add condition.
	if(group_name != NULL){
	//modified end.
		setgrent();
		while ((p = getgrent()) != NULL) {
			if (compare_case_string(group_name, p->gr_name)==0) {
				bRet = TRUE;
				break;
			}
		}
		endgrent();		// close /etc/group file
	}
	return bRet;
}

//-------------------------------------------------------------
// Check the specified user is belong to the specified group
// if belong to return 1 otherwise return 0
//-------------------------------------------------------------
BOOL User_Belongs_To_Group(char *user_name, char *group_name)
{
	struct group *p;
	int i = 0;
	char grpname[GROUP_NAME_LENGTH] = {0};
	//john modified 2001-09-13:add condition
	if(user_name != NULL && group_name != NULL){
	//modified end.
		
		// 2004.12.09, Johnson Cheng add for multi-domain AD
		strcpy(grpname, Get_Group_Name_By_User_Name(user_name));
		if(compare_case_string(grpname, group_name) == 0)
			return TRUE;
		// 2004.12.09, Johnson Cheng Add End
		
		// Catherine 12/14/01
		strncpy(grpname, group_name, GROUP_NAME_LENGTH-1);
		grpname[GROUP_NAME_LENGTH-1]=0x0;	//Andy 05/26/03
		if (SUCCESS!=Get_Exact_System_Group_Name(grpname)) 
			return FALSE;
		//if ((p = getgrnam(group_name)) == NULL)
		if ((p = getgrnam(grpname)) == NULL)
			return FALSE;

		while (*(p->gr_mem + i) != NULL) {
			if (compare_case_string(user_name, *(p->gr_mem + i)) == 0)
				return TRUE;
			i++;
		}
	}
	// Kevin Liao 2001-07-10: Fix creating user error
	// free(p);
	// Kevin Liao 2001-07-10: End of fix
	// meiji ?? memory leak ??

	return FALSE;
}


//------------------------------------------------------------
//  Add one specified local user information
//  You should assign the user name & his plaintext password
//   & his primary group
//  & it will call crypt() to encrypt it
//  Return:   ERROR_ALREAY_EXIST --> User alreay exist
//            UG_ADD_LOCAL_USER --> Call useradd CMD fail
//            SUCCESS             --> Successful
//------------------------------------------------------------
//int Add_Local_User(char *user_name, char *pass_wd, char *pri_grp) {
int Create_System_User(char *user_name, char *passwd, char *group_name, char *home_dir)
{
	char *ep;	// encrypted passwd buffer
	char buf[2*BUF_SIZE];
	char check_user_name[2*USER_NAME_LENGTH];

	// Check the user is exist or not ?
	if (Is_System_User(user_name)) return ERROR_ALREADY_EXISTED;

	if (EMPTY_STRING(group_name)) strcpy(group_name, GROUP_EVERYONE);

	if (!Is_System_Group(group_name))	// create non-existed group automatically.
		Create_System_Group(group_name);

	if (get_total_user_count() >= MAX_USER_NUMBER) // too many system users
		return ERROR_USERNUM_EXCEED_ACCOUNT;

	// to get encrypted passwd
	if (EMPTY_STRING(passwd))
		ep = "";
	else
		ep = passwd;
		//ep = NASCRYPT(passwd, CRYPT_SALT_NO);

	// add local user  -M: doesn't create home DIR.
	Str_Trans_CmdStr(user_name, check_user_name);
	if (EMPTY_STRING(home_dir)){		// home set to / , no shell
		sprintf(buf,"%s -G \"%s\" -h / -H -p \"%s\" \"%s\" > /dev/null",
			ADDUSER_PATH, group_name, ep, check_user_name);
	}
	else
		sprintf(buf,"%s -G \"%s\" -h %s -p \"%s\" \"%s\" > /dev/null",
			ADDUSER_PATH, group_name, home_dir, ep, check_user_name);


	if (system(buf) !=0) {
		return ERROR_EXEC_FAIL;
	}

	// Meiji: 2001-03-14 explicitly add this user to the group list in /etc/group
	Add_NAS_User_To_Group(user_name, group_name);

	return SUCCESS;
}



//-------------------------------------------------------------
// Create a new smb user
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------
int Create_Samba_User(char *user_name, char *pass_wd)
{
	char buf[BUF_SIZE], checked_passwd[BUF_SIZE];
	char check_user_name[2*USER_NAME_LENGTH];

	Str_Trans_CmdStr(user_name, check_user_name);
	if (EMPTY_STRING(pass_wd))
//		sprintf(buf, "%s -a -n \"%s\" > /dev/null", SMBPASSWD_PATH, user_name);
		/* if no arg2, smbpasswd will set "no password" to this user */
//		sprintf(buf, "%s -a \"%s\" \"\" > /dev/null", SMBPASSWD_PATH, user_name);
		sprintf(buf, "%s -a -n \"%s\" > /dev/null", SMBPASSWD_PATH, check_user_name);
	else {
		memset(checked_passwd, 0, sizeof(checked_passwd));
		Str_Trans_CmdStr(pass_wd, checked_passwd);
//		sprintf(buf, "%s -a \"%s\" \"%s\" > /dev/null", SMBPASSWD_PATH, user_name, checked_passwd);
		sprintf(buf, "%s -a \"%s\" \"%s\" > /dev/null", SMBPASSWD_PATH, check_user_name, checked_passwd);
	}

#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Create_Samba_User: buf size too small.\n");
#endif

	if (system(buf) !=0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Create_Samba_User: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}


//------------------------------------------------------------------
//  Create a new user: It will create a local & smb user by call
//  Add_Local_User() & Create_Smb_User()
//   primary group.
//  If hasn't pass_wd, it should be assigned as NULL or NULL char
//  Add_Local_User() Return code:
//  Return:   ERROR_ALREAY_EXIST --> User alreay exist
//            UG_ADD_LOCAL_USER --> Call useradd CMD fail
//            SUCCESS             --> Successful
//
//  Create_Smb_User() Return code:
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//------------------------------------------------------------------
//int Create_One_User(char *user_name, char *pass_wd, char *pri_grp) {
int Create_NAS_User(char *user_name, char *pass_wd)
{
	int ret = SUCCESS;
	char group_name[GROUP_NAME_LENGTH];
	char username[USER_NAME_LENGTH];

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03

	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;	
	if (Is_Reserved_User(username))	// check whether the name is reserved for the system
		return ERROR_RESERVED_USER;
	if (Is_System_User(username))		// check whether the name exists or not 
		return ERROR_ALREADY_EXISTED;

	strcpy(group_name, GROUP_EVERYONE);
	ret = Create_System_User(username, pass_wd, group_name, NULL);
	if (ret == SUCCESS) {
		ret = Create_Samba_User(username, pass_wd);
		if (ret==SUCCESS)
			ret = change_netware_user_password(username, pass_wd);
	}

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	//Update_Flash_Data(SAMBA_PASS_PATH);
	//Update_Flash_Data("/etc/config/nwpwd");	// Catherine 2003/04/30 for Netware user list

	return ret;
}


//--------------------------------------------------------
//  Del one specified local user information
//  If the user currently login in Linux then userdel
//  cannot delete it, How about SAMBA, if login on samba
//  Call userdel CMD which will delete the user's groups
//  in the /etc/group file
//  this time it can still delete him.
//  Return:   ERROR_NOT_FOUND   --> User doesn't exist
//            UG_DEL_LOCAL_USER --> call userdel CMD fail
//                 may be the is logging to the Linux
//            ERROR_CANNOT_DEL: the user cannot been deleted
//            SUCCESS          --> Successful
//--------------------------------------------------------
//int Del_Local_User(char *user_name) {
int Remove_System_User(char *user_name)
{
	char buf[BUF_SIZE];
	char username[USER_NAME_LENGTH];
	char check_user_name[2*USER_NAME_LENGTH];
	struct passwd *p;
	uid_t uid=0;

	if ((p = getpwnam(user_name)) == NULL)
		return ERROR_NOT_EXISTED;
	uid = p->pw_uid;

	// Check the User is admin or guest, they cannot been deleted
	if (Is_Reserved_User(user_name)) return ERROR_RESERVED_USER;

	//---- Check the user is exist or not ? ----
	// Catherine 2001/12/14 -- replace by 'Get_Exact_System_User_Name()'
	//if (!Is_System_User(user_name)) return ERROR_NOT_EXISTED;
	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	if (SUCCESS!=Get_Exact_System_User_Name(username)) return ERROR_NOT_EXISTED;

	Str_Trans_CmdStr(username, check_user_name);

	sprintf(buf, "%s \"%s\" > /dev/null", DELUSER_PATH, check_user_name);
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Remove_System_User: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Remove_System_User: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	change_user_owned_files_to_admin(uid);

	return SUCCESS;
}

//-------------------------------------------------------------
// Delete a smb user
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------
//int Del_Smb_User(char *smb_name) {
int Remove_Samba_User(char *user_name)
{
	char buf[BUF_SIZE];
	char username[USER_NAME_LENGTH];
	char check_user_name[2*USER_NAME_LENGTH];

	// Catherine 2001/12/14
	strncpy(username, user_name,USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03

	if (SUCCESS!=Get_Exact_System_User_Name(username)) return ERROR_NOT_EXISTED;
	
	// Meiji: need Samba 2.0.7 for -x support...
	Str_Trans_CmdStr(username, check_user_name);
//	sprintf(buf,"%s -x \"%s\" > /dev/null", SMBPASSWD_PATH, username);
	sprintf(buf,"%s -x \"%s\" > /dev/null", SMBPASSWD_PATH, check_user_name);
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Remove_Samba_User: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Remove_Samba_User: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}


//------------------------------------------------------------------
//  Delete a user: It will delete a local & smb user by call
//  Del_Local_User() & Del_Smb_User()
//  Del_Local_User() Return code:
//  Return:   ERROR_NOT_FOUND   --> User doesn't exist
//            UG_DEL_LOCAL_USER --> call userdel CMD fail
//                 may be the is logging to the Linux
//            ERROR_CANNOT_DEL: the user cannot been deleted
//            SUCCESS          --> Successful
//
//  Del_Smb_User() Return code:
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
// 2000/9/19: You cannot delete the User "admin", "guest" & "root"
// 2001/12/03: samba 2.2.2 will check /etc/passwd exist or not first.
//		if not exist, it will not remove samba user
//------------------------------------------------------------------
//int Del_One_User(char *user_name) {
int Remove_NAS_User(char *user_name)
{
	int ret;
	NAS_USER_GROUP_TYPE type;

printf("11111\n");
	if (Is_Reserved_User(user_name)) return ERROR_RESERVED_USER;
	
	// Catherine 2001/12/14 : check if is a PDC user
printf("22222\n");
	type = NAS_Get_User_Type(user_name);
	
	if (type==NAS_USER_GROUP_NONE)
		return ERROR_NOT_EXISTED;
	else if (type==NAS_USER_GROUP_PDC)
		ret = SUCCESS;
	else  { // remove samba user passwd first 
		ret = Remove_Samba_User(user_name);
		ret = remove_netware_user(user_name);
	}
printf("3333\n");
		
	//if (ret == SUCCESS ) {
//		ret = Remove_User_Share_Privilege(user_name);
		Remove_User_From_Section_Ex(user_name);
		//if (ret == SUCCESS) 
		ret = Remove_System_User(user_name);
	//}

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	//Update_Flash_Data(SAMBA_PASS_PATH);
	//Update_Flash_Data("/etc/config/nwpwd");

	return ret;
}


//---------------------------------------------------------
//  Modify one specified local user passwd
//  If the user is login to the Linux, it can still change
//  his passwd
//  Return:   ERROR_NOT_FOUND      --> User doesn't exist
//            UG_MOD_LOCAL_USER --> call usermod CMD fail
//            SUCCESS             --> Successful
//---------------------------------------------------------
//int Mod_Local_User_Passwd(char *user_name, char *pass_wd) {
int Change_System_User_Password(char *user_name, char *pass_wd)
{
	char *ep;
	char buf[BUF_SIZE];
	char username[USER_NAME_LENGTH];
	char check_user_name[2*USER_NAME_LENGTH];

	// Check the user is exist or not
	// Catherine 2001/12/14 -- replace by 'Get_Exact_System_User_Name()'
	//if (!Is_System_User(user_name)) return ERROR_NOT_EXISTED;
	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if (SUCCESS!=Get_Exact_System_User_Name(username)) return ERROR_NOT_EXISTED;
	
	// to get encrypted passwd
	Str_Trans_CmdStr(username, check_user_name);
	if (EMPTY_STRING(pass_wd)) {	// Check this...
//		sprintf(buf, "%s -p \"\" \"%s\" > /dev/null", USERMOD_PATH, username);
		sprintf(buf, "%s -p \"\" \"%s\" > /dev/null", PASSWD_PATH, check_user_name);
	}
	else {
		//ep = NASCRYPT(pass_wd, CRYPT_SALT_NO);
		ep = pass_wd;
//		sprintf(buf, "%s  -p \"%s\" \"%s\" > /dev/null", USERMOD_PATH, ep, username);
		sprintf(buf, "%s  -p \"%s\" \"%s\" > /dev/null", PASSWD_PATH, ep, check_user_name);
	}

#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Change_System_User_Password: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Change_System_User_Password: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;

}


//-------------------------------------------------------------
// Modify smb password
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------
//int Mod_Smb_Passwd(char *user_name, char *pass_wd) {
int Change_Samba_User_Passwd(char *user_name, char *pass_wd)
{
	char buf[BUF_SIZE], checked_passwd[BUF_SIZE];
	char check_user_name[2*USER_NAME_LENGTH];

	Str_Trans_CmdStr(user_name, check_user_name);
	if (EMPTY_STRING(pass_wd)) {
//		sprintf(buf,"%s -n \"%s\" > /dev/null", SMBPASSWD_PATH, user_name);
//		sprintf(buf, "%s \"%s\" \"\" > /dev/null", SMBPASSWD_PATH, user_name);
		sprintf(buf, "%s -n \"%s\" > /dev/null", SMBPASSWD_PATH, check_user_name);
	}
	else {
		memset(checked_passwd, 0, sizeof(checked_passwd));
		Str_Trans_CmdStr(pass_wd, checked_passwd);
//		sprintf(buf,"%s \"%s\" \"%s\" > /dev/null", SMBPASSWD_PATH, user_name, checked_passwd);
		sprintf(buf,"%s \"%s\" \"%s\" > /dev/null", SMBPASSWD_PATH, check_user_name, checked_passwd);
	}
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Change_Samba_User_Passwd: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Change_Samba_User_Passwd: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}


//-------------------------------------------------------------------
//  Modify a user password: It will modify a local & smb user passwd
//  by call Mod_Local_User_Passwd & Mod_Smb_Passwd
//  Mod_Local_User_Passwd() & Mod_Smb_Passwd()
//  Mod_Local_User_Passwd() Return code:
//  Return:   ERROR_NOT_FOUND      --> User doesn't exist
//            UG_MOD_LOCAL_USER --> call usermod CMD fail
//	ERROR_PDC_USER_GROUP --> the user is a PDC user
//            SUCCESS             --> Successful
//
//  Mod_Smb_Passwd() Return code:
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------------
//int Mod_User_Passwd(char *user_name, char *pass_wd) {
int Change_NAS_User_Password(char *user_name, char *pass_wd)
{
	int ret;
	char username[USER_NAME_LENGTH];
	NAS_USER_GROUP_TYPE type;

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03

	if(EMPTY_STRING(user_name)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
          
	type = NAS_Get_User_Type(username);

	if (type==NAS_USER_GROUP_PDC)
		return ERROR_PDC_USER_GROUP;	// PDC users are not allowed to change the password

	ret = Change_System_User_Password(username, pass_wd);
	if (ret == SUCCESS) {
		ret = Change_Samba_User_Passwd(username, pass_wd);
		if (ret == SUCCESS)
			ret = change_netware_user_password(username, pass_wd);
	}

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	//Update_Flash_Data(SAMBA_PASS_PATH);
	//Update_Flash_Data("/etc/config/nwpwd");

	return ret;
}



//-------------------------------------------------------
//  Get one specified local user information
//  Retunr: ERROR_NOT_FOUND --> the user does not exist
//          ERROR_EXEC_FAIL --> call getpwnam() fail
//          SUCCESS        --> Successful
//-------------------------------------------------------
//int Get_User_Info(struct local_user *user_ptr) {
int Get_NAS_User_Info(NAS_USER_INFO *user_info_ptr)
{
	struct passwd *p;

	// Catherine 12/14/01
	if (SUCCESS!=Get_Exact_System_User_Name(user_info_ptr->user_name))
		return  ERROR_NOT_EXISTED;

	if ((p = getpwnam(user_info_ptr->user_name)) == NULL) {
		strcpy(user_info_ptr->password, "");
		strcpy(user_info_ptr->primary_group, "");
		user_info_ptr->uid = user_info_ptr->gid = (ULONG) -1;
		if (errno == ENOENT)
			return ERROR_NOT_EXISTED;	// not found the user in passwd file
		else
			return ERROR_EXEC_FAIL;		// function execution fail
	}
	else {
		struct group *gp;

		strncpy(user_info_ptr->password, p->pw_passwd, PASSWORD_LENGTH-1);
		(user_info_ptr->password)[PASSWORD_LENGTH-1]=0x0;
		//Andy	05/26/03
		
		user_info_ptr->uid = p->pw_uid;
		user_info_ptr->gid = p->pw_gid;

		// first find out the primary group
		gp = getgrgid(p->pw_gid);
		if (gp == NULL)
			strcpy(user_info_ptr->primary_group, "");
		else
			strncpy(user_info_ptr->primary_group, gp->gr_name, PASSWORD_LENGTH-1);
			(user_info_ptr->primary_group)[PASSWORD_LENGTH-1]=0x0;
			//Andy	05/26/03	
	}

	return SUCCESS;
}







BOOL Is_Reserved_Group(char *group_name)
{
	if (compare_case_string(group_name, GROUP_ADMINISTRATORS) == 0) return TRUE;
	if (compare_case_string(group_name, GROUP_EVERYONE) == 0) return TRUE;
	if (compare_case_string(group_name, GROUP_GUEST) == 0) return TRUE;

	return FALSE;
}

BOOL Is_Hidden_Group(char *group_name)
{
	if (compare_case_string(group_name, GROUP_GUEST) == 0) return TRUE;

	return FALSE;
}






//------------------------------------------------------------
// Create a new Group
//  Return:   ERROR_ALREADY_EXISTED --> The group already exist
//            ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//------------------------------------------------------------
//int Create_One_Grp(char *grp_name) {
int Create_System_Group(char *group_name)
{
	char buf[2*BUF_SIZE];
	char check_group_name[2*GROUP_NAME_LENGTH];

	// Check the Group Name already exist in /etc/group file ?
	if (Is_System_Group(group_name)) 	// group exist, return error
		return ERROR_ALREADY_EXISTED;

	Str_Trans_CmdStr(group_name, check_group_name);
//	sprintf(buf, "%s \"%s\" > /dev/null", ADDGROUP_PATH, group_name);
	sprintf(buf, "%s \"%s\" 2>/dev/null 1>/dev/null", ADDGROUP_PATH, check_group_name);
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Create_System_Group: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Create_System_Group: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}


int Create_NAS_Group(char *group_name)
{
	int ret;
	char groupname[GROUP_NAME_LENGTH];

	strncpy(groupname, group_name, GROUP_NAME_LENGTH-1);
	groupname[GROUP_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if(EMPTY_STRING(groupname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(groupname); 	// Remove the extra blank in front, middle and back
	if (!Is_Legal_String(groupname))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	if (Is_Reserved_Group(groupname))	// check whether the name is reserved for the system
		return ERROR_RESERVED_GROUP;
	if (Is_System_Group(groupname))	// check whether the name exists or not 
		return ERROR_ALREADY_EXISTED;

	ret = Create_System_Group(groupname);

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);

	return ret;
}



//---------------------------------------------------------------
// Delete one Group
// Call groupdel, if the group is a primary group of one user
// this time groupdel will return error
// Cannot delete the default group "Administrator" & "everyone"
// Because all local users whose primary group either
// "Administrator" or "everyone", so upper description doesn't
// occupy on our project
// If the group doesn't exist, it just return OK
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
// 2000/9/19: Cannot delete the "administrator" & "everyone" grp
//---------------------------------------------------------------
//int Del_One_Grp(char *grp_name) {
int Remove_System_Group(char *group_name)
{
	char buf[2*BUF_SIZE];
	char grpname[GROUP_NAME_LENGTH];
	char check_group_name[2*GROUP_NAME_LENGTH];

	// Check if the Group Name exist in /etc/group file
	// Catherine 2001/12/14 -- replace by 'Get_Exact_System_Group_Name()'
	//if (!Is_System_Group(group_name))  // group not exist, return error
	strncpy(grpname, group_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if (SUCCESS!=Get_Exact_System_Group_Name(grpname))  // group not exist, return error
		return ERROR_NOT_EXISTED;

	if (Is_Reserved_Group(grpname)) return ERROR_RESERVED_GROUP;

	Str_Trans_CmdStr(grpname, check_group_name);
//	sprintf(buf, "%s \"%s\" > /dev/null", DELGROUP_PATH, grpname);
	sprintf(buf, "%s \"%s\" 2>/dev/null 1>/dev/null", DELGROUP_PATH, check_group_name);
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Remove_System_Group: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Remove_System_Group: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}

int Remove_NAS_Group(char *group_name)
{	int ret;
	ret = Remove_System_Group(group_name);
	if (ret == SUCCESS) {
		ret = Remove_Group_Share_Privilege(group_name);
	}

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);

	return ret;
}


int Change_System_Group_Name(char *old_group_name, char *new_group_name)
{
	char buf[3*BUF_SIZE];
	char old_grp_name[GROUP_NAME_LENGTH];
	char buf1[2*GROUP_NAME_LENGTH], buf2[2*GROUP_NAME_LENGTH];

	// Check if the Group Name exist in /etc/group file
	// Catherine 2001/12/14 -- replace
	//if (!Is_System_Group(old_group_name))  // group not exist, return error
	strncpy(old_grp_name, old_group_name, GROUP_NAME_LENGTH-1);
	old_grp_name[GROUP_NAME_LENGTH-1]=0x0;		//Andy	05/26/03
	
	if (SUCCESS!=Get_Exact_System_Group_Name(old_grp_name))  // group not exist, return error
		return ERROR_NOT_EXISTED;

	if (Is_Reserved_Group(old_grp_name)) return ERROR_RESERVED_GROUP;

	Str_Trans_CmdStr(new_group_name, buf1);
	Str_Trans_CmdStr(old_group_name, buf2);
//	sprintf(buf, "%s -n \"%s\" \"%s\" > /dev/null", GROUPMOD_PATH, new_group_name, old_grp_name);
	sprintf(buf, "%s -n \"%s\" \"%s\" > /dev/null", GROUPMOD_PATH, buf1, buf2);
#ifdef DEBUG		// For checking if the buf overflow happened...
	if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Change_System_Group_Name: buf size too small.\n");
#endif

	if (system(buf) != 0) {
#ifdef DEBUG		// dump more error message...
		fprintf(stderr, "Change_System_Group_Name: system(%s) failed.\n", buf);
#endif
		return ERROR_EXEC_FAIL;
	}

	return SUCCESS;
}


//-------------------------------------------------------------
// Modify group name
//  Return:   ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//-------------------------------------------------------------
//int Mod_Grp_Name(char *old_grp_name, char *new_grp_name) {
int Change_NAS_Group_Name(char *old_group_name, char *new_group_name)
{
	int ret;

	ret = Change_System_Group_Name(old_group_name, new_group_name);
	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	return ret;
}


/*
//------------------------------------------------------------
// Check the specified user is belong to the specified group
// if belong to return 1 otherwise return 0
//------------------------------------------------------------
int Is_My_User(char *grp_name, char *user_name) {
    struct group *grp_ptr;
    int curr_list=0;
    int found_flag = 0;

    grp_ptr = getgrnam(grp_name);
    if (grp_ptr == NULL) return 0;

    while ( *(grp_ptr->gr_mem+curr_list) ) {
          if ( strcmp(user_name,*(grp_ptr->gr_mem+curr_list))==0 )
              return 1;
          curr_list++;
    }

    return 0;
}
*/




//-------------------------------------------------------------
// Add one group's member
//-------------------------------------------------------------
//int Add_One_GrpUSER(char *group_name, char *user_name) {
int Add_NAS_Group_To_User(char *group_name, char *user_name)
{
	return	Add_NAS_User_To_Group(user_name, group_name);
}

//-------------------------------------------------------------
// Del one group's member
//-------------------------------------------------------------
//int Del_One_GrpUSER(char *grp_name, char *user_name) {
int Remove_NAS_Group_From_User(char *group_name, char *user_name)
{
    return Remove_NAS_User_From_Group(user_name, group_name);
}


//--------------------------------------------------------------
// Get_Grp_Info(): get one specified group's Information
//  & his all mmbers
//  Return:   ERROR_NOT_FOUND    --> The group not exist
//            ERROR_EXEC_FAIL    --> call system function fail
//            ERROR_BUFFER_TOO_SMALL     --> You pass in smaller buffer
//            SUCCESS           --> Successful
//--------------------------------------------------------------
//int Get_Grp_Info(struct local_grp *grp_ptr) {
int Get_NAS_Group_Info(NAS_GROUP_INFO *group_info_ptr)
{
	struct group *pg;
	int i, n;

	//--- Check the Group Name already exist in /etc/group file ? ---
/* Catherine 12/14/01--replace by 'Get_Exact_System_Group_Name()'
	if (!Is_System_Group(group_info_ptr->group_name)) // group not exist, return error
		return ERROR_NOT_EXISTED;
*/
	if (SUCCESS!=Get_Exact_System_Group_Name(group_info_ptr->group_name))
		return ERROR_NOT_EXISTED;

	if ((pg = getgrnam(group_info_ptr->group_name)) == NULL) return ERROR_EXEC_FAIL;

	group_info_ptr->gid = pg->gr_gid;

	i = n = 0;
	while (*(pg->gr_mem + i) != NULL) {

		// if user name is root skip it
		// Meiji ????? SKIP ALL RESERVED USERS HERE... TBD ??????
//		if (strcasecmp(*(pg->gr_mem + i), "root") !=0) {
		if (!Is_Hidden_User(*(pg->gr_mem + i))) {
			if (n < group_info_ptr->list_count) {
				strncpy((char *) (group_info_ptr->user_list + n), *(pg->gr_mem + i), USER_NAME_LENGTH-1);
				(*(group_info_ptr->user_list + n))[USER_NAME_LENGTH-1]=0x0;
				//Andy	05/26/03
				n++;
			}
			else
				return ERROR_BUFFER_TOO_SMALL;
		}
		i++;
	}
	group_info_ptr->user_count = n;
	return SUCCESS;
}


//------------------------------------------------------------
//   Get All Local Users Name
//   It will return the total user No. the Linux has
//   So you should check the return value is small/equal
//   than you pass in buffer size
//   Return: ERROR_BUFFER_TOO_SMALL  --> you pass in buffer is too small
//           nnn          --> users number of the Linux has
// 2000/9/19: If the user name is "root", don't display he
//------------------------------------------------------------
//int Get_All_Users(char (*list_ptr)[USER_NAME_LENGTH], int list_cnt) {
int Get_NAS_User_List(char (*user_list)[USER_NAME_LENGTH], int list_cnt)
{
	struct passwd *p;
	int i = 0, n = 0;

	setpwent();
	while ((p = getpwent()) != NULL) {

		// Meiji ??? TBD... What user to be hidden here.,,, ????
		/*----- add to check root user start --*/
//		if (strcasecmp(p->pw_name, "root")==0)  continue;
		if (Is_Hidden_User(p->pw_name)) continue;
		/*----- add to check root user end   --*/
//2001/7/19 add by louistsai
		if (is_pdc_name(p->pw_name)) continue;
//end

		if (i < list_cnt) {
			strncpy((char *) (user_list + i), p->pw_name, USER_NAME_LENGTH-1);
			(*(user_list + i))[USER_NAME_LENGTH-1]=0x0;
			//Andy	05/26/03
			i++;
		}
		n++;	// n represent total count of the visible users...
	}	// end of while
	endpwent();    // close /etc/passwd

	if (i < n) return ERROR_BUFFER_TOO_SMALL;	// i == n, if space is enough

	// sort area
	Sort_Name_List((char **) user_list, USER_NAME_LENGTH, i);

	return i;	// return total count of the users listed...
}


//--------------------------------------------------------------
//   Get All Local Group Name
//  Return:   ERROR_BUFFER_TOO_SMALL       --> You pass in smaller buffer
//            SUCCESS             --> Successful
//--------------------------------------------------------------
//int Get_All_Grps(char (*list_ptr)[GROUP_NAME_LENGTH], int list_cnt) {
int Get_NAS_Group_List(char (*group_list)[GROUP_NAME_LENGTH], int list_cnt)
{
	struct group *gp;
	int i = 0, n = 0;

	setgrent();
	while ((gp = getgrent()) != NULL) {

		// Meiji??? Any hidden group can be filtered here...???
		if (Is_Hidden_Group(gp->gr_name)) continue;
		// TBD...
//2001/7/19 add by louistsai
		if (is_pdc_name(gp->gr_name)) continue;
//end
		if (i < list_cnt) {
			strncpy(*(group_list + i), gp->gr_name, GROUP_NAME_LENGTH-1);
			(*(group_list + i))[GROUP_NAME_LENGTH-1]=0x0;
			//Andy	05/26/03
			i++;
		}
		n++;
	}
	endgrent();

	if (i < n) return ERROR_BUFFER_TOO_SMALL;	// i == n, if space is enough

	Sort_Name_List((char **) group_list, GROUP_NAME_LENGTH, i);
	return i;
}


//------------------------------------------------------------------
//  Get one specified local user's groups, it will call getgrent()
//  If successful, the first member of the grp_list_ptr is the
//   primary group.
//  Return:   ERROR_NOT_FOUND      --> User doesn't exist
//            ERROR_EXEC_FAIL      --> call system function fail
//            ERROR_BUFFER_TOO_SMALL       --> you pass in buffer is too small
//            nnn               --> The groups he has
// 2000/10/9: The grp_max_cnt & grp_curr_cnt is equal, so del the
  //          grp_max_cnt
//------------------------------------------------------------------
//int Get_All_UserGRP(char *user_name, char (*grp_list_ptr)[GROUP_NAME_LENGTH], int grp_list_cnt) {
int Get_NAS_Group_List_Of_User(char *user_name, char (*group_list)[GROUP_NAME_LENGTH], int list_cnt)
{
	//NAS_USER_INFO ui;
	//int ret;
	struct group *gp;
	int i = 0;
	char username[USER_NAME_LENGTH];
	
	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	/*Marked by Shone (2004,12,15)
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	*/	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	setgrent();
	while ((gp = getgrent()) != NULL) {
		int j = 0;
		while (*(gp->gr_mem + j) != NULL) {	// search all groups
			if (compare_case_string(*(gp->gr_mem + j), username) == 0) { // find user_name in this group
				if (i <= list_cnt) {

					if (Is_Hidden_Group(gp->gr_name)) continue;

					//-- the primary grp already add to the first --
					//-- don't need add again                     --
					//if (compare_string(ui.primary_group, gp->gr_name) !=0) {
						strncpy(*(group_list + i), gp->gr_name, GROUP_NAME_LENGTH-1);
						(*(group_list+i))[GROUP_NAME_LENGTH-1]=0x0;
						//Andy	05/26/03
						i++;
					//}
				}
				else {	// pass in array is too small
					endgrent();
					return ERROR_BUFFER_TOO_SMALL;
				}
			}
			j++;
		}
	}
	endgrent();

	if (i > 0)
		Sort_Name_List((char **) group_list, GROUP_NAME_LENGTH, i);
	return i;
}


//------------------------------------------------------------
// Get one goups's members
// It will call Get_Grp_Info)
//  Return:   ERROR_NOT_FOUND    --> The group not exist
//            ERROR_EXEC_FAIL    --> call system function fail
//            ERROR_BUFFER_TOO_SMALL     --> You pass in smaller buffer
//            nnn (member cnt)--> Successful
//------------------------------------------------------------
//int Get_Grp_Users(char *grp_name, char (* user_list)[USER_NAME_LENGTH], int user_list_cnt ) {
int Get_NAS_User_List_Of_Group(char *group_name, char (*user_list)[USER_NAME_LENGTH], int list_cnt)
{
	NAS_GROUP_INFO gi;
	int ret;
	char groupname[GROUP_NAME_LENGTH];
	
	strncpy(groupname, group_name, GROUP_NAME_LENGTH-1);
	groupname[GROUP_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if(EMPTY_STRING(groupname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(groupname); 	// Remove the extra blank in front, middle and back
	/*Marked by Shone (2004,12,15)
	if (!Is_Legal_String(groupname))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	*/	
	if (!Is_System_Group(groupname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	strncpy(gi.group_name, groupname, GROUP_NAME_LENGTH-1);
	(gi.group_name)[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	gi.user_list = user_list;
	gi.list_count = list_cnt;
	gi.user_count = 0;
	if ((ret = Get_NAS_Group_Info(&gi)) != SUCCESS) return ret;

	Sort_Name_List((char **) user_list, USER_NAME_LENGTH, gi.user_count);
	return gi.user_count;
}

// 2004.12.09, Johnson Cheng add for multi-domain AD
char* Get_Group_Name_By_User_Name(char* user_name)
{
	struct passwd *user;
	struct group *data;
	
	if (SUCCESS != Get_Exact_System_User_Name(user_name)) 
		return NULL;
	if((user = getpwnam(user_name)) != NULL)
		if((data = getgrgid(user->pw_gid)) != NULL)
			return data->gr_name;
	
	return NULL;
}

//--------------------------------------------------------------------
// Add a specified NAS user to one group
//
//  Return:   ERROR_NOT_FOUND      --> User or Group doesn't exist
//            ERROR_LOCK_FILE      --> group file be locked
//	      ERROR_OPEN_FILE	   --> open group file failed
//            SUCCESS              --> Successful
//
// 2000/4/11 Louis Tsai modified comment
//--------------------------------------------------------------------
int Add_NAS_User_To_Group(char *user_name, char *grp_name)
{

	int str_len,name_len,fld_len,new_len;
	int result;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	// check user name
	if(!IsLocalUser(username))            //Shone added, check for AD domain user
	       return ERROR_NOT_LOCAL_USER;    
	if(!IsLocalGroup(grpname))            //Shone added, check for AD domain group
	       return ERROR_NOT_LOCAL_GROUP;       
	       
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check group name
	if(EMPTY_STRING(grpname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(grpname))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_Group(grpname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check the user_name belongs to the grp_name member already?
	if (User_Belongs_To_Group(username, grpname))
		return SUCCESS;

	// Set string comparision with case insensitive
	Conf_Case_Sensitive(0);
	name_len =  strlen (username);

	if( (str_len = SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,NULL,0)) > 0) {
		char out_str[str_len];
		SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,out_str,str_len);
		if((fld_len = Get_String_Field (out_str,4,':',NULL,0)) > 0) {
			char fld_str[fld_len];
			char set_str[fld_len+name_len];
			Get_String_Field (out_str,4,':',fld_str,fld_len);

			if( compare_case_string(fld_str,"") == 0 ) //no member in this group
				sprintf(set_str,"%s%s",fld_str,username);
			else
				sprintf(set_str,"%s,%s",fld_str,username);

			if ( (new_len = Set_String_Field(out_str,4,':',set_str,NULL,0)) > 0) {
				char new_str[new_len];
				Set_String_Field(out_str,4,':',set_str,new_str,new_len);
				result = SYS_Set_One_String(GROUP_CONF_PATH,1,':',grpname,new_str);
			}
		}
		else
			result = fld_len;
	}
	else
		result = str_len;

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	//Update_Flash_Data(SAMBA_PASS_PATH);

	return result;
}

//--------------------------------------------------------------------
// Remove a specified NAS user from one group
//
//  Return:   ERROR_NOT_FOUND      --> User or Group doesn't exist
//            ERROR_LOCK_FILE      --> group file be locked
//	      ERROR_OPEN_FILE	   --> open group file failed
//            SUCCESS              --> Successful
//
// 2000/4/11 Louis Tsai modified comment
//--------------------------------------------------------------------

int Remove_NAS_User_From_Group(char *user_name, char *grp_name)
{
 	char buf[USER_NAME_LENGTH];
	int str_len,name_len,fld_len,new_len;
	int result;
	int i=0;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	// check user name
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check group name
	if(EMPTY_STRING(grpname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(grpname))		// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;

	if (!Is_System_Group(grpname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check the user_name belongs to the grp_name member already?
	if (! User_Belongs_To_Group(username, grpname))
		return ERROR_NOT_EXISTED;

	name_len =  strlen (username);
	// Set string comparision with case insensitive
	Conf_Case_Sensitive(0);

	if( (str_len = SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,NULL,0)) > 0) {
		char out_str[str_len];
		SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,out_str,str_len);
		if((fld_len = Get_String_Field (out_str,4,':',NULL,0)) > 0) {
			char fld_str[fld_len];
			Get_String_Field (out_str,4,':',fld_str,fld_len);
			do {
				i++;
				Get_String_Field(fld_str,i,',',buf,USER_NAME_LENGTH);
				if( compare_case_string(username,buf) == 0 ) break;
				if ( i>65534 ) return ERROR_NOT_EXISTED; // prevent system crazy
			} while(1);

			Remove_String_Field(fld_str,i,',');

			if ( (new_len = Set_String_Field(out_str,4,':',fld_str,NULL,0)) > 0) {
				char new_str[new_len];
				Set_String_Field(out_str,4,':',fld_str,new_str,new_len);
				result = SYS_Set_One_String(GROUP_CONF_PATH,1,':',grpname,new_str);
			}

		}
		else
			result = fld_len;
	}
	else
		result = str_len;

	//Update_Flash_Data(PASS_CONF_FILE);
	//Update_Flash_Data(GROUP_CONF_PATH);
	//Update_Flash_Data(SAMBA_PASS_PATH);

	return result;
}

//== Group Fnction ===



//=============================================================


//=============   SHARE FUNCTIONS   ====================


//---------------------------------------------------------------
// int Get_All_SubDIR_Cnt_Len()
// The directory is "." will be igrone
// If top_dir same as dir, then ingore the ".." directory
//---------------------------------------------------------------
int  Get_All_SubDIR_Cnt_Len(char *top_dir, char *dir, int *cnt, int *max_len)
{
     DIR *dp;
     struct dirent *entry;
     struct stat statbuf;
	 char curr_path_name[PATH_MAX];

     if((dp = opendir(dir)) == NULL) {
         fprintf(stderr,"cannot open directory: %s\n", dir);
         return ERROR_NOT_FOUND;
     }

     *cnt = *max_len = 0 ;
    //     Save current DIR -
	//     curr_path_name = get_current_dir_name();
	//     if ( curr_path_name==NULL) {
	if (NULL == getcwd(curr_path_name, sizeof(curr_path_name))) {
		return ERROR_EXEC_FAIL;
	}

     chdir(dir);
     while((entry = readdir(dp)) != NULL) {
         lstat(entry->d_name,&statbuf);
         if(S_ISDIR(statbuf.st_mode)) {
             /* Found a directory, but ignore . and .. */
			 // Modified by Louis 03/20/01 strncmp->strcmp
			 if( strcmp(".",entry->d_name)==0) continue;
			 //if top_dir=dir,ignore ".."
// Catherine			 if( strcmp("..",entry->d_name) == 0 && strcmp(top_dir,dir)==0 )  continue;
			 if( strcmp("..",entry->d_name) == 0 && compare_string(top_dir,dir)==0 )  continue;

			 (*cnt)++;
             if ( (strlen(entry->d_name)+1) > *max_len) *max_len=(strlen(entry->d_name)+1);
         }
     }
     chdir(curr_path_name);
     closedir(dp);

     return SUCCESS;
}

//---------------------------------------------------------------
// char **Get_All_SubDIR(char *path_name)
// you must free the char **after the return is not NULL Value
// The directory "." will be igrone
// If top_dir same as dir, then ingore the ".." directory
//---------------------------------------------------------------
int  Get_All_SubDIR(char *top_dir, char *dir, char *ptr, int ptr_cnt, int array_len)
{
     DIR *dp;
     struct dirent *entry;
     struct stat statbuf;
     int cnt=0;
	 char curr_path_name[PATH_MAX];

     if((dp = opendir(dir)) == NULL) {
         fprintf(stderr,"cannot open directory: %s\n", dir);
         return ERROR_NOT_FOUND;
     }
    //--- Save current DIR -
	//    curr_path_name = get_current_dir_name();
	//    if ( curr_path_name==NULL) {
	if (NULL == getcwd(curr_path_name, sizeof(curr_path_name))) {
		return ERROR_EXEC_FAIL;
	}

     chdir(dir);
     while((entry = readdir(dp)) != NULL) {
         lstat(entry->d_name,&statbuf);
         if(S_ISDIR(statbuf.st_mode)) {
             /* Found a directory, but ignore '.' */
			 // Modified by Louis 03/20/01 strncmp->strcmp
			 if( strcmp(".",entry->d_name)==0) continue;
			 //if top_dir=dir,ignore ".."
// Catherine			 if( strcmp("..",entry->d_name) == 0 && strcmp(top_dir,dir)==0 )  continue;
			 if( strcmp("..",entry->d_name) == 0 && compare_string(top_dir,dir)==0 )  continue;
             // check allocated memory size is overflow...
			 if (cnt >= ptr_cnt) {
               chdir(curr_path_name);
               return ERROR_BUFFER_TOO_SMALL;
             }
			 // Modified by Louis 03/20/01 strcpy->strncpy
             strncpy( (ptr+cnt*array_len),entry->d_name,array_len);
//			 printf("%s\n",ptr+cnt*array_len);
             cnt++;
         }
     }
     chdir(curr_path_name);
     closedir(dp);

     return SUCCESS;
}



//----------------------------------------------------------------------
// 2000/10/9: Created
// Is_My_SubDIR(char *search_DIR, char *target_DIR)
// return SUCCESS: It is my sub-directory
// because it is recursive function you should return continuly if the
// the static varable f_flag == 1
//----------------------------------------------------------------------
int Is_My_SubDIR(char *search_DIR, char *target_DIR)
{
    DIR *dp;
    struct dirent *entry;
    struct stat statbuf;
    char   l_dir[255];
    int    f_flag=0;  //found flag
	char curr_path_name[PATH_MAX];

// Catherine    if ( strcmp(search_DIR, target_DIR)==0 ) return SUCCESS;
    if (compare_string(search_DIR, target_DIR)==0 ) return SUCCESS;

    if((dp = opendir(search_DIR)) == NULL) {
        #ifdef DEBUG
          printf("cannot open directory: %s\n", search_DIR);
        #endif
        return ERROR_OPEN_FILE;
    }
    //--- Save current DIR -
	//    curr_path_name = get_current_dir_name();
	//    if ( curr_path_name==NULL) {
	if (NULL == getcwd(curr_path_name, sizeof(curr_path_name))) {
		return ERROR_EXEC_FAIL;
	}

    chdir(search_DIR);
    while((entry = readdir(dp)) != NULL) {
        lstat(entry->d_name,&statbuf);
        if(S_ISDIR(statbuf.st_mode)) {
             /* Found a directory, but ignore . and .. */
			// Modified by Louis 03/20/01 strncmp->strcmp
            if(strcmp(".",entry->d_name) == 0 || strcmp("..",entry->d_name) == 0)
				continue;
            //printf("%*s%s\n",depth,"",entry->d_name);
			if(strcmp(search_DIR,"/")!=0)
               sprintf(l_dir,"%s/%s",search_DIR,entry->d_name);
			else
               sprintf(l_dir,"%s%s",search_DIR,entry->d_name);
			#ifdef DEBUG
              printf("l_dir=%s,len=%d, target_DIR=%s,len=%d,\n",l_dir,strlen(l_dir),target_DIR,strlen(target_DIR));
            #endif
// Catherine            if ( strcmp(l_dir,target_DIR)==0 ) {
            if ( compare_string(l_dir,target_DIR)==0 ) {
               f_flag = 1;
               break;

            }
            /* Recurse at a new indent level */
            if(Is_My_SubDIR(l_dir,target_DIR)==SUCCESS) {
				f_flag = 1;
				break;
			}
        }
    }
    chdir(curr_path_name);
    closedir(dp);
    if ( f_flag)   return SUCCESS;
    else           return ERROR_NOT_FOUND;
}



//=================================================================
//====                Internal Functions                       ====
//=================================================================





//=============   GROUP FUNCTIONS   =================


//=========    SMB USER FUNCTIONS      ========================



/*
 * 2000/10/17: change a file/dir owner & group
 *
 */

//int Change_Owner_Grp(char *path, char *owner, char*grp) {
int Change_File_Attribute(char *path, char *owner, char *group)
{
	uid_t uid;
	gid_t gid;
	struct passwd *pp;
	struct group  *gp;
	char uname[USER_NAME_LENGTH], gname[GROUP_NAME_LENGTH];

	// get uid
	strncpy(uname, owner, USER_NAME_LENGTH-1);
	uname[USER_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	if (SUCCESS!=Get_Exact_System_User_Name(uname))
		return ERROR_NOT_FOUND;	
	pp = getpwnam(uname);
	if (pp == NULL) return ERROR_NOT_FOUND;  // cannot find the owner
	else uid = pp->pw_uid;

	// get gid
	strncpy(gname, group, GROUP_NAME_LENGTH-1);
	gname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	if (SUCCESS!=Get_Exact_System_Group_Name(gname))
		return ERROR_NOT_FOUND;
	gp = getgrnam(gname);
	if (gp == NULL) return ERROR_NOT_FOUND; //cannot find the group
	else gid = gp->gr_gid;

	if (chown(path, uid, gid) !=0) return ERROR_EXEC_FAIL; // chown() fail

	return SUCCESS;
}


// Return the uid of the specified user name.
// Return:
//	uid			on success
//	ERROR_BAD_USERNAME	not valid username
int Get_System_User_ID(char *name)
{
	struct passwd *pwd;
	char uname[USER_NAME_LENGTH];
	// Catherine 12/14/01
	strncpy(uname,name, USER_NAME_LENGTH-1);
	uname[USER_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	if (SUCCESS!=Get_Exact_System_User_Name(uname))
		return  ERROR_BAD_USERNAME;

	pwd = getpwnam(uname);
	if (NULL == pwd) return ERROR_BAD_USERNAME;
	return pwd->pw_uid;
}

int Check_Local_User_Password(char *user, char *password)
{
	FILE	*fp;
	int	ret;
	char uname[USER_NAME_LENGTH];
	// Catherine 12/14/01
	strncpy(uname, user, USER_NAME_LENGTH-1);
	uname[USER_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	if (SUCCESS!=Get_Exact_System_User_Name(uname))
		return  ERROR_BAD_USERNAME;
		
        if (((fp = fopen("/etc/shadow", "r")) == NULL) && errno == ENOENT)
	{	// file not found!
                // handle /etc/passwd
                struct passwd *pwd;

                if ((pwd = getpwnam(uname)) == (struct passwd *) NULL)
                        return ERROR_BAD_USERNAME;
                ret=strcmp((char *)NASCRYPT(password, pwd->pw_passwd), pwd->pw_passwd); // encrypy cleartext passwd
		/* add by Kent			*/
		/* to prevent no password	*/
		if (pwd->pw_passwd[0]==0x0 && password[0]!=0x0)
			return ERROR_BAD_PASSWORD;
                if ((ret) || (pwd->pw_passwd=='\0') || (password == '\0'))
                        return ERROR_BAD_PASSWORD;
        }
        else
	{
                struct spwd *shadow;

                // compare (name,passwd) pairs to the tuple in /etc/shadow
                // first get user information by getspnam(username)........
                // The user information will be filled in struct-->shadow
                if (fp != NULL)
			fclose(fp);
                if ((shadow = getspnam(uname)) == (struct spwd *) NULL)
                        return ERROR_BAD_USERNAME;
		/* can shadow use NASCRYPT ????? */
                ret = strcmp((char *) crypt(password, shadow->sp_pwdp), shadow->sp_pwdp); // encrypy cleartext passwd
                /* add by Kent                  */
                /* to prevent no password       */
                if (shadow->sp_pwdp[0]==0x0 && password[0]!=0x0)
                        return ERROR_BAD_PASSWORD;
                if ((ret) || (shadow->sp_pwdp == '\0') || (password == '\0'))
                        return ERROR_BAD_PASSWORD;
        }
        return 0;
}

// Check the username & password to see if it's valid for the local system
// Return:
//	0			pass
//	ERROR_BAD_USERNAME	not valid username
//	ERROR_BAD_PASSWORD	not correct password
int Check_System_User_Password(char *user_name, char *password)
{
//	NAS_USER_GROUP_TYPE type;

	if (!strcasecmp(user_name, "administrator") || !strcasecmp(user_name, "guest"))
	{
		if (Check_Local_User_Password(user_name, password)==0)
			return 0;
		else
		{
			if (PDC_Is_Enabled()) {
				//type=NAS_Get_User_Type(user_name);
				//if (type==NAS_USER_GROUP_BOTH)
					return PDC_Check_User_Password(user_name, password, 1);
			}
			else
				return ERROR_BAD_PASSWORD;
		}
	}
	
	// 2004.12.09, Johnson Cheng modified for multi-domain AD
	/*if (PDC_Is_Enabled())
	{
		type=NAS_Get_User_Type(user_name);
		
		switch (type) {
			case NAS_USER_GROUP_PDC: 
			     {
				if (PDC_Check_User_Password(user_name,password, 1)==0)
				{	// check success
					return 0;
				}
				else 
					return ERROR_BAD_PASSWORD;
			      }
			      break;
			      
			case NAS_USER_GROUP_BOTH: 
			     {
				if (PDC_Check_User_Password(user_name,password, 1)==0)
					return 0;
			     }
			     break;
			     
			case NAS_USER_GROUP_NONE:
			     return ERROR_BAD_USERNAME;
			     break;
			     
			default:
			
		}
	}
	
	return Check_Local_User_Password(user_name, password);*/
	if(IsLocalUser(user_name))
		return Check_Local_User_Password(user_name, password);
	else if (PDC_Is_Enabled())
		return PDC_Check_User_Password(user_name, password, 1);
	else
		return ERROR_BAD_USERNAME;
	
/*
local_check:
	if (((fp = fopen("/etc/shadow", "r")) == NULL) && errno == ENOENT) {  // file not found!
		// handle /etc/passwd
		struct passwd *pwd;

		if ((pwd = getpwnam(user_name)) == (struct passwd *) NULL) {
			return ERROR_BAD_USERNAME;
		}

		ret = strcmp((char *) NASCRYPT(password, pwd->pw_passwd), pwd->pw_passwd); // encrypy cleartext passwd
		if ((ret) || (pwd->pw_passwd=='\0') || (password == '\0')) {        // check passwd
			return ERROR_BAD_PASSWORD;
		}
	}
	else {
		struct spwd *shadow;

		// compare (name,passwd) pairs to the tuple in /etc/shadow
		// first get user information by getspnam(username)........
		// The user information will be filled in struct-->shadow
		if (fp != NULL) fclose(fp);

		if ((shadow = getspnam(user_name)) == (struct spwd *) NULL) {
			return ERROR_BAD_USERNAME;
		}

		ret = strcmp((char *)NASCRYPT(password, shadow->sp_pwdp), shadow->sp_pwdp); // encrypy cleartext passwd

		if ((ret) || (shadow->sp_pwdp == '\0') || (password == '\0')) {       // check passwd
			return ERROR_BAD_PASSWORD;
		}
	}

	return 0;*/
}


// Check the username & password to see if it's valid and with administrator privilege
// To have administrator privilege, the uid has to be zero...
// Return:
//	0			pass
//	ERROR_BAD_USERNAME	not valid username
//	ERROR_BAD_PASSWORD	not correct password
//	ERROR_NO_PRIVILEGE	no admin privilege
int Check_NAS_Administrator_Password(char *admin_name, char *password)
{
	int ret;

	// Meiji 00-12-28: assume that all the username is in lower case...
//	for (i = 0; admin_name[i]; i++)
//		admin_name[i] = tolower(admin_name[i]);

	ret = Check_System_User_Password(admin_name, password);
	if (ret == 0) {
		// privl 1-->root, 0 -->user
//		if (0 != Get_System_User_ID(admin_name)) ret = ERROR_NO_PRIVILEGE;
		if (!User_Belongs_To_Group(admin_name, GROUP_ADMINISTRATORS)) ret = ERROR_NO_PRIVILEGE;
	}
	return ret;
}



// Check the username & password to see if it's valid and with administrator privilege
// To have administrator privilege, the uid has to be zero...
// Return:
//	0			pass
//	ERROR_BAD_USERNAME	not valid username
//	ERROR_BAD_PASSWORD	not correct password
int Check_NAS_User_Password(char *user_name, char *password)
{
	return Check_System_User_Password(user_name, password);
	// ????????????  check remote PDC users??????????????
}

///////////////////////////////////////////////////////////////////////////////////////////
// locate_my_field
// description: locate the real start position of the field (separated by delimeter),
//		and set the end of this field '\0';
//		This function starts seeking from (buf_start+field_ref_offset),
//		if (buf_start+prev_end) exceeds (buf_start+field_ref_offset),
//		then this field doesn't start until the first delimeter is met.
// input:
//		buf_start -- the start position of the string
//		prev_end -- the "real" offset of the previous field's "end-of-field" ('\0')
//		field_ref_offset -- the offset of the start of the desired field,
//					but for reference only
//		delimeter -- the character that is used to separate the fields
///////////////////////////////////////////////////////////////////////////////////////////
char* locate_my_field(char* buf_start, int prev_end, int field_ref_offset, char delimeter)
{
	char *result = NULL, *p = NULL;
	int offset = (prev_end + 1 > field_ref_offset) ? (prev_end + 1) : field_ref_offset;

	if (buf_start == NULL || field_ref_offset<0) return NULL;

	result = buf_start + offset;
	if (result == NULL || *result == '\0') return NULL;

	while (result && *result && *result == delimeter) result++;

	if (*result == '\0') return NULL;

	p = result;
	while (p && *p && (*p != delimeter || *(p+1)!=delimeter)) p++;//Shone Modified to solve name-with-space problem
	     
	if (*p == delimeter) *p = '\0';
	return result;
}

typedef struct {
	char	pid[10];
	char	user[USER_NAME_LENGTH];
	char	group[GROUP_NAME_LENGTH];
	char	ip[USER_NAME_LENGTH];
} s_client_table;

typedef struct {
	char	service[SHARE_NAME_LENGTH];
	char	pid[10];
	char	machine[USER_NAME_LENGTH];
	char	connect_time[BUF_SIZE];
} s_service_table;

//------------------------------------------------------------------------//
//get samba status
//
//Reurn	:
//	NNN>0 			, number of active connection
//	ERROR_EXEC_FAIL		, excute system command error
//	ERROR_BUFFER_TOO_SMALL	, input buffer too small
//
//Commnet:
//	if u_list = NULL , return value is equal to necessary list count
//------------------------------------------------------------------------//

static int m_datacmp(const void *a, const void *b)
{
	SAMBA_STATUS_INFO *c,*d;
	c = (SAMBA_STATUS_INFO *) a;
	d = (SAMBA_STATUS_INFO *) b;
	return(strcasecmp(c->service,d->service));
}

int Get_Samba_Status(SAMBA_STATUS_INFO *u_list, int list_cnt) //modified by YFHuang 05/04/2004
{
	char cmd_str[BUF_SIZE],*str_ptr;
	FILE *fp;
	int tmp_cnt=0;
	char buf[BUF_SIZE];
	int found_tag = 0;
	int ser_pos = 0,ip_pos = 0,pid_pos = 0,user_pos = 0,from_pos = 0,login_pos = 0,group_pos = 0;
	int smb_version;
	int part=0;
	int ct_size = 0, st_size = 0, i, j;
	s_client_table *pc = NULL;
	s_service_table *ps = NULL;
        //Shone added parameter '-p',for samba 3.0.10 will show only share folders using parameter '-S'			
	sprintf(cmd_str,"%s -p > %s;%s -S >> %s",SMBSTATUS_PATH,SMBREPORT_PATH,SMBSTATUS_PATH,SMBREPORT_PATH);

	//excute smbstatus -S to create active user report
	if (system(cmd_str) !=0)
		return ERROR_EXEC_FAIL;

	if ( ((fp=fopen(SMBREPORT_PATH,"r")) == NULL) && errno == ENOENT )
		return ERROR_OPEN_FILE;
				
	//Samba version 2 & 3
	while( fgets(buf,BUF_SIZE,fp) != NULL) {
		if (buf[0] == '\n') continue;
		
		if( (str_ptr=strstr(buf, "Samba version 2")) != NULL)
		{
			smb_version = 2;
			continue;
		}
		if( (str_ptr=strstr(buf, "Samba version 3")) != NULL)
		{
			smb_version = 3;
			continue;
		}

		if(smb_version == 2){
			if ( found_tag == 0 ) {
				if ( (str_ptr=strstr(buf,"Service")) != NULL) {
					ser_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"pid");
					pid_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"uid");
					user_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"ip");
					ip_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"machine");
					from_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"Time");
					login_pos = (int)(str_ptr-buf);
	
					found_tag = 1;
					fgets(buf,BUF_SIZE,fp); // ignore ------------
					continue;
				}
			}
			else {
				if( (tmp_cnt+1 > list_cnt) && (u_list != NULL)) {
					fclose(fp);
					return ERROR_BUFFER_TOO_SMALL;
				}
				if ( u_list != NULL) {
					buf[pid_pos-1]  = '\0';
					buf[user_pos-1] = '\0';
					buf[from_pos-1] = '\0';
					buf[ip_pos-1]	= '\0';
					buf[login_pos-1]= '\0';
	
					strcpy((u_list+tmp_cnt)->service,buf+ser_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->service);
	
					strcpy((u_list+tmp_cnt)->pid,buf+pid_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->pid);
	
					strcpy((u_list+tmp_cnt)->user_name,buf+user_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->user_name);
	
					strcpy((u_list+tmp_cnt)->from_name,buf+from_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->from_name);
	
					strcpy((u_list+tmp_cnt)->ip,buf+ip_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->ip);
	
					strcpy((u_list+tmp_cnt)->login_time,buf+login_pos);
					stripe_white_space_tail((u_list+tmp_cnt)->login_time);
				}
	
				tmp_cnt++;
			}
		}
		else if(smb_version == 3)
		{
			void *ptmp = NULL;

			if ( (str_ptr = strstr(buf,"PID")) != NULL)
			{ // start of client table
				found_tag = 0;
				part = 1;
			}
			else if ( (str_ptr = strstr(buf,"Service")) != NULL)
			{ // start of service table
				found_tag = 0;
				part = 2;
			}
			
			if ( found_tag == 0 ) { // the table header line
				if ( part == 1 ) {
					pid_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"Username");
					user_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"Group");
					group_pos = (int)(str_ptr-buf);
					
					str_ptr = strstr(buf,"Machine");
					ip_pos = (int)(str_ptr-buf);

					found_tag = 1;
					tmp_cnt = 0;
					fgets(buf,BUF_SIZE,fp); // ignore ------------
					continue;
				}
				if ( part == 2 ) {
					ser_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"pid");
					pid_pos = (int)(str_ptr-buf);
					
					str_ptr = strstr(buf,"machine");
					from_pos = (int)(str_ptr-buf);
	
					str_ptr = strstr(buf,"Connected at");
					login_pos = (int)(str_ptr-buf);
	
					found_tag = 1;
					tmp_cnt = 0;
					fgets(buf,BUF_SIZE,fp); // ignore ------------
					continue;
				}
			}
			else { // table body
				int prev_end = -1; // Catherine for test
				char *field = NULL; // Catherine for test
				int ret;

				if (st_size+1 > list_cnt && u_list != NULL) {
					fclose(fp);
					return ERROR_BUFFER_TOO_SMALL;
				}
				if (part == 1)
				{
					if (u_list != NULL) {
						int pos = 0;

						ptmp = realloc((void*)pc, sizeof(s_client_table) * (ct_size + 1));
						if (ptmp == NULL) {
							ret = ERROR_OUT_OF_MEMORY;
							goto part_one_error_out;
						}
						pc = (struct s_client_table*)ptmp;

	 					// PID
						if ((field = locate_my_field(buf, prev_end, pid_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_one_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						strcpy(pc[ct_size].pid, field);
						stripe_white_space_tail(pc[ct_size].pid);

						// User
						if ((field = locate_my_field(buf, prev_end, user_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_one_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						strcpy(pc[ct_size].user, field);
						stripe_white_space_tail(pc[ct_size].user);

						// Group -- not in use, but we need its end
						if ((field = locate_my_field(buf, prev_end, group_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_one_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						/* group not in use --> skip
						strcpy(pc[ct_size].group, field);
						stripe_white_space_tail(pc[ct_size].group); */

						// IP -- last field (may contain ' ')
						ip_pos = prev_end + 1;
						while(buf[ip_pos + pos] != ')')
						{
							if(buf[ip_pos + pos ] == '\n' || buf[ip_pos + pos] == '\0')
								break;
							pos++;
						}
					
						buf[ip_pos + pos] = '\0';
	
						while(buf[ip_pos] != '(')
						{
							if(buf[ip_pos] == '\n' || buf[ip_pos] == '\0')
								break;
							ip_pos++;
						}

						strcpy(pc[ct_size].ip,buf + ip_pos + 1);
						stripe_white_space_tail(pc[ct_size].ip);
						goto part_one_out;

					part_one_error_out:
						if (pc) free(pc);
						if (ps) free(ps);
						return ret;
					part_one_out: 
						;
					} // u_list not null

					ct_size++;
				} // part one finish
				else if (part == 2)
				{
					if (u_list != NULL) {
						ptmp = realloc((void*)ps, sizeof(s_service_table) * (st_size + 1));
						if (ptmp == NULL) {
							ret = ERROR_OUT_OF_MEMORY;
							goto part_two_error_out;
						}
						ps = (struct s_server_table *)ptmp;

						// Share (service)
						if ((field = locate_my_field(buf, prev_end, ser_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_two_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						strcpy(ps[st_size].service, field);
						stripe_white_space_tail(ps[st_size].service);
                                                //Shone added to add tail "..." after folder name having more than 20 char 
						{
							char *strptr=ps[st_size].service;
							int count=0;
							
							while(*(strptr+count) != 0x0) 
							       {
							       	 if (is_tc_word(strptr)||is_jis_word(strptr)
                                                                                       ||is_korea_word(strptr))
							                    count+=2;
							               else
							       	      count++;
							        }	
							if (count>20 ) 
							   {
							   	int flag=0;
							   	if (is_tc_word(strptr+19)||is_jis_word(strptr+19)
                                                                                         ||is_korea_word(strptr+19))
							   	   flag=1;
								 *(strptr+20+flag)='.';
								 *(strptr+21+flag)='.'; 
								 *(strptr+22+flag)='.';   
								 *(strptr+23+flag)=0x0;   
							   }
						}

						// PID
						if ((field = locate_my_field(buf, prev_end, pid_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_two_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						strcpy(ps[st_size].pid, field);
						stripe_white_space_tail(ps[st_size].pid);

						// Machine name
						if ((field = locate_my_field(buf, prev_end, from_pos, ' ')) == NULL) {
							ret = ERROR_FAIL;
							goto part_two_error_out;
						}
						prev_end = (int)(field + strlen(field) - buf);
						strcpy(ps[st_size].machine, field);
						stripe_white_space_tail(ps[st_size].machine);

						// Connect time
						field = buf + prev_end + 1;
						prev_end = (int)(field - buf);
						while (*(buf+prev_end) != '\0' && *(buf+prev_end) != '\n')
							prev_end++;
						*(buf+prev_end) = '\0';

						strcpy(ps[st_size].connect_time, field);
						stripe_white_space_tail(ps[st_size].connect_time);
						goto part_two_out;

					part_two_error_out:
						if (pc) free(pc);
						if (ps) free(ps);
						return ret;
					part_two_out: 
						;
					} // u_list not NULL

					st_size++;
				} // part two finish
			} // table body finish
		} // samba 2 or 3 ?
	}
	fclose(fp);

	// combine the client & service tables
	if (u_list != NULL) {
		for (i=0; i<ct_size; i++)
		{
			for (j=0; j<st_size; j++) {
				if (!strcmp(pc[i].pid, ps[j].pid))
				{
					strcpy((u_list+tmp_cnt)->service, ps[j].service);
					strcpy((u_list+tmp_cnt)->pid, pc[i].pid);
					strcpy((u_list+tmp_cnt)->user_name, pc[i].user);
					strcpy((u_list+tmp_cnt)->from_name, ps[j].machine);
					strcpy((u_list+tmp_cnt)->login_time, ps[j].connect_time);
					strcpy((u_list+tmp_cnt)->ip, pc[i].ip);
					tmp_cnt++;
				}
			}
		}
		if (pc) free(pc);
		if (ps) free(ps);
	}
	else
		tmp_cnt = st_size;

	if (found_tag == 0 )
		return ERROR_OPEN_FILE;
	if(u_list != NULL)
		qsort((void *)u_list,tmp_cnt,sizeof(SAMBA_STATUS_INFO),m_datacmp);
	return tmp_cnt;
}

/************************************************************
* Name : Create_System_User_Ex
* Description : a new version of Create_System_User;
*                         by the type, we can decide whether other fields are useful.
*                        (1) 'pdc' : to create a new user with only user_name & group_name valid
*                        (2) 'local' : to create a new user with every parameters
* Return : 
*            ERROR_INVALID_PARAM --> Invalid parameter
*            ERROR_ALREAY_EXIST --> User alreay exist
*            UG_ADD_LOCAL_USER --> Call useradd CMD fail
*            SUCCESS             --> Successful
************************************************************/
int Create_System_User_Ex(USER_DATA *usrinfo)
{
	char *ep;	// encrypted passwd buffer
	char buf[2*BUF_SIZE];
	char check_user_name[2*USER_NAME_LENGTH];

	// Check the user is exist or not ?
	if (Is_System_User(usrinfo->name)) return ERROR_ALREADY_EXISTED;

	if (EMPTY_STRING(usrinfo->group)) usrinfo->group = GROUP_EVERYONE;

	if (get_total_user_count() >= MAX_USER_NUMBER) // too many system users
		return ERROR_USERNUM_EXCEED_ACCOUNT;

	Str_Trans_CmdStr(usrinfo->name, check_user_name);
	switch (usrinfo->type) {
	    case NAS_USER_GROUP_PDC: {
#ifdef __QNAP_NAS_ADS__
	    		if (usrinfo->uid > 0)
	    			sprintf(buf,"%s -u %d -G \"%s\" -h / -H -g \"%s\" \"%s\" > /dev/null",
				ADDUSER_PATH, usrinfo->uid, GROUP_EVERYONE, NAS_USER_PDC_CMT, check_user_name);
	    		else
#endif // __QNAP_NAS_ADS__
			sprintf(buf,"%s -G \"%s\" -h / -H -g \"%s\" \"%s\" > /dev/null",
				ADDUSER_PATH, GROUP_EVERYONE, NAS_USER_PDC_CMT, check_user_name);

			if (system(buf) !=0) {
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex: system(%s) failed.\n", buf);
#endif
				return ERROR_EXEC_FAIL;
			}

			if (Add_NAS_User_To_Group(usrinfo->name, GROUP_EVERYONE) != SUCCESS) {
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex: fail to add user %s to group %s.\n", usrinfo->name, usrinfo->group);
#endif
			}

			if (compare_case_string(GROUP_EVERYONE, usrinfo->group)) {
				if (Add_NAS_User_To_Group(usrinfo->name, usrinfo->group) != SUCCESS) {
#ifdef DEBUG		// dump more error message...
					fprintf(stderr, "Create_System_User_Ex: fail to add user %s to group %s.\n", usrinfo->name, usrinfo->group);
#endif
				}
			}
		}
		break;
	    case NAS_USER_GROUP_LOCAL: {
			if (!Is_System_Group(usrinfo->group))	// create non-existed group automatically.
				Create_System_Group(usrinfo->group);
		
			// to get encrypted passwd
			if (EMPTY_STRING(usrinfo->passwd))
				ep = "";
			else
				ep = usrinfo->passwd;
				//ep = NASCRYPT(usrinfo->passwd, CRYPT_SALT_NO);
		
			// add local user  -M: doesn't create home DIR.
			if (EMPTY_STRING(usrinfo->home_dir))		// home set to / , no shell
				sprintf(buf,"%s -G \"%s\" -h / -H -p \"%s\" -g \"%s\" \"%s\" > /dev/null",
					ADDUSER_PATH, usrinfo->group, ep, NAS_USER_LOCAL_CMT, check_user_name);
			else
				sprintf(buf,"%s -G \"%s\" -h %s -p \"%s\" -g \"%s\" \"%s\" > /dev/null",
					ADDUSER_PATH, usrinfo->group, usrinfo->home_dir, ep, NAS_USER_LOCAL_CMT, usrinfo->name);
		
#ifdef DEBUG		// For checking if the buf overflow happened...
			if (strlen(buf) >= sizeof(buf)) fprintf(stderr, "Create_System_User_Ex: buf size too small.\n");
#endif

			if (system(buf) !=0) {
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex: system(%s) failed.\n", buf);
#endif
				return ERROR_EXEC_FAIL;
			}

			if (Add_NAS_User_To_Group(usrinfo->name, usrinfo->group) != SUCCESS) {   
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex: fail to add user %s to group %s.\n", usrinfo->name, usrinfo->group);
#endif
			}
		}
		break;

	    default:  return ERROR_INVALID_PARAM;
	}
	
	return SUCCESS;	
}

/************************************************************
* Name : NAS_Get_User_Type
* Description : to return the user type
* Return : 
*            NAS_USER_GROUP_PDC --> PDC user
*            NAS_USER_GROUP_LOCAL --> local user
*            NAS_USER_GROUP_BOTH --> both PDC & local user
*            NAS_USER_GROUP_NONE --> not exist
************************************************************/
NAS_USER_GROUP_TYPE NAS_Get_User_Type(char* user_name)
{

	struct passwd *p;
	int flag=NAS_USER_GROUP_NONE;
	
	setpwent();
	while ((p=getpwent())!=NULL) {
		if (compare_case_string(user_name, p->pw_name)==0) {
			if (compare_string(p->pw_gecos, NAS_USER_PDC_CMT)==0) 
				flag=NAS_USER_GROUP_PDC;
			else if (compare_string(p->pw_gecos, NAS_USER_BOTH_CMT)==0) 
				flag=NAS_USER_GROUP_BOTH;
			else
				flag=NAS_USER_GROUP_LOCAL;
			break;
		}
	}
	endpwent();

	return (flag);
}

/************************************************************
* Name : NAS_Get_Group_Type
* Description : to return the group type
* Return : 
*            NAS_USER_GROUP_PDC --> PDC group
*            NAS_USER_GROUP_LOCAL --> local group
*            NAS_USER_GROUP_BOTH --> both PDC & local group
*            NAS_USER_GROUP_NONE --> not exist
************************************************************/
NAS_USER_GROUP_TYPE NAS_Get_Group_Type(char* grp_name)
{
	int ret;
	NAS_USER_GROUP_TYPE type=NAS_USER_GROUP_LOCAL;
	char buf[BUF_SIZE];

	if (!Is_System_Group(grp_name)) return NAS_USER_GROUP_NONE;

	ret = Conf_Get_Field(PDC_GROUP_FILE, NULL, grp_name, buf, BUF_SIZE);
	
	if (ret==0)
		type = atoi(buf);

	return type;
}

//------------------------------------------------------------
//   Get All NAS Users Names
//   Return: ERROR_BUFFER_TOO_SMALL  --> you pass in buffer is too small
//           nnn          --> users number
//------------------------------------------------------------

int Get_NAS_User_List_Ex(char (**user_list_ptr)[USER_NAME_LENGTH])
{
	struct passwd *p;
	char (*user_list)[USER_NAME_LENGTH]=NULL;
	int count = 0;

	setpwent();
	while ((p = getpwent()) != NULL) {
		if (Is_Hidden_User(p->pw_name)) continue;
		
		count++;
		user_list = (char (*)[USER_NAME_LENGTH]) realloc(
				(void*)user_list,
				count * USER_NAME_LENGTH * sizeof(char));
		
		if (user_list==NULL) return ERROR_BUFFER_TOO_SMALL;

		strncpy((char *) (user_list + count -1), p->pw_name, USER_NAME_LENGTH-1);
		(*(user_list + count -1))[USER_NAME_LENGTH-1]=0x0;
		//Andy	05/26/03
	}	// end of while
	endpwent();    // close /etc/passwd

	// sort area
	Sort_Name_List((char **) user_list, USER_NAME_LENGTH, count);

	*user_list_ptr = user_list;
	return count;	// return total count of the users listed...
}


//--------------------------------------------------------------
//   Get All Local Group Name
//  Return:   ERROR_BUFFER_TOO_SMALL       --> You pass in smaller buffer
//            group count
//--------------------------------------------------------------
int Get_NAS_Group_List_Ex(char (**group_list_ptr)[GROUP_NAME_LENGTH])
{
	struct group *gp;
	char (*group_list)[GROUP_NAME_LENGTH]=NULL;
	int count = 0;

	setgrent();
	while ((gp = getgrent()) != NULL) {
		if (Is_Hidden_Group(gp->gr_name)) continue;
		// TBD...
		count++;
		group_list = (char (*)[GROUP_NAME_LENGTH]) realloc(
				(void*)group_list,
				count * GROUP_NAME_LENGTH * sizeof(char));

		if (group_list==NULL)  return ERROR_BUFFER_TOO_SMALL;
		
		strncpy(*(group_list +count - 1), gp->gr_name, GROUP_NAME_LENGTH-1);
		(*(group_list +count - 1))[GROUP_NAME_LENGTH-1]=0x0;
		//Andy	05/26/03
	}
	endgrent();


	Sort_Name_List((char **) group_list, GROUP_NAME_LENGTH, count);
	*group_list_ptr = group_list;
	return count;
}

//--------------------------------------------------------------
//   Free the allocated memory
//--------------------------------------------------------------
void Release_List(void* list_ptr)
{
	free(list_ptr);
}

//--------------------------------------------------------------
// Get_Grp_Info_Ex(): get one specified group's information
//  & all its mmbers
//  Return:   ERROR_NOT_FOUND    --> The group not exist
//            ERROR_EXEC_FAIL    --> call system function fail
//            SUCCESS           --> Successful
//--------------------------------------------------------------
int Get_NAS_Group_Info_Ex(NAS_GROUP_INFO *group_info_ptr)
{
	struct group *pg;
	int i,n, cnt, bFound;
	char (*user_list)[USER_NAME_LENGTH]=NULL;
	struct passwd *user;

	if (SUCCESS!=Get_Exact_System_Group_Name(group_info_ptr->group_name))
		return ERROR_NOT_EXISTED;

	if ((pg = getgrnam(group_info_ptr->group_name)) == NULL) return ERROR_EXEC_FAIL;

	group_info_ptr->gid = pg->gr_gid;

	i=n=0;
	while (*(pg->gr_mem + i) != NULL) {
		if (!Is_Hidden_User(*(pg->gr_mem + i))) {
			user_list = (char (*)[USER_NAME_LENGTH]) realloc(
					(void*)user_list,
					(n+1)*USER_NAME_LENGTH*sizeof(char));
			strncpy((char *) (user_list + n), *(pg->gr_mem + i), USER_NAME_LENGTH-1);
			(*(user_list + n))[USER_NAME_LENGTH-1]=0x0;
			//Andy	05/26/03
			n++;
		}
		i++;
	}
	
	// 2004.12.09, Johnson Cheng add for multi-domain AD
	setpwent();
	while((user = getpwent()) != 0)
	{
		if((user->pw_gid == group_info_ptr->gid) && !Is_Reserved_User(user->pw_name))
		{
			bFound = 0;
			for(cnt = 0; cnt < n; cnt++)
			{
				if(compare_case_string(user->pw_name, (char *) (user_list + cnt)) == 0)
				{
					bFound = 1;
					break;
				}
			}
			if(!bFound)
			{
				user_list = (char (*)[USER_NAME_LENGTH]) realloc((void*)user_list, (n+1)*USER_NAME_LENGTH*sizeof(char));
				strncpy((char *) (user_list + n), user->pw_name, USER_NAME_LENGTH-1);
				(*(user_list + n))[USER_NAME_LENGTH-1]=0x0;
				n++;
			}
		}
	}
	endpwent();
	
	group_info_ptr->user_list = user_list;
	group_info_ptr->user_count = n;
	return SUCCESS;
}

//------------------------------------------------------------
// Get one goups's members
// It will call Get_Grp_Info
//  Return:   ERROR_NOT_FOUND    --> The group not exist
//            ERROR_EXEC_FAIL    --> call system function fail
//            ERROR_BUFFER_TOO_SMALL     --> You pass in smaller buffer
//            nnn (member cnt)--> Successful
//------------------------------------------------------------
int Get_NAS_User_List_Of_Group_Ex(char *group_name, char (**user_list_ptr)[USER_NAME_LENGTH])
{
	NAS_GROUP_INFO gi;
	int ret;
	char groupname[GROUP_NAME_LENGTH];

	strncpy(groupname, group_name, GROUP_NAME_LENGTH-1);
	groupname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	
	if(EMPTY_STRING(groupname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(groupname); 	// Remove the extra blank in front, middle and back
	// 2004.12.09, Johnson Cheng marked for multi-domain AD
	//if (!Is_Legal_String(groupname))	// check whether the name is legal or not
	//	return ERROR_ILLEGAL_NAME;
	if (!Is_System_Group(groupname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	strncpy(gi.group_name, groupname, GROUP_NAME_LENGTH-1);
	(gi.group_name)[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	gi.user_count = 0;

	if ((ret = Get_NAS_Group_Info_Ex(&gi)) != SUCCESS)
		return ret;

	Sort_Name_List((char **)gi.user_list, USER_NAME_LENGTH, gi.user_count);
	
	*user_list_ptr=gi.user_list;
	return (gi.user_count);
}

//------------------------------------------------------------------
//  Get one specified local user's groups, it will call getgrent()
//  If successful, the first member of the grp_list_ptr is the
//   primary group.
//  Return:   ERROR_NOT_FOUND      --> User doesn't exist
//            ERROR_EXEC_FAIL      --> call system function fail
//            nnn               --> The groups he has
//------------------------------------------------------------------
int Get_NAS_Group_List_Of_User_Ex(char *user_name, char (**group_list_ptr)[GROUP_NAME_LENGTH])
{
	struct group *gp;
	int i = 0, j;
	char username[USER_NAME_LENGTH];
	char (*group_list)[GROUP_NAME_LENGTH]=NULL;
	char *group = NULL;
	
	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	/*Marked by Shone (2004,12,15)
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	*/	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// 2004.12.09, Johnson Cheng add for multi-domain AD
	group = Get_Group_Name_By_User_Name(user_name);
	if(group)
	{
		group_list=(char (*)[GROUP_NAME_LENGTH])realloc((void*)group_list, (i+1)*GROUP_NAME_LENGTH*sizeof(char));
		strncpy(*(group_list + i), group, GROUP_NAME_LENGTH-1);
		(*(group_list+i))[GROUP_NAME_LENGTH-1]=0x0;
		i++;
	}
	
	setgrent();
	while ((gp = getgrent()) != NULL) {
		// 2004.12.09, Johnson Cheng add for multi-domain AD
		if(compare_case_string(group, gp->gr_name) == 0)
			continue;
		j = 0;
		while (*(gp->gr_mem + j) != NULL) {	// search all groups
			if (compare_case_string(*(gp->gr_mem+j),username)==0) {
				// find user_name in this group
				if (Is_Hidden_Group(gp->gr_name)) continue;

				group_list=(char (*)[GROUP_NAME_LENGTH])realloc(
						(void*)group_list,
						(i+1)*GROUP_NAME_LENGTH*sizeof(char));
				strncpy(*(group_list + i), gp->gr_name, GROUP_NAME_LENGTH-1);
				(*(group_list+i))[GROUP_NAME_LENGTH-1]=0x0;
				//Andy	05/26/03
				
				i++;
				break;
			}
			j++;
		}
	}
	endgrent();
	if (i > 0) Sort_Name_List((char **) group_list, GROUP_NAME_LENGTH, i);
	*group_list_ptr = group_list;
	return i;
}

int int2str(int value, char* buff, int buff_size)
{
	
	int base = 10, remainder, quotient = value, i=0;   	 
   	memset(buff, '\0', sizeof(buff));

   	 if(value==0)
	{
// Cat	  	strcpy(buff, "0");
		memset(buff, '0', buff_size);
		return 1;
	}
	
	for(i=buff_size-1; i>=0 ; i--)
	{
		if (quotient)
			remainder=quotient%base;		
		buff[i]=(!quotient)?'0':'0'+remainder;
		quotient/=base;
	}	
	return 1;
}

void Free_Mutiple_Exist_user(MUTIPLE_EXIST_USER* user)
{
	MUTIPLE_EXIST_USER* tmp;

	tmp=user;
	while (user)
	{
		tmp=(MUTIPLE_EXIST_USER*)user->next;		
		free(user);
		user=tmp;
	}
}

MUTIPLE_EXIST_USER* Create_NAS_Mutiple_Users(char* prefix_name, char* account, char* passwd, int num, int* ret)
{
	char* str;
	char (*user_list)[USER_NAME_LENGTH]=NULL;
	MUTIPLE_EXIST_USER* exist_user=NULL,*start_user=NULL;
	int i,j,n=0, e=0;
	unsigned long quota_size;
	char buf[BUF_SIZE];

	*ret = SUCCESS;
	str = calloc(1, strlen(account));
	i = atoi(account);
	
#ifdef DEBUG	
	printf("len is %d\n", strlen(account));
#endif	
	if (!Is_Legal_String(prefix_name)){	
		*ret =  ERROR_ILLEGAL_NAME;
		return NULL;
	}	
	// Catherine 2003/08/15 ==>
	sprintf(buf, "%d", i + num - 1);
	if (strlen(buf) > strlen(account)) {
		char format[BUF_SIZE];

		sprintf(format, "%%0%dd", strlen(buf));
		sprintf(account, format, i);
	}
	// <== Catherine 2003/08/15

	for(j=0;j<num;j++){
#ifdef DEBUG		
		printf("int %d = %d\n", i+j,i+j);
#endif		
		int2str(i+j, str, strlen(account));
#ifdef DEBUG		
		printf("str %d = \"%s\"\n",j, str);
#endif		
		user_list = (char (*)[USER_NAME_LENGTH]) realloc(
					(void*)user_list,
					(n+1)*USER_NAME_LENGTH*sizeof(char));
		strncpy((char *) (user_list + n), prefix_name, USER_NAME_LENGTH-1);
		(*(user_list+n))[USER_NAME_LENGTH-1]=0x0;
		strncat((char *) (user_list + n), str, USER_NAME_LENGTH-1);
		(*(user_list+n))[USER_NAME_LENGTH-1]=0x0;
		//Andy	05/26/03
#ifdef DEBUG		
		printf("user_list %d = \"%s\"\n",j, *(user_list+n));
#endif 		
		if (Create_NAS_User((char *)(user_list + n), passwd) == ERROR_ALREADY_EXISTED){
			if (e==0){
				exist_user=calloc(1, sizeof(MUTIPLE_EXIST_USER));
				
				start_user=exist_user;
				e++;
			}
			else{
				exist_user->next=calloc(1, sizeof(MUTIPLE_EXIST_USER));
				
				exist_user=(MUTIPLE_EXIST_USER*)exist_user->next;
			}							
			strncpy(exist_user->exist_user, (char *)(user_list + n), USER_NAME_LENGTH-1);						
			(exist_user->exist_user)[USER_NAME_LENGTH-1]=0x0;
			//Andy	05/26/03
		}
		else{
			if (Is_User_Quota_Enabled(&quota_size))
				Set_One_User_Quota((char *)(user_list + n), quota_size, 0, 0);
			else
				Set_One_User_Quota((char *)(user_list + n), 0, 0, 0);
		}	
		n++;		
	}	
	free(str);
	
	Release_List(user_list);
	return (MUTIPLE_EXIST_USER*)start_user;
}

#ifdef	LARGE_NO_OF_USERS_SUPPORT	// Catherine 2003/06/18

// is_new_user_allowed
///////////////////////////////////////////////////
// description: 3 conditions must be fulfilled ---
//	1. 'usrname' is not a reserved name
//	2. 'usrname' doesn't exist
//	3. user count < MAX_USER_NUMBER
///////////////////////////////////////////////////
BOOL is_new_user_allowed(char* usrname)
{
	struct passwd *pwd;
	BOOL bRet = FALSE;
	int usercnt = 0;

	if (usrname == NULL || Is_Reserved_User(usrname))
		return FALSE;

	setpwent();
	while ((pwd = getpwent()) != NULL) {
		usercnt ++;
		if (compare_case_string(usrname, pwd->pw_name)==0) {
			bRet = TRUE;
			break;
		}
	}
	endpwent();		// close /etc/passwd file

	// already exist
	if (bRet) return FALSE;

	// too many users
	if (usercnt >= MAX_USER_NUMBER) return FALSE;

	return TRUE;
}

BOOL is_digital_string(char* str, int strmaxlen)
{
	int i = 0;

#ifdef	DEBUG
printf("is_digital_string (%s, %d) starts!\n", str, strmaxlen);
#endif
	if (str == NULL || strmaxlen <= 0)
		return FALSE;
#ifdef	DEBUG
printf("is_digital_string step 1\n");
#endif
	while (i<strmaxlen && (uint) *(str+i) != 0x0 &&
		(uint) *(str+i) != 0xa && (uint) *(str+i) != 0xd)
	{
#ifdef	DEBUG
printf("\tchar %d value = 0x%x\n", i, (uint) *(str+i));
#endif

		if ((i > 0 || (uint) *(str+i) != '-') && !isdigit(*(str+i)))
			return FALSE;
		i++;
	}
#ifdef	DEBUG
printf("is_digital_string step 2 : i=%d\n", i);
#endif

	if (i==0) return FALSE;

	return TRUE;
}

int Get_Progress_From_Status_File(char* stsfile, PROGRESS_STATUS* psts)
{
	int result = SUCCESS;
	FILE *fp;
	char line[BUF_SIZE];
	int found = 0;

	if (stsfile == NULL || psts == NULL)
		return ERROR_INVALID_PARAM;

#ifdef	DEBUG
	printf("Get_Progress_From_Status_File(%s, %d, %d, %d) starts!\n",
		stsfile, psts->status, (psts->plist == NULL) ? -1 : psts->plist->progress,
		(psts->plist == NULL || psts->plist->next==NULL)? -1 : psts->plist->next->progress);
#endif
	if (NAS_File_Lock(stsfile, O_RDONLY)==0)
		return ERROR_OPEN_FILE;

#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 1!\n");
#endif
	if ((fp = fopen(stsfile, "r"))==NULL)
	{
		NAS_File_Unlock(stsfile);
		return ERROR_OPEN_FILE;
	}
#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 2!\n");
#endif
	while (fgets(line, sizeof(line), fp)) {
#ifdef	DEBUG
		printf("\tstep 2.1 : line len = %d [%s]\n", strlen(line), line);
		printf("\t\tis_digital_string(%s, %d) == TRUE ? %s\n",
			line, sizeof(line),
			(is_digital_string(line, sizeof(line))==TRUE) ? "YES" : "NO");
#endif
		if (strlen(line) == 0)
			continue;

		if (is_digital_string(line, sizeof(line))) {
#ifdef	DEBUG
			printf("\tstep 2.2 : a digital line\n");
#endif
			psts->status = atoi(line);
			found ++;
			break;
		}
	}
#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 3!\n");
#endif
	if (found) {
		PROGRESS_LIST	*ptr1, *ptr2;

#ifdef	DEBUG
		printf("\tstep 3.1\n");
#endif
		ptr1 = psts->plist;
		while (ptr1 != NULL) {
			ptr2 = (PROGRESS_LIST*) ptr1->next;
			free(ptr1);
			ptr1 = ptr2;
		}
		psts->plist = NULL;
		ptr1 = NULL;
#ifdef	DEBUG
		printf("\tstep 3.2\n");
#endif
		while (fgets(line, sizeof(line), fp)) {
#ifdef	DEBUG
			printf("\tstep 3.2.1 : line [%s]\n", line);
#endif
			if (strlen(line) == 0)
				continue;
#ifdef	DEBUG
			printf("\tstep 3.2.2\n");
#endif
			if (is_digital_string(line, sizeof(line))) {
#ifdef	DEBUG
				printf("\tstep 3.2.2.1 : a digital line\n");
#endif
				if (ptr1 == NULL) {
					ptr1 = (PROGRESS_LIST*) calloc(1, sizeof(PROGRESS_LIST));
					if (ptr1 == NULL) {
#ifdef	DEBUG
						printf("get_progress_from_status_file("
							"%s) : fails to allocate memory!\n",
							stsfile);	
#endif
						result = ERROR_OUT_OF_MEMORY;
						goto get_progress_from_status_file_out;
					}
					psts->plist = ptr1;
				}
				else {
					ptr2 = (PROGRESS_LIST*) calloc(1, sizeof(PROGRESS_LIST));
					if (ptr2 == NULL) {
#ifdef	DEBUG
						printf("get_progress_from_status_file("
							"%s) : fails to allocate memory!\n",
							stsfile);	
#endif
						result = ERROR_OUT_OF_MEMORY;
						goto get_progress_from_status_file_out;
					}
					ptr1->next = ptr2;
					ptr1 = ptr2;
				}
				ptr1->progress = atoi(line);
				ptr1->next = NULL;	// dummy, just in case
			}
		}
	}

get_progress_from_status_file_out:
#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 4\n");
#endif
	fclose(fp);
	NAS_File_Unlock(stsfile);
#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 5\n");
#endif
	if (result != SUCCESS) {
		//free the allocated memory
		PROGRESS_LIST	*ptr = psts->plist;

		while (psts->plist != NULL) {
			psts->plist = ptr->next;
			free(ptr);
			ptr = psts->plist;
		}
		psts->plist = NULL;
	}
#ifdef	DEBUG
	printf("Get_Progress_From_Status_File step 6: result=%d\n", result);
#endif
	return result;
}

int Report_Progress_To_File(char* stsfile, PROGRESS_STATUS* pslist)
{
	int result = SUCCESS;
	FILE *fp;
	char tmpstr[BUF_SIZE];
	PROGRESS_LIST* tmp_ptr;

	if (stsfile == NULL || pslist == NULL)
		return ERROR_INVALID_PARAM;

	if (NAS_File_Lock(stsfile, O_RDWR)==0)
		return ERROR_OPEN_FILE;

	if ((fp = fopen(stsfile, "w+"))==NULL)
	{
		NAS_File_Unlock(stsfile);
		return ERROR_OPEN_FILE;
	}

	// piece the status and the progress percentages together
	sprintf(tmpstr, "%d", pslist->status);
	tmp_ptr = pslist->plist;
	while (tmp_ptr) {
		sprintf(tmpstr, "%s\n%d", tmpstr, tmp_ptr->progress);
		tmp_ptr = tmp_ptr->next;
	}
	strcat(tmpstr, "\n");

	if (fputs(tmpstr, fp) < 0) {
		result = ERROR_WRITE_FILE;
	}

	fclose(fp);
	NAS_File_Unlock(stsfile);
	return result;
}

int Create_System_User_Ex_2(USER_DATA *usrinfo, BOOL update_flash)
{
	char *ep;	// encrypted passwd buffer
	char buf[2*BUF_SIZE];
	char check_user_name[2*USER_NAME_LENGTH];

	// Check the user is exist or not ?
	if (Is_System_User(usrinfo->name)) return ERROR_ALREADY_EXISTED;

	if (EMPTY_STRING(usrinfo->group)) usrinfo->group = GROUP_EVERYONE;

	if (get_total_user_count() >= MAX_USER_NUMBER) // too many system users
		return ERROR_USERNUM_EXCEED_ACCOUNT;

	Str_Trans_CmdStr(usrinfo->name, check_user_name);
	switch (usrinfo->type) {
	    case NAS_USER_GROUP_PDC: {
#ifdef __QNAP_NAS_ADS__
			if (usrinfo->uid > 0)
				sprintf(buf,"%s -u %d -G \"%s\" -h / "
					"-H -g \"%s\" \"%s\" > /dev/null",
					ADDUSER_PATH, usrinfo->uid, GROUP_EVERYONE, NAS_USER_PDC_CMT,
					check_user_name);
			else			
#endif // __QNAP_NAS_ADS__
			sprintf(buf,"%s -G \"%s\" -h / "
				"-H -g \"%s\" \"%s\" > /dev/null",
				ADDUSER_PATH, GROUP_EVERYONE, NAS_USER_PDC_CMT,
				check_user_name);

			if (system(buf) !=0) {
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex_2:"
					" system(%s) failed.\n", buf);
#endif
				return ERROR_EXEC_FAIL;
			}

			if (Add_NAS_User_To_Group_Ex(usrinfo->name, GROUP_EVERYONE,
				update_flash) != SUCCESS)
			{
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex_2:"
					" fail to add user %s to group %s.\n",
					usrinfo->name, usrinfo->group);
#endif
			}

			if (compare_case_string(GROUP_EVERYONE, usrinfo->group)) {
				if (Add_NAS_User_To_Group_Ex(usrinfo->name,
					usrinfo->group, update_flash) != SUCCESS)
				{
#ifdef DEBUG		// dump more error message...
					fprintf(stderr, "Create_System_User_Ex_2:"
						" fail to add user %s to group %s.\n",
						usrinfo->name, usrinfo->group);
#endif
				}
			}
		}
		break;
	    case NAS_USER_GROUP_LOCAL: {
			if (!Is_System_Group(usrinfo->group))	// create non-existed group automatically.
				Create_System_Group(usrinfo->group);
		
			// to get encrypted passwd
			if (EMPTY_STRING(usrinfo->passwd))
				ep = "";
			else
				ep = usrinfo->passwd;
				//ep = NASCRYPT(usrinfo->passwd, CRYPT_SALT_NO);
		
			// add local user  -M: doesn't create home DIR.
			if (EMPTY_STRING(usrinfo->home_dir))		// home set to / , no shell
				sprintf(buf,"%s -G \"%s\" -h /"
					" -H -p \"%s\" -g \"%s\" \"%s\" > /dev/null",
					ADDUSER_PATH, usrinfo->group, ep,
					NAS_USER_LOCAL_CMT, check_user_name);
			else
				sprintf(buf,"%s -G \"%s\" -h %s -p \"%s\" -g"
					" \"%s\" \"%s\" > /dev/null",
					ADDUSER_PATH, usrinfo->group,
					usrinfo->home_dir, ep, NAS_USER_LOCAL_CMT,
					usrinfo->name);
		
#ifdef DEBUG		// For checking if the buf overflow happened...
			if (strlen(buf) >= sizeof(buf))
				fprintf(stderr, "Create_System_User_Ex_2:"
					" buf size too small.\n");
#endif

			if (system(buf) !=0) {
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex_2:"
					" system(%s) failed.\n", buf);
#endif
				return ERROR_EXEC_FAIL;
			}

			if (Add_NAS_User_To_Group_Ex(usrinfo->name, usrinfo->group,
				update_flash) != SUCCESS)
			{
#ifdef DEBUG		// dump more error message...
				fprintf(stderr, "Create_System_User_Ex_2:"
					" fail to add user %s to group %s.\n",
					usrinfo->name, usrinfo->group);
#endif
			}
		}
		break;

	    default:  return ERROR_INVALID_PARAM;
	}
	
	return SUCCESS;	
}

int Create_NAS_User_Ex(char *user_name, char *pass_wd, BOOL update_flash)
{
	int ret = SUCCESS;
	char group_name[GROUP_NAME_LENGTH];
	char username[USER_NAME_LENGTH];
	USER_DATA udata;

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03

	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	if (Is_Reserved_User(username))	// check whether the name is reserved for the system
		return ERROR_RESERVED_USER;
	if (Is_System_User(username))		// check whether the name exists or not 
		return ERROR_ALREADY_EXISTED;

	strcpy(group_name, GROUP_EVERYONE);

	udata.name = username;
	udata.passwd = pass_wd;
	udata.group = group_name;
	udata.home_dir = NULL;
	udata.type = NAS_USER_GROUP_LOCAL;
#ifdef __QNAP_NAS_ADS__
	udata.uid = -1;
#endif
	ret = Create_System_User_Ex_2(&udata, update_flash);
	if (ret == SUCCESS) {
		ret = Create_Samba_User(username, pass_wd);
		if (ret==SUCCESS)
			ret = change_netware_user_password(username, pass_wd);
	}

	if (update_flash) {
#ifdef	DEBUG
		printf("Create_NAS_User_Ex update flash files\n"
			"\t%s\n\t%s\n\t%s\n\t/etc/config/nwpwd\n",
			PASS_CONF_FILE, GROUP_CONF_PATH, SAMBA_PASS_PATH);
#endif
		//Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		//Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		//Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
		//Update_Flash_Data_Ex("/etc/config/nwpwd", TRUE);
	}

	return ret;
}

int Remove_NAS_User_Ex(char *user_name, BOOL update_flash)
{
	int ret;
	NAS_USER_GROUP_TYPE type;

	if (user_name == NULL)
		return ERROR_INVALID_PARAM;
#ifdef	DEBUG
	printf("Remove_NAS_User_Ex(%s, %d) starts.\n",
		user_name, update_flash);
#endif
	if (Is_Reserved_User(user_name)) return ERROR_RESERVED_USER;

#ifdef	DEBUG
	printf("Remove_NAS_User_Ex step 1\n");
#endif
	type = NAS_Get_User_Type(user_name);

#ifdef	DEBUG
	printf("Remove_NAS_User_Ex step 2 : type = %d\n", type);
#endif
	if (type==NAS_USER_GROUP_NONE)
		return ERROR_NOT_EXISTED;
	else if (type==NAS_USER_GROUP_PDC)
		ret = SUCCESS;
	else  { // remove samba user passwd first 
		ret = Remove_Samba_User(user_name);
		ret = remove_netware_user(user_name);
	}
#ifdef	DEBUG
	printf("Remove_NAS_User_Ex step 3\n");
#endif
	Remove_User_Share_Privilege_Ex(user_name, update_flash);
#ifdef	DEBUG
	printf("Remove_NAS_User_Ex step 4\n");
#endif
	ret = Remove_System_User(user_name);

	if (update_flash) {
#ifdef	DEBUG
		printf("Remove_NAS_User_Ex step 4.5\n");
#endif
		//Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		//Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		//Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
		//Update_Flash_Data_Ex("/etc/config/nwpwd", TRUE);
	}
#ifdef	DEBUG
	printf("Remove_NAS_User_Ex step 5\n");
#endif
	return ret;
}

int Change_NAS_User_Password_Ex(char *user_name, char *pass_wd, BOOL update_flash)
{
	int ret;
	char username[USER_NAME_LENGTH];
	NAS_USER_GROUP_TYPE type;

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;	//Andy	05/26/03

	if(EMPTY_STRING(user_name)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
	
	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;

	type = NAS_Get_User_Type(username);

	if (type==NAS_USER_GROUP_PDC)
		return ERROR_PDC_USER_GROUP;	// PDC users are not allowed to change the password

	ret = Change_System_User_Password(username, pass_wd);
	if (ret == SUCCESS) {
		ret = Change_Samba_User_Passwd(username, pass_wd);
		if (ret == SUCCESS)
			ret = change_netware_user_password(username, pass_wd);
	}

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
		Update_Flash_Data_Ex("/etc/config/nwpwd", TRUE);
	}*/

	return ret;
}

int Create_NAS_Group_Ex(char *group_name, BOOL update_flash)
{
	int ret;
	char groupname[GROUP_NAME_LENGTH];

	strncpy(groupname, group_name, GROUP_NAME_LENGTH-1);
	groupname[GROUP_NAME_LENGTH-1]=0x0;	//Andy	05/26/03
	
	if(EMPTY_STRING(groupname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(groupname); 	// Remove the extra blank in front, middle and back
	if (!Is_Legal_String(groupname))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	if (Is_Reserved_Group(groupname))	// check whether the name is reserved for the system
		return ERROR_RESERVED_GROUP;
	if (Is_System_Group(groupname))	// check whether the name exists or not 
		return ERROR_ALREADY_EXISTED;

	ret = Create_System_Group(groupname);

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
	}*/

	return ret;
}

int Remove_NAS_Group_Ex(char *group_name, BOOL update_flash)
{
	int ret;

	ret = Remove_System_Group(group_name);
	if (ret == SUCCESS) {
		ret = Remove_Group_Share_Privilege(group_name);
	}

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
	}*/

	return ret;
}

int Change_NAS_Group_Name_Ex(char *old_group_name, char *new_group_name, BOOL update_flash)
{
	int ret;

	ret = Change_System_Group_Name(old_group_name, new_group_name);
	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
	}*/
	return ret;
}

int Add_NAS_Group_To_User_Ex(char *group_name, char *user_name, BOOL update_flash)
{
	return	Add_NAS_User_To_Group_Ex(user_name, group_name, update_flash);
}

int Add_NAS_User_To_Group_Ex(char *user_name, char *grp_name, BOOL update_flash)
{
	int str_len,name_len,fld_len,new_len;
	int result;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	// check user name
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check group name
	if(EMPTY_STRING(grpname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(grpname))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;

	if (!Is_System_Group(grpname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check the user_name belongs to the grp_name member already?
	if (User_Belongs_To_Group(username, grpname))
		return SUCCESS;

	// Set string comparision with case insensitive
	Conf_Case_Sensitive(0);
	name_len =  strlen (username);

	if( (str_len = SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,NULL,0)) > 0) {
		char out_str[str_len];
		SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,out_str,str_len);
		if((fld_len = Get_String_Field (out_str,4,':',NULL,0)) > 0) {
			char fld_str[fld_len];
			char set_str[fld_len+name_len];
			Get_String_Field (out_str,4,':',fld_str,fld_len);

			if( compare_case_string(fld_str,"") == 0 ) //no member in this group
				sprintf(set_str,"%s%s",fld_str,username);
			else
				sprintf(set_str,"%s,%s",fld_str,username);

			if ( (new_len = Set_String_Field(out_str,4,':',set_str,NULL,0)) > 0) {
				char new_str[new_len];
				Set_String_Field(out_str,4,':',set_str,new_str,new_len);
				result = SYS_Set_One_String(GROUP_CONF_PATH,1,':',grpname,new_str);
			}
		}
		else
			result = fld_len;
	}
	else
		result = str_len;

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
	}*/

	return result;
}

int Remove_NAS_Group_From_User_Ex(char *group_name, char *user_name, BOOL update_flash)
{
	return Remove_NAS_User_From_Group_Ex(user_name, group_name, update_flash);
}

int Remove_NAS_User_From_Group_Ex(char *user_name, char *grp_name, BOOL update_flash)
{
 	char buf[USER_NAME_LENGTH];
	int str_len,name_len,fld_len,new_len;
	int result;
	int i=0;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];

	strncpy(username, user_name, USER_NAME_LENGTH-1);
	username[USER_NAME_LENGTH-1]=0x0;
	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;
	//Andy	05/26/03
	// check user name
	if(EMPTY_STRING(username)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(username))	// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_User(username))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check group name
	if(EMPTY_STRING(grpname)) 		// Null string check
		return ERROR_EMPTY_STRING;
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back

	if (!Is_Legal_String(grpname))		// check whether the name is legal or not
		return ERROR_ILLEGAL_NAME;
	
	if (!Is_System_Group(grpname))		// check whether the name exists or not 
		return ERROR_NOT_EXISTED;

	// check the user_name belongs to the grp_name member already?
	if (! User_Belongs_To_Group(username, grpname))
		return ERROR_NOT_EXISTED;

	name_len =  strlen (username);
	// Set string comparision with case insensitive
	Conf_Case_Sensitive(0);

	if( (str_len = SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,NULL,0)) > 0) {
		char out_str[str_len];
		SYS_Get_One_String(GROUP_CONF_PATH,1,':',grpname,out_str,str_len);
		if((fld_len = Get_String_Field (out_str,4,':',NULL,0)) > 0) {
			char fld_str[fld_len];
			Get_String_Field (out_str,4,':',fld_str,fld_len);
			do {
				i++;
				Get_String_Field(fld_str,i,',',buf,USER_NAME_LENGTH);
				if( compare_case_string(username,buf) == 0 ) break;
				if ( i>65534 ) return ERROR_NOT_EXISTED; // prevent system crazy
			} while(1);

			Remove_String_Field(fld_str,i,',');

			if ( (new_len = Set_String_Field(out_str,4,':',fld_str,NULL,0)) > 0) {
				char new_str[new_len];
				Set_String_Field(out_str,4,':',fld_str,new_str,new_len);
				result = SYS_Set_One_String(GROUP_CONF_PATH,1,':',grpname,new_str);
			}

		}
		else
			result = fld_len;
	}
	else
		result = str_len;

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
	}*/

	return result;
}

MUTIPLE_EXIST_USER* Create_NAS_Multiple_Users_Ex(
	char* prefix_name, char* account, char* passwd,
	int num, int* ret, BOOL update_flash, char* stsfile)
{
	char group_name[GROUP_NAME_LENGTH] = {GROUP_EVERYONE};
	char username[USER_NAME_LENGTH], prefix[USER_NAME_LENGTH];
	char *str = NULL, *ep;
	MUTIPLE_EXIST_USER* exist_user=NULL, *start_user=NULL;
	int i, j, e=0, result, cnt, name_len;
	unsigned long quota_size;
	char buf[BUF_SIZE];
	PROGRESS_STATUS mystatus;

	*ret = SUCCESS;
	i = atoi(account);

	if (stsfile) {
		mystatus.plist = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (mystatus.plist == NULL) {
#ifdef	DEBUG
			printf("Create_NAS_Multiple_Users_Ex: fails to allocate memory!\n");
#endif
			*ret = ERROR_OUT_OF_MEMORY;
			mystatus.status = *ret;
			Report_Progress_To_File(stsfile, &mystatus);
			return NULL;
		}

		mystatus.plist->progress = 0;
		mystatus.plist->next = NULL;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(stsfile, &mystatus);
	}

	if (!Is_Legal_String(prefix_name)) {
		*ret =  ERROR_ILLEGAL_NAME;
		goto create_nas_multiple_users_ex_out;
	}	
	Remove_Blank_From_String(prefix_name); 	// Remove the extra blank in front, middle and back

	// Catherine 2003/08/15 ==>
	sprintf(buf, "%d", i + num - 1);
	if (strlen(buf) > strlen(account)) {
		char format[BUF_SIZE];

		sprintf(format, "%%0%dd", strlen(buf));
		sprintf(account, format, i);
	}
	// <== Catherine 2003/08/15

	if (strlen(prefix_name) + strlen(account) >= USER_NAME_LENGTH) {
		*ret = ERROR_BUFFER_TOO_SMALL;
		goto create_nas_multiple_users_ex_out;
	}

	Str_Trans_CmdStr(prefix_name, prefix);

	// to get encrypted passwd
	if (EMPTY_STRING(passwd))
		ep = "";
	else
		ep = passwd;
		//ep = NASCRYPT(passwd, CRYPT_SALT_NO);

	str = (char*) calloc(1, strlen(account) + 1);
	for(j=0;j<num;j++){
		if (stsfile) {
			mystatus.plist->progress = 100*j/num;
			mystatus.status = S_PROCESSING;
			Report_Progress_To_File(stsfile, &mystatus);
		}

		int2str(i + j, str, strlen(account));
		strcpy(username, "");
		snprintf(username, USER_NAME_LENGTH, "%s%s", prefix, str);
		name_len = strlen(username);

		if (is_new_user_allowed(username))
		{
			// 1 -- add system user
			sprintf(buf, "%s -G \"%s\" -h / -H -p \"%s\" "
				"\"%s\" > /dev/null",
				ADDUSER_PATH, group_name, ep, username);

			if (system(buf) != 0)
				result = ERROR_EXEC_FAIL;
			else
				result = SUCCESS;

			// 2 -- add the new user to group "everyone"
			if( result == SUCCESS &&
				(cnt = SYS_Get_One_String(GROUP_CONF_PATH,1, ':',
					group_name, NULL, 0)) > 0 )
			{
				char out_str[cnt];
				int fld_len;

				SYS_Get_One_String(GROUP_CONF_PATH, 1, ':', group_name,
					out_str, cnt);
				if((fld_len = Get_String_Field (out_str, 4,':', NULL, 0)) > 0)
				{
					char fld_str[fld_len];
					char set_str[fld_len + name_len];
					int new_len;

					// get group members
					Get_String_Field (out_str,4,':',fld_str,fld_len);

					// no member in this group ??
					if( compare_case_string(fld_str,"") == 0 )
						sprintf(set_str,"%s%s",fld_str,username);
					else
						sprintf(set_str,"%s,%s",fld_str,username);

					// set group members
					if ( (new_len = Set_String_Field(out_str,4,':',set_str,NULL,0)) > 0) {
						char new_str[new_len];

						Set_String_Field(out_str, 4, ':', set_str, new_str, new_len);
						result = SYS_Set_One_String(
							GROUP_CONF_PATH, 1, ':',
							group_name, new_str);
					}
				}
				else	// cannot get group members of "everyone"
					result = (fld_len == SUCCESS) ? ERROR_FAIL : fld_len;
			}
			else // cannot get group members' length of "everyone"
				result = (cnt == SUCCESS) ? ERROR_FAIL : cnt;

			if (result == SUCCESS) {
				// 3 --  add samba user
				result = Create_Samba_User(username, passwd);
				if (result==SUCCESS) {
					// 4 -- add Novell user
					result = change_netware_user_password(username, passwd);
				}
			}

			// 5 -- add quota control
			if (Is_User_Quota_Enabled(&quota_size))
				Set_One_User_Quota(username, quota_size, 0, 0);
		}
		else {	// user already exists in NAS or exceed max system user count
			if (e==0) {	// the list is empty
				exist_user=calloc(1, sizeof(MUTIPLE_EXIST_USER));

				start_user=exist_user;
				e++;
			}
			else {
				exist_user->next=calloc(1, sizeof(MUTIPLE_EXIST_USER));
				
				exist_user=(MUTIPLE_EXIST_USER*)exist_user->next;
			}							
			strncpy(exist_user->exist_user, username, USER_NAME_LENGTH);						
		}
	}
create_nas_multiple_users_ex_out:
	if (stsfile) {
		mystatus.plist->progress = 100;
		mystatus.status = *ret;
		Report_Progress_To_File(stsfile, &mystatus);
		if (mystatus.plist) free(mystatus.plist);
	}
	if (str) free(str);

	/*if (update_flash && *ret == SUCCESS) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
		Update_Flash_Data_Ex("/etc/config/nwpwd", TRUE);
	}*/

	return (MUTIPLE_EXIST_USER*)start_user;
}

MUTIPLE_EXIST_USER* Remove_NAS_Multiple_Users(
	char (*ulist)[USER_GROUP_NAME_LENGTH], int ucnt,
	int *errcode, BOOL update_flash, char* sts_file)
{
	int ret, i;
	NAS_USER_GROUP_TYPE	type;
	char user_name[USER_GROUP_NAME_LENGTH];
	PROGRESS_STATUS		mystatus;
	MUTIPLE_EXIST_USER	*start_user = NULL, *cur_user = NULL;

#ifdef	DEBUG
	fprintf(stdout, "Remove_NAS_Multiple_Users: %d users, update flash ? %s, status file = %s\n",
		ucnt, (update_flash) ? "YES" : "NO", (sts_file==NULL) ? "NONE" : sts_file);
#endif
	if (sts_file) {
		mystatus.plist = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (mystatus.plist == NULL) {
#ifdef	DEBUG
			fprintf(stdout, "Remove_NAS_Multiple_Users: fails to allocate memory!\n");
#endif
			*errcode = ERROR_OUT_OF_MEMORY;
			goto remove_nas_multiple_users_out;
		}

		mystatus.plist->progress = 0;
		mystatus.plist->next = NULL;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(sts_file, &mystatus);
	}
#ifdef	DEBUG
	fprintf(stdout, "Remove_NAS_Multiple_Users: step 1\n");
#endif
	*errcode = SUCCESS;
	for (i=0; i<ucnt; i++) {
#ifdef	DEBUG
		fprintf(stdout, "\t1.1 user : %s (len=%d)\n", ulist[i], strlen(ulist[i]));
#endif
		if (sts_file) {
			mystatus.status = S_PROCESSING;
			mystatus.plist->progress = 100*i/ucnt;
			Report_Progress_To_File(sts_file, &mystatus);
		}
#ifdef	DEBUG
		fprintf(stdout, "\t1.2\n");
#endif
		strncpy(user_name, ulist[i], USER_GROUP_NAME_LENGTH-1);
		user_name[USER_GROUP_NAME_LENGTH-1]=0x0;
		Remove_Blank_From_String(user_name);

		if (Is_Reserved_User(user_name)) {
#ifdef	DEBUG
			fprintf(stdout, "\t%s is a reserved user!\n", user_name);
#endif
			if (start_user==NULL) {	// the list is empty
				cur_user = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
				start_user = cur_user;
			}
			else {
				cur_user->next = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
				cur_user = (MUTIPLE_EXIST_USER*)cur_user->next;
			}							
			strncpy(cur_user->exist_user, user_name, USER_NAME_LENGTH-1);
			continue;
		}
	
		type = NAS_Get_User_Type(user_name);
#ifdef	DEBUG
		fprintf(stdout, "\t%s type = %d\n", user_name, (int)type);
#endif
		if (type==NAS_USER_GROUP_NONE) {
#ifdef	DEBUG
			fprintf(stdout, "\t%s is NOT a NAS user!\n", user_name);
#endif
			if (start_user == NULL) {	// the list is empty
				cur_user = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
				start_user = cur_user;
			}
			else {
				cur_user->next=calloc(1, sizeof(MUTIPLE_EXIST_USER));
				cur_user=(MUTIPLE_EXIST_USER*)cur_user->next;
			}							
			strncpy(cur_user->exist_user, user_name, USER_NAME_LENGTH);
			continue;
		}
		else if (type!=NAS_USER_GROUP_PDC) {
			// remove samba user passwd first 
			Remove_Samba_User(user_name);
			remove_netware_user(user_name);
		}
#ifdef	DEBUG
		fprintf(stdout, "\t1.3\n");
#endif
		Remove_User_Share_Privilege(user_name);
		ret = Remove_System_User(user_name);
		if (ret<0) {
#ifdef	DEBUG
			fprintf(stdout, "\tRemove_System_User(%s) fails, err=%d\n", user_name, ret);
#endif
			if (start_user == NULL) {	// the list is empty
				cur_user=calloc(1, sizeof(MUTIPLE_EXIST_USER));

				start_user = cur_user;
			}
			else {
				cur_user->next=calloc(1, sizeof(MUTIPLE_EXIST_USER));
				cur_user=(MUTIPLE_EXIST_USER*)cur_user->next;
			}
			strncpy(cur_user->exist_user, user_name, USER_NAME_LENGTH);
		}
	}

remove_nas_multiple_users_out:
#ifdef	DEBUG
	fprintf(stdout, "Remove_NAS_Multiple_Users: step 2\n");
#endif
	// update flash ?
	/*if (update_flash && *errcode == SUCCESS) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
		Update_Flash_Data_Ex("/etc/config/nwpwd", TRUE);
	}*/
#ifdef	DEBUG
	fprintf(stdout, "Remove_NAS_Multiple_Users: step 3\n");
#endif
	// reports over ?
	if (sts_file) {
		mystatus.status = *errcode;
		mystatus.plist->progress = 100;
		Report_Progress_To_File(sts_file, &mystatus);

		if (mystatus.plist) free(mystatus.plist);
	}
#ifdef	DEBUG
	fprintf(stdout, "Remove_NAS_Multiple_Users: step 4\n");
#endif
	return (MUTIPLE_EXIST_USER*)start_user;
}

MUTIPLE_EXIST_USER* Add_NAS_User_List_To_Group(
	char *grp_name, char (*ulist)[USER_GROUP_NAME_LENGTH], int ucnt,
	int *errcode, BOOL update_flash, char* stsfile)
{
	int str_len, name_len, fld_len, new_len;
	int outbuf_len = 0, i;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];
	char *outbuf = NULL;
	PROGRESS_STATUS mystatus;
	MUTIPLE_EXIST_USER *start_user = NULL, *cur_user = NULL;

#ifdef	DEBUG
	printf("Add_NAS_User_List_To_Group: starts!\n");
#endif
	if (stsfile) {
		mystatus.plist = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (mystatus.plist == NULL) {
#ifdef	DEBUG
			printf("Add_NAS_User_List_To_Group: fails to allocate memory!\n");
#endif
			*errcode = ERROR_OUT_OF_MEMORY;
			mystatus.status = *errcode;
			Report_Progress_To_File(stsfile, &mystatus);
			return NULL;
		}

		mystatus.plist->progress = 0;
		mystatus.plist->next = NULL;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(stsfile, &mystatus);
	}

	if (grp_name == NULL || ulist == NULL || ucnt <=0) {
#ifdef	DEBUG
		printf("\tgroup %s, %d users, update flash? %s, status file = %s\n"
			"\tinvalid input(s)\n",
			(grp_name==NULL) ? "NULL" : grp_name, ucnt,
			(update_flash) ? "YES" : "NO", (stsfile==NULL) ? "NULL" : stsfile);
#endif
		*errcode = ERROR_INVALID_PARAM;
		goto add_nas_user_list_to_group_out;
	}
	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;

	// check group name
	if(EMPTY_STRING(grpname)) { 		// Null string check
		*errcode = ERROR_EMPTY_STRING;
		goto add_nas_user_list_to_group_out;
	}
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back
	/*Marked by Shone (2004,12,15)
	if (!Is_Legal_String(grpname)) {	// check whether the name is legal or not
		*errcode = ERROR_ILLEGAL_NAME;
		goto add_nas_user_list_to_group_out;
	}*/
	if (!Is_System_Group(grpname)) {	// check whether the name exists or not 
		*errcode = ERROR_NOT_EXISTED;
		goto add_nas_user_list_to_group_out;
	}

	*errcode = SUCCESS;
	for (i=0; i<ucnt; i++) {
		if (stsfile) {
			mystatus.status = S_PROCESSING;
			mystatus.plist->progress = 100*i/ucnt;
			Report_Progress_To_File(stsfile, &mystatus);
		}

		strncpy(username, ulist[i], USER_NAME_LENGTH-1);
		username[USER_NAME_LENGTH-1]=0x0;
		Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
                                                    //Marked by Shone(2004,12,15)
		if (EMPTY_STRING(username) || /*(!Is_Legal_String(username)) ||*/
			(!Is_System_User(username)))
		{ 	// something wrong with the user name string
			if (start_user == NULL) {	// the list is empty
				cur_user = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
				start_user = cur_user;
			}
			else {
				cur_user->next = calloc(1, sizeof(MUTIPLE_EXIST_USER));
				cur_user = (MUTIPLE_EXIST_USER*)cur_user->next;
			}
			strncpy(cur_user->exist_user, username, USER_NAME_LENGTH);
			continue;
		}

		// check the user_name belongs to the grp_name member already?
		if (User_Belongs_To_Group(username, grpname))
			continue;

		// Set string comparision with case insensitive
		Conf_Case_Sensitive(0);
		name_len =  strlen (username);

		if (outbuf == NULL) {
			outbuf = (char*)calloc(1, BUF_SIZE * sizeof(char));
			if (outbuf == NULL) {
				*errcode = ERROR_OUT_OF_MEMORY;
				goto add_nas_user_list_to_group_out;
			}
			outbuf_len = BUF_SIZE;
		}

		if (strlen(outbuf) + name_len >= outbuf_len) {
			char* tmpbuf = NULL;

			tmpbuf = (char*) realloc(outbuf, (outbuf_len+BUF_SIZE) * sizeof(char));
			if (tmpbuf == NULL) {
				if (outbuf) free(outbuf);
				*errcode = ERROR_OUT_OF_MEMORY;
				goto add_nas_user_list_to_group_out;
			}
			outbuf = tmpbuf;
			outbuf_len += BUF_SIZE;
		}
		if (strlen(outbuf) > 0)
			sprintf(outbuf, "%s,%s", outbuf, username);
		else
			strcpy(outbuf, username);
	}

	// set the group member list
	if( (str_len = SYS_Get_One_String(GROUP_CONF_PATH,1, ':', grpname, NULL,0)) > 0) {
		char out_str[str_len];

		SYS_Get_One_String(GROUP_CONF_PATH, 1, ':', grpname, out_str, str_len);
		if((fld_len = Get_String_Field (out_str,4,':',NULL,0)) > 0) {
			char fld_str[fld_len];
			char set_str[fld_len + outbuf_len];

			Get_String_Field (out_str,4,':',fld_str,fld_len);

			if( compare_case_string(fld_str,"") == 0 ) //no member in this group
				sprintf(set_str, "%s%s", fld_str, outbuf);
			else
				sprintf(set_str, "%s,%s", fld_str, username);

			if ( (new_len = Set_String_Field(out_str, 4, ':', set_str, NULL, 0)) > 0) {
				char new_str[new_len];

				Set_String_Field(out_str, 4, ':', set_str, new_str, new_len);
				*errcode = SYS_Set_One_String(GROUP_CONF_PATH,1,':',grpname,new_str);
			}
		}
		else
			*errcode = fld_len;
	}
	else
		*errcode = str_len;

	if (outbuf) free(outbuf);

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
	}*/

add_nas_user_list_to_group_out:
	if (stsfile) {
		mystatus.status = *errcode;
		mystatus.plist->progress = 100;
		Report_Progress_To_File(stsfile, &mystatus);
		free(mystatus.plist);
	}

	return (MUTIPLE_EXIST_USER*)start_user;
}

MUTIPLE_EXIST_USER* Remove_NAS_User_List_From_Group(
	char *grp_name, char (*ulist)[USER_GROUP_NAME_LENGTH], int ucnt,
	int* errcode, BOOL update_flash, char* sts_file)
{
 	char buf[USER_NAME_LENGTH];
	int str_len, fld_len, new_len, u, i, found;
	char username[USER_NAME_LENGTH];
	char grpname[GROUP_NAME_LENGTH];
	char tmp_grp_file[BUF_SIZE], tmpcmd[BUF_SIZE];
	char *out_str = NULL, *fld_str = NULL;
	PROGRESS_STATUS mystatus;
	MUTIPLE_EXIST_USER *start_user = NULL, *cur_user = NULL;

#ifdef	DEBUG
	printf("Remove_NAS_User_List_From_Group: starts!\n");
#endif
	if (sts_file) {
		mystatus.plist = (PROGRESS_LIST*) malloc(sizeof(PROGRESS_LIST));
		if (mystatus.plist == NULL) {
#ifdef	DEBUG
			printf("Remove_NAS_User_List_From_Group:"
				" fails to allocate memory!\n");
#endif
			*errcode = ERROR_OUT_OF_MEMORY;
			mystatus.status = *errcode;
			Report_Progress_To_File(sts_file, &mystatus);
			return NULL;
		}

		mystatus.plist->progress = 0;
		mystatus.plist->next = NULL;
		mystatus.status = S_PROCESSING;
		Report_Progress_To_File(sts_file, &mystatus);
	}

	if (grp_name == NULL || ulist == NULL || ucnt <=0) {
#ifdef	DEBUG
		printf("\tgroup %s, %d users, update flash? %s, status file = %s\n"
			"\tinvalid input(s)\n",
			(grp_name==NULL) ? "NULL" : grp_name, ucnt,
			(update_flash) ? "YES" : "NO", (sts_file==NULL) ? "NULL" : sts_file);
#endif
		*errcode = ERROR_INVALID_PARAM;
		goto remove_nas_user_list_from_group_out;
	}

	strncpy(grpname, grp_name, GROUP_NAME_LENGTH-1);
	grpname[GROUP_NAME_LENGTH-1]=0x0;

	// check group name
	if(EMPTY_STRING(grpname)) { 		// Null string check
		*errcode = ERROR_EMPTY_STRING;
		goto remove_nas_user_list_from_group_out;
	}
	Remove_Blank_From_String(grpname); 	// Remove the extra blank in front, middle and back
	/*Marked by Shone (2004,12,15)
	if (!Is_Legal_String(grpname)) {		// check whether the name is legal or not
		*errcode =ERROR_ILLEGAL_NAME;
		goto remove_nas_user_list_from_group_out;
	}*/
	if (!Is_System_Group(grpname)) {		// check whether the name exists or not 
		*errcode =ERROR_NOT_EXISTED;
		goto remove_nas_user_list_from_group_out;
	}

	sprintf(tmp_grp_file, "%s.tmp", GROUP_CONF_PATH);
	sprintf(tmpcmd, "cp %s %s", GROUP_CONF_PATH, tmp_grp_file);
	if (system(tmpcmd) != 0) {
		*errcode = ERROR_FAIL;
		goto remove_nas_user_list_from_group_out;
	}

	if (NAS_File_Lock(GROUP_CONF_PATH, O_RDWR)==0) {
		*errcode = ERROR_OPEN_FILE;
		unlink(tmp_grp_file);
		goto remove_nas_user_list_from_group_out;
	}
	if ((str_len = SYS_Get_One_String(tmp_grp_file, 1, ':', grpname, NULL, 0)) > 0) {
		out_str = (char*) calloc(1, str_len * sizeof(char));
		if (out_str == NULL) {
			*errcode = ERROR_OUT_OF_MEMORY;
			NAS_File_Unlock(GROUP_CONF_PATH);
			goto remove_nas_user_list_from_group_out;
		}
		SYS_Get_One_String(tmp_grp_file, 1, ':', grpname, out_str, str_len);

		if ((fld_len = Get_String_Field (out_str, 4, ':', NULL, 0)) > 0) {
			fld_str = (char*) calloc(1, fld_len * sizeof(char));
			if (fld_str == NULL) {
				*errcode = ERROR_OUT_OF_MEMORY;
				NAS_File_Unlock(GROUP_CONF_PATH);
				goto remove_nas_user_list_from_group_out;
			}
			Get_String_Field (out_str, 4, ':', fld_str, fld_len);
		}
		else {
			*errcode = ERROR_NOT_FOUND;
			NAS_File_Unlock(GROUP_CONF_PATH);
			goto remove_nas_user_list_from_group_out;
		}
	}
	else {
		*errcode = ERROR_NOT_FOUND;
		NAS_File_Unlock(GROUP_CONF_PATH);
		goto remove_nas_user_list_from_group_out;
	}

	*errcode = SUCCESS;
	for (u=0; u<ucnt; u++) {
		if (sts_file) {
			mystatus.status = S_PROCESSING;
			mystatus.plist->progress = 100*i/ucnt;
			Report_Progress_To_File(sts_file, &mystatus);
		}

		strncpy(username, ulist[u], USER_NAME_LENGTH-1);
		username[USER_NAME_LENGTH-1]=0x0;

		// check user name
		Remove_Blank_From_String(username); 	// Remove the extra blank in front, middle and back
		                                               //Marked by Shone(2004,12,15)
		if (EMPTY_STRING(username) ||/* (!Is_Legal_String(username)) ||*/
			(!Is_System_User(username)))
		{		// Null string check
			if (start_user == NULL) {	// the list is empty
				cur_user = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
				start_user = cur_user;
			}
			else {
				cur_user->next = calloc(1, sizeof(MUTIPLE_EXIST_USER));
				cur_user = (MUTIPLE_EXIST_USER*)cur_user->next;
			}
			strncpy(cur_user->exist_user, username, USER_NAME_LENGTH);
			continue;
		}
		// Set string comparision with case insensitive
		Conf_Case_Sensitive(0);

		i = 0;
		found = 0;
		do {
			i++;
			Get_String_Field(fld_str,i,',',buf,USER_NAME_LENGTH);
			if( compare_case_string(username,buf) == 0 ) {
				found ++;
				break;
			}

			if ( i>MAX_USER_NUMBER ) {
				if (start_user == NULL) {	// the list is empty
					cur_user = (MUTIPLE_EXIST_USER*)calloc(1, sizeof(MUTIPLE_EXIST_USER));
					start_user = cur_user;
				}
				else {
					cur_user->next = calloc(1, sizeof(MUTIPLE_EXIST_USER));
					cur_user = (MUTIPLE_EXIST_USER*)cur_user->next;
				}
				strncpy(cur_user->exist_user, username, USER_NAME_LENGTH);
				break;
			}
		} while(1);

		if (found > 0)
			Remove_String_Field(fld_str,i,',');
	}

	if ((new_len = Set_String_Field(out_str, 4, ':', fld_str, NULL, 0)) > 0) {
		char new_str[new_len];

		Set_String_Field(out_str, 4, ':', fld_str, new_str, new_len);
		if ((*errcode = SYS_Set_One_String(tmp_grp_file, 1, ':', grpname, new_str)) < 0)
			goto remove_nas_user_list_from_group_out;
	}
	NAS_File_Unlock(GROUP_CONF_PATH);
	sprintf(tmpcmd, "mv %s %s", tmp_grp_file, GROUP_CONF_PATH);
	if (system(tmpcmd) != 0) {
		*errcode = ERROR_FAIL;
		goto remove_nas_user_list_from_group_out;
	}

	/*if (update_flash) {
		Update_Flash_Data_Ex(PASS_CONF_FILE, TRUE);
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(SAMBA_PASS_PATH, TRUE);
	}*/

remove_nas_user_list_from_group_out:
	if (out_str) free(out_str);
	if (fld_str) free(fld_str);

	if (sts_file) {
		mystatus.status = *errcode;
		mystatus.plist->progress = 100;
		Report_Progress_To_File(sts_file, &mystatus);
		if (mystatus.plist) free(mystatus.plist);
	}

	return (MUTIPLE_EXIST_USER*)start_user;
}

#endif

int Get_NAS_User_Count(void)
{
	return (get_total_user_count());
}

#ifdef __QNAP_NAS_ADS__
int Create_System_Group_Ex(GROUP_DATA *g, BOOL update_flash)
{
	char buf[BUF_SIZE];
	char check_group_name[2*USER_GROUP_NAME_LENGTH];

	// Check the Group Name already exist in /etc/group file ?
	if (Is_System_Group(g->name)) 	// group exist, return error
		return ERROR_ALREADY_EXISTED;

	Str_Trans_CmdStr(g->name, check_group_name);
	if (g->gid > 0)
		sprintf(buf, "%s -g %d \"%s\" 2>/dev/null 1>/dev/null", ADDGROUP_PATH, g->gid, check_group_name);
	else
		sprintf(buf, "%s \"%s\" 2>/dev/null 1>/dev/null", ADDGROUP_PATH, check_group_name);
	if (system(buf) != 0) {
#ifdef DEBUG
		fprintf(stderr, "%s: system(%s) failed.\n", __FUNCTION__, buf);
#endif
		return ERROR_EXEC_FAIL;
	}
	if (g->type != NAS_USER_GROUP_LOCAL)
		Set_NAS_Group_Type(g->name, g->type);

	/*if (update_flash) {
		Update_Flash_Data_Ex(GROUP_CONF_PATH, TRUE);
		Update_Flash_Data_Ex(PDC_GROUP_FILE, TRUE);
	}*/
	return SUCCESS;
}
#endif // __QNAP_NAS_ADS__
