/* mini-lpd (See RFC1179)
 * written for WDT
 * Nico Schottelius
 * Copying: GPL
 * Version: 0.4
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>
#include <unistd.h>
#include <stdio.h>
#include <signal.h>
#include <fcntl.h>
#include <errno.h>
#include <limits.h>
#include <dirent.h>
#include <string.h>
#include <stdlib.h>
#include <sys/poll.h>  /* poll */
#include <syslog.h>
//#include <netinet/ip.h> /* wherefore did I need that? */


//#define PORT         1024+515
#define PORT         515
#define MY_IP        "0.0.0.0"   /* where we listen to */
#define SOCK_QUEUE   32 /* maximum elements in queue */
#define LF           0x0a  /* line feed */


/* DEBUG */
#ifdef DEBUG
# define D_PRINTF(x)  ( printf("[%s:%d]: %s\n",__FILE__,__LINE__,x) )
#else
# define D_PRINTF(x)  if(0) 
#endif


#define CONFIG_DIR   "/etc/mini-lpd"
/* #define CONFIG_DIR   "/home/user/nico/mini-lpd.test" */
#define DEVNAME      "device"
#define SLASH        "/"
#define MAX_PRINTERS 5	//one is LPT PRINTER and the other one is usb printer

struct mlqueue {
   char *name;       /* name of queue */
   int   dev;        /* fd to device */
//   int   filter;   /* fd or char * to filter program */ /* not implemented */
};

/* global variables */
struct mlqueue queues[MAX_PRINTERS];
int    qcnt;

/* whitespaces */
#define  WS_VT 0x0b  /* vertical tab */
#define  WS_HT 0x09  /* horizontal tab */
#define  WS_SP 0x20  /* space */
#define  WS_FF 0x0c  /* form feed */

#define	WAIT_PRINT		0	//Wait fo jobs
#define PRINTING		1	//Printing
#define PRINT_FINISH	2	//Print finish
#define PRINT_ERROR		3	//Print Error happen
#define UNKNOWN_CMD		4	//Unknown command

#define DB25_PRINTER	"lp"
#define USB_PRINTER		"lpusb"

//For Printer status (other define in amit.h) 
#define PTR_STAT_NOT_READY		1	
#define PTR_STAT_OOP			3	//off-line or Out of Paper
#define PTR_STAT_PRINTING		4
#define PTR_STAT_READY			5
#define PTR_STAT_DEV_ERR		7	//device error
#define PTR_STAT_UNKNOW_ERR		9	//unknow error


/************************************************/
/*	Purpose:print out printing status for UI	*/
/*				written by Laroche 2005/11/29	*/
/************************************************/
int lp_status(int status, char* lp_name)
{
	FILE *fd;
	
		if( !strncmp(lp_name,USB_PRINTER, strlen(USB_PRINTER)) )
			fd = fopen("/var/log/print_usb","w+");
		else if(!strncmp(lp_name, DB25_PRINTER, strlen(DB25_PRINTER)) )
			fd = fopen("/var/log/print_db25","w+"); 
		else
			return 0;	//error
			
		if (fd != NULL)
		{	
			switch (status)
			{
				case WAIT_PRINT:
					fprintf(fd,"%d",PTR_STAT_READY);
					break;
				case PRINTING:
					fprintf(fd,"%d",PTR_STAT_PRINTING);
					break;
				case PRINT_FINISH:
					fprintf(fd,"%d",PTR_STAT_READY);
					break;
				case PRINT_ERROR:
					fprintf(fd,"%d",PTR_STAT_OOP);
					break;
				case UNKNOWN_CMD:
					fprintf(fd,"%d",PTR_STAT_UNKNOW_ERR);
					break;
				default:
					break;
			}
			fclose (fd);
		}
#if 0
		fd = fopen("/var/log/print_status","w+");
		if (fd)
		{	
			switch (status)
			{
				case 0:
					fprintf(fd,"Waiting for Jobs\n");
					break;
				case 1:
					fprintf(fd,"Printing a job\n");
					break;
				case 2:
					fprintf(fd,"Finish a job\n");
					break;
				case 3:
					fprintf(fd,"Print Error\n");
					break;
				case 4:
					fprintf(fd,"Unknown Command\n");
					break;
				default:
					break;
			}
			fclose (fd);
			status_record = status;
		}
#endif
	return 1;

}

char *whitespace(char *string)
{
   char *sp;

   sp = string; while(*sp != 0) { if (*sp == WS_VT) return sp; sp++; };
   sp = string; while(*sp != 0) { if (*sp == WS_HT) return sp; sp++; };
   sp = string; while(*sp != 0) { if (*sp == WS_SP) return sp; sp++; };
   sp = string; while(*sp != 0) { if (*sp == WS_FF) return sp; sp++; };

   return NULL;
}

/* we will allocate max+1 buf (\0) */
char *read_file(int socket, int bytes)
{
   char *queue = NULL, buf;
   int tmp, cnt = 0;
 
   while( (tmp = read(socket, &buf, 1)) ) {
      if(tmp == -1) {
         perror("read");
		 //lp_status(PRINT_ERROR);
         return NULL;
      }
      cnt++;
      
      queue = realloc(queue,cnt);
      if(queue == NULL) {
         return queue;
      }
      queue[cnt-1] = buf;

      if(cnt == bytes) break;
   }
   queue = realloc(queue,cnt+1);
   if(queue == NULL) {
      return queue;
   }
   queue[cnt] = '\0'; /* add terminating \0 */
   
   return queue;
}

char *read_line(int socket, char end)
{
   char *queue = NULL, buf;
   int tmp, cnt = 0;
 
   while( (tmp = read(socket, &buf, 1)) ) {
      if(tmp == -1) {
         perror("read");
		 //lp_status(PRINT_ERROR);
         return NULL;
      }
      cnt++;
      
      queue = realloc(queue,cnt);
      if(queue == NULL) {
         return queue;
      }
      queue[cnt-1] = buf;

      if(buf == end) { /* end of queue */
         break;
      }
   }
   if(cnt) {
      queue[cnt-1] = '\0'; /* replace LF with \0 */
   }

   return queue;
}

int print_jobs(int socket, char *queue)
{
   D_PRINTF(queue);
   free(queue);
   return 1;
} /* done */

#define  ACK   0
#define  NACK  1

int sent_ack(int socket,char code)
{
   if (write(socket,&code,1) == -1) {
     if(errno != EINVAL) {
         perror("write ack");
      }
      return 0;
   } else 
      return 1;
}

int recv_ack(int socket)
{
   char buf;

   if (read(socket,&buf,1) == -1) {
     if(errno != EINVAL) {
         perror("write ack");
      }
      return 0;
   } else {
      if(buf == 0) {
         return 1;
      } else {
         return 0;
      }
   }
}

/* recieve the job: data/control files */
int recieve_job(int socket, char *queue)
{
   char buf = NACK, *cmd = NULL, *tbuf;
   int tmp, num = 0;
   
   D_PRINTF(queue);
   while(num < qcnt) {
      //if(! strcmp(queue,queues[num].name) ) {
      //jpg
	  if(! strcasecmp(queue,queues[num].name) ) {
         D_PRINTF(queues[num].name);
         buf = ACK;
         break;
      } else {
         num++;
      }
   }

   sent_ack(socket,buf);
   if(buf != ACK) return 0;

   while( (tmp = read(socket,&buf,1) ) ) {
      D_PRINTF("lese subkommando");
      if(tmp == -1) {
         if(errno != EINVAL) {
            perror("rj: subcommand");
            return 0;
         }
         return 1;
      }
      
      
      /* read subcommand parameters, acknowledge*/
      D_PRINTF("lese parameter");
      cmd = read_line(socket,LF);
      if(cmd == NULL) {
         sent_ack(socket,NACK);
         return 0;
      }
      if( !sent_ack(socket,ACK) ) return 0;
      
      D_PRINTF("subkommando-parameter:");
      D_PRINTF(cmd);
     
	  lp_status(PRINTING,queues[num].name);
	  
      switch(buf) {
         case 1: /* abort job */
            /* read LF */
            tmp = read(socket,&buf,1);
            if(tmp == -1 || buf != LF) {
               perror("read lf");
            }
            return 0;
            break;
         case 2: /* Recieve controle file: 02 | Count | SP | Name | LF */
            cmd = read_line(socket, 0);
            D_PRINTF("cf");
            D_PRINTF(cmd);
            
            if( !sent_ack(socket,ACK) ) return 0;

            switch(cmd[0]) {
               case 'C': /* C - Class for banner page */
               case 'H': /* H - Host name */
               case 'J': /* J - Job name for banner page */
               case 'P': /* P - User identification */
               case 'I': /* I - Indent Printing */
               case 'L': /* L - Print banner page */
 //                 printf("%c: %s\n",cmd[0],&cmd[1]);
                  break;

               case 'M': /* M - Mail When Printed */
                  break;

               case 'N': /* N - Name of source file */
                  break;

               case 'S': /* S - Symbolic link data */
                  break;

               case 'T': /* T - Title for pr */
                  break;

               case 'U': /* U - Unlink data file */
                  break;

               case 'W': /* W - Width of output */
                  break;

               case '1': /* 1 - troff R font */
                  break;
               /* 7.14 - 7.18 ignored */
            }
            free(cmd);
            break;
         case 3: /* Recieve data file 03 | Count | SP | Name | LF */
            D_PRINTF("bis weisszeichen");
            tbuf = whitespace(cmd);
            if(!tbuf) return 0;

            *tbuf = 0;
            tmp = atoi(cmd);
            free(cmd);
            D_PRINTF("nach befreiung");

            /* read the content of controllfile */
            cmd = read_file(socket,tmp);
            D_PRINTF("nach datei lesen");

            if(write(queues[num].dev,cmd,tmp) == -1) {
               perror("write data");
			   //lp_status(PRINT_ERROR);
			   lp_status(PRINT_ERROR, queues[num].name);
               return 0;
            }
			
            free(cmd);
            D_PRINTF("nach 2. befreiung");
            if( !recv_ack(socket) ) return 0;
            if( !sent_ack(socket,ACK) ) return 0;
            D_PRINTF("nach ack");
            break;
      }
   }
   lp_status(WAIT_PRINT,queues[num].name);
   return 1;
}

/* we are called, if one or _more_ connections are waiting */
void sigio(int sock)
{
   int len;
   int nsock;
   char buf, *cmd = NULL;

   while( (nsock = accept(sock,(struct sockaddr *) NULL, (socklen_t *) NULL) )
           != -1) {
      if( nsock == -1) {
         D_PRINTF("-1 von accept");
         if (errno != EAGAIN) {
            perror("accept");
            _exit(1);
         } else {
            D_PRINTF("returne");
            return;
         }
      }

      /* read command */
      len = read(nsock,&buf,1);
      if(len == -1) break;
      D_PRINTF("nach read kommano / vor cmd read");
      
      /* read command parameters (until LF) */
      cmd = read_line(nsock, LF);

      D_PRINTF("params");
      D_PRINTF(cmd);
	  //lp_status(PRINTING);
      switch(buf) {
         case 1: /* print waiting jobs */
            D_PRINTF("print waiting jobs");
            print_jobs(nsock,cmd);
            break;
         case 2: /* recieve a job */
            D_PRINTF("get a job");
            recieve_job(nsock,cmd);
            break;
         case 3: /* send queue state (short) */
            D_PRINTF("send queue state short");
            break;
         case 4: /* send queue state (long) */
            D_PRINTF("send queue state long");
            D_PRINTF(cmd);
            break;
         case 5: /* Remove Job */
            D_PRINTF("rm job");
            break;
         default:
            break;
      }
      close(nsock);
      //lp_status(PRINT_FINISH);
   }
}

/* config: /etc/mini-lpd/
 * 
 * every queue has a directory below configdir
 * in this queue-dir is:
 * device -> link to device I should write to
 * filter -> program to pipe input to before writing to device
 */

int read_config()
{
   DIR *d_tmp = NULL;
   struct dirent *tdirent;
   char pathbuf[PATH_MAX];

   d_tmp = opendir(CONFIG_DIR);
   if(d_tmp == NULL) {
      return 0;
   }

   qcnt = 0;

   while( (tdirent = readdir(d_tmp) ) != NULL) {
      if ( *(tdirent->d_name) == '.') continue; /* skip "^\..*" */

      if(qcnt == MAX_PRINTERS) break;

      queues[qcnt].name = malloc( strlen(tdirent->d_name) + 1);
      if(queues[qcnt].name == NULL) return 0;
      strcpy(queues[qcnt].name,tdirent->d_name);

      D_PRINTF(queues[qcnt].name);
      
      strcpy(pathbuf,CONFIG_DIR);
      strcat(pathbuf,SLASH);
      strcat(pathbuf,queues[qcnt].name);
      strcat(pathbuf,SLASH);
      strcat(pathbuf,DEVNAME);
	  //printf("pathbuf = %s\n",pathbuf);
      queues[qcnt].dev = open(pathbuf,O_RDWR|O_APPEND);
		
      if(queues[qcnt].dev == -1) {
         perror("open printer");
         syslog(LOG_INFO,"Mini-lpd:Open %s Failed",pathbuf);
		 lp_status(PRINT_ERROR,queues[qcnt].name);
		 return 0;
      }
	  else{
		 lp_status(WAIT_PRINT,queues[qcnt].name);
	  }
	  //printf("open pathbuf = %s\n",pathbuf);
	  
      qcnt++;
   }
 
   return 1;
}


/* to call when exiting */
void sigexit(int signal)
{
   while(qcnt--) {
      close(queues[qcnt].dev);
   }
   unlink("/var/run/lpd.pid");
   syslog(LOG_INFO,"Mini-lpd:Receive Exit Singal\n");
   _exit(0);
}

#define BANNER "mini-lpd 0.4\n"

int main()
{
   int tmp, sock;
   char ip[32] = MY_IP;
   struct sigaction sa;
   struct sockaddr_in ins;
   struct pollfd plist;
   FILE *fs = NULL; 
   D_PRINTF("0.4");
 
   syslog(LOG_INFO,"Starting mini-lpd-0.4\n");
   
   if(!read_config()) return 23;
   
	//write lpd.pid
	if((fs = fopen("/var/run/lpd.pid","w")) != NULL) 
	{
		fprintf(fs,"%d",getpid());
		fclose(fs);              
	}
   /* signalhandler for TERM, INT */
   sa.sa_handler=sigexit;
   sigaction(SIGTERM, &sa, NULL);
   sigaction(SIGINT, &sa, NULL);

   /* from gpm: clear structure  */
   memset(&ins, 0, sizeof(ins));

   ins.sin_family = AF_INET;
   ins.sin_port   = htons(PORT);
   if( ! inet_aton(ip, &(ins.sin_addr) ) ) {
      write(2,"ip kaputt\n",10);
      _exit(1);
   }

   sock = socket(PF_INET,SOCK_STREAM,0);
   if( sock == -1 ) {
      perror("socket");
      _exit(1);
   }

   tmp = 1;
   // lose the pesky "Address already in use" error message
   if (setsockopt(sock,SOL_SOCKET,SO_REUSEADDR,&tmp,sizeof(tmp)) == -1) {
      perror("setsockopt");
      _exit(1);
    }

   if(bind(sock,(struct sockaddr *)&ins,sizeof(ins)) == -1) {
      perror("bind");
      _exit(1);
   }

   /* start listening */
   if(listen(sock,SOCK_QUEUE) == -1) {
      perror("listen");
      _exit(1);
   }

   write(1,BANNER,strlen(BANNER));

   syslog(LOG_INFO,"Mini-lpd:Successful\n");
   plist.fd = sock;
   plist.events = POLLIN | POLLPRI;
   while(1) {
	  //lp_status(WAIT_PRINT);	//wait for jobs
      if(poll(&plist, 1, -1) != -1) {
         if( (plist.revents & POLLIN)  == POLLIN ||
             (plist.revents & POLLPRI) == POLLPRI) {
            /* look if a packet reached us */
            D_PRINTF("sigio starten");
            sigio(sock);
         }
      }
   }
   return 0;
}
