/******************************************************************************
 * $Id: transmissioncli.c 2862 2007-08-19 02:02:10Z charles $
 *
 * Copyright (c) 2005-2006 Transmission authors and contributors
 *
 * Permission is hereby granted, free of charge, to any person obtaining a
 * copy of this software and associated documentation files (the "Software"),
 * to deal in the Software without restriction, including without limitation
 * the rights to use, copy, modify, merge, publish, distribute, sublicense,
 * and/or sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
 * DEALINGS IN THE SOFTWARE.
 *****************************************************************************/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <getopt.h>
#include <signal.h>
#include <time.h>
#include <sys/stat.h> 
#include "transmission.h"
//#include "makemeta.h"

#ifdef __BEOS__
    #include <kernel/OS.h>
    #define wait_msecs(N)  snooze( (N) * 1000 )
    #define wait_secs(N)   sleep( (N) )
#elif defined(WIN32)
    #include <windows.h>
    #define wait_msecs(N)  Sleep( (N) )
    #define wait_secs(N)   Sleep( (N) * 1000 )
#else
    #define wait_msecs(N)  usleep( (N) * 1000 )
    #define wait_secs(N)   sleep( (N) )
#endif

/* macro to shut up "unused parameter" warnings */
#ifdef __GNUC__
#define UNUSED                  __attribute__((unused))
#else
#define UNUSED
#endif

const char * USAGE =
"Usage: %s [options] file.torrent [options]\n\n"
"Options:\n"
"  -c, --create-from <file>  Create torrent from the specified source file.\n"
"  -a, --announce <url> Used in conjunction with -c.\n"
"  -r, --private        Used in conjunction with -c.\n"
"  -m, --comment <text> Adds an optional comment when creating a torrent.\n"
"  -d, --download <int> Maximum download rate (-1 = no limit, default = -1)\n"
"  -f, --finish <shell script> Command you wish to run on completion\n" 
"  -h, --help           Print this help and exit\n" 
"  -i, --info           Print metainfo and exit\n"
"  -n  --nat-traversal  Attempt NAT traversal using NAT-PMP or UPnP IGD\n"
"  -p, --port <int>     Port we should listen on (default = %d)\n"
#if 0
"  -s, --scrape         Print counts of seeders/leechers and exit\n"
#endif
"  -u, --upload <int>   Maximum upload rate (-1 = no limit, default = 20)\n"
"  -v, --verbose <int>  Verbose level (0 to 2, default = 0)\n";

static int           showHelp      = 0;
static int           showInfo      = 0;
#if 0
static int           showScrape    = 0;
#endif
static int           isPrivate     = 0;
static int           verboseLevel  = 0;
static int           bindPort      = TR_DEFAULT_PORT;
static int           uploadLimit   = 20;
static int           downloadLimit = -1;
static char          * torrentPath = NULL;
static int           natTraversal  = 0;
static sig_atomic_t  gotsig        = 0;
static tr_torrent_t  * tor;

//add for AMIT download assistant
static char          * Dest         = NULL;
char             	 * Job_name 	= NULL;
char             	 * File_ENC     = NULL;
//end AMIT add 

static char          * finishCall   = NULL;
static char          * announce     = NULL;
static char          * sourceFile   = NULL;
static char          * comment      = NULL;

static int  parseCommandLine ( int argc, char ** argv );
static void sigHandler       ( int signal );

char * getStringRatio( float ratio )
{
    static char string[20];

    if( ratio == TR_RATIO_NA )
        return "n/a";
    snprintf( string, sizeof string, "%.3f", ratio );
    return string;
}

#define LINEWIDTH 128

int main( int argc, char ** argv )
{
    int i, error;
    tr_handle_t  * h;
    const tr_stat_t    * s;
    tr_handle_status_t * hstat;
	
	//Laroche add for dest.
	char dest[256];
	
    //printf( "Transmission %s - http://transmission.m0k.org/\n\n",
    //        LONG_VERSION_STRING );

    /* Get options */
    if( parseCommandLine( argc, argv ) )
    {
        printf( USAGE, argv[0], TR_DEFAULT_PORT );
        return EXIT_FAILURE;
    }
	
	BT_LOG( "Transmission %s - http://transmission.m0k.org/",
			LONG_VERSION_STRING);
	BT_LOG("Transmission Starting...");	
	//record pid to /var/run for AMIT Download Assistant
	if (Job_name != NULL)
	{
		char *p=NULL;
		char f_ind[128];
		FILE *pid_fd = NULL;
		struct stat buf;
		p = strrchr(Job_name,'/');
		memset(f_ind,0,sizeof(f_ind));
		sprintf(f_ind,"/var/run/.dl/");
		strncat(f_ind+strlen(f_ind),p+1,strlen(p+1));
		strncat(f_ind,".pid",4);
		if (stat("/var/run/.dl",&buf) !=0)
		{
			mkdir("/var/run/.dl",0700);
		}
		pid_fd = fopen(f_ind,"w");
		if (pid_fd)
		{
			fprintf(pid_fd,"%d",getpid());
			fclose(pid_fd);
		}
		
	}
	

    if( showHelp )
    {
        printf( USAGE, argv[0], TR_DEFAULT_PORT );
        return EXIT_SUCCESS;
    }

    if( verboseLevel < 0 )
    {
        verboseLevel = 0;
    }
    else if( verboseLevel > 9 )
    {
        verboseLevel = 9;
    }
    if( verboseLevel )
    {
        static char env[11];
        snprintf( env, sizeof env, "TR_DEBUG=%d", verboseLevel );
        putenv( env );
    }

    if( bindPort < 1 || bindPort > 65535 )
    {
        printf( "Invalid port '%d'\n", bindPort );
        return EXIT_FAILURE;
    }
    if (Dest != NULL)
	{
		char *p=NULL;
		char job[128];
		p = strrchr(torrentPath,'/');
		memset(job,0,sizeof(job));
		if (p)
		{
			strncpy(job,p+1,strlen(p+1)-8); //.torrent not include
		}
		sprintf(dest,"%s/%s",Dest,job);
		//tr_torrentSetFolder( tor, buf );
	}
	else
	{
		sprintf(dest,".");
	}
	

    /* Initialize libtransmission */
    h = tr_init( "cli" );
#if 0
    if( sourceFile && *sourceFile ) /* creating a torrent */
    {
        int ret;
        tr_metainfo_builder_t* builder = tr_metaInfoBuilderCreate( h, sourceFile );
        tr_makeMetaInfo( builder, torrentPath, announce, comment, isPrivate );
        while( !builder->isDone ) {
            wait_msecs( 1 );
            printf( "." );
        }
        ret = !builder->failed;
        tr_metaInfoBuilderFree( builder );
        return ret;
    }
#endif
    /* Open and parse torrent file */
    //if( !( tor = tr_torrentInit( h, torrentPath, ".", 0, &error ) ) )
    if( !( tor = tr_torrentInit( h, torrentPath, dest, 0, &error ) ) )
    {
        //printf( "Failed opening torrent file `%s'\n", torrentPath );
        BT_LOG( "Failed opening torrent file [%s]", strrchr(torrentPath,'/')+1 );
        tr_close( h );
        return EXIT_FAILURE;
    }

    if( showInfo )
    {
        const tr_info_t * info = tr_torrentInfo( tor );

        s = tr_torrentStat( tor );

        /* Print torrent info (quite  la btshowmetainfo) */
        printf( "hash:     " );
        for( i = 0; i < SHA_DIGEST_LENGTH; i++ )
        {
            printf( "%02x", info->hash[i] );
        }
        printf( "\n" );
        printf( "tracker:  %s:%d\n",
                s->tracker->address, s->tracker->port );
        printf( "announce: %s\n", s->tracker->announce );
        printf( "size:     %"PRIu64" (%"PRIu64" * %d + %"PRIu64")\n",
                info->totalSize, info->totalSize / info->pieceSize,
                info->pieceSize, info->totalSize % info->pieceSize );
        if( info->comment[0] )
        {
            printf( "comment:  %s\n", info->comment );
        }
        if( info->creator[0] )
        {
            printf( "creator:  %s\n", info->creator );
        }
        if( TR_FLAG_PRIVATE & info->flags )
        {
            printf( "private flag set\n" );
        }
        printf( "file(s):\n" );
        for( i = 0; i < info->fileCount; i++ )
        {
            printf( " %s (%"PRIu64")\n", info->files[i].name,
                    info->files[i].length );
        }

        goto cleanup;
    }

#if 0
    if( showScrape )
    {
        int seeders, leechers, downloaded;

        if( tr_torrentScrape( tor, &seeders, &leechers, &downloaded ) )
        {
            printf( "Scrape failed.\n" );
        }
        else
        {
            printf( "%d seeder(s), %d leecher(s), %d download(s).\n",
                    seeders, leechers, downloaded );
        }

        goto cleanup;
    }
#endif

    signal( SIGINT, sigHandler );
    signal( SIGTERM, sigHandler );

    tr_setBindPort( h, bindPort );
  
    tr_setGlobalSpeedLimit   ( h, TR_UP,   uploadLimit );
    tr_setUseGlobalSpeedLimit( h, TR_UP,   uploadLimit > 0 );
    tr_setGlobalSpeedLimit   ( h, TR_DOWN, downloadLimit );
    tr_setUseGlobalSpeedLimit( h, TR_DOWN, downloadLimit > 0 );

    tr_natTraversalEnable( h, natTraversal );
    
    tr_torrentStart( tor );

    for( ;; )
    {
        char string[LINEWIDTH];
        int  chars = 0;
        int result;

        wait_secs( 1 );

        if( gotsig )
        {
            gotsig = 0;
            tr_torrentStop( tor );
            tr_natTraversalEnable( h, 0 );
        }

        s = tr_torrentStat( tor );

        if( s->status & TR_STATUS_CHECK_WAIT )
        {
            chars = snprintf( string, sizeof string,
                "Waiting to check files... [%.2f %%]", 100.0 * s->percentDone );
        }
        else if( s->status & TR_STATUS_CHECK )
        {
            chars = snprintf( string, sizeof string,
                "Checking files... [%.2f %%]", 100.0 * s->percentDone );
        }
        else if( s->status & TR_STATUS_DOWNLOAD )
        {
            chars = snprintf( string, sizeof string,
                "Progress: [%.2f] %%, %d peer%s, download from [%d] [%.2f KB/s], "
                "upload to [%d] [%.2f KB/s] [%s]", 100.0 * s->percentDone,
                s->peersConnected, ( s->peersConnected == 1 ) ? "" : "s",
                s->peersSendingToUs, s->rateDownload,
                s->peersGettingFromUs, s->rateUpload,
                getStringRatio(s->ratio) );
        }
        else if( s->status & TR_STATUS_SEED )
        {
            chars = snprintf( string, sizeof string,
                "Seeding: uploading to [%d] of [%d] peers, [%.2f KB/s] [%s]",
                s->peersGettingFromUs, s->peersTotal,
                s->rateUpload, getStringRatio(s->ratio) );
        }
        else if( s->status & TR_STATUS_STOPPING )
        {
            chars = snprintf( string, sizeof string, "Stopping..." );
        }
        else if( s->status & TR_STATUS_INACTIVE )
        {
            break;
        }
        if( ( signed )sizeof string > chars )
        {
            memset( &string[chars], ' ', sizeof string - 1 - chars );
        }
        string[sizeof string - 1] = '\0';
        //fprintf( stderr, "\r%s", string );
		BT_LOG("%s",string);
        if( s->error )
        {
        //    fprintf( stderr, "\n%s\n", s->errorString );
        	BT_LOG("%s",s->errorString);
        }
        else if( verboseLevel > 0 )
        {
            fprintf( stderr, "\n" );
        }
        
        if( tr_getDone(tor) || tr_getComplete(tor) )
        {
			char cmd[128];
			char job[64];
			char *j = NULL;
			FILE *ff;
			char fn[256];
           //result = system(finishCall);
		   //for AMIT download email alert at finishing
			if ((j = strrchr(Job_name,'/')) != NULL)
			{
				memset(job,0,sizeof(job));
				strncpy(job,j+1,strlen(j+1)-4);
				sprintf(cmd,"amit_bt_mail \"%s\"",job);
		   		system(cmd);
				//write the end time
				sprintf(fn,"/mnt/C/.dl/.btfinish/%s.dlt",job);
				ff = fopen(fn,"w");
				if (ff)
				{
					fprintf(ff,"%d\0",time((time_t *) NULL));
					fclose(ff);
				}
			}
        }
    }
    fprintf( stderr, "\n" );

    /* Try for 5 seconds to delete any port mappings for nat traversal */
    tr_natTraversalEnable( h, 0 );
    for( i = 0; i < 10; i++ )
    {
        hstat = tr_handleStatus( h );
        if( TR_NAT_TRAVERSAL_DISABLED == hstat->natTraversalStatus )
        {
            /* Port mappings were deleted */
            break;
        }
        wait_msecs( 500 );
    }
    
cleanup:
    tr_torrentClose( tor );
    tr_close( h );

    return EXIT_SUCCESS;
}

static int parseCommandLine( int argc, char ** argv )
{
    for( ;; )
    {
        static struct option long_options[] =
          { { "help",     no_argument,       NULL, 'h' },
            { "info",     no_argument,       NULL, 'i' },
            { "scrape",   no_argument,       NULL, 's' },
            { "private",  no_argument,       NULL, 'r' },
            { "verbose",  required_argument, NULL, 'v' },
            { "port",     required_argument, NULL, 'p' },
            { "upload",   required_argument, NULL, 'u' },
            { "download", required_argument, NULL, 'd' },
            { "finish",   required_argument, NULL, 'f' },
            { "create",   required_argument, NULL, 'c' },
            { "comment",  required_argument, NULL, 'm' },
            { "announce", required_argument, NULL, 'a' },
            { "Job",      required_argument, NULL, 'J' },//Dest and Job for AMIT download assistant
		    { "Dest",     required_argument, NULL, 'D' },//add by laroche_hsieh@amit.com.tw
		    { "ENC",	  required_argument, NULL, 'E' },//add by laroche_hsieh@amit.com.tw
            { "nat-traversal", no_argument,  NULL, 'n' },
            { 0, 0, 0, 0} };

        int c, optind = 0;
        c = getopt_long( argc, argv, "hisrv:p:u:d:f:c:m:a:D:J:E:n",
                         long_options, &optind );
        if( c < 0 )
        {
            break;
        }
        switch( c )
        {
            case 'h':
                showHelp = 1;
                break;
            case 'i':
                showInfo = 1;
                break;
#if 0
            case 's':
                showScrape = 1;
                break;
#endif
            case 'r':
                isPrivate = 1;
                break;
            case 'v':
                verboseLevel = atoi( optarg );
                break;
            case 'p':
                bindPort = atoi( optarg );
                break;
            case 'u':
                uploadLimit = atoi( optarg );
                break;
            case 'd':
                downloadLimit = atoi( optarg );
                break;
            case 'f':
                finishCall = optarg;
                break;
            case 'c':
                sourceFile = optarg;
                break;
            case 'm':
                comment = optarg;
                break;
            case 'a':
                announce = optarg;
                break;
			//for AMIT download assistant
			case 'D':
				Dest = optarg;
				break;
			case 'J':
				Job_name = optarg;
				break;
			case 'E':
				File_ENC = optarg;
				break;
			//end AMIT added
            case 'n':
                natTraversal = 1;
                break;
            default:
                return 1;
        }
    }

    if( optind > argc - 1  )
    {
        return !showHelp;
    }

    torrentPath = argv[optind];
    return 0;
}

static void sigHandler( int signal )
{
    switch( signal )
    {
        case SIGINT:
		case SIGTERM:
			gotsig = 1;
            break;

        default:
            break;
    }
}
