/*
 * DB25 printer driver (GPIO)
 *
 * contributors : 
 * 				stanley  95/01/23
 */

#include <linux/init.h>
#include <linux/kernel.h>
#include <linux/proc_fs.h>
#include <linux/types.h>
#include <linux/poll.h>
#include <linux/module.h>
#include <linux/miscdevice.h>
#include <asm/delay.h>		/* udelay() */
#include <asm/io.h>		/* outb */
#include <linux/ioport.h>

#include "amit_lp.h"

#define DATA_PORT	0x70	/* IO Port */
#define DATA_WIDTH	1	
#define CS_GPIO_NO	25	/* chip select 1 GPIO number */
#define DEV_MAJOR	6 /* device major number */
#define DEV_NAME	"lp0"	/* device name */
#define LP_PROC_NAME	"db25"

#define _HW_RDC3210
//#define FOR_DEBUG
//#undef _DEBUG
#ifdef FOR_DEBUG
#define Dprintk	printk
#else
#define Dprintk(stuff...)
#endif

#ifdef FOR_DEBUG
//static unsigned int prn_count = 0;
#endif
//static int lp_major = DEV_MAJOR;
//static char *lp_devname = "lp0";
static unsigned int ioport = DATA_PORT;
//static devfs_handle_t devfs_handle = NULL;
static int Is_Printing = 0;

typedef struct lp_device {
	//wait_queue_head_t wait;
	struct semaphore sem;
	char *lp_buffer;
	int flags;
} lp_device;

static lp_device *lp_dev;

#define lp_busy()		get_gpio(PRN_BUSY)
#define lp_ready(status)	( (status & (LP_PBUSY|LP_POUTPA|LP_PERRORP|LP_PSELECT)) == (LP_PERRORP|LP_PSELECT))		
#define lp_offline(status)	( (status & (LP_PBUSY|LP_POUTPA|LP_PERRORP|LP_PSELECT|LP_PACK)) == (LP_PBUSY|LP_POUTPA|LP_PERRORP|LP_PSELECT|LP_PACK) )
#define lp_read_ack()		get_gpio(PRN_ACK)
#define lp_write_data(x)	outb(x,ioport)


static inline unsigned int ms_to_jiffies(unsigned int ms)
{
	unsigned int j;
	j = (ms * HZ + 500) / 1000;
	return (j > 0) ? j : 1;
}
/*
static inline void enable_gpio(int gpio_no, int flag)
{
#if defined(_HW_RDC3210)
	int reg_val;
	outl(0x80003848, 0xcf8);
	reg_val = inl(0xcfc);
	if(flag == 0)	// disable
		reg_val &= ~(1<<gpio_no);
	else	// enable
		reg_val |= (1<<gpio_no);
	outl(reg_val, 0xcfc);
#endif
}
*/
static inline int get_status()
{
#if defined(_HW_RDC3210)
	outl(0x8000384C, 0xcf8);
	return (inl(0xcfc));
#else
	return 0;
#endif	
}

static inline void set_cs1()
{
#if defined(_HW_RDC3210)
	int reg_val;
	outl(0x80003890, 0xcf8);
	outl( ((ioport << 8) | 0x01) , 0xcfc); /* use IO map */

	//enable chip select1
	outl(0x80003884,0xcf8);
	reg_val = inl(0xcfc);
	reg_val &= ~(1<<25);
	outl(reg_val, 0xcfc);
#endif
}

#ifdef FOR_DEBUG
static inline void get_cs1()
{
	int reg_val;
	outl(0x80003890, 0xcf8);
	reg_val = inl(0xcfc);
	Dprintk("cs1 base addr = %08x\n",reg_val);

	outl(0x80003894,0xcf8);
	reg_val = inl(0xcfc);
	Dprintk("cs1 base addr mask = %08x\n",reg_val);

	outl(0x80003884,0xcf8);
	reg_val = inl(0xcfc);
	Dprintk("CS1_n = %d\n",(reg_val & 0x02000000)?1:0);

}
#endif
/*
static inline int get_gpio(int gpio_no)
{
#if defined(_HW_RDC3210)
	int reg_val;
	outl(0x8000384C, 0xcf8);
	reg_val = inl(0xcfc);
	return (reg_val & (1<<gpio_no))?1:0;
#else
	return 0;
#endif
}

static inline void write_gpio(int gpio_no, int value)
{
#if defined(_HW_RDC3210)
        int reg_val;
        outl(0x8000384C, 0xcf8);
        reg_val = inl(0xcfc);
        if(value==0)
                reg_val &= ~(1<<gpio_no);
        else
                reg_val |= (1<<gpio_no);
        outl(reg_val, 0xcfc);
#endif
}
*/
static inline void lp_set_strobe(int set)
{
#if defined(_HW_RDC3210)
        int reg_val;
        outl(0x8000384C, 0xcf8);
        reg_val = inl(0xcfc);
        if(set==0)
                reg_val &= ~LP_PSTROBE;
        else
                reg_val |= LP_PSTROBE;
        outl(reg_val, 0xcfc);
#endif
}

static int lp_reset()
{
	write_gpio(PRN_INIT,0);
    udelay(LP_DELAY);
	write_gpio(PRN_INIT,1);
        return 1;
}

static inline int lp_char(char lpchar)
{

#if defined(_HW_RDC3210)
	unsigned long count;
	unsigned int status;
	//mm_segment_t orig_fs;
//Retry:
	count = 0;
	while(count < LP_INIT_CHAR )
	//while(1)
	{
		status = get_status();
		if(lp_ready(status)) break;
		//if(!lp_busy()) break;
		count++;
		if (need_resched)
			schedule ();
	} 

	if(count == LP_INIT_CHAR) {
		//if( status & LP_PBUSY) printk("printer busy\n");
		//if( status & LP_POUTPA) printk("out of paper\n");
		//if( !(status & LP_PERRORP)) printk("printer error\n");
		return 0;
	}
	__cli();
	lp_write_data(lpchar);
	
	udelay(1);
	lp_set_strobe(0);
	udelay(1);
	lp_set_strobe(1);

	//wait printer ack
	/*count = 0;
	while(count < LP_WAIT_ACK_TIMEOUT)
	{
		if( get_gpio(PRN_ACK) ==0 )break;
		count ++;
		if(need_resched) schedule();
		//udelay(100);
	}
	
	if(count == LP_WAIT_ACK_TIMEOUT ) {
		//set_current_state(TASK_INTERRUPTIBLE);
		//schedule_timeout(ms_to_jiffies(LP_WAIT_BUSY));
	
		if(lp_busy()){
			printk("(%u) no ack, char = %d\n",prn_count,lpchar);
			//lp_write_data(lpchar);
		}
		//goto Retry;
	}*/
#ifdef FOR_DEBUG
	//prn_count ++;
#endif
	udelay(1);
	__sti();
	return 1;
#else
	Dprintk("%c",lpchar);
	udelay(10);
	return 1;
#endif
}

static inline int lp_write_buf(lp_device *dev, const char *buf, int count)
{
	unsigned long copy_size;
	unsigned long total_bytes_written = 0;
	unsigned long bytes_written;
	unsigned int status;
	unsigned int busy_loop = 0;
	//int i;
	do {
		copy_size = (count <= LP_BUFFER_SIZE ? count : LP_BUFFER_SIZE);
		copy_from_user(dev->lp_buffer, buf, copy_size);

		bytes_written = 0;
		//for(i=0;i<copy_size;i++)
		while(bytes_written < copy_size)
		{
			//if(lp_char(dev->lp_buffer[i]) ==0)
			if(lp_char(dev->lp_buffer[bytes_written]) ==0)
			{
				/*if( (total_bytes_written + bytes_written) >  0) 
					return (total_bytes_written + bytes_written);
				else*/
				{
					status = get_status();
					if(status & LP_PBUSY) {
						//Dprintk("printer busy\n");
						set_current_state(TASK_INTERRUPTIBLE);
						schedule_timeout(ms_to_jiffies(LP_WAIT_BUSY));
						if(busy_loop++ > 300){
							if(total_bytes_written + bytes_written > 0)
								return (total_bytes_written + bytes_written);
							else{
								Dprintk("lp0: write error\n");
								return -EIO;
							}
						}

						continue;
					}
					else if(status & LP_POUTPA) {
						Dprintk("out of paper\n");
						set_current_state(TASK_INTERRUPTIBLE);
						schedule_timeout(ms_to_jiffies(LP_WAIT_OUTPA));
						//return -ENOSPC;
						continue;
					}
					else if(!(status & LP_PERRORP)) {
						printk("printer error\n");
						return -EIO;
					}
					else{
						printk("other error\n");
						return -EIO;
					}
				}
				//return total_bytes_written ? total_bytes_written : -EIO;
			}
			busy_loop = 0;
			bytes_written ++;
		}
		total_bytes_written += copy_size;
		buf += copy_size;
		count -= copy_size;					
	}while(count > 0);	
	return total_bytes_written;
}

static int lp_ioctl (struct inode *inode, struct file *filp,
                 unsigned int cmd, unsigned long arg)
{
	unsigned int status;
	switch(cmd)
	{
		case IOCTL_GETSTATUS:
			status = get_status();
			if( (status & LP_PERRORP) != LP_PERRORP) return S_DEVICE_ERROR;
			else if(Is_Printing && (status & LP_POUTPA)) return S_NO_PAPER;
			else if(lp_offline(status)) return S_NOT_READY;
			break;
	}
	return S_READY;
}	                 

static ssize_t lp_write(struct file *filp, const char * buf, 
			  size_t count, loff_t *ppos)
{
	int retv;
	//Dprintk("%s : write %u bytes\n",lp_devname,count);

	if( (filp->f_flags & O_NONBLOCK) && Is_Printing )
	{
		Dprintk("printer is printing..\n");
		return -EAGAIN;
	}

	if(Is_Printing) Dprintk("printing..\n");
	
	if (down_interruptible (&lp_dev->sem)){
		return -ERESTARTSYS;
	}
	
	Is_Printing = 1;
	retv = lp_write_buf(lp_dev, buf, count);
	
	Is_Printing = 0;	
	
	up (&lp_dev->sem);
	//if(retv >= 0) Dprintk("%s : write %d bytes OK\n",lp_devname,retv);
	return retv;
}

static int lp_open(struct inode * inode, struct file *filp)
{

	//unsigned int status;
	//if(lp_dev->flags & LP_BUSY) return -EBUSY;
	
	/*
	status = get_status();
	if(status & LP_PBUSY) {
		Dprintk("%s: printer busy\n",DEV_NAME);
		return -EIO;
	}
	else if(!(status & LP_PERRORP)) {
		printk("%s: printer error\n",DEV_NAME);
		return -EIO;
	}
	*/

	//lp_dev->flags |= LP_BUSY;
	Dprintk("%s : open\n",DEV_NAME);
	
	//lp_reset();
#ifdef FOR_DEBUG
	//prn_count =0;
#endif
	//MOD_INC_USE_COUNT;
	
	return 0;
}

int lp_release(struct inode *inode, struct file *filp)
{
	Dprintk("%s : release\n",DEV_NAME);
	//MOD_DEC_USE_COUNT;
	//lp_dev->flags &= ~LP_BUSY;
	//Dprintk("total print: %u\n",prn_count);
	return 0;
}

int lp_read_proc(char *buf, char **start, off_t offset,
		int count, int *eof, void *data)
{
	unsigned int status;
	//printk("lp: read status\n");
	status = get_status();
	if( (status & LP_PERRORP) != LP_PERRORP)
		sprintf(buf,"3");
	else if(Is_Printing && (status & LP_POUTPA))
		sprintf(buf,"2");
	else if(lp_offline(status)) 
		sprintf(buf,"1");
	else
		sprintf(buf,"0");
	
	return 1;
}

static struct file_operations lp_fops = {
	llseek:		NULL,
	read:		NULL,
	write:		lp_write,
	mmap:		NULL,
	open:		lp_open,
	poll:		NULL,
	ioctl:		lp_ioctl,
	release:	lp_release,
	fasync:		NULL,
};

MODULE_PARM(ioport,"b");

int __init lp_init(void)
{
	int status;
	Dprintk("%s : init module\n",DEV_NAME);

    if( (status = check_region(ioport,DATA_WIDTH))<0 ){
		Dprintk("%s: ioport 0x%02x already in use\n",DEV_NAME,ioport);
		return status;
	}

	status = register_chrdev(DEV_MAJOR, DEV_NAME, &lp_fops);
	if (status < 0) {
		printk("%s: register_chrdev error\n",DEV_NAME); 
		return(status);
	}
	
	lp_dev = kmalloc(sizeof (lp_device), GFP_KERNEL);
	if (!lp_dev) 
	{
		unregister_chrdev(DEV_MAJOR, DEV_NAME);
		return  -ENOMEM;
	}
	memset((char*)lp_dev,0,sizeof(lp_device));
	lp_dev->flags = 0;
	lp_dev->lp_buffer = (char *)kmalloc( LP_BUFFER_SIZE , GFP_KERNEL);
	if(!lp_dev->lp_buffer)
	{
		unregister_chrdev(DEV_MAJOR, DEV_NAME);
		kfree(lp_dev);
		return  -ENOMEM;
	}
	memset(lp_dev->lp_buffer,0,LP_BUFFER_SIZE);
    
	sema_init (&lp_dev->sem, 1);	
	//init_waitqueue_head(&lp_dev->wait);

	/*
	enable_gpio(PRN_ACK,1);
	enable_gpio(PRN_BUSY,1);
	enable_gpio(PRN_PE,1);
	enable_gpio(PRN_SLCT,1);
	enable_gpio(PRN_ERROR,1);
	enable_gpio(PRN_INIT,1);
	enable_gpio(PRN_STROBE,1);
	*/

	//lp_set_strobe(0);
	lp_reset();

	set_cs1();
	
	#ifdef FOR_DEBUG
	get_cs1();
	
	printk("lp control: strobe = %d\n",get_gpio(PRN_STROBE));
	printk("lp control: init = %d\n",get_gpio(PRN_INIT));

	printk("lp status: busy = %d\n",get_gpio(PRN_BUSY));
	printk("lp status: pe = %d\n",get_gpio(PRN_PE));
	printk("lp status: slct = %d\n",get_gpio(PRN_SLCT));
	printk("lp status: error = %d\n",get_gpio(PRN_ERROR));
	printk("lp status: ack = %d\n",get_gpio(PRN_ACK));

	
	#endif

	request_region(ioport,DATA_WIDTH,"parallel");

	create_proc_read_entry(LP_PROC_NAME,0,NULL,lp_read_proc,NULL);
	
	return(0);
}	


void __exit lp_cleanup(void)
{
	Dprintk("%s : remove module\n",DEV_NAME);
	//if (devfs_handle != NULL)
	//	devfs_unregister(devfs_handle);

	remove_proc_entry(LP_PROC_NAME,NULL);
	
	unregister_chrdev(DEV_MAJOR, DEV_NAME);
	kfree(lp_dev->lp_buffer);
	kfree(lp_dev);

	/*
	enable_gpio(PRN_ACK,0);
	enable_gpio(PRN_BUSY,0);
	enable_gpio(PRN_PE,0);
	enable_gpio(PRN_SLCT,0);
	enable_gpio(PRN_ERROR,0);
	enable_gpio(PRN_INIT,0);
	enable_gpio(PRN_STROBE,0);
	*/

	release_region(ioport,DATA_WIDTH);
}	

module_init(lp_init);
module_exit(lp_cleanup);

MODULE_LICENSE("GPL");

