/*
 * Copyright (C) 1999-2002 Free Software Foundation, Inc.
 * This file is part of the GNU LIBICONV Library.
 *
 * The GNU LIBICONV Library is free software; you can redistribute it
 * and/or modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * The GNU LIBICONV Library is distributed in the hope that it will be
 * useful, but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with the GNU LIBICONV Library; see the file COPYING.LIB.
 * If not, write to the Free Software Foundation, Inc., 59 Temple Place -
 * Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * CP932 extensions
 */

static const unsigned short cp932ext_2uni_page81[188] = {
  /* 0x81 */
  0x3000, 0x3001, 0x3002, 0xff0c, 0xff0e, 0x30fb, 0xff1a, 0xff1b,
  0xff1f, 0xff01, 0x309b, 0x309c, 0x00b4, 0xff40, 0x00a8, 0xff3e,
  0xffe3, 0xff3f, 0x30fd, 0x30fe, 0x309d, 0x309e, 0x3003, 0x4edd,
  0x3005, 0x3006, 0x3007, 0x30fc, 0x2015, 0x2010, 0xff0f, 0xff3c,
  0xff5e, 0x2225, 0xff5c, 0x2026, 0x2025, 0x2018, 0x2019, 0x201c,
  0x201d, 0xff08, 0xff09, 0x3014, 0x3015, 0xff3b, 0xff3d, 0xff5b,
  0xff5d, 0x3008, 0x3009, 0x300a, 0x300b, 0x300c, 0x300d, 0x300e,
  0x300f, 0x3010, 0x3011, 0xff0b, 0xff0d, 0x00b1, 0x00d7, 0x00f7,
  0xff1d, 0x2260, 0xff1c, 0xff1e, 0x2266, 0x2267, 0x221e, 0x2234,
  0x2642, 0x2640, 0x00b0, 0x2032, 0x2033, 0x2103, 0xffe5, 0xff04,
  0xffe0, 0xffe1, 0xff05, 0xff03, 0xff06, 0xff0a, 0xff20, 0x00a7,
  0x2606, 0x2605, 0x25cb, 0x25cf, 0x25ce, 0x25c7, 0x25c6, 0x25a1,
  0x25a0, 0x25b3, 0x25b2, 0x25bd, 0x25bc, 0x203b, 0x3012, 0x2192,
  0x2190, 0x2191, 0x2193, 0x3013, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
  0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x2208,
  0x220b, 0x2286, 0x2287, 0x2282, 0x2283, 0x222a, 0x2229, 0xfffd,
  0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x2227,
  0x2228, 0xffe2, 0x21d2, 0x21d4, 0x2200, 0x2203, 0xfffd, 0xfffd,
  0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd,
  0xfffd, 0x2220, 0x22a5, 0x2312, 0x2202, 0x2207, 0x2261, 0x2252,
  0x226a, 0x226b, 0x221a, 0x223d, 0x221d, 0x2235, 0x222b, 0x222c,
  0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x212b,
  0x2030, 0x266f, 0x266d, 0x266a, 0x2020, 0x2021, 0x00b6, 0xfffd,
  0xfffd, 0xfffd, 0xfffd, 0x25ef,
};
static const unsigned short cp932ext_2uni_page87[92] = {
  /* 0x87 */
  0x2460, 0x2461, 0x2462, 0x2463, 0x2464, 0x2465, 0x2466, 0x2467,
  0x2468, 0x2469, 0x246a, 0x246b, 0x246c, 0x246d, 0x246e, 0x246f,
  0x2470, 0x2471, 0x2472, 0x2473, 0x2160, 0x2161, 0x2162, 0x2163,
  0x2164, 0x2165, 0x2166, 0x2167, 0x2168, 0x2169, 0xfffd, 0x3349,
  0x3314, 0x3322, 0x334d, 0x3318, 0x3327, 0x3303, 0x3336, 0x3351,
  0x3357, 0x330d, 0x3326, 0x3323, 0x332b, 0x334a, 0x333b, 0x339c,
  0x339d, 0x339e, 0x338e, 0x338f, 0x33c4, 0x33a1, 0xfffd, 0xfffd,
  0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0xfffd, 0x337b, 0x301d,
  0x301f, 0x2116, 0x33cd, 0x2121, 0x32a4, 0x32a5, 0x32a6, 0x32a7,
  0x32a8, 0x3231, 0x3232, 0x3239, 0x337e, 0x337d, 0x337c, 0x2252,
  0x2261, 0x222b, 0x222e, 0x2211, 0x221a, 0x22a5, 0x2220, 0x221f,
  0x22bf, 0x2235, 0x2229, 0x222a,
};
static const unsigned short cp932ext_2uni_pageed[376] = {
  /* 0xed */
  0x7e8a, 0x891c, 0x9348, 0x9288, 0x84dc, 0x4fc9, 0x70bb, 0x6631,
  0x68c8, 0x92f9, 0x66fb, 0x5f45, 0x4e28, 0x4ee1, 0x4efc, 0x4f00,
  0x4f03, 0x4f39, 0x4f56, 0x4f92, 0x4f8a, 0x4f9a, 0x4f94, 0x4fcd,
  0x5040, 0x5022, 0x4fff, 0x501e, 0x5046, 0x5070, 0x5042, 0x5094,
  0x50f4, 0x50d8, 0x514a, 0x5164, 0x519d, 0x51be, 0x51ec, 0x5215,
  0x529c, 0x52a6, 0x52c0, 0x52db, 0x5300, 0x5307, 0x5324, 0x5372,
  0x5393, 0x53b2, 0x53dd, 0xfa0e, 0x549c, 0x548a, 0x54a9, 0x54ff,
  0x5586, 0x5759, 0x5765, 0x57ac, 0x57c8, 0x57c7, 0xfa0f, 0xfa10,
  0x589e, 0x58b2, 0x590b, 0x5953, 0x595b, 0x595d, 0x5963, 0x59a4,
  0x59ba, 0x5b56, 0x5bc0, 0x752f, 0x5bd8, 0x5bec, 0x5c1e, 0x5ca6,
  0x5cba, 0x5cf5, 0x5d27, 0x5d53, 0xfa11, 0x5d42, 0x5d6d, 0x5db8,
  0x5db9, 0x5dd0, 0x5f21, 0x5f34, 0x5f67, 0x5fb7, 0x5fde, 0x605d,
  0x6085, 0x608a, 0x60de, 0x60d5, 0x6120, 0x60f2, 0x6111, 0x6137,
  0x6130, 0x6198, 0x6213, 0x62a6, 0x63f5, 0x6460, 0x649d, 0x64ce,
  0x654e, 0x6600, 0x6615, 0x663b, 0x6609, 0x662e, 0x661e, 0x6624,
  0x6665, 0x6657, 0x6659, 0xfa12, 0x6673, 0x6699, 0x66a0, 0x66b2,
  0x66bf, 0x66fa, 0x670e, 0xf929, 0x6766, 0x67bb, 0x6852, 0x67c0,
  0x6801, 0x6844, 0x68cf, 0xfa13, 0x6968, 0xfa14, 0x6998, 0x69e2,
  0x6a30, 0x6a6b, 0x6a46, 0x6a73, 0x6a7e, 0x6ae2, 0x6ae4, 0x6bd6,
  0x6c3f, 0x6c5c, 0x6c86, 0x6c6f, 0x6cda, 0x6d04, 0x6d87, 0x6d6f,
  0x6d96, 0x6dac, 0x6dcf, 0x6df8, 0x6df2, 0x6dfc, 0x6e39, 0x6e5c,
  0x6e27, 0x6e3c, 0x6ebf, 0x6f88, 0x6fb5, 0x6ff5, 0x7005, 0x7007,
  0x7028, 0x7085, 0x70ab, 0x710f, 0x7104, 0x715c, 0x7146, 0x7147,
  0xfa15, 0x71c1, 0x71fe, 0x72b1,
  /* 0xee */
  0x72be, 0x7324, 0xfa16, 0x7377, 0x73bd, 0x73c9, 0x73d6, 0x73e3,
  0x73d2, 0x7407, 0x73f5, 0x7426, 0x742a, 0x7429, 0x742e, 0x7462,
  0x7489, 0x749f, 0x7501, 0x756f, 0x7682, 0x769c, 0x769e, 0x769b,
  0x76a6, 0xfa17, 0x7746, 0x52af, 0x7821, 0x784e, 0x7864, 0x787a,
  0x7930, 0xfa18, 0xfa19, 0xfa1a, 0x7994, 0xfa1b, 0x799b, 0x7ad1,
  0x7ae7, 0xfa1c, 0x7aeb, 0x7b9e, 0xfa1d, 0x7d48, 0x7d5c, 0x7db7,
  0x7da0, 0x7dd6, 0x7e52, 0x7f47, 0x7fa1, 0xfa1e, 0x8301, 0x8362,
  0x837f, 0x83c7, 0x83f6, 0x8448, 0x84b4, 0x8553, 0x8559, 0x856b,
  0xfa1f, 0x85b0, 0xfa20, 0xfa21, 0x8807, 0x88f5, 0x8a12, 0x8a37,
  0x8a79, 0x8aa7, 0x8abe, 0x8adf, 0xfa22, 0x8af6, 0x8b53, 0x8b7f,
  0x8cf0, 0x8cf4, 0x8d12, 0x8d76, 0xfa23, 0x8ecf, 0xfa24, 0xfa25,
  0x9067, 0x90de, 0xfa26, 0x9115, 0x9127, 0x91da, 0x91d7, 0x91de,
  0x91ed, 0x91ee, 0x91e4, 0x91e5, 0x9206, 0x9210, 0x920a, 0x923a,
  0x9240, 0x923c, 0x924e, 0x9259, 0x9251, 0x9239, 0x9267, 0x92a7,
  0x9277, 0x9278, 0x92e7, 0x92d7, 0x92d9, 0x92d0, 0xfa27, 0x92d5,
  0x92e0, 0x92d3, 0x9325, 0x9321, 0x92fb, 0xfa28, 0x931e, 0x92ff,
  0x931d, 0x9302, 0x9370, 0x9357, 0x93a4, 0x93c6, 0x93de, 0x93f8,
  0x9431, 0x9445, 0x9448, 0x9592, 0xf9dc, 0xfa29, 0x969d, 0x96af,
  0x9733, 0x973b, 0x9743, 0x974d, 0x974f, 0x9751, 0x9755, 0x9857,
  0x9865, 0xfa2a, 0xfa2b, 0x9927, 0xfa2c, 0x999e, 0x9a4e, 0x9ad9,
  0x9adc, 0x9b75, 0x9b72, 0x9b8f, 0x9bb1, 0x9bbb, 0x9c00, 0x9d70,
  0x9d6b, 0xfa2d, 0x9e19, 0x9ed1, 0xfffd, 0xfffd, 0x2170, 0x2171,
  0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177, 0x2178, 0x2179,
  0xffe2, 0xffe4, 0xff07, 0xff02,
};
static const unsigned short cp932ext_2uni_pagefa[388] = {
  /* 0xfa */
  0x2170, 0x2171, 0x2172, 0x2173, 0x2174, 0x2175, 0x2176, 0x2177,
  0x2178, 0x2179, 0x2160, 0x2161, 0x2162, 0x2163, 0x2164, 0x2165,
  0x2166, 0x2167, 0x2168, 0x2169, 0xffe2, 0xffe4, 0xff07, 0xff02,
  0x3231, 0x2116, 0x2121, 0x2235, 0x7e8a, 0x891c, 0x9348, 0x9288,
  0x84dc, 0x4fc9, 0x70bb, 0x6631, 0x68c8, 0x92f9, 0x66fb, 0x5f45,
  0x4e28, 0x4ee1, 0x4efc, 0x4f00, 0x4f03, 0x4f39, 0x4f56, 0x4f92,
  0x4f8a, 0x4f9a, 0x4f94, 0x4fcd, 0x5040, 0x5022, 0x4fff, 0x501e,
  0x5046, 0x5070, 0x5042, 0x5094, 0x50f4, 0x50d8, 0x514a, 0x5164,
  0x519d, 0x51be, 0x51ec, 0x5215, 0x529c, 0x52a6, 0x52c0, 0x52db,
  0x5300, 0x5307, 0x5324, 0x5372, 0x5393, 0x53b2, 0x53dd, 0xfa0e,
  0x549c, 0x548a, 0x54a9, 0x54ff, 0x5586, 0x5759, 0x5765, 0x57ac,
  0x57c8, 0x57c7, 0xfa0f, 0xfa10, 0x589e, 0x58b2, 0x590b, 0x5953,
  0x595b, 0x595d, 0x5963, 0x59a4, 0x59ba, 0x5b56, 0x5bc0, 0x752f,
  0x5bd8, 0x5bec, 0x5c1e, 0x5ca6, 0x5cba, 0x5cf5, 0x5d27, 0x5d53,
  0xfa11, 0x5d42, 0x5d6d, 0x5db8, 0x5db9, 0x5dd0, 0x5f21, 0x5f34,
  0x5f67, 0x5fb7, 0x5fde, 0x605d, 0x6085, 0x608a, 0x60de, 0x60d5,
  0x6120, 0x60f2, 0x6111, 0x6137, 0x6130, 0x6198, 0x6213, 0x62a6,
  0x63f5, 0x6460, 0x649d, 0x64ce, 0x654e, 0x6600, 0x6615, 0x663b,
  0x6609, 0x662e, 0x661e, 0x6624, 0x6665, 0x6657, 0x6659, 0xfa12,
  0x6673, 0x6699, 0x66a0, 0x66b2, 0x66bf, 0x66fa, 0x670e, 0xf929,
  0x6766, 0x67bb, 0x6852, 0x67c0, 0x6801, 0x6844, 0x68cf, 0xfa13,
  0x6968, 0xfa14, 0x6998, 0x69e2, 0x6a30, 0x6a6b, 0x6a46, 0x6a73,
  0x6a7e, 0x6ae2, 0x6ae4, 0x6bd6, 0x6c3f, 0x6c5c, 0x6c86, 0x6c6f,
  0x6cda, 0x6d04, 0x6d87, 0x6d6f,
  /* 0xfb */
  0x6d96, 0x6dac, 0x6dcf, 0x6df8, 0x6df2, 0x6dfc, 0x6e39, 0x6e5c,
  0x6e27, 0x6e3c, 0x6ebf, 0x6f88, 0x6fb5, 0x6ff5, 0x7005, 0x7007,
  0x7028, 0x7085, 0x70ab, 0x710f, 0x7104, 0x715c, 0x7146, 0x7147,
  0xfa15, 0x71c1, 0x71fe, 0x72b1, 0x72be, 0x7324, 0xfa16, 0x7377,
  0x73bd, 0x73c9, 0x73d6, 0x73e3, 0x73d2, 0x7407, 0x73f5, 0x7426,
  0x742a, 0x7429, 0x742e, 0x7462, 0x7489, 0x749f, 0x7501, 0x756f,
  0x7682, 0x769c, 0x769e, 0x769b, 0x76a6, 0xfa17, 0x7746, 0x52af,
  0x7821, 0x784e, 0x7864, 0x787a, 0x7930, 0xfa18, 0xfa19, 0xfa1a,
  0x7994, 0xfa1b, 0x799b, 0x7ad1, 0x7ae7, 0xfa1c, 0x7aeb, 0x7b9e,
  0xfa1d, 0x7d48, 0x7d5c, 0x7db7, 0x7da0, 0x7dd6, 0x7e52, 0x7f47,
  0x7fa1, 0xfa1e, 0x8301, 0x8362, 0x837f, 0x83c7, 0x83f6, 0x8448,
  0x84b4, 0x8553, 0x8559, 0x856b, 0xfa1f, 0x85b0, 0xfa20, 0xfa21,
  0x8807, 0x88f5, 0x8a12, 0x8a37, 0x8a79, 0x8aa7, 0x8abe, 0x8adf,
  0xfa22, 0x8af6, 0x8b53, 0x8b7f, 0x8cf0, 0x8cf4, 0x8d12, 0x8d76,
  0xfa23, 0x8ecf, 0xfa24, 0xfa25, 0x9067, 0x90de, 0xfa26, 0x9115,
  0x9127, 0x91da, 0x91d7, 0x91de, 0x91ed, 0x91ee, 0x91e4, 0x91e5,
  0x9206, 0x9210, 0x920a, 0x923a, 0x9240, 0x923c, 0x924e, 0x9259,
  0x9251, 0x9239, 0x9267, 0x92a7, 0x9277, 0x9278, 0x92e7, 0x92d7,
  0x92d9, 0x92d0, 0xfa27, 0x92d5, 0x92e0, 0x92d3, 0x9325, 0x9321,
  0x92fb, 0xfa28, 0x931e, 0x92ff, 0x931d, 0x9302, 0x9370, 0x9357,
  0x93a4, 0x93c6, 0x93de, 0x93f8, 0x9431, 0x9445, 0x9448, 0x9592,
  0xf9dc, 0xfa29, 0x969d, 0x96af, 0x9733, 0x973b, 0x9743, 0x974d,
  0x974f, 0x9751, 0x9755, 0x9857, 0x9865, 0xfa2a, 0xfa2b, 0x9927,
  0xfa2c, 0x999e, 0x9a4e, 0x9ad9,
  /* 0xfc */
  0x9adc, 0x9b75, 0x9b72, 0x9b8f, 0x9bb1, 0x9bbb, 0x9c00, 0x9d70,
  0x9d6b, 0xfa2d, 0x9e19, 0x9ed1,
};

static int
cp932ext_mbtowc (conv_t conv, ucs4_t *pwc, const unsigned char *s, int n)
{
  unsigned char c1 = s[0];
  if ((c1 == 0x81) || (c1 == 0x87) || (c1 >= 0xed && c1 <= 0xee) || (c1 >= 0xfa && c1 <= 0xfc)) {
    if (n >= 2) {
      unsigned char c2 = s[1];
      if ((c2 >= 0x40 && c2 < 0x7f) || (c2 >= 0x80 && c2 < 0xfd)) {
        unsigned int i = 188 * (c1 - (c1 >= 0xe0 ? 0xc1 : 0x81)) + (c2 - (c2 >= 0x80 ? 0x41 : 0x40));
        unsigned short wc = 0xfffd;
        if (i < 1128) {
          if (i < 188)
            wc = cp932ext_2uni_page81[i];
        } else if (i < 8272) {
          if (i < 1220)
            wc = cp932ext_2uni_page87[i-1128];
        } else if (i < 10716) {
          if (i < 8648)
            wc = cp932ext_2uni_pageed[i-8272];
        } else {
          if (i < 11104)
            wc = cp932ext_2uni_pagefa[i-10716];
        }
        if (wc != 0xfffd) {
          *pwc = (ucs4_t) wc;
          return 2;
        }
      }
      return RET_ILSEQ;
    }
    return RET_TOOFEW(0);
  }
  return RET_ILSEQ;
}

static const unsigned short cp932ext_2charset[594] = {
  0x8198, 0x814e, 0x818b, 0x817d, 0x814c, 0x81f7, 0x817e, 0x8180,
  0x815d, 0x815c, 0x8165, 0x8166, 0x8167, 0x8168, 0x81f5, 0x81f6,
  0x8164, 0x8163, 0x81f1, 0x818c, 0x818d, 0x81a6, 0x818e, 0x8782,
  0x8784, 0x81f0, 0x8754, 0x8755, 0x8756, 0x8757, 0x8758, 0x8759,
  0x875a, 0x875b, 0x875c, 0x875d, 0xfa40, 0xfa41, 0xfa42, 0xfa43,
  0xfa44, 0xfa45, 0xfa46, 0xfa47, 0xfa48, 0xfa49, 0x81a9, 0x81aa,
  0x81a8, 0x81ab, 0x81cb, 0x81cc, 0x81cd, 0x81dd, 0x81ce, 0x81de,
  0x81b8, 0x81b9, 0x8794, 0x81e3, 0x81e5, 0x8187, 0x8798, 0x81da,
  0x8161, 0x81c8, 0x81c9, 0x81bf, 0x81be, 0x81e7, 0x81e8, 0x8793,
  0x8188, 0x81e6, 0x81e4, 0x81e0, 0x8182, 0x81df, 0x8185, 0x8186,
  0x81e1, 0x81e2, 0x81bc, 0x81bd, 0x81ba, 0x81bb, 0x81db, 0x8799,
  0x81dc, 0x8740, 0x8741, 0x8742, 0x8743, 0x8744, 0x8745, 0x8746,
  0x8747, 0x8748, 0x8749, 0x874a, 0x874b, 0x874c, 0x874d, 0x874e,
  0x874f, 0x8750, 0x8751, 0x8752, 0x8753, 0x81a1, 0x81a0, 0x81a3,
  0x81a2, 0x81a5, 0x81a4, 0x819f, 0x819e, 0x819b, 0x819d, 0x819c,
  0x81fc, 0x819a, 0x8199, 0x818a, 0x8189, 0x81f4, 0x81f3, 0x81f2,
  0x8140, 0x8141, 0x8142, 0x8156, 0x8158, 0x8159, 0x815a, 0x8171,
  0x8172, 0x8173, 0x8174, 0x8175, 0x8176, 0x8177, 0x8178, 0x8179,
  0x817a, 0x81a7, 0x81ac, 0x816b, 0x816c, 0x8780, 0x8781, 0x814a,
  0x814b, 0x8154, 0x8155, 0x8145, 0x815b, 0x8152, 0x8153, 0x878a,
  0x878b, 0x878c, 0x8785, 0x8786, 0x8787, 0x8788, 0x8789, 0x8765,
  0x8769, 0x8760, 0x8763, 0x8761, 0x876b, 0x876a, 0x8764, 0x876c,
  0x8766, 0x876e, 0x875f, 0x876d, 0x8762, 0x8767, 0x8768, 0x877e,
  0x878f, 0x878e, 0x878d, 0x8772, 0x8773, 0x876f, 0x8770, 0x8771,
  0x8775, 0x8774, 0x8783, 0xfa68, 0x8157, 0xfa69, 0xfa6a, 0xfa6b,
  0xfa6c, 0xfa6d, 0xfa6e, 0xfa70, 0xfa6f, 0xfa72, 0xfa71, 0xfa61,
  0xfa73, 0xfa76, 0xfa77, 0xfa75, 0xfa74, 0xfa7a, 0xfa78, 0xfa79,
  0xfa7b, 0xfa7d, 0xfa7c, 0xfa7e, 0xfa80, 0xfa81, 0xfa82, 0xfa83,
  0xfa84, 0xfa85, 0xfa86, 0xfb77, 0xfa87, 0xfa88, 0xfa89, 0xfa8a,
  0xfa8b, 0xfa8c, 0xfa8d, 0xfa8e, 0xfa8f, 0xfa92, 0xfa91, 0xfa93,
  0xfa94, 0xfa95, 0xfa96, 0xfa97, 0xfa98, 0xfa9a, 0xfa99, 0xfa9d,
  0xfa9e, 0xfa9f, 0xfaa0, 0xfaa1, 0xfaa2, 0xfaa3, 0xfaa4, 0xfaa5,
  0xfaa6, 0xfaa7, 0xfaa9, 0xfaaa, 0xfaab, 0xfaac, 0xfaad, 0xfaae,
  0xfaaf, 0xfab2, 0xfab0, 0xfab3, 0xfab4, 0xfab5, 0xfab6, 0xfab7,
  0xfab8, 0xfa67, 0xfab9, 0xfaba, 0xfabb, 0xfabc, 0xfabd, 0xfabe,
  0xfac0, 0xfabf, 0xfac2, 0xfac3, 0xfac1, 0xfac5, 0xfac4, 0xfac6,
  0xfac7, 0xfac8, 0xfac9, 0xfaca, 0xfacb, 0xfacc, 0xfacd, 0xface,
  0xfad1, 0xfacf, 0xfad3, 0xfad4, 0xfad2, 0xfa63, 0xfad0, 0xfad6,
  0xfad7, 0xfad5, 0xfad9, 0xfada, 0xfadb, 0xfadc, 0xfadd, 0xfade,
  0xfa66, 0xfadf, 0xfae1, 0xfae2, 0xfae4, 0xfae5, 0xfae6, 0xfae3,
  0xfa64, 0xfae7, 0xfae9, 0xfaeb, 0xfaec, 0xfaed, 0xfaef, 0xfaee,
  0xfaf0, 0xfaf1, 0xfaf2, 0xfaf3, 0xfaf4, 0xfaf5, 0xfaf6, 0xfaf8,
  0xfaf7, 0xfaf9, 0xfafa, 0xfafc, 0xfafb, 0xfb40, 0xfb41, 0xfb42,
  0xfb44, 0xfb43, 0xfb45, 0xfb48, 0xfb46, 0xfb49, 0xfb47, 0xfb4a,
  0xfb4b, 0xfb4c, 0xfb4d, 0xfb4e, 0xfb4f, 0xfb50, 0xfb51, 0xfb52,
  0xfa62, 0xfb54, 0xfb53, 0xfb56, 0xfb57, 0xfb55, 0xfb59, 0xfb5a,
  0xfb5b, 0xfb5c, 0xfb5d, 0xfb5f, 0xfb60, 0xfb61, 0xfb64, 0xfb62,
  0xfb63, 0xfb66, 0xfb65, 0xfb67, 0xfb69, 0xfb68, 0xfb6a, 0xfb6b,
  0xfb6c, 0xfb6d, 0xfb6e, 0xfaa8, 0xfb6f, 0xfb70, 0xfb73, 0xfb71,
  0xfb72, 0xfb74, 0xfb76, 0xfb78, 0xfb79, 0xfb7a, 0xfb7b, 0xfb7c,
  0xfb81, 0xfb83, 0xfb84, 0xfb85, 0xfb87, 0xfb88, 0xfb8a, 0xfb8b,
  0xfb8d, 0xfb8c, 0xfb8e, 0xfb8f, 0xfa5c, 0xfb90, 0xfb91, 0xfb93,
  0xfb94, 0xfb95, 0xfb96, 0xfb97, 0xfb98, 0xfb99, 0xfa60, 0xfb9a,
  0xfb9b, 0xfb9c, 0xfb9e, 0xfba1, 0xfba2, 0xfa5d, 0xfba3, 0xfba4,
  0xfba5, 0xfba6, 0xfba7, 0xfba8, 0xfbaa, 0xfbab, 0xfbac, 0xfbad,
  0xfbae, 0xfbaf, 0xfbb0, 0xfbb2, 0xfbb5, 0xfbb6, 0xfbb8, 0xfbb9,
  0xfbbb, 0xfbba, 0xfbbc, 0xfbbf, 0xfbc0, 0xfbbd, 0xfbbe, 0xfbc1,
  0xfbc3, 0xfbc2, 0xfbca, 0xfbc4, 0xfbc6, 0xfbc5, 0xfbc7, 0xfbc9,
  0xfbc8, 0xfbcb, 0xfbcd, 0xfbce, 0xfa5f, 0xfbcc, 0xfbd2, 0xfbd6,
  0xfbd4, 0xfbd0, 0xfbd1, 0xfbd5, 0xfbcf, 0xfa65, 0xfbd9, 0xfbdc,
  0xfbde, 0xfbdd, 0xfbdb, 0xfbd8, 0xfbd7, 0xfa5e, 0xfbe0, 0xfbdf,
  0xfbe1, 0xfbe2, 0xfbe3, 0xfbe4, 0xfbe5, 0xfbe6, 0xfbe7, 0xfbe8,
  0xfbeb, 0xfbec, 0xfbed, 0xfbee, 0xfbef, 0xfbf0, 0xfbf1, 0xfbf2,
  0xfbf3, 0xfbf4, 0xfbf5, 0xfbf8, 0xfbfa, 0xfbfb, 0xfbfc, 0xfc40,
  0xfc42, 0xfc41, 0xfc43, 0xfc44, 0xfc45, 0xfc46, 0xfc48, 0xfc47,
  0xfc4a, 0xfc4b, 0xfae0, 0xfbe9, 0xfa90, 0xfa9b, 0xfa9c, 0xfab1,
  0xfad8, 0xfae8, 0xfaea, 0xfb58, 0xfb5e, 0xfb75, 0xfb7d, 0xfb7e,
  0xfb80, 0xfb82, 0xfb86, 0xfb89, 0xfb92, 0xfb9d, 0xfb9f, 0xfba0,
  0xfba9, 0xfbb1, 0xfbb3, 0xfbb4, 0xfbb7, 0xfbd3, 0xfbda, 0xfbea,
  0xfbf6, 0xfbf7, 0xfbf9, 0xfc49, 0x8149, 0xfa57, 0x8194, 0x8190,
  0x8193, 0x8195, 0xfa56, 0x8169, 0x816a, 0x8196, 0x817b, 0x8143,
  0x817c, 0x8144, 0x815e, 0x8146, 0x8147, 0x8183, 0x8181, 0x8184,
  0x8148, 0x8197, 0x816d, 0x815f, 0x816e, 0x814f, 0x8151, 0x814d,
  0x816f, 0x8162, 0x8170, 0x8160, 0x8191, 0x8192, 0x81ca, 0x8150,
  0xfa55, 0x818f,
};

static const Summary16 cp932ext_uni2indx_page00[16] = {
  /* 0x0000 */
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 },
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0000 },
  {    0, 0x0000 }, {    0, 0x0000 }, {    0, 0x0180 }, {    2, 0x0053 },
  {    6, 0x0000 }, {    6, 0x0080 }, {    7, 0x0000 }, {    7, 0x0080 },
};
static const Summary16 cp932ext_uni2indx_page20[103] = {
  /* 0x2000 */
  {    8, 0x0000 }, {    8, 0x3321 }, {   14, 0x0063 }, {   18, 0x080d },
  {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 },
  {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 },
  {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 }, {   22, 0x0000 },
  /* 0x2100 */
  {   22, 0x0008 }, {   23, 0x0040 }, {   24, 0x0802 }, {   26, 0x0000 },
  {   26, 0x0000 }, {   26, 0x0000 }, {   26, 0x03ff }, {   36, 0x03ff },
  {   46, 0x0000 }, {   46, 0x000f }, {   50, 0x0000 }, {   50, 0x0000 },
  {   50, 0x0000 }, {   50, 0x0014 }, {   52, 0x0000 }, {   52, 0x0000 },
  /* 0x2200 */
  {   52, 0x098d }, {   58, 0xe402 }, {   63, 0x5fa1 }, {   72, 0x2030 },
  {   75, 0x0000 }, {   75, 0x0004 }, {   76, 0x0cc3 }, {   82, 0x0000 },
  {   82, 0x00cc }, {   86, 0x0000 }, {   86, 0x0020 }, {   87, 0x8000 },
  {   88, 0x0000 }, {   88, 0x0000 }, {   88, 0x0000 }, {   88, 0x0000 },
  /* 0x2300 */
  {   88, 0x0000 }, {   88, 0x0004 }, {   89, 0x0000 }, {   89, 0x0000 },
  {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 },
  {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 },
  {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 },
  /* 0x2400 */
  {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0x0000 },
  {   89, 0x0000 }, {   89, 0x0000 }, {   89, 0xffff }, {  105, 0x000f },
  {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 },
  {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 },
  /* 0x2500 */
  {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 },
  {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0000 },
  {  109, 0x0000 }, {  109, 0x0000 }, {  109, 0x0003 }, {  111, 0x300c },
  {  115, 0xc8c0 }, {  120, 0x0000 }, {  120, 0x8000 }, {  121, 0x0000 },
  /* 0x2600 */
  {  121, 0x0060 }, {  123, 0x0000 }, {  123, 0x0000 }, {  123, 0x0000 },
  {  123, 0x0005 }, {  125, 0x0000 }, {  125, 0xa400 },
};
static const Summary16 cp932ext_uni2indx_page30[16] = {
  /* 0x3000 */
  {  128, 0xffef }, {  143, 0xa03f }, {  151, 0x0000 }, {  151, 0x0000 },
  {  151, 0x0000 }, {  151, 0x0000 }, {  151, 0x0000 }, {  151, 0x0000 },
  {  151, 0x0000 }, {  151, 0x7800 }, {  155, 0x0000 }, {  155, 0x0000 },
  {  155, 0x0000 }, {  155, 0x0000 }, {  155, 0x0000 }, {  155, 0x7800 },
};
static const Summary16 cp932ext_uni2indx_page32[29] = {
  /* 0x3200 */
  {  159, 0x0000 }, {  159, 0x0000 }, {  159, 0x0000 }, {  159, 0x0206 },
  {  162, 0x0000 }, {  162, 0x0000 }, {  162, 0x0000 }, {  162, 0x0000 },
  {  162, 0x0000 }, {  162, 0x0000 }, {  162, 0x01f0 }, {  167, 0x0000 },
  {  167, 0x0000 }, {  167, 0x0000 }, {  167, 0x0000 }, {  167, 0x0000 },
  /* 0x3300 */
  {  167, 0x2008 }, {  169, 0x0110 }, {  171, 0x08cc }, {  176, 0x0840 },
  {  178, 0x2600 }, {  181, 0x0082 }, {  183, 0x0000 }, {  183, 0x7800 },
  {  187, 0xc000 }, {  189, 0x7000 }, {  192, 0x0002 }, {  193, 0x0000 },
  {  193, 0x2010 },
};
static const Summary16 cp932ext_uni2indx_page4e[121] = {
  /* 0x4e00 */
  {  195, 0x0000 }, {  195, 0x0000 }, {  195, 0x0100 }, {  196, 0x0000 },
  {  196, 0x0000 }, {  196, 0x0000 }, {  196, 0x0000 }, {  196, 0x0000 },
  {  196, 0x0000 }, {  196, 0x0000 }, {  196, 0x0000 }, {  196, 0x0000 },
  {  196, 0x0000 }, {  196, 0x2000 }, {  197, 0x0002 }, {  198, 0x1000 },
  /* 0x4f00 */
  {  199, 0x0009 }, {  201, 0x0000 }, {  201, 0x0000 }, {  201, 0x0200 },
  {  202, 0x0000 }, {  202, 0x0040 }, {  203, 0x0000 }, {  203, 0x0000 },
  {  203, 0x0400 }, {  204, 0x0414 }, {  207, 0x0000 }, {  207, 0x0000 },
  {  207, 0x2200 }, {  209, 0x0000 }, {  209, 0x0000 }, {  209, 0x8000 },
  /* 0x5000 */
  {  210, 0x0000 }, {  210, 0x4000 }, {  211, 0x0004 }, {  212, 0x0000 },
  {  212, 0x0045 }, {  215, 0x0000 }, {  215, 0x0000 }, {  215, 0x0001 },
  {  216, 0x0000 }, {  216, 0x0010 }, {  217, 0x0000 }, {  217, 0x0000 },
  {  217, 0x0000 }, {  217, 0x0100 }, {  218, 0x0000 }, {  218, 0x0010 },
  /* 0x5100 */
  {  219, 0x0000 }, {  219, 0x0000 }, {  219, 0x0000 }, {  219, 0x0000 },
  {  219, 0x0400 }, {  220, 0x0000 }, {  220, 0x0010 }, {  221, 0x0000 },
  {  221, 0x0000 }, {  221, 0x2000 }, {  222, 0x0000 }, {  222, 0x4000 },
  {  223, 0x0000 }, {  223, 0x0000 }, {  223, 0x1000 }, {  224, 0x0000 },
  /* 0x5200 */
  {  224, 0x0000 }, {  224, 0x0020 }, {  225, 0x0000 }, {  225, 0x0000 },
  {  225, 0x0000 }, {  225, 0x0000 }, {  225, 0x0000 }, {  225, 0x0000 },
  {  225, 0x0000 }, {  225, 0x1000 }, {  226, 0x8040 }, {  228, 0x0000 },
  {  228, 0x0001 }, {  229, 0x0800 }, {  230, 0x0000 }, {  230, 0x0000 },
  /* 0x5300 */
  {  230, 0x0081 }, {  232, 0x0000 }, {  232, 0x0010 }, {  233, 0x0000 },
  {  233, 0x0000 }, {  233, 0x0000 }, {  233, 0x0000 }, {  233, 0x0004 },
  {  234, 0x0000 }, {  234, 0x0008 }, {  235, 0x0000 }, {  235, 0x0004 },
  {  236, 0x0000 }, {  236, 0x2000 }, {  237, 0x0000 }, {  237, 0x0000 },
  /* 0x5400 */
  {  237, 0x0000 }, {  237, 0x0000 }, {  237, 0x0000 }, {  237, 0x0000 },
  {  237, 0x0000 }, {  237, 0x0000 }, {  237, 0x0000 }, {  237, 0x0000 },
  {  237, 0x0400 }, {  238, 0x1000 }, {  239, 0x0200 }, {  240, 0x0000 },
  {  240, 0x0000 }, {  240, 0x0000 }, {  240, 0x0000 }, {  240, 0x8000 },
  /* 0x5500 */
  {  241, 0x0000 }, {  241, 0x0000 }, {  241, 0x0000 }, {  241, 0x0000 },
  {  241, 0x0000 }, {  241, 0x0000 }, {  241, 0x0000 }, {  241, 0x0000 },
  {  241, 0x0040 },
};
static const Summary16 cp932ext_uni2indx_page57[44] = {
  /* 0x5700 */
  {  242, 0x0000 }, {  242, 0x0000 }, {  242, 0x0000 }, {  242, 0x0000 },
  {  242, 0x0000 }, {  242, 0x0200 }, {  243, 0x0020 }, {  244, 0x0000 },
  {  244, 0x0000 }, {  244, 0x0000 }, {  244, 0x1000 }, {  245, 0x0000 },
  {  245, 0x0180 }, {  247, 0x0000 }, {  247, 0x0000 }, {  247, 0x0000 },
  /* 0x5800 */
  {  247, 0x0000 }, {  247, 0x0000 }, {  247, 0x0000 }, {  247, 0x0000 },
  {  247, 0x0000 }, {  247, 0x0000 }, {  247, 0x0000 }, {  247, 0x0000 },
  {  247, 0x0000 }, {  247, 0x4000 }, {  248, 0x0000 }, {  248, 0x0004 },
  {  249, 0x0000 }, {  249, 0x0000 }, {  249, 0x0000 }, {  249, 0x0000 },
  /* 0x5900 */
  {  249, 0x0800 }, {  250, 0x0000 }, {  250, 0x0000 }, {  250, 0x0000 },
  {  250, 0x0000 }, {  250, 0x2808 }, {  253, 0x0008 }, {  254, 0x0000 },
  {  254, 0x0000 }, {  254, 0x0000 }, {  254, 0x0010 }, {  255, 0x0400 },
};
static const Summary16 cp932ext_uni2indx_page5b[46] = {
  /* 0x5b00 */
  {  256, 0x0000 }, {  256, 0x0000 }, {  256, 0x0000 }, {  256, 0x0000 },
  {  256, 0x0000 }, {  256, 0x0040 }, {  257, 0x0000 }, {  257, 0x0000 },
  {  257, 0x0000 }, {  257, 0x0000 }, {  257, 0x0000 }, {  257, 0x0000 },
  {  257, 0x0001 }, {  258, 0x0100 }, {  259, 0x1000 }, {  260, 0x0000 },
  /* 0x5c00 */
  {  260, 0x0000 }, {  260, 0x4000 }, {  261, 0x0000 }, {  261, 0x0000 },
  {  261, 0x0000 }, {  261, 0x0000 }, {  261, 0x0000 }, {  261, 0x0000 },
  {  261, 0x0000 }, {  261, 0x0000 }, {  261, 0x0040 }, {  262, 0x0400 },
  {  263, 0x0000 }, {  263, 0x0000 }, {  263, 0x0000 }, {  263, 0x0020 },
  /* 0x5d00 */
  {  264, 0x0000 }, {  264, 0x0000 }, {  264, 0x0080 }, {  265, 0x0000 },
  {  265, 0x0004 }, {  266, 0x0008 }, {  267, 0x2000 }, {  268, 0x0000 },
  {  268, 0x0000 }, {  268, 0x0000 }, {  268, 0x0000 }, {  268, 0x0300 },
  {  270, 0x0000 }, {  270, 0x0001 },
};
static const Summary16 cp932ext_uni2indx_page5f[458] = {
  /* 0x5f00 */
  {  271, 0x0000 }, {  271, 0x0000 }, {  271, 0x0002 }, {  272, 0x0010 },
  {  273, 0x0020 }, {  274, 0x0000 }, {  274, 0x0080 }, {  275, 0x0000 },
  {  275, 0x0000 }, {  275, 0x0000 }, {  275, 0x0000 }, {  275, 0x0080 },
  {  276, 0x0000 }, {  276, 0x4000 }, {  277, 0x0000 }, {  277, 0x0000 },
  /* 0x6000 */
  {  277, 0x0000 }, {  277, 0x0000 }, {  277, 0x0000 }, {  277, 0x0000 },
  {  277, 0x0000 }, {  277, 0x2000 }, {  278, 0x0000 }, {  278, 0x0000 },
  {  278, 0x0420 }, {  280, 0x0000 }, {  280, 0x0000 }, {  280, 0x0000 },
  {  280, 0x0000 }, {  280, 0x4020 }, {  282, 0x0000 }, {  282, 0x0004 },
  /* 0x6100 */
  {  283, 0x0000 }, {  283, 0x0002 }, {  284, 0x0001 }, {  285, 0x0081 },
  {  287, 0x0000 }, {  287, 0x0000 }, {  287, 0x0000 }, {  287, 0x0000 },
  {  287, 0x0000 }, {  287, 0x0100 }, {  288, 0x0000 }, {  288, 0x0000 },
  {  288, 0x0000 }, {  288, 0x0000 }, {  288, 0x0000 }, {  288, 0x0000 },
  /* 0x6200 */
  {  288, 0x0000 }, {  288, 0x0008 }, {  289, 0x0000 }, {  289, 0x0000 },
  {  289, 0x0000 }, {  289, 0x0000 }, {  289, 0x0000 }, {  289, 0x0000 },
  {  289, 0x0000 }, {  289, 0x0000 }, {  289, 0x0040 }, {  290, 0x0000 },
  {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 },
  /* 0x6300 */
  {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 },
  {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 },
  {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 },
  {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0000 }, {  290, 0x0020 },
  /* 0x6400 */
  {  291, 0x0000 }, {  291, 0x0000 }, {  291, 0x0000 }, {  291, 0x0000 },
  {  291, 0x0000 }, {  291, 0x0000 }, {  291, 0x0001 }, {  292, 0x0000 },
  {  292, 0x0000 }, {  292, 0x2000 }, {  293, 0x0000 }, {  293, 0x0000 },
  {  293, 0x4000 }, {  294, 0x0000 }, {  294, 0x0000 }, {  294, 0x0000 },
  /* 0x6500 */
  {  294, 0x0000 }, {  294, 0x0000 }, {  294, 0x0000 }, {  294, 0x0000 },
  {  294, 0x4000 }, {  295, 0x0000 }, {  295, 0x0000 }, {  295, 0x0000 },
  {  295, 0x0000 }, {  295, 0x0000 }, {  295, 0x0000 }, {  295, 0x0000 },
  {  295, 0x0000 }, {  295, 0x0000 }, {  295, 0x0000 }, {  295, 0x0000 },
  /* 0x6600 */
  {  295, 0x0201 }, {  297, 0x4020 }, {  299, 0x4010 }, {  301, 0x0802 },
  {  303, 0x0000 }, {  303, 0x0280 }, {  305, 0x0020 }, {  306, 0x0008 },
  {  307, 0x0000 }, {  307, 0x0200 }, {  308, 0x0001 }, {  309, 0x8004 },
  {  311, 0x0000 }, {  311, 0x0000 }, {  311, 0x0000 }, {  311, 0x0c00 },
  /* 0x6700 */
  {  313, 0x4000 }, {  314, 0x0000 }, {  314, 0x0000 }, {  314, 0x0000 },
  {  314, 0x0000 }, {  314, 0x0000 }, {  314, 0x0040 }, {  315, 0x0000 },
  {  315, 0x0000 }, {  315, 0x0000 }, {  315, 0x0000 }, {  315, 0x0800 },
  {  316, 0x0001 }, {  317, 0x0000 }, {  317, 0x0000 }, {  317, 0x0000 },
  /* 0x6800 */
  {  317, 0x0002 }, {  318, 0x0000 }, {  318, 0x0000 }, {  318, 0x0000 },
  {  318, 0x0010 }, {  319, 0x0004 }, {  320, 0x0000 }, {  320, 0x0000 },
  {  320, 0x0000 }, {  320, 0x0000 }, {  320, 0x0000 }, {  320, 0x0000 },
  {  320, 0x8100 }, {  322, 0x0000 }, {  322, 0x0000 }, {  322, 0x0000 },
  /* 0x6900 */
  {  322, 0x0000 }, {  322, 0x0000 }, {  322, 0x0000 }, {  322, 0x0000 },
  {  322, 0x0000 }, {  322, 0x0000 }, {  322, 0x0100 }, {  323, 0x0000 },
  {  323, 0x0000 }, {  323, 0x0100 }, {  324, 0x0000 }, {  324, 0x0000 },
  {  324, 0x0000 }, {  324, 0x0000 }, {  324, 0x0004 }, {  325, 0x0000 },
  /* 0x6a00 */
  {  325, 0x0000 }, {  325, 0x0000 }, {  325, 0x0000 }, {  325, 0x0001 },
  {  326, 0x0040 }, {  327, 0x0000 }, {  327, 0x0800 }, {  328, 0x4008 },
  {  330, 0x0000 }, {  330, 0x0000 }, {  330, 0x0000 }, {  330, 0x0000 },
  {  330, 0x0000 }, {  330, 0x0000 }, {  330, 0x0014 }, {  332, 0x0000 },
  /* 0x6b00 */
  {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 },
  {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 },
  {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 }, {  332, 0x0000 },
  {  332, 0x0000 }, {  332, 0x0040 }, {  333, 0x0000 }, {  333, 0x0000 },
  /* 0x6c00 */
  {  333, 0x0000 }, {  333, 0x0000 }, {  333, 0x0000 }, {  333, 0x8000 },
  {  334, 0x0000 }, {  334, 0x1000 }, {  335, 0x8000 }, {  336, 0x0000 },
  {  336, 0x0040 }, {  337, 0x0000 }, {  337, 0x0000 }, {  337, 0x0000 },
  {  337, 0x0000 }, {  337, 0x0400 }, {  338, 0x0000 }, {  338, 0x0000 },
  /* 0x6d00 */
  {  338, 0x0010 }, {  339, 0x0000 }, {  339, 0x0000 }, {  339, 0x0000 },
  {  339, 0x0000 }, {  339, 0x0000 }, {  339, 0x8000 }, {  340, 0x0000 },
  {  340, 0x0080 }, {  341, 0x0040 }, {  342, 0x1000 }, {  343, 0x0000 },
  {  343, 0x8000 }, {  344, 0x0000 }, {  344, 0x0000 }, {  344, 0x1104 },
  /* 0x6e00 */
  {  347, 0x0000 }, {  347, 0x0000 }, {  347, 0x0080 }, {  348, 0x1200 },
  {  350, 0x0000 }, {  350, 0x1000 }, {  351, 0x0000 }, {  351, 0x0000 },
  {  351, 0x0000 }, {  351, 0x0000 }, {  351, 0x0000 }, {  351, 0x8000 },
  {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 },
  /* 0x6f00 */
  {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 },
  {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 }, {  352, 0x0000 },
  {  352, 0x0100 }, {  353, 0x0000 }, {  353, 0x0000 }, {  353, 0x0020 },
  {  354, 0x0000 }, {  354, 0x0000 }, {  354, 0x0000 }, {  354, 0x0020 },
  /* 0x7000 */
  {  355, 0x00a0 }, {  357, 0x0000 }, {  357, 0x0100 }, {  358, 0x0000 },
  {  358, 0x0000 }, {  358, 0x0000 }, {  358, 0x0000 }, {  358, 0x0000 },
  {  358, 0x0020 }, {  359, 0x0000 }, {  359, 0x0800 }, {  360, 0x0800 },
  {  361, 0x0000 }, {  361, 0x0000 }, {  361, 0x0000 }, {  361, 0x0000 },
  /* 0x7100 */
  {  361, 0x8010 }, {  363, 0x0000 }, {  363, 0x0000 }, {  363, 0x0000 },
  {  363, 0x00c0 }, {  365, 0x1000 }, {  366, 0x0000 }, {  366, 0x0000 },
  {  366, 0x0000 }, {  366, 0x0000 }, {  366, 0x0000 }, {  366, 0x0000 },
  {  366, 0x0002 }, {  367, 0x0000 }, {  367, 0x0000 }, {  367, 0x4000 },
  /* 0x7200 */
  {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x0000 },
  {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x0000 },
  {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x0000 }, {  368, 0x4002 },
  {  370, 0x0000 }, {  370, 0x0000 }, {  370, 0x0000 }, {  370, 0x0000 },
  /* 0x7300 */
  {  370, 0x0000 }, {  370, 0x0000 }, {  370, 0x0010 }, {  371, 0x0000 },
  {  371, 0x0000 }, {  371, 0x0000 }, {  371, 0x0000 }, {  371, 0x0080 },
  {  372, 0x0000 }, {  372, 0x0000 }, {  372, 0x0000 }, {  372, 0x2000 },
  {  373, 0x0200 }, {  374, 0x0044 }, {  376, 0x0008 }, {  377, 0x0020 },
  /* 0x7400 */
  {  378, 0x0080 }, {  379, 0x0000 }, {  379, 0x4640 }, {  383, 0x0000 },
  {  383, 0x0000 }, {  383, 0x0000 }, {  383, 0x0004 }, {  384, 0x0000 },
  {  384, 0x0200 }, {  385, 0x8000 }, {  386, 0x0000 }, {  386, 0x0000 },
  {  386, 0x0000 }, {  386, 0x0000 }, {  386, 0x0000 }, {  386, 0x0000 },
  /* 0x7500 */
  {  386, 0x0002 }, {  387, 0x0000 }, {  387, 0x8000 }, {  388, 0x0000 },
  {  388, 0x0000 }, {  388, 0x0000 }, {  388, 0x8000 }, {  389, 0x0000 },
  {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 },
  {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 },
  /* 0x7600 */
  {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 },
  {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 }, {  389, 0x0000 },
  {  389, 0x0004 }, {  390, 0x5800 }, {  393, 0x0040 }, {  394, 0x0000 },
  {  394, 0x0000 }, {  394, 0x0000 }, {  394, 0x0000 }, {  394, 0x0000 },
  /* 0x7700 */
  {  394, 0x0000 }, {  394, 0x0000 }, {  394, 0x0000 }, {  394, 0x0000 },
  {  394, 0x0040 }, {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0000 },
  {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0000 },
  {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0000 },
  /* 0x7800 */
  {  395, 0x0000 }, {  395, 0x0000 }, {  395, 0x0002 }, {  396, 0x0000 },
  {  396, 0x4000 }, {  397, 0x0000 }, {  397, 0x0010 }, {  398, 0x0400 },
  {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0000 },
  {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0000 },
  /* 0x7900 */
  {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0000 }, {  399, 0x0001 },
  {  400, 0x0000 }, {  400, 0x0000 }, {  400, 0x0000 }, {  400, 0x0000 },
  {  400, 0x0000 }, {  400, 0x0810 }, {  402, 0x0000 }, {  402, 0x0000 },
  {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 },
  /* 0x7a00 */
  {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 },
  {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 },
  {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 }, {  402, 0x0000 },
  {  402, 0x0000 }, {  402, 0x0002 }, {  403, 0x0880 }, {  405, 0x0000 },
  /* 0x7b00 */
  {  405, 0x0000 }, {  405, 0x0000 }, {  405, 0x0000 }, {  405, 0x0000 },
  {  405, 0x0000 }, {  405, 0x0000 }, {  405, 0x0000 }, {  405, 0x0000 },
  {  405, 0x0000 }, {  405, 0x4000 },
};
static const Summary16 cp932ext_uni2indx_page7d[43] = {
  /* 0x7d00 */
  {  406, 0x0000 }, {  406, 0x0000 }, {  406, 0x0000 }, {  406, 0x0000 },
  {  406, 0x0100 }, {  407, 0x1000 }, {  408, 0x0000 }, {  408, 0x0000 },
  {  408, 0x0000 }, {  408, 0x0000 }, {  408, 0x0001 }, {  409, 0x0080 },
  {  410, 0x0000 }, {  410, 0x0040 }, {  411, 0x0000 }, {  411, 0x0000 },
  /* 0x7e00 */
  {  411, 0x0000 }, {  411, 0x0000 }, {  411, 0x0000 }, {  411, 0x0000 },
  {  411, 0x0000 }, {  411, 0x0004 }, {  412, 0x0000 }, {  412, 0x0000 },
  {  412, 0x0400 }, {  413, 0x0000 }, {  413, 0x0000 }, {  413, 0x0000 },
  {  413, 0x0000 }, {  413, 0x0000 }, {  413, 0x0000 }, {  413, 0x0000 },
  /* 0x7f00 */
  {  413, 0x0000 }, {  413, 0x0000 }, {  413, 0x0000 }, {  413, 0x0000 },
  {  413, 0x0080 }, {  414, 0x0000 }, {  414, 0x0000 }, {  414, 0x0000 },
  {  414, 0x0000 }, {  414, 0x0000 }, {  414, 0x0002 },
};
static const Summary16 cp932ext_uni2indx_page83[44] = {
  /* 0x8300 */
  {  415, 0x0002 }, {  416, 0x0000 }, {  416, 0x0000 }, {  416, 0x0000 },
  {  416, 0x0000 }, {  416, 0x0000 }, {  416, 0x0004 }, {  417, 0x8000 },
  {  418, 0x0000 }, {  418, 0x0000 }, {  418, 0x0000 }, {  418, 0x0000 },
  {  418, 0x0080 }, {  419, 0x0000 }, {  419, 0x0000 }, {  419, 0x0040 },
  /* 0x8400 */
  {  420, 0x0000 }, {  420, 0x0000 }, {  420, 0x0000 }, {  420, 0x0000 },
  {  420, 0x0100 }, {  421, 0x0000 }, {  421, 0x0000 }, {  421, 0x0000 },
  {  421, 0x0000 }, {  421, 0x0000 }, {  421, 0x0000 }, {  421, 0x0010 },
  {  422, 0x0000 }, {  422, 0x1000 }, {  423, 0x0000 }, {  423, 0x0000 },
  /* 0x8500 */
  {  423, 0x0000 }, {  423, 0x0000 }, {  423, 0x0000 }, {  423, 0x0000 },
  {  423, 0x0000 }, {  423, 0x0208 }, {  425, 0x0800 }, {  426, 0x0000 },
  {  426, 0x0000 }, {  426, 0x0000 }, {  426, 0x0000 }, {  426, 0x0001 },
};
static const Summary16 cp932ext_uni2indx_page88[109] = {
  /* 0x8800 */
  {  427, 0x0080 }, {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 },
  {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 },
  {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 },
  {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0000 }, {  428, 0x0020 },
  /* 0x8900 */
  {  429, 0x0000 }, {  429, 0x1000 }, {  430, 0x0000 }, {  430, 0x0000 },
  {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 },
  {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 },
  {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 }, {  430, 0x0000 },
  /* 0x8a00 */
  {  430, 0x0000 }, {  430, 0x0004 }, {  431, 0x0000 }, {  431, 0x0080 },
  {  432, 0x0000 }, {  432, 0x0000 }, {  432, 0x0000 }, {  432, 0x0200 },
  {  433, 0x0000 }, {  433, 0x0000 }, {  433, 0x0080 }, {  434, 0x4000 },
  {  435, 0x0000 }, {  435, 0x8000 }, {  436, 0x0000 }, {  436, 0x0040 },
  /* 0x8b00 */
  {  437, 0x0000 }, {  437, 0x0000 }, {  437, 0x0000 }, {  437, 0x0000 },
  {  437, 0x0000 }, {  437, 0x0008 }, {  438, 0x0000 }, {  438, 0x8000 },
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 },
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 },
  /* 0x8c00 */
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 },
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 },
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 },
  {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0000 }, {  439, 0x0011 },
  /* 0x8d00 */
  {  441, 0x0000 }, {  441, 0x0004 }, {  442, 0x0000 }, {  442, 0x0000 },
  {  442, 0x0000 }, {  442, 0x0000 }, {  442, 0x0000 }, {  442, 0x0040 },
  {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 },
  {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 },
  /* 0x8e00 */
  {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 },
  {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 },
  {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 }, {  443, 0x0000 },
  {  443, 0x8000 },
};
static const Summary16 cp932ext_uni2indx_page90[238] = {
  /* 0x9000 */
  {  444, 0x0000 }, {  444, 0x0000 }, {  444, 0x0000 }, {  444, 0x0000 },
  {  444, 0x0000 }, {  444, 0x0000 }, {  444, 0x0080 }, {  445, 0x0000 },
  {  445, 0x0000 }, {  445, 0x0000 }, {  445, 0x0000 }, {  445, 0x0000 },
  {  445, 0x0000 }, {  445, 0x4000 }, {  446, 0x0000 }, {  446, 0x0000 },
  /* 0x9100 */
  {  446, 0x0000 }, {  446, 0x0020 }, {  447, 0x0080 }, {  448, 0x0000 },
  {  448, 0x0000 }, {  448, 0x0000 }, {  448, 0x0000 }, {  448, 0x0000 },
  {  448, 0x0000 }, {  448, 0x0000 }, {  448, 0x0000 }, {  448, 0x0000 },
  {  448, 0x0000 }, {  448, 0x4480 }, {  451, 0x6030 }, {  455, 0x0000 },
  /* 0x9200 */
  {  455, 0x0440 }, {  457, 0x0001 }, {  458, 0x0000 }, {  458, 0x1600 },
  {  461, 0x4001 }, {  463, 0x0202 }, {  465, 0x0080 }, {  466, 0x0180 },
  {  468, 0x0100 }, {  469, 0x0000 }, {  469, 0x0080 }, {  470, 0x0000 },
  {  470, 0x0000 }, {  470, 0x02a9 }, {  475, 0x0081 }, {  477, 0x8a00 },
  /* 0x9300 */
  {  480, 0x0004 }, {  481, 0x6000 }, {  483, 0x0022 }, {  485, 0x0000 },
  {  485, 0x0100 }, {  486, 0x0080 }, {  487, 0x0000 }, {  487, 0x0001 },
  {  488, 0x0000 }, {  488, 0x0000 }, {  488, 0x0010 }, {  489, 0x0000 },
  {  489, 0x0040 }, {  490, 0x4000 }, {  491, 0x0000 }, {  491, 0x0100 },
  /* 0x9400 */
  {  492, 0x0000 }, {  492, 0x0000 }, {  492, 0x0000 }, {  492, 0x0002 },
  {  493, 0x0120 }, {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 },
  {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 },
  {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 },
  /* 0x9500 */
  {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 },
  {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 }, {  495, 0x0000 },
  {  495, 0x0000 }, {  495, 0x0004 }, {  496, 0x0000 }, {  496, 0x0000 },
  {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 },
  /* 0x9600 */
  {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 },
  {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 }, {  496, 0x0000 },
  {  496, 0x0000 }, {  496, 0x2000 }, {  497, 0x8000 }, {  498, 0x0000 },
  {  498, 0x0000 }, {  498, 0x0000 }, {  498, 0x0000 }, {  498, 0x0000 },
  /* 0x9700 */
  {  498, 0x0000 }, {  498, 0x0000 }, {  498, 0x0000 }, {  498, 0x0808 },
  {  500, 0xa008 }, {  503, 0x0022 }, {  505, 0x0000 }, {  505, 0x0000 },
  {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 },
  {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 },
  /* 0x9800 */
  {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 }, {  505, 0x0000 },
  {  505, 0x0000 }, {  505, 0x0080 }, {  506, 0x0020 }, {  507, 0x0000 },
  {  507, 0x0000 }, {  507, 0x0000 }, {  507, 0x0000 }, {  507, 0x0000 },
  {  507, 0x0000 }, {  507, 0x0000 }, {  507, 0x0000 }, {  507, 0x0000 },
  /* 0x9900 */
  {  507, 0x0000 }, {  507, 0x0000 }, {  507, 0x0080 }, {  508, 0x0000 },
  {  508, 0x0000 }, {  508, 0x0000 }, {  508, 0x0000 }, {  508, 0x0000 },
  {  508, 0x0000 }, {  508, 0x4000 }, {  509, 0x0000 }, {  509, 0x0000 },
  {  509, 0x0000 }, {  509, 0x0000 }, {  509, 0x0000 }, {  509, 0x0000 },
  /* 0x9a00 */
  {  509, 0x0000 }, {  509, 0x0000 }, {  509, 0x0000 }, {  509, 0x0000 },
  {  509, 0x4000 }, {  510, 0x0000 }, {  510, 0x0000 }, {  510, 0x0000 },
  {  510, 0x0000 }, {  510, 0x0000 }, {  510, 0x0000 }, {  510, 0x0000 },
  {  510, 0x0000 }, {  510, 0x1200 }, {  512, 0x0000 }, {  512, 0x0000 },
  /* 0x9b00 */
  {  512, 0x0000 }, {  512, 0x0000 }, {  512, 0x0000 }, {  512, 0x0000 },
  {  512, 0x0000 }, {  512, 0x0000 }, {  512, 0x0000 }, {  512, 0x0024 },
  {  514, 0x8000 }, {  515, 0x0000 }, {  515, 0x0000 }, {  515, 0x0802 },
  {  517, 0x0000 }, {  517, 0x0000 }, {  517, 0x0000 }, {  517, 0x0000 },
  /* 0x9c00 */
  {  517, 0x0001 }, {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 },
  {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 },
  {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 },
  {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 },
  /* 0x9d00 */
  {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0000 },
  {  518, 0x0000 }, {  518, 0x0000 }, {  518, 0x0800 }, {  519, 0x0001 },
  {  520, 0x0000 }, {  520, 0x0000 }, {  520, 0x0000 }, {  520, 0x0000 },
  {  520, 0x0000 }, {  520, 0x0000 }, {  520, 0x0000 }, {  520, 0x0000 },
  /* 0x9e00 */
  {  520, 0x0000 }, {  520, 0x0200 }, {  521, 0x0000 }, {  521, 0x0000 },
  {  521, 0x0000 }, {  521, 0x0000 }, {  521, 0x0000 }, {  521, 0x0000 },
  {  521, 0x0000 }, {  521, 0x0000 }, {  521, 0x0000 }, {  521, 0x0000 },
  {  521, 0x0000 }, {  521, 0x0002 },
};
static const Summary16 cp932ext_uni2indx_pagef9[19] = {
  /* 0xf900 */
  {  522, 0x0000 }, {  522, 0x0000 }, {  522, 0x0200 }, {  523, 0x0000 },
  {  523, 0x0000 }, {  523, 0x0000 }, {  523, 0x0000 }, {  523, 0x0000 },
  {  523, 0x0000 }, {  523, 0x0000 }, {  523, 0x0000 }, {  523, 0x0000 },
  {  523, 0x0000 }, {  523, 0x1000 }, {  524, 0x0000 }, {  524, 0x0000 },
  /* 0xfa00 */
  {  524, 0xc000 }, {  526, 0xffff }, {  542, 0x3fff },
};
static const Summary16 cp932ext_uni2indx_pageff[15] = {
  /* 0xff00 */
  {  556, 0xfffe }, {  571, 0xfc00 }, {  577, 0x0001 }, {  578, 0xf800 },
  {  583, 0x0001 }, {  584, 0x7800 }, {  588, 0x0000 }, {  588, 0x0000 },
  {  588, 0x0000 }, {  588, 0x0000 }, {  588, 0x0000 }, {  588, 0x0000 },
  {  588, 0x0000 }, {  588, 0x0000 }, {  588, 0x003f },
};

static int
cp932ext_wctomb (conv_t conv, unsigned char *r, ucs4_t wc, int n)
{
  if (n >= 2) {
    const Summary16 *summary = NULL;
    if (wc >= 0x0000 && wc < 0x0100)
      summary = &cp932ext_uni2indx_page00[(wc>>4)];
    else if (wc >= 0x2000 && wc < 0x2670)
      summary = &cp932ext_uni2indx_page20[(wc>>4)-0x200];
    else if (wc >= 0x3000 && wc < 0x3100)
      summary = &cp932ext_uni2indx_page30[(wc>>4)-0x300];
    else if (wc >= 0x3200 && wc < 0x33d0)
      summary = &cp932ext_uni2indx_page32[(wc>>4)-0x320];
    else if (wc >= 0x4e00 && wc < 0x5590)
      summary = &cp932ext_uni2indx_page4e[(wc>>4)-0x4e0];
    else if (wc >= 0x5700 && wc < 0x59c0)
      summary = &cp932ext_uni2indx_page57[(wc>>4)-0x570];
    else if (wc >= 0x5b00 && wc < 0x5de0)
      summary = &cp932ext_uni2indx_page5b[(wc>>4)-0x5b0];
    else if (wc >= 0x5f00 && wc < 0x7ba0)
      summary = &cp932ext_uni2indx_page5f[(wc>>4)-0x5f0];
    else if (wc >= 0x7d00 && wc < 0x7fb0)
      summary = &cp932ext_uni2indx_page7d[(wc>>4)-0x7d0];
    else if (wc >= 0x8300 && wc < 0x85c0)
      summary = &cp932ext_uni2indx_page83[(wc>>4)-0x830];
    else if (wc >= 0x8800 && wc < 0x8ed0)
      summary = &cp932ext_uni2indx_page88[(wc>>4)-0x880];
    else if (wc >= 0x9000 && wc < 0x9ee0)
      summary = &cp932ext_uni2indx_page90[(wc>>4)-0x900];
    else if (wc >= 0xf900 && wc < 0xfa30)
      summary = &cp932ext_uni2indx_pagef9[(wc>>4)-0xf90];
    else if (wc >= 0xff00 && wc < 0xfff0)
      summary = &cp932ext_uni2indx_pageff[(wc>>4)-0xff0];
    if (summary) {
      unsigned short used = summary->used;
      unsigned int i = wc & 0x0f;
      if (used & ((unsigned short) 1 << i)) {
        unsigned short c;
        /* Keep in `used' only the bits 0..i-1. */
        used &= ((unsigned short) 1 << i) - 1;
        /* Add `summary->indx' and the number of bits set in `used'. */
        used = (used & 0x5555) + ((used & 0xaaaa) >> 1);
        used = (used & 0x3333) + ((used & 0xcccc) >> 2);
        used = (used & 0x0f0f) + ((used & 0xf0f0) >> 4);
        used = (used & 0x00ff) + (used >> 8);
        c = cp932ext_2charset[summary->indx + used];
        r[0] = (c >> 8); r[1] = (c & 0xff);
        return 2;
      }
    }
    return RET_ILUNI;
  }
  return RET_TOOSMALL;
}
