/*
 *  linux/drivers/char/s3530a-rtc.c
 *
 *  I2C Real Time Clock Client Driver for Seiko Instrumentes S-3530A RTC/Calendar
 *
 *  Copyright 2004 Sernet Technologies
 *  Author:  Sernet Technologies
 *     	prakity@sercomm.com.cn
 *
 *  This program is free software; you can redistribute  it and/or modify it
 *  under  the terms of  the GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the  License, or (at your
 *  option) any later version.
 *
 *  THIS  SOFTWARE  IS PROVIDED   ``AS  IS'' AND   ANY  EXPRESS OR IMPLIED
 *  WARRANTIES,   INCLUDING, BUT NOT  LIMITED  TO, THE IMPLIED WARRANTIES OF
 *  MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED.  IN
 *  NO  EVENT  SHALL   THE AUTHOR  BE    LIABLE FOR ANY   DIRECT, INDIRECT,
 *  INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT
 *  NOT LIMITED   TO, PROCUREMENT OF  SUBSTITUTE GOODS  OR SERVICES; LOSS OF
 *  USE, DATA,  OR PROFITS; OR  BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 *  ANY THEORY OF LIABILITY, WHETHER IN  CONTRACT, STRICT LIABILITY, OR TORT
 *  (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
 *  THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 *  You should have received a copy of the  GNU General Public License along
 *  with this program; if not, write  to the Free Software Foundation, Inc.,
 *  675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  TODO:
 *    - implement alarm and periodic IRQ support.
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/types.h>
#include <linux/miscdevice.h>
#include <linux/fcntl.h>
#include <linux/poll.h>
#include <linux/fs.h>
#include <linux/init.h>
#include <linux/i2c.h>
#include <linux/slab.h>
#include <linux/string.h>
#include <linux/rtc.h>
#include <linux/proc_fs.h>
#include <linux/spinlock.h>
#include <asm/uaccess.h>
#include <asm/system.h>
#include <asm/arch-ixp425/time.h>

#undef DEBUG_S3530

#ifdef DEBUG_S3530
#define	dbg(fmt, args...) printk(KERN_INFO "%s: " fmt, __func__, ## args)
#else
#define	dbg(fmt, args...)
#endif

#define S3530_MODULE_NAME "S3530"
#define PFX S3530_MODULE_NAME

#define err(format, arg...) printk(KERN_ERR PFX ": " format , ## arg)
#define info(format, arg...) printk(KERN_INFO PFX ": " format , ## arg)
#define warn(format, arg...) printk(KERN_WARNING PFX ": " format , ## arg)
#define emerg(format, arg...) printk(KERN_EMERG PFX ": " format , ## arg)

#define EPOCH 2000

#undef BCD_TO_BIN
#define BCD_TO_BIN(val) (((val)&15) + ((val)>>4)*10)

#undef BIN_TO_BCD
#define BIN_TO_BCD(val) ((((val)/10)<<4) + (val)%10)

typedef struct 
{
	unsigned char year;
	unsigned char month;
	unsigned char day;
	unsigned char weekDay;
	unsigned char hours;
	unsigned char minutes;
	unsigned char second;
} RTCDATA;

#define DEVICE_COMMAND_RESET		0x30
#define DEVICE_COMMAND_GET_STATUS	0x31
#define DEVICE_COMMAND_SET_STATUS	0x31
#define DEVICE_COMMAND_GET_DATA		0x32
#define DEVICE_COMMAND_SET_DATA		0x32

#define I2C_DRIVERID_S3530 1
#define SYS_EPOCH		1900
#define EPOCH 			2000

#define DEVID_RTC          0x6F

static struct i2c_driver s3550_driver;

static int s3550_use_count = 0;

static struct i2c_client *this_client = NULL;

static int s3550_set_time(struct i2c_client *client, const struct rtc_time *tm);
static int rtc_read_proc(char *page, char **start, off_t off,
                         int count, int *eof, void *data);


/* flip bit  order from LSB to MSB */
inline static unsigned char flipBitOrderInByte(unsigned char byte)
{
	int i;
	unsigned char flippedByte;
	
	flippedByte = 0;
	for (i = 0; i < 8; i++)
	{
		if (byte & (1 << i))
			flippedByte |= 1<< (7-i);
	}
	return flippedByte;
}


static int
s3550_get_time(struct i2c_client *client, struct rtc_time *tm)
{	
	RTCDATA rtc;
	int ret;
	
	client->addr = DEVICE_COMMAND_GET_DATA;

	ret = i2c_master_recv(client, (char *) &rtc, sizeof (RTCDATA));
	
	if (ret < sizeof (RTCDATA)) 
	{
		ret = -ENXIO;
		dbg("couldn't read all RTC data\n");
	}

	dbg("IN: year=%02x, month=%02x, day=%02x, hour=%02x, "
	    "min=%02x, sec=%02x\n",
	    rtc.year, rtc.month, rtc.day, rtc.hours,
	    rtc.minutes, rtc.second);
	
	tm->tm_year = BCD_TO_BIN(flipBitOrderInByte (rtc.year)); // 0 - 99
	tm->tm_year += EPOCH - SYS_EPOCH;
	tm->tm_mon  = BCD_TO_BIN(flipBitOrderInByte (rtc.month));   // 1 - 12
	tm->tm_mon--;                       /* tm_mon is 0 to 11 */
	tm->tm_mday = BCD_TO_BIN(flipBitOrderInByte (rtc.day));  // 1 - 31
	
	tm->tm_hour = BCD_TO_BIN(flipBitOrderInByte (rtc.hours & ~0x01));
	if (rtc.hours & 0x01) 
	{
		// PM format
		tm->tm_hour += 12; // Make 24 hour format
	}
	else
	{
	}
	
	tm->tm_min = BCD_TO_BIN(flipBitOrderInByte (rtc.minutes));
	tm->tm_sec = BCD_TO_BIN(flipBitOrderInByte (rtc.second));

	dbg("OUT: year=%d, mon=%d, day=%d, hour=%d, min=%d, sec=%d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);
	
	return 0;
}

static int 
s3550_set_time(struct i2c_client *client, const struct rtc_time *tm)
{
	RTCDATA rtc;
	int ret;

	dbg("IN: year=%d, mon=%d, day=%d, hour=%d, min=%d, sec=%d\n",
	    tm->tm_year, tm->tm_mon, tm->tm_mday, tm->tm_hour,
	    tm->tm_min, tm->tm_sec);

	rtc.year    = flipBitOrderInByte (BIN_TO_BCD(tm->tm_year - (EPOCH - SYS_EPOCH) ));
	rtc.month   = flipBitOrderInByte (BIN_TO_BCD(tm->tm_mon + 1)); /* tm_mon is 0 to 11 */
	rtc.day     = flipBitOrderInByte (BIN_TO_BCD(tm->tm_mday));
	
	if (tm->tm_hour > 11)
	{
		rtc.hours = flipBitOrderInByte (BIN_TO_BCD(tm->tm_hour-12) | 0x80); /* 12 hour format */
	}
	else
	{
		rtc.hours = flipBitOrderInByte (BIN_TO_BCD(tm->tm_hour));
	}

	rtc.minutes  = flipBitOrderInByte (BIN_TO_BCD(tm->tm_min));
	rtc.second  = flipBitOrderInByte (BIN_TO_BCD(tm->tm_sec));
	
	dbg("OUT: year=%02x, mon=%02x, day=%02x, hour=%02x, "
	    "min=%02x, sec=%02x\n",
	    rtc.year, rtc.month, rtc.day, rtc.hours,
	    rtc.minutes, rtc.second);
	
	/* write RTC registers */
	
	client->addr = DEVICE_COMMAND_SET_DATA;
	ret = i2c_master_send(client, (char *) &rtc, sizeof (RTCDATA));

	if (ret != sizeof (RTCDATA)) 
	{
		dbg("couldn't write RTC\n");
		return ret;
	}

	return 0;
}


static int 
s3550_rtc_reset (struct i2c_client *client)
{
	unsigned char status;
	int ret;

	client->addr = DEVICE_COMMAND_GET_STATUS;

	ret = i2c_master_recv(client, &status, sizeof (status));
	
	if (ret < sizeof (status)) 
	{
		ret = -ENXIO;
		dbg("couldn't read all RTC data\n");
	}

	dbg("STATUS: status=%02x\n", status);
	    
	/* if set , do reset */
	if(status & 0x80)
	{	
		client->addr = DEVICE_COMMAND_RESET;
		ret = i2c_master_send(client, NULL, 0);
	}
	
	return ret;
}

static int
s3550_probe(struct i2c_adapter *adap)
{
	int ret;
	struct rtc_time dummy_tm;
	
	if (this_client != NULL)
		return -EBUSY;
	
	this_client = kmalloc(sizeof(*this_client), GFP_KERNEL);
	if (this_client == NULL) {
		return -ENOMEM;
	}

	strcpy(this_client->name, S3530_MODULE_NAME);
	this_client->id		= s3550_driver.id;
	this_client->flags	= 0;
	this_client->addr	= DEVID_RTC;
	this_client->adapter	= adap;
	this_client->driver	= &s3550_driver;
	this_client->data	= NULL;

	
	if ((ret = s3550_rtc_reset(this_client)) < 0) 
	{
		kfree(this_client);
		this_client = NULL;
		return ret;
	}
	
	/*
	 * use s3550_get_time() to probe for an s3550 on this bus.
	 */
	if ((ret = s3550_get_time(this_client, &dummy_tm)) < 0) {
		kfree(this_client);
		this_client = NULL;
		return ret;
	}

	info("found s3550 on %s\n", adap->name);

#ifdef CONFIG_ARCH_SC806DVI
	dummy_tm.tm_sec = 0;
	s3550_set_time(this_client, &dummy_tm);
#endif
	
	/* attach it. */
	return i2c_attach_client(this_client);
}

static int
s3550_detach(struct i2c_client *client)
{
	i2c_detach_client(client);

	if (this_client != NULL) {
		kfree(this_client);
		this_client = NULL;
	}

	return 0;
}

int rtc_open(struct inode *minode, struct file *mfile)
{
	/*if(MOD_IN_USE)*/
	if(s3550_use_count > 0) {
		return -EBUSY;
	}
	MOD_INC_USE_COUNT;
	++s3550_use_count;
	return 0;
}

int rtc_release(struct inode *minode, struct file *mfile)
{
	MOD_DEC_USE_COUNT;
	--s3550_use_count;
	return 0;
}

static loff_t rtc_llseek(struct file *mfile, loff_t offset, int origint)
{
	return -ESPIPE;
}

static int
s3550_command(struct i2c_client *client, unsigned int cmd, void *arg)
{
	return -EINVAL;
}

static int rtc_ioctl(struct inode *inode, struct file *file, unsigned int cmd,
		     unsigned long arg)
{
	struct rtc_time rtc_tm;
	int ret;
	
	switch (cmd) {
	case RTC_RD_TIME:	/* Read the time/date from RTC  */
		if ((ret = s3550_get_time(this_client, &rtc_tm)) < 0)
			return ret;
		return copy_to_user((void *)arg, &rtc_tm, sizeof(rtc_tm)) ? 
			-EFAULT : 0;
	case RTC_SET_TIME:	/* Set the RTC */
		if (!capable(CAP_SYS_TIME))
			return -EACCES;

		if (copy_from_user(&rtc_tm, 
				   (struct rtc_time *) arg,
		                   sizeof(struct rtc_time))) 
			return -EFAULT;

		return s3550_set_time(this_client, &rtc_tm);
	default:
		return -EINVAL;
	}
}


static struct i2c_driver s3550_driver = {
	name:			S3530_MODULE_NAME,
	id:				I2C_DRIVERID_S3530,
	flags:			I2C_DF_NOTIFY,
	attach_adapter:	s3550_probe,
	detach_client:	s3550_detach,
	command:		s3550_command
};

static struct file_operations rtc_fops = {
	owner:		THIS_MODULE,
	llseek:		rtc_llseek,
	ioctl:		rtc_ioctl,
	open:		rtc_open,
	release:	rtc_release,
};

static struct miscdevice s3550rtc_miscdev = {
	RTC_MINOR,
	"rtc",
	&rtc_fops
};

static __init int s3550_init(void)
{
	int ret;

	info("I2C based RTC driver.\n");
	ret = i2c_add_driver(&s3550_driver);
	if (ret) {
		err("Register I2C driver failed, errno is %d\n", ret);
		return ret;
	}
	ret = misc_register(&s3550rtc_miscdev);
	if (ret) {
		err("Register misc driver failed, errno is %d\n", ret);
		ret = i2c_del_driver(&s3550_driver);
		if (ret) {
			err("Unregister I2C driver failed, errno is %d\n",
			    ret);
		}
		return ret;
	}

	create_proc_read_entry("driver/rtc", 0, 0, rtc_read_proc, NULL);

	return 0;
}

static void __exit s3550_exit(void)
{
        remove_proc_entry("driver/rtc", NULL);
        misc_deregister(&s3550rtc_miscdev);
	i2c_del_driver(&s3550_driver);
}


module_init(s3550_init);
module_exit(s3550_exit);

/*
 *	Info exported via "/proc/driver/rtc".
 */

static int rtc_proc_output(char *buf)
{
	char *p;
	struct rtc_time tm;
	int ret;
	
	if ((ret = s3550_get_time(this_client, &tm)) < 0)
		return ret;

	p = buf;

	/*
	 * There is no way to tell if the luser has the RTC set for local
	 * time or for Universal Standard Time (GMT). Probably local though.
	 */
	p += sprintf(p,
		     "rtc_time\t: %02d:%02d:%02d\n"
		     "rtc_date\t: %04d-%02d-%02d\n"
		     "rtc_epoch\t: %04d\n",
		     tm.tm_hour, tm.tm_min, tm.tm_sec,
		     tm.tm_year + SYS_EPOCH, tm.tm_mon + 1,
		     tm.tm_mday, EPOCH);

	return p - buf;
}

static int rtc_read_proc(char *page, char **start, off_t off,
			 int count, int *eof, void *data)
{
	int len = rtc_proc_output(page);
	if (len <= off + count)
		*eof = 1;
	*start = page + off;
	len -= off;
	if (len > count)
		len = count;
	if (len < 0)
		len = 0;
	return len;
}

MODULE_AUTHOR("Philip Rakity");
MODULE_LICENSE("GPL");

