/* sem.c
 * Common Utility Library
 *
 *---------------------------------------------------------------------------
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 *---------------------------------------------------------------------------
 *
 * system: Common utility
 * subsystem: Common
 * Author: Intel Corporation
 * ---------------------------------------------------------------------
 */
#include <stdlib.h>
#include "ostype.h"
#include "sem.h"

#if (OS == WIN_32)
#include<wtypes.h>
#include<winbase.H>


sem_T Sem_Init(unsigned int initCount)
{
   return((sem_T)CreateSemaphore(NULL, initCount, 10, NULL));
}


int Sem_Destroy(sem_T sem)
{
   CloseHandle((HANDLE)sem);
   return(SEM_SUCCESS);
}

int Sem_Wait(sem_T sem, unsigned int timeout)
{
   unsigned int status;
   if((status = WaitForSingleObject((HANDLE)sem, timeout)) == WAIT_FAILED)
	   return(SEM_FAILED);
   if(status == WAIT_TIMEOUT) return(SEM_TIMEDOUT);
   return (SEM_SUCCESS);
}

int Sem_Post(sem_T sem)
{
   if(ReleaseSemaphore((HANDLE)sem, 1, NULL)) return(SEM_FAILED);
   return(SEM_SUCCESS);
}
#endif /* OS == WIN_32 */


#if (OS == VXWORKS)
#include <semLib.h>
sem_T Sem_Init(unsigned int initCount)
{
   return((sem_T)semCCreate(SEM_Q_FIFO, initCount));
}

int Sem_Destroy(sem_T sem)
{
   semDelete((SEM_ID)sem);
   return(SEM_SUCCESS);
}

int Sem_Wait(sem_T sem, unsigned int timeout)
{
   return(semTake((SEM_ID)sem, timeout));
}

int Sem_Post(sem_T sem)
{
   return(semGive((SEM_ID)sem));
}
#endif /* OS == VXWORKS */


#if (OS == UCOS)
#include "os.h"

sem_T Sem_Init(unsigned int initCount)
{
   return((sem_T)OSSemCreate(initCount));
}

int Sem_Destroy(sem_T sem)
{
   ;/*dj: Henry we need ucos function to remove semaphores! */
   return(SEM_SUCCESS);
}

int Sem_Wait(sem_T sem, unsigned int timeout)
{
   unsigned int err;

   OSSemPend((OS_EVENT *)sem, timeout, &err);
   if(err == OS_NO_ERR) return(SEM_SUCCESS);
   if(err == OS_TIMEOUT) return(SEM_TIMEDOUT);
   return(SEM_FAILED);
}

int Sem_Post(sem_T sem)
{
   if(OSSemPost((OS_EVENT *) sem) == OS_NO_ERR) return(SEM_SUCCESS);
   return(SEM_FAILED);
}
#endif /* OS == UCOS */



#if (OS == UNIX)
#include <pthread.h>
typedef struct semx_S{
   unsigned int		count;			/* number of SemPost called */
   unsigned int		waiting;		/* number of Sem_Wait called */  
   pthread_mutex_t	mutex;			/* protect member variables */
   pthread_cond_t	condCount;		/* condition variable to block on */
}semx_T;


sem_T Sem_Init(unsigned int initCount)
{
   semx_T *sem;

   if(!(sem = (semx_T *)malloc(sizeof(semx_T)))) return(NULL);
   pthread_mutex_init(&sem->mutex, NULL);
   pthread_cond_init(&sem->condCount, NULL);
   sem->count = initCount;
   sem->waiting = 0;
   return((sem_T) sem);
}

int Sem_Wait(sem_T sem, unsigned int timeout)
{
   semx_T *semx = (semx_T *)sem;
   struct timespec abstime, delta;
   int status;
   
   if(timeout >= 1000) delta.tv_sec = timeout/1000;
   else delta.tv_sec = 0;
   delta.tv_nsec = (timeout - (delta.tv_sec * 1000)) * 1000000;

   pthread_get_expiration_np(&delta, &abstime);
					    
   pthread_mutex_lock(&semx->mutex);			/* obtain the semaphore mutex */
   semx->waiting++;   					/* indicate that we are waiting */

   /* wait for condition count to be > 0 and to be signaled or timed out */
   while(semx->count == 0)
	  if((status = pthread_cond_timedwait(&semx->condCount, &semx->mutex, &abstime))){
         semx->waiting--;
         pthread_mutex_unlock(&semx->mutex);
         if(status == ETIMEDOUT) return(SEM_TIMEDOUT);
         return(SEM_FAILED);
	   }

   /* We've gotten the mutex, so... decrement counters and release mutex */
   semx->waiting--;
   semx->count--;
   pthread_mutex_unlock(&semx->mutex);
   return(SEM_SUCCESS);
}


int Sem_Post(sem_T sem)
{
   semx_T *semx = (semx_T *)sem;

   pthread_mutex_lock(&semx->mutex);

   /* if threads are waiting, then signal one to continue */
   if(semx->waiting > 0) pthread_cond_signal(&semx->condCount);

   /* increment the semaphore's count and released the mutex */
   semx->count++;

   pthread_mutex_unlock(&semx->mutex);

   return(SEM_SUCCESS);
}

int Sem_Destroy(sem_T sem)
{
   semx_T *semx = (semx_T *)sem;

   pthread_mutex_destroy(&semx->mutex);
   pthread_cond_destroy(&semx->condCount);
   free(semx);
   return(SEM_SUCCESS);
}

#endif /* OS == UNIX */
