/**
 * ============================================================================
 * = COPYRIGHT
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * = PRODUCT
 *      Intel(r) IXP425 Software Release
 *
 * = LIBRARY
 *      OSSL ( Operating System Services)  Library
 *
 * = MODULE
 *      OSSL Utilities
 *
 * = FILENAME
 *      osslUtils.c
 *
 * = DESCRIPTION
 *      
 *   OSSL API functions for timers and memory management:
 *
 *	    ix_ossl_sleep
 *        ix_ossl_time_get
 *        ix_ossl_malloc
 *        ix_ossl_free
 *        ix_ossl_memcpy
 *        ix_ossl_memset
 *        ix_ossl_tick_get
 *      
 *
 * = AUTHOR
 *     Intel Corporation
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      1/9/2002 1:56:24 PM
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 */

#include "ix_ossl.h" 

#ifdef __linux
#include <linux/sched.h>
#include <linux/time.h>
#else
#include <stdio.h>
#include "internal-osslThreadUtils.h"
#include "os_api.h"
#include "string.h"
#endif

/**
 * NAME:   ix_ossl_sleep
 *
 * DESCRIPTION: This function causes the calling thread to sleep for the 
 *              specified time.
 *
 * @Param:  - IN sleeptime_ms:  sleep time specified in milliseconds.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */

ix_error ix_ossl_sleep(ix_uint32 sleeptime_ms)
{
#ifdef __linux
    current->state = TASK_INTERRUPTIBLE;
    schedule_timeout((sleeptime_ms * HZ) / 1000);
    return IX_OSSL_ERROR_SUCCESS;
#else
	ix_error ixError = 0;
	os_error osError;

	ixError = os_sleep(sleeptime_ms,&osError);

	if(ixError != OS_SUCCESS) {
#ifdef DEBUG
		printf("ix_ossl_sleep: failed! %d] %s \n",osError.errnum, osError.errname);
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(ixError,IX_ERROR_LEVEL_LOCAL);		
	}  
	return IX_OSSL_ERROR_SUCCESS;
#endif
}


/**
 * NAME:    ix_ossl_sleep_tick 
 *
 * DESCRIPTION: This function causes the calling thread to sleep for the 
 *              time specified in OS ticks.
 *
 * @Param:  - IN sleeptime_ticks:  sleep time specified in os ticks.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_sleep_tick(ix_uint32 sleeptime_ticks)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	ix_error ixError = 0;
	os_error osError;

	ixError = os_sleep_tick(sleeptime_ticks,&osError);

	if(ixError != OS_SUCCESS) {
#ifdef DEBUG
		printf("ix_ossl_sleep_tick: failed! %d] %s \n",osError.errnum, osError.errname);
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(ixError,IX_ERROR_LEVEL_LOCAL);
	}    
	return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:      ix_ossl_time_get 
 *
 * DESCRIPTION: This function places the current value of a timer,
 *              in seconds and  nano-seconds, into an ix_ossl_time_t structure 
 *              pointed by 'ptime'. This function does not provide a 
 *              time-of-day. The intention is to provide a nano-second 
 *              resolution time.
 *
 * @Param:  - OUT ptime :  pointer to 'ix_ossl_time_t' structure where data will be
 *              returned.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_time_get(ix_ossl_time_t *ptime)
{

#ifdef __linux
        do_gettimeofday((struct timeval *)ptime);
        ptime->tv_nsec *= 1000;
        return IX_OSSL_ERROR_SUCCESS;
#else
	ix_error ixError = 0;
	os_error osError;

	ixError = os_time_get(ptime,&osError);
	if(ixError != OS_SUCCESS) {
#ifdef DEBUG
		printf("ix_ossl_time_get: failed! %d] %s \n",osError.errnum, osError.errname);
#endif /* DEBUG */	
		return IX_OSSL_ERROR_NEW(ixError,IX_ERROR_LEVEL_LOCAL);
	}
	return IX_OSSL_ERROR_SUCCESS;

#endif

}


/**
 * NAME:      ix_ossl_tick_get 
 *
 * DESCRIPTION: This function returns the number of os ticks per second.
 *
 * @Param:  - OUT pticks :  pointer to location  where data will be returned.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS.
 */
ix_error ix_ossl_tick_get(int *pticks)
{

#ifdef __linux

        *pticks = HZ;
        
#else
	*pticks = sysClkRateGet();

#endif
	return IX_OSSL_ERROR_SUCCESS;

}


/**
*   osApi functions for ossl utilities
*   Port to VxWorks
*
*
*	       os_sleep
*              os_sleep_tick
*              os_time_get
*              
**/


/**
 * NAME: os_sleep
 *
 * DESCRIPTION: This function causes the calling thread to sleep for the 
 *              specified time.
 *
 * @Param:  - IN  sleeptime_ms  sleep time specified in milliseconds.
 * @Param:  - OUT osError       pointer to the datastructure where the 
 *                              error conditions are returned.
 *
 */
int os_sleep(ix_uint32 sleeptime_ms, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
	
    int retVal;
    ix_uint64 sleeptime_ticks=0;

     /* convert ms to nanoseconds */
#ifdef DEBUG
    struct timespec sleeptime;
    ix_uint64 sleeptime_ns=0;
    ldiv_t result; /* ldivt: struct { long quot, rem; } containing both 
	              the quotient and the remainder */

    sleeptime_ns = sleeptime_ms * 1000000;
    printf("before ldiv_r \n");
    ldiv_r(sleeptime_ns,1000000000,&result); 
    printf("after ldiv_r \n");
    sleeptime.tv_sec = result.quot ;     
    sleeptime.tv_nsec = result.rem;
    printf(" sleeptime.: sec %d nsec  %d \n",sleeptime.tv_sec,sleeptime.tv_nsec);
#endif /* DEBUG */

#ifdef DEBUG	
   if(!TV_VALID(sleeptime)){	
	   printf(" invalid time \n");
	   retVal = EINVAL;	
	   goto exit;
    }
#endif /* DEBUG */ 
  

#ifdef DEBUG
     TV_CONVERT_TO_TICK(sleeptime_ticks,sleeptime);
	 printf(" sleeptime_ticks = %d \n",sleeptime_ticks);
#endif /* DEBUG */


#define MS_PER_TICK (1000/CLOCKS_PER_SEC)
   sleeptime_ticks = sleeptime_ms/MS_PER_TICK;

   retVal = taskDelay(sleeptime_ticks);

   if(retVal == VXWORKS_SUCCESS) {   
	   /* thread sleep is successfull */  
#ifdef DEBUG
	   printf("os_sleep is completed \n"); 
#endif /* DEBUG */
      return OS_SUCCESS;	
    }

#ifdef DEBUG
exit:   
#endif

	if (retVal == EINVAL){ 	
		/* The value specified by attr is invalid */   
		osError->errnum = retVal;
		strcpy(osError->errname,"EINVAL");
		return OS_INVALID_ATTR;   
	}
	else if (retVal == EINTR){ 
		/* Error Signal Handler  */     
		osError->errnum = retVal;
		strcpy(osError->errname,"EINTR");
		return OS_ERR_INTR;     
    }
    else  {      
		osError->errnum = retVal; 
		strcpy(osError->errname,"UNKNOWN_ERROR");
		return OS_FAILURE; 
	}	
#endif
}


/**
 * NAME: os_sleep_tick
 *
 * DESCRIPTION: This function causes the calling thread to sleep for the 
 *  	           time specified in OS ticks.
 *
 * @Param:  - IN    sleeptime_ticks  sleep time specified in OS ticks.
 * @Param:  - OUT   osError          pointer to the datastructure where the 
 *                                   error conditions are returned.
 *
 */
int os_sleep_tick(ix_uint32 sleeptime_ticks, os_error *osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;

    retVal = taskDelay(sleeptime_ticks);	 
	if(retVal == VXWORKS_SUCCESS){
		/* thread is created successfully */ 
#ifdef DEBUG
		printf("os_sleep is completed \n"); 
#endif /* DEBUG */		
		return OS_SUCCESS;	   
	}
   
    else if (retVal == EINTR){
		/* The value specified by attr is invalid */    
		osError->errnum = retVal;
		strcpy(osError->errname,"EINTR");
		return OS_ERR_INTR;     
    }
    else    {     
		osError->errnum = retVal;
		strcpy(osError->errname,"UNKNOWN_ERROR");
		return OS_FAILURE; 
	}	
#endif
}

/**
 * NAME: os_time_get
 *
 * DESCRIPTION: This function places the current value of time, in
 *              (seconds, nanoseconds), into the '*ptime' structure. This 
 *              function does not provide a time-of-day. The purpose is 
 *              to provide a nano-second resolution time.
 *
 * @Param:  - OUT ptime         addres to the time structure.
 * @Param:  - OUT osError       pointer to the datastructure where the 
 *                               error conditions are returned.
 *
 */
int os_time_get(ix_ossl_time_t *ptime, os_error *osError)
{  

#ifdef __vxworks
        int retVal;

    	/* Uses a  posix call to get time */ 
	retVal = clock_gettime(CLOCK_REALTIME,(struct timespec *)ptime);

#ifdef DEBUG
	printf("clock_gettime: failed! %d  \n",retVal);
#endif 
      
   if(retVal == VXWORKS_SUCCESS)     {   
	   /* clock gettime is successfull */     
	   return OS_SUCCESS;	
    }   
    else if (retVal == EINVAL)    {   
		/* The value specified by attr is invalid */      
		osError->errnum = retVal; 
		strcpy(osError->errname,"EINVL"); 
		return OS_INVALID_ATTR;     
    }
    else  if(retVal == EFAULT)    {   
		/* The value specified by attr is invalid */     
		osError->errnum = retVal;     
		strcpy(osError->errname,"EFAULT"); 
		return OS_INVALID_ADDR;     
    }
    else    {     
		osError->errnum = retVal;
		strcpy(osError->errname,"UNKNOWN_ERROR");
		return OS_FAILURE;
	}
#endif
	return 0;
}


/**
 * NAME: ix_ossl_malloc
 *
 * DESCRIPTION: This function will allocate a memory block.
 *          the function returns a void pointer to the allocated space, or NULL if there is 
 *          insufficient memory available. To return a pointer to a type other than void, use
 *          a type cast on the return value. The storage space pointed to by the return value 
 *          is guaranteed to be suitably aligned for storage of any type of object. If size is 0,
 *          ix_ossl_malloc allocates a zero-length item in the heap and returns a valid pointer 
 *          to that item. Always check the return from ix_ossl_malloc, even if the amount of 
 *          memory requested is small.
 * 
 * @Param:  - IN arg_Size - the size of the memory block requested.
 *
 * @Return: Returns a valid address if successful or a NULL for failure.
 */
void* ix_ossl_malloc(
                      ix_ossl_size_t arg_Size
                    )
{
#ifdef __linux
    return NULL;

#else
void* mem_ptr;

#ifdef DEBUG
	printf(" in ix_ossl_malloc \n");
#endif /* DEBUG */

	mem_ptr = malloc( arg_Size );
	return mem_ptr;
#endif
}


/**
 * NAME: ix_ossl_free
 *
 * DESCRIPTION: This function will free a memory block specified by the passed address.
 *          The ix_ossl_free function deallocates a memory block (arg_pMemory) that was previously 
 *          allocated by a call to ix_ossl_malloc.If arg_pMemory is NULL, the pointer is ignored 
 *          and ix_ossl_free immediately returns. Attempting to free an invalid pointer (a 
 *          pointer to a memory block that was not allocated by ix_ossl_malloc) may affect 
 *          subsequent allocation requests and cause errors.
 * 
 * @Param:  - IN arg_pMemory - address of the memory block to be deallocated.
 */
void ix_ossl_free(
                   void* arg_pMemory
                 )
{
#ifdef __linux
#else

#ifdef DEBUG
	printf(" in ix_ossl_free \n");
#endif /* DEBUG */

	free(arg_pMemory);		
#endif
}



/**
 * NAME: ix_ossl_memcpy
 *
 * DESCRIPTION: This function will copy memory bytes between buffers.
 *          The ix_ossl_memcpy function copies count bytes of arg_pSrc to arg_pDest. If the source and 
 *          destination overlap, this function does not ensure that the original source bytes in the 
 *          overlapping region are copied before being overwritten. 
 * 
 * @Param:  - INOUT arg_pDest - destination buffer address 
 * @Param:  - IN arg_pSrc - source buffer address
 * @Param:  - IN arg_Count - number of bytes to copy
 *
 * @Return: Returns the address of the destination buffer.
 */
void* ix_ossl_memcpy(
                      void* arg_pDest, 
                      const void* arg_pSrc,
                      ix_ossl_size_t arg_Count 
                    )
{
#ifdef __linux
	return NULL;

#else
	void* copy_result;

#ifdef DEBUG
	printf(" in ix_ossl_memcpy \n");
#endif /* DEBUG */

	copy_result = memcpy( arg_pDest, arg_pSrc, arg_Count );
	return copy_result;
#endif
}


/**
 * NAME: ix_ossl_memset
 *
 * DESCRIPTION: This function sets buffers to a specified character.
 *          The ix_ossl_memset function sets the first arg_Count bytes of arg_pDest to the 
 *          character arg_Char.
 * 
 * @Param:  - INOUT arg_pDest - destination buffer address 
 * @Param:  - IN arg_pChar - character to set
 * @Param:  - IN arg_Count - number of characters to set
 *
 * @Return: Returns the address of the destination buffer.
 */
void* ix_ossl_memset(
                      void* arg_pDest, 
                      int arg_pChar,
                      ix_ossl_size_t arg_Count 
                    )
{		
#ifndef __linux
	return NULL;

#else
	void* retPtr;
#ifdef DEBUG
	printf(" in ix_ossl_memset \n");
#endif /* DEBUG */

	retPtr = memset( arg_pDest, arg_pChar, arg_Count );
	return(retPtr);
#endif
}



/**
 * Message logging API
 */

/**
 * NAME: ix_ossl_message_log_init
 *
 * DESCRIPTION: This function is used to initialize the error message logging. For each OS the 
 *              messages will be logged into an implementation dependent stream. Further details 
 *              will be provided on where the messages will be logged! This function should be called 
 *              before any call to ix_ossl_message_log().
 * 
 * @Return: IX_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_message_log_init(void)
{
#ifdef __linux
	return IX_OSSL_ERROR_FAILURE;

#else
	printf(" in ix_ossl_message_log_init, currently this is under definition \n");
	/* Nothing Needs to be done here */
	return IX_OSSL_ERROR_SUCCESS;
#endif
}



/**
 * NAME: ix_ossl_message_log
 *
 * DESCRIPTION: This routine is used to log a specified message.  This routine's  syntax is similar to
 *              printf() - a format string is followed by  a variable number of arguments which will 
 *              be interpreted and formated according to the fmt_string passed as the first argument.
 *              Further details will be provided on where the messages will be logged!
 *
 * @Param: - IN arg_pFmtString  -  format string for the log message
 *
 * @Return: IX_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_message_log(
                              char* arg_pFmtString, 
                              ...
                            )
{
#ifdef __linux
	return IX_OSSL_ERROR_FAILURE;

#else
	printf(" in ix_ossl_message_log: currently this is under definition \n");

#if 0		
	EventLogMsg(MYAPP_CANNOT_COPY_FILE, 2, pszSourceFile, pszDestFile, 1, dwError);
#endif 

	return IX_OSSL_ERROR_SUCCESS;
#endif
}
