/**
 * ============================================================================
 * = COPYRIGHT
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * = PRODUCT
 *      Intel(r) IXP425 Software Release
 *
 * = LIBRARY
 *      OSSL ( Operating System Services)  Library
 *
 * = MODULE
 *      OSSL Threads
 *
 * = FILENAME
 *      osslThread.c
 *
 * = DESCRIPTION
 *      
 *   This file contains the implementation of wrapper functions around
 *   the  "thread" calls. To spawn a new thread the programmer
 *   will have to pass the entry point function for that new thread.
 *   This entry point function has to adhere to the defined prototype
 *   which will enable that thread to return both an ix_error token
 *   and a pointer to a caller-defined object to the thread's parent.
 *   In addition, functions are defined that allow for the standard
 *   wait, exit, and kill functionality.
 *
 *   OSSL Thread API Functions:
 *
 *	     ix_ossl_thread_create
 *         ix_ossl_thread_get_id
 *         ix_ossl_thread_exit
 *         ix_ossl_thread_kill
 *         ix_ossl_thread_set_priority
 *         ix_ossl_thread_delay
 *
 * = AUTHOR
 *      Intel Corporation
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      1/9/2002 1:56:24 PM
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 */

/* INCLUDES */
#include "ix_ossl.h" 

#ifdef __linux
#include <linux/sched.h>
#else
#include <stdio.h>
#include "internal-osslThreadUtils.h"
#include "os_api.h"
#include "string.h"
#endif

#ifdef __linux
static struct 
{
    ix_ossl_thread_entry_point_t entryPoint;
    void* arg;
} threadData;

DECLARE_MUTEX(tdMutex);

static int thread_internal(void *unused)
{
    ix_ossl_thread_entry_point_t entryPoint = threadData.entryPoint;
    void *arg = threadData.arg;

    daemonize();
    exit_files(current);

    snprintf(current->comm, sizeof(current->comm), "ixp425_csr");

    up(&tdMutex);
    return entryPoint(arg, NULL);
}
#endif
 
/* FUNCTIONS */


/**
 * NAME: ix_ossl_thread_create
 *
 * DESCRIPTION: This function creates a cancellable thread that will 
 *              execute the user-provided entry point function. Custom 
 *              arguments can be passed to this function using the "arg"
 *              argument. 
 *
 * @Param:  - IN entryPoint:  thread's entry point function.
 * @Param:  - IN  arg      :  pointer to custom arguments that
 *                            will be passed to entry point function
 *                            as the first argument.
 *
 * @Param:  - OUT ptrTid   :  address at which the thread id of the
 *                            newly created thread will be returned
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_thread_create(
                                ix_ossl_thread_entry_point_t entryPoint,
                                void* arg,
	                            ix_ossl_thread_t* ptrTid
                              )
{

#ifdef __linux

    down(&tdMutex);
    threadData.entryPoint = entryPoint;
    threadData.arg = arg;
    
    *ptrTid = kernel_thread(thread_internal, NULL, CLONE_SIGHAND);
    if (*ptrTid < 0)
    {
        up(&tdMutex);
        return IX_OSSL_ERROR_OS_CALL_FAILURE;
    }

    return IX_OSSL_ERROR_SUCCESS;

#else
    ix_error ixError = 0;
    int retVal;
    os_error osError;
    ix_ossl_thread_main_info_t *ptrThreadInfo = NULL;
    
    
    /* 
    * Allocate memory for our internal structure that will contain
    * the actual function and arguments that the caller wants to invoke
    */  
 
	IX_OSSL_ERROR_MALLOC(ptrThreadInfo,
                         ix_ossl_thread_main_info_t*,
                         sizeof(ix_ossl_thread_main_info_t),
                         ixError
                        );     

    if(ixError != 0)
        return ixError;

	ptrThreadInfo->threadMain = entryPoint;
    ptrThreadInfo->threadArg = arg;
    
    /*
    * Spawn new thread with our VxWorks Task -compliant entry point.
    * Note that we are not running the caller's entry point
    * function just yet.
    */
    retVal = os_thread_create(ix_ossl_thread_main_wrapper, ptrThreadInfo,ptrTid,&osError);   
    if(retVal != OS_SUCCESS){
#ifdef DEBUG
			printf("ix_ossl_thread_create: os_thread_create() failed! %d] %s \n",
	                osError.errnum, osError.errname);
#endif /* DEBUG */
								 
			return IX_OSSL_ERROR_NEW(IX_OSSL_ERROR_THREAD_CALL_FAILURE,IX_ERROR_LEVEL_LOCAL);         
				}

    return IX_OSSL_ERROR_SUCCESS;
#endif
}


/**
 * NAME: ix_ossl_thread_get_id
 *
 * DESCRIPTION: This function returns id of the calling thread.
 *
 * @Param:  - OUT ptrTid   :  address at which the id of the calling 
 *                            thread will be returned
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_thread_get_id(ix_ossl_thread_t* ptrTid)
{
#ifdef __linux
   return IX_OSSL_ERROR_FAILURE;
#else

   /* Argument checking */
   int retVal;

   retVal = os_thread_get_id(ptrTid);   
   return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
*   ix_ossl_thread_main_wrapper
*
*   Purpose:    This funciton provides the needed pthread-compliant
*               entry point function. It basically acts as a wrapper
*               around the user-provided function. It still does one
*               important thing which is to set the cancellation type
*               of the thread to ASYNCHRONOUS (which translates into
*               "instantaneous")
*
*   In:         ptrThreadInfo   pointer to our temporary structure
*                               containing pointers to the thread's
*                               entry point function and its
*                               argument structure
*
*   Out:        void
*/
void* ix_ossl_thread_main_wrapper(void* ptrThreadInfo)
{
#ifdef __linux
    return NULL;

#else
    ix_error ixError = 0;
    ix_ossl_thread_main_info_t threadInfo;
    ix_ossl_thread_main_info_t *ptrInfo = NULL;
    void *ptrRetObj = NULL;

    ptrInfo = (ix_ossl_thread_main_info_t*)ptrThreadInfo;

    /*
    * We can't use the usual argument checking macro here because
    * this function does not return an ix_error token
    */
    if(ptrInfo == NULL)   {
				printf("ix_ossl_thread_main_wrapper:pointer to thread"
                       "information structure is null\n");
                goto exit;
    }

    if(ptrInfo->threadMain == NULL){
				printf( "ix_ossl_thread_main_wrapper: "
                        "pointer to thread's entry point function is null\n");  
			    goto exit;
    }

    /*
    * We now copy the user's thread info (thread entry point,
    * and context) on the stack. This will allow us to free
    * the data structure that was on the heap. The reason for
    * all this is that if we don't free it now, we will never
    * be able to free it if the user exits from the thread
    * by calling ix_ossl_thread_exit().
    */
    threadInfo.threadMain   = ptrInfo->threadMain;
    threadInfo.threadArg    = ptrInfo->threadArg;
    IX_OSSL_FREE_POINTER(ptrInfo); 

    
    /* Call the "real" entry point of the thread */
#ifdef DEBUG
    printf(" in wrapper *arg  %d \n", (*(int *)(threadInfo.threadArg)));
#endif /* DEBUG */
    ixError = threadInfo.threadMain(threadInfo.threadArg, &ptrRetObj);

exit:
    /* Terminate thread passing back the ix_error and status object */
    ix_ossl_thread_exit(ixError, ptrRetObj); 
    return NULL;
#endif
}

/**
 * NAME: ix_ossl_thread_exit
 *
 * DESCRIPTION:  This function causes the calling thread to exit. It
 *               gives the opportunity to the caller to pass back to 
 *               a waiting parent a pointer to a custom data structure 
 *               and an ix_error token.
 *
 * @Param:  - IN retError  ix_error token to be returned to the waiting
 *                         parent (0 if no error is returned)
 *
 * @Param:  - IN retObj   pointer to custom data structure returned to
 *                        the waiting parent on thread exit.It is used for
 *                        post-mortem debugging.(null if no data structure 
 *                        is returned)
 *
 */
void ix_ossl_thread_exit(ix_error retError, void* retObj)
{
#ifdef __linux
    /* do nothing */    
#else
    ix_error ixError = 0;
    int retVal;
    int  *ptrRetObj = NULL;
   
    /*
    * Allocate memory for our custom structure for return objects.
    * This structure contains both the error code returned by the
    * thread as well as any optional return object.
    *
    * Note: We are fully aware that allocating resources in an exit
    *       function is bad design practice since the memory allocation
    *       might fail, thus preventing us from returning any kind of
    *       status. 
				*/
    
	IX_OSSL_ERROR_MALLOC(ptrRetObj,
                         int*,
                         sizeof(int),
                         ixError
                        );
	if(ixError != 0)
        goto dump_and_exit;
    
 	  /* Now the thread is officially dead */
				  
	retVal = os_thread_exit((void*)ptrRetObj); 
    IX_OSSL_FREE_POINTER(ptrRetObj);  
	return;
dump_and_exit:
    /* ix_error_dump(stderr, ixError); */
    os_thread_exit(NULL);
    return;

#endif
}



/**
 * NAME: ix_ossl_thread_kill
 *
 * DESCRIPTION:  Kills the running thread specified by its thread id.
 *               Because the thread will be killed instantly, the caller 
 *               must be extremely careful when using this function as 
 *               the thread will not have time to release any of the 
 *               resources it is currently owning. ix_ossl_thread_exit
 *               should be used to delete a thread and its resources instead!. 
 *
 * @Param:  - IN tid:   id of the thread to be killed
 *
 * @Return:  IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_thread_kill(ix_ossl_thread_t tid)
{
#ifdef __linux
    kill_proc(tid, SIGKILL, 1);
    return IX_OSSL_ERROR_SUCCESS;
#else

    int retVal;
    os_error osError;

    retVal = os_thread_kill(tid, &osError);
    if(retVal != OS_SUCCESS)    {
        /* This error code is returned when the tid is invalid */     
#ifdef DEBUG	
		printf("ix_ossl_thread_kill  failed : !\n"
               "[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(IX_OSSL_ERROR_THREAD_CALL_FAILURE,IX_ERROR_LEVEL_LOCAL);                          
    }

    return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:  ix_ossl_thread_set_priority
 *
 * DESCRIPTION:This function sets the priority of the indicated thread.
 *     	       Possible values for 'priority' are IX_OSSL_THREAD_PRI_HIGH, 
 *             IX_OSSL_THREAD_PRI_MED, and IX_OSSL_THREAD_PRI_LOW. 
 *
 * @Param:  - IN tid:   id of the thread 
 * @Param:  - IN priority:  valid priority values are :- IX_OSSL_THREAD_PRI_HIGH, 
 *            IX_OSSL_THREAD_PRI_MED, and IX_OSSL_THREAD_PRI_LOW. 
 *
 * @Return:  IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_thread_set_priority(
                                     ix_ossl_thread_t tid, 
                                     ix_ossl_thread_priority priority
                                    )
{
#ifdef __linux
    return IX_OSSL_ERROR_SUCCESS;
#else

    int retVal;
    os_error osError;

    retVal = os_thread_set_priority(&tid,priority,&osError);
    if(retVal != OS_SUCCESS){
#ifdef DEBUG
    	printf("ix_ossl_thread_set_priority  failed : !\n"
   	                             "[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(IX_OSSL_ERROR_THREAD_CALL_FAILURE,IX_ERROR_LEVEL_LOCAL);
     }

     return IX_OSSL_ERROR_SUCCESS;

#endif
}


/**
 * NAME: ix_ossl_thread_delay
 *
 * DESCRIPTION: This function causes the current task to delay for the specified
 *              number of OS ticks. Control of the CPU is relinquished during this time
 *              allowing other system tasks a chance to execute.
 *
 *
 * @Param:  - IN entryPoint:  number of OS ticks to delay task.
 
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or IX_OSSL_ERROR_FAILURE for failure.
 */
ix_error ix_ossl_thread_delay(
                                int ticks
                              )
{

#ifdef __linux
    
    if ( ticks != 0 )
	{
	    current->state = TASK_INTERRUPTIBLE;
	    schedule_timeout(ticks);
	}
    else
	{
	    schedule();
	}
    return IX_OSSL_ERROR_SUCCESS;

#else
    
    int retVal;
   
    retVal = taskDelay(ticks);

    if (retVal == VXWORKS_SUCCESS)
	{
	    return IX_OSSL_ERROR_SUCCESS;
	}
    else
	{
	    return IX_OSSL_ERROR_OS_CALL_FAILURE;
        }
#endif
}




/**
*   OS Api Calls
*
*   This file contains the implementation of wrapper functions around
*   the standard "pthread" calls. To spawn a new thread the programmer
*   will have to pass the entry point function for that new thread.
*   This entry point function has to adhere to the defined prototype
*   which will enable that thread to return both an ix_error token
*   and a pointer to a caller-defined object to the thread's parent.
*   In addition, functions are defined that allow for the standard
*   wait, exit, and kill functionality.
*
*   Functions:  os_thread_create
*               os_thread_get_id
*               os_thread_exit
*               os_thread_kill
*
 * Author: Intel Corporation
*
*/

#define _DefStackSize_			(16*1024)
#define _DefPriority_			90


/* FUNCTIONS */

/**
 * NAME: os_thread_create
 *
 * DESCRIPTION: Create a cancellable thread that will execute the
 *               user-provided entry point function. Custom arguments
 *               can be passed to this function using the "arg"
 *               argument. 
 *
 * @Param:  - IN       start_routine  pointer to thread's entry point function
 *
 * @Param:  - IN       ptrThreadInfo  pointer to custom argument structure that
 *                                      will be passed to entry point function
 *
 * @Param:  - OUT      ptrTid         address at which the thread id of the
 *                                    newly created thread will be returned
 * @Param:  - OUT      osError        pointer to the datastructure where OS 
 *                                    error codes are returned.
 */
int  os_thread_create(
                      void * (*start_routine)(void *),
                      ix_ossl_thread_main_info_t *ptrThreadInfo,
                      ix_ossl_thread_t* ptrTid, 
                      os_error* osError
                     )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
 
    int retVal;
	WIND_TCB   *pNewTcb;
	char	     *pStack;
	void       *ptrRetObj = NULL;
	char	     tname[11]; /* need 11 bytes to store a hex pointer like "0xNNNNNNNN" */

#ifdef DEBUG 
	printf(" in os_thread_create \n");
#endif /* DEBUG */
      
	/* Step 1: Allocate memory for new thread's WIND_TCB and stack */
	pStack = (char *) malloc((_DefStackSize_) + sizeof(WIND_TCB));

	if (NULL == pStack)	{
				retVal = ENOMEM;
				osError->errnum = retVal;
				strcpy(osError->errname,"EINVAL");  
				return OS_INVALID_ATTR; 		
	}
	
	pNewTcb = (WIND_TCB *) ((char *) pStack + (_DefStackSize_));
	if (ptrThreadInfo == NULL) {
				printf(" in os_thread_create ptrThreadInfo is null \n");
				retVal = EINVAL;
				goto exit;
	}      
	
	sprintf(tname, "0x%08x", (unsigned int) ptrThreadInfo->threadMain);

	retVal = taskInit(pNewTcb,
                      tname,
			          90,
			          0,                  
#if (_STACK_DIR == _STACK_GROWS_DOWN)                	 
			   		  (pStack+(_DefStackSize_)),   
#else
					  pStack,                     
#endif
					  (_DefStackSize_),
					  (FUNCPTR)start_routine,
					  (int)(ptrThreadInfo), (int) &(ptrRetObj),
#ifdef DEBUG 
					  (int)(ptrThreadInfo)->threadMain,
					  ptrThreadInfo->threadArg,&(ptrRetObj),
#endif /* DEBUG */
					  0,0, 0, 0, 0, 0, 0, 0);
	
	if (OK != retVal){
			free((void *)pStack);	
			osError->errnum = retVal; 
			strcpy(osError->errname,"EAGAIN");
			return OS_RESOURCE_SHORTAGE;
	}

	if (OK == retVal){	
			(void) taskActivate((int)pNewTcb);
			*ptrTid = (ix_ossl_thread_t) pNewTcb;
#ifdef DEBUG	
			printf(" *ptrTid  %d \n",(*ptrTid));
#endif /* DEBUG */
	}
     
exit: 
	if(retVal == VXWORKS_SUCCESS){ 
	/* thread is created successfully */
#ifdef DEBUG   
			printf("os_thread_create is successful \n"); 
#endif /* DEBUG */ 
			return OS_SUCCESS; 
	}
    if (retVal == EINVAL){
		
	/* The value specified by attr is invalid */ 
				osError->errnum = retVal;
				strcpy(osError->errname,"EINVAL");  
				return OS_INVALID_ATTR; 
	}
    else if(retVal == EAGAIN){						
					strcpy(osError->errname,"EAGAIN");     
					return OS_RESOURCE_SHORTAGE;
      
     }

    /*  Handle the default case */
	osError->errnum = retVal;    	
	strcpy(osError->errname,"UNKNOWN_ERROR");
    return OS_FAILURE;
#endif 
}

/**
 * NAME: os_thread_get_id
 *
 * DESCRIPTION: Returns the thread id of the calling thread
 *
 * @Param:  - OUT    ptrTid    address at which the thread id of the
 *                             inquiring thread will be returned
 * 
 */
int os_thread_get_id(ix_ossl_thread_t* ptrTid)
{ 
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	/* the thread_self function returns the thread ID of
     the calling thread */ 
#ifdef DEBUG	
	printf(" in os get thread id \n"); 
#endif /* DEBUG */ 
	*ptrTid = taskIdSelf();  
	/* No errors returned */   
	return OS_SUCCESS;
#endif
}


/**
 * NAME: os_thread_exit
 *
 * DESCRIPTION: This function causes the calling thread to exit. It
 *              gives the opportunity (this is not a requirement!) to
 *              the caller to pass back to a waiting parent a pointer
 *              to a custom data structure and an ix_error token.
 *
 *
 * @Param:  - OUT    ptrRetObj  pointer to custom data structure
 *                   (null if no data structure is returned)
 *
 */
int os_thread_exit(void* ptrRetObj)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;	
	ix_ossl_thread_t tid;

#ifdef DEBUG 
	printf(" in os exit thread \n");
#endif /* DEBUG */
	
	tid = taskIdSelf();
	retVal = taskDelete(tid);
	if (retVal == OK)
				return OS_SUCCESS;

	else {
				*(int *)(ptrRetObj) = retVal; 	
				return OS_FAILURE;
	}

#endif
}



/**
 * NAME: os_thread_kill
 *
 * DESCRIPTION: Kills the running thread specified by its thread id.
 *
 * @Param:  - IN         tid     id of the thread to be killed
 *
 * @Param:  - OUT       osError  pointer to the datastructure where OS 
 *                               error conditions are returned. 
 */
int  os_thread_kill(ix_ossl_thread_t tid, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;
#ifdef DEBUG  
	printf(" in os kill thread tid %d \n",tid);
#endif /* DEBUG */
	retVal = taskDelete(tid);

	if (OK != retVal){
			osError->errnum = retVal;
        	strcpy(osError->errname,"ESRCH"); 
			return OS_INVALID_THREADID;
	}
#ifdef DEBUG
	else {
			printf(" in os_thread_kill : tid %d is killed \n",tid);
	}
#endif /* DEBUG */
	
	/* default error value */ 
	osError->errnum = retVal;   
	strcpy(osError->errname,"UNKNOWN_ERROR");
	return OS_FAILURE;
#endif
}


/**
 * NAME: os_thread_set_priority
 *
 * DESCRIPTION: This function sets the priority of the indicated thread.
 *     	        possible values for 'priority' are IX_OSSL_PRI_HIGH, 
 *              IX_OSSL_PRI_MED, and IX_OSSL_PRI_LOW. The effect of priority
 *              is OS dependant.  	  
 *
 * @Param:  - IN    pointer to the thread object 
 * @Param:  - IN    priority  priority level.
 * @Param:  - OUT   osError   pointer to the datastructure where OS 
 *                  error conditions are returned.
 *
 */
ix_error os_thread_set_priority(
                                ix_ossl_thread_t* tid, 
                                ix_ossl_thread_priority priority,
			                    os_error* osError
                               )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;

    retVal = taskPrioritySet(*((int *)tid),priority);
	
    if (OK == retVal) return OS_SUCCESS;

    osError->errnum = retVal;
    strcpy(osError->errname,"UNKNOWN_ERROR");
    return OS_FAILURE;
#endif
}
