
/**
 * ============================================================================
 * = COPYRIGHT
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
 * = PRODUCT
 *      Intel(r) IXP425 Software Release
 *
 * = LIBRARY
 *      OSSL ( Operating System Services)  Library
 *
 * = MODULE
 *      OSSL Mutexes
 *
 * = FILENAME
 *      osslMutex.c
 *
 * = DESCRIPTION
 *      
 *
 *   Contains the VxWorks C wrapper functions that allows for creation,
 *   locking, unlocking and deletion of thread mutexes.
 *
 *   Function: ix_ossl__mutex_init
 *             ix_ossl_mutex_lock
 *             ix_ossl_mutex_unlock
 *             ix_ossl_mutex_fini
 *
 * = AUTHOR
 *      Intel Corporation
 *
 * = AKNOWLEDGEMENTS
 *      
 *
 * = CREATION TIME
 *      1/9/2002 1:56:24 PM
 *
 * = CHANGE HISTORY
 *
 * ============================================================================
 */


/* Header Files */
#include "ix_ossl.h"

#ifndef __linux
#include "internal-osslThreadUtils.h"
#include "os_api.h"
#include <string.h>
#endif


/* FUNCTIONS */

/**
 * NAME:    ix_ossl_mutex_init
 *
 * DESCRIPTION: This function initializes a new mutex. 'mid' is a pointer
 *              to an ix_ossl_mutex_t. Upon success, '*mid' will contain
 *              the mutex id. 'start_state' is the initial locking
 *              state.
 *
 * @Param:  - IN start_state: 'start_state' is  initial locking state.
 *                            Valid values are :-
 *                            'IX_OSSL_MUTEX_LOCK': Lock the mutex
 *                            'IX_OSSL_MUTEX_UNLOCK':  mutex unlocked
 *
 * @Param:  - OUT mid:  pointer to id of the mutex created
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_mutex_init(
                            ix_ossl_mutex_state start_state,
                            ix_ossl_mutex_t* mid
                           )
{
#ifdef __linux
    return ix_ossl_sem_init(!start_state, (ix_ossl_sem_t *)mid);
#else
	int retVal;
    os_error osError;

    /* Call a osapi function for mutex creation */
    retVal = os_thread_mutex_create(start_state,mid, &osError);

    if(retVal != OS_SUCCESS)  {
		/* Mutex creation is not successful */
#ifdef DEBUG
			printf("ix_ossl_mutex_init: pthread_mutex_init() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */
	    	return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);
   	}    
	return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:   ix_ossl_mutex_lock
 *
 * DESCRIPTION: This function locks the mutex. If the mutex is already
 *              locked, the thread will block. If the time indicated in
 *              'timeout' is reached, the thread will unblock and return
 *              error indication. If the timeout is set to 'IX_OSSL_WAIT_NONE', 
 *              the thread will never block (this is trylock). If the timeout
 *              is set to 'IX_OSSL_WAIT_FOREVER',  the thread will block until
 *              the mutex is  unlocked. 
 *
 * @Param:  - IN mid:  mutex id.
 * @Param:  - IN timeout:  timeout value of type ix_uint32 expressed in miliseconds.
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_mutex_lock(
                            ix_ossl_mutex_t mid,
                            ix_uint32 timeout
                           )
{ 	
#ifdef __linux
    return ix_ossl_sem_take((ix_ossl_sem_t)mid, timeout);
#else

	int retVal;
    os_error osError;
   
	/* Call a osapi function for mutex lock */
    retVal = os_thread_mutex_lock(&mid, timeout, &osError);

    if(retVal != OS_SUCCESS)   { 
		/* Mutex lock failed */
#ifdef DEBUG
	   printf("ix_ossl_mutex_lock: os_thread_mutex_lock() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */	
	   return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);  
	}
	return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:  ix_ossl_mutex_unlock
 *
 * DESCRIPTION: This function unlocks the mutex.  'mid' is the mutex id.
 *              If there are threads pending on the mutex, the next one is 
 *              given the lock. If there are no pending threads, then the 
 *              mutex is unlocked.
 *
 * @Param:  - IN mid:  mutex id
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_mutex_unlock(ix_ossl_mutex_t  mid)
{
#ifdef __linux
    return ix_ossl_sem_give((ix_ossl_sem_t)mid);
#else
	int retVal;
	os_error osError;
   
	/* Call the osapi function for mutex unlock */
    retVal = os_thread_mutex_unlock(&mid, &osError);

    if(retVal != OS_SUCCESS) {    
		/* Mutex unlock failed */
#ifdef DEBUG
		printf( "ix_ossl_mutex_unlock: os_thread_mutex_unlock() failed\n"
                            "\t[%d] %s\n", osError.errnum, osError.errname);  
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);
	}    
	return IX_OSSL_ERROR_SUCCESS;
#endif
}

/**
 * NAME:   ix_ossl_mutex_fini
 *
 * DESCRIPTION: This function frees a mutex.'mid' is the id mutex id.
 *              The mutex is deleted, all resources are freed. Any threads
 *              pending on this mutex will be unblocked and return an error.
 *
 * @Param:  - IN mid:  mutex id
 *
 * @Return: IX_OSSL_ERROR_SUCCESS if successful or a valid ix_error token for failure.
 */
ix_error ix_ossl_mutex_fini(ix_ossl_mutex_t mid)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

    int retVal;
    os_error osError;

    /* Call the osapi function for mutex destroy */
    retVal = os_thread_mutex_destroy(&mid, &osError);

    if(retVal != OS_SUCCESS){
		/* mutex fini failed */
#ifdef DEBUG
		printf("ix_ossl_mutex_destroy: os_thread_mutex_destroy() failed\n"
               "\t[%d] %s\n", osError.errnum, osError.errname);
#endif /* DEBUG */
		return IX_OSSL_ERROR_NEW(retVal,IX_ERROR_LEVEL_LOCAL);
	}

	return IX_OSSL_ERROR_SUCCESS;
#endif
}


/**
*   OS Mutex APIs
*
*   Contains the APIs for VxWorks C wrapper functions that allows
*   for creation, locking, unlocking and deletion of thread mutexes.
*
*   Function: os_thread_mutex_create
*             os_thread_mutex_lock
*             os_thread_mutex_unlock
*             os_thread_mutex_destroy
*
 * Author: Intel Corporation
*
*/



/**
 * NAME: os_thread_mutex_create
 *
 * DESCRIPTION: Create a thread mutex object.
 *
 * @Param:  - IN      start_state  the value that the mutex state should be initialized to.
 *
 * @Param:  - OUT     mid          id of the thread mutex created.
 * @Param:  - OUT     osError      pointer to the datastructure where the
 *                                 error conditions are returned.
 *
 */
int  os_thread_mutex_create(
                            ix_ossl_mutex_state start_state,
                            ix_ossl_mutex_t* mid,
                            os_error* osError
                           )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;
	SEM_ID mutex;

	/* ceate a mutual exclusion semaphore */
	mutex =   semMCreate(SEM_Q_FIFO);
	if (mutex == NULL){ 
		/* error creating the mutex */	
		retVal = OS_NOMEM;
		osError->errnum = retVal;
     	strcpy(osError->errname,"ENOMEM");
      	return retVal;	
	}
 
	else {       
	/* Mutex creation is successful. store the
       mutex id in mid */
		
		*mid = (ix_ossl_mutex_t) mutex;	
		retVal = OS_SUCCESS;
	
		/* check for the start state of the mutex */
		if (start_state == IX_OSSL_MUTEX_LOCK)	{ 	
			/* Lock the mutex */	 
			retVal =   semTake(*(SEM_ID *)mutex,NO_WAIT);	
			/* check the retVal for the error condition returned */
			switch(retVal)	 {

			case SEMA_SUCCESS :  /* Successful. */
				{
					return OS_SUCCESS;
				}
			case S_objLib_OBJ_TIMEOUT :  /* A Timeout occured */
				{
					osError->errnum = retVal;
					strcpy(osError->errname,"S_objLib_OBJ_TUMEOUT");
					/* return OS_TIMEDOUT; */
                    return OS_SUCCESS;
				}
			case S_objLib_OBJ_ID_ERROR :
				{
					/* The value specified by attr is invalid */
					osError->errnum = retVal;
					strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");
					return OS_INVALID_ATTR;
				}
			case S_objLib_OBJ_UNAVAILABLE :
				{
					/* Invalid Object */
					osError->errnum = retVal;
					strcpy(osError->errname,"S_objLib_OBJ_UNAVAILABLE");
					return OS_NOSYS;
				}
			case S_intLib_NOT_ISR_CALLABLE :
				{
					/* Error calling this function */
					osError->errnum = retVal;
					strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE");
					return OS_ERR_INTR;
				}
			default:
				{
					osError->errnum = retVal;
					strcpy(osError->errname,"UNKNOWN_ERROR");
					return OS_FAILURE;
				}
			}   /* end switch */
		}  /* end if start_state == IX_OSSL_MUTEX_LOCK */
		return retVal;
	}  /* end else */

	/* default error */

	osError->errnum = retVal;
	strcpy(osError->errname,"UNKNOWN_ERROR");
	return OS_FAILURE;
#endif
}

/**
 * NAME: os_thread_mutex_lock
 *
 * DESCRIPTION: This function locks the mutex. If the mutex is already
 *              locked, the task will block. If the time indicated in
 *              'timeout' is reached, the task will unblock and retun
 *              error indication. If timeout is set to '0', the task will
 *              never block.
 *
 * @Param:  - IN  mutex      pointer to the thread mutex object.
 * @Param:  - IN  timeout   'timeout' value
 * @Param:  - OUT osError   pointer to the datastructure where the
 *                          error conditions are returned.
 */
int  os_thread_mutex_lock(
                          ix_ossl_mutex_t* mutex,
                          ix_uint32 timeout,
                          os_error* osError
                         )
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
  	
	int retVal;
#ifdef DEBUG	
	printf(" in os_thread_mutex_lock : timeout %d mutex id %d \n",timeout,*mutex);
#endif /* DEBUG */	
	retVal =   semTake(*(SEM_ID *)mutex,timeout);

	switch(retVal){	  
	case SEMA_SUCCESS :
		{    	  
			return OS_SUCCESS;     
    	} 
	case S_objLib_OBJ_TIMEOUT :
		{    	  
			osError->errnum = retVal;     	 
			strcpy(osError->errname,"S_objLib_OBJ_TUMEOUT");     	 
			return OS_TIMEDOUT;    	 
		}		
  	case S_objLib_OBJ_ID_ERROR :  	 
		{         	  
			/* The value specified by attr is invalid */    	 
			osError->errnum = retVal;  
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");    
			return OS_INVALID_ATTR;  
		}    	
	case S_objLib_OBJ_UNAVAILABLE :  	
		{        	  
			osError->errnum = retVal;    	
			strcpy(osError->errname,"S_objLib_OBJ_UNAVAILABLE"); 
			return OS_NOSYS;  
		}  	
	case S_intLib_NOT_ISR_CALLABLE :   	
		{
			osError->errnum = retVal;     	 
			strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE");      	
			return OS_ERR_INTR;    	
		}	 
	default: 	
		{    	  
			osError->errnum = retVal;     	
			strcpy(osError->errname,"UNKNOWN_ERROR");      	
			return OS_FAILURE;
    	}  	
	}    
#endif
}


/**
 * NAME: os_thread_mutex_unlock
 *
 * DESCRIPTION: This function unlocks the mutex. If there are tasks
 *              pending on the mutex, the next one is given the lock.
 *              If there are no pending tasks, then the mutex is
 *              unlocked.
 *
 * @Param:  - IN   mutex       pointer to the thread mutex object.
 * @Param:  - OUT  osError     pointer to the datastructure where the
 *                             error conditions are returned.
 */
int  os_thread_mutex_unlock(ix_ossl_mutex_t* mutex, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else

	int retVal;

	retVal = semGive(*(SEM_ID *)mutex);

	switch(retVal){
	case SEMA_SUCCESS  :
		{     
			return OS_SUCCESS;   
		}   
	case  S_objLib_OBJ_ID_ERROR :
		{       
			/* The value specified by attr is invalid */
			osError->errnum = retVal; 
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");   
			return OS_INVALID_ATTR;   
		}  
	case  S_semLib_INVALID_OPERATION :  
		{          
			osError->errnum = retVal;   
			strcpy(osError->errname,"S_semLib_INVALID_OPERATION");   
			return OS_INVALID_OPERATION; 
		}  
	case  S_intLib_NOT_ISR_CALLABLE :
		{
			osError->errnum = retVal; 
			strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE");    
			return OS_ERR_INTR;   
		}
	default:  
		{  
			osError->errnum = retVal; 
			strcpy(osError->errname,"UNKNOWN_ERROR"); 
			return OS_FAILURE;    
		} 
	}   
#endif
}



/**
 * NAME: os_thread_mutex_destroy
 *
 * DESCRIPTION: Destroy the thread mutex object.
 *
 * @Param:  - IN    mutex       pointer to the thread mutex object.
 * @Param:  - OUT   osError   pointer to the datastructure where the
 *                            error conditions are returned.
 */
int  os_thread_mutex_destroy(ix_ossl_mutex_t* mutex, os_error* osError)
{
#ifdef __linux
    return IX_OSSL_ERROR_OS_CALL_FAILURE;
#else
 
	int retVal;
 
	retVal = semDelete(*(SEM_ID *)mutex); 
 
	switch(retVal){
	case SEMA_SUCCESS  :   
		{  
			return OS_SUCCESS;        
		}  
	case  S_objLib_OBJ_ID_ERROR : 
		{         
			/* The value specified by attr is invalid */  
			osError->errnum = retVal;   
			strcpy(osError->errname,"S_objLib_OBJ_ID_ERROR");    
			return OS_INVALID_ATTR; 
		}  
	case  S_smObjLib_NO_OBJECT_DESTROY :  
		{                
			osError->errnum = retVal;
			strcpy(osError->errname,"S_semLib_INVALID_OPERATION"); 
			return OS_INVALID_OPERATION;   
		}   
	case  S_intLib_NOT_ISR_CALLABLE :    
		{    
			osError->errnum = retVal; 
			strcpy(osError->errname,"S_intLib_NOT_ISR_CALLABLE");   
			return OS_ERR_INTR;   
		} 
	default:    
		{     
			osError->errnum = retVal;     
			strcpy(osError->errname,"UNKNOWN_ERROR");     
			return OS_FAILURE;
		}    
	}
#endif
}

