/**
 * @file    IxNpeA.h
 *
 * @date    22-Mar-2002
 *
 * @brief   Header file for the IXP425 ATM NPE API
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

/**
 * @defgroup IxNpeA IXP425 NPE-A (IxNpeA) API
 *
 * @brief The Public API for the IXP425 NPE-A
 *
 * @{
 */

#ifndef IX_NPE_A_H
#define IX_NPE_A_H

#include "IxTypes.h"
#include "IxOsBuffMgt.h"

/* General Message Ids */

/* ATM Message Ids */

/**
 * @def IX_NPE_A_MSSG_ATM_UTOPIA_CONFIG_WRITE
 *
 * @brief ATM Message ID command to write the data to the offset in the
 * Utopia Configuration Table
 */
#define IX_NPE_A_MSSG_ATM_UTOPIA_CONFIG_WRITE       0x20

/**
 * @def IX_NPE_A_MSSG_ATM_UTOPIA_CONFIG_LOAD
 *
 * @brief ATM Message ID command triggers the NPE to copy the Utopia
 * Configuration Table to the Utopia coprocessor
 */
#define IX_NPE_A_MSSG_ATM_UTOPIA_CONFIG_LOAD        0x21

/**
 * @def IX_NPE_A_MSSG_ATM_UTOPIA_STATUS_UPLOAD
 *
 * @brief ATM Message ID command triggers the NPE to read-back the Utopia
 * status registers and update the Utopia Status Table.
 */
#define IX_NPE_A_MSSG_ATM_UTOPIA_STATUS_UPLOAD      0x22

/**
 * @def IX_NPE_A_MSSG_ATM_UTOPIA_STATUS_READ
 *
 * @brief ATM Message ID command to read the Utopia Status Table at the
 * specified offset.
 */
#define IX_NPE_A_MSSG_ATM_UTOPIA_STATUS_READ        0x23

/**
 * @def IX_NPE_A_MSSG_ATM_TX_ENABLE
 *
 * @brief ATM Message ID command triggers the NPE to re-enable processing
 * of any entries on the TxVcQ for this port.
 *
 * This command will be ignored for a port already enabled
 */
#define IX_NPE_A_MSSG_ATM_TX_ENABLE                 0x25

 /**
 * @def IX_NPE_A_MSSG_ATM_TX_DISABLE
 *
 * @brief ATM Message ID command triggers the NPE to disable processing on
 * this port
 *
 * This command will be ignored for a port already disabled
 */
#define IX_NPE_A_MSSG_ATM_TX_DISABLE                0x26

/**
 * @def IX_NPE_A_MSSG_ATM_RX_ENABLE
 *
 * @brief ATM Message ID command triggers the NPE to process any received
 * cells for this VC according to the VC Lookup Table.
 *
 * Re-issuing this command with different contents for a VC that is not
 * disabled will cause unpredictable behavior.
 */
#define IX_NPE_A_MSSG_ATM_RX_ENABLE                 0x27

/**
 * @def IX_NPE_A_MSSG_ATM_RX_DISABLE
 *
 * @brief ATM Message ID command triggers the NPE to disable processing for
 * this VC.
 *
 * This command will be ignored for a VC already disabled
 */
#define IX_NPE_A_MSSG_ATM_RX_DISABLE                0x28

/**
 * @def IX_NPE_A_MSSG_ATM_STATUS_READ
 *
 * @brief ATM Message ID command to read the ATM status. The data is returned via
 * a response message
 */
#define IX_NPE_A_MSSG_ATM_STATUS_READ               0x29

/*--------------------------------------------------------------------------
 * HSS Message IDs
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_MSSG_HSS_PORT_CONFIG_WRITE
 *
 * @brief HSS Message ID command writes the ConfigWord value to the location
 * in the HSS_CONFIG_TABLE specified by offset for HSS port hPort.
 */
#define IX_NPE_A_MSSG_HSS_PORT_CONFIG_WRITE         0x40

/**
 * @def IX_NPE_A_MSSG_HSS_PORT_CONFIG_LOAD
 *
 * @brief HSS Message ID command triggers the NPE to copy the contents of the
 * HSS Configuration Table to the appropriate configuration registers in the
 * HSS coprocessor for the port specified by hPort.
 */
#define IX_NPE_A_MSSG_HSS_PORT_CONFIG_LOAD          0x41

/**
 * @def IX_NPE_A_MSSG_HSS_PORT_ERROR_READ
 *
 * @brief HSS Message ID command triggers the NPE to return an HssErrorReadResponse
 * message for HSS port hPort.
 */
#define IX_NPE_A_MSSG_HSS_PORT_ERROR_READ           0x42

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_FLOW_ENABLE
 *
 * @brief HSS Message ID command triggers the NPE to reset internal status and
 * enable the HssChannelized operation on the HSS port specified by hPort.
 */
#define IX_NPE_A_MSSG_HSS_CHAN_FLOW_ENABLE          0x43

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_FLOW_DISABLE
 *
 * @brief HSS Message ID command triggers the NPE to disable the HssChannelized
 * operation on the HSS port specified by hPort.
 */
#define IX_NPE_A_MSSG_HSS_CHAN_FLOW_DISABLE         0x44

/**
 * @def  IX_NPE_A_MSSG_HSS_CHAN_IDLE_PATTERN_WRITE
 *
 * @brief HSS Message ID command writes the HSSnC_IDLE_PATTERN value for HSS
 * port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_IDLE_PATTERN_WRITE   0x45

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_NUM_CHANS_WRITE
 *
 * @brief HSS Message ID command writes the HSSnC_NUM_CHANNELS value for HSS
 * port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_NUM_CHANS_WRITE      0x46

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_RX_BUF_ADDR_WRITE
 *
 * @brief HSS Message ID command writes the HSSnC_RX_BUF_ADDR value for HSS
 * port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_RX_BUF_ADDR_WRITE    0x47

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_RX_BUF_CFG_WRITE
 *
 * @brief HSS Message ID command  writes the HSSnC_RX_BUF_SIZEB and
 * HSSnC_RX_TRIG_PERIOD values for HSS port hPort.  (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_RX_BUF_CFG_WRITE     0x48

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_TX_BLK_CFG_WRITE
 *
 * @brief HSS Message ID command writes the HSSnC_TX_BLK1_SIZEB,
 * HSSnC_TX_BLK1_SIZEW, HSSnC_TX_BLK2_SIZEB, and HSSnC_TX_BLK2_SIZEW  values
 * for HSS port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_TX_BLK_CFG_WRITE     0x49

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_TX_BUF_ADDR_WRITE
 * @brief HSS Message ID command writes the HSSnC_TX_BUF_ADDR value for HSS
 * port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_TX_BUF_ADDR_WRITE    0x4A

/**
 * @def IX_NPE_A_MSSG_HSS_CHAN_TX_BUF_SIZE_WRITE
 *
 * @brief HSS Message ID command writes the HSSnC_TX_BUF_SIZEN value for HSS
 * port hPort. (n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_CHAN_TX_BUF_SIZE_WRITE    0x4B

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_FLOW_ENABLE
 *
 * @brief HSS Message ID command triggers the NPE to reset internal status and
 * enable the HssPacketized operation for the flow specified by pPipe on
 * the HSS port specified by hPort.
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_FLOW_ENABLE      0x50

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_FLOW_DISABLE
 * @brief HSS Message ID command triggers the NPE to disable the HssPacketized
 * operation for the flow specified by pPipe on the HSS port specified by hPort.
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_FLOW_DISABLE     0x51

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_NUM_PIPES_WRITE
 * @brief HSS Message ID command writes the HSSnP_NUM_PIPES value for HSS
 * port hPort.(n=hPort)
 */
#define IX_NPE_A_MSSG_HSS_PKT_NUM_PIPES_WRITE       0x52

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_FIFO_SIZEW_WRITE
 *
 * @brief HSS Message ID command writes the HSSnP_PIPEp_FIFOSIZEW value for
 * packet-pipe pPipe on HSS port hPort.  (n=hPort, p=pPipe)
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_FIFO_SIZEW_WRITE 0x53

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_HDLC_CFG_WRITE
 *
 * @brief HSS Message ID command writes the HSSnP_PIPEp_HDLC_RXCFG and
 * HSSnP_PIPEp_HDLC_TXCFG values for packet-pipe pPipe on HSS port hPort.
 * (n=hPort, p=pPipe)
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_HDLC_CFG_WRITE   0x54

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_IDLE_PATTERN_WRITE
 *
 * @brief HSS Message ID command writes the HSSnP_PIPEp_IDLE_PATTERN value
 * for packet-pipe pPipe on HSS port hPort.  (n=hPort, p=pPipe)
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_IDLE_PATTERN_WRITE 0x55

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_RX_SIZE_WRITE
 *
 * @brief HSS Message ID command writes the HSSnP_PIPEp_RXSIZEB value for
 * packet-pipe pPipe on HSS port hPort.  (n=hPort, p=pPipe)
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_RX_SIZE_WRITE    0x56

/**
 * @def IX_NPE_A_MSSG_HSS_PKT_PIPE_MODE_WRITE
 *
 * @brief HSS Message ID command writes the HSSnP_PIPEp_MODE value for
 * packet-pipe pPipe on HSS port hPort.  (n=hPort, p=pPipe)
 */
#define IX_NPE_A_MSSG_HSS_PKT_PIPE_MODE_WRITE       0x57



/* Queue Entry Masks */

/*--------------------------------------------------------------------------
 *  ATM Descriptor Structure offsets
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_RXDESCRIPTOR_STATUS_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Status field
 *
 * It is used for descriptor error reporting.
 */
#define IX_NPE_A_RXDESCRIPTOR_STATUS_OFFSET          0

/**
 * @def IX_NPE_A_RXDESCRIPTOR_VCID_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor VC ID field
 *
 * It is used to hold an identifier number for this VC
 */
#define IX_NPE_A_RXDESCRIPTOR_VCID_OFFSET            1

/**
 * @def IX_NPE_A_RXDESCRIPTOR_CURRMBUFSIZE_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Current Mbuf
 * Size field
 *
 * Number of bytes the current mbuf data buffer can hold
 */
#define IX_NPE_A_RXDESCRIPTOR_CURRMBUFSIZE_OFFSET    2

/**
 * @def IX_NPE_A_RXDESCRIPTOR_ATMHEADER_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor ATM Header
 */
#define IX_NPE_A_RXDESCRIPTOR_ATMHEADER_OFFSET       4

/**
 * @def IX_NPE_A_RXDESCRIPTOR_CURRMBUFLEN_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Current MBuf length
 *
 *
 * RX - Initialized to zero.  The NPE updates this field as each cell is received and
 * zeroes it with every new mbuf for chaining. Will not be bigger than currBbufSize.
 */
#define IX_NPE_A_RXDESCRIPTOR_CURRMBUFLEN_OFFSET    12

/**
 * @def IX_NPE_A_RXDESCRIPTOR_TIMELIMIT__OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Time Limit field
 *
 * Contains the Payload Reassembly Time Limit (used for aal0_xx only)
 */
#define IX_NPE_A_RXDESCRIPTOR_TIMELIMIT_OFFSET        14

/**
 * @def IX_NPE_A_RXDESCRIPTOR_PCURRMBUFF_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Current MBuf Pointer
 *
 * The current mbuf pointer of a chain of mbufs.
 */
#define IX_NPE_A_RXDESCRIPTOR_PCURRMBUFF_OFFSET     20

/**
 * @def IX_NPE_A_RXDESCRIPTOR_PCURRMBUFDATA_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Current MBuf Pointer
 *
 * Pointer to the next byte to be read or next free location to be written.
 */
#define IX_NPE_A_RXDESCRIPTOR_PCURRMBUFDATA_OFFSET  24

/**
 * @def IX_NPE_A_RXDESCRIPTOR_PNEXTMBUF_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Next MBuf Pointer
 *
 * Pointer to the next MBuf in a chain of MBufs.
 */
#define IX_NPE_A_RXDESCRIPTOR_PNEXTMBUF_OFFSET      28

/**
 * @def IX_NPE_A_RXDESCRIPTOR_TOTALLENGTH_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Total Length
 *
 * Total number of bytes written to the chain of MBufs by the NPE
 */
#define IX_NPE_A_RXDESCRIPTOR_TOTALLENGTH_OFFSET    32

/**
 * @def IX_NPE_A_RXDESCRIPTOR_AAL5CRCRESIDUE_OFFSET
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor AAL5 CRC Residue
 *
 * Current CRC value for a PDU
 */
#define IX_NPE_A_RXDESCRIPTOR_AAL5CRCRESIDUE_OFFSET 36

/**
 * @def IX_NPE_A_RXDESCRIPTOR_SIZE
 *
 * @brief ATM Descriptor structure offset for Receive Descriptor Size
 *
 * The size of the Receive descriptor
 */
#define IX_NPE_A_RXDESCRIPTOR_SIZE                  40

/**
 * @def IX_NPE_A_TXDESCRIPTOR_PORT_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Port
 *
 * Port identifier.
 */
#define IX_NPE_A_TXDESCRIPTOR_PORT_OFFSET            0

/**
 * @def IX_NPE_A_TXDESCRIPTOR_RSVD_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor RSVD
 */
#define IX_NPE_A_TXDESCRIPTOR_RSVD_OFFSET            1

/**
 * @def IX_NPE_A_TXDESCRIPTOR_CURRMBUFLEN_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Current MBuf Length
 *
 * TX - Initialized by the XScale to the number of bytes in the current MBuf data buffer.
 * The NPE decrements this field for every transmitted cell.  Thus, when the NPE writes a
 * descriptor the TxDone queue, this field will equal zero.
 */
#define IX_NPE_A_TXDESCRIPTOR_CURRMBUFLEN_OFFSET     2

/**
 * @def IX_NPE_A_TXDESCRIPTOR_ATMHEADER_OFFSET
 * @brief ATM Descriptor structure offset for Transmit Descriptor ATM Header
 */
#define IX_NPE_A_TXDESCRIPTOR_ATMHEADER_OFFSET       4

/**
 * @def IX_NPE_A_TXDESCRIPTOR_PCURRMBUFF_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Pointer to the current MBuf chain
 */
#define IX_NPE_A_TXDESCRIPTOR_PCURRMBUFF_OFFSET      8

/**
 * @def IX_NPE_A_TXDESCRIPTOR_PCURRMBUFDATA_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Pointer to the current MBuf Data
 *
 * Pointer to the next byte to be read or next free location to be written.
 */
#define IX_NPE_A_TXDESCRIPTOR_PCURRMBUFDATA_OFFSET  12

/**
 * @def IX_NPE_A_TXDESCRIPTOR_PNEXTMBUF_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Pointer to the Next MBuf chain
 */
#define IX_NPE_A_TXDESCRIPTOR_PNEXTMBUF_OFFSET      16

/**
 * @def IX_NPE_A_TXDESCRIPTOR_TOTALLENGTH_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Total Length
 *
 * Total number of bytes written to the chain of MBufs by the NPE
 */
#define IX_NPE_A_TXDESCRIPTOR_TOTALLENGTH_OFFSET    20

/**
 * @def IX_NPE_A_TXDESCRIPTOR_AAL5CRCRESIDUE_OFFSET
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor AAL5 CRC Residue
 *
 * Current CRC value for a PDU
 */
#define IX_NPE_A_TXDESCRIPTOR_AAL5CRCRESIDUE_OFFSET 24

/**
 * @def IX_NPE_A_TXDESCRIPTOR_SIZE
 *
 * @brief ATM Descriptor structure offset for Transmit Descriptor Size
 */
#define IX_NPE_A_TXDESCRIPTOR_SIZE                  28

/* Global compiler switch to select between 3 possible NPE Modes */
/* Default(No Switch) = MultiPHY Utopia2 */
/* IX_UTOPIAMODE=1 for single Phy Utopia1 */
/* IX_MPHYSINGLEPORT=1 for single Phy Utopia2 */
/** @brief Define this macro to enable MPHY mode */
#define IX_NPE_MPHYMULTIPORT 1
#if IX_UTOPIAMODE == 1
#undef  IX_NPE_MPHYMULTIPORT
#endif
#if IX_MPHYSINGLEPORT == 1
#undef  IX_NPE_MPHYMULTIPORT
#endif
/*--------------------------------------------------------------------------
 *  Hardware queue IDs relating to Atm Part-I
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_TXDONE_QUEUE_HIGHWATERMARK
 *
 * @brief The NPE reserves the High Watermark for its operation. But it must be set by the Xscale
 */
#define IX_NPE_A_TXDONE_QUEUE_HIGHWATERMARK  2

/**
 * @def IX_NPE_A_QMQ_ATM_TX_DONE
 *
 * @brief Queue ID for ATM Transmit Done queue
 */
#define IX_NPE_A_QMQ_ATM_TX_DONE       1

/**
 * @def IX_NPE_A_QMQ_ATM_TX0
 *
 * @brief Queue ID for ATM transmit Queue in a single phy configuration
 */
#define IX_NPE_A_QMQ_ATM_TX0           2


/**
 * @def IX_NPE_A_QMQ_ATM_TXID_MIN
 *
 * @brief Queue Manager Queue ID for ATM transmit Queue with minimum number of queue
 *
 */
/**
 * @def IX_NPE_A_QMQ_ATM_TXID_MAX
 *
 * @brief Queue Manager Queue ID for ATM transmit Queue with maximum number of queue
 *
 */
/**
 * @def IX_NPE_A_QMQ_ATM_RX_HI
 *
 * @brief Queue Manager Queue ID for ATM Receive high Queue
 *
 */
/**
 * @def IX_NPE_A_QMQ_ATM_RX_LO
 *
 * @brief Queue Manager Queue ID for ATM Receive low Queue
 */
#ifdef IX_NPE_MPHYMULTIPORT
/**
 * @def IX_NPE_A_QMQ_ATM_TX1
 *
 * @brief Queue ID for ATM transmit Queue Multiphy from 1 to 11
 */
#define IX_NPE_A_QMQ_ATM_TX1           IX_NPE_A_QMQ_ATM_TX0+1
#define IX_NPE_A_QMQ_ATM_TX2           IX_NPE_A_QMQ_ATM_TX1+1
#define IX_NPE_A_QMQ_ATM_TX3           IX_NPE_A_QMQ_ATM_TX2+1
#define IX_NPE_A_QMQ_ATM_TX4           IX_NPE_A_QMQ_ATM_TX3+1
#define IX_NPE_A_QMQ_ATM_TX5           IX_NPE_A_QMQ_ATM_TX4+1
#define IX_NPE_A_QMQ_ATM_TX6           IX_NPE_A_QMQ_ATM_TX5+1
#define IX_NPE_A_QMQ_ATM_TX7           IX_NPE_A_QMQ_ATM_TX6+1
#define IX_NPE_A_QMQ_ATM_TX8           IX_NPE_A_QMQ_ATM_TX7+1
#define IX_NPE_A_QMQ_ATM_TX9           IX_NPE_A_QMQ_ATM_TX8+1
#define IX_NPE_A_QMQ_ATM_TX10          IX_NPE_A_QMQ_ATM_TX9+1
#define IX_NPE_A_QMQ_ATM_TX11          IX_NPE_A_QMQ_ATM_TX10+1
#define IX_NPE_A_QMQ_ATM_TXID_MIN      IX_NPE_A_QMQ_ATM_TX0
#define IX_NPE_A_QMQ_ATM_TXID_MAX      IX_NPE_A_QMQ_ATM_TX11
#define IX_NPE_A_QMQ_ATM_RX_HI         21
#define IX_NPE_A_QMQ_ATM_RX_LO         22
#else
#define IX_NPE_A_QMQ_ATM_TXID_MIN      IX_NPE_A_QMQ_ATM_TX0
#define IX_NPE_A_QMQ_ATM_TXID_MAX      IX_NPE_A_QMQ_ATM_TX0
#define IX_NPE_A_QMQ_ATM_RX_HI         10
#define IX_NPE_A_QMQ_ATM_RX_LO         11
#endif /* MPHY */

/*--------------------------------------------------------------------------
 * Hardware queue IDs relating to HSS Port 0
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_QMQ_HSS0_CHL_RX_TRIG
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Channelized Receive trigger
 */
#define IX_NPE_A_QMQ_HSS0_CHL_RX_TRIG   12

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_RX
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Receive
 */
#define IX_NPE_A_QMQ_HSS0_PKT_RX        13

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_TX0
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Transmit queue 0
 */
#define IX_NPE_A_QMQ_HSS0_PKT_TX0       14

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_TX1
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Transmit queue 1
 */
#define IX_NPE_A_QMQ_HSS0_PKT_TX1       15

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_TX2
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Transmit queue 2
 */
#define IX_NPE_A_QMQ_HSS0_PKT_TX2       16

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_TX3
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Transmit queue 3
 */
#define IX_NPE_A_QMQ_HSS0_PKT_TX3       17

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_RX_FREE0
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Receive Free queue 0
 */
#define IX_NPE_A_QMQ_HSS0_PKT_RX_FREE0  18

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_RX_FREE1
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Receive Free queue 1
 */
#define IX_NPE_A_QMQ_HSS0_PKT_RX_FREE1  19

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_RX_FREE2
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Receive Free queue 2
 */
#define IX_NPE_A_QMQ_HSS0_PKT_RX_FREE2  20

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_RX_FREE3
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Receive Free queue 3
 */
#define IX_NPE_A_QMQ_HSS0_PKT_RX_FREE3  21

/**
 * @def IX_NPE_A_QMQ_HSS0_PKT_TX_DONE
 *
 * @brief Hardware QMgr Queue ID for HSS Port 0 Packetized Transmit Done queue
 */
#define IX_NPE_A_QMQ_HSS0_PKT_TX_DONE   22


/*--------------------------------------------------------------------------
 * Hardware queue IDs relating to HSS Port 1
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_QMQ_HSS1_CHL_RX_TRIG
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Channelized Receive trigger
 */
#define IX_NPE_A_QMQ_HSS1_CHL_RX_TRIG   10

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_RX
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Receive
 */
#define IX_NPE_A_QMQ_HSS1_PKT_RX        0

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_TX0
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Transmit queue 0
 */
#define IX_NPE_A_QMQ_HSS1_PKT_TX0       5

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_TX1
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Transmit queue 1
 */
#define IX_NPE_A_QMQ_HSS1_PKT_TX1       6

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_TX2
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Transmit queue 2
 */
#define IX_NPE_A_QMQ_HSS1_PKT_TX2       7

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_TX3
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Transmit queue 3
 */
#define IX_NPE_A_QMQ_HSS1_PKT_TX3       8

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_RX_FREE0
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Receive Free queue 0
 */
#define IX_NPE_A_QMQ_HSS1_PKT_RX_FREE0  1

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_RX_FREE1
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Receive Free queue 1
 */
#define IX_NPE_A_QMQ_HSS1_PKT_RX_FREE1  2

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_RX_FREE2
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Receive Free queue 2
 */
#define IX_NPE_A_QMQ_HSS1_PKT_RX_FREE2  3

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_RX_FREE3
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Receive Free queue 3
 */
#define IX_NPE_A_QMQ_HSS1_PKT_RX_FREE3  4

/**
 * @def IX_NPE_A_QMQ_HSS1_PKT_TX_DONE
 *
 * @brief Hardware QMgr Queue ID for HSS Port 1 Packetized Transmit Done queue
 */
#define IX_NPE_A_QMQ_HSS1_PKT_TX_DONE   9

/*--------------------------------------------------------------------------
 * Hardware queue IDs relating to Atm Part-II
 *--------------------------------------------------------------------------*/

/**
 * @def IX_NPE_A_QMQ_ATM_FREE_VC0
 *
 * @brief Hardware QMgr Queue ID for ATM Free VC Queue.
 *
 * There are 32 Hardware QMgr Queue ID; from IX_NPE_A_QMQ_ATM_FREE_VC1 to
 * IX_NPE_A_QMQ_ATM_FREE_VC30
 */
#define IX_NPE_A_QMQ_ATM_FREE_VC0      32
#define IX_NPE_A_QMQ_ATM_FREE_VC1      IX_NPE_A_QMQ_ATM_FREE_VC0+1
#define IX_NPE_A_QMQ_ATM_FREE_VC2      IX_NPE_A_QMQ_ATM_FREE_VC1+1
#define IX_NPE_A_QMQ_ATM_FREE_VC3      IX_NPE_A_QMQ_ATM_FREE_VC2+1
#define IX_NPE_A_QMQ_ATM_FREE_VC4      IX_NPE_A_QMQ_ATM_FREE_VC3+1
#define IX_NPE_A_QMQ_ATM_FREE_VC5      IX_NPE_A_QMQ_ATM_FREE_VC4+1
#define IX_NPE_A_QMQ_ATM_FREE_VC6      IX_NPE_A_QMQ_ATM_FREE_VC5+1
#define IX_NPE_A_QMQ_ATM_FREE_VC7      IX_NPE_A_QMQ_ATM_FREE_VC6+1
#define IX_NPE_A_QMQ_ATM_FREE_VC8      IX_NPE_A_QMQ_ATM_FREE_VC7+1
#define IX_NPE_A_QMQ_ATM_FREE_VC9      IX_NPE_A_QMQ_ATM_FREE_VC8+1
#define IX_NPE_A_QMQ_ATM_FREE_VC10     IX_NPE_A_QMQ_ATM_FREE_VC9+1
#define IX_NPE_A_QMQ_ATM_FREE_VC11     IX_NPE_A_QMQ_ATM_FREE_VC10+1
#define IX_NPE_A_QMQ_ATM_FREE_VC12     IX_NPE_A_QMQ_ATM_FREE_VC11+1
#define IX_NPE_A_QMQ_ATM_FREE_VC13     IX_NPE_A_QMQ_ATM_FREE_VC12+1
#define IX_NPE_A_QMQ_ATM_FREE_VC14     IX_NPE_A_QMQ_ATM_FREE_VC13+1
#define IX_NPE_A_QMQ_ATM_FREE_VC15     IX_NPE_A_QMQ_ATM_FREE_VC14+1
#define IX_NPE_A_QMQ_ATM_FREE_VC16     IX_NPE_A_QMQ_ATM_FREE_VC15+1
#define IX_NPE_A_QMQ_ATM_FREE_VC17     IX_NPE_A_QMQ_ATM_FREE_VC16+1
#define IX_NPE_A_QMQ_ATM_FREE_VC18     IX_NPE_A_QMQ_ATM_FREE_VC17+1
#define IX_NPE_A_QMQ_ATM_FREE_VC19     IX_NPE_A_QMQ_ATM_FREE_VC18+1
#define IX_NPE_A_QMQ_ATM_FREE_VC20     IX_NPE_A_QMQ_ATM_FREE_VC19+1
#define IX_NPE_A_QMQ_ATM_FREE_VC21     IX_NPE_A_QMQ_ATM_FREE_VC20+1
#define IX_NPE_A_QMQ_ATM_FREE_VC22     IX_NPE_A_QMQ_ATM_FREE_VC21+1
#define IX_NPE_A_QMQ_ATM_FREE_VC23     IX_NPE_A_QMQ_ATM_FREE_VC22+1
#define IX_NPE_A_QMQ_ATM_FREE_VC24     IX_NPE_A_QMQ_ATM_FREE_VC23+1
#define IX_NPE_A_QMQ_ATM_FREE_VC25     IX_NPE_A_QMQ_ATM_FREE_VC24+1
#define IX_NPE_A_QMQ_ATM_FREE_VC26     IX_NPE_A_QMQ_ATM_FREE_VC25+1
#define IX_NPE_A_QMQ_ATM_FREE_VC27     IX_NPE_A_QMQ_ATM_FREE_VC26+1
#define IX_NPE_A_QMQ_ATM_FREE_VC28     IX_NPE_A_QMQ_ATM_FREE_VC27+1
#define IX_NPE_A_QMQ_ATM_FREE_VC29     IX_NPE_A_QMQ_ATM_FREE_VC28+1
#define IX_NPE_A_QMQ_ATM_FREE_VC30     IX_NPE_A_QMQ_ATM_FREE_VC29+1
#define IX_NPE_A_QMQ_ATM_FREE_VC31     IX_NPE_A_QMQ_ATM_FREE_VC30+1

/**
 * @def IX_NPE_A_QMQ_ATM_RXFREE_MIN
 *
 * @brief The minimum queue ID for FreeVC queue
 */
#define IX_NPE_A_QMQ_ATM_RXFREE_MIN  IX_NPE_A_QMQ_ATM_FREE_VC0

/**
 * @def IX_NPE_A_QMQ_ATM_RXFREE_MAX
 *
 * @brief The maximum queue ID for FreeVC queue
 */
#define IX_NPE_A_QMQ_ATM_RXFREE_MAX  IX_NPE_A_QMQ_ATM_FREE_VC31

/**
 * @def IX_NPE_A_QMQ_OAM_FREE_VC
 * @brief OAM Rx Free queue ID
 */
#ifdef IX_NPE_MPHYMULTIPORT
#define IX_NPE_A_QMQ_OAM_FREE_VC       14
#else
#define IX_NPE_A_QMQ_OAM_FREE_VC       3
#endif

/**
 * @def IX_NPE_A_CHAIN_DESC_COUNT_MAX
 *
 * @brief Maximum number of chained MBufs that can be chained together
 */
#define IX_NPE_A_CHAIN_DESC_COUNT_MAX            256



/*
 *  Definition of the ATM cell header
 *
 * This would most conviently be defined as the bit field shown below.
 * Endian portability prevents this, therefore a set of macros
 * are defined to access the fields within the cell header assumed to
 * be passed as a UINT32.
 *
 * Changes to field sizes or orders must be reflected in the offset
 * definitions above.
 *
 *    typedef struct
 *    {
 *       unsigned int gfc:4;
 *       unsigned int vpi:8;
 *       unsigned int vci:16;
 *       unsigned int pti:3;
 *       unsigned int clp:1;
 *    } IxNpeA_AtmCellHeader;
 *
 */

/** Mask to acess GFC */
#define GFC_MASK        0xf0000000

/** return GFC from ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_GFC_GET( header ) \
(((header) & GFC_MASK) >> 28)

/** set GFC into ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_GFC_SET( header,gfc ) \
do { \
    (header) &= ~GFC_MASK; \
    (header) |= (((gfc) << 28) & GFC_MASK); \
} while(0)

/** Mask to acess VPI */
#define VPI_MASK        0x0ff00000

/** return VPI from ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_VPI_GET( header ) \
(((header) & VPI_MASK) >> 20)

/** set VPI into ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_VPI_SET( header, vpi ) \
do { \
    (header) &= ~VPI_MASK; \
    (header) |= (((vpi) << 20) & VPI_MASK); \
} while(0)

/** Mask to acess VCI */
#define VCI_MASK        0x000ffff0

/** return VCI from ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_VCI_GET( header ) \
(((header) & VCI_MASK) >> 4)

/** set VCI into ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_VCI_SET( header, vci ) \
do { \
    (header) &= ~VCI_MASK; \
    (header) |= (((vci) << 4) & VCI_MASK); \
} while(0)

/** Mask to acess PTI */
#define PTI_MASK        0x0000000e

/** return PTI from ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_PTI_GET( header ) \
(((header) & PTI_MASK) >> 1)

/** set PTI into ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_PTI_SET( header, pti ) \
do { \
    (header) &= ~PTI_MASK; \
    (header) |= (((pti) << 1) & PTI_MASK); \
} while(0)

/** Mask to acess CLP */
#define CLP_MASK        0x00000001

/** return CLP from ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_CLP_GET( header ) \
((header) & CLP_MASK)

/** set CLP into ATM cell header */
#define IX_NPE_A_ATMCELLHEADER_CLP_SET( header, clp ) \
do { \
    (header) &= ~CLP_MASK; \
    (header) |= ((clp) & CLP_MASK); \
} while(0)


/*
* Definition of the Rx bitfield
*
* This would most conviently be defined as the bit field shown below.
* Endian portability prevents this, therefore a set of macros
* are defined to access the fields within the rxBitfield assumed to
* be passed as a UINT32.
*
* Changes to field sizes or orders must be reflected in the offset
* definitions above.
*
* Rx bitfield
*    struct
*    {   IX_NPEA_RXBITFIELD(
*        unsigned int status:1,
*        unsigned int port:7,
*        unsigned int vcId:8,
*        unsigned int currMbufSize:16);
*    } rxBitField;
*
*/

/** Mask to acess the rxBitField status */
#define STATUS_MASK     0x80000000

/** return the rxBitField status */
#define IX_NPE_A_RXBITFIELD_STATUS_GET( rxbitfield ) \
(((rxbitfield) & STATUS_MASK) >> 31)

/** set the rxBitField status */
#define IX_NPE_A_RXBITFIELD_STATUS_SET( rxbitfield, status ) \
do { \
    (rxbitfield) &= ~STATUS_MASK; \
    (rxbitfield) |= (((status) << 31) & STATUS_MASK); \
} while(0)

/** Mask to acess the rxBitField port */
#define PORT_MASK       0x7f000000

/** return the rxBitField port */
#define IX_NPE_A_RXBITFIELD_PORT_GET( rxbitfield ) \
(((rxbitfield) & PORT_MASK) >> 24)

/** set the rxBitField port */
#define IX_NPE_A_RXBITFIELD_PORT_SET( rxbitfield, port ) \
do { \
    (rxbitfield) &= ~PORT_MASK; \
    (rxbitfield) |= (((port) << 24) & PORT_MASK); \
} while(0)

/** Mask to acess the rxBitField vcId */
#define VCID_MASK       0x00ff0000

/** return the rxBitField vcId */
#define IX_NPE_A_RXBITFIELD_VCID_GET( rxbitfield ) \
(((rxbitfield) & VCID_MASK) >> 16)

/** set the rxBitField vcId */
#define IX_NPE_A_RXBITFIELD_VCID_SET( rxbitfield, vcid ) \
do { \
    (rxbitfield) &= ~VCID_MASK; \
    (rxbitfield) |= (((vcid) << 16) & VCID_MASK); \
} while(0)

/** Mask to acess the rxBitField mbuf size */
#define CURRMBUFSIZE_MASK       0x0000ffff

/** return the rxBitField mbuf size */
#define IX_NPE_A_RXBITFIELD_CURRMBUFSIZE_GET( rxbitfield ) \
((rxbitfield) & CURRMBUFSIZE_MASK)

/** set the rxBitField mbuf size */
#define IX_NPE_A_RXBITFIELD_CURRMBUFSIZE_SET( rxbitfield, currmbufsize ) \
do { \
    (rxbitfield) &= ~CURRMBUFSIZE_MASK; \
    (rxbitfield) |= ((currmbufsize) & CURRMBUFSIZE_MASK); \
} while(0)



/**
 * @brief Tx Descriptor definition
 */
typedef struct
{
    UINT8 port;				/**< Tx Port number */
    UINT8 aalType; 			/**< AAL Type */
    UINT16 currMbufLen;			/**< mbuf length */
    UINT32 atmCellHeader;		/**< ATM cell header */
    IX_MBUF *pCurrMbuf;			/**< pointer to mbuf */
    unsigned char *pCurrMbufData;	/**< Pointer to mbuf->dat */
    IX_MBUF *pNextMbuf;			/**< Pointer to next mbuf */
    UINT32  totalLen;			/**< Total Length */
    UINT32  aal5CrcResidue;		/**< AAL5 CRC Residue */
} IxNpeA_TxAtmVc;

/* Changes to field sizes or orders must be reflected in the offset
 * definitions above. */




/**
 * @brief Rx Descriptor definition
 */
typedef struct
{
    UINT32  rxBitField;			/**< Recieved bit field */
    UINT32  atmCellHeader;		/**< ATM Cell Header */
    UINT32  rsvdWord0;                  /**< Reserved field */
    UINT16  currMbufLen;		/**< Mbuf Length */
    UINT8   timeLimit; 			/**< Payload Reassembly timeLimit (used for aal0_xx only) */
    UINT8   rsvdByte0;                  /**< Reserved field */ 
    UINT32  rsvdWord1;   		/**< Reserved field */
    IX_MBUF *pCurrMbuf;			/**< Pointer to current mbuf */
    unsigned char *pCurrMbufData;	/**< Pointer to current mbuf->data */
    IX_MBUF *pNextMbuf;			/**< Pointer to next mbuf */
    UINT32  totalLen;			/**< Total Length */
    UINT32  aal5CrcResidue;		/**< AAL5 CRC Residue */
} IxNpeA_RxAtmVc;


/**
 * @brief NPE-A AAL Type
 */
typedef enum
{
    IX_NPE_A_AAL_TYPE_INVALID = 0,	/**< Invalid AAL type */
    IX_NPE_A_AAL_TYPE_0_48    = 0x1,	/**< AAL0 - 48 byte */
    IX_NPE_A_AAL_TYPE_0_52    = 0x2,	/**< AAL0 - 52 byte */
    IX_NPE_A_AAL_TYPE_5       = 0x5,	/**< AAL5 */
    IX_NPE_A_AAL_TYPE_OAM     = 0xF	/**< OAM */
} IxNpeA_AalType;

/**
 * @brief NPE-A Payload format 52-bytes & 48-bytes
 */
typedef enum
{
    IX_NPE_A_52_BYTE_PAYLOAD = 0,	/**< 52 byte payload */
    IX_NPE_A_48_BYTE_PAYLOAD		/**< 48 byte payload */
} IxNpeA_PayloadFormat;

/**
 * @brief  HSS Packetized NpePacket Descriptor Structure
 */
typedef struct
{
    UINT8   status;		/**< Status of the packet passed to the client */
    UINT8   errorCount;		/**< Number of errors */
    UINT8   chainCount;		/**< Mbuf chain count e.g. 0 - No mbuf chain */
    UINT8   rsvdByte0;		/**< Reserved byte to make the descriptor word align */

    UINT16  packetLength;	/**< Packet Length */
    UINT16  rsvdShort0;		/**< Reserved short to make the descriptor a word align */

    IX_MBUF *pRootMbuf;		/**< Pointer to Root mbuf */
    IX_MBUF *pNextMbuf;		/**< Pointer to next mbuf */
    UINT8   *pMbufData;		/**< Pointer to the current mbuf->data */
    UINT32  mbufLength;		/**< Current mbuf length */

} IxNpeA_NpePacketDescriptor;


#endif
/**
 *@}
 */
