/**
 * @file IxFeatureCtrl.h
 *
 * @date 30-Jan-2003

 * @brief This file contains the public API of the IXP425 Feature Control
 *        component.
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/
/* ------------------------------------------------------
   Doxygen group definitions
   ------------------------------------------------------ */
/**
 * @defgroup IxFeatureCtrlAPI IXP425 Feature Control (IxFeatureCtrl) API
 *
 * @brief The Public API for the IXP425 Feature Control.
 * 
 * @{
 */

#ifndef IXFEATURECTRL_H
#define IXFEATURECTRL_H

/*
 * User defined include files
 */
#include "IxTypes.h"
#include "ixp425.h"

/*
 * #defines and macros
 */

/*************************************************************
 * The following are IxFeatureCtrlComponentCheck return values.
 ************************************************************/

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def  IX_FEATURE_CTRL_COMPONENT_DISABLED
 *
 * @brief Hardware Component is disabled/unavailable.
 *        Return status by ixFeatureCtrlComponentCheck()  
 */
#define  IX_FEATURE_CTRL_COMPONENT_DISABLED 0

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def  IX_FEATURE_CTRL_COMPONENT_ENABLED
 *
 * @brief Hardware Component is available. 
 *        Return status by ixFeatureCtrlComponentCheck()  
 */
#define  IX_FEATURE_CTRL_COMPONENT_ENABLED  1

/************************************************************************
 * Product ID in XScale CP15 - Register 0
 *  - It contains information on the maximum XScale Core Frequency and
 *    Silicon Stepping.  
 *  - XScale Core Frequency Id indicates only the maximum XScale frequency
 *    achievable and not the running XScale frequency (maybe stepped down).    
 *  - The register is read by using ixFeatureCtrlProductIdRead.
 *  - Usage example: 
 *          productId = ixFeatureCtrlProductIdRead();
 *          if( (productId & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
 *              IX_FEATURE_CTRL_SILICON_TYPE_A0 )
 *          if( (productId & IX_FEATURE_CTRL_XSCALE_FREQ_MASK) == 
 *              IX_FEATURE_CTRL_XSCALE_FREQ_533 )    
 * 
 *  31 28 27 24 23 20 19 16 15 12 11                  4 3              0  
 *  -------------------------------------------------------------------- 
 * | 0x6 | 0x9 | 0x0 | 0x5 | 0x4 | XScale Core Freq Id | Si Stepping Id |    
 *  --------------------------------------------------------------------
 *
 *   Maximum Achievable XScale Core Frequency Id :  533MHz  - 0x1C
 *                                                  400MHz  - 0x1D 
 *                                                  266MHz  - 0x1F
 * 
 *  Si Stepping Id            :  A       - 0x0    
 *                               B       - 0x1 
 ************************************************************************/

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_SILICON_TYPE_A0
 *
 * @brief This is the value of A0 Silicon in product ID. 
 */
#define IX_FEATURE_CTRL_SILICON_TYPE_A0   0

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_SILICON_TYPE_B0
 *
 * @brief This is the value of B0 Silicon in product ID.
 */
#define IX_FEATURE_CTRL_SILICON_TYPE_B0   1

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_SILICON_STEPPING_MASK
 *
 * @brief This is the mask of silicon stepping in product ID. 
 */
#define IX_FEATURE_CTRL_SILICON_STEPPING_MASK  0xF 

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_XSCALE_FREQ_533
 *
 * @brief This is the value of 533MHz XScale Core in product ID.
 */
#define IX_FEATURE_CTRL_XSCALE_FREQ_533  ((0x1C)<<4)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_XSCALE_FREQ_400
 *
 * @brief This is the value of 400MHz XScale Core in product ID.
 */
#define IX_FEATURE_CTRL_XSCALE_FREQ_400  ((0x1D)<<4)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_XSCALE_FREQ_266
 *
 * @brief This is the value of 266MHz XScale Core in product ID.
 */
#define IX_FEATURE_CTRL_XSCALE_FREQ_266 ((0x1F)<<4)   

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURE_CTRL_XSCALE_FREQ_MASK
 *
 * @brief This is the mask of XScale Core in product ID.
 */
#define IX_FEATURE_CTRL_XSCALE_FREQ_MASK ((0xFF)<<4)  


/************************************************************************
 * IXP425 Feature Control Register  
 * - It contains the information (available/unavailable) of IXP425
 *   hardware components in their corresponding bit location. 
 * - Bit value of 0 means the hardware component is available 
 *   or not software disabled. Hardware component that is available 
 *   can be software disabled.  
 * - Bit value of 1 means the hardware is unavailable or software  
 *   disabled.Hardware component that is unavailable cannot be software 
 *   enabled. 
 * - Use ixFeatureCtrlHwCapabilityRead() to read the hardware component's
 *   availability.  
 * - Use ixFeatureCtrlRead() to get the current IXP425 feature control  
 *   register value.    
 *
 *   Bit            Field Description (Hardware Component Availability) 
 *   ---            ---------------------------------------------------
 *    0             RComp Circuitry     
 *    1             USB Controller
 *    2             Hashing Coprocessor
 *    3             AES Coprocessor 
 *    4             DES Coprocessor 
 *    5             HDLC Coprocessor
 *    6             AAL Coprocessor
 *    7             HSS Coprocesspr
 *    8             Utopia Coprocessor
 *    9             Ethernet 0 Coprocessor
 *   10             Ethernet 1 Coprocessor
 *   11             NPE A 
 *   12             NPE B  
 *   13             NPE C
 *   14             PCI Controller
 *   15             Reserved
 *  16-17           Utopia PHY Limit Status : 0x0 - 32 PHY 
 *                                            0x1 - 16 PHY
 *                                            0x2 -  8 PHY
 *                                            0x3 -  4 PHY  
 *  18-31           Reserved    
 ************************************************************************/
/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_RCOMP
 *
 * @brief The bit location for RComp Circuitry
 */
#define IX_FEATURECTRL_REG_LOC_RCOMP 0

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_USB
 *
 * @brief The bit location for USB Controller
 */
#define IX_FEATURECTRL_REG_LOC_USB 1

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_HASH
 *
 * @brief The bit location for Hashing Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_HASH 2

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_AES
 *
 * @brief The bit location for AES Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_AES 3

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_DES
 *
 * @brief The bit location for DES Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_DES 4

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_HDLC
 *
 * @brief The bit location for HDLC Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_HDLC 5

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_AAL
 *
 * @brief The bit location for AAL Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_AAL 6

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_HSS
 *
 * @brief The bit location for HSS Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_HSS 7

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_UTOPIA
 *
 * @brief The bit location for UTOPIA Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_UTOPIA 8

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_ETH0
 *
 * @brief The bit location for Ethernet 0 Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_ETH0 9

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_ETH1
 *
 * @brief The bit location for Ethernet 1 Coprocessor
 */
#define IX_FEATURECTRL_REG_LOC_ETH1 10

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_NPEA
 *
 * @brief The bit location for NPE A 
 */
#define IX_FEATURECTRL_REG_LOC_NPEA 11

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_NPEB
 *
 * @brief The bit location for NPE B
 */
#define IX_FEATURECTRL_REG_LOC_NPEB 12

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_NPEC
 *
 * @brief The bit location for NPE C
 */
#define IX_FEATURECTRL_REG_LOC_NPEC 13

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_PCI
 *
 * @brief The bit location for PCI Controller
 */
#define IX_FEATURECTRL_REG_LOC_PCI 14

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_LOC_UTOPIA_PHY_LIMIT
 *
 * @brief The bit location for Utopia PHY Limit Status
 */
#define IX_FEATURECTRL_REG_LOC_UTOPIA_PHY_LIMIT 16

/**
 * Component Type Defines. 
 * - These defines are used as input argument to ixFeatureCtrlComponentCheck().
 */ 

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_RCOMP
 *
 * @brief The Component Name for RCOMP Circuitry. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_RCOMP    (1<<IX_FEATURECTRL_REG_LOC_RCOMP)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_USB
 *
 * @brief The Component Name for USB Controller. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_USB    (1<<IX_FEATURECTRL_REG_LOC_USB)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_HASH
 *
 * @brief The Component Name for Hashing Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_HASH    (1<<IX_FEATURECTRL_REG_LOC_HASH)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_AES
 *
 * @brief The Component Name for AES Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_AES    (1<<IX_FEATURECTRL_REG_LOC_AES)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_DES
 *
 * @brief The Component Name for DES Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_DES    (1<<IX_FEATURECTRL_REG_LOC_DES)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_HDLC
 *
 * @brief The Component Name for HDLC Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_HDLC    (1<<IX_FEATURECTRL_REG_LOC_HDLC)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_AAL
 *
 * @brief The Component Name for AAL Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_AAL    (1<<IX_FEATURECTRL_REG_LOC_AAL)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_HSS
 *
 * @brief The Component Name for HSS Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_HSS    (1<<IX_FEATURECTRL_REG_LOC_HSS)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_UTOPIA
 *
 * @brief The Component Name for Utopia Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_UTOPIA    (1<<IX_FEATURECTRL_REG_LOC_UTOPIA)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_ETH0
 *
 * @brief The Component Name for Ethernet 0 Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_ETH0    (1<<IX_FEATURECTRL_REG_LOC_ETH0)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_ETH1
 *
 * @brief The Component Name for Ethernet 1 Coprocessor. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_ETH1    (1<<IX_FEATURECTRL_REG_LOC_ETH1)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_NPEA
 *
 * @brief The Component Name for NPE A. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_NPEA    (1<<IX_FEATURECTRL_REG_LOC_NPEA)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_NPEB
 *
 * @brief The Component Name for NPE B. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_NPEB    (1<<IX_FEATURECTRL_REG_LOC_NPEB)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_NPEC
 *
 * @brief The Component Name for NPE C. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_NPEC   (1<<IX_FEATURECTRL_REG_LOC_NPEC)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_PCI
 *
 * @brief The Component Name for PCI Controller. The name will be used for 
 * ixFeatureCtrlComponentCheck(). 
 */
#define IX_FEATURECTRL_PCI   (1<<IX_FEATURECTRL_REG_LOC_PCI)

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_UTOPIA_32PHY
 *
 * @brief Maximum  UTOPIA PHY available to IXP425 is 32.  
 * 
 */
#define IX_FEATURECTRL_REG_UTOPIA_32PHY  0x0

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_UTOPIA_16PHY
 *
 * @brief Maximum  UTOPIA PHY available to IXP425 is 16.  
 * 
 */
#define IX_FEATURECTRL_REG_UTOPIA_16PHY  0x1

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_UTOPIA_8PHY
 *
 * @brief Maximum  UTOPIA PHY available to IXP425 is 8.  
 * 
 */
#define IX_FEATURECTRL_REG_UTOPIA_8PHY   0x2

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @def IX_FEATURECTRL_REG_UTOPIA_4PHY
 *
 * @brief Maximum  UTOPIA PHY available to IXP425 is 4.  
 * 
 */
#define IX_FEATURECTRL_REG_UTOPIA_4PHY   0x3

/**
 * @defgroup IxFeatureCtrlSwConfig Software Configuration for Access Component
 *
 * @ingroup IxFeatureCtrlAPI 
 *
 * @brief This section describes software configuration in access component. The
 *        configuration can be changed at run-time. ixFeatureCtrlSwConfigurationCheck( )
 *        will be used across applicable access component to check the configuration.
 *        ixFeatureCtrlSwConfigurationWrite( ) is used to write the software configuration.
 *
 * @note <b>All software configurations are default to be enabled.</b> 
 *
 * @{
 */
/**
 * @ingroup IxFeatureCtrlSwConfig
 *
 * @def IX_FEATURE_CTRL_SWCONFIG_DISABLED
 *
 * @brief Software configuration is disabled. 
 * 
 */
#define IX_FEATURE_CTRL_SWCONFIG_DISABLED 0  

/**
 * @ingroup IxFeatureCtrlSwConfig
 *
 * @def IX_FEATURE_CTRL_SWCONFIG_ENABLED
 *
 * @brief Software configuration is enabled. 
 * 
 */
#define IX_FEATURE_CTRL_SWCONFIG_ENABLED 1  

/**
 * @ingroup IxFeatureCtrlSwConfig
 *
 * @enum IxFeatureCtrlSwConfig
 *
 * @brief Enumeration for software configuration in access components.
 *
 */
/*
 * Entry for new run-time software configuration should be added here.
 */
typedef enum
{
    IX_FEATURECTRL_ETH_LEARNING,       /**< EthDB Learning Feature */
    IX_FEATURECTRL_SWCONFIG_MAX        /**< Maximum boudary for IxFeatureCtrlSwConfig  */
} IxFeatureCtrlSwConfig;


/**
 * @} addtogroup IxFeatureCtrlSwConfig
 */

/*
 * Typedefs
 */

/**
 * @ingroup IxFeatureCtrlAPI 
 * 
 * @typedef IxFeatureCtrlReg
 *
 * @brief Feature Control Register that contains hardware components'
 *        availability information.
 */
typedef UINT32 IxFeatureCtrlReg;

/**
 * @ingroup IxFeatureCtrlAPI 
 * 
 * @typedef IxFeatureCtrlProductId
 *
 * @brief Product ID of Silicon that contains Silicon Stepping and 
 *        Maximum XScale Core Frequency information.  
 */
typedef UINT32 IxFeatureCtrlProductId;

/**
 * @ingroup IxFeatureCtrlAPI 
 * 
 * @typedef IxFeatureCtrlComponentType
 *
 * @brief The component type used for ixFeatureCtrlComponentCheck().
 *        
 */
typedef UINT32 IxFeatureCtrlComponentType;

/*
 * Prototypes for interface functions
 */

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn IxFeatureCtrlReg ixFeatureCtrlRead (void)
 * 
 * @brief This function reads out the CURRENT value of Feature Control Register.
 *        The current value may not be the same as that of the hardware component 
 *        availability.    
 * 
 * The bit location of each hardware component is defined above. 
 * A value of '1' in bit means the hardware component is not available.  A value of '0'   
 * means the hardware component is available.
 *
 * @return 
 *      - IxFeatureCtrlReg - the current value of IXP425 Feature Control Register
 */ 
IxFeatureCtrlReg
ixFeatureCtrlRead (void);

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn IxFeatureCtrlReg ixFeatureCtrlHwCapabilityRead (void)
 * 
 * @brief This function reads out the hardware capability of a silicon type as defined in 
 * feature control register.This value is different from that returned by 
 * ixFeatureCtrlRead() because this function returns the actual hardware component
 * availability.     
 *
 * The bit location of each hardware component is defined above. 
 * A value of '1' in bit means the hardware component is not available.  A value of '0'   
 * means the hardware component is available.
 *
 * @return 
 *      - IxFeatureCtrlReg - the hardware capability of IXP425. 
 *
 * @warning
 *      - This function must not be called when IXP425 is running as the result
 *        is undefined.    
 */ 
IxFeatureCtrlReg
ixFeatureCtrlHwCapabilityRead (void);

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn void ixFeatureCtrlWrite (IxFeatureCtrlReg expUnitReg)
 * 
 * @brief This function write the value stored in IxFeatureCtrlReg expUnitReg  
 *        to the Feature Control Register. 
 * 
 * The bit location of each hardware component is defined above.
 * The write is only effective on available hardware components. Writing '1' in a  
 * bit will software disable the respective hardware component. A '0' will mean that  
 * the hardware component will remain to be operable. 
 *
 * @param IxFeatureCtrlReg (in)expUnitReg - The value to be written to feature control 
 *                                          register.
 *
 * @return none
 *
 */ 
void
ixFeatureCtrlWrite (IxFeatureCtrlReg expUnitReg);

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn IX_STATUS ixFeatureCtrlComponentCheck (IxFeatureCtrlComponentType componentType)
 * 
 * @brief This function will check the availability of hardware component specified
 *        as componentType value. 
 *
 *        Usage Example:<br> 
 *         -  if(IX_FEATURE_CTRL_COMPONENT_DISABLED != 
 *              ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0)) <br>
 *         -  if(IX_FEATURE_CTRL_COMPONENT_ENABLED == 
 *              ixFeatureCtrlComponentCheck(IX_FEATURECTRL_PCI)) <br>
 *
 * This function is typically called during component initialization time. 
 *
 * @param IxFeatureCtrlComponentType (in)componentType - the type of a component as
 *        defined above as IX_FEATURECTRL_XXX (Exp: IX_FEATURECTRL_PCI, IX_FEATURECTRL_ETH0)           
 *        
 * @return 
 *      - IX_FEATURE_CTRL_COMPONENT_ENABLED if component is available 
 *      - IX_FEATURE_CTRL_COMPONENT_DISABLED if component is unavailable            
 */ 
IX_STATUS
ixFeatureCtrlComponentCheck (IxFeatureCtrlComponentType componentType);

/**
 * @ingroup IxFeatureCtrlAPI 
 * 
 * @fn IxFeatureCtrlProductId ixFeatureCtrlProductIdRead (void)
 * 
 * @brief This function will return IXP425 product ID i.e. CP15,
 *        Register 0.
 *                                                
 * @return 
 *      - IxFeatureCtrlProductId - the value of product ID.
 *
 */ 
IxFeatureCtrlProductId
ixFeatureCtrlProductIdRead (void) ;

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn IX_STATUS ixFeatureCtrlSwConfigurationCheck (IxFeatureCtrlSwConfig swConfigType)
 * 
 * @brief This function checks whether the specified software configuration is
 *        enabled or disabled. 
 *
 *        Usage Example:<br> 
 *        - if(IX_FEATURE_CTRL_SWCONFIG_DISABLED != 
 *            ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_ETH_LEARNING)) <br>
 *        - if(IX_FEATURE_CTRL_SWCONFIG_ENABLED == 
 *            ixFeatureCtrlSwConfigurationCheck(IX_FEATURECTRL_ETH_LEARNING)) <br>
 *
 * This function is typically called during access component initialization time. 
 *
 * @param IxFeatureCtrlSwConfig (in)swConfigType - the type of a software configuration
 *        defined in IxFeatureCtrlSwConfig enumeration.          
 *        
 * @return 
 *      - IX_FEATURE_CTRL_SWCONFIG_ENABLED if software configuration is enabled. 
 *      - IX_FEATURE_CTRL_SWCONFIG_DISABLED if software configuration is disabled.            
 */ 
IX_STATUS
ixFeatureCtrlSwConfigurationCheck (IxFeatureCtrlSwConfig swConfigType);

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn void ixFeatureCtrlSwConfigurationWrite (IxFeatureCtrlSwConfig swConfigType, BOOL enabled)
 * 
 * @brief This function enable/disable the specified software configuration.  
 *
 *        Usage Example:<br> 
 *        - ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_ETH_LEARNING, TRUE) is used
 *          to enable Ethernet Learning Feature <br>
 *        - ixFeatureCtrlSwConfigurationWrite(IX_FEATURECTRL_ETH_LEARNING, FALSE) is used
 *          to disable Ethernet Learning Feature <br> 
 *
 * @param IxFeatureCtrlSwConfig (in)swConfigType - the type of a software configuration
 *        defined in IxFeatureCtrlSwConfig enumeration. 
 * @param BOOL (in)enabled - To enable(TRUE) / disable (FALSE) the specified software
 *                           configuration.            
 *
 * @return none
 *          
 */ 
void
ixFeatureCtrlSwConfigurationWrite (IxFeatureCtrlSwConfig swConfigType, BOOL enabled);

/**
 * @ingroup IxFeatureCtrlAPI 
 *
 * @fn void ixFeatureCtrlIxp400SwVersionShow (void)
 * 
 * @brief This function shows the current software release information for IXP400 
 *          
 * @return none
 *          
 */ 
void
ixFeatureCtrlIxp400SwVersionShow (void);

#endif /* IXFEATURECTRL_H */

/**
 * @} defgroup IxFeatureCtrlAPI
 */
