/**
 * @file IxAdsl.h
 * 
 * @brief Header file for the ADSL Driver
 * 
 * @par
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 1999-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

/**
 * @defgroup IxAdsl IXP425 ADSL Driver API
 *
 * @brief The public API for the IXP425 ADSL Driver
 *
 * @{
 */

#ifndef IXADSL_H
#define IXADSL_H

#include "IxTypes.h"

/**
 * @ingroup IxAdsl
 *
 * @enum IxAdslStatus
 *
 * @brief These status will be used by the APIs to return to the client
 *
 */
 
typedef enum
{
    IX_ADSL_STATUS_SUCCESS = IX_SUCCESS, /**< Successful API execution. */
    IX_ADSL_STATUS_FAIL = IX_FAIL,       /**< Failed API execution. */
    IX_ADSL_STATUS_UNSUPPORTED_MODE,     /**< 
                                          * Unsupported mode type for 
                                          * IxAdslLineOpen function.
                                          */ 
    IX_ADSL_STATUS_ALREADY_DOWN          /**< Line is already down. */

} IxAdslStatus;

/**
 * @ingroup IxAdsl
 *
 * @enum IxAdslLineState
 *
 * @brief These status will be used to indicate the line state.
 *
 */

typedef enum
{
    IX_ADSL_LINE_STATE_UP_DUAL_LATENCY = 0,  /**<
                                              * The line is in
                                              * showtime state.
                                              * Fast & Interleaved
                                              * Channel
                                              */
    IX_ADSL_LINE_STATE_WAIT_FOR_ACTIVATING, /**<
                                              * The line is waiting for the
                                              * peer to activate.
                                              */
    IX_ADSL_LINE_STATE_ACTIVATING,          /**<
                                              * The line is negotiating
                                              * with its peer.
                                              */
    IX_ADSL_LINE_STATE_DOWN,                /**< The line is down. */
    IX_ADSL_LINE_STATE_UP_FASTCHANNEL,       /**<
                                              * The line is in
                                              * showtime state.
                                              * Fast Channel
                                              */
    IX_ADSL_LINE_STATE_UP_INTERLEAVECHANNEL, /**<
                                              * The line is in
                                              * showtime state.
                                              * Interleaved Channel
                                              */
    IX_ADSL_LINE_STATE_INVALID              /**<
                                              * ADSL line in an unknown state.
                                              */

} IxAdslLineState;

/**
 * @ingroup IxAdsl
 *
 * @enum IxAdslLineType
 *
 * @brief Used to indicate the type of ADSL line type used
 *
 */
typedef enum
{
    IX_ADSL_AUTOSELECT     =   0, /**<
                                * This is an auto-select mode for
                                * the CPE to auto-configure based on the
                                * CO/DSLAM line type; DMT, ANSI or G.lite.
                                */
    IX_ADSL_GLITE          =   1, /**< G.lite line type. */
    IX_ADSL_DMT            =   2, /**< DMT line type */
    IX_ADSL_ANSI           =   3, /**< ANSI line type */
    IX_ADSL_LOOPBACK       =   4, /**< Utopia Loopback line type */
    IX_ADSL_INVALID_MODE   =   5  /**<
                                * Used internally to indicate
                                * last valid enum
                                */
} IxAdslLineType;

/**
 * @ingroup IxAdsl
 *
 * @enum IxAdslPhyType
 *
 * @brief   Used to indicate the ADSL physical type - CPE
 *
 * @note    IxAdslPhyType is declared as an enum due to forward
 *          compatibility to support CO (fast and interleaved mode)
 *          in the future.
 *
 */
typedef enum
{
    IX_ADSL_PHY_CPE = 0,        /**< Adsl type is CPE. */
#if IX_ADSL_CO_ENABLE
    IX_ADSL_PHY_CO_FAST,        /**< Adsl type is CO Fast. */
    IX_ADSL_PHY_CO_INTERLEAVED, /**< Adsl type is CO Interleaved. */
#endif
    IX_ADSL_PHY_INVALID         /**< Adsl type is invalid. */
} IxAdslPhyType;

/**
 * @ingroup IxAdsl
 *
 * @brief   Callback function to indicate of the changes on the 
 *          line state.
 *
 */
typedef void (*IxAdslStateChangeCallback)(
    UINT32 lineNum,
    IxAdslLineState lineState);


/** 
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineOpen ( 
    UINT32 lineNum,
    IxAdslLineType lineType,
    IxAdslPhyType phyType)
 *
 * @brief Open the given ADSL line in the specified mode and type.
 *
 * Opens the given ADSL line in the specified mode and puts it in the 
 * 'Showtime' state, i.e. available to carry user data. 
 *
 * @note \n
 *
 * @li  The parameter lineNum exists for future Multi-PHY support. 
 *      Only lineNum = 0 is valid.
 *
 * @li  The parameter phyType exists for future CO support.
 *
 * Blocking :   This call may block for several seconds while the link is 
 *              established.
 *
 * Impacts On Global Data: Sets the lineEnable State .
 *
 * Pre-Conditions:  The code must only be called once the operating system 
 *                  is running, i.e. do not call as part of the hardware 
 *                  init as this code requires base services such as Atmd, 
 *                  Atmm, Atm scheduler and Utopia. The code should only
 *                  be called from task level.
 *
 * Post-Conditions: No cleanup after this call is required.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is 
 *          being used.
 * @param   lineType [in] indicates type of ADSL to be opened.
 * @param   phyType [in] is the type of Phy used - CPE.
 * 
 * @return @li  IX_ADSL_STATUS_SUCCESS - Line was opened and is in 'showtime' 
 *              state
 * @return @li  IX_ADSL_STATUS_FAILED - Line failed to open properly.
 * @return @li  IX_ADSL_STATUS_UNSUPPORTED_MODE - Illegal ADSL type for 
 *              IxAdslLineOpen function.
 *
 */
PUBLIC IxAdslStatus
ixAdslLineOpen (
    UINT32 lineNum,
    IxAdslLineType lineType,
    IxAdslPhyType phyType);


/** 
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineClose (UINT32 lineNum)
 *
 * @brief Closes a previously opened ADSL line. 
 *
 * The line will closed and put in the idle state.
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *  
 * Blocking : Non-blocking   
 *
 * Impacts On Global Data: Sets the lineEnable State . Notifies a callback routine.
 *
 * Pre-Conditions: Code should only be called from task level.
 *
 * Post-Conditions: No cleanup after this call is required.
 * 
 * @param   lineNum [in] is the parameter showing which ADSL line is
 *          being used and to be closed.
 *
 * @return @li  IX_ADSL_STATUS_SUCCESS - Line was closed successfully.
 * @return @li  IX_ADSL_STATUS_FAILED - Line failed to close properly.
 * @return @li  IX_ADSL_STATUS_ALREADY_DOWN - Line was not open before close.
 *
 */
PUBLIC IxAdslStatus
ixAdslLineClose (UINT32 lineNum);

/** 
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineStateChangeCallbackRegister(
    UINT32 lineNum,
    IxAdslStateChangeCallback lineChangeCallbackFn)
 *
 * @brief This is a notification registration procedure that gets called
 * if the line state of the given ADSL line changes.
 * The maximum callbacks that can be registered is defined as 
 * IX_ADSL_SIZEOF_CALLBACK_LIST (The default value is 10).
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * Blocking : This call is a non-blocking function.
 *
 * Impacts On Global Data: This sets a global callback handler.
 *
 * Pre-Conditions: There are no pre conditions to this call.
 *
 * Post-Conditions: A global line state handler for ADSL line state changes
 *                  shall be registered. It is advisable to register the
 *                  callback before the given ADSL line is opened.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is
 *          being used.
 * @param   lineChangeCallbackFn [in] is the callback function that will
 *          be invoked when there is a state change
 *
 * @return @li  IX_ADSL_STATUS_SUCCESS - The callback function was registered
 *              successfully.
 * @return @li  IX_ADSL_STATUS_FAILED - Internal error, registration of the
 *              callback function failed.
 *
 */
PUBLIC IxAdslStatus
ixAdslLineStateChangeCallbackRegister (
   UINT32 lineNum,
   IxAdslStateChangeCallback lineChangeCallbackFn);

/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineStateGet (UINT32 lineNum)
 *
 * @brief Returns the current state of the given ADSL line.
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * Blocking : This call is a non-blocking
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is
 *          being used.
 *
 * @return @li  IX_ADSL_LINE_STATE_UP_FAST - The line is in show time state.
 * @return @li  IX_ADSL_LINE_STATE_WAIT_FOR_ACTIVATING - The line is
 *              waiting fo the peer to activate.
 * @return @li  IX_ADSL_LINE_STATE_ACTIVATING - The line is negotiating
 *              with its peer.
 * @return @li	IX_ADSL_LINE_STATE_DOWN - The line is idle.
 * @return @li	IX_ADSL_LINE_STATE_INVALID - The line is in an unknown
 *              state.
 *
 */
PUBLIC IxAdslLineState
ixAdslLineStateGet (UINT32 lineNum);

/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineRateUpstreamGet (UINT32 lineNum)
 *
 * @brief Return the current upstream line speed of the given ADSL line.
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * The data returned by this API represents the transmit (Tx) rate of the
 * line from the ATU device.
 *
 * Blocking : This call is a non-blocking
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is
 *          being used.
 *
 * @return @li  Integer - bit rate in kbits/second.
 *              N.B. Returns zero if the line is not in 'Showtime' state.
 *
 */
PUBLIC UINT32
ixAdslLineRateUpstreamGet (UINT32 lineNum);

/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslLineRateDownstreamGet (UINT32 lineNum)
 *
 * @brief Return the current downstream line speed of the given ADSL line.
 *
 * The data returned by this API is the received (Rx) rate of the line
 * the ATU device.
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * Blocking : This call is a non-blocking
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param lineNum [in] is the parameter showing which ADSL line is being used.
 *
 * @return @li  Integer - bit rate in kbits/second.
 *              N.B. Returns zero if the line is not in 'Showtime' state.
 *
 */
PUBLIC UINT32
ixAdslLineRateDownstreamGet (UINT32 lineNum);

/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslDyingGaspEnable (UINT32 lineNum)
 * 
 * @brief Enables the function that informs ATU-C when condition that 
 * leads to shutdown of the given Adsl line has been detected. 
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * Blocking : This call is a non-blocking
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only. Only Available in ATU-R.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param lineNum [in] is the parameter showing which ADSL line is being used.
 * 
 * @return @li  IX_ADSL_STATUS_SUCCESS - Dying Gasp is enabled.
 *              successfully.
 * @return @li  IX_ADSL_STATUS_FAILED - Failed to enable Dying Gasp.
 *
 */
PUBLIC IxAdslStatus
ixAdslDyingGaspEnable (UINT32 lineNum);


/** 
 * @ingroup IxAdsl
 *
 * @fn ixAdslVendorCodeSet (
    UINT32 lineNum,
    UINT8 ixAdslItuVendoridCountrycode,
    UINT8 ixAdslItuVendoridVendorcode1,
    UINT8 ixAdslItuVendoridVendorcode2,
    UINT8 ixAdslItuVendoridVendorcode3,
    UINT8 ixAdslItuVendoridVendorcode4,
    UINT8 ixAdslItuVendoridVendorspecific1,
    UINT8 ixAdslItuVendoridVendorspecific2)
 *
 * @brief Set the vendor specific bytes in the given ADSL line. 
 *
 * The vendor ID must be set before the line is open, if not the default
 * vendor ID shall be sent to the peer modem upon request.
 *
 * @note \n
 * 
 * @li  Vendor specific values are taken by the phy and linked  
 *      together to form a single code
 *
 * @li  Note that the parameter lineNum exists for future Multi-PHY support.
 *      Only lineNum = 0 is valid.
 *
 * Blocking : This call is a non-blocking
 *
 * Impacts On Global Data: Sets an internal ADSL global data structure. 
 * This shall be used for all subsequent ADSL line open commands.
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is being 
            used.
 * @param   ixAdslItuVendoridCountrycode [in] is the vendor country code  
 *          that are predefined in standards. 
 * @param   ixAdslItuVendoridVendorcode1 [in] is the vendor code 1 that  
 *          are predefined in standards.
 * @param   ixAdslItuVendoridVendorcode2 [in] is the vendor code 2 that are 
 *          predefined in standards. 
 * @param   ixAdslItuVendoridVendorcode3 [in] is the vendor code 3 that are  
 *          predefined in standards. 
 * @param   ixAdslItuVendoridVendorcode4 [in] is the vendor code 4 that are 
 *          predefined in standards.  
 * @param   ixAdslItuVendoridVendorspecific1 [in] is the vendor specific 1 
 *          that are predefined in standards. 
 * @param   ixAdslItuVendoridVendorspecific2 [in] is the vendor specific 2 
 *          that are predefined in standards. 
 *
 * 
 * @return @li  IX_ADSL_STATUS_SUCCESS - Set Vendor Code successful.
 * @return @li  IX_ADSL_STATUS_FAILED - Set Vendor Code failed because 
 *              the line is up.
 *                   
 */
PUBLIC IxAdslStatus
ixAdslVendorCodeSet (
    UINT32 lineNum,
    UINT8 ixAdslItuVendoridCountrycode,
    UINT8 ixAdslItuVendoridVendorcode1,
    UINT8 ixAdslItuVendoridVendorcode2,
    UINT8 ixAdslItuVendoridVendorcode3,
    UINT8 ixAdslItuVendoridVendorcode4,
    UINT8 ixAdslItuVendoridVendorspecific1,
    UINT8 ixAdslItuVendoridVendorspecific2);


/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslShow (UINT32 lineNum)
 *
 * @brief This function will show the current statistics associated with
 * the given ADSL Line.
 *
 * @note    The parameter lineNum exists for future Multi-PHY support.
 *          Only lineNum = 0 is valid.
 *
 * The list of statistics to be shown by IxAdslShow:
 *
 * @li  Controller SW Version
 * @li  ADSL Line State
 * @li  Line Number
 * @li  Upstream and Downstream Rate
 * @li  Training Statistics for Upstream and Downstream Rates
 * @li  ADSL Near End Operational Data such as Upstream Relative
 *      Capacity Occupancy, Noise Margin Upstream, Output Pwr Downstream,
 *      Attenuation Upstream, Downstream Fast Bitrate, Downstream
 *      Interleaved Bitrate, Near-end defect bitmap, Loss of Frame (secs),
 *      Loss of Cell delineation (secs), Loss of Signal (secs), Loss of
 *      Margin (secs), Errored seconds, HEC and FEC Errors.
 * @li  ADSL Far End Operational Data such as Downstream Relative
 *      Capacity Occupancy, Noise Margin Downstream, Output Pwr Upstream,
 *      Attenuation Downstream, Upstream Fast Bitrate, Upstream
 *      Interleaved Bitrate, Far-end defect bitmap
 * @li  Tx and Rx ATM Cell Counters
 *
 * Blocking : This call is non-blocking.
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param   lineNum [in] is the parameter showing which ADSL line is
 *          being used.
 *
 * @return None
 *
 */
PUBLIC void
ixAdslShow (UINT32 lineNum);
/**
 * @ingroup IxAdsl
 *
 * @fn ixAdslMemoryUnmap (void)
 *
 * @brief This function will unmap the dynamically allocated addresses
 *
 *
 * Blocking : This call is non-blocking.
 *
 * Impacts On Global Data: None
 *
 * Pre-Conditions: Task level calls only.
 *
 * Post-Conditions: None.
 *
 * Exceptions: None.
 *
 * @param   None
 *
 * @return None
 *
 */
PUBLIC void
ixAdslMemoryUnmap(void);
#endif /* ndef IXADSL_H */

/**
 *@}
 */






