/*
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
*/
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#define WAIT_CNT    (5000000)
#define SECTOR_SIZE (128 * 1024)
#define RETRY_CNT   (100)

#define DEFAULT_FILE_SIZE (512 * 1024) /* 512 kb */

/* external data and prototypes */
extern unsigned char ixFlashBinArrayStart[], ixFlashBinArrayEnd[];
extern int ixFlashTftpGet(char *server, char *file, unsigned char *tftpDestBuffer);

/* Function prototypes */
int ixFlashProgram(unsigned char* src, char* dest, int len);
static void ixFlashProgressShow(int percentage);
static void ixFlashUsageShow(void);
int ixFlashUpgrade(char *option, char *server, char *file, int memorySize);


int ixFlashUpgrade(char *option, char *server, char *file, int memorySize)
{
  unsigned char *startAddr;
  int length;
  
  int download;
  int result;
  
  if (option == 0)
  {
    ixFlashUsageShow();
    
    return 1;
  }
  
  if (strcmp(option, "-b") == 0)
  {
    startAddr = ixFlashBinArrayStart;
    length    = (int) ixFlashBinArrayEnd - (int) ixFlashBinArrayStart;
    download  = 0;

    if (length == 0)
    {
      printf("\nInternal image is empty, please convert a valid bootrom.bin to assembly\n");
      printf("using binToAsm and copy the resulting bootrom.s over the default\n");
      printf("blank one in xscale_sw/src/flashUpgrade.\n");
      printf("\nAlternatively, simply use TFTP to load a remote image\n\n");
    
      return 1;
    }
  }
  else if (strcmp(option, "-l") == 0 && server != 0 && file != 0)
  {
    if (memorySize == 0)
    {
      memorySize = DEFAULT_FILE_SIZE;
    }
      
    startAddr = malloc(memorySize);

    if (startAddr == 0)
    {
      printf("Error: could not allocate memory for download (%d bytes)\n", memorySize);

      return 1;
    }

    length = ixFlashTftpGet(server, file, startAddr);

    if (length < 0)
    {
      printf("Error downloading file, aborted\n");

      free(startAddr);

      return 1;
    }
    else if (length == 0)
    {
      printf("Null size reply from server, aborted\n");

      free(startAddr);

      return 1;
    }

    download = 1;
  }
  else
  {
    ixFlashUsageShow();
    
    return 1;
  }
    
  printf("Programming flash using %s image at: start 0x%08x - dest 0x%08x - len 0x%08x\n", download ? "downloaded" : "built-in", (int) startAddr, 0x50000000, length);
  
  if ((result = ixFlashProgram(startAddr, (char *) 0x50000000, length)) != 0)
  {
    printf("Flash programming failed with error code %d\n", result);
    
    if (download) free(startAddr);
    
    return 1;
  };
  
  if (download) free(startAddr);
  
  return 0;
}

static void ixFlashUsageShow(void)
{
  printf("Usage: IxFlashUpgrade <option> [<server> <file> [<memory size>]]\n");
  printf("\n");
  printf("option  \"-b\" use built-in bootrom.s image (the other parameters are ignored)\n");
  printf("        \"-l\" load remote <file> from <server> using TFTP\n");
  printf("               <memory size> is optional in this case and defaults to %dkb\n", DEFAULT_FILE_SIZE / 1024);
  printf("               use only when the expected file size exceeds the default\n");
  printf("\n");
  printf("Please note that Tornado 2.1.1 bootrom.ful images need to be processed through\n");
  printf("the swapbytes utility and they can be loaded only via TFTP\n\n");
}

static void ixFlashProgressShow(int percentage)
{
  int i;
  const char fullEraseStr[] = { 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0x8, 0 };
  const char pctEraseStr[]  = { 0x8, 0x8, 0x8, 0x8, 0 };
  static int lastPercentage = -1;
  static int lastFullUpdatePercentage = -1;
  
  if (lastPercentage == -1 || (percentage - lastFullUpdatePercentage) >= 10)
  {
    if (lastPercentage != -1) printf("%s", fullEraseStr);

    printf("[");
    
    for (i = 0 ; i < percentage / 10 ; i++)  printf("=");
    for (; i < 10 ; i++) printf(".");

    printf("] %3d%%", percentage);
    
    lastFullUpdatePercentage = lastPercentage = percentage;
  }
  else if (percentage != lastPercentage)
  {
    printf("%s", pctEraseStr);
    printf("%3d%%", percentage);
    
    lastPercentage = percentage;
  }
  
  if (percentage == 100) lastPercentage = lastFullUpdatePercentage = -1;
}

int ixFlashProgram(unsigned char *src, char *dest, int len)
{
  volatile short *address = (short *) dest; 
  volatile short *verPtr  = (short *) dest;
  short *originalSrc      = (short *) src;

  int i, retry, wait, x, z;
  
  int currSectorNum = 0;
  int sectorSize    = SECTOR_SIZE;
  int numSectors    = (len % SECTOR_SIZE) == 0 ? len / SECTOR_SIZE : (len / SECTOR_SIZE) + 1;
  
  unsigned volatile long *ptr;
  
  printf("Checking source...");
  
  address = (short *) src;
  
  /* copy all the source into a 2-byte sink - this might data abort if the addresses aren't correct */
  for (i = 0 ; i < len / 2 ; i++)
  {
    short tmpData = *address;
    short tmpDest, *tmpDestPtr = &tmpDest;
    
    *tmpDestPtr = tmpData;
    
    address++;
  }
  
  printf("done\n");

  printf("Upgrading bootrom, please wait\n");
  
  ptr = (unsigned long *) 0xc4000000;
  *ptr = 0xbfff3c02;
  
  address  = (short *) dest;
  *address = 0xff; /*Set flash to read state*/

  printf("Erasing flash\n");
  
  ixFlashProgressShow(0);
  
  do
  {
    /*unlock sector*/
    *address = 0x60; /*unlock*/
    *address = 0xD0; /*confirm*/
    
    wait = WAIT_CNT;
    
    while ((((*address) & 0x80) != 0x80) && wait > 0)
    {
    	wait--;
    }
    
    if (wait == 0)
    {
     	printf("\nError waiting for sector unlock\n");
      
    	return -1;
    }
    
    wait = 50000;
    while (wait--);
    
    /*erase Sector*/
    *address = 0x20; /*erase*/
    *address = 0xD0; /*confirm*/
    
    wait = WAIT_CNT;
    
    while ((((*address) & 0x80) != 0x80) && wait > 0)
    {
    	wait--;
    }
    
    if(wait == 0)
    {
   	  printf("\nError waiting for sector erase\n");

	    return -1;
    } 

    address += sectorSize / 2; /* inc word pointer */
    currSectorNum++;
    
    ixFlashProgressShow(currSectorNum * 100 / numSectors);
    
  }  while (currSectorNum < numSectors);

  printf("\n");
  
  printf("Done, checking flash\n");

  address = (short *) dest;
  
  for (i = 0 ; i < len / 2 ; i++)
  {  
     *address = 0xff;
     ++address;
  }

  ixFlashProgressShow(0);

  for(i = 0; i < len / 2 ; i++)
  {  
    unsigned short val = (unsigned short) *verPtr;
    
    ixFlashProgressShow(i * 200 / len);

    if (val != (unsigned short) 0xffff)
    {
      return (-3);
    }
   
    ++verPtr;
  }
  
  ixFlashProgressShow(100);
  
  printf("\n");

  printf("Flash erased successfully, loading image\n");
  
  ixFlashProgressShow(0);

  address = (short *) dest;
  
  for (i = 0 ; i < len / 2 ; i++)
  {
    for(x = 0 ; x <= RETRY_CNT ; ++x)
    {
      *address = 0xff;            /* put the flash in read mode */
      *address = 0x40;            /* flash command write */
      *address = *(short *) src;  /* word to be written */
      
      {
        for (z = 0 ; z < 1024 ; ++z);
      }
      
      *address = 0x70; /* Put in status read mode */

      retry = WAIT_CNT;
      
      while ((((*address) & 0x80) != 0x80) && retry--);        

      if (retry < 0)
      {  
       	 printf("\nTimeout waiting for programming\n"); 

         return -1; 
      } 
      
      *address = 0xff; /* Put in read mode */
      
      if (*address == *((short *) src))
      {
        address++;  /* Inc word pointer */
        src++;
        src++;      /* Src is a byte pointer to inc twice */
        
        break;
      }
      
      if (x % 100 == 0)
      {
        int y;

        for (y = 0 ; y < 5000 ; ++y)
        {
	        *address = 0xff;
        }

      }

      if (x == RETRY_CNT - 1)
      {
     	  return -2; 
      }
    }
    
    ixFlashProgressShow((i * 200) / len);
  }
  
  ixFlashProgressShow(100);
  
  printf("\n");

  printf("Wrote [%d] bytes, verifying...\n", len);
  
  ixFlashProgressShow(0);

  address = (short *) dest;

  for (i = 0 ; i < len / 2 ; i++)
  {
    *address = 0xff;
    
    if (*originalSrc != *address)
    {
      printf("\nError, data compare incorrect at word# %d\n", i);
          
      return i;
    }
    
    address++;
    originalSrc++;
    
    ixFlashProgressShow(i * 200 / len);
  }
  
  ixFlashProgressShow(100);
  
  printf("\n");
  
  printf("Burn completed - reboot to load new image\n"); 
  
  return 0;
}
