/**
 * @file IxEthAccMii.c
 *
 * @author Intel Corporation
 * @date 
 *
 * @brief  MII control functions
 *
 * Design Notes:
 *
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include <stdio.h>

#include "IxOsServices.h"
#include "IxOsServicesMemMap.h"

#include "IxEthAcc.h"
#include "IxEthAcc_p.h"
#include "IxTypes.h"
#include "IxEthAccMac_p.h"
#include "IxEthAccMii_p.h"


PRIVATE UINT32 miiBaseAddressVirt;
PRIVATE IxMutex miiAccessLock;

/* -----------------------------------
 * private function prototypes
 */
PRIVATE void
ixEthAccMdioCmdWrite(UINT32 mdioCommand);

PRIVATE void
ixEthAccMdioCmdRead(UINT32 *data);

PRIVATE void
ixEthAccMdioStatusRead(UINT32 *data);


PRIVATE void
ixEthAccMdioCmdWrite(UINT32 mdioCommand)
{
    REG_WRITE(miiBaseAddressVirt,
	      IX_ETH_ACC_MAC_MDIO_CMD_1,
	      mdioCommand & 0xff);

    REG_WRITE(miiBaseAddressVirt,
	      IX_ETH_ACC_MAC_MDIO_CMD_2,
	      (mdioCommand >> 8) & 0xff);

    REG_WRITE(miiBaseAddressVirt,
	      IX_ETH_ACC_MAC_MDIO_CMD_3,
	      (mdioCommand >> 16) & 0xff);

    REG_WRITE(miiBaseAddressVirt,
	      IX_ETH_ACC_MAC_MDIO_CMD_4,
	      (mdioCommand >> 24) & 0xff);
}

PRIVATE void
ixEthAccMdioCmdRead(UINT32 *data)
{
    UINT32 regval;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_CMD_1,
	     regval);

    *data = regval & 0xff;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_CMD_2,
	     regval);

    *data |= (regval & 0xff) << 8;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_CMD_3,
	     regval);

    *data |= (regval & 0xff) << 16;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_CMD_4,
	     regval);

    *data |= (regval & 0xff) << 24;

}

PRIVATE void
ixEthAccMdioStatusRead(UINT32 *data)
{
    UINT32 regval;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_STS_1,
	     regval);

    *data = regval & 0xff;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_STS_2,
	     regval);

    *data |= (regval & 0xff) << 8;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_STS_3,
	     regval);

    *data |= (regval & 0xff) << 16;

    REG_READ(miiBaseAddressVirt,
	     IX_ETH_ACC_MAC_MDIO_STS_4,
	     regval);
    
    *data |= (regval & 0xff) << 24;
    
}


/********************************************************************
 * ixEthAccMiiInit
 */
IxEthAccStatus
ixEthAccMiiInit()
{
    if(ixOsServMutexInit(&miiAccessLock)!= IX_SUCCESS)
    {
	return IX_ETH_ACC_FAIL;
    }
    if(ixOsServMutexUnlock(&miiAccessLock)!= IX_SUCCESS)
    {
	return IX_ETH_ACC_FAIL;
    }
    
    miiBaseAddressVirt = (UINT32) IX_OSSERV_MEM_MAP(IX_ETH_ACC_MAC_0_BASE, IX_OSSERV_ETHA_MAP_SIZE);
    
    if (miiBaseAddressVirt == 0)
    {
      ixOsServLog(LOG_FATAL, "EthAcc: Could not map MII I/O mapped memory\n", 0, 0, 0, 0, 0, 0);
      
      return IX_ETH_ACC_FAIL;
    }
    
    return IX_ETH_ACC_SUCCESS;
}

void
ixEthAccMiiUnload(void)
{
  IX_OSSERV_MEM_UNMAP(miiBaseAddressVirt);
  
  miiBaseAddressVirt = 0;
}

/*********************************************************************
 * ixEthAccMiiReadRtn - read a 16 bit value from a PHY
 */
IxEthAccStatus      
ixEthAccMiiReadRtn (UINT8 phyAddr, 
		    UINT8 phyReg, 
		    UINT16 *value)
{
    UINT32 mdioCommand;
    UINT32 regval;
    UINT32 miiTimeout;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    if ((phyAddr >= IXP425_ETH_ACC_MII_MAX_ADDR) 
	|| (phyReg >= IXP425_ETH_ACC_MII_MAX_REG))
    {
	return (IX_ETH_ACC_FAIL);
    }

    if (value == NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }

    ixOsServMutexLock(&miiAccessLock);
    mdioCommand = phyReg <<  IX_ETH_ACC_MII_REG_SHL 
	| phyAddr << IX_ETH_ACC_MII_ADDR_SHL;
    mdioCommand |= IX_ETH_ACC_MII_GO;

    ixEthAccMdioCmdWrite(mdioCommand);
    
    miiTimeout = IX_ETH_ACC_MII_TIMEOUT_10TH_SECS;

    while(miiTimeout)
    {
	
	ixEthAccMdioCmdRead(&regval);
     
	if((regval & IX_ETH_ACC_MII_GO) == 0x0)
	{	    
	    break;
	}
	/*Sleep for 10th of a second*/
	ixOsServTaskSleep(IX_ETH_ACC_MII_10TH_SEC_IN_MILLIS);
	miiTimeout--;
    }
    

    
    if(miiTimeout == 0)
    {	
	ixOsServMutexUnlock(&miiAccessLock);
	*value = 0xffff;
	return IX_ETH_ACC_FAIL;
    }
    
    
    ixEthAccMdioStatusRead(&regval);
    if(regval & IX_ETH_ACC_MII_READ_FAIL)
    {
	ixOsServMutexUnlock(&miiAccessLock);
	*value = 0xffff;
	return IX_ETH_ACC_FAIL;
    }

    *value = regval & 0xffff;
    ixOsServMutexUnlock(&miiAccessLock);
    return IX_ETH_ACC_SUCCESS;
    
}


/*********************************************************************
 * ixEthAccMiiWriteRtn - write a 16 bit value to a PHY
 */
IxEthAccStatus
ixEthAccMiiWriteRtn (UINT8 phyAddr, 
		     UINT8 phyReg, 
		     UINT16 value)
{
    UINT32 mdioCommand;
    UINT32 regval;
    UINT16 readVal;
    UINT32 miiTimeout;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    if ((phyAddr >= IXP425_ETH_ACC_MII_MAX_ADDR) 
	|| (phyReg >= IXP425_ETH_ACC_MII_MAX_REG))
    {
	return (IX_ETH_ACC_FAIL);
    }
   
    /* ensure that a PHY is present at this address */
    if(ixEthAccMiiReadRtn(phyAddr,
                          IX_ETH_ACC_MII_CTRL_REG,
                          &readVal) != IX_ETH_ACC_SUCCESS)
    {
        return (IX_ETH_ACC_FAIL);
    }

    ixOsServMutexLock(&miiAccessLock);
    mdioCommand = phyReg << IX_ETH_ACC_MII_REG_SHL
	| phyAddr << IX_ETH_ACC_MII_ADDR_SHL ;
    mdioCommand |= IX_ETH_ACC_MII_GO | IX_ETH_ACC_MII_WRITE | value;

    ixEthAccMdioCmdWrite(mdioCommand);
    
    miiTimeout = IX_ETH_ACC_MII_TIMEOUT_10TH_SECS;

    while(miiTimeout)
    {
	
	ixEthAccMdioCmdRead(&regval);

	/*The "GO" bit is reset to 0 when the write completes*/
	if((regval & IX_ETH_ACC_MII_GO) == 0x0)
	{	    	    
	    break;
	}
	/*Sleep for 100 milliseconds*/
	ixOsServTaskSleep(IX_ETH_ACC_MII_10TH_SEC_IN_MILLIS);
        miiTimeout--;
    }
    
    ixOsServMutexUnlock(&miiAccessLock);
    if(miiTimeout == 0)
    {
	return IX_ETH_ACC_FAIL;
    }
    return IX_ETH_ACC_SUCCESS;
}


/*****************************************************************
 *
 *  Phy query functions
 *
 */
IxEthAccStatus
ixEthAccMiiStatsShow (UINT32 phyAddr)
{
    UINT16 regval;
    printf("Regisers on PHY at address 0x%x\n", phyAddr);
    ixEthAccMiiReadRtn(phyAddr, IX_ETH_ACC_MII_CTRL_REG, &regval);
    printf("    Control Register                  :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_STAT_REG, &regval);
    printf("    Status Register                   :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_PHY_ID1_REG, &regval);
    printf("    PHY ID1 Register                  :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_PHY_ID2_REG, &regval);
    printf("    PHY ID2 Register                  :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_AN_ADS_REG, &regval);
    printf("    Auto Neg ADS Register             :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_AN_PRTN_REG, &regval);
    printf("    Auto Neg Partner Ability Register :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_AN_EXP_REG, &regval);
    printf("    Auto Neg Expansion Register       :      0x%4.4x\n", regval);
    ixEthAccMiiReadRtn(phyAddr,  IX_ETH_ACC_MII_AN_NEXT_REG, &regval);
    printf("    Auto Neg Next Register            :      0x%4.4x\n", regval);

    return IX_ETH_ACC_SUCCESS;
}


/*****************************************************************
 *
 *  Interface query functions
 *
 */
IxEthAccStatus
ixEthAccMdioShow (void)
{
    UINT32 regval;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }

    ixOsServMutexLock(&miiAccessLock);
    ixEthAccMdioCmdRead(&regval);
    ixOsServMutexUnlock(&miiAccessLock);
    
    printf("MDIO command register\n");
    printf("    Go bit      : 0x%x\n", (regval & BIT(31)) >> 31);
    printf("    MDIO Write  : 0x%x\n", (regval & BIT(26)) >> 26);
    printf("    PHY address : 0x%x\n", (regval >> 21) & 0x1f);
    printf("    Reg address : 0x%x\n", (regval >> 16) & 0x1f);
	
    ixOsServMutexLock(&miiAccessLock);
    ixEthAccMdioStatusRead(&regval);
    ixOsServMutexUnlock(&miiAccessLock);

    printf("MDIO status register\n");
    printf("    Read OK     : 0x%x\n", (regval & BIT(31)) >> 31);
    printf("    Read Data   : 0x%x\n", (regval >> 16) & 0xff);

    return IX_ETH_ACC_SUCCESS;   
}

