/**
 * @file IxEthDataPlane.c
 *
 * @author Intel Corporation
 * @date 12-Feb-2002
 *
 * @brief This file contains the implementation of the IXPxxx 
 * Ethernet Access Data plane component
 *
 * Design Notes:
 *
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#include "IxAssert.h"
#include "IxNpeMh.h"
#include "IxEthAcc.h"
#include "IxOsServices.h"
#include "IxOsServicesMemAccess.h"
#include "IxEthDBPortDefs.h"
#include "IxFeatureCtrl.h"
#include "IxEthAcc_p.h"
#include "IxEthAccQueueAssign_p.h"

#include <stdio.h>

/**
 * private functions prototype
 */
PRIVATE IX_MBUF *
ixEthAccEntryFromQConvert(UINT32 qEntry, UINT32 mask);

PRIVATE UINT32 
ixEthAccMbufRxQPrepare(IX_MBUF *mbuf);

PRIVATE UINT32 
ixEthAccMbufTxQPrepare(IX_MBUF *mbuf);

PRIVATE IxEthAccStatus
ixEthAccTxSwQHighestPriorityGet(IxEthAccPortId portId,
				IxEthAccTxPriority *priorityPtr);

PRIVATE IxEthAccStatus
ixEthAccTxFromSwQ(IxEthAccPortId portId,
		  IxEthAccTxPriority priority);

PRIVATE IxEthAccStatus
ixEthAccRxFreeFromSwQ(IxEthAccPortId portId);

PRIVATE void 
ixEthAccMbufFromTxQ(IX_MBUF *mbuf);

PRIVATE void 
ixEthAccMbufFromRxQ(IX_MBUF *mbuf);

PRIVATE void 
ixEthAccMbufFromSwRxQ(IX_MBUF *mbuf);

PRIVATE IX_STATUS 
ixEthAccQmgrLockTxWrite(IxEthAccPortId portId, 
			UINT32 qBuffer);

PRIVATE IX_STATUS 
ixEthAccQmgrLockRxWrite(IxEthAccPortId portId, 
			UINT32 qBuffer);

PRIVATE IX_STATUS 
ixEthAccQmgrTxWrite(IxEthAccPortId portId, 
		    UINT32 qBuffer, 
		    UINT32 priority);

/**
 * @addtogroup IxEthAccPri
 *@{
 */

#define TX_INC(port,field) \
        ixEthAccPortData[port].ixEthAccTxData.stats.field++
#define RX_INC(port,field) \
        ixEthAccPortData[port].ixEthAccRxData.stats.field++
#define TX_STATS_INC(port,field) \
        IX_ETH_ACC_STATS_INC(ixEthAccPortData[port].ixEthAccTxData.stats.field)
#define RX_STATS_INC(port,field) \
        IX_ETH_ACC_STATS_INC(ixEthAccPortData[port].ixEthAccRxData.stats.field)

extern ixEthAccPortDataInfo   ixEthAccPortData[];

PRIVATE IxEthAccDataPlaneStats     ixEthAccDataStats;

PRIVATE IxFastMutex txWriteMutex[IX_ETH_ACC_NUMBER_OF_PORTS];
PRIVATE IxFastMutex rxWriteMutex[IX_ETH_ACC_NUMBER_OF_PORTS];

/**
 *
 * @brief function to retrieve the correct pointer from
 * a queue entry posted by the NPE
 *
 * @param qEntry : entry from qmgr queue 
 *        mask : applicable mask for this queue
 *        (4 most significant bits are used for additional informations)
 *
 * @return IX_MBUF * pointer to mbuf header
 *
 * @internal
 */
PRIVATE IX_MBUF * 
ixEthAccEntryFromQConvert(UINT32 qEntry, UINT32 mask)
{
    IX_MBUF *mbufPtr;

    if (qEntry != 0)
    {
	/* mask NPE bits */
	qEntry &= mask;
#if IX_ACC_DRAM_PHYS_OFFSET != 0
	/* restore the original address pointer (if PHYS_OFFSET is not 0) */
	qEntry |= (IX_ACC_DRAM_PHYS_OFFSET & ~IX_ETHNPE_QM_Q_RXENET_ADDR_MASK);
#endif
	/* phys2virt mbuf */
	mbufPtr = (IX_MBUF *)IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION(qEntry);
    
#if (CPU!=SIMSPARCSOLARIS)
	/* preload one cache line (mbuf headers are aligned
	 * and their size is 1 cache line)
	 */
	__asm__ (" pld [%0]\n": : "r" (mbufPtr));
#else
	/* preload not implemented on different processor */
#endif
    }
    else
    {
	mbufPtr = NULL;
    }

    return mbufPtr;
}

/**
 *
 * @brief Mbuf header conversion macros : they implement the 
 *  different conversions using a temporary value. They also double-check
 *  that the parameters can be converted to/from NPE format. 
 *
 */
#define PTR_VIRT2NPE(type,ptr) \
  do { UINT32 temp; \
      IX_ENSURE(sizeof(type) == sizeof(ptr), "Wrong parameter type"); \
      IX_ENSURE(sizeof(ptr) == sizeof(UINT32), "Wrong parameter type"); \
      temp = (UINT32)IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(ptr); \
      (ptr) = (type)IX_OSSERV_SWAP_NPE_SHARED_LONG(temp); } \
  while(0)
	
#define LEN_VIRT2NPE(len) \
  do { \
      IX_ENSURE(sizeof(len) == sizeof(UINT32), "Wrong parameter type"); \
      (len) = IX_OSSERV_SWAP_NPE_SHARED_LONG(len); } \
  while(0)

#define PTR_NPE2VIRT(type,ptr) \
  do { UINT32 temp; \
      IX_ENSURE(sizeof(type) == sizeof(ptr), "Wrong parameter type"); \
      IX_ENSURE(sizeof(ptr) == sizeof(UINT32), "Wrong parameter type"); \
      temp = IX_OSSERV_SWAP_NPE_SHARED_LONG((UINT32)(ptr)); \
      (ptr) = (type)IX_MMU_PHYSICAL_TO_VIRTUAL_TRANSLATION((void *)temp); } \
  while(0)

#define LEN_NPE2VIRT(len) \
  do { \
      IX_ENSURE(sizeof(len) == sizeof(UINT32), "Wrong parameter type"); \
      (len) = IX_OSSERV_SWAP_NPE_SHARED_LONG(len); } \
  while(0)

/* Convert the mbuf header for NPE transmission */
PRIVATE UINT32 
ixEthAccMbufTxQPrepare(IX_MBUF *mbuf)
{
    UINT32 qbuf;

    /* test for unchained mbufs */
    if (IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) == NULL)
    {
	/* unchained mbufs */
	IX_ETH_ACC_STATS_INC(ixEthAccDataStats.unchainedTxMBufs);

	/* endianess swap only the necessary fields */
	PTR_VIRT2NPE(char *,IX_MBUF_MDATA(mbuf));
	LEN_VIRT2NPE(IX_MBUF_MLEN(mbuf));

	/* unchained mbufs : the frame length is the mbuf length */
	IX_MBUF_PKT_LEN(mbuf) = IX_MBUF_MLEN(mbuf);

	/* flush MBUF header after all address conversions */
	IX_ACC_DATA_CACHE_FLUSH(mbuf, sizeof(IX_MBUF));
    }
    else
    {
	/* chained mbufs */
	IX_MBUF *ptr = mbuf;
	IX_MBUF *nextPtr;

	/* frame length conversion */
	LEN_VIRT2NPE(IX_MBUF_PKT_LEN(mbuf));

	do
	{
	    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.chainedTxMBufs);

	    /* we must save virtual next chain pointer */
	    nextPtr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr);
 
	    /* endianess swap only the necessary fields */
	    PTR_VIRT2NPE(IX_MBUF *,IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr));
	    PTR_VIRT2NPE(char *,IX_MBUF_MDATA(ptr));
	    LEN_VIRT2NPE(IX_MBUF_MLEN(ptr));
	    
	    /* flush MBUF header after all address conversions */
	    IX_ACC_DATA_CACHE_FLUSH(ptr, sizeof(IX_MBUF));

	    /* next mbuf in the chain */
	    ptr = nextPtr;
	}
	while(ptr != NULL);
    }
    
    /* virt2phys mbuf itself */
    qbuf = (UINT32)IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(mbuf);

    /* Ensure the bits which are reserved to exchange information with
     * the NPE are cleared 
     *
     * If the mbuf address is not correctly aligned, or from an
     * incompatible memory range, there is no point to continue
     */
    IX_ENSURE(((qbuf & ~IX_ETHNPE_QM_Q_TXENET_ADDR_MASK) == 0), 
	      "Invalid address range");

    return qbuf;
}

/* Convert the mbuf header for NPE reception */
PRIVATE UINT32 
ixEthAccMbufRxQPrepare(IX_MBUF *mbuf)
{
    UINT32 len;
    UINT32 qbuf;

    if (IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) == NULL)
    {
	/* unchained mbufs */
	IX_ETH_ACC_STATS_INC(ixEthAccDataStats.unchainedRxFreeMBufs);

	/* endianess swap only the necessary fields */
	PTR_VIRT2NPE(char *,IX_MBUF_MDATA(mbuf));

	/* round down the length to a multiple of 64 and convert */
	len = (UINT32)IX_MBUF_MLEN(mbuf);
	len &= IX_ETHNPE_ACC_RXFREE_BUFFER_LENGTH_MASK;
	len = IX_OSSERV_SWAP_NPE_SHARED_LONG(len);
	IX_MBUF_MLEN(mbuf) = len;
	
	/* flush MBUF header after all address conversions */
	IX_ACC_DATA_CACHE_FLUSH(mbuf, sizeof(IX_MBUF));
	
	/* remove the mbuf cache lines */
	IX_ACC_DATA_CACHE_INVALIDATE(mbuf, sizeof(IX_MBUF));
    }
    else
    {
	/* chained mbufs */
	IX_MBUF *ptr = mbuf;
	IX_MBUF *nextPtr;

	do
	{
	    /* unchained mbufs */
	    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.chainedRxFreeMBufs);

	    /* we must save virtual next chain pointer */
	    nextPtr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr);
 
	    /* endianess swap only the necessary fields */
	    PTR_VIRT2NPE(IX_MBUF *,IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr));
	    PTR_VIRT2NPE(char *,IX_MBUF_MDATA(ptr));

	    /* convert length and round down to a multiple of 64 */
	    len = (UINT32)IX_MBUF_MLEN(ptr);
	    len &= IX_ETHNPE_ACC_RXFREE_BUFFER_LENGTH_MASK;
	    len = IX_OSSERV_SWAP_NPE_SHARED_LONG(len);
	    IX_MBUF_MLEN(ptr) = len;

	    /* flush MBUF header after all address conversions */
	    IX_ACC_DATA_CACHE_FLUSH(ptr, sizeof(IX_MBUF));

	    /* remove the mbuf cache lines */
	    IX_ACC_DATA_CACHE_INVALIDATE(ptr, sizeof(IX_MBUF));

	    ptr = nextPtr;
	}
	while(ptr != NULL);
    }

    /* virt2phys mbuf itself */
    qbuf = (UINT32)IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION(mbuf);

    /* Ensure the bits which are reserved to exchange information with
     * the NPE are cleared 
     *
     * If the mbuf address is not correctly aligned, or from an
     * incompatible memory range, there is no point to continue
     */
    IX_ENSURE(((qbuf & ~IX_ETHNPE_QM_Q_RXENET_ADDR_MASK) == 0), 
	      "Invalid address range");

    return qbuf;
}

/* Convert the mbuf header after NPE transmission */
PRIVATE void 
ixEthAccMbufFromTxQ(IX_MBUF *mbuf)
{
    /* test for unchained mbufs */
    if (IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) == NULL)
    {
	/* unchained mbufs */
	IX_ETH_ACC_STATS_INC(ixEthAccDataStats.unchainedTxDoneMBufs);

	/* endianess swap only the necessary fields */
	PTR_NPE2VIRT(char *,IX_MBUF_MDATA(mbuf));
	LEN_NPE2VIRT(IX_MBUF_MLEN(mbuf));
	IX_MBUF_PKT_LEN(mbuf) = IX_MBUF_MLEN(mbuf);
    }
    else
    {
	/* chained mbufs */
	IX_MBUF *ptr = mbuf;

	/* frame length conversion */
	LEN_NPE2VIRT(IX_MBUF_PKT_LEN(mbuf));

	do
	{
	    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.chainedTxDoneMBufs);

	    /* endianess swap only the necessary fields */
	    PTR_NPE2VIRT(char *,IX_MBUF_MDATA(ptr));
	    LEN_NPE2VIRT(IX_MBUF_MLEN(ptr));
	    PTR_NPE2VIRT(IX_MBUF *,IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr));
	    ptr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr);
	}
	while (ptr != NULL);
    }
}

/* Convert the mbuf header after NPE reception */
PRIVATE void 
ixEthAccMbufFromRxQ(IX_MBUF *mbuf)
{
    UINT32 len;
    UINT32 mflags;

    /* test for unchained mbufs */
    if (IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(mbuf) == NULL)
    {
	/* unchained mbufs */
	IX_ETH_ACC_STATS_INC(ixEthAccDataStats.unchainedRxMBufs);
    
	/* endianess swap only the necessary fields */
	PTR_NPE2VIRT(char *,IX_MBUF_MDATA(mbuf));
        
	/* for performances, the NPE stores the frame length in 
	 * the second word of the mbuf header
	 * together with the flags
	 * - m_flag is stored in the lower nibble of the msb.
	 * - frmlen is stored in the least significant half word
	 */
	len = (UINT32)IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(mbuf);
	len = IX_OSSERV_SWAP_NPE_SHARED_LONG(len);

	/* set the flags */
	mflags = (len >> IX_ETHNPE_QM_Q_FIELD_FLAG_R);
	IX_MBUF_FLAGS(mbuf) |= (mflags & IX_ETHNPE_QM_Q_FIELD_FLAG_MASK);

	/* set the packet and frame length */
	len &= IX_ETHNPE_QM_Q_RXENET_LENGTH_MASK;
	IX_MBUF_PKT_LEN(mbuf) = IX_MBUF_MLEN(mbuf) = len;
        
        /* clears the next packet field */
	IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(mbuf) = NULL;
    }
    else
    {
	IX_MBUF *ptr = mbuf;
	UINT32 pktlen = 0;
	IX_MBUF *lastMbufPtr;

	/* chained mbufs */
	do
	{
	    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.chainedRxMBufs);
    
	    /* save the current pointer */
	    lastMbufPtr = ptr;

	    /* endianess swap only the necessary fields */
	    PTR_NPE2VIRT(char *,IX_MBUF_MDATA(ptr));
 	    PTR_NPE2VIRT(IX_MBUF *,IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr));

	    ptr = (IX_MBUF *)IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr);
	    if (ptr)
	    {
		/* not the last mbuf of a chain */
		LEN_NPE2VIRT(IX_MBUF_MLEN(lastMbufPtr));
		pktlen += IX_MBUF_MLEN(lastMbufPtr);
	    }
	}
	while (ptr != NULL);

	/* last mbuf : for performances, the NPE stores the 
	 * frame length in the second word of the mbuf header
	 * together with the flags
	 * - m_flag is stored in the lower nibble of the msb.
	 * - frmlen is stored in the least significant half word
	 */
	len = (UINT32)IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(lastMbufPtr);
	len = IX_OSSERV_SWAP_NPE_SHARED_LONG(len);

	/* set the flags in the first mbuf */
	mflags = (len >> IX_ETHNPE_QM_Q_FIELD_FLAG_R);
	IX_MBUF_FLAGS(mbuf) |= (mflags & IX_ETHNPE_QM_Q_FIELD_FLAG_MASK);

	/* set the length of the last mbuf */
	len &= IX_ETHNPE_QM_Q_RXENET_LENGTH_MASK;
	IX_MBUF_MLEN(lastMbufPtr) = len - pktlen;

	/* update the total length of the first mbuf */
	IX_MBUF_PKT_LEN(mbuf) = len;
 
        /* clears the next packet field */
	IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(lastMbufPtr) = NULL;
    }
}

/* Convert the mbuf header when retrieved from a sw RxFree queue */
PRIVATE void 
ixEthAccMbufFromSwRxQ(IX_MBUF *mbufPtr)
{
    IX_MBUF *ptr = mbufPtr;

    do
    {
	/* convert only the necessary fields */
	PTR_NPE2VIRT(char *,IX_MBUF_MDATA(ptr));
	IX_MBUF_PKT_LEN(ptr) = 0;
	IX_MBUF_MLEN(ptr) = 0;
        IX_MBUF_FLAGS(ptr) = 0;
        IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(ptr) = NULL;
	PTR_NPE2VIRT(IX_MBUF *,IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr));
	ptr = IX_MBUF_NEXT_BUFFER_IN_PKT_PTR(ptr);
    }
    while (ptr != NULL);
}

/* write to qmgr if possible and report an overflow if not passible 
 * This way, the tx feature is reentrant.
 */
PRIVATE IX_STATUS
ixEthAccQmgrLockTxWrite(IxEthAccPortId portId, UINT32 qBuffer)
{
    IX_STATUS qStatus;
    if (ixOsServFastMutexTryLock(&txWriteMutex[portId]) == IX_SUCCESS)
    {
	qStatus = ixQMgrQWrite(
	       IX_ETH_ACC_PORT_TO_TX_Q_ID(portId),
	       &qBuffer);
#ifndef NDEBUG  
	if (qStatus != IX_SUCCESS)
	{
	    TX_STATS_INC(portId, txOverflow);	    
	}
#endif
	ixOsServFastMutexUnlock(&txWriteMutex[portId]);
    }
    else
    {
	TX_STATS_INC(portId, txLock);
	qStatus = IX_QMGR_Q_OVERFLOW;
    }
    return qStatus;
}

/* write to qmgr if possible and report an overflow if not passible 
 * This way, the Rx feature is reentrant.
 */
PRIVATE IX_STATUS
ixEthAccQmgrLockRxWrite(IxEthAccPortId portId, UINT32 qBuffer)
{
    IX_STATUS qStatus;
    if (ixOsServFastMutexTryLock(&rxWriteMutex[portId]) == IX_SUCCESS)
    {
	qStatus = ixQMgrQWrite(
	       IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId),
	       &qBuffer);	    
#ifndef NDEBUG  
	if (qStatus != IX_SUCCESS)
	{
	    RX_STATS_INC(portId, rxFreeOverflow);	    
	}
#endif
	ixOsServFastMutexUnlock(&rxWriteMutex[portId]);
    }
    else
    {
	RX_STATS_INC(portId, rxFreeLock);
	qStatus = IX_QMGR_Q_OVERFLOW;
    }
    return qStatus;
}

/* 
 * Set the priority and write to a qmgr queue.
 */
PRIVATE IX_STATUS
ixEthAccQmgrTxWrite(IxEthAccPortId portId, UINT32 qBuffer, UINT32 priority)
{
    /* fill the priority field */
    qBuffer |= (priority << IX_ETHNPE_QM_Q_FIELD_PRIOR_R);

    return ixEthAccQmgrLockTxWrite(portId, qBuffer);	    
}

/**
 *
 * @brief This function will discover the highest priority S/W Tx Q that
 *        has entries in it
 *
 * @param portId - (in) the id of the port whose S/W Tx queues are to be searched
 *        priorityPtr - (out) the priority of the highest priority occupied q will be written
 *                      here
 *
 * @return IX_ETH_ACC_SUCCESS if an occupied Q is found
 *         IX_ETH_ACC_FAIL if no Q has entries
 *
 * @internal
 */
PRIVATE IxEthAccStatus
ixEthAccTxSwQHighestPriorityGet(IxEthAccPortId portId,
				IxEthAccTxPriority *priorityPtr)
{
    if (ixEthAccPortData[portId].ixEthAccTxData.schDiscipline 
	== FIFO_NO_PRIORITY)
    {
	if(IX_ETH_ACC_DATAPLANE_IS_Q_EMPTY(ixEthAccPortData[portId].
	       ixEthAccTxData.txQ[IX_ETH_ACC_TX_DEFAULT_PRIORITY]))
	{
	    return IX_ETH_ACC_FAIL;
	}
	else
	{
	    *priorityPtr = IX_ETH_ACC_TX_DEFAULT_PRIORITY;
	    TX_STATS_INC(portId,txPriority[*priorityPtr]);
	    return IX_ETH_ACC_SUCCESS;
	}
    }
    else
    {
	IxEthAccTxPriority highestPriority = IX_ETH_ACC_TX_PRIORITY_7;       
	while(1)
	{
	    if(!IX_ETH_ACC_DATAPLANE_IS_Q_EMPTY(ixEthAccPortData[portId].
	       ixEthAccTxData.txQ[highestPriority]))
	    {

		*priorityPtr = highestPriority;
		TX_STATS_INC(portId,txPriority[highestPriority]);
		return IX_ETH_ACC_SUCCESS;

	    }
	    if (highestPriority == IX_ETH_ACC_TX_PRIORITY_0)
	    {
		return IX_ETH_ACC_FAIL;
	    }
	    highestPriority--;
	}
    }
}

/**
 *
 * @brief This function will take a buffer from a TX S/W Q and attempt
 *        to add it to the relevant TX H/W Q
 *
 * @param portId - the port whose TX queue is to be written to
 *        priority - identifies the queue from which the entry is to be read
 *
 * @internal
 */
PRIVATE IxEthAccStatus
ixEthAccTxFromSwQ(IxEthAccPortId portId,
		  IxEthAccTxPriority priority)
{
    IX_MBUF        *mbuf;
    IX_STATUS	   qStatus;

    IX_ENSURE((UINT32)priority <= (UINT32)7, "Invalid priority");

    IX_ETH_ACC_DATAPLANE_REMOVE_MBUF_FROM_Q_HEAD(
	ixEthAccPortData[portId].ixEthAccTxData.txQ[priority], 
	mbuf);
    
    if (mbuf != NULL)
    {
        /*
        * the sw queue modifies the next packet field, reset it
        */
        IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(mbuf) = NULL;

	/*
	 * Add the Tx buffer to the H/W Tx Q  
	 */

	/* 
	 * We do not need to flush here as it is already done
	 * in TxFrameSubmit()
	 */
	qStatus = ixEthAccQmgrTxWrite(portId, 
	      IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION((UINT32)mbuf),
	      priority);
	
	if (qStatus == IX_SUCCESS)
	{
	    TX_STATS_INC(portId,txFromSwQOK);
	    return IX_SUCCESS;
	}
	else if (qStatus == IX_QMGR_Q_OVERFLOW)
	{
	    /*
	     * H/W Q overflow, need to save the buffer 
	     * back on the s/w Q.
	     * we must put it back on the head of the q to avoid 
	     * reordering packet tx
	     */
	    TX_STATS_INC(portId,txFromSwQDelayed);
	    IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_HEAD(
		ixEthAccPortData[portId].ixEthAccTxData.txQ[priority], 
		mbuf);

	    /*enable Q notification*/
	    qStatus = ixQMgrNotificationEnable(
		IX_ETH_ACC_PORT_TO_TX_Q_ID(portId),
		IX_ETH_ACC_PORT_TO_TX_Q_SOURCE(portId));	 
        
            if (qStatus != IX_SUCCESS && qStatus != IX_QMGR_WARNING)
            {
		TX_INC(portId,txUnexpectedError);
		IX_ETH_ACC_FATAL_LOG(
	            "IXETHACC:ixEthAccPortTxFromSwQ:Error: %u\n", 
	            qStatus, 0, 0, 0, 0, 0);
            }
	}
	else 
	{
	    TX_INC(portId,txUnexpectedError);

	    /* recovery attempt */
	    IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_HEAD(
		ixEthAccPortData[portId].ixEthAccTxData.txQ[priority], 
		mbuf);
	    
	    IX_ETH_ACC_FATAL_LOG(
		"IXETHACC:ixEthAccTxFromSwQ:Error: QM status 0x%08X\n", 
		qStatus, 0, 0, 0, 0, 0);	
	}
    }
    else
    {
	/* sw queue is empty */
    }
    return IX_ETH_ACC_FAIL;
}

/**
 *
 * @brief This function will take a buffer from a RXfree S/W Q and attempt
 *        to add it to the relevant RxFree H/W Q
 *
 * @param portId - the port whose RXFree queue is to be written to
 *
 * @internal
 */
PRIVATE IxEthAccStatus
ixEthAccRxFreeFromSwQ(IxEthAccPortId portId)
{
    IX_MBUF        *mbuf;
    IX_STATUS	   qStatus = IX_SUCCESS;

    IX_ETH_ACC_DATAPLANE_REMOVE_MBUF_FROM_Q_HEAD(
	  ixEthAccPortData[portId].ixEthAccRxData.freeBufferList, 
	  mbuf);
    if (mbuf != NULL)
    {
	/* flush the mbuf header (modified in the sw Q) */
	IX_ACC_DATA_CACHE_FLUSH(mbuf, sizeof(IX_MBUF));
	/* remove the mbuf from the cache) */
	IX_ACC_DATA_CACHE_INVALIDATE(mbuf, sizeof(IX_MBUF));
		
	/*
	 * Add The Rx Buffer to the H/W Free buffer Q if possible  
	 */
	qStatus = ixEthAccQmgrLockRxWrite(portId, 
	  IX_MMU_VIRTUAL_TO_PHYSICAL_TRANSLATION((UINT32)mbuf));

	if (qStatus == IX_SUCCESS)
	{
	    RX_STATS_INC(portId,rxFreeRepFromSwQOK);
	    /*
	     * Buffer added to h/w Q.
	     */
	    return IX_SUCCESS;
	}
	else if (qStatus == IX_QMGR_Q_OVERFLOW)
	{
	    /*
	     * H/W Q overflow, need to save the buffer back on the s/w Q.
	     */
	    RX_STATS_INC(portId,rxFreeRepFromSwQDelayed);

	    IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_HEAD(
		   ixEthAccPortData[portId].ixEthAccRxData.freeBufferList, 
		   mbuf);
	}
	else 
	{
	    /* unexpected qmgr error */
	    RX_INC(portId,rxUnexpectedError);

	    IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_HEAD(
		    ixEthAccPortData[portId].ixEthAccRxData.freeBufferList, 
		    mbuf);

	    IX_ETH_ACC_FATAL_LOG("IXETHACC:rxfreeFromSwQ:Error: QM status 0x%08X\n", 
				 qStatus, 0, 0, 0, 0, 0);
	}
    }
    else
    {
	/* sw queue is empty */
    }
    return IX_ETH_ACC_FAIL;
}


IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccInitDataPlane()
{	
    UINT32 portId;
    
    /* 
     * Initialize the service and register callback to other services.
     */

    IX_ETH_ACC_MEMSET(&ixEthAccDataStats, 0, sizeof(ixEthAccDataStats));

    for(portId=IX_ETH_PORT_1; portId<=IX_ETH_PORT_2; portId++)
    {
	ixOsServFastMutexInit(&txWriteMutex[portId]);
	ixOsServFastMutexInit(&rxWriteMutex[portId]);
    }

    return (IX_ETH_ACC_SUCCESS);
}


IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccPortTxDoneCallbackRegister(IxEthAccPortId portId, 
						  IxEthAccPortTxDoneCallback
						  txCallbackFn, 
						  UINT32 callbackTag)
{	
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	return (IX_ETH_ACC_INVALID_PORT);
    }

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot register TxDone Callback.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    if (txCallbackFn == 0)
	/* Check for null function pointer here. */
    {
	return (IX_ETH_ACC_INVALID_ARG);
    }
    ixEthAccPortData[portId].ixEthAccTxData.txBufferDoneCallbackFn = txCallbackFn;
    ixEthAccPortData[portId].ixEthAccTxData.txCallbackTag = callbackTag;
    return (IX_ETH_ACC_SUCCESS);
}


IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccPortRxCallbackRegister(IxEthAccPortId portId, 
					      IxEthAccPortRxCallback
					      rxCallbackFn, UINT32
					      callbackTag)
{	
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	return (IX_ETH_ACC_INVALID_PORT);
    }

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot register Rx Callback.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }
 
    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    
    /* Check for null function pointer here. */
    if (rxCallbackFn == NULL)
    {
	return (IX_ETH_ACC_INVALID_ARG);
    }
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn = rxCallbackFn;
    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag = callbackTag;
    return (IX_ETH_ACC_SUCCESS);
}

IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccPortTxFrameSubmit(IxEthAccPortId portId, 
					 IX_MBUF *buffer, 
					 IxEthAccTxPriority priority)
{	
    IX_STATUS	qStatus = IX_SUCCESS;
    UINT32      qBuffer;
    IxEthAccTxPriority highestPriority;
    IxQMgrQStatus txQStatus;

#ifndef NDEBUG 
    if (buffer == NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	return (IX_ETH_ACC_INVALID_PORT);
    }

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_FATAL_LOG("EthAcc: Unavailable Eth %d: Cannot submit Tx Frame.\n",
			     (INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_PORT_UNINITIALIZED ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    if ((UINT32)priority > (UINT32)IX_ETH_ACC_TX_PRIORITY_7)
    {
	return (IX_ETH_ACC_INVALID_ARG);
    }
    if (IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(buffer) != NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }
#endif 

    /*
     * Need to Flush the MBUF and its contents (data) as it may be
     * read from the NPE. Convert virtual addresses to physical addresses also.
     */
    qBuffer = ixEthAccMbufTxQPrepare(buffer);

    /*
     * If no fifo priority ...
     */
    if (ixEthAccPortData[portId].ixEthAccTxData.schDiscipline == 
	FIFO_NO_PRIORITY)
    {
	/*
	 * Add The Tx Buffer to the H/W Tx Q if possible  
	 */
	qStatus = ixEthAccQmgrTxWrite(portId,
				      qBuffer,
				      IX_ETH_ACC_TX_DEFAULT_PRIORITY);

	if (qStatus == IX_SUCCESS)
	{
	    TX_STATS_INC(portId,txQOK);

	    /*  
	     * Happy day scenario : Buffer added to h/w Q.
	     */
	    return (IX_SUCCESS);
	}
	else if (qStatus == IX_QMGR_Q_OVERFLOW)
	{
	    /*
	     * We were unable to write the buffer to the 
	     * appropriate H/W Q,  Save it in the sw Q.
	     * (use the default priority queue regardless of 
	     * input parameter)
	     */
	    priority = IX_ETH_ACC_TX_DEFAULT_PRIORITY;
	}
	else 
	{
	    /* unexpected qmgr error */
	    TX_INC(portId,txUnexpectedError);
	    IX_ETH_ACC_FATAL_LOG(
		"IXETHACC:ixEthAccPortTxFrameSubmit:Error: qStatus = %u\n", 
		(UINT32)qStatus, 0, 0, 0, 0, 0);
	    return (IX_ETH_ACC_FAIL);
	}
    }
    else if (ixEthAccPortData[portId].ixEthAccTxData.schDiscipline == 
	     FIFO_PRIORITY)
    {
	
	/*
	 * For priority transmission, put the frame directly on the H/W queue
	 * if the H/W queue is empty, otherwise, put it in a S/W Q
	 */
	ixQMgrQStatusGet(IX_ETH_ACC_PORT_TO_TX_Q_ID(portId), &txQStatus);
	if((txQStatus & IX_QMGR_Q_STATUS_E_BIT_MASK) != 0)
	{
	    /*The tx queue is empty, check whether there are buffers on the s/w queues*/
	    if(ixEthAccTxSwQHighestPriorityGet(portId,  &highestPriority)
	       !=IX_ETH_ACC_FAIL)
	    {
		/*there are buffers on the s/w queues, submit them*/
		ixEthAccTxFromSwQ(portId, highestPriority);

		/* the queue was empty, 1 buffer is supplied
		 * but is likely to be transmitted now and the 
		 * hw queue is empty again, so submit more from
		 * the sw queues
		 */
		if(ixEthAccTxSwQHighestPriorityGet(portId,  &highestPriority)
		   !=IX_ETH_ACC_FAIL)
		{
		    ixEthAccTxFromSwQ(portId, highestPriority);
		    /*
		     * and force the buffer supplied to be placed 
		     * on a priority queue
		     */
		    qStatus = IX_QMGR_Q_OVERFLOW;
		}
		else
		{
		    /*there are no buffers in the s/w queues, submit directly*/
		    qStatus = ixEthAccQmgrTxWrite(portId, qBuffer, priority);
		}
	    }
	    else
	    {
		/*there are no buffers in the s/w queues, submit directly*/
		qStatus = ixEthAccQmgrTxWrite(portId, qBuffer, priority);
	    }
	}
	else
	{
	    qStatus = IX_QMGR_Q_OVERFLOW;
	}
    }
    else 
    {
	TX_INC(portId,txUnexpectedError);
	IX_ETH_ACC_FATAL_LOG(
	    "IXETHACC:ixEthAccPortTxFrameSubmit:Error: wrong schedule discipline setup\n", 
	    0, 0, 0, 0, 0, 0);
	return (IX_ETH_ACC_FAIL);
    }
    
    if(qStatus == IX_SUCCESS ) 
    {
	TX_STATS_INC(portId,txQOK);
	return IX_ETH_ACC_SUCCESS;
    }
    else if(qStatus == IX_QMGR_Q_OVERFLOW)
    {
	TX_STATS_INC(portId,txQDelayed);
	/*
	 * We were unable to write the buffer to the 
	 * appropriate H/W Q,  Save it in a s/w Q.
	 */
	IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_TAIL(
		ixEthAccPortData[portId].
		ixEthAccTxData.txQ[priority], 
		buffer);
	    
	qStatus = ixQMgrNotificationEnable(
		IX_ETH_ACC_PORT_TO_TX_Q_ID(portId),
		IX_ETH_ACC_PORT_TO_TX_Q_SOURCE(portId));

        if (qStatus != IX_SUCCESS)
	{
	    if (qStatus == IX_QMGR_WARNING)
	    {
		/* notification is enabled for a queue 
		 * which is already empty (the condition is already met)
		 * and there will be no more queue event to drain the sw queue
		 */
		TX_STATS_INC(portId,txLateNotificationEnabled);
		
		/* pull a buffer from the sw queue */
		if(ixEthAccTxSwQHighestPriorityGet(portId,  &highestPriority)
		   !=IX_ETH_ACC_FAIL)
		{
		    /*there are buffers on the s/w queues, submit from them*/
		    ixEthAccTxFromSwQ(portId, highestPriority);
		}
	    }
	    else
	    {
		TX_INC(portId,txUnexpectedError);
		IX_ETH_ACC_FATAL_LOG(
		     "IXETHACC:ixEthAccPortTxFrameSubmit:Error: %u\n", 
		     qStatus, 0, 0, 0, 0, 0);
	    }
        }
    }
    else
    {
	TX_INC(portId,txUnexpectedError);
	IX_ETH_ACC_FATAL_LOG(
	     "IXETHACC:ixEthAccPortTxFrameSubmit:Error: %u\n", 
	     qStatus, 0, 0, 0, 0, 0);
	return (IX_ETH_ACC_FAIL);
    }

    return (IX_ETH_ACC_SUCCESS);
}


/**
 *
 * @brief replenish: convert a chain of mbufs to the format 
 *        expected by the NPE
 *
  */

IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccPortRxFreeReplenish(IxEthAccPortId portId, 
					   IX_MBUF *buffer)
{	
    IX_STATUS	qStatus = IX_SUCCESS;
    UINT32      qBuffer;
    
    /* 
     * Check buffer is valid.
     */
    
#ifndef NDEBUG 
    /* check parameter value */
    if (buffer == 0)
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	return (IX_ETH_ACC_INVALID_PORT);
    }

    /* check initialisation is done */
    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_FATAL_LOG("EthAcc: Unavailable Eth %d: Cannot replenish Rx Free Q.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_PORT_UNINITIALIZED ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    /* check boundaries and constraints */
    if (IX_MBUF_MLEN(buffer) < IX_ETHNPE_ACC_RXFREE_BUFFER_LENGTH_MIN)
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (IX_MBUF_NEXT_PKT_IN_CHAIN_PTR(buffer) != NULL)
    {
	return (IX_ETH_ACC_FAIL);
    }
#endif 
    
    qBuffer = ixEthAccMbufRxQPrepare(buffer);

    /*
     * Add The Rx Buffer to the H/W Free buffer Q if possible  
     */
    qStatus = ixEthAccQmgrLockRxWrite(portId, qBuffer);

    if (qStatus == IX_SUCCESS)
    {
	RX_STATS_INC(portId,rxFreeRepOK);
	/* 
	 * Buffer added to h/w Q.
	 */
	return (IX_SUCCESS);
    }
    else if (qStatus == IX_QMGR_Q_OVERFLOW)
    {
	RX_STATS_INC(portId,rxFreeRepDelayed);
	/*
	 * We were unable to write the buffer to the approprate H/W Q, 
	 * Save it in a s/w Q.
	 */
	IX_ETH_ACC_DATAPLANE_ADD_MBUF_TO_Q_HEAD(
	    ixEthAccPortData[portId].ixEthAccRxData.freeBufferList, 
	    buffer);

	qStatus = ixQMgrNotificationEnable(
	    IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId),
	    IX_ETH_ACC_PORT_TO_RX_FREE_Q_SOURCE(portId));
        
        if (qStatus != IX_SUCCESS)
	{
	    if (qStatus == IX_QMGR_WARNING)
	    {
		/* notification is enabled for a queue 
		 * which is already empty (the condition is already met)
		 * and there will be no more queue event to drain the sw queue
		 * move an entry from the sw queue to the hw queue */
		RX_STATS_INC(portId,rxFreeLateNotificationEnabled);
		ixEthAccRxFreeFromSwQ(portId);
	    }
	    else
	    {
		RX_INC(portId,rxUnexpectedError);
		IX_ETH_ACC_FATAL_LOG(
		     "IXETHACC:ixEthAccRxPortFreeReplenish:Error: %u\n", 
		     qStatus, 0, 0, 0, 0, 0);
	    }
        }
    }
    else 
    {
	RX_INC(portId,rxUnexpectedError);
	IX_ETH_ACC_FATAL_LOG(
	    "IXETHACC:ixEthAccRxPortFreeReplenish:Error: qStatus = %u\n", 
	    (UINT32)qStatus, 0, 0, 0, 0, 0);
        return(IX_ETH_ACC_FAIL);
    }
    return (IX_ETH_ACC_SUCCESS);
}


IX_ETH_ACC_PUBLIC
IxEthAccStatus ixEthAccTxSchedulingDisciplineSet(IxEthAccPortId portId, 
						 IxEthAccTxSchedulerDiscipline
						 sched)
{	
    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return (IX_ETH_ACC_FAIL);
    }
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	return (IX_ETH_ACC_INVALID_PORT);
    }

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: Cannot set Tx Scheduling Discipline.\n",(INT32)portId,0,0,0,0,0);
        return IX_ETH_ACC_SUCCESS ;
    }

    if (!IX_ETH_IS_PORT_INITIALIZED(portId))
    {
	return (IX_ETH_ACC_PORT_UNINITIALIZED);
    }
    ixEthAccPortData[portId].ixEthAccTxData.schDiscipline = sched;
    return (IX_ETH_ACC_SUCCESS);
}
/**
 * @fn ixEthRxFrameQMCallback 
 *
 * @brief receive callback for Frame receive Q from NPE
 *
 * @param @ref IxQMgrCallback
 *
 * @return none
 *
 * @internal
 *
 * Design note : while processing the entry X, entry X+1 is preloaded
 * into memory to reduce the number of stall cycles
 *
 * @warning If NDEBUG is set
 * any portID returned via the received callback is invalid if
 * greater than IX_ETH_DB_NUMBER_OF_PORTS,
 * This problem is the result of the NPE Rx descriptor format
 * limitation, which can report valid ports only between 0 and 5, 
 * and will use 6 for "reserved" and "7" for "not found", while
 * the XScale Ethernet Database can use these numbers for legitimate
 * ports.
 */
void ixEthRxFrameQMCallback(IxQMgrQId qId, IxQMgrCallbackId callbackId)
{	
    IX_MBUF    *mbufPtr;
    IX_MBUF    *nextMbufPtr;
    UINT32     qEntry;
    UINT32     nextQEntry;
    UINT32     *qEntryPtr;
    UINT32     portId;
    UINT32     npeId;
    UINT32     rxQReadStatus;
    /*
     * Design note : entries are read in a static buffer, This buffer contains
     * an extra entyry (which is zeroed by the compiler), so the loop will
     * always terminate on a null entry, whatever the result of Burst read is.
     */
    static UINT32 rxQEntry[IX_ETH_ACC_MAX_RX_FRAME_CONSUME_PER_CALLBACK + 1];

    /* 
     * Indication that there are Rx frames in the QM Rx Frame Q.
     */
    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.rxCallbackCounter);

    do
    {
	qEntryPtr = rxQEntry;
	rxQReadStatus = ixQMgrQBurstRead(IX_ETH_ACC_RX_FRAME_ETH_Q, 
		 IX_ETH_ACC_MAX_RX_FRAME_CONSUME_PER_CALLBACK, 
		 qEntryPtr);

#ifndef NDEBUG
	if ((rxQReadStatus != IX_QMGR_Q_UNDERFLOW)
	    && (rxQReadStatus != IX_SUCCESS))
	{
	    ixEthAccDataStats.unexpectedError++;
	    /*major error*/
	    IX_ETH_ACC_FATAL_LOG(
		"IXETHACC:ixEthRxFrameQMCallback:Error: %u\n", 
		(UINT32)rxQReadStatus, 0, 0, 0, 0, 0);
	    return;
	}
#endif

	/* convert and preload the first entry
	 * (the conversion function takes care about null pointers which
	 * are used to mark the end of the loop)
	 */
	nextQEntry = *qEntryPtr;
	nextMbufPtr = ixEthAccEntryFromQConvert(nextQEntry, 
			  IX_ETHNPE_QM_Q_RXENET_ADDR_MASK);

	while(nextQEntry != 0)
	{
	    /* get the next entry */
	    qEntry = nextQEntry;
	    mbufPtr = nextMbufPtr;

#ifndef NDEBUG
	    if (mbufPtr == (IX_MBUF *) NULL)
	    {
		ixEthAccDataStats.unexpectedError++;
		IX_ETH_ACC_FATAL_LOG(
		    "IXETHACC:ixEthRxQMCallback:Error: Null Mbuf Ptr\n", 
		    0, 0, 0, 0, 0, 0);
		return;
	    }
#endif

	    /* convert and preload the next entry
	     * (the conversion function takes care about null pointers which
	     * are used to mark the end of the loop)
	     */
	    nextQEntry = *(++qEntryPtr);
	    nextMbufPtr = ixEthAccEntryFromQConvert(nextQEntry, 
			      IX_ETHNPE_QM_Q_RXENET_ADDR_MASK);

	    /* convert fields from mbuf header */
	    ixEthAccMbufFromRxQ(mbufPtr);
	    
	    /*
	     * Get Port and Npe ID from message.
	     */	    
	    portId = ((IX_ETHNPE_QM_Q_RXENET_PORTID_MASK & 
		       qEntry) >> 
		      IX_ETHNPE_QM_Q_FIELD_PORTID_R);
	    
	    npeId = ((IX_ETHNPE_QM_Q_RXENET_NPEID_MASK & 
		      qEntry) >> 
		     IX_ETHNPE_QM_Q_FIELD_NPEID_R);
   
	    /* 
	     * Call user level callback.
	     */
	    RX_STATS_INC(npeId,rxFrameClientCallback);
	    ixEthAccPortData[npeId].ixEthAccRxData.rxCallbackFn(
		ixEthAccPortData[npeId].ixEthAccRxData.rxCallbackTag, 
		mbufPtr, 
		portId);
	}
    } while (rxQReadStatus == IX_SUCCESS);
}

/**
 * @brief  rxFree low event handler
 *
 */
void ixEthRxFreeQMCallback(IxQMgrQId qId, IxQMgrCallbackId callbackId)
{	
    IxEthAccPortId	portId = (IxEthAccPortId) callbackId;
    int		        lockVal;
    UINT32		maxQWritesToPerform = IX_ETH_ACC_MAX_RX_FREE_BUFFERS_LOAD;
    IX_STATUS	        qStatus = IX_SUCCESS;

    /*
     * We have reached a low threshold on one of the Rx Free Qs
     */
    
    /*note that due to the fact that we are working off an Empty threshold, this callback
      need only write a single entry to the Rx Free queue in order to re-arm the notification
    */
    
    RX_STATS_INC(portId,rxFreeLowCallback);
    
    /* 
     * Get buffers from approprite S/W Rx freeBufferList Q.
     */
    
#ifndef NDEBUG 
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	IX_ETH_ACC_FATAL_LOG(
	    "IXETHACC:ixEthRxFreeQMCallback:Error: Invalid Port 0x%08X\n", 
	    portId, 0, 0, 0, 0, 0);
	return;
    }
#endif
    IX_ETH_ACC_DATA_PLANE_LOCK(lockVal);
    if (IX_ETH_ACC_DATAPLANE_IS_Q_EMPTY(ixEthAccPortData[portId].
					ixEthAccRxData.freeBufferList))
    {
	/*  
	 * Turn off Q callback notification for Q in Question.
	 */
	qStatus = ixQMgrNotificationDisable(
	    IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId));
	

	IX_ETH_ACC_DATA_PLANE_UNLOCK(lockVal);

	if (qStatus != IX_SUCCESS)
	{
	    RX_INC(portId,rxUnexpectedError);
	    IX_ETH_ACC_FATAL_LOG(
		"IXETHACC:ixEthRxFreeQMCallback:Error: QM status 0x%08X\n", 
		qStatus, 0, 0, 0, 0, 0);
	    return;
	}
    }
    else 
    {
	IX_ETH_ACC_DATA_PLANE_UNLOCK(lockVal);
	/*
	 * Load the H/W Q with buffers from the s/w Q.
	 */
	
	do
	{
	    /* 
	     * Consume Q entries. - Note Q contains Physical addresss, 
	     * and have already been flushed to memory,
	     * And endianess converted if required.
	     */
	    if (ixEthAccRxFreeFromSwQ(portId) != IX_SUCCESS)
	    {
		/*
		 * No more entries in s/w Q.
		 * Turn off Q callback indication
		 */
		
		IX_ETH_ACC_DATA_PLANE_LOCK(lockVal);
		if (IX_ETH_ACC_DATAPLANE_IS_Q_EMPTY(ixEthAccPortData[portId].
		    ixEthAccRxData.freeBufferList))
		{
		    qStatus = ixQMgrNotificationDisable(
			IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId));
		}
		IX_ETH_ACC_DATA_PLANE_UNLOCK(lockVal);
		break;
	    }
	}
	while (--maxQWritesToPerform);
    }
}
/**
 * @fn Tx queue low event handler
 *
 */
void 
ixEthTxFrameQMCallback(IxQMgrQId qId, IxQMgrCallbackId callbackId)
{
    IxEthAccPortId portId = (IxEthAccPortId) callbackId;
    int		   lockVal;
    UINT32	   maxQWritesToPerform = IX_ETH_ACC_MAX_TX_FRAME_TX_CONSUME_PER_CALLBACK;
    IX_STATUS	   qStatus = IX_SUCCESS;
    IxEthAccTxPriority highestPriority;

    
    /*
     * We have reached a low threshold on the Tx Q, and are being asked to 
     * supply a buffer for transmission from our S/W TX queues
     */
    TX_STATS_INC(portId,txLowThreshCallback);

    /* 
     * Get buffers from approprite Q.
     */
    
#ifndef NDEBUG
    if (!IX_ETH_ACC_IS_PORT_VALID(portId))
    {
	IX_ETH_ACC_FATAL_LOG(
	    "IXETHACC:ixEthTxFrameQMCallback:Error: Invalid Port 0x%08X\n", 
	    portId, 0, 0, 0, 0, 0);
	return;
    }
#endif

    do
    {
	/* 
	 * Consume Q entries. - Note Q contains Physical addresss, 
	 * and have already been flushed to memory,
	 * And endianess converted if required.
	 */
	
	IX_ETH_ACC_DATA_PLANE_LOCK(lockVal);
	
	if(ixEthAccTxSwQHighestPriorityGet(portId, &highestPriority) ==
	   IX_ETH_ACC_FAIL)
	{
	    /*
	     * No more entries in s/w Q.
	     * Turn off Q callback indication
	     */
	    qStatus = ixQMgrNotificationDisable(
		IX_ETH_ACC_PORT_TO_TX_Q_ID(portId));

	    IX_ETH_ACC_DATA_PLANE_UNLOCK(lockVal);

	    if (qStatus != IX_SUCCESS)
	    {
		ixEthAccDataStats.unexpectedError++;
		IX_ETH_ACC_FATAL_LOG(
		    "IXETHACC:ixEthTxFrameQMCallback:Error: QM status 0x%08X\n", 
		    qStatus, 0, 0, 0, 0, 0);
	    }
	    
	    return;	    
	}
	else
	{
	    IX_ETH_ACC_DATA_PLANE_UNLOCK(lockVal);
	    if (ixEthAccTxFromSwQ(portId,highestPriority)!=IX_SUCCESS)
	    {
		RX_INC(portId,rxUnexpectedError);
		return;
	    }
	}	
    }  	
    while (--maxQWritesToPerform);
}

/**
 * @brief TxDone event handler
 *
 * Design note : while processing the entry X, entry X+1 is preloaded
 * into memory to reduce the number of stall cycles
 *
 */

void 
ixEthTxFrameDoneQMCallback(IxQMgrQId qId, IxQMgrCallbackId callbackId)
{	
    IX_MBUF    *mbufPtr;
    IX_MBUF    *nextMbufPtr;
    UINT32     qEntry;
    UINT32     nextQEntry;
    UINT32     *qEntryPtr;
    UINT32     txDoneQReadStatus;
    UINT32     portId;
    /*
     * Design note : entries are read in a static buffer, This buffer contains
     * an extra entyry (which is zeroed by the compiler), so the loop will
     * always terminate on a null entry, whatever the result of Burst read is.
     */
    static UINT32 txDoneQEntry[IX_ETH_ACC_MAX_TX_FRAME_DONE_CONSUME_PER_CALLBACK + 1];

    /* 
     * Indication that Tx frames have been transmitted from the NPE.
     */
    
    IX_ETH_ACC_STATS_INC(ixEthAccDataStats.txDoneCallbackCounter);
    
    do{
	qEntryPtr = txDoneQEntry;
	txDoneQReadStatus = ixQMgrQBurstRead(IX_ETH_ACC_TX_FRAME_DONE_ETH_Q, 
		     IX_ETH_ACC_MAX_TX_FRAME_DONE_CONSUME_PER_CALLBACK,
		     qEntryPtr);

#ifndef NDEBUG
	if (txDoneQReadStatus != IX_QMGR_Q_UNDERFLOW
	    && (txDoneQReadStatus != IX_SUCCESS))
	{
	    /*major error*/
	    ixEthAccDataStats.unexpectedError++;
	    IX_ETH_ACC_FATAL_LOG(
		"IXETHACC:ixEthTxFrameDoneQMCallback:Error: %u\n", 
		(UINT32)txDoneQReadStatus, 0, 0, 0, 0, 0);
	    return;
	}
#endif

	/* convert and preload the first entry 
	 * (the conversion function takes care about null pointers which
	 * are used to mark the end of the loop)
	 */
	nextQEntry = *qEntryPtr;
	nextMbufPtr = ixEthAccEntryFromQConvert(nextQEntry,
			  IX_ETHNPE_QM_Q_TXENET_ADDR_MASK);

	while(nextQEntry != 0)
	{	    
	    /* get the next entry */
	    qEntry = nextQEntry;
	    mbufPtr = nextMbufPtr;

#ifndef NDEBUG
	    if (mbufPtr == (IX_MBUF *) NULL)
	    {
		ixEthAccDataStats.unexpectedError++;
		IX_ETH_ACC_FATAL_LOG(
		    "IXETHACC:ixEthTxFrameDoneQMCallback:Error: Null Mbuf Ptr\n", 
		    0, 0, 0, 0, 0, 0);
		return;
	    }
#endif
	    
	    /* convert and preload the next entry 
	     * (the conversion function takes care about null pointers which
	     * are used to mark the end of the loop)
	     */
	    nextQEntry = *(++qEntryPtr);
	    nextMbufPtr = ixEthAccEntryFromQConvert(nextQEntry,
		      IX_ETHNPE_QM_Q_TXENET_ADDR_MASK);

	    /* convert fields from mbuf header */
	    ixEthAccMbufFromTxQ(mbufPtr);

	    /*
	     * Get NPE id from message, then convert to portId.
	     */
	    portId = ((IX_ETHNPE_QM_Q_TXENETDONE_NPEID_MASK &
		       qEntry) >> 
		      IX_ETHNPE_QM_Q_FIELD_NPEID_R);
	    portId = IX_ETH_ACC_QM_NPE_ID_TO_PORT_ID(portId);

#ifndef NDEBUG
	    /* Prudent to at least check the port is within range */
	    if (portId >= IX_ETH_ACC_NUMBER_OF_PORTS)
	    {
		ixEthAccDataStats.unexpectedError++;
		IX_ETH_ACC_FATAL_LOG(
		    "IXETHACC:ixEthTxFrameDoneQMCallback: Illegal port: %u\n", 
		    (UINT32)portId, 0, 0, 0, 0, 0);
		return;
	    }
#endif

	    TX_STATS_INC(portId,txDoneClientCallback);
            
	    /* 
	     * Call user level callback.
	     */
	    ixEthAccPortData[portId].ixEthAccTxData.txBufferDoneCallbackFn(
		ixEthAccPortData[portId].ixEthAccTxData.txCallbackTag, 
		mbufPtr);
	}
    } while( txDoneQReadStatus == IX_SUCCESS );
}

void 
ixEthAccRecoverRxFreeQBuffers(IxEthAccPortId portId )
{
    IX_STATUS   qStatus;
    /*
     * Design note : entries are read in a static buffer, This buffer contains
     * an extra entyry (which is zeroed by the compiler), so the loop will
     * always terminate on a null entry, whatever the result of Burst read is.
     */
    static UINT32 rxFreeQEntry[IX_QMGR_Q_SIZE128 + 1];
    UINT32 *qEntryPtr;
    IX_MBUF    *buffer;
    UINT32 qEntry;

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: We should not have any RxFree Q Buffers.\n",(INT32)portId,0,0,0,0,0);
        return ; 
    }
    
    /* disable rx free Q notifications */
    ixQMgrNotificationDisable(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId));

    /*
     * Drain Rx Replenish Q.
     */
    do
    {
	qEntryPtr = rxFreeQEntry;
	qStatus = ixQMgrQBurstRead(IX_ETH_ACC_PORT_TO_RX_FREE_Q_ID(portId), 
		   IX_QMGR_Q_SIZE128,
		   qEntryPtr);
	/*
	 * Recover Entries in Rx Replenish Q.
	 */
	while (*qEntryPtr != 0)
	{
	    qEntry = *qEntryPtr++;
	    buffer = ixEthAccEntryFromQConvert(qEntry,
			 IX_ETHNPE_QM_Q_RXENET_ADDR_MASK);
	    ixEthAccMbufFromSwRxQ(buffer);
	    
	    RX_STATS_INC(portId,rxDuringDisable);
	    ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn(
	         ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag, 
	         buffer,
  	         IX_ETH_DB_NUMBER_OF_PORTS /* port not found */);
	}
    }
    while (qStatus == IX_SUCCESS);
}

void  
ixEthAccRecoverTxSubmittedQBuffers(IxEthAccPortId portId)
{
    IX_STATUS   qStatus = IX_SUCCESS;
    /*
     * Design note : entries are read in a static buffer, This buffer contains
     * an extra entyry (which is zeroed by the compiler), so the loop will
     * always terminate on a null entry, whatever the result of Burst read is.
     */
    static UINT32 txQEntry[IX_QMGR_Q_SIZE128 + 1];
    UINT32 *qEntryPtr;
    UINT32     qEntry;
    IX_MBUF    *buffer;

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: We should not have any Tx submitted Q Buffers.\n",(INT32)portId,0,0,0,0,0);
        return; 
    }

    /* disable tx Q notifications */
    ixQMgrNotificationDisable(IX_ETH_ACC_PORT_TO_TX_Q_ID(portId));

    /*
     * Drain tx Q.
     */
    do
    {
	qEntryPtr = txQEntry;
	qStatus = ixQMgrQBurstRead(IX_ETH_ACC_PORT_TO_TX_Q_ID(portId),
		   IX_QMGR_Q_SIZE128,
		   qEntryPtr);

	/*
	 * Recover Entries in Rx Replenish Q.
	 */
	while (*qEntryPtr != 0)
	{
	    qEntry = *qEntryPtr++;
	    
            /* convert fields from mbuf header */
            buffer = ixEthAccEntryFromQConvert(qEntry, 
			 IX_ETHNPE_QM_Q_TXENET_ADDR_MASK);
            ixEthAccMbufFromTxQ(buffer);
            
            TX_STATS_INC(portId,txDoneDuringDisable);
            
            ixEthAccPortData[portId].ixEthAccTxData.txBufferDoneCallbackFn(
                ixEthAccPortData[portId].ixEthAccTxData.txCallbackTag, 
                buffer);
	}
    }
    while (qStatus == IX_SUCCESS);
}

void  
ixEthAccRecoverSwQBuffers(IxEthAccPortId portId)
{
    IxEthAccTxPriority priority;
    IX_MBUF *mbuf;

    if (IX_ETH_ACC_SUCCESS != ixEthAccSingleEthNpeCheck(portId))
    {
        IX_ETH_ACC_WARNING_LOG("EthAcc: Unavailable Eth %d: We should not have Sw Q Buffers.\n",(INT32)portId,0,0,0,0,0);
        return; 
    }

    /* tx queues */
    for (priority = IX_ETH_ACC_TX_PRIORITY_0 ; 
	 priority <= IX_ETH_ACC_TX_PRIORITY_7 ;
	 priority++)
    {
        do
        {
	    /* get sw q entry */
            IX_ETH_ACC_DATAPLANE_REMOVE_MBUF_FROM_Q_HEAD(
		 ixEthAccPortData[portId].ixEthAccTxData.txQ[priority], mbuf);

            if (mbuf != NULL)
            {
		TX_STATS_INC(portId,txPriority[priority]);
                TX_STATS_INC(portId,txDoneSwQDuringDisable);
		
		/* convert to mbuf header fields to virtual address */
		ixEthAccMbufFromTxQ(mbuf);

	        ixEthAccPortData[portId].ixEthAccTxData.txBufferDoneCallbackFn(
                    ixEthAccPortData[portId].ixEthAccTxData.txCallbackTag, 
                    mbuf);
            }
        } 
        while (mbuf != NULL);
    }

    /* rx queue */
    do
    {
	/* get sw q entry */
        IX_ETH_ACC_DATAPLANE_REMOVE_MBUF_FROM_Q_HEAD(ixEthAccPortData[portId].
	     ixEthAccRxData.freeBufferList, mbuf);

        if (mbuf != NULL)
        {
	    /* convert to mbuf header fields to virtual address */
	    ixEthAccMbufFromSwRxQ(mbuf);

	    RX_STATS_INC(portId,rxSwQDuringDisable);

            ixEthAccPortData[portId].ixEthAccRxData.rxCallbackFn(
                ixEthAccPortData[portId].ixEthAccRxData.rxCallbackTag, 
                mbuf,
                IX_ETH_DB_NUMBER_OF_PORTS /* port not found */);

        } 
    }
    while (mbuf != NULL);
}


IX_ETH_ACC_PUBLIC
void ixEthAccDataPlaneShow(void)
{
    UINT32 numTx0Entries;
    UINT32 numTx1Entries;
    UINT32 numTxDoneEntries;
    UINT32 numRxEntries;
    UINT32 numRxFree0Entries;
    UINT32 numRxFree1Entries;
    UINT32 portId;
#ifndef NDEBUG
    UINT32 priority;
    UINT32 numBuffersInRx=0;
    UINT32 numBuffersInTx=0;
    UINT32 numBuffersInSwQ=0;
    UINT32 totalBuffers=0;
    UINT32 rxFreeCallbackCounter = 0;
    UINT32 txCallbackCounter = 0;
#endif
    UINT32 key;

    /* snapshot of stats */
    ixEthAccTxDataStats tx[IX_ETH_ACC_NUMBER_OF_PORTS];
    ixEthAccRxDataStats rx[IX_ETH_ACC_NUMBER_OF_PORTS];
    IxEthAccDataPlaneStats stats;

    if (!IX_ETH_ACC_IS_SERVICE_INITIALIZED())
    {
	return;
    }

    /* get a reliable snapshot */
    key = ixOsServIntLock();

    numTx0Entries = 0;
    ixQMgrQNumEntriesGet(IX_ETH_ACC_TX_FRAME_ENET0_Q, &numTx0Entries);
    numTx1Entries = 0;
    ixQMgrQNumEntriesGet(IX_ETH_ACC_TX_FRAME_ENET1_Q, &numTx1Entries);
    numTxDoneEntries = 0;
    ixQMgrQNumEntriesGet( IX_ETH_ACC_TX_FRAME_DONE_ETH_Q, &numTxDoneEntries);
    numRxEntries = 0;
    ixQMgrQNumEntriesGet(IX_ETH_ACC_RX_FRAME_ETH_Q, &numRxEntries);
    numRxFree0Entries = 0;
    ixQMgrQNumEntriesGet(IX_ETH_ACC_RX_FREE_BUFF_ENET0_Q, &numRxFree0Entries);
    numRxFree1Entries = 0;
    ixQMgrQNumEntriesGet(IX_ETH_ACC_RX_FREE_BUFF_ENET1_Q, &numRxFree1Entries);

    for(portId=IX_ETH_PORT_1; portId<=IX_ETH_PORT_2; portId++)
    {
	memcpy(&tx[portId], 
	       &ixEthAccPortData[portId].ixEthAccTxData.stats,
	       sizeof(tx[portId]));
	memcpy(&rx[portId], 
	       &ixEthAccPortData[portId].ixEthAccRxData.stats,
	       sizeof(rx[portId]));
    }
    memcpy(&stats, &ixEthAccDataStats, sizeof(stats));
    
    ixOsServIntUnlock(key);

#ifdef NDEBUG
    printf("Detailed statistics collection not supported in this load\n");
#endif

    /* print snapshot */
    for(portId=IX_ETH_PORT_1; portId<=IX_ETH_PORT_2; portId++)
    {
       if ((ixFeatureCtrlProductIdRead() & IX_FEATURE_CTRL_SILICON_STEPPING_MASK) == 
            IX_FEATURE_CTRL_SILICON_TYPE_B0) 
       {  
            if ((IX_ETH_PORT_1 == portId) && 
                (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH0) ==  
                 IX_FEATURE_CTRL_COMPONENT_DISABLED)) 
            {							      
               continue ;
            } 
            if ((IX_ETH_PORT_2 == portId) && 
                (ixFeatureCtrlComponentCheck(IX_FEATURECTRL_ETH1) == 
                 IX_FEATURE_CTRL_COMPONENT_DISABLED)) 
            { 
	       continue ;
            } 
        }        

	printf("PORT %u --------------------------------\n",
	       portId);
#ifndef NDEBUG 
	printf("Tx Done Frames                : %u\n", 
	       tx[portId].txDoneClientCallback + 
	       tx[portId].txDoneSwQDuringDisable + 
	       tx[portId].txDoneDuringDisable);
	printf("Tx Frames                     : %u\n", 
	       tx[portId].txQOK + tx[portId].txQDelayed);
	printf("Tx H/W Q Added OK             : %u\n", 
	       tx[portId].txQOK);
	printf("Tx H/W Q Delayed              : %u\n", 
	       tx[portId].txQDelayed);
	printf("Tx From S/W Q Added OK        : %u\n", 
	       tx[portId].txFromSwQOK);
	printf("Tx From S/W Q Delayed         : %u\n", 
	       tx[portId].txFromSwQDelayed);
	printf("Tx Overflow                   : %u\n", 
	       tx[portId].txOverflow);
	printf("Tx Mutual Lock                : %u\n", 
	       tx[portId].txLock);
	printf("Tx Late Ntf Enabled           : %u\n", 
	       tx[portId].txLateNotificationEnabled);
	printf("Tx Low Thresh CB              : %u\n", 
	       tx[portId].txLowThreshCallback);
	printf("Tx Done from H/W Q (Disable)  : %u\n", 
	       tx[portId].txDoneDuringDisable);
	printf("Tx Done from S/W Q (Disable)  : %u\n", 
	       tx[portId].txDoneSwQDuringDisable);
	for (priority = IX_ETH_ACC_TX_PRIORITY_0;
	     priority <= IX_ETH_ACC_TX_PRIORITY_7;
	     priority++)
	{
	    if (tx[portId].txPriority[priority])
	    {
		printf("Tx Priority %u                 : %u\n", 
		       priority,
		       tx[portId].txPriority[priority]);
	    }
	}
#endif
	printf("Tx unexpected errors          : %u (should be 0)\n", 
	       tx[portId].txUnexpectedError);	
	
#ifndef NDEBUG
	printf("Rx Frames                     : %u\n", 
	       rx[portId].rxFrameClientCallback + 
	       rx[portId].rxSwQDuringDisable+ 
	       rx[portId].rxDuringDisable);
	printf("Rx Free Replenish             : %u\n", 
	       rx[portId].rxFreeRepOK + rx[portId].rxFreeRepDelayed);
	printf("Rx Free H/W Q Added OK        : %u\n", 
	       rx[portId].rxFreeRepOK);
	printf("Rx Free H/W Q Delayed         : %u\n", 
	       rx[portId].rxFreeRepDelayed);
	printf("Rx Free From S/W Q Added OK   : %u\n", 
	       rx[portId].rxFreeRepFromSwQOK);
	printf("Rx Free From S/W Q Delayed    : %u\n", 
	       rx[portId].rxFreeRepFromSwQDelayed);
	printf("Rx Free Overflow              : %u\n", 
	       rx[portId].rxFreeOverflow);
	printf("Rx Free Mutual Lock           : %u\n", 
	       rx[portId].rxFreeLock);
	printf("Rx Free Late Ntf Enabled      : %u\n", 
	       rx[portId].rxFreeLateNotificationEnabled);
	printf("Rx Free Low CB                : %u\n",
	       rx[portId].rxFreeLowCallback);
	printf("Rx From H/W Q (Disable)       : %u\n", 
	       rx[portId].rxDuringDisable);
	printf("Rx From S/W Q (Disable)       : %u\n", 
	       rx[portId].rxSwQDuringDisable);
#endif
	printf("Rx unexpected errors          : %u (should be 0)\n", 
	       rx[portId].rxUnexpectedError);	
	
#ifndef NDEBUG
	numBuffersInTx = tx[portId].txQOK + 
	    tx[portId].txQDelayed -
	    tx[portId].txDoneClientCallback - 
	    tx[portId].txDoneSwQDuringDisable - 
	    tx[portId].txDoneDuringDisable;

	printf("# Tx Buffers currently for transmission : %u\n", 
	       numBuffersInTx);
	
	numBuffersInRx = rx[portId].rxFreeRepOK + 
	    rx[portId].rxFreeRepDelayed - 
	    rx[portId].rxFrameClientCallback - 
	    rx[portId].rxSwQDuringDisable - 
	    rx[portId].rxDuringDisable;

	printf("# Rx Buffers currently for reception    : %u\n", 
	       numBuffersInRx);	

	totalBuffers += numBuffersInRx + numBuffersInTx;
#endif
    }
    
    printf("---------------------------------------\n");

#ifndef NDEBUG
    printf("\n");
    printf("Mbufs :\n");   
    printf("Tx Unchained mbufs            : %u\n", 
	   stats.unchainedTxMBufs);
    printf("Tx Chained bufs               : %u\n", 
	   stats.chainedTxMBufs);
    printf("TxDone Unchained mbufs        : %u\n", 
	   stats.unchainedTxDoneMBufs);
    printf("TxDone Chained bufs           : %u\n", 
	   stats.chainedTxDoneMBufs);
    printf("RxFree Unchained mbufs        : %u\n", 
	   stats.unchainedRxFreeMBufs);
    printf("RxFree Chained bufs           : %u\n", 
	   stats.chainedRxFreeMBufs);
    printf("Rx Unchained mbufs            : %u\n", 
	   stats.unchainedRxMBufs);
    printf("Rx Chained bufs               : %u\n", 
	   stats.chainedRxMBufs);

    printf("\n");
    printf("Software queue usage :\n");   
    printf("Buffers added to S/W Q        : %u\n", 
	   stats.addToSwQ);
    printf("Buffers removed from S/W Q    : %u\n", 
	   stats.removeFromSwQ);

    printf("\n");
    printf("Hardware queues callbacks :\n");   

    for(portId=IX_ETH_PORT_1; portId<=IX_ETH_PORT_2; portId++)
    {
	rxFreeCallbackCounter += rx[portId].rxFreeLowCallback;
	txCallbackCounter += tx[portId].txLowThreshCallback;
    }
    printf("Tx Done QM Callback invoked   : %u\n", 
	   stats.txDoneCallbackCounter);
    printf("Tx QM Callback invoked        : %u\n", 
	   txCallbackCounter);
    printf("Rx QM Callback invoked        : %u\n", 
	   stats.rxCallbackCounter);
    printf("Rx Free QM Callback invoked   : %u\n", 
	   rxFreeCallbackCounter);
#endif
    printf("Unexpected errors in CB       : %u (should be 0)\n", 
	   stats.unexpectedError);
    printf("\n");

    printf("Hardware queues levels :\n");   
    printf("Transmit Port 1 Q             : %u \n",numTx0Entries);
    printf("Transmit Port 2 Q             : %u \n",numTx1Entries);
    printf("Transmit Done Q               : %u \n",numTxDoneEntries);
    printf("Receive Q                     : %u \n",numRxEntries);
    printf("Receive Free Port 1 Q         : %u \n",numRxFree0Entries);
    printf("Receive Free Port 2 Q         : %u \n",numRxFree1Entries);

#ifndef NDEBUG
    printf("\n");
    printf("# Total Buffers accounted for : %u\n", 
	   totalBuffers);

    numBuffersInSwQ = ixEthAccDataStats.addToSwQ - 
	ixEthAccDataStats.removeFromSwQ;

    printf("    Buffers in S/W Qs         : %u\n", 
	   numBuffersInSwQ);
    printf("    Buffers in H/W Qs or NPEs : %u\n", 
	   totalBuffers - numBuffersInSwQ);
#endif
}





