/*
 * File Version: $Revision: 1.1.1.1 $
 * 
 * -- Intel Copyright Notice --
 * 
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * -- End Intel Copyright Notice --
*/
#include <linux/config.h>
#include <linux/module.h>
#include <linux/kernel.h>
#include <linux/pci.h>
#include <linux/init.h>
#include <linux/ioport.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>
#include <linux/delay.h>
#include <linux/ethtool.h>
#include <linux/mii.h>
#include <asm/io.h>

#include <IxUSBRNDIS.h>
#include "IxUSBRNDISEnd.h"
/* Hardcoded parameters */

#define MODULE_NAME "ixp_usb"
#define MODULE_VERSION "0.0.1"

MODULE_DESCRIPTION("IXP425 USB Rndis driver");

/* Private device data */
typedef struct {
    spinlock_t lock;
    /* XXX Check Intel holds this for us? */
    struct net_device_stats stats;
} priv_data_t;

/* This callback is called when new packet received from MAC
 * and ready to be transfered up-stack
 */
static void rx_cb(struct net_device *dev, struct sk_buff *buffer)
{
    priv_data_t *priv = dev->priv;
    int irq_state;

    spin_lock_irqsave(&priv->lock, irq_state);
    priv->stats.rx_packets++;
    priv->stats.rx_bytes += buffer->len;
    spin_unlock_irqrestore(&priv->lock, irq_state);

    buffer->dev = dev;
    buffer->protocol = eth_type_trans(buffer, dev);
    netif_rx(buffer);
}

void rndisInt(void *_dev, int _packet)
{
    rx_cb(_dev, (struct sk_buff *)_packet);
}

/* Set promiscuous/multycase mode for the MAC */
static void dev_set_multicast_list(struct net_device *dev)
{
    printk("Don't know how to enter promiscuous/multicast mode on '%s'\n",
	dev->name);

    /* XXX Should check if->flags IFF_PROMISC | IFF_ALLMULTI or
     * dev->mc_count in order to setup the device
     * Current Intel codelet (Beta) does not support this.
     * CHECK again when new versions are released.
     */
}

/* Open the device.
 * Request resources and start interrupts
 */
static int do_dev_open(struct net_device *dev)
{
    /* TODO: Reset controler
     * TODO: Config RNDIS (Promisc/Mcast add list)
     * Depends on Intel codelet implementation. Current version does not
     * support.
     */

    /* load the RNDIS layer */
    ixUSBRNDISInit();
    ixUSBRNDISLayerInit(dev);

    netif_start_queue(dev);

    return 0;
}

/* Close the device.
 * Free resources acquired in dev_start
 */
static int do_dev_stop(struct net_device *dev)
{
    netif_stop_queue(dev);
    return 0;
}

/* this is called by kernel to transmit packet */
static int dev_hard_start_xmit(struct sk_buff *skb, struct net_device *dev)
{
    priv_data_t *priv = dev->priv;
    int irq_state;

    spin_lock_irqsave(&priv->lock, irq_state);
    priv->stats.tx_packets++;
    priv->stats.tx_bytes += skb->len;
    spin_unlock_irqrestore(&priv->lock, irq_state);

    ixUSBRNDISSendDataPacket(skb);
    dev_kfree_skb(skb);

    return 0;
}

static struct net_device_stats *dev_get_stats(struct net_device *dev)
{
    priv_data_t *priv = dev->priv;
    return &priv->stats;
}

/* Initialize device structs.
 * Resource allocation is deffered until do_dev_open
 */
static int __devinit do_init(struct net_device *dev)
{
    priv_data_t *priv;

    SET_MODULE_OWNER(dev);

    printk("Loading RNDIS...\n");

    ether_setup(dev);

    /* allocate and initialize priv struct */
    priv = dev->priv = kmalloc(sizeof(priv_data_t), GFP_KERNEL);
    if (dev->priv == NULL)
    {
	unregister_netdevice(dev);
	return -ENOMEM;
    }
    memset(dev->priv, 0, sizeof(priv_data_t));

    /* TODO: Set MAC */


    /* TODO: Set device flags */

    /* fill in dev struct callbacks */
    dev->open = do_dev_open;
    dev->stop = do_dev_stop;
    dev->hard_start_xmit = dev_hard_start_xmit;
    dev->get_stats = dev_get_stats;
    dev->set_multicast_list = dev_set_multicast_list;

    /* Fill in priv data */
    priv->lock = SPIN_LOCK_UNLOCKED;

    printk("Done loading RNDIS\n");

    return 0;
}

/* Module initialization and cleanup */

static struct net_device usb_dev;

static int __init do_module_init(void)
{
    int res;

    printk("%s address: %p\n", __FUNCTION__, do_module_init);
    usb_dev.init = do_init;
    sprintf(usb_dev.name, "usb%d", 0);
    res = register_netdev(&usb_dev);
    if (res)
	printk("Failed to register USB netdev. res = %d\n", res);
    
    return res;
}

static void __exit do_module_exit(void)
{
    if (usb_dev.priv != NULL)
    {
        ixUSBRNDISUnload();
      
	unregister_netdev(&usb_dev);
	kfree(usb_dev.priv);
	usb_dev.priv = NULL;
    }
}

module_init(do_module_init);
module_exit(do_module_exit);

