/** @file
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2002-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
*/
/**
 * @brief USB specific functionality (enumeration) is implemented here
 */

#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include <net/mbuf.h>
#include <netBufLib.h>
#include <logLib.h>

#include "IxOsServices.h"
#include "IxOsServicesMemMap.h"

#include "usb.h"
#include "usbdeviceparam.h"
#include "usbstd.h"

#include "IxUSBRNDIS.h"
#include "IxUSBRNDISDescriptors.c"

/* imported data */
extern IX_USB_MBLK *rndisResponse;

/* data */
USBDevice usbDevice;

/* log control */
#ifndef NDEBUG
int rndisEnumTrace        = 1;
int rndisEnumVerboseTrace = 1;
#else
int rndisEnumTrace        = 0;
int rndisEnumVerboseTrace = 0;
#endif

/* prototypes */
PRIVATE IX_STATUS ixUSBRNDISProcessGetDescriptorRequest(USBDevice *device, UINT32 type, UINT32 index1, UINT32 languageID, UINT32 len);
PRIVATE IX_STATUS ixUSBRNDISProcessSetConfigurationRequest(USBDevice *device, UINT32 configuration);
PRIVATE IX_STATUS ixUSBRNDISProcessSetInterfaceRequest(USBDevice *device, UINT32 requestedInterface, UINT32 alternateInterface);
PRIVATE void ixUSBRNDISSetupCallback(USBDevice *device, const char *packet);
PRIVATE void ixUSBRNDISReceiveCallback(USBDevice *device, UINT16 endpoint, IX_USB_MBLK *receiveBuffer);
PRIVATE void ixUSBRNDISEventCallback(USBDevice *device, USBEventSet eventSet);
PRIVATE IX_STATUS ixUSBRNDISErrorExit(const char *msg);

/* external prototypes */
IX_USB_MBLK* ixUSBBufferAlloc(size_t size);

PUBLIC IX_USB_MBLK*
ixUSBRNDISCreateMBuf(UINT8 *buffer, UINT32 len)
{
    IX_USB_MBLK *mbuf = ixUSBBufferAlloc(len);

    /* debug */
    ENUM_VERBOSE_TRACE("=> allocated %d bytes into IX_USB_MBLK %p [ixUSBRNDISCreateMBuf]\n", len, mbuf);

    if (mbuf != NULL)
    {
        memcpy(IX_USB_MBLK_DATA(mbuf), buffer, len);

        IX_USB_MBLK_LEN(mbuf) = len;
    }

    return mbuf;
}

PRIVATE IX_STATUS
ixUSBRNDISProcessGetDescriptorRequest(USBDevice *device, UINT32 type, UINT32 index1, UINT32 languageID, UINT32 len)
{
    BOOL result;
    UINT32 transferLen = 0 ;
    IX_USB_MBLK *buf;
    UINT8 *usbDescriptor = NULL;

    switch (type)
    {
        case USB_DEVICE_DESCRIPTOR:
            transferLen   = MIN(len, sizeof(ixUSBRNDISDeviceDescriptor));
            usbDescriptor = ixUSBRNDISDeviceDescriptor;

            ENUM_VERBOSE_TRACE("USB: sending device descriptor, %d bytes\n", transferLen );

            break;
        case USB_CONFIGURATION_DESCRIPTOR:
            transferLen   = MIN(len, sizeof(ixUSBRNDISConfigurationDescriptor));
            usbDescriptor = ixUSBRNDISConfigurationDescriptor;

            ENUM_VERBOSE_TRACE("USB: sending configuration descriptor, %d bytes\n", transferLen);

            break;
        case USB_STRING_DESCRIPTOR:
            switch (index1)
            {
                case 0:
                    transferLen         = MIN(len, sizeof(ixUSBRNDISLangIdDescriptor));
                    usbDescriptor       = ixUSBRNDISLangIdDescriptor;
                    break;
                case 1:
                    transferLen         = MIN(len, sizeof(ixUSBRNDISManufacturerStringDescriptor));
                    usbDescriptor       = ixUSBRNDISManufacturerStringDescriptor;
                    break;
                case 2:
                    transferLen         = MIN(len, sizeof(ixUSBRNDISProductStringDescriptor));
                    usbDescriptor       = ixUSBRNDISProductStringDescriptor;
                    break;
                case 3:
                    transferLen         = MIN(len, sizeof(ixUSBRNDISSerialNumberStringDescriptor));
                    usbDescriptor       = ixUSBRNDISSerialNumberStringDescriptor;
                    break;
                case 4:
                    transferLen         = MIN(len, sizeof(ixUSBRNDISConfigurationStringDescriptor));
                    usbDescriptor       = ixUSBRNDISConfigurationStringDescriptor;
                    break;
                default:
                    ; /* unknown */
            }

            break;

        default:
            ; /* unsupported type */
    }

    if (usbDescriptor != NULL)
    {
        buf    = ixUSBRNDISCreateMBuf(usbDescriptor, transferLen);
        result = ixUSBBufferSubmit(&usbDevice, RNDIS_CONTROL_ENDPOINT, buf);

        if (result != IX_SUCCESS && buf != NULL)
        {
            RNDIS_FREE(buf);
        }
    }
    else
    {
        result = IX_FAIL;
    }

    return result;
}

PRIVATE IX_STATUS
ixUSBRNDISProcessSetConfigurationRequest(USBDevice *device, UINT32 configuration)
{
    ENUM_VERBOSE_TRACE("USB: active configuration set to %d\n", configuration);

    return IX_SUCCESS;
}

PRIVATE IX_STATUS
ixUSBRNDISProcessSetInterfaceRequest(USBDevice *device, UINT32 requestedInterface, UINT32 alternateInterface)
{
    ENUM_VERBOSE_TRACE("USB: active interface set to %d (alternate setting %d)\n", requestedInterface, alternateInterface);

    return IX_SUCCESS;
}

PRIVATE void
ixUSBRNDISSetupCallback(USBDevice *device, const char *packet)
{
    USBSetupPacket *setupPacket = (USBSetupPacket *) packet;
    IX_STATUS result = IX_FAIL;

    ENUM_VERBOSE_TRACE("USB: setup [bmRequestType (0x%02X) bRequest: (0x%02X) wValue: (0x%04X) wIndex: (0x%04X) wLength: (0x%04X)]\n", setupPacket->bmRequestType, setupPacket->bRequest, setupPacket->wValue, setupPacket->wIndex, setupPacket->wLength);

    ENUM_VERBOSE_TRACE("req type 0x%04X, mask 0x%04X, result 0x%04X, standard is 0x%04X\n", setupPacket->bmRequestType, USB_REQ_TYPE_MASK, (setupPacket->bmRequestType & USB_REQ_TYPE_MASK), USB_REQ_TYPE_STANDARD);
    
    if ((setupPacket->bmRequestType & USB_REQ_TYPE_MASK) == USB_REQ_TYPE_STANDARD)
    {
        switch (setupPacket->bRequest)
        {
            case GET_DESCRIPTOR_REQUEST:
                ENUM_VERBOSE_TRACE("USB: processing GET_DESCRIPTOR request..\n");
                result = ixUSBRNDISProcessGetDescriptorRequest(device, setupPacket->wValue >> 8, setupPacket->wValue & 0xFF, setupPacket->wIndex, setupPacket->wLength);
                break;
            case SET_CONFIGURATION_REQUEST:
                ENUM_VERBOSE_TRACE("USB: processing SET_CONFIGURATION request..\n");
                result = ixUSBRNDISProcessSetConfigurationRequest(device, setupPacket->wValue);
                break;
            case SET_DESCRIPTOR_REQUEST:
                ENUM_VERBOSE_TRACE("USB: processing SET_DESCRIPTOR request.. \n");
                result = IX_FAIL; /* not supported */
                break;
            case SET_INTERFACE_REQUEST:
                ENUM_VERBOSE_TRACE("USB: processing SET_INTERFACE request..\n");
                result = ixUSBRNDISProcessSetInterfaceRequest(device, setupPacket->wIndex, setupPacket->wValue);
                break;
            case SYNCH_FRAME_REQUEST:
                ENUM_VERBOSE_TRACE("USB: processing SYNCH_FRAME request..\n");
                result = IX_FAIL; /* not supported */
                break;
            default:
                ;
        }
    }
    else if (setupPacket->bmRequestType & USB_REQ_TYPE_CLASS)
    {
        ENUM_VERBOSE_TRACE("USB: processing class specific request...\n");

        switch (setupPacket->bRequest)
        {
            case RNDIS_SEND_ENCAPSULATED_COMMAND:
                RNDIS_VERBOSE_TRACE("USB: processing RNDIS_SEND_ENCAPSULATED_COMMAND..\n");

                result = ixUSBRNDISSignalEncapsulatedCommand();
                break;
            case RNDIS_GET_ENCAPSULATED_RESPONSE:
                RNDIS_VERBOSE_TRACE("USB: processing RNDIS_GET_ENCAPSULATED_RESPONSE..\n");
    
                /* ixUSBRNDISDumpPacket(); */

                result = ixUSBBufferSubmit(&usbDevice, RNDIS_CONTROL_ENDPOINT, rndisResponse);

                if (result != IX_SUCCESS && rndisResponse != NULL)
                {
                    RNDIS_FREE(rndisResponse);
                }

                rndisResponse = NULL;

                break;
            default:
                result = IX_FAIL;
        }
    }
    else
    {
        ENUM_TRACE("USB: mangled setup data\n");

        result = IX_FAIL;
    }

    if (result != IX_SUCCESS)
    {
        ENUM_TRACE("USB: an error occurred, stalling endpoint 0\n");

        /* stall control endpoint on protocol error */
        ixUSBEndpointStall(device, RNDIS_CONTROL_ENDPOINT, TRUE);
    }
}

PRIVATE void
ixUSBRNDISReceiveCallback(USBDevice *device, UINT16 endpoint, IX_USB_MBLK *receiveBuffer)
{
    ENUM_VERBOSE_TRACE("USB: packet received on endpoint %d (%d bytes)\n", endpoint, IX_USB_MBLK_LEN(receiveBuffer));

    if (endpoint == RNDIS_CONTROL_ENDPOINT)
    {
        /* encapsulated command */
        ixUSBRNDISProcessEncapsulatedCommand(receiveBuffer);
    }
    else
    {
        RNDIS_VERBOSE_TRACE("USB: RNDIS data message, forwarding to ixUSBRNDISProcessDataPacket\n");

        /* RNDIS data packet */
        ixUSBRNDISProcessDataPacket(receiveBuffer);
    }
}

PRIVATE void 
ixUSBRNDISEventCallback(USBDevice *device, USBEventSet eventSet)
{
    if (eventSet & USB_RESET)
    {
        ENUM_TRACE("USB: reset received\n");
    }
    
    if (eventSet & USB_SUSPEND)
    {
        ENUM_TRACE("USB: suspend received\n");
    }
    
    if (eventSet & USB_RESUME)
    {
        ENUM_TRACE("USB: resume received\n");
    }
}

PRIVATE IX_STATUS
ixUSBRNDISErrorExit(const char *msg)
{
    ixOsServLog(LOG_ERROR, (char *)msg, 0, 0, 0, 0, 0, 0);
    ixOsServLog(LOG_ERROR, ": ", 0, 0, 0, 0, 0, 0);
    ixOsServLog(LOG_ERROR, (char *)ixUSBErrorStringGet(usbDevice.lastError), 0, 0, 0, 0, 0, 0);
    ixOsServLog(LOG_ERROR, "\n", 0, 0, 0, 0, 0, 0);

    return IX_FAIL;
}

PUBLIC IX_STATUS
ixUSBRNDISInit(void)
{
    /* init hardware specific fields */
    usbDevice.interruptLevel = UDC_IRQ;
    
    /* Rx and Tx sequencing are enabled */
    usbDevice.flags = 1;

    /* init device */
    if (ixUSBDriverInit(&usbDevice) != IX_SUCCESS)
    {
        return ixUSBRNDISErrorExit("Device init failed");
    }

    /* set setup callback */
    if (ixUSBSetupCallbackRegister(&usbDevice, ixUSBRNDISSetupCallback) != IX_SUCCESS)
    {
        return ixUSBRNDISErrorExit("Could not register setup callback");
    }

    /* set receive callback */
    if (ixUSBReceiveCallbackRegister(&usbDevice, ixUSBRNDISReceiveCallback) != IX_SUCCESS)
    {
        return ixUSBRNDISErrorExit("Could not register receive callback");
    }

    /* set event callback */
    if (ixUSBEventCallbackRegister(&usbDevice, ixUSBRNDISEventCallback, USB_DEVICE_EVENTS) != IX_SUCCESS)
    {
        return ixUSBRNDISErrorExit("Could not register event callback");
    }

    /* enable device */
    if (ixUSBDeviceEnable(&usbDevice, TRUE) != IX_SUCCESS)
    {
        return ixUSBRNDISErrorExit("Device enabling failure");
    }

    printf("USB: RNDIS device initialized\n");

    return IX_SUCCESS;
}

PUBLIC void
ixUSBRNDISUnload(void)
{
  ixOsServIntUnbind(usbDevice.interruptLevel);
  IX_OSSERV_MEM_UNMAP(usbDevice.baseIOAddress);
}

/*
 * Message trace function
 */
int 
enum_verbose_log_trace(char *_fmt_,...) 
{  
    char _buf_[1024];  
    int retStatus = 0; 
    va_list vap;  
 
    va_start(vap, _fmt_);  
    retStatus = vsprintf(_buf_,_fmt_,vap); 
    if (retStatus > 0) {   
#ifdef __vxworks 
        logMsg(_buf_,0,0,0,0,0,0);  
#else 
        printf(_buf_); 
#endif 
    }  
    va_end(vap);  
    return retStatus; 
}
