/**
 * @file IxAdslUtil.h
 *
 * @date 4-Jun-2003
 *
 * @brief  Header file for Hardware interface functions in the IxAdsl component
 *
 *
 * Design Notes:  this Adsl Driver only Supports Utopia Level 1
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2000-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */

#ifndef IXADSLUTIL_H
#define	IXADSLUTIL_H
#include "IxTypes.h"
#include "IxAdsl.h"
#include <IxOsServicesMemMap.h> 
#include <IxOsServicesMemAccess.h> 
#include <IxOsServices.h>
#define	RECTRLE_BUFFSIZE 128  /**< Default size for CtrlE interface arrays */
#define IX_ADSL_BOOT_TIMEOUT_COUNTER 1000     /**< Number or times to poll for bootcode completion timeout */
#define IX_ADSL_BOOT_TIMEOUT_POLL_INTERVAL 10 /**< OS Ticks */

/* ----------------------------------------------------------------------*/
/*  Externs for fast mutex locks          -------------------------------*/
/* ----------------------------------------------------------------------*/
extern IxFastMutex ixAdslCtrleAccessLock;

/* ----------------------------------------------------------------------*/
/*  Defines for parallel CtrlE Interface  -------------------------------*/
/* ----------------------------------------------------------------------*/
/**
 * @def CTRLE_READ_REG(x)
 *
 * @brief Macro to read CtrlE Addresses
 *
 * @param [in] x - Address of CtrlE buffer to read from
 *
 *
 */
#define CTRLE_READ_REG(x)    IX_OSSERV_READ_BYTE(x) 
/**
 * @def CTRLE_WRITE_REG(x)
 *
 * @brief Macro to write CtrlE Addresses
 *
 * @param [in] x - Address of CtrlE buffer to write
 * @param [in] val - Value to write to given address
 *
 */
#define CTRLE_WRITE_REG(x,val) IX_OSSERV_WRITE_BYTE(x,val) 

#define INIT_COMPLETE    0x80 /**< Used to determine when the init
                                * image has finished running and the
                                * application image can be loaded.
                                */

#define CTRLE_LOCALWRITEBUFFERSIZE    0x80 /**< Size of CtrlE put[] array */
#define CTRLE_LOCALREADBUFFERSIZE     0x80 /**< Size of CtrlE get[] array */


/* Base address of Adsl component dynamically mapped to Expansion bus CS1 during initialisation*/
#define IXP425_ADSL_BASE_ADDRESS    ixAdslBaseAddressGet()

#define  MODULE_SIZE 0x200000    /**< Space allocated for one Adsl Phy
                                      2^25 : 24 Address bits per Phy */

/**
 * @def CTRLE_READ_REG(x)
 *
 * @brief Macro to compute the base address for a given Adsl line
 *
 * @param [in] N - line number of Adsl Line
 *
 * @return none
 *
 */
#define IXP425_ADSL_LINE_ADDR(N)      (IXP425_ADSL_BASE_ADDRESS+N*MODULE_SIZE)

/* Macros to compute the CtrlE register addresses for a given Adsl line */
#define CTRLE_TX_REG_ADDR(N)          (IXP425_ADSL_LINE_ADDR(N)+(0x00))  /**< Compute Transmit Command
                                                                              register address */
#define CTRLE_RX_REG_ADDR(N)          (IXP425_ADSL_LINE_ADDR(N)+ (0x01)) /**< Compute Receive Command
                                                                              register address */
#define CTRLE_TX_COM_AV_ADDR(N)       (IXP425_ADSL_LINE_ADDR(N)+ (0x02)) /**< Compute Transmit Command
                                                                              Available register address */
#define CTRLE_RX_COM_AV_ADDR(N)       (IXP425_ADSL_LINE_ADDR(N)+ (0x03)) /**< Compute Receive Command
                                                                              Available register address */
#define CTRLE_SEMAPHORE_ADDR(N)       (IXP425_ADSL_LINE_ADDR(N)+ (0x04)) /**< Compute Semaphore
                                                                              register address */
#define CTRLE_CONFIG_DATA_ADDR(N)     (IXP425_ADSL_LINE_ADDR(N)+ (0x40)) /**< Compute config data
                                                                              register address */
#define CTRLE_MODEM_STATUS_ADDR(N)    (IXP425_ADSL_LINE_ADDR(N)+ (0x100))/**< Compute Modem status
                                                                              register address */
#define CTRLE_MODEM_AVAIL_DATA_ADDR(N)(IXP425_ADSL_LINE_ADDR(N)+ (0x101))/**< Compute Modem available data
                                                                              register address */
/* Macros to read CtrlE registers for a given Adsl line */
#define CTRLE_TX_REG(N)               (CTRLE_READ_REG(CTRLE_TX_REG_ADDR(N)))/**< Read Transmit Command
                                                                                 register */
#define CTRLE_RX_REG(N)               (CTRLE_READ_REG(CTRLE_RX_REG_ADDR(N)))/**< Read Receive Command
                                                                                 register */
#define CTRLE_TX_COM_AV(N)            (CTRLE_READ_REG(CTRLE_TX_COM_AV_ADDR(N)))/**< Read Transmit Command
                                                                                    Available register */
#define CTRLE_RX_COM_AV(N)            (CTRLE_READ_REG(CTRLE_RX_COM_AV_ADDR(N)))/**< Read Receive Command
                                                                                    Available register */
#define CTRLE_SEMAPHORE(N)            (CTRLE_READ_REG(CTRLE_SEMAPHORE_ADDR(N)))/**< Read Semaphore
                                                                                    register */

#define CTRLE_SLEEP_TIME 10       /**< Time for CtrlE delay: OS ticks */

#define CTRLE_WAIT_FOR_CMD_REG_CNT 30  /**< Countdown value for CtrlE timeout */

#define CTRLE_SLEEP      { taskDelay(CTRLE_SLEEP_TIME); }   /**< Macro for CtrlE delay */

/* ----------------------------------------------------------------------*/
/* For IXDP425 Only Softreset -------------------------------------------*/
/* ----------------------------------------------------------------------*/

#define GPIO_BASE ixAdslGpioBaseAddressGet() /**< GPIO base address */
#define TMR_BASE  ixAdslTimerBaseAddressGet() /**< TIMER base address */
#define OSTS      (TMR_BASE + 0x0)  /**< Compute Continuous TimeStamp Register addres */
#define PERIPHERAL_BUS_CLOCK (66)          /**< Bus clock speed in MHz */
#define  GPIO_PIN_MAX 15	/* 16 pins (incl. 0) */
/* GPIO pin mappings */
#define  GPIO_PIN_16 16
#define  GPIO_PIN_15 15
#define  GPIO_PIN_14 14
#define  GPIO_PIN_13 13
#define  GPIO_PIN_12 12
#define  GPIO_PIN_11 11
#define  GPIO_PIN_10 10
#define  GPIO_PIN_9  9
#define  GPIO_PIN_8  8
#define  GPIO_PIN_7  7
#define  GPIO_PIN_6  6
#define  GPIO_PIN_5  5
#define  GPIO_PIN_4  4
#define  GPIO_PIN_3  3
#define  GPIO_PIN_2  2
#define  GPIO_PIN_1  1
/* GPIO register addresses */
#define GPIO_GPOUTR      (GPIO_BASE + 0x0000)     /**< Compute GPIO_GPOUTR Address */
#define GPIO_GPOER       (GPIO_BASE + 0x0004)     /**< Compute GPIO_GPOER Address */
#define GPIO_GPINR       (GPIO_BASE + 0x0008)     /**< Compute GPIO_GPINR Address */
#define GPIO_GPISR       (GPIO_BASE + 0x000C)     /**< Compute GPIO_GPISR Address */
#define GPIO_GPIT1R      (GPIO_BASE + 0x0010)     /**< Compute GPIO_GPIT1R Address */
#define GPIO_GPIT2R      (GPIO_BASE + 0x0014)     /**< Compute GPIO_GPIT2R Address */
#define GPIO_GPCLKR      (GPIO_BASE + 0x0018)     /**< Compute GPIO_GPCLKR Address */
#define GPIO_GPDBSELR    (GPIO_BASE + 0x001C)     /**< Compute GPIO_GPDBSELR Address */
/* GPIO pin types */
#define  GPIO_OUT 0x1
#define  GPIO_IN  0x2
/* GPIO interrupt types */
#define  GPIO_ACTIVE_HIGH 0x4		/* Default */
#define  GPIO_ACTIVE_LOW  0x8
#define  GPIO_RISING_EDGE 0x10
#define  GPIO_FALLING_EDGE 0x20
#define  GPIO_TRANSITIONAL 0x40

/* Interrupt styles, these refer to actual values used in reg */
#define GPIO_STYLE_ACTIVE_HIGH   0x0
#define GPIO_STYLE_ACTIVE_LOW    0x1
#define GPIO_STYLE_RISING_EDGE   0x2
#define GPIO_STYLE_FALLING_EDGE  0x3
#define GPIO_STYLE_TRANSITIONAL  0x4

/* Mask used to clear interrupt styles */
#define GPIO_STYLE_CLEAR 0x7
#define GPIO_INTSTYLE_MASK 0x7
#define GPIO_STYLE_SIZE  3
/**
 * @def GPIO_REG_READ(regPtr,res)
 *
 * @brief Macro to read GPIO Register
 *
 * @param [in] regPtr - Pointer to GPIO register
 * @param [out] res - Variable to copy register contents to
 *
 */
#define GPIO_REG_READ(regPtr,res) \
        ((res)=IX_OSSERV_READ_LONG(regPtr))
/**
 * @def GPIO_REG_WRITE(regPtr,val)
 *
 * @brief Macro to write GPIO Register
 *
 * @param [in] regPtr - Pointer to GPIO register
 * @param [in] val - Variable to copy to GPIO register
 *
 */
#define GPIO_REG_WRITE(regPtr,val) \
        IX_OSSERV_WRITE_LONG(regPtr,val)
/* ----------------------------------------------------------------------*/
/* For MTK20150 Firmware       ------------------------------------------*/
/* ----------------------------------------------------------------------*/
#ifdef IX_USE_ADSL_20150 
extern unsigned char ixASW_3_9_22_adsl_init[];
extern unsigned int  ixASW_3_9_22_adsl_init_len;
extern unsigned char ixASW_R3_9_22_adsl_load[];
extern unsigned int  ixASW_R3_9_22_adsl_load_len;
extern unsigned char ixASW_3_8_124_adsl_init[];
extern unsigned int  ixASW_3_8_124_adsl_init_len;
extern unsigned char ixASW_C3_8_121_adsl_load[];
extern unsigned int  ixASW_C3_8_121_adsl_load_len;

#define IX_ASW_CPE_INIT                 ixASW_3_9_22_adsl_init     /**< CPE initialisation firmware byte array */
#define IX_ASW_CPE_INIT_LEN             ixASW_3_9_22_adsl_init_len /**< CPE initialisation firmware byte array
                                                                       length */
#define IX_ASW_CPE_LOAD                 ixASW_R3_9_22_adsl_load    /**< CPE modem firmware byte array */
#define IX_ASW_CPE_LOAD_LEN             ixASW_R3_9_22_adsl_load_len/**< CPE modem firmware byte array
                                                                       length */
#define IX_ASW_CO_INIT                  ixASW_3_8_124_adsl_init     /**< CO initialisation firmware byte array */
#define IX_ASW_CO_INIT_LEN              ixASW_3_8_124_adsl_init_len /**< CO initialisation firmware byte array
                                                                       length */
#define IX_ASW_CO_LOAD                  ixASW_C3_8_121_adsl_load    /**< CO modem firmware byte array */
#define IX_ASW_CO_LOAD_LEN              ixASW_C3_8_121_adsl_load_len/**< CO modem firmware byte array
                                                                       length */

#define IX_ASW_ADSLINIT_FW_LOAD_ADDR      0x00000000     /**< Initialisation firmware load address */
#define IX_ASW_ADSLINIT_FW_JUMP_ADDR      0x00000000     /**< Initialisation firmware jump address */
#define IX_ASW_ADSLPHY_FW_LOAD_ADDR       0x00000000     /**< Modem firmware jump address */
#define IX_ASW_ADSLPHY_FW_JUMP_ADDR       0x00000000     /**< Modem firmware jump address */
#endif /* IX_USE_ADSL_20150 */
/**
 * @def IX_ADSL_CS_ENABLE
 *
 * @brief Macro to enable Chip select for Adsl device on IXDP425
 *
 * @note The chipselect value should match the hardware configuration
 *
 */
#ifdef __linux
#define IX_ADSL_CS_ENABLE *((unsigned long*)IXP425_EXP_CS1)=0xbfff3c03
#else
#ifdef __vxworks
#define IX_ADSL_CS_ENABLE *((unsigned long*)IXP425_EXP_CS1_REG)=0xbfff3c03
#endif
#endif
/* ----------------------------------------------------------------------*/
/* For 20170 Firmware ---------------------------------------------------*/
/* ----------------------------------------------------------------------*/
#ifdef IX_USE_ADSL_20170 
static unsigned char dummy_adsl_load[1];
static unsigned int  dummy_adsl_load_len;
extern unsigned char ixASW_13_9_22_adsl_init[];
extern unsigned int  ixASW_13_9_22_adsl_init_len;
extern unsigned char ixASW_R13_9_22_adsl_load[];
extern unsigned int  ixASW_R13_9_22_adsl_load_len;

#define IX_ASW_CPE_INIT                 ixASW_13_9_22_adsl_init     /**< CPE initialisation firmware byte array */
#define IX_ASW_CPE_INIT_LEN             ixASW_13_9_22_adsl_init_len /**< CPE initialisation firmware byte array
                                                                       length */
#define IX_ASW_CPE_LOAD                 ixASW_R13_9_22_adsl_load    /**< CPE modem firmware byte array */
#define IX_ASW_CPE_LOAD_LEN             ixASW_R13_9_22_adsl_load_len/**< CPE modem firmware byte array
                                                                       length */
#define IX_ASW_CO_INIT                  dummy_adsl_init             /**< 20170 does not support CO */
#define IX_ASW_CO_INIT_LEN              dummy_adsl_init_len         /**< 20170 does not support CO */
#define IX_ASW_CO_LOAD                  dummy_adsl_load             /**< 20170 does not support CO */
#define IX_ASW_CO_LOAD_LEN              dummy_adsl_load_len         /**< 20170 does not support CO */
#define IX_ASW_ADSLINIT_FW_LOAD_ADDR      0x08000000  /**< Initialisation firmware load address */
#define IX_ASW_ADSLINIT_FW_JUMP_ADDR      0x08000000  /**< Initialisation firmware jump address */
#define IX_ASW_ADSLPHY_FW_LOAD_ADDR       0x00000000  /**< Modem firmware jump address */
#define IX_ASW_ADSLPHY_FW_JUMP_ADDR       0x00000000  /**< Modem firmware jump address */
#endif /* IX_USE_ADSL_20170 */

/* ----------------------------------------------------------------------*/
/* Type Definitions  ----------------------------------------------------*/
/* ----------------------------------------------------------------------*/
/**
 * @enum IxAdslGpioSig
 *
 * @brief   GPIO Signal definitions
 *
 */
typedef enum
{
	GPIO_LOW =0,   /**< Gpio pin signal is Low */
	GPIO_HIGH      /**< Gpio pin signal is High */
}  IxAdslGpioSig;
/* ----------------------------------------------------------------------*/
/*  Function Prototypes  ------------------------------------------------*/
/* ----------------------------------------------------------------------*/
/**
 * @fn ixAdslUtilCtrleWrite
 *
 * @brief   Function to write a number of bytes from local buffer into CtrlE address
 *
 * @param   lineNum [in] is the parameter showing which ADSL line to write to
 * @param   pLocalBuffer [in] pointer to Local Buffer
 * @param   address [in] is the CtrlE address to write to
 * @param   number [in] number of bytes to transfer
 *
 * @return TRUE - CtrlE write was successful
 * @return FALSE - CtrlE write failed
 *
 */
BOOL ixAdslUtilCtrleWrite( UINT32 lineNum,
                           const UINT8 *pLocalBuffer,
                           const UINT32 address,
                           const UINT32 number);
/**
 * @fn ixAdslUtilCtrleRead
 *
 * @brief   Function to read a number of bytes to local buffer from CtrlE address
 *
 * @param   lineNum [in] is the parameter showing which ADSL line to write to
 * @param   pLocalBuffer [in] pointer to Local Buffer
 * @param   address [in] is the CtrlE address to read from
 * @param   number [in] number of bytes to transfer
 *
 * @return TRUE - CtrlE read was successful
 * @return FALSE - CtrlE read failed
 *
 */
BOOL ixAdslUtilCtrleRead ( UINT32 lineNum,
                           UINT8 *pLocalBuffer,
                           const UINT32 address,
                           const UINT32 number);
/**
 * @fn ixAdslUtilDeviceDownload
 *
 * @brief Function to download initFirmware and phyFirmware into Adsl phy
 *
 * @param   lineNum [in] is the parameter indicating which ADSL line to write to
 * @param   initFirmware [in] Pointer to Initialisation firmware
 * @param   initFwLen [in] Length of Initialisation firmware
 * @param   phyFirmware [in] Pointer to Modem application firmware
 * @param   phyFwLen [in] Length of Modem application firmware
 *
 * @return IX_ADSL_STATUS_SUCCESS - firmware download succesful
 * @return IX_ADSL_STATUS_FAIL - firmware download failed
 *
 */
IxAdslStatus
ixAdslUtilDeviceDownload ( UINT32 lineNum,
		                   UINT8 *initFirmware,
		                   UINT32 initFwLen,
		                   UINT8 *phyFirmware,
		                   UINT32 phyFwLen);
/**
 * @fn ixAdslUtilMicrosecDelay
 *
 * @brief Function to delay in microseconds
 *
 * @param microseconds [in] number of microseconds to delay
 *
 * @return none
 *
 */
void ixAdslUtilMicrosecDelay(UINT32 microseconds);
/**
 * @fn ixAdslSoftReset
 *
 * @brief Function to reset all adsl phys in IXDP425 platform only
 *
 * @param none
 *
 * @return none
 *
 */
void ixAdslSoftReset(void);
/**
 * @fn ixAdslUtilMemoryUnmap
 *
 * @brief Function to unmap all dynamically mapped addresses
 *
 * @param none
 *
 * @return none
 *
 */
void ixAdslUtilMemoryUnmap(void);
/**
 *
 * ixAdslBaseAddressGet gets the dynamically mapped base address of the adsl component
 *
 */
UINT32 ixAdslBaseAddressGet(void);

/**
 * ixAdslGpioBaseAddressGet gets the dynamically mapped base address of the ixdp425 Gpio
 */
UINT32 ixAdslGpioBaseAddressGet(void);

/**
 * ixAdslTimerBaseAddressGet gets the dynamically mapped base address of the ixdp425 Timer
 */
UINT32 ixAdslTimerBaseAddressGet(void);

#endif
