/**
 * @file IxAdslUtil.c
 *
 * @date 4-Jun-2003
 *
 * @brief  File for Hardware interface functions in the IxAdsl component
 *
 *
 * Design Notes:  this Adsl Driver only Supports Utopia Level 1
 *
 * @version $Revision: 1.1.1.1 $
 * 
 * @par
 * -- Intel Copyright Notice --
 * 
 * @par
 * Copyright 2000-2003 Intel Corporation All Rights Reserved.
 * 
 * @par
 * The source code contained or described herein and all documents
 * related to the source code ("Material") are owned by Intel Corporation
 * or its suppliers or licensors.  Title to the Material remains with
 * Intel Corporation or its suppliers and licensors.
 * 
 * @par
 * The Material is protected by worldwide copyright and trade secret laws
 * and treaty provisions. No part of the Material may be used, copied,
 * reproduced, modified, published, uploaded, posted, transmitted,
 * distributed, or disclosed in any way except in accordance with the
 * applicable license agreement .
 * 
 * @par
 * No license under any patent, copyright, trade secret or other
 * intellectual property right is granted to or conferred upon you by
 * disclosure or delivery of the Materials, either expressly, by
 * implication, inducement, estoppel, except in accordance with the
 * applicable license agreement.
 * 
 * @par
 * Unless otherwise agreed by Intel in writing, you may not remove or
 * alter this notice or any other notice embedded in Materials by Intel
 * or Intel's suppliers or licensors in any way.
 * 
 * @par
 * For further details, please see the file README.TXT distributed with
 * this software.
 * 
 * @par
 * -- End Intel Copyright Notice --
 */


#include <stdio.h>
#include <taskLib.h>
#include "IxAdsl.h"
#include "IxAdslCtrleLib.h"
#include "IxAdsl_p.h"
#include "IxAdslUtil.h"
#include "IxAdslCtrleConstants.h"


/* ----------------------------------------------------------------------------
* Download a Block of data into CTRLE Buffer
*/
PRIVATE IxAdslStatus ixAdslUtilDownload ( UINT32 lineNum,
                                UINT8 *pLocalBuffer,
                                UINT32 len,
                                UINT8 *pRemoteBuffer,
                                UINT8 *pJumpAddress );
/* ----------------------------------------------------------------------------
* Write a Block of data into CTRLE Buffer
*/
PRIVATE IxAdslStatus ixAdslUtilBlockWrite( UINT32 lineNum,
                        UINT8 *pLocalBuffer,
                        UINT8 *pRemoteBuffer,
                        UINT32 number);
/* ----------------------------------------------------------------------------
* Utility to configure GPIO line
*/
PRIVATE IxAdslStatus ixAdslUtilGpioLineConfig ( UINT8 lineNo,
                                                UINT32 style);
/* ----------------------------------------------------------------------------
* Utility to set GPIO line
*/
PRIVATE IxAdslStatus ixAdslUtilGpioLineSet ( UINT8 lineNo,
                                             IxAdslGpioSig value);
/**
 *
 * ixAdslBaseAddress stores the dynamically mapped base address of the adsl component
 *
 */
static UINT32 ixAdslBaseAddress = 0;
/**
 * ixAdslGpioBaseAddress stores the dynamically mapped base address of the ixdp425 Gpio
 */
static UINT32 ixAdslGpioBaseAddress = 0;
/**
 * ixAdslTimerBaseAddress stores the dynamically mapped base address of the ixdp425 Timer
 */
static UINT32 ixAdslTimerBaseAddress = 0;


/**
 *
 * Fast mutex locks for ctrle read and ctrle write access 
 *
 */
IxFastMutex ixAdslCtrleAccessLock;
/* ----------------------------------------------------------------------------
* Write Data to CTRLE Buffer
*/
BOOL ixAdslUtilCtrleWrite ( UINT32 lineNum,
                            const UINT8 *pLocalBuffer,
                            const UINT32 address,
                            const UINT32 number)
{ 
    BOOL   rc=TRUE; /* Success by default.      */
    UINT32 addrOffset;
    UINT32 numBytes=number;
    while(ixOsServFastMutexTryLock(&ixAdslCtrleAccessLock)==IX_FAIL)
    {
        ixOsServTaskSleep(IX_ADSL_POLL_DELAY);
    };

    /* Delay if not ready for write */
    if ( !CTRLE_TX_COM_AV(lineNum) )
    {
        CTRLE_SLEEP;
    }
    /* -------------------------------------------------------------------------
     * Write Data to CTRLE Data Buffer
     */
    addrOffset = address;
    while ( numBytes )
    {
        CTRLE_WRITE_REG((IXP425_ADSL_LINE_ADDR(lineNum)+ addrOffset),*pLocalBuffer++);
        addrOffset++;
        numBytes--;
    }

    ixOsServFastMutexUnlock(&ixAdslCtrleAccessLock);
    return(rc);
}

/* ----------------------------------------------------------------------------
* Read Data from CTRLE Buffer
*/
BOOL ixAdslUtilCtrleRead (  UINT32 lineNum,
                            UINT8 *buffer,
                            const UINT32 address,
                            const UINT32 number)
{
    UINT32  addrOffset;
    UINT32 x;
    BOOL rxCmdAvail;
    UINT32 numBytes=number;
    while(ixOsServFastMutexTryLock(&ixAdslCtrleAccessLock)==IX_FAIL)
    {
        ixOsServTaskSleep(IX_ADSL_POLL_DELAY);
    };
    /* Check if this is request for one byte read from RxCommand Register */
    if ((RXCOMMANDREG == address)&& (numBytes == 1))
    {
        /* Check if there is a new value waiting */
        rxCmdAvail = FALSE;
        x = CTRLE_WAIT_FOR_CMD_REG_CNT;
        while(--x)
        {
            if (CTRLE_RX_COM_AV(lineNum) )
            {
                /* There is an Rx command available. */
                *buffer = CTRLE_RX_REG(lineNum);
                ixOsServFastMutexUnlock(&ixAdslCtrleAccessLock);
                return (TRUE);
            }
            CTRLE_SLEEP;
        }
        /* Nothing to read or timeout has occured */
        ixOsServFastMutexUnlock(&ixAdslCtrleAccessLock);
        return(FALSE);
    }  /* End of RxCommand Register read */

    /* Below is one or more byte read from CtrlE address space */
    addrOffset = address;
    while (numBytes)
    {
        *buffer++ = CTRLE_READ_REG(IXP425_ADSL_LINE_ADDR(lineNum)+addrOffset ) ;
        addrOffset++;
        --numBytes;
    }
    ixOsServFastMutexUnlock(&ixAdslCtrleAccessLock);
    return(TRUE);
}

/* ----------------------------------------------------------------------------
* Entry point to Download the ADSL controller via CtrlE interface.
*/
IxAdslStatus
ixAdslUtilDeviceDownload ( UINT32 lineNum,
                        UINT8* initFirmware,
                        UINT32 initFwLen,
                        UINT8* phyFirmware,
                        UINT32 phyFwLen)
{
    IxAdslStatus status;
    UINT8 rxAv,rxCom,txCom,txAv;
    UINT32 tmocnt;
    UINT8 initResult = 0;
    DEBUG_CMD(DEBUG_LOG("Downloading the init code: size  = %d ...\n", initFwLen));
    /* ----------------------------------------------------------------------------
    * Check boot code parameters
    */
    if ( (0==initFwLen) || (NULL==initFirmware) )
    {
        /* Bad firmware specified */
        DEBUG_CMD(DEBUG_LOG("Bad bootcode parameters specified...\n"));
        return (IX_ADSL_STATUS_FAIL);
    }
    else
    {
        DEBUG_CMD(DEBUG_LOG("\nDownloading bootcode: %d bytes\n", initFwLen));
        /* ----------------------------------------------------------------------------
        * Download boot code into Adsl Phy
        */
        status = ixAdslUtilDownload( lineNum,
                                    initFirmware,
                                    initFwLen,
                                    (UINT8*)IX_ASW_ADSLINIT_FW_LOAD_ADDR,
                                    (UINT8*)IX_ASW_ADSLINIT_FW_JUMP_ADDR);
    }
    if(IX_ADSL_STATUS_FAIL == status)
    {
        DEBUG_CMD(DEBUG_LOG("\nBootcode download failed! \n"));
        return (status);
    }
    DEBUG_CMD(DEBUG_LOG("\nBootcode download successful! Waiting for boot to complete...\n"));
    /* ---------------------------------------------------------------------------
    * Wait for Boot to complete
    */
    tmocnt= (UINT32)IX_ADSL_BOOT_TIMEOUT_COUNTER;
    while ((INIT_COMPLETE != initResult) && (tmocnt > 0))
    {
    	tmocnt --;
        initResult = CTRLE_READ_REG(CTRLE_RX_REG_ADDR(lineNum));
        ixOsServTaskSleep(IX_ADSL_BOOT_TIMEOUT_POLL_INTERVAL);
    }

    DEBUG_CMD(DEBUG_LOG("\nBoot returns : 0x%02x\n", initResult));
    /* If Rx Command NOT Available then we are clear to download modem code */
    tmocnt= (UINT32)IX_ADSL_BOOT_TIMEOUT_COUNTER;
    while ((0x1 == (CTRLE_RX_COM_AV(lineNum) & 0x1)) && (tmocnt > 0) )
    {
        tmocnt--;
        DEBUG_CMD(DEBUG_LOG("+"));
        ixOsServTaskSleep(IX_ADSL_BOOT_TIMEOUT_POLL_INTERVAL);
    }
    if (0==tmocnt)
    {
        DEBUG_CMD(DEBUG_LOG("\nTimeout between boot load and application load.\n"));
        return(IX_ADSL_STATUS_FAIL); /* Failed */
    }
    /* Clear all registers */
    rxCom = CTRLE_RX_REG(lineNum);
    txCom = CTRLE_TX_REG(lineNum);
    rxAv = CTRLE_RX_COM_AV(lineNum);
    txAv = CTRLE_TX_COM_AV(lineNum);
    /* ------------------------------------------------------------------------
    * Check firmware parameters
    */
    if ((0==phyFwLen)||(NULL==phyFirmware))
    {
        /* Bad firmware specified */
        DEBUG_CMD(DEBUG_LOG("\nBad modem firmware parameters specified: size  = %d ...\n",phyFwLen));
        status = IX_ADSL_STATUS_FAIL;
    }
    else
    {
        /* --------------------------------------------------------------------
        * Download firmware into Adsl Phy
        */
        DEBUG_CMD(DEBUG_LOG("\nDownloading application code: size  = %d ...\n",phyFwLen));
        status = ixAdslUtilDownload( lineNum,
                                    phyFirmware,
                                    phyFwLen,
                                    (UINT8*)IX_ASW_ADSLPHY_FW_LOAD_ADDR,
                                    (UINT8*)IX_ASW_ADSLPHY_FW_JUMP_ADDR);
    }
    if(IX_ADSL_STATUS_FAIL == status)
    {
        DEBUG_CMD(DEBUG_LOG("\nModem firmware download failed! \n"));
        return (status);
    }
    DEBUG_CMD(DEBUG_LOG("\nModem firmware download successful! \n"));
    return(status);
}
/* ----------------------------------------------------------------------------
* Download local buffer into CtrleE Buffer
*/
IxAdslStatus
ixAdslUtilDownload( UINT32 lineNum,
                    UINT8 *pLocalBuffer,
                    UINT32 len,
                    UINT8 *pRemoteBuffer,
                    UINT8 *pJumpAddress )
{
    IxAdslStatus status= IX_ADSL_STATUS_SUCCESS;
    int offset=0;
    int i;

    /*********download***************************/

    while (len>0 && (IX_ADSL_STATUS_SUCCESS==status))
    {
        DEBUG_CMD(DEBUG_LOG("."));
        if (len >= CTRLE_LOCALWRITEBUFFERSIZE)
        {
            status = ixAdslUtilBlockWrite(  lineNum,
                                            pLocalBuffer+offset,
                                            pRemoteBuffer+offset,
                                            CTRLE_LOCALWRITEBUFFERSIZE);
            offset+=CTRLE_LOCALWRITEBUFFERSIZE;
            len-=CTRLE_LOCALWRITEBUFFERSIZE;
        }
        else
        {
            status = ixAdslUtilBlockWrite(  lineNum,
                                            pLocalBuffer+offset,
                                            pRemoteBuffer+offset,
                                            len);
            len=0;
        }
    }

    if ( IX_ADSL_STATUS_FAIL == status )
    {
        DEBUG_CMD(DEBUG_LOG("Download failed.."));
        return(status);
    }

    /*
    * Jump to relevant code.
    */

    /*If download is ok then jump to code, no acknowledge required*/
    if (IX_ADSL_STATUS_SUCCESS==status)
    {
        for( i=0;i<IX_ADSL_SIZE4BYTE;i++)
        {
            CTRLE_WRITE_REG((IXP425_ADSL_LINE_ADDR(lineNum) + (BLCKBADDRREG+i) ),
                            ((UINT32)pJumpAddress) % IX_ADSL_0x100);
            pJumpAddress = ((UINT8*)(((UINT32)pJumpAddress)/IX_ADSL_0x100));
        }
        CTRLE_WRITE_REG(CTRLE_TX_REG_ADDR(lineNum),MODEM_JUMP_ADDR);
    }
    return(status); /* o.k */
}

/* ----------------------------------------------------------------------------
* Write a Block of data into CTRLE Buffer
*/
IxAdslStatus
ixAdslUtilBlockWrite( UINT32 lineNum,
        UINT8 *pLocalBuffer,
        UINT8 *pRemoteBuffer,
        UINT32 number)
{
    IxAdslStatus status= IX_ADSL_STATUS_SUCCESS;
    UINT32 pRemoteBufferint=(UINT32)pRemoteBuffer;
    UINT32 lnumber=number;
    UINT32 tmocnt;
    UINT32 i;
    UINT8 rxResp=0;
    /********issue dump block command****************/
    for(i=0;i<IX_ADSL_SIZE4BYTE;i++)
    {
        CTRLE_WRITE_REG( (IXP425_ADSL_LINE_ADDR(lineNum)+(BLCKBADDRREG+i)),
                          pRemoteBufferint % IX_ADSL_0x100);
    	pRemoteBufferint/=IX_ADSL_0x100;
    }
    for(i=0;i<IX_ADSL_SIZE2BYTE;i++)
    {
        CTRLE_WRITE_REG( (IXP425_ADSL_LINE_ADDR(lineNum)+(BLCKLENREG+i)),
                          lnumber % IX_ADSL_0x100);
	    lnumber=lnumber / IX_ADSL_0x100;
    }
    /********************************************/
    for(i=0;i<number;i++)
    {
        CTRLE_WRITE_REG( (IXP425_ADSL_LINE_ADDR(lineNum)+(BLCKDATAREG+i)),
                         *(pLocalBuffer+i));
    }
    CTRLE_WRITE_REG( CTRLE_TX_REG_ADDR(lineNum),MODEM_DUMP_BLOCK);

    /*********wait for acknowledge*******************/
    tmocnt=IX_ADSL_BOOT_TIMEOUT_COUNTER;

    while (  (CTRLE_RX_COM_AV(lineNum) & 0x1) != 0x1 && tmocnt > 0 )
    {
        tmocnt--;
    }
    if (0==tmocnt)
    {
        /* Try a slower delay since quick polling did not do it
        */
        tmocnt = IX_ADSL_BOOT_TIMEOUT_COUNTER/10;
        while (  (CTRLE_RX_COM_AV(lineNum) & 0x1) != 0x1 && tmocnt > 0 )
        {
            DEBUG_CMD(DEBUG_LOG("!"));
            tmocnt--;
            ixOsServTaskSleep(IX_ADSL_BOOT_TIMEOUT_POLL_INTERVAL);
        }
    }
    if (0==tmocnt)
    {
        status=IX_ADSL_STATUS_FAIL;
        DEBUG_CMD(DEBUG_LOG("ALERT:status Timed out..\n"));
    }
    else if (IX_ADSL_STATUS_SUCCESS == status) /* No timeout. */
        {
            /* Check return code */
            rxResp = CTRLE_READ_REG(CTRLE_RX_REG_ADDR(lineNum));
#ifdef IX_USE_ADSL_20150
            if ( rxResp  != IX_ASW_BLOCKDUMP_ACK )
            {
                status = IX_ADSL_STATUS_FAIL;
            }
#else
#ifdef IX_USE_ADSL_20170
            if ( rxResp  != IX_ASW_BLOCKDUMP_ACK_166 )
            {
                status = IX_ADSL_STATUS_FAIL;
            }
#endif
#endif
        }
    if(IX_ADSL_STATUS_FAIL==status)
    {
        DEBUG_CMD(DEBUG_LOG("%s:ALERT: Invalid value for Modem_Block_Dump_Ack = 0x%02x\n", __FUNCTION__, rxResp));
    }
    return status;
}
/*****************************************************************************
* ixAdslSoftReset - Reset Adsl Chip
*
* RETURNS: N/A
*/
void ixAdslSoftReset(void )
{
#ifdef IX_USE_ADSL_20150
    ixAdslUtilGpioLineConfig(GPIO_PIN_12,GPIO_OUT);
    ixAdslUtilGpioLineSet(GPIO_PIN_12,GPIO_HIGH);
    ixOsServSleep(1);
    ixAdslUtilGpioLineSet(GPIO_PIN_12,GPIO_LOW);
    ixOsServSleep(500);
    ixAdslUtilGpioLineSet(GPIO_PIN_12,GPIO_HIGH);
    ixOsServSleep(1);
#endif
}

/******************************************************************************
*
* ixAdslUtilGpioLineConfig - configure a GPIO pin
*
* This routine configures a GPIO pin/line for use as either an input or output.
* Line config for input is not supported since this is not applicable for softreset
*
* RETURNS: OK when line is successfully configured; ERROR if the line number
*	   passed is out of range, or an unknown style is requested for an
*	   interrupt.
*/
IxAdslStatus ixAdslUtilGpioLineConfig (UINT8 lineNo, UINT32 style)
{
    volatile UINT32 enableReg;

    if (lineNo >  GPIO_PIN_MAX)
    {
    	return (IX_ADSL_STATUS_FAIL);
    }
    GPIO_REG_READ( GPIO_GPOER, enableReg);

    if (style &  GPIO_OUT)
    {
        enableReg &= ~(0x1 << lineNo);
    }
    else	/* Unknown style options */
    {
        return (IX_ADSL_STATUS_FAIL);
    }
    GPIO_REG_WRITE(GPIO_GPOER, enableReg);
    return (IX_ADSL_STATUS_SUCCESS);
}

/******************************************************************************
*
* ixAdslUtilGpioLineSet - set the value of a GPIO pin configured for output
*
* RETURNS: OK when pin is successfully set; ERROR if the pin number
*	   passed is out of range.
*/
IxAdslStatus ixAdslUtilGpioLineSet (UINT8 lineNo,  IxAdslGpioSig value)
{
    UINT32 registerValue;
    if (lineNo > GPIO_PIN_MAX)
    {
        return (IX_ADSL_STATUS_FAIL);
    }
    if (GPIO_HIGH==value)
    {
        GPIO_REG_READ(GPIO_GPOUTR, registerValue);
        GPIO_REG_WRITE(GPIO_GPOUTR, (registerValue |= (0x1 << (lineNo))) );
    }
    else if (GPIO_LOW==value)
    {
        GPIO_REG_READ(GPIO_GPOUTR, registerValue);
        GPIO_REG_WRITE(GPIO_GPOUTR, (registerValue &= ~(0x1 << (lineNo))) );
    }
    return (IX_ADSL_STATUS_SUCCESS);
}

/*****************************************************************************
* ixAdslUtilMicrosecDelay - Microsecond delay
*
* RETURNS: N/A
*/

void ixAdslUtilMicrosecDelay(UINT32 microseconds)
{
    UINT32 delay = 0;
    UINT32 lastTimestamp = *((volatile int *)OSTS);
    UINT32 currentTimestamp;
    UINT32 delta;

    while (delay < microseconds * PERIPHERAL_BUS_CLOCK)
    {
        currentTimestamp = *((volatile int *)OSTS);
        delta = currentTimestamp > lastTimestamp ? currentTimestamp - lastTimestamp :
        0xffffffff - lastTimestamp + currentTimestamp;
        delay += delta;
        lastTimestamp = currentTimestamp;
    }
}
/*****************************************************************************
* ixAdslUtilMemoryUnmap - Release Dynamically Allocated Memory
*
* RETURNS: N/A
*/
void ixAdslUtilMemoryUnmap(void)
{
#ifdef IX_USE_ADSL_20150
    IX_OSSERV_MEM_UNMAP(ixAdslGpioBaseAddress);
    IX_OSSERV_MEM_UNMAP(ixAdslTimerBaseAddress);
    ixAdslGpioBaseAddress = 0;
    ixAdslTimerBaseAddress = 0;
#endif
    IX_OSSERV_MEM_UNMAP(ixAdslBaseAddress);
    ixAdslBaseAddress = 0;
}
/**
 *
 * ixAdslBaseAddressGet gets the dynamically mapped base address of the adsl component
 *
 */
UINT32 ixAdslBaseAddressGet(void)
{
    if(0==ixAdslBaseAddress)
    {
    /* Map Adsl Base address that is used in for Adsl CtrlE interface */
#ifdef IX_USE_ADSL_20150
    ixAdslBaseAddress = (UINT32)IX_OSSERV_MEM_MAP(IX_OSSERV_EXP_BUS_CS1_PHYS_BASE,IX_OSSERV_EXP_BUS_CS1_MAP_SIZE);
#endif /* IX_USE_ADSL_20150 */
    }
    return ixAdslBaseAddress;
}

/**
 * ixAdslGpioBaseAddressGet gets the dynamically mapped base address of the ixdp425 Gpio
 */
UINT32 ixAdslGpioBaseAddressGet(void)
{
#ifdef IX_USE_ADSL_20150
    if(0==ixAdslGpioBaseAddress)
    {
        ixAdslGpioBaseAddress =(UINT32)IX_OSSERV_MEM_MAP( IX_OSSERV_GPIO_PHYS_BASE,
                                                          IX_OSSERV_GPIO_MAP_SIZE);
    }
#endif
    return ixAdslGpioBaseAddress;
}

/**
 * ixAdslTimerBaseAddressGet gets the dynamically mapped base address of the ixdp425 Timer
 */
UINT32 ixAdslTimerBaseAddressGet(void)
{
#ifdef IX_USE_ADSL_20150
    if(0==ixAdslTimerBaseAddress)
    {
        ixAdslTimerBaseAddress = (UINT32)IX_OSSERV_MEM_MAP( IX_OSSERV_OSTS_PHYS_BASE,
                                                            IX_OSSERV_OSTS_MAP_SIZE);
    }
#endif
    return ixAdslTimerBaseAddress;
}


